// SVGALib graphics driver
//
// Fri Jan 30 17:19:54 EST 1998
// Patrick McCarthy
// Finished primary creation, with keyboard and mode handling.
// No mouse stuff yet though.
//
// Fri Jan 30 16:25:54 EST 1998
// Patrick McCarthy
// Time of creation.

#ifdef HAVE_SVGALIB
#include <stdlib.h>
#include <stdio.h>
#include <vga.h>
#include <vgakeyboard.h>
#include <vgamouse.h>

#include "error.h"
#include "gr.h"
#include "mono.h"
#include "key.h"
#include "timer.h"

// First we have the constants.
const int svgalib_vidmode_table[SM_Max] = {
G320x200x256,		// SM_320x200C
G320x200x256,		// SM_320x200U
G320x240x256,		// SM_320x240U
-1,			// SM_360x200U
-1,			// SM_360x240U
-1,			// SM_376x282U
G320x400x256,		// SM_320x400U
-1,			// SM_320x480U
-1,			// SM_360x400U
G360x480x256,		// SM_360x480U
-1,			// SM_360x360U
-1,			// SM_376x308U
-1,			// SM_376x564U
-1,			// SM_640x400V	What in the?... we should have this.
G640x480x256,		// SM_640x480V
G800x600x256,		// SM_800x600V
G1024x768x256,		// SM_1024x768V
G640x480x32K,		// SM_640x480V15
G800x600x32K,		// SM_800x600V15
};

// Since descent's bitchy about keys, we merge them all to be descent's
// format. (scancode+(0x80*E0code))
const int svgalib_scancode_to_descent[128] = {
0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,0x0a,0x0b,0x0c,0x0d,0x0e,0x0f,
0x10,0x11,0x12,0x13,0x14,0x15,0x16,0x17,0x18,0x19,0x1a,0x1b,0x1c,0x1d,0x1e,0x1f,
0x20,0x21,0x22,0x23,0x24,0x25,0x26,0x27,0x28,0x29,0x2a,0x2b,0x2c,0x2d,0x2e,0x2f,
0x30,0x31,0x32,0x33,0x34,0x35,0x36,0x37,0x38,0x39,0x3a,0x3b,0x3c,0x3d,0x3e,0x3f,
0x40,0x41,0x42,0x43,0x44,0x45,0x46,0x47,0x48,0x49,0x4a,0x4b,0x4c,0x4d,0x4e,0x4f,
0x50,0x51,0x52,0x53,0x54,0x55,0x56,0x57,0x58,0x59,0x5a,0x5b,0x5c,0x5d,0x5e,0x5f,
0x9c,0x9d,0xb5,0xb7,0xb8,0x61,0xc7,0xc8,0xc9,0xcb,0xcd,0xcf,0xd0,0xd1,0xd2,0xd3,
0x70,0x71,0x72,0x73,0x74,0x75,0x76,0x61,0x78,0x79,0x7a,0x7b,0x7c,0x7d,0x7e,0x7f,
};

// Locals
static int current_draw_page=0;
static int sxsize=0,sysize=0;
static int granularity=0;
static int running=0;
static int bpp=0,bm_type=-1;

// Graphics shtuff
int svgalib_set_mode(int mode) {
	vga_modeinfo *modeinfo;

	if (mode<0) {
		vga_setmode(TEXT); // This can't fail.
		running=0; bm_type=-1;
		return 0;
	}
	if (svgalib_vidmode_table[mode]<0) return 0;
	
	if (vga_setmode(svgalib_vidmode_table[mode])<0) {
		vga_setmode(TEXT);
		mprintf((0,"vga_setmode failure on mode %d!",mode));
		return -1;
	}
	
	if (!(modeinfo=vga_getmodeinfo(svgalib_vidmode_table[mode]))) {
		vga_setmode(TEXT);
		mprintf((0,"vga_getmodeinfo failure on mode %d!",mode));
		return -1;
	}

	sxsize=modeinfo->width;
	sysize=modeinfo->height;
	bpp=modeinfo->bytesperpixel<<3;
	// *shrug*
	granularity=65536;
	
	if (modeinfo->flags&IS_MODEX) bm_type=BM_MODEX;
	else if ((modeinfo->width*modeinfo->height)>granularity) bm_type=BM_SVGA;
	else bm_type=BM_LINEAR;
	
	running=1;
	
	// Should check for modeinfo here.
	return 0;
}

void svgalib_set_palette(char *pal) {
	register int i;
	
	if (!running) return;
	for (i=0;i<256;i++) {
		int r=*pal++,g=*pal++,b=*pal++;
		vga_setpalette(i,r,g,b);
	}
	return;
}

void svgalib_get_palette(char *pal) {
	register int i;
	
	if (!running) return;
	for (i=0;i<256;i++) {
		int r,g,b;
		vga_getpalette(i,&b,&g,&r);
		*pal++=r; *pal++=g; *pal++=b;
	}
	return;
}

void svgalib_set_drawing_page(int page) {
	if (!running) return;
	current_draw_page=(sxsize*sysize*((bpp+7)/8))*page;
	if (current_draw_page) {current_draw_page/=granularity; current_draw_page+=page; }
}

void svgalib_set_viewing_page(int page) {
	int vpage;
	if (!running) return;
	vpage=(sxsize*sysize*((bpp+7)/8))*page;
	if (vpage) { vpage/=granularity; vpage+=page; }
	vga_setdisplaystart(vpage<<granularity);
}

void svgalib_bank(int bank) {
	if (!running) return;
	vga_setpage(current_draw_page+bank);
}

// I can't believe I have to do this. I thought powerful operating systems
// were supposed to *eliminate* the need for things like this. -- PCM
// 02/04/98: D'OH!! port_out(number,port) **NOT** port_out(port,number)
//           Having your system lock and spurt out eth0 not found messages
//           isn't fun.
#ifdef __alpha__
static void port_out(int value, int port) {
    outb (value, port);
}
#else
static __inline__ void port_out(int value, int port) {
	__asm__ volatile ("outb %0,%1"
		::"a"((unsigned char) value),"d"((unsigned short) port));
}
#endif

void svgalib_modex_plane(int plane) {
	if (!running||plane<0||plane>3) return;
	port_out(0x02,0x3C4); port_out(1<<plane,0x3C5);
}

unsigned char *svgalib_get_buffer() {
	return vga_getgraphmem();
}

int svgalib_get_gran() {
	return granularity;
}

void svgalib_update() {
}

int svgalib_bm_type() {
	return bm_type;
}

// SVGALib keyboard handler..
static void svgalib_keyboard_handler(int scancode, int newstate) {
	int dcode;
	
	if (scancode<0||scancode>127) {
		mprintf((0,"svgalib_keyboard_handler: scancode %d out of range 0-127!\n",scancode));
		return;
	}
	
	dcode=svgalib_scancode_to_descent[scancode];
	if (newstate) {
		// keydown
		keyd_last_pressed=dcode;
		keyd_time_when_last_pressed=timer_get_fixed_secondsX();
		if (!keyd_pressed[dcode]) {
			key_data.NumDowns[dcode]++;
			keyd_pressed[dcode]=1;
			key_data.TimeKeyWentDown[dcode]=timer_get_fixed_secondsX();
		} else if (!keyd_repeat)
			dcode=0xAA;
		
		if (dcode!=0xAA) {
			int kcode,temp;
			// Handle annoying repeat stuff
			kcode=dcode
#ifndef NDEBUG
				|(keyd_pressed[KEY_DELETE]?KEY_DEBUGGED:0)
#endif
				|(keyd_pressed[KEY_LSHIFT]||keyd_pressed[KEY_RSHIFT]?KEY_SHIFTED:0)
				|(keyd_pressed[KEY_LALT]||keyd_pressed[KEY_RALT]?KEY_ALTED:0)
				|(keyd_pressed[KEY_LCTRL]||keyd_pressed[KEY_RCTRL]?KEY_CTRLED:0);
			if ((temp=key_data.keytail+1)>=KEY_BUFFER_SIZE) temp=0;
			if (temp!=key_data.keyhead) {
				key_data.keybuffer[key_data.keytail]=kcode;
				key_data.time_pressed[key_data.keytail]=keyd_time_when_last_pressed;
				key_data.keytail=temp;
			}
		}
	} else {
		// keyup
		keyd_last_released=newstate;
		keyd_pressed[dcode]=0;
		key_data.NumUps[dcode]++;
		key_data.TimeKeyHeldDown[dcode]+=timer_get_fixed_secondsX()-key_data.TimeKeyWentDown[dcode];
	}
	return;
}

int svgalib_keyboard_init(void) {
	if (keyboard_init()) {
		fprintf(stderr,"Couldn't initialize keyboard! AIIEEEEEEE!! Bailing!\n");
		exit(1);
	}
	keyboard_translatekeys(DONT_CATCH_CTRLC);
	keyboard_seteventhandler(svgalib_keyboard_handler);
	return 0;
}

void svgalib_keyboard_update(void) {
	keyboard_update();
}

void svgalib_keyboard_close(void) {
	keyboard_setdefaulteventhandler();
	keyboard_close();
	return;
}

// Mouse stuff should be down here.
void svgalib_mouse_update(int *deltax, int *deltay, int *buttons)
{
   int vga_but;
   mouse_update();
   *deltax = mouse_getx();
   *deltay = mouse_gety();
   vga_but = mouse_getbutton();
   /* map bit 0,1,2 (R,M,L) to 1,2,0 */
   *buttons = ((vga_but & 1) << 1) | ((vga_but & 2) << 1) | 
    ((vga_but & 4) >> 2);
   mouse_setposition(0,0);
}

int svgalib_mouse_init(void) {
   int fd;
   
   fd=mouse_init_return_fd("/dev/mouse", vga_getmousetype(), 150);
   if (fd<0) {
       perror("mouse_init");
       fprintf(stderr,"SVGALib: failed to open mouse device %d\n",fd);
       return 1;
   }
   mouse_setxrange(-32768,32767);
   mouse_setyrange(-32768,32767);
   mouse_setposition(0,0);
   return 0;
}

void svgalib_mouse_close(void) {
	mouse_close();
}

#endif
