#ifdef WANT_JOYSTICK
#include <stdlib.h>
#include <stdio.h>
#include "joystick.h"
#include <sys/ioctl.h>
#include <fcntl.h>
#include <unistd.h>

#include "types.h"
#include "mono.h"
#include "joy.h"

#define MAX_AXES	32
#define MAX_BUTTONS	64

char joy_installed = 0;
char joy_present = 0;


typedef struct j_Device {
	int		device_number;
	int		version;
	int		buffer;
	char		num_buttons;
	char		num_axes;
} j_Device;

typedef struct j_Axis {
	int		value;
	int		min;
	int		center;
	int		max;
	int		joydev;
} j_Axis;


typedef struct j_Button {
	ubyte		state;
	ubyte		last_state;
	int		timedown;
	ubyte		downcount;
	int		joydev;
} j_Button;


j_Device joystick[4];
j_Axis axis[MAX_AXES];
j_Button button[MAX_BUTTONS];
int num_axes = 0, num_buttons = 0;
int timer_rate;


int j_Get_axis_number (int joydev_number, int joy_axis_number) {
	int i, all_axis_number = 0;
	for (i = 0; i < joydev_number; i++) {
		all_axis_number += joystick[i].num_axes;
	}
	all_axis_number += joy_axis_number;
	return all_axis_number;
}


int j_Get_button_number (int joydev_number, int joy_button_number) {
	int i, all_button_number = 0;
	for (i = 0; i < joydev_number; i++) {
		all_button_number += joystick[i].num_buttons;
	}
	all_button_number += joy_button_number;

	return all_button_number;
}


int j_Get_joydev_axis_number (int all_axis_number) {
	int i, joy_axis_number = all_axis_number;

	for (i = 0; i < axis[all_axis_number].joydev; i++) {
		joy_axis_number -= joystick[i].num_axes;
	}		

	return joy_axis_number;
}


int j_Get_joydev_button_number (int all_button_number) {
	int i, joy_button_number = all_button_number;

	for (i = 0; i < button[all_button_number].joydev; i++) {
		joy_button_number -= joystick[i].num_buttons;
	}		

	return joy_button_number;
}


int j_Get_joydev_version (int joydev) {
	return joystick[joydev].version;
}


int j_Get_button_joydev (int button_number) {
	return button[button_number].joydev;
}


int j_Get_axis_joydev (int axis_number) {
	return axis[axis_number].joydev;
}


int j_Get_num_buttons () {
	return num_buttons;
}


int j_Get_num_axes () {
	return num_axes;
}


int j_Get_axis_range_of_motion (int axisnum) {
	return (axis[axisnum].max - axis[axisnum].min);
}


int j_Update_state () {
	int num_processed = 0, current_joystick, i;
	struct js_event current_event;
	struct JS_DATA_TYPE joy_data;

	for (current_joystick = 0; current_joystick < 4; current_joystick++) {
		if (joystick[current_joystick].buffer) {
			if (joystick[current_joystick].version) {
				while (read (joystick[current_joystick].buffer, &current_event, sizeof (struct js_event)) > 0) {
					num_processed++;
					switch (current_event.type & ~JS_EVENT_INIT) {
						case JS_EVENT_AXIS:
							axis[j_Get_axis_number (current_joystick, current_event.number)].value = current_event.value;
							break;
						case JS_EVENT_BUTTON:
							if (current_event.value == 1) button[j_Get_button_number(current_joystick, current_event.number)].downcount++;
							button[j_Get_button_number(current_joystick, current_event.number)].state = current_event.value;
							break;
					}
				}
			} else {
				read (joystick[current_joystick].buffer, &joy_data, JS_RETURN);
				axis[j_Get_axis_number (current_joystick, 0)].value = joy_data.x;
				axis[j_Get_axis_number (current_joystick, 1)].value = joy_data.y;
				button[j_Get_button_number (current_joystick, 0)].state = (joy_data.buttons & 0x01);
				button[j_Get_button_number (current_joystick, 1)].state = (joy_data.buttons & 0x02) >> 1;
//				button[j_Get_button_number (current_joystick, 2)].state = (joy_data.buttons & 0x04) >> 2;
//				button[j_Get_button_number (current_joystick, 3)].state = (joy_data.buttons & 0x08) >> 3;
			}
		}
	}

	for (i = 0; i < num_buttons; i++) {
		if (button[i].last_state == button[i].state && button[i].state) button[i].timedown = 10000;
		else button[i].timedown = 0;
		button[i].last_state = button[i].state;
	}

	return num_processed;
}


void joy_set_cal_vals(int *axis_min, int *axis_center, int *axis_max) {
	int i;

	for (i = 0; i < num_axes; i++) {
		if (!joystick[axis[i].joydev].version) {
			axis[i].center = axis_center[i];
			axis[i].min = axis_min[i];
			axis[i].max = axis_max[i];
		}
	}
}


void joy_get_cal_vals(int *axis_min, int *axis_center, int *axis_max) {
	int i;

	for (i = 0; i < num_axes; i++) {
		axis_center[i] = axis[i].center;
		axis_min[i] = axis[i].min;
		axis_max[i] = axis[i].max;
	}
}


void joy_set_min (int axis_number, int value) {
	axis[axis_number].min = value;
}


void joy_set_center (int axis_number, int value) {
	axis[axis_number].center = value;
}


void joy_set_max (int axis_number, int value) {
	axis[axis_number].max = value;
}


ubyte joy_get_present_mask() {
	return 1;
}


void joy_set_timer_rate (int max_value) {
	timer_rate = max_value;
}


int joy_get_timer_rate() {
	return timer_rate;
}


void joy_flush() {
	int i;

	if (!joy_installed) return;

	for (i = 0; i < num_buttons; i++) {
		button[i].state = 0;	
		button[i].last_state = 0;
		button[i].timedown = 0;	
		button[i].downcount = 0;	
	}
	
}


ubyte joystick_read_raw_axis (ubyte mask, int *axes) {
	int i;
	
	j_Update_state();

	for (i = 0; i <= num_axes; i++) {
		axes[i] = axis[i].value;
	}

	return 0;
}


int joy_init () {
	int i, j;

	if (joy_installed) return 0;
	joy_flush ();

	if (!joy_installed)	{

		printf ("Initializing joystick... ");

		joystick[0].buffer = open ("/dev/js0", O_NONBLOCK);
		joystick[1].buffer = open ("/dev/js1", O_NONBLOCK);
		joystick[2].buffer = open ("/dev/js2", O_NONBLOCK);
		joystick[3].buffer = open ("/dev/js3", O_NONBLOCK);
		
		if (joystick[0].buffer >= 0 || joystick[1].buffer >= 0 || joystick[2].buffer >= 0 || joystick[3].buffer >= 0) {
			printf ("found: ");

			for (i = 0; i < 4; i++) {
				if (joystick[i].buffer >= 0) {
					ioctl (joystick[i].buffer, JSIOCGAXES, &joystick[i].num_axes);
					ioctl (joystick[i].buffer, JSIOCGBUTTONS, &joystick[i].num_buttons);
					ioctl (joystick[i].buffer, JSIOCGVERSION, &joystick[i].version);
					
					if (!joystick[i].version) {
						joystick[i].num_axes = 2;
						joystick[i].num_buttons = 2;
						printf ("js%d (v0.x)  " , i);
					} else {
						printf ("js%d (v%d.%d.%d)  ", i, (joystick[i].version & 0xff0000) >> 16, (joystick[i].version & 0xff00) >> 8, joystick[i].version & 0xff);
					}						

//					printf ("%d axes, %d buttons", joystick[i].num_axes, joystick[i].num_buttons);

					for (j = num_axes; j < num_axes + joystick[i].num_axes; j++) {
						axis[j].joydev = i;
						if (joystick[i].version) {
							axis[j].center = 0;
							axis[j].max = 32767;
							axis[j].min = -32767;
						}
					}
					for (j = num_buttons; j < num_buttons + joystick[i].num_buttons; j++) {
						button[j].joydev = i;
					}

					num_axes += joystick[i].num_axes;
					num_buttons += joystick[i].num_buttons;
					
				} else {
					joystick[i].num_buttons = 0;
					joystick[i].num_axes = 0;
				}	
			}
		} else {
			printf ("no joysticks found\n");
			return 0;
		}		

		if (num_axes > MAX_AXES)
			num_axes = MAX_AXES;
		if (num_buttons > MAX_BUTTONS)
			num_buttons = MAX_BUTTONS;

		printf ("\n");
//		printf ("%d axes, %d buttons\n", num_axes, num_buttons);

		joy_present = 1;
		joy_installed = 1;
		return 1;
	}

	return 1;
}


void joy_close() {
	int i;

	if (!joy_installed) return;

	for (i = 0; i < 4; i++) {
		if (joystick[i].buffer>=0) {
			printf ("closing js%d\n", i);
			close (joystick[i].buffer);
		}
		joystick[i].buffer=-1;
	}

	joy_present=0;
	joy_installed=0;
}


void joy_set_cen() {
}


int joy_get_scaled_reading(int raw, int axis_num) {
	int d, x;

	raw -= axis[axis_num].center;

	if (raw < 0) d = axis[axis_num].center - axis[axis_num].min;
	else if (raw > 0) d = axis[axis_num].max - axis[axis_num].center;
	else d = 0;

	if (d) x = ((raw << 7) / d);
	else x = 0;

	if ( x < -128 ) x = -128;
	if ( x > 127 ) x = 127;

	return x;
}


void joy_get_pos(int *x, int *y) {
	int axis[MAX_AXES];

	if ((!joy_installed)||(!joy_present)) { *x=*y=0; return; }

	joystick_read_raw_axis (JOY_ALL_AXIS, axis);

	*x = joy_get_scaled_reading( axis[0], 0 );
	*y = joy_get_scaled_reading( axis[1], 1 );
}


int joy_get_btns () {
	return 0;
}


int joy_get_button_state (int btn) {
        j_Update_state ();

        return button[btn].state;
}


int joy_get_button_down_cnt (int btn) {
	int downcount;

	j_Update_state ();

	downcount = button[btn].downcount;
	button[btn].downcount = 0;
	
	return downcount;
}


fix joy_get_button_down_time(int btn)  {
	j_Update_state ();

	return button[btn].timedown;
}


void joy_poll() {

}


void joy_set_slow_reading(int flag) {

}

#else //						** else !HAVE_JOYSTICK
#include "types.h"
#include "fix.h"

char joy_present=0;

void joy_get_cal_vals(int *axis_min, int *axis_center, int *axis_max) { }
void joy_set_cal_vals(int *axis_min, int *axis_center, int *axis_max) { }
ubyte joy_get_present_mask() { return 0; }
void joy_set_timer_rate(int max_value )	{ }
int joy_get_timer_rate() { return 0; }
void joy_flush() { }
void joy_set_slow_reading(int flag) { }
ubyte joystick_read_raw_axis(ubyte mask, int * axis) { return 0; }
int joy_init() { return 0; }
void joy_close() { }
void joy_set_cen() { }
int joy_get_scaled_reading(int raw, int axn) { return 0; }
void joy_get_pos(int *x, int *y) { }
int joy_get_btns() { return 0; }
int joy_get_button_state(int btn) { return 0; }
int joy_get_button_down_cnt(int btn) { return 0; }
fix joy_get_button_down_time(int btn) { return 0; }
void joy_poll() { }

int j_Get_axis_range_of_motion (int axisnum) { return 0; }
int j_Get_joydev_version (int joydev) { return 0; }
int j_Get_joydev_axis_number (int all_axis_number) { return 0; }
int j_Get_joydev_button_number (int all_button_number) { return 0; }
int j_Get_button_joydev (int button_num) { return 0; }
int j_Get_axis_joydev (int axis_number) { return 0; }
int j_Get_axis_number (int joydev_number, int joy_axis_number) { return 0; }
int j_Get_button_number (int joydev_number, int joy_button_number) { return 0; }
int j_Get_num_buttons () { return 0; }
int j_Get_num_axes () { return 0; }
int j_Update_state () { return 0; }
void joy_set_min (int axis_number, int value) {}
void joy_set_center (int axis_number, int value) {}
void joy_set_max (int axis_number, int value) {}

#endif
