// 
// FreeBSD Joystick driver, Stephen Hocking
//

#ifdef WANT_JOYSTICK
#include <stdlib.h>
#include <stdio.h>
#include <machine/joystick.h>
#include <fcntl.h>
#include <unistd.h>

#include "types.h"
#include "mono.h"
#include "joy.h"

#define JS_RETURN sizeof(struct joystick)
#define NUM_BUTTONS 28

char joy_installed = 0;
char joy_present = 0;

typedef struct Button_info {
  ubyte		ignore;
  ubyte		state;
  ubyte		last_state;
  int		timedown;
  ubyte		downcount;
  ubyte		upcount;
} Button_info;

typedef struct Joy_info {
  ubyte			present_mask;
  ubyte			slow_read;
  int			max_timer;
  int			read_count;
  ubyte			last_value;
  Button_info		buttons[NUM_BUTTONS];
  int			axis_min[4];
  int			axis_center[4];
  int			axis_max[4];
} Joy_info;

Joy_info joystick;

extern int joy_retries;

int joystick_fd0=-1,joystick_fd1=-1;

void joy_get_cal_vals(int *axis_min, int *axis_center, int *axis_max)
{
  int i;
  
  for (i=0; i<4; i++)		{
    axis_min[i] = joystick.axis_min[i];
    axis_center[i] = joystick.axis_center[i];
    axis_max[i] = joystick.axis_max[i];
  }
}

void joy_set_cal_vals(int *axis_min, int *axis_center, int *axis_max)
{
  int i;
  
  for (i=0; i<4; i++)		{
    joystick.axis_min[i] = axis_min[i];
    joystick.axis_center[i] = axis_center[i];
    joystick.axis_max[i] = axis_max[i];
  }
}


ubyte joy_get_present_mask()	{
  return joystick.present_mask;
}

void joy_set_timer_rate(int max_value )	{
  joystick.max_timer = max_value;
}

int joy_get_timer_rate()	{
  return joystick.max_timer;
}

void joy_flush()	{
  int i;
  
  if (!joy_installed) return;
  
  for (i=0; i<NUM_BUTTONS; i++ )	{
    joystick.buttons[i].ignore = 0;
    joystick.buttons[i].state = 0;	
    joystick.buttons[i].timedown = 0;	
    joystick.buttons[i].downcount = 0;	
    joystick.buttons[i].upcount = 0;	
  }
  
}

int joy_buttons;

ubyte joy_read_raw_buttons()	{
  return joy_buttons;
}

void joy_set_slow_reading(int flag)
{
  joystick.slow_read |= flag;
  joy_set_cen();
}

ubyte joystick_read_raw_axis(ubyte mask, int * axis) {
  struct joystick js;
  int status;

  if (!joy_installed||!joy_present) return 0;

  joy_buttons=0;
  if (joystick_fd0>=0) {
    if ((status=read(joystick_fd0,&js,JS_RETURN))!=JS_RETURN) {
      perror("joystick_read_new_axis");
      close(joystick_fd0);
      joystick_fd0=-1;
      return 0;
    }
    axis[0]=js.x; axis[1]=js.y;
    joy_buttons |= (js.b1 | (js.b2 << 1));
  } else {
    axis[0]=0; axis[1]=0;
  }
  
  axis[2] = axis[3] = 0;
  if (joystick_fd1>=0) {
    if ((status=read(joystick_fd1,&js,JS_RETURN))!=JS_RETURN) {
      perror("joystick_read_new_axis");
      close(joystick_fd1);
      joystick_fd1=-1;
      return 0;
    }
    axis[2]=js.x; axis[3]=js.y;
    joy_buttons|=((js.b1|js.b2)&3)<<2;
  } else {
    axis[2]=0; axis[3]=0;
  }
  
  {
    int i,value,state;
    Button_info *button;
    
    value=joy_buttons;
    // We do some fancy checks here because we don't have joy_handler
    // nor autotimers to handle such things
    for (i=0; i<NUM_BUTTONS;i++) {
      button=&joystick.buttons[i];
      if (button->ignore) continue;
      if (i<5) state=(value>>i)&1;
      else if (i==(value+4))
	state=1;
      else state = 0;
      if (button->last_state==state) {
	if (state) button->timedown+=10000;
      } else {
	if (state) {
	  button->downcount+=state;
	  button->state=1;
	} else {
	  button->upcount+=button->state;
	  button->state=0;
	}
	button->last_state=state;
      }
    }
  }
  return (axis[0]?1:0)|(axis[1]?2:0)|(axis[2]?4:0)|(axis[3]?8:0);
}

int joy_init() {
  int i;
  int temp_axis[4];
  
  if (joy_installed) return 0;
  joy_flush();
  
  for (i=0; i<NUM_BUTTONS; i++ )	
    joystick.buttons[i].last_state = 0;
  
  if ( !joy_installed )	{
    joystick_fd0=open("/dev/joy0",O_RDONLY);
    if (joystick_fd0<0) {
      perror("Joystick_fd0");
      return 0;
    }
    
    joystick_fd1=open("/dev/joy1",O_RDONLY);
    /* commented out because not everyone uses both js0 and js1
       if (joystick_fd1<0) {
       perror("Joystick_fd1");
       return 0;
       }
    */
    
    joy_present = 1;
    joy_installed = 1;
    // joystick.max_timer = 65536;
    joystick.slow_read = 0;
    joystick.read_count = 0;
    joystick.last_value = 0;
  }
  
  // Do initial cheapy calibration...
  joystick.present_mask = JOY_ALL_AXIS;		// Assume they're all present
  joystick.present_mask = joystick_read_raw_axis( JOY_ALL_AXIS, temp_axis );
  
  return 1;
}

void joy_close() {
  if (!joy_installed) return;
  
  if (joystick_fd0>=0) close(joystick_fd0);
  if (joystick_fd1>=0) close(joystick_fd1);
  joystick_fd0=-1; joystick_fd1=-1;
  joy_present=0; joy_installed=0;
}

void joy_set_ul() {
  joystick.present_mask = JOY_ALL_AXIS;		// Assume they're all present
  joystick.present_mask = joystick_read_raw_axis( JOY_ALL_AXIS, joystick.axis_min );
  if ( joystick.present_mask & 3 )
    joy_present = 1;
  else
    joy_present = 0;
}

void joy_set_lr() {
  joystick.present_mask = JOY_ALL_AXIS;		// Assume they're all present
  joystick.present_mask = joystick_read_raw_axis( JOY_ALL_AXIS, joystick.axis_max );
  
  if ( joystick.present_mask & 3 )
    joy_present = 1;
  else
    joy_present = 0;
}

void joy_set_cen() {
  joystick.present_mask = JOY_ALL_AXIS;		// Assume they're all present
  joystick.present_mask = joystick_read_raw_axis( JOY_ALL_AXIS, joystick.axis_center );
  
  if ( joystick.present_mask & 3 )
    joy_present = 1;
  else
    joy_present = 0;
}

void joy_set_cen_fake(int channel) {
  int i,n=0;
  int minx, maxx, cenx;
  
  minx=maxx=cenx=0;
  
  for (i=0; i<4; i++ )	{
    if ( (joystick.present_mask & (1<<i)) && (i!=channel) )	{
      n++;
      minx += joystick.axis_min[i];
      maxx += joystick.axis_max[i];
      cenx += joystick.axis_center[i];
    }
  }
  minx /= n;
  maxx /= n;
  cenx /= n;
  
  joystick.axis_min[channel] = minx;
  joystick.axis_max[channel] = maxx;
  joystick.axis_center[channel] = cenx;
}

int joy_get_scaled_reading(int raw, int axn) {
  int x, d;
  
  // Make sure it's calibrated properly.
  if ( joystick.axis_center[axn] - joystick.axis_min[axn] < 5 ) return 0;
  if ( joystick.axis_max[axn] - joystick.axis_center[axn] < 5 ) return 0;
  
  raw -= joystick.axis_center[axn];
  
  if ( raw < 0 )	{
    d = joystick.axis_center[axn]-joystick.axis_min[axn];
  } else {
    d = joystick.axis_max[axn]-joystick.axis_center[axn];
  }
  
  if ( d )
    x = (raw << 7) / d;
  else 
    x = 0;
  
  if ( x < -128 ) x = -128;
  if ( x > 127 ) x = 127;
  return x;
}

int last_reading[4] = { 0, 0, 0, 0 };

void joy_get_pos(int *x, int *y) {
  ubyte flags;
  int axis[4];
  
  if ((!joy_installed)||(!joy_present)) { *x=*y=0; return; }
  
  flags=joystick_read_raw_axis( JOY_1_X_AXIS+JOY_1_Y_AXIS, axis );
  
  last_reading[0] = axis[0];
  last_reading[1] = axis[1];
  
  if ( flags & JOY_1_X_AXIS )
    *x = joy_get_scaled_reading( axis[0], 0 );
  else
    *x = 0;
  
  if ( flags & JOY_1_Y_AXIS )
    *y = joy_get_scaled_reading( axis[1], 1 );
  else
    *y = 0;
}

ubyte joy_read_stick(ubyte masks, int *axis) {
  ubyte flags;
  int raw_axis[4];
  
  if ((!joy_installed)||(!joy_present)) { 
    axis[0] = 0; axis[1] = 0;
    axis[2] = 0; axis[3] = 0;
    return 0;  
  }
  
  flags=joystick_read_raw_axis( masks, raw_axis );
  
  last_reading[0] = axis[0];
  last_reading[1] = axis[1];
  last_reading[2] = axis[2];
  last_reading[3] = axis[3];
  
  if ( flags & JOY_1_X_AXIS )
    axis[0] = joy_get_scaled_reading( raw_axis[0], 0 );
  else
    axis[0] = 0;
  
  if ( flags & JOY_1_Y_AXIS )
    axis[1] = joy_get_scaled_reading( raw_axis[1], 1 );
  else
    axis[1] = 0;
  
  if ( flags & JOY_2_X_AXIS )
    axis[2] = joy_get_scaled_reading( raw_axis[2], 2 );
  else
    axis[2] = 0;
  
  if ( flags & JOY_2_Y_AXIS )
    axis[3] = joy_get_scaled_reading( raw_axis[3], 3 );
  else
    axis[3] = 0;
  
  return flags;
}


int joy_get_btns() {
  int moo[4];
  if ((!joy_installed)||(!joy_present)) return 0;
  
  joystick_read_raw_axis(0,moo);
  
  return joy_read_raw_buttons();
}

void joy_get_btn_down_cnt(int *btn0, int *btn1) {
  if ((!joy_installed)||(!joy_present)) { *btn0=*btn1=0; return; }
  
  *btn0=joystick.buttons[0].downcount;
  joystick.buttons[0].downcount=0;
  *btn1=joystick.buttons[1].downcount;
  joystick.buttons[1].downcount=0;
}

int joy_get_button_state(int btn) {
  int count;
  int moo[4];
  
  if ((!joy_installed)||(!joy_present)) return 0;
  
  if (btn>=NUM_BUTTONS) return 0;
  
  joystick_read_raw_axis(0,moo);
  
  count=joystick.buttons[btn].state;
  
  return count;
}

int joy_get_button_up_cnt(int btn) {
  int count;
  int moo[4];
  
  if ((!joy_installed)||(!joy_present)) return 0;
  
  if (btn>=NUM_BUTTONS) return 0;
  
  joystick_read_raw_axis(0,moo);
  
  count=joystick.buttons[btn].upcount;
  joystick.buttons[btn].upcount=0;
  
  return count;
}

int joy_get_button_down_cnt(int btn) {
  int count;
  int moo[4];
  
  if ((!joy_installed)||(!joy_present)) return 0;
  if (btn>=NUM_BUTTONS) return 0;
  
  joystick_read_raw_axis(0,moo);
  
  count=joystick.buttons[btn].downcount;
  joystick.buttons[btn].downcount=0;
  
  return count;
}

fix joy_get_button_down_time(int btn)  {
  fix count;
  int moo[4];
  
  if ((!joy_installed)||(!joy_present)) return 0;
  
  if (btn>=NUM_BUTTONS) return 0;
  
  joystick_read_raw_axis(0,moo);
  
  count=joystick.buttons[btn].timedown;
  joystick.buttons[btn].timedown=0;
  
  return fixmuldiv(count,65536,1193180);
}

void joy_get_btn_up_cnt(int *btn0, int *btn1) {
  if ((!joy_installed)||(!joy_present)) { *btn0=*btn1=0; return; }
  
  *btn0=joystick.buttons[0].upcount;
  joystick.buttons[0].upcount=0;
  *btn1=joystick.buttons[1].upcount;
  joystick.buttons[1].upcount=0;
}

void joy_set_btn_values(int btn,int state,fix timedown,int downcount, int upcount) {
  joystick.buttons[btn].ignore=1;
  joystick.buttons[btn].state=state;
  joystick.buttons[btn].timedown=fixmuldiv(timedown,1193180,65536);
  joystick.buttons[btn].downcount=downcount;
  joystick.buttons[btn].upcount=upcount;
}

void joy_poll() {
}

#else //						** else !HAVE_JOYSTICK
#include "types.h"
#include "fix.h"

char joy_present=0;

void joy_get_cal_vals(int *axis_min, int *axis_center, int *axis_max) { }
void joy_set_cal_vals(int *axis_min, int *axis_center, int *axis_max) { }
ubyte joy_get_present_mask() { return 0; }
void joy_set_timer_rate(int max_value )	{ }
int joy_get_timer_rate() { return 0; }
void joy_flush() { }
ubyte joy_read_raw_buttons() { return 0; }
void joy_set_slow_reading(int flag) { }
ubyte joystick_read_raw_axis(ubyte mask, int * axis) { return 0; }
int joy_init() { return 0; }
void joy_close() { }
void joy_set_ul() { }
void joy_set_lr() { }
void joy_set_cen() { }
void joy_set_cen_fake(int channel) { }
int joy_get_scaled_reading(int raw, int axn) { return 0; }
void joy_get_pos(int *x, int *y) { }
ubyte joy_read_stick(ubyte masks, int *axis) { return 0; }
int joy_get_btns() { return 0; }
void joy_get_btn_down_cnt(int *btn0, int *btn1) { }
int joy_get_button_state(int btn) { return 0; }
int joy_get_button_up_cnt(int btn) { return 0; }
int joy_get_button_down_cnt(int btn) { return 0; }
fix joy_get_button_down_time(int btn) { return 0; }
void joy_get_btn_up_cnt(int *btn0, int *btn1) { }
void joy_set_btn_values(int btn,int state,fix timedown,int downcount, int upcount) { }
void joy_poll() { }

int j_Get_axis_range_of_motion (int axisnum) { return 0; }
int j_Get_joydev_version (int joydev) { return 0; }
int j_Get_joydev_axis_number (int all_axis_number) { return 0; }
int j_Get_joydev_button_number (int all_button_number) { return 0; }
int j_Get_button_joydev (int button_num) { return 0; }
int j_Get_axis_joydev (int axis_number) { return 0; }
int j_Get_axis_number (int joydev_number, int joy_axis_number) { return 0; }
int j_Get_button_number (int joydev_number, int joy_button_number) { return 0; }
int j_Get_num_buttons () { return 0; }
int j_Get_num_axes () { return 0; }
int j_Update_state () { return 0; }
void joy_set_min (int axis_number, int value) {}
void joy_set_center (int axis_number, int value) {}
void joy_set_max (int axis_number, int value) {}

#endif
