
/* 
 * Dos/PC Emulator
 * Copyright (C) 1991 Jim Hudgens
 * 
 * 
 * The file is part of GDE.
 * 
 * GDE is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 1, or (at your option)
 * any later version.
 * 
 * GDE is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GDE; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.  
 *
 */

#include "gde.h"

/*
  system file table, similar in function to the 
  MS/PCDOS  system file table.  Algorithms manipulating 
  this table are not necessarily clear and certainly undocumented
  in DOS.  All files which are opened, modify or add an entry
  to this table.  
  
  */


struct doss_file_handle   doss_sys_file[DOSS_MAX_SYS_FH];


int syscall_errno;

/*
  
  drive table, 
  
  This structure, which defines a disk or disk like device, 
  is a static initialization of the various fields
  of the drive table.
  
  */

u_int16  sys_base;

struct doss_drive         doss_sys_drives[DOSS_MAX_DRIVES];

/* system configuration parser */

/* note, this is a quick hack, and needs to be either made into a 
   lex/yacc form --- though that seems like overkill --- or at least
   allow  the input file to be more freeform.
   
   */

#define SAMESTRING(x,y)  (strncmp(x,y,strlen(y)) == 0)

static char *syscnf_keywords[] = 
      {
#define KWD_DEBUG 0
	"DEBUG",
#define KWD_MEMSIZE 1
	"MEMSIZE",
#define KWD_FILESYSTEM 2
	"FILESYSTEM",
#define KWD_ENVIRONMENT 3
	"ENVIRONMENT",
#define KWD_CDEVPARM 4
	"CDEVPARM",
#define KWD_DOSVERSION 5
	"DOSVERSION",
#define KWD_MEMBASE  6
	"MEMBASE",
#define KWD_CODEPAGE 7
	"CODEPAGE",
#define KWD_KEYBOARD 8
	"KEYBOARD",
	NULL,
      };

static char * EXFUN(extract_quoted_string,(char *s1,char *s2));
static int EXFUN(syscnf_kwdlookup,(char *s));
static void EXFUN(syscnf_error,(char *s));


void DEFUN(sys_config,(file,m), char *file AND PC_ENV *m)
{

  FILE *f;
  char line[1024];
  char tmppath[1024];
  char *p;
  int drive;
  enum fs_type dtype;
  int len;
  int i;


#ifdef DEBUG  
  m->debug=0;
#endif

#ifdef DOSS_SUPPORT
  m->mem_size = 1024  * 1024;
  m->cpage_active = m->cpage_system = 367;	/* ASCII */
  m->kbd_ops = &kbd_ops_termios;
  sys_base = 0x60;
#endif

  f = fopen(file,"r");
  
  if (f == NULL)
      {
	fprintf(stderr,"sys_config: error opening %s\n",file);
	return;
      }
  
  while (fgets(line,sizeof(line),f) != NULL )
      {
	if (line[0] == '#')
	  continue;
	
	switch(syscnf_kwdlookup(line))
	    {
	    case KWD_DEBUG:  
#ifdef DEBUG
	      /* one line, only. */
	      m->debug = 0;
	      p = index(line,'=');
	      if (p == NULL) 
		  {
		    syscnf_error("invalid DEBUG line\n");
		    continue;
		  }
	      while(p != NULL)
		  {
		    p++;   /* skip over the current character. 
			      either a "=" or a "," */
		    if (SAMESTRING(p,"DBGSVC"))
		      m->debug |= DEBUG_SVC_F;
		    else if (SAMESTRING(p,"DBGDECODE"))
		      m->debug |= DEBUG_DECODE_F;
		    else if (SAMESTRING(p,"DBGSTEP"))
		      m->debug |= DEBUG_STEP_F;
		    else if (SAMESTRING(p,"DBGFS"))
		      m->debug |= DEBUG_FS_F;
		    else if (SAMESTRING(p,"DBGPROC"))
		      m->debug |= DEBUG_PROC_F;
		    else if (SAMESTRING(p,"DBGSYSINT"))
		      m->debug |= DEBUG_SYSINT_F;
		    else if (SAMESTRING(p,"DBGTRACECALL"))
		      m->debug |= DEBUG_TRACECALL_F;
		    else if (SAMESTRING(p,"DBGTRACE"))
		      m->debug |= DEBUG_TRACE_F;
		    else if (SAMESTRING(p,"DBGDISASS"))
		      m->debug |= DEBUG_DISASSEMBLE_F;
		    else if (SAMESTRING(p,"DBGKEYB"))
		      m->debug |= DEBUG_KEYBOARD_F;
		    p = index (p,',');
		  }
#endif
	      break;

	    case KWD_KEYBOARD:
	      p = index(line,'=');
	      if (p == NULL) 
		  {
		    syscnf_error("invalid KEYBOARD line\n");
		    continue;
		  }
	      p++;
	      if (SAMESTRING(p,"STDIO"))
		      m->kbd_ops = &kbd_ops_stdio;
	      else if (SAMESTRING(p,"TERMIOS"))
		      m->kbd_ops = &kbd_ops_termios;
	      break;	


            case KWD_CODEPAGE:
	      p = index(line,'=');
	      if (p == NULL) 
		  {
		    syscnf_error("invalid CODEPAGE line\n");
		    continue;
		  }
	      m->cpage_active = m->cpage_system = atoi(p+1);
	      break;

	      
	    case KWD_MEMSIZE:
	
	      /* lines of the form:
		 
		 MEMSIZE=1024

		 Implicit is the scaling by 1K.
	       */

	      p = index(line,'=');
	      if (p == NULL) 
		  {
		    syscnf_error("invalid MEMSIZE line\n");
		    continue;
		  }
	      m->mem_size = atoi(p+1) * 1024;
	      if (m->mem_size < 64*1024 )
		m->mem_size = 64*1024;
	      break;

      
	    case KWD_FILESYSTEM:
	      /* lines of the form:
		 FILESYSTEM=A,UFS,"~/dos"
		 FILESYSTEM=B,FFS,"~/C:",
		 FILESYSTEM=C,DFS,"/dev/rfd0a",
	       */
		 
#ifdef DOSS_SUPPORT 		 
	      p = index(line,'=');
	      if (p == NULL) 
		  {
		    syscnf_error("invalid FILESYSTEM line\n");
		    continue;
		  }
	      p++;     /* SKIP the "=" */
	      
	      if (isupper(*p)) *p = tolower(*p);
	      
	      drive = *p - 'a';
	      
	      if ( drive < 0 || drive >= DOSS_MAX_DRIVES ||
		  (doss_sys_drives[drive].flags & DOSS_VALID_DRIVE) != 0)
		  {
		    syscnf_error("invalid FILESYSTEM line, bogus drive\n");
		    continue;
		  }
	      
	      p = index(p,',');
	      if (p == NULL) 
		  {
		    syscnf_error("invalid FILESYSTEM line\n");
		    continue;
		  }
	      p++;
	      
	      if (SAMESTRING(p,"UFS"))
		{
		  dtype=doss_sys_drives[drive].fs_type = unix_fs;
		  doss_sys_drives[drive].fs_ops = &ufs;
		}		    
	      else if (SAMESTRING(p,"FFS"))
		{
		  dtype=doss_sys_drives[drive].fs_type = ufile_fs;
		  doss_sys_drives[drive].fs_ops = &gde_ffs;
		}		    
	      else if (SAMESTRING(p,"DFS"))
		dtype = udevice_fs;
	      else
		  {
		    syscnf_error("invalid FS type\n");
		    continue;
		  }
	      
      
	      p = index(p,',');
	      if (p == NULL) 
		  {
		    syscnf_error("invalid FILESYSTEM line\n");
		    continue;
		  }
	      p++;
	      
	      switch(dtype)
		  {
		   case unix_fs:
		   case ufile_fs:
 		   
		    /* expecting a string delimiter */
		    
		    if (*p != '"')
			{
			  syscnf_error("invalid FILESYSTEM line\n");
			  continue;
			}
		    
		    /* grab string for drive->path */
		    p = extract_quoted_string(p,tmppath);
		    
		    /* expand ~ to $HOME */
		    /* save the string. */
		    
		    if (tmppath[0] == '~') 
			{
			  if (getenv("HOME") == NULL)
			      {
				syscnf_error("invalid UFS expansion of ~\n");
				continue;
			      }
			  else
			      {
				
				len = strlen(getenv("HOME"))+
				  strlen(tmppath) + 
				    10;
				
				doss_sys_drives[drive].path =  malloc(len);
				
				if (doss_sys_drives[drive].path == NULL) 
				    {
				      sys_fatal(errno,
						"sys_config: malloc failed\n");
				    }
				
				sprintf(doss_sys_drives[drive].path,"%s%s",
					getenv("HOME"),
					tmppath+1);
			      }
			}
		    else
			{
			  len = strlen(tmppath) +  10;
			  doss_sys_drives[drive].path =  malloc(len);
			  
			  if (doss_sys_drives[drive].path == NULL) 
			      {
				sys_fatal(errno,
					  "sys_config: malloc failed\n");
			      }
			  
			  strcpy(doss_sys_drives[drive].path,
				 tmppath);	
			}
		    
		    break; /* UFS type */
		  case udevice_fs:
		    syscnf_error("unsupported FS type\n");
		    continue;
		  default:
		    syscnf_error("unknown FS type\n");
		    continue;
		  }   /* switch dtype */

	      /* set it to be valid if we reach this point */
	      doss_sys_drives[drive].flags |= DOSS_VALID_DRIVE;
	      doss_sys_drives[drive].unit = drive;
	      
	      break;  /* KWD_FILESYSTEM */
#endif

	    case KWD_ENVIRONMENT:
	      /* slightly different here.  One environment string
		 per line, and the continuation character is a ','
	       
		 ENVIRONMENT="PATH=....",
	                     "TMP=C:\",
			     "FOO=bar"
               */
#ifdef DOSS_SUPPORT
	      /* YYY temporary hardcoded limit of 100 strings. */
	      m->doss_envp = (char **) malloc(sizeof(char *) * 100);
	      if (m->doss_envp == NULL)
		sys_fatal(errno,
			  "sys_config: malloc failed");	
      
	      for (i=0; i<100; i++) 
		m->doss_envp[i] = NULL;

	      p = index(line,'=');
	      if (p == NULL) 
		  {
		    syscnf_error("invalid MEMSIZE line\n");
		    continue;
		  }
	      p++;

	      i = 0;
	      do     /* at least once */
		  {
		    if (*p != '"')
			{
			  syscnf_error("invalid ENVIRONMENT line\n");
			  break;   /* exit from loop */
			}

		    p = extract_quoted_string(p,tmppath);
		    m->doss_envp[i] = malloc(strlen(tmppath)+1);
		    if (m->doss_envp[i] == NULL)
		      sys_fatal(errno,
				"sys_config: malloc failed");
		    strcpy(m->doss_envp[i],tmppath);
		    i++;
		    p = index(p,',');
		    if (p == NULL)
		      break;      /* exit from loop */
		    if (fgets(line,sizeof(line),f)== NULL) 
		      break;  /* exit from loop */
		    p = line;
		    
		    while(*p != '"') p++;
		  }
	      while (1);
#endif	      
	      break;
	    case KWD_MEMBASE:
	      /* lines of the form:
		 
		 MEMBASE=60

		 Implicit is the scaling by "paragraphs".
	       */

	      p = index(line,'=');
	      if (p == NULL) 
		  {
		    syscnf_error("invalid MEMSIZE line\n");
		    continue;
		  }
	      sys_base = atoi(p+1);
	      break;

	      
	    }  /* switch */
      }       /* while */
}


void DEFUN(sys_config_def,(m),  PC_ENV *m)
{

  char tmppath[1024];
  int drive;
  int i;


#ifdef DEBUG
  m->debug = 0;
#endif

  m->mem_size = 1024 * 1024;
  sys_base = 0x60;
	      
/* FILESYSTEMS */

#ifdef DOSS_SUPPORT 		 

  drive = 0;
  doss_sys_drives[drive].fs_type = unix_fs;
  doss_sys_drives[drive].fs_ops = &ufs;
  doss_sys_drives[drive].path =  ".";
  doss_sys_drives[drive].flags |= DOSS_VALID_DRIVE;
  doss_sys_drives[drive].unit = drive;

  if (getenv("HOME") != NULL)
      {
	drive = 1;
	doss_sys_drives[drive].fs_type = unix_fs;
	doss_sys_drives[drive].fs_ops = &ufs;
	doss_sys_drives[drive].flags |= DOSS_VALID_DRIVE;
	doss_sys_drives[drive].unit = drive;
	sprintf(tmppath,"%s/%s",getenv("HOME"),"dos");
	doss_sys_drives[drive].path =  malloc(strlen(tmppath)+1);	
	strcpy(doss_sys_drives[drive].path,tmppath);
      }
#endif

/* ENVIRONMENT */
#ifdef DOSS_SUPPORT
  m->doss_envp = (char **) malloc(sizeof(char *) * 10);
  if (m->doss_envp == NULL)
    sys_fatal(errno,
	      "sys_config: malloc failed");	
  for (i=0; i<10; i++) 
    m->doss_envp[i] = NULL;
  m->doss_envp[0] = "PATH=a:\\;b:\\";
#endif	      
}




static char *DEFUN(extract_quoted_string,(s1,s2),char *s1 AND char *s2)
{

  char *p,*q;
    
  p = s1;
  q = s2;
  
  if (*p != '"')
      {
	*s2 = '\0';
	return s2;
      }
  
  p++;
  
  while (*p != '"' && *p != '\0')
      {
	*q = *p;
	p++;
	q++;
      }

  *q = '\0';

  if (*p == '\0')
    syscnf_error("unterminated string in config file\n");

  return p+1;
}


static void DEFUN(syscnf_error,(s),char *s)
{
  sys_warn("sys_config: %s",s);
}

  

static int DEFUN(syscnf_kwdlookup,(s),char *s)
{
  int i;
  for (i=0;syscnf_keywords[i]; i++)
      {
	if(SAMESTRING(s,syscnf_keywords[i]))
	  return i;
      }

  return -1;

}

/* below used for testing purposes */

#ifdef TEST

main()
{
  
  PC_ENV m;
  int i;
  
  
  sys_config("configure.gde",&m);
  
  
  printf(" memory size set to: %dK\n",m.mem_size/1024);
  printf(" debug options set to: %x\n",m.debug);
  printf("environment:\n");
  
  for(i=0; m.doss_envp[i]; i++)
    printf("\"%s\"\n",m.doss_envp[i]);
  
  printf("\n\nfilesystems:\n");
  
  for (i=0; i<DOSS_MAX_DRIVES; i++)
      {
	if (doss_sys_drives[i].flags & DOSS_VALID_DRIVE)
	    {
	      printf("unit=%d type=%d path=%s\n",
		     doss_sys_drives[i].unit,
		     doss_sys_drives[i].fs_type,
		     doss_sys_drives[i].path);
	    }
      }
}

void	sys_fatal(va_alist)
     va_dcl
{
  char *fmt;
  int  error;
  va_list	p;
	
  va_start(p);
  error = va_arg(p,int);
  fmt = va_arg(p,char *);
  fprintf(stderr, "Fatal error: ");
  if (error != 0)
      {
	fprintf(stderr, "<%d>",error);
	fprintf(stderr,"%s",STRERROR(error));
      }

  vfprintf(stderr, fmt, p);
  va_end(p);
  fprintf(stderr, "\nExiting...\n");
  exit(1);
}

void	sys_warn(va_alist)
     va_dcl
{
  char *fmt;
  va_list	p;
	
  va_start(p);
  fmt = va_arg(p,char *);
  fprintf(stderr, "Warning: ");
  vfprintf(stderr, fmt, p);
  va_end(p);
}

struct doss_fs ufs;

#endif

  

