
/* 
 * Dos/PC Emulator
 * Copyright (C) 1991 Jim Hudgens
 * 
 * 
 * The file is part of GDE.
 * 
 * GDE is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 1, or (at your option)
 * any later version.
 * 
 * GDE is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GDE; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.  
 *
 */

#ifdef DOSS_SUPPORT
static char rcsid[]=
  "$Id: doss_ufs.c,v 1.3 1991/07/30 01:58:52 hudgens Exp hudgens $";

/* $Log: doss_ufs.c,v $
 * Revision 1.3  1991/07/30  01:58:52  hudgens
 * added copyright.
 *
 * Revision 1.2  1991/07/20  15:25:42  hudgens
 * modified the dos to unix mapping of attribute bits.
 *
 * Revision 1.1  1991/07/20  04:08:35  hudgens
 * Initial revision
 *
 *
 */

/* DOSS_UFS.C

   This module implements a mapping of the doss filehandle
   operations into the unix filesystem.  The mapping is predicated
   on a doss drive being equated to a unix directory.  
   
   These operations assume that the correct drive has already
   been determined, and all operations here are called via
   a switch table, based on the operations contained in the
   drive's fs_ops field.  That is, given a struct drive *d,
   and a request for an open_file, the operation ufs_open_file
   would be invoked as:
   
        (*d->fs_ops->open_file)(...);

   In this way, other mappings of doss filesystems 
   (implemented as either a single unix file, or as a unix
   device) can be added without much trouble.  

 */

#include "gde.h"
#include <dirent.h>
#include <time.h>


static void DEFUN(ufs_error,(m,s),PC_ENV *m AND char *s)
 {
    debug_printf(m, "ufs error: %s\n", s);
 }


static int DEFUN(is_legal_filename,(f),char *f)
 {
    char *p;
    char t[15];
    
    
    if (strlen(f) > 12)   /*  8 + '.' + 3 */
      return 0;


    /* probably should look for illegal characters here. */

    strcpy(t,f);

    p = index (t, '.');
    
    if (p == NULL && strlen (t) > 8)
      return 0;
    
    else if (p != NULL)
	{
	   *p = '\0';      /* eliminate the dot. */
	   p += 1;
	   if (index(p,'.'))
	     return 0;     /* more than one dot in the name. */
	                   /* this handles dot-dot. */

	   if (strlen(t) > 8 || strlen(p) > 3) /* check sizes */
	     return 0;
	     
	}
    else   /* p == NULL */
	{
	   if (strlen(t) > 8) return 0;
	}
	   
    return 1;
 }
    
    

/* convert '\' to '/'.  Remove Drive designations. */
static char *DEFUN(parse_path,(s),char *s)
 {
    char *p,*q;
    
    p = index (s,':');
    if (p == NULL) p = s;   /* no drive specification in the path */
    else  p+=1;             /* remove the drive and colon from the path */
    
    q = p;
    
    /* now change all '\\'s into '/'s. */
    while (*q != '\0')
	{
	   if (*q == '\\') *q ='/';
	   q++;
	}
    return p;
 }



/* 
  
  conversion routines. Takes one from the unix view of things 
  to the dos view of things, as far as file permissions, times, etc.
  go.  

 */

static int DEFUN(ufs_mode_to_attrib,(mode),int mode)
 {
    int res=0;
    
    /* the dir bit. */
/*    if (S_ISDIR(mode))  */

    if ((mode & S_IFDIR) == S_IFDIR)
      res |= DOSS_FS_ATTR_DIR;

    /* the RO bit. */
    if ((mode & S_IWUSR)==0)
      res |= DOSS_FS_ATTR_RO;
    
    /* SYS BIT */
    if (mode & (S_IXGRP))
      res |= DOSS_FS_ATTR_SYS;
    
    /* ARC bit */
    if (mode & (S_IROTH))
      res |= DOSS_FS_ATTR_ARC;
    
    /* HIDDEN BIT */
    if ((mode & (S_IRGRP)) == 0)
      res |= DOSS_FS_ATTR_HID;
    
    /* VOLUME BIT */

    if ((mode &  S_IFLNK) == S_IFLNK)
      {
/*	 printf("is a link.\n");*/
	 res |= DOSS_FS_ATTR_VOL;
      }
    
	     
    return res;
 }


/* 
  
  conversion routines. Takes one from the unix view of things 
  to the dos view of things, as far as file permissions, times, etc.
  go.  

 */

static int 
  DEFUN(attrib_to_ufs_mode,(mode),int mode)
 {
    int res;

    res = 0440;   /* all starts out as readable/writable */
    

    /* the dir bit. */
    if (mode & DOSS_FS_ATTR_DIR)
      res |= S_IFDIR;

    /* the RO bit. */

    if ((mode & DOSS_FS_ATTR_RO)==0)
      res |= S_IWUSR;  /* clear the write bits */
    
    /* SYS BIT */
    if (mode & DOSS_FS_ATTR_SYS)
      res |= S_IXGRP;

    /* ARC bit */
    if (mode & DOSS_FS_ATTR_ARC)
      res |= S_IROTH;
    
    /* HIDDEN BIT */
    if (mode & DOSS_FS_ATTR_HID)
      res &= ~S_IRGRP;

    /* VOLUME BIT */
    if (mode & DOSS_FS_ATTR_VOL)
      res |= S_IFLNK;

    return res;
 }

static int
  DEFUN(epoch_to_doss_date,(t),time_t t)
 {
    struct tm *p;
    int res;
   
    p = localtime (&t);

    res = p->tm_mday;  /* bits 0..4 */
    res |= ((p->tm_mon+1) << 5);  /* 5..8 */ 
    res |= ((p->tm_year-80) << 9);  /* 9..f */
   return res;
 }


static int 
  DEFUN(epoch_to_doss_time,(t),time_t t)
 {
    struct tm *p;
    int res;
   
    p = localtime (&t);

    res = p->tm_sec/2;  /* bits 0..4 */
    res |= ((p->tm_min) << 5);  /* 5..9 */
    res |= ((p->tm_hour) << 11);  /* a..f */
    return res;
 }
    

static int 
  DEFUN(doss_datetime_to_epoch,(dd,dt),
	int dd AND int dt)
 {
    struct tm tm,*ttm;
    time_t res;
    
    res = time(0);
    ttm = localtime(&res);

    
    tm.tm_sec = 2 * (dt & 0x1f);
    tm.tm_min = ((dt>>5)&0x3f);
    tm.tm_hour = ((dt>>11)&0x1f);
    
/*    printf("dt=%x = %0d:%0d:%0d\n",
	   dt,
	   tm.tm_hour,
	   tm.tm_min,
	   tm.tm_sec);*/
    
    tm.tm_mday = (dd & 0x1f);
    tm.tm_mon = ((dd >> 5) & 0xf) - 1;
    tm.tm_year = ((dd >> 9) & 0x7f) + 80;

/*    printf("dd=%x %d/%d/%d\n",dd,
	   tm.tm_mon,
	   tm.tm_mday,
	   tm.tm_year); */

    res = 0;
  /*   res = localtime(&tm);	!!!! This needs work!*/

/*    printf("%s",ctime(&res));*/

    return res;
 }


/*
  ufs_initialize. 

  Set up all the parameters to this filesystem.  Called at 
  system initialization time, this essentially mapps 
  a unix directory into a dos drive.

  not much to do in this case.  Lets just make sure that the 
  "mount" point exists and is a directory.  If that is true, 
  then return success.  If the path to the directory exists 
  and it is, in fact, a directory, then we are OK.  No other 
  initializations need be done.

 */


static int 
  DEFUN(ufs_initialize,
	(m,dr),
	PC_ENV *m AND struct doss_drive *dr)
 {
    int r,err = 0,retv;
    struct stat sb;
    char dr_path[PATH_MAX];
    char cwd_path[PATH_MAX];   
    char cwd[PATH_MAX];

    getcwd(cwd, PATH_MAX);
    realpath(cwd,      cwd_path);
    realpath(dr->path, dr_path);
 
    r = lstat(dr->path,&sb);   
    
    if (r == 0 && ((sb.st_mode&S_IFMT)==S_IFDIR))
      retv = FSOP_SUCCESS;
    else
	{
	   err = errno;
	   retv = FSOP_FAILURE;
	}

    sprintf(dr->curr_path,"\\");
    /* If we're starting off somewhere below the fs root, initialise cwd */
    if (!strncmp(dr_path, cwd_path, strlen(dr_path)))
    {
	char *p;
	strncpy(dr->curr_path, cwd_path + strlen(dr_path), DOSS_MAX_PATH);
	/* Change / to \ */
	while ((p = strchr(dr->curr_path, '/')) != NULL)
	{
		*p = '\\';
	}
    }
 
#ifdef DEBUG
	if (DEBUG_FS(m))
	{
		debug_printf(m, "UFSD %d init\n",dr->unit);
		debug_printf(m, "Mount point is: %s ..", dr->path);
	   if  (retv == FSOP_SUCCESS) 
			debug_printf(m, "success\n");
	   else
	       {
			debug_printf(m, "failed\n");
		  if (r == 0)
		     {
			errno = err;
			perror("UFS_MOUNT");
		     }
	       }
	}
#endif
    return retv;
 }



static int 
  DEFUN(ufs_create_file,
	(m,dr,fh),
	PC_ENV *m AND
	struct doss_drive *dr AND
	struct doss_file_handle *fh)
 {
    char newpath[DOSS_MAX_PATH*2];
    int mode,flags;
    extern int errno;
    int err;
    char pp[DOSS_MAX_PATH];
    
   
    strcpy(pp,fh->path);
    
    sprintf(newpath,"%s/%s", dr->path,parse_path(pp));
    
    flags =  O_CREAT | O_TRUNC | O_RDWR;
    if (fh -> attrib & 0x1) 
	{
	   mode = 0440;
	}
    else
	{
	   mode = 0640;
	}

    /* How does msdos handle this??? I dunno. Punting. */
    fh->access = 2;
	       

    /* no clean and easy way to handle system, hidden, and archive
       bits.  So discard them and hope noone notices.
     */

    fh -> fd = open(newpath, flags, mode);
    err = errno;
    fh->curr_pos = 0;
   
#ifdef DEBUG
    if (DEBUG_FS(m))
	{
	   debug_printf(m, "UFSD %d: CREATE FILE %s", dr->unit, newpath);
	   if  (fh->fd > 0) 
	     debug_printf(m, " OK\n");
	   else
	       {
		  debug_printf(m, " FAILED\n");
		  errno = err;
		  perror("CREATE");
	       }
	}
#endif
    
    if (fh->fd >= 0) 
      return FSOP_SUCCESS;
    else
      return FSOP_FAILURE;
    
 }
    

    
/****************************************************************/


static int
  DEFUN(ufs_open_file,
	(m,dr,fh),
	PC_ENV *m AND
	struct doss_drive *dr AND
	struct doss_file_handle *fh)
 {
    char newpath[DOSS_MAX_PATH*2];
    int flags;
    extern int errno;
    int err;
    char pp[DOSS_MAX_PATH];
    
   
    strcpy(pp,fh->path);
    sprintf(newpath,"%s/%s", dr->path,parse_path(pp));
    
    flags =  0;
    
    switch(fh ->access & 0x3) 
	{
	   
	 case 0:
	   flags |= O_RDONLY;
	   break;
	 case 1:
	   flags |= O_WRONLY;
	   break;
	 case 2:
	   flags |= O_RDWR;
	   break;
	   
	}
        
    fh -> fd = open(newpath, flags);
    err = errno;
    fh->curr_pos = 0;
   
#ifdef DEBUG
    if (DEBUG_FS(m))
	{
	   debug_printf(m, "UFSD %d: OPEN %s",dr->unit, newpath);
	   if  (fh->fd > 0) 
	     debug_printf(m, " OK\n");
	   else
	       {
		  debug_printf(m, " FAILED\n");
		  errno = err;
		  perror("OPEN");
	       }
	}
#endif

    
    
    if (fh->fd >= 0) 
      return FSOP_SUCCESS;
    else
	{
	   switch(errno)
	       {
		case EACCES:
		case EDQUOT:
		case EEXIST:
		  syscall_errno = 5;  /* access denied */
		  break;
		case ENOENT:          
		  syscall_errno = 2;  /* file not found */
		  break;
		 }
	   
	   return FSOP_FAILURE;
	}
    
    
 }
   


static int
  DEFUN(ufs_close_file,
	(m,dr,fh),
	PC_ENV *m AND
	struct doss_drive *dr AND
	struct doss_file_handle *fh)
{
   
    extern int errno;
    int rerr,err;
    char pp[DOSS_MAX_PATH];
    
    rerr=close(fh->fd);
    err = errno;
    
    fh->fd = -1;
    
#ifdef DEBUG
    if (DEBUG_FS(m))
	{
	   strcpy(pp,fh->path);
	   
	   debug_printf(m, "UFSD %d: CLOSE %s",
		  dr->unit,
		  parse_path(pp));

	   if  (rerr >= 0) 
	     debug_printf(m, " OK\n");
	   else
	       {
		  debug_printf(m, " FAILED\n");
		  errno = err;
		  perror("CLOSE");
	       }
	}
#endif
    
    if (rerr >= 0) 
      return FSOP_SUCCESS;
    else
      return FSOP_FAILURE;

}




static int 
  DEFUN(ufs_seek_file,
	(m,dr,fh,ocode,offset,fp),
	PC_ENV *m AND
	struct doss_drive *dr AND
	struct doss_file_handle *fh AND
	int32 ocode AND 
	int32 offset AND
	int32* fp)
{
    extern int errno;
    int rerr,err = 0;
    int32 at,end;
    char pp[DOSS_MAX_PATH];

    /* save current location */
    /* seek to end of file */
    at = lseek(fh->fd, 0L, 1);

    /* find end of file */
    end = lseek(fh->fd, 0L, 2);

    if ((ocode == 0 && (/*offset > end ||*/ offset < 0)) ||
	(ocode == 1 && (/*offset + at > end ||*/ offset + at < 0)) || 
	(ocode == 2 && (/*offset > 0 ||*/ end + offset < 0)) ) 
	{
	   syscall_errno = DOSS_EINVALFUNC;  /* illegal offset */
	   end = lseek(fh->fd, at, 0);
	   *fp = at;
	   goto quick_exit;
	}

    /* return to original position for those cases where the
       process does a seek(-,0,1) to determine the current position */
    at = lseek(fh->fd, at, 0);
  
    rerr = lseek(fh->fd, offset, ocode);

    if (rerr>0) 
      fh->curr_pos = rerr;
    
#ifdef DEBUG
    if (DEBUG_FS(m))
	{
	   strcpy(pp,fh->path);
	   debug_printf(m, "UFSD %d: SEEK %s\n",
		  dr->unit,
		  parse_path(pp));
	   if  (rerr >= 0) 
	       {
		  debug_printf(m, "\tOK cpos=%d\n", rerr);
	       }
	   else
	       {
		  debug_printf(m, " FAILED\n"); 
		  errno = err; 
		  perror("LSEEK");
	       }
	}
#endif
    
    *fp = rerr;    /* return the original */
    if (rerr >= 0) 
      return FSOP_SUCCESS;
    else
      return FSOP_FAILURE;

quick_exit:
    return FSOP_FAILURE;
    
   
}


static int
  DEFUN(ufs_read_file,
	(m,dr,fh,buf,n,nread),
	PC_ENV *m AND
	struct doss_drive *dr AND
	struct doss_file_handle *fh AND
	char *buf AND 
	int n AND
	int *nread)
{
   
    extern int errno;
    int rerr,err;
    char pp[DOSS_MAX_PATH];
    
    
    /* check access for permission to do operation */
    
    switch(fh->access&0x3)
	{
	 case 0:
	 case 2:
	   break;
	 case 1:
	   /* XXX errno */
	   goto quick_exit;
	}
    
	   
    rerr=read(fh->fd,buf,n);
    err = errno;

    if (rerr>0) 
      fh->curr_pos += rerr;
    
    
#ifdef DEBUG
    if (DEBUG_FS(m))
	{
	   strcpy(pp,fh->path);
	   
	   debug_printf(m, "UFSD %d: READ %s",
		  dr->unit,
		  parse_path(pp));

	   if  (rerr >= 0) 
	       {
		  debug_printf(m, " OK: %d\n", rerr);
	       }
	   else
	       {
		  debug_printf(m, " FAILED\n"); 
		  errno = err; 
		  perror("READ");
	       }
	}
#endif
    
    *nread = rerr;    
    if (rerr >= 0) 
      return FSOP_SUCCESS;
    else
      return FSOP_FAILURE;

quick_exit:
    return FSOP_FAILURE;
    
}



       
static int 
  DEFUN(ufs_write_file,
	(m,dr,fh,buf,n,nwrote),
	PC_ENV *m AND
	struct doss_drive *dr AND
	struct doss_file_handle *fh AND
	char *buf AND
	int n AND
	int *nwrote)
{
   
    extern int errno;
    int rerr,err;
    char pp[DOSS_MAX_PATH];
    
   
    /* check access for permission to do operation */
    
    switch(fh->access&0x3)
	{
	 case 2:
	 case 3:
	   break;
	 case 0:
	   /* XXX errno */
	   goto quick_exit;
	}
    
	   
    rerr=write(fh->fd,buf,n);
    err = errno;

    if (rerr > 0 ) fh->curr_pos += rerr;
    
    
#ifdef DEBUG
    if (DEBUG_FS(m))
	{
	   strcpy(pp,fh->path);
	   debug_printf(m, "UFSD %d: WRITE %s",
		  dr->unit,
		  parse_path(pp));

	   if  (rerr >= 0) 
	       {
		  debug_printf(m, " OK: %d\n", rerr);
	       }
	   else
	       {
		  debug_printf(m, " FAILED\n"); 
		  errno = err;
		  perror("WRITE");
	       }
	}
#endif
    
    *nwrote = rerr;    
    if (rerr >= 0) 
      return FSOP_SUCCESS;
    else
      return FSOP_FAILURE;

quick_exit:
    return FSOP_FAILURE;
    
}

static int
  DEFUN(ufs_rename_file,(m),PC_ENV *m)
{ return FSOP_FAILURE; }


static int
  DEFUN(ufs_delete_file,
	(m,dr,s),
	PC_ENV *m AND
	struct doss_drive *dr AND
	char *s)
{
   
    extern int errno;
    int rerr,err;
    char name[DOSS_MAX_PATH*2];
    char pp[DOSS_MAX_PATH];
    
    strcpy(pp,s);
    sprintf(name,"%s/%s",
	    dr->path,
	    parse_path(pp));

    rerr=unlink(name);
    err = errno;
        
#ifdef DEBUG
    if (DEBUG_FS(m))
	{
	   debug_printf(m, "UFSD %d: DELETE %s",
		  dr->unit,
		  s);

	   if  (rerr >= 0) 
	     debug_printf(m, " OK\n");
	   else
	       {
		  debug_printf(m, " FAILED\n");
		  errno = err;
		  perror("DELETE");
	       }
	}
#endif
    
    if (rerr >= 0) 
      return FSOP_SUCCESS;
    else
      return FSOP_FAILURE;


}



static int 
  DEFUN(ufs_get_attr,
	(m,dr,fh),
	PC_ENV *m AND 
	struct doss_drive *dr AND
	struct doss_file_handle *fh)
{
    extern int errno;
    int rerr,err;
    char name[DOSS_MAX_PATH*2];
    struct stat sb;
    char pp[DOSS_MAX_PATH];
    
    strcpy(pp,fh->path);
    sprintf(name,"%s/%s",  dr->path,   parse_path(pp));

    rerr =  lstat(name,&sb);
    err = errno;
    
#ifdef DEBUG
    if (DEBUG_FS(m))
	{
	   debug_printf(m, "UFSD %d: GET ATTR %s",
		  dr->unit, name);

	   if  (rerr >= 0) 
	     debug_printf(m, " OK\n");
	   else
	       {
		  debug_printf(m, " FAILED\n");
		  errno = err;
		  perror("STAT FILE");
	       }
	}
#endif
    

    if (rerr == 0) 
	{
	   fh->size = sb.st_size;
	   fh->attrib = ufs_mode_to_attrib(sb.st_mode);
	   fh->mtime = epoch_to_doss_time(sb.st_mtime);
	   fh->mdate = epoch_to_doss_date(sb.st_mtime);	   
	   return FSOP_SUCCESS;
	}
    else
	{
	   return FSOP_FAILURE;
	}
 }


static int 
  DEFUN(ufs_set_attr,
	(m,dr,fh),
	PC_ENV *m AND
	struct doss_drive *dr AND
	struct doss_file_handle *fh)
{
    int rerr;
    char name[DOSS_MAX_PATH*2];
    struct stat sb;
    char pp[DOSS_MAX_PATH];
    int attrib,mdate,mtime;
    struct timeval tv[2];
    
    strcpy(pp,fh->path);
    sprintf(name,"%s/%s",  dr->path,   parse_path(pp));

    rerr =  lstat(name,&sb);

#ifdef DEBUG
    if (DEBUG_FS(m))
	{
	   debug_printf(m, "UFSD %d: SET ATTR %s \n",
		  dr->unit, name);
	}
#endif

    if (rerr == 0) 
	{

	   mdate = epoch_to_doss_date(sb.st_mtime);
	   mtime = epoch_to_doss_time(sb.st_mtime);
	   attrib = ufs_mode_to_attrib(sb.st_mode);

	   if (fh->mdate != mdate || 
	       fh->mtime != mtime)
	       {
		  tv[0].tv_usec = 0;
		  tv[0].tv_sec = doss_datetime_to_epoch(fh->mdate,
						     fh->mtime);
		  tv[1].tv_usec = 0;
		  tv[1].tv_sec = tv[0].tv_sec;
		  
		  rerr = utimes(name, tv);

#ifdef DEBUG
		  if (DEBUG_SYS(m))
		    debug_printf(m, "%ssetting mtime to:%s",
			    (rerr==0)?"":"Failed ",
			    ctime(&tv[0].tv_sec));
#endif		  

		  if (rerr) 
		      {
			 syscall_errno = 5;
			 return FSOP_FAILURE;
		      }
	       }
	   
	   if (fh->attrib != attrib)
	       {
		  attrib = attrib_to_ufs_mode(fh->attrib);
		  /* make sure we do not change anything below 
		     the bottom 9 bits . */
		  
		  attrib &= 0777;
		  rerr = chmod(name, attrib);
#ifdef DEBUG
		  if (DEBUG_SYS(m))
		    debug_printf(m, "%ssetting mode to:0%o",
			    (rerr==0)?"":"Failed ",
			    attrib);
#endif		  
		  if (rerr) 
		      {
			 syscall_errno = 5;
			 return FSOP_FAILURE;
		      }
		  return FSOP_SUCCESS;
	       }
	   return FSOP_SUCCESS;
	}
    else
	{
	   syscall_errno = 2;
	   return FSOP_FAILURE;
	}
    

 }



static int 
  DEFUN(ufs_dprint_file,
	(m,dr,fh),
	PC_ENV *m AND
	struct doss_drive *dr AND
	struct doss_file_handle *fh)
{
      struct stat sb;
      char name[DOSS_MAX_PATH*2];
      char pp[DOSS_MAX_PATH*2];

      strcpy(pp,fh->path);
      sprintf(name,"%s/%s",
	      dr->path,
	      parse_path(pp));
      lstat(name,&sb);
#ifdef DEBUG
	if (DEBUG_FS(m))
	  {
		debug_printf(m, "UFS%d: %s\t",dr->unit,fh->path);
		debug_printf(m, "%6d ", sb.st_size);
		if (sb.st_mode & 0400)
			debug_printf(m, "r");
		if (sb.st_mode & 0200)
			debug_printf(m, "w");
		debug_printf(m, "\n");
	     
	  }
#endif
	return FSOP_SUCCESS;
}

   
static int   
  DEFUN(ufs_create_dir,
	(m,dr,s),
	PC_ENV *m AND
	struct doss_drive *dr AND
	char *s)
{
    extern int errno;
    int rerr,err;
    char name[DOSS_MAX_PATH*2];
    char pp[DOSS_MAX_PATH*2];

    strcpy(pp,s);
    
    sprintf(name,"%s/%s",
	    dr->path,
	    parse_path(pp));

    rerr=mkdir(name,0740);
    err = errno;
    
#ifdef DEBUG
    if (DEBUG_FS(m))
	{
	   debug_printf(m, "UFSD %d: MKDIR %s",
		  dr->unit, s);
	   if  (rerr >= 0) 
	     debug_printf(m, " OK\n");
	   else
	       {
		  debug_printf(m, " FAILED\n");
		  errno = err;
		  perror("MKDIR");
	       }
	}
#endif
    
    if (rerr >= 0) 
      return FSOP_SUCCESS;
    else
	{
	   if (err == ENOENT)
	     syscall_errno = 3;
	   else if (err == EEXIST || err == EACCES)
	     syscall_errno = 5;
	   return FSOP_FAILURE;
	}


 }


static int 
  DEFUN(ufs_delete_dir,
	(m,dr,s),
	PC_ENV *m AND
	struct doss_drive *dr AND
	char *s)
{
    extern int errno;
    int rerr,err;
    char name[DOSS_MAX_PATH*2];
    char pp[DOSS_MAX_PATH*2];

    strcpy(pp,s);
    
    sprintf(name,"%s/%s",
	    dr->path,
	    parse_path(pp));

    rerr=rmdir(name);
    err = errno;
        
#ifdef DEBUG
    if (DEBUG_FS(m))
	{
	   debug_printf(m, "UFSD %d: RMDIR %s",
		  dr->unit, s);

	   if  (rerr >= 0) 
	     debug_printf(m, " OK\n");
	   else
	       {
		  debug_printf(m, " FAILED\n");
		  errno = err;
		  perror("RMDIR");
	       }
	}
#endif
    
    if (rerr >= 0) 
      return FSOP_SUCCESS;
    else
	{
	   if (err == EACCES)
	     syscall_errno = 5; /* access denied */
	   else if (err == ENOTEMPTY)
	     syscall_errno = 5;
	   else if (err == ENOTDIR)
	     syscall_errno = 3;
	   else if (err == ENOENT)
	     syscall_errno = 3;
	   return FSOP_FAILURE;
	}
 }


/* note that all state information about a directory
   is contained in a doss_file_handle structure.  
   This includes the name of the directory we are 
   attempting to open.
 */

static int 
  DEFUN(ufs_open_dir,
	(m,dr,fh),
	PC_ENV *m AND
	struct doss_drive *dr AND
	struct doss_file_handle *fh)
{
    extern int errno;
    int rerr,err;
    char name[DOSS_MAX_PATH*2];
    char pp[DOSS_MAX_PATH*2];

    strcpy(pp,fh->path);
    sprintf(name,"%s/%s",
	    dr->path,
	    parse_path(pp));

    /* XXXX */
    rerr = fh->fd = (int) opendir(name);
    err = errno;
    
#ifdef DEBUG
    if (DEBUG_FS(m))
	{
	   debug_printf(m, "UFSD %d: OPENDIR %s",
		  dr->unit, name);

	   if  (rerr >= 0) 
	     debug_printf(m, " OK\n");
	   else
	       {
		  debug_printf(m, " FAILED\n");
		  errno = err;
		  perror("OPENDIR");
	       }
	}
#endif
    if (rerr > 0) 
      return FSOP_SUCCESS;
    else
	{
	   syscall_errno = 3;
	   return FSOP_FAILURE;
	}
 }

	     
static int   
  DEFUN(ufs_read_dir,
	(m,dr,dfh,fh),
	PC_ENV *m AND
	struct doss_drive *dr AND
	struct doss_file_handle *dfh AND
	struct doss_file_handle *fh)
/* dfh for directory */
{
    extern int errno;
    int err;
    struct dirent  *de;
    char *p;

    /* XXXX really need to fix this code here. This is undoubtedly
       illegal C, as well as almost immoral.  No time, though.*/

    /* note that NEVER does a malformed filename get returned. */

    do 
	{
	   de  = readdir((DIR*)(dfh->fd));
	   err = errno;
	   if (de == 0)
	       {
		  /* break out completely on error. */
		  return FSOP_FAILURE;
	       }
	   if (is_legal_filename(de->d_name))
	     break;
	}
    while(1);
    
#ifdef DEBUG
    if (DEBUG_FS(m))
	{
	   debug_printf(m, "UFSD %d: READDIR: '%s'",
		  dr->unit,dfh->path);

	   if  (de != 0) 
	     debug_printf(m, " OK: '%s'\n",de->d_name);
	   else
	       {
		  debug_printf(m, " FAILED\n");
		  errno = err;
		  perror("READDIR");
	       }
	}
#endif

    if (de != 0)
	{
	   /* all we fill in is the name. */
	   sprintf(fh->path,"%s\\%s",dfh->path,de->d_name);
	   for(p=fh->path; *p; p++)
	     if (*p == '/') *p = '\\';
	   
	   return FSOP_SUCCESS;
	}
    else
	{
	   return FSOP_FAILURE;
	}
 }

static int   
  DEFUN(ufs_seek_dir,
	(m,dr,dfh,dta),
	PC_ENV *m AND
	struct doss_drive *dr AND
	struct doss_file_handle *dfh AND
	u_int8 *dta)

/* dfh for directory */
{
   
   int where;
   u_int16 word;
   
   
   FETCH_WORD(word, dta, 0);
   where = word;
   FETCH_WORD(word, dta, 2);
   where |= (word << 16); 

   seekdir((DIR*)(dfh->fd),where);
    
#ifdef DEBUG
    if (DEBUG_FS(m))
	{
	   debug_printf(m, "UFSD %d: SEEKDIR: %s",
		  dr->unit,dfh->path);
	   /* no way to tell if error occurred. */
	}
    
#endif

    return FSOP_SUCCESS;

 }



static int  
  DEFUN(ufs_tell_dir,(m,dr,dfh,dta),
	PC_ENV *m AND
	struct doss_drive *dr AND
	struct doss_file_handle *dfh AND
	u_int8 *dta)

/* dfh for directory */
{
   u_int16 word;
   int currp;

   currp = telldir((DIR*)(dfh->fd));

#ifdef DEBUG
    if (DEBUG_FS(m))
	{
	   debug_printf(m, "UFSD %d: TELLDIR: %s",
		  dr->unit,dfh->path);
	   /* no way to tell if error occurred. */
	}
    
#endif

   word = currp & 0xffff;
   STORE_WORD(dta, 0, word);
   word = (currp >> 16) & 0xffff;
   STORE_WORD(dta, 2, word);
   
   return FSOP_SUCCESS;

 }


static int  
  DEFUN(ufs_close_dir,(m,dr,dfh),
	PC_ENV *m AND 
	struct doss_drive *dr AND
	struct doss_file_handle *dfh)
/* dfh for directory */
{
   int err,rerr;
   extern int errno;   
   rerr = closedir((DIR*)(dfh->fd));
   err = errno;
   
#ifdef DEBUG
    if (DEBUG_FS(m))
	{
	   debug_printf(m, "UFSD %d: CLOSEDIR: %s",
		  dr->unit,dfh->path);
	   if  (rerr >= 0) 
	     debug_printf(m, " OK\n");
	   else
	       {
		  debug_printf(m, " FAILED\n");
		  errno = err;
		  perror("CLOSEDIR");
	       }
	}
    
#endif
   if (rerr >= 0)
     return FSOP_SUCCESS;
   else
     return FSOP_FAILURE;
 }

/* basically unimplemented functions */

static int 
  DEFUN(ufs_read_block,(m,dr,lbn,buff),
	PC_ENV *m AND struct doss_drive *dr AND int lbn AND char *buff)
 {
    ufs_error(m, "ufs read block unimplemented\n");
    return FSOP_FAILURE;
 }
    
    
static int  
  DEFUN(ufs_write_block,(m,dr,lbn,buff),
	PC_ENV *m AND struct doss_drive *dr AND int lbn AND char *buff)
 {
    ufs_error(m, "ufs write block unimplemented\n");
    return FSOP_FAILURE;
 }
    
    
    
static int  
  DEFUN(ufs_format,(m,dr),
	PC_ENV *m AND struct doss_drive *dr)  /* XXXXX */
 {
    ufs_error(m, "ufs format unimplemented\n");
    return FSOP_FAILURE;
 }
    
    
static int  
  DEFUN(ufs_disk_space,(m,dr),
	PC_ENV *m AND struct doss_drive *dr)  /* XXXXX */
 {
    return FSOP_SUCCESS;
 }

static int  
  DEFUN(ufs_get_fs_attr,(m,dr),
	PC_ENV *m AND struct doss_drive *dr)  /* XXXXX */
 {
    return FSOP_SUCCESS;
 }


/* 
  here's our switch table. 
  all calls will need a minimum of:
    a drive descriptor
    a path component or file table entry handle
  The specific information about the configuration 
  is contained in the drive info.  In particular, the 
  "mount point" corresponding to the drive will be 
  associated with the drive and not the fs.
  */

struct doss_fs   ufs = {
    unix_fs,

    ufs_initialize,     /* DONE   */

    ufs_read_block,     /* Unimpl */
    ufs_write_block,    /* Unimpl */
    ufs_format,         /* Unimpl */
    ufs_disk_space,     /* XXXXXX */
    ufs_get_fs_attr,    /* XXXXXX */

    ufs_create_file,    /* DONE  X */
    ufs_open_file,      /* DONE  X */
    ufs_close_file,     /* DONE  X */
    ufs_seek_file,      /* DONE  X */
    ufs_read_file,      /* DONE  X */
    ufs_write_file,     /* DONE  X */
    ufs_rename_file,
    ufs_delete_file,    /* DONE  X */
/*    ufs_stat_file,       DONE  X */
/*    ufs_dprint_file,     DONE  X (debugging) */

    ufs_get_attr,       /* DONE  X */
    ufs_set_attr,       /* DONE  X */

    ufs_create_dir,     /* DONE  X */
    ufs_delete_dir,     /* DONE  X */
    ufs_open_dir,       /* DONE  X */
    ufs_read_dir,       /* DONE  X */
    ufs_seek_dir,       /* DONE  X */
    ufs_tell_dir,       /* DONE  X */
    ufs_close_dir,      /* DONE  X */
 };



#ifdef TEST
#include "test/t_doss_ufs.c"
#endif

#endif  /* DOSS_SUPPORT */
