/* 
 * Dos/PC Emulator
 * Copyright (C) 1991 Jim Hudgens
 * 
 * 
 * The file is part of GDE.
 * 
 * GDE is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 1, or (at your option)
 * any later version.
 * 
 * GDE is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GDE; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.  
 *
 */

static char rcsid[]=
  "$Id: doss_svc_tab.c,v 1.7 1991/07/30 01:56:36 hudgens Exp hudgens $";

/* $Log: doss_svc_tab.c,v $
 * Revision 1.7  1991/07/30  01:56:36  hudgens
 * added copyright.
 *
 * Revision 1.6  1991/07/21  18:27:38  hudgens
 * Added support for service 33, which is a temporary hack.
 *
 * Revision 1.5  1991/07/21  04:49:28  hudgens
 * Added  preliminary version of service x38 --- get/set country
 * Fixed sign extension problem in read/write file.
 *
 * Revision 1.4  1991/07/20  23:00:09  hudgens
 * no major changes.
 *
 * Revision 1.3  1991/07/20  18:36:28  hudgens
 * added more checking to the interrupt setting/getting services.
 *
 * Revision 1.2  1991/07/20  17:58:39  hudgens
 * added more debugging messages.
 *
 * Revision 1.1  1991/07/20  04:08:14  hudgens
 * Initial revision
 *
 *
 */


#ifdef DOSS_SUPPORT
#include "gde.h"


static char *SUCCESS="SUCCESS";
static char *FAIL="FAIL";


/*----------------------------------------------------------------------*/
/*----------------------------------------------------------------------*/
/*----------------------------------------------------------------------*/

static void  DEFUN(doss_svc_tab_00,(m),PC_ENV *m)
    {

#ifdef DEBUG
      if (DEBUG_SVC(m))
	    {
	      decode_printf(m,"\n---\nDOS SVC 0x00: EXIT\n");    
	    }
#endif

       doss_exit(m,0);
    }


/*----------------------------------------------------------------------*/
/* character input with echo.                                           */
/*----------------------------------------------------------------------*/


static void  DEFUN(doss_svc_tab_01,(m),PC_ENV *m)
    {
       u_int8 chr;
       int rv,nr,nw;
       int ioflags;

       if (doss_conio_is_conin(m,1) == SYSCALL_SUCCESS)
	   {
	      ioflags = DOSS_CONIO_IECHO |
		        DOSS_CONIO_IBLK |
		        DOSS_CONIO_CTLBRK ;

	      rv = doss_conio_in_chr(m,&chr,ioflags);  
	      m->R_AL = chr;
	   }
       else
	   {

       /* this has got to be the most goddamned brain-damaged I/O scheme ever
	  invented.  In PC_system_programming by M Tischer, 
	  it states:  "When the function is called but the character
	  doesn't exist, the function waits until a character is 
	  available.  Since input and output can be redirected ...
	  If the character comes from a file, input doesn't redirect
	  to the keyboard once it reaches end of file".  I take that 
	  to mean that this is a nice tight little infinite loop.
	*/
	      nr = 0;
	      while (nr == 0)
		  {
		     /* file handle 0, read one character. */	      
		     rv = doss_fsop_fread(m,0,&chr,1,&nr);  
		  }
	      /* echo it back XXXXX??? */
	      if (rv == SYSCALL_SUCCESS) 
		rv = doss_fsop_fwrite(m,1,&chr,1,&nw);
	      m->R_AL = chr;
	   }
	      
#if defined(DEBUG)
      if (DEBUG_SVC(m))
	{
	   debug_printf(m,"\n---\nDOS SVC 0x1:  CHARACTER INPUT\n");	
	   debug_printf(m,"<<<%x\n",chr);
	}
#endif
       /* no error checking.  No return codes.  Nuthin. */
 }


/*----------------------------------------------------------------------*/
/* write character to stdout  (fhl == 1)                                */
/*----------------------------------------------------------------------*/


static void  DEFUN(doss_svc_tab_02,(m),PC_ENV *m)
    {
       u_int8 chr;
       int rv,nw;
       int ioflags;
       
#if defined(DEBUG)
      if (DEBUG_SVC(m))
	{
	   debug_printf(m,"\n---\nDOS SVC 0x2:  CHARACTER OUTPUT\n");
	   debug_printf(m,">>>%x\n",m->R_DL);
	}
#endif

       if (doss_conio_is_conout(m,1) == SYSCALL_SUCCESS)
	   {
	      chr = m->R_DL;
	      ioflags = DOSS_CONIO_CTLBRK | DOSS_CONIO_OCOOK ;
	      doss_conio_out_chr(m,chr,ioflags);  
	   }
       else
	   {
       
	      chr = m->R_DL;
	      rv = doss_fsop_fwrite(m,1,&chr,1,&nw);  /* file handle 1,
							 one character. */
	   }
       /* no error checking.  No return codes.*/
 }


/*
 * Noone probably uses services #3 and #4, which are aux-input and 
 * aux-output, respectively.  Therefore, we'll leave them undefined
 * by design.
 */


/*----------------------------------------------------------------------*/
/* write character to printer                                           */
/*----------------------------------------------------------------------*/


static void  DEFUN(doss_svc_tab_05,(m),PC_ENV *m)
    {
       u_int8 chr;
       int rv,nw;
       struct timeval ts;

#if defined(DEBUG)
      if (DEBUG_SVC(m))
	{
	   debug_printf(m,"\n---\nDOS SVC 0x5:  PRN CHARACTER OUTPUT\n");
	   debug_printf(m,">>>%x\n",m->R_DL);
	}
#endif
       chr = m->R_DL;
       if (m->printer_desc[0].lpd_fd_stream != (FILE  *) NULL)
	   {
	      rv = fwrite(&chr, 1, 1, 
			 m->printer_desc[0].lpd_fd_stream);
	      /* on success, update timestamp and return */
	      gettimeofday(&ts,(struct timezone *)NULL);
	      m->printer_desc[0].lpd_fd_last_access 
		    = ts.tv_sec;
	   }
       /* no error checking.  No return codes.*/
 }


/*----------------------------------------------------------------------*/
/*  direct console I/O service.                                         */
/*----------------------------------------------------------------------*/


static void  DEFUN(doss_svc_tab_06,(m),PC_ENV *m)
    {
       u_int8 chr;
       int rv,nw,nr;
       int ioflags;

#if defined(DEBUG)
    if (DEBUG_SVC(m))
	{
	   debug_printf(m,"\n---\nDOS SVC 0x6:  CONIO\n");
	   debug_printf(m,">>>%x\n",m->R_DL);
	}
#endif

    if (m->R_DL != 0xff) 
	{
	   chr = m->R_DL;
	   if (doss_conio_is_conout(m,1) == SYSCALL_SUCCESS)
	       {
		  ioflags = 0;
		  doss_conio_out_chr(m,chr,ioflags);  
	       }
	   else
	       {
		  chr = m->R_DL;
		  rv = doss_fsop_fwrite(m,1,&chr,1,&nw);  /* file handle 1,
							     one character. */
	       }
	   /* no error checking.  No return codes.*/
	   /* maybe it went out, maybe it didn't */
	}
    else 
	{
	   if (doss_conio_is_conin(m,1) == SYSCALL_SUCCESS)
	       {
		  ioflags = 0;   /* no ctl-brk, noblock, noecho */
		  rv = doss_conio_in_chr(m,&chr,ioflags);  
		  if (rv == SYSCALL_SUCCESS)
		    SET_FLAG(m,F_ZF);
		  else
		      {
			 CLEAR_FLAG(m,F_ZF);
			 m->R_AL = chr;
		      }
	       }
	   else  /* redirected from a file */
	       {
		  rv = doss_fsop_fread(m,0,&chr,1,&nr);  
		  if (rv == SYSCALL_SUCCESS)
		    SET_FLAG(m,F_ZF);
		  else
		      {
			 CLEAR_FLAG(m,F_ZF);
			 m->R_AL = chr;
		      }
	       }
	}
 }

/*----------------------------------------------------------------------*/
/*  unfiltered character input.                                         */
/*----------------------------------------------------------------------*/

static void  DEFUN(doss_svc_tab_07,(m),PC_ENV *m)
    {
       u_int8 chr;
       int rv,nr,nw,ioflags;
       

       if (doss_conio_is_conin(m,1) == SYSCALL_SUCCESS)
	   {
	      ioflags = DOSS_CONIO_IBLK ; 
	      rv = doss_conio_in_chr(m,&chr,ioflags);  
	      m->R_AL = chr;
	   }
       else
	   {
	      nr = 0;
	      while (nr == 0)
		  {
		     /* file handle 0, read one character. */
		     rv = doss_fsop_fread(m,0,&chr,1,&nr);  
		  }
	      m->R_AL = chr;
	   }
       

#if defined(DEBUG)
      if (DEBUG_SVC(m))
	{
	   debug_printf(m,
			"\n---\nDOS SVC 0x7:  CHARACTER INPUT, NO ECHO\n");
	   debug_printf(m,"<<<%x\n",chr);
	}
#endif
       /* no error checking.  No return codes. */
 }

/*----------------------------------------------------------------------*/
/*  filtered character input.                                           */
/*----------------------------------------------------------------------*/


static void  DEFUN(doss_svc_tab_08,(m),PC_ENV *m)
    {
       u_int8 chr;
       int rv,nr,nw,ioflags;
       

       if (doss_conio_is_conin(m,1) == SYSCALL_SUCCESS)
	   {
	      ioflags = DOSS_CONIO_IBLK | DOSS_CONIO_CTLBRK; 
	      rv = doss_conio_in_chr(m,&chr,ioflags);  
	      m->R_AL = chr;
	   }
       else
	   {
	      nr = 0;
	      while (nr == 0)
		  {
		     /* file handle 0, read one character. */
		     rv = doss_fsop_fread(m,0,&chr,1,&nr);  
		  }
	      m->R_AL = chr;
	   }

#if defined(DEBUG)
      if (DEBUG_SVC(m))
	{
	   debug_printf(m,
			"\n---\nDOS SVC 0x7:  CHARACTER INPUT, NO ECHO\n");
	   debug_printf(m,"<<<%x\n",chr);
	}
#endif
       /* no error checking.  No return codes. */
 }

/*----------------------------------------------------------------------*/
/*                                                                      */
/*----------------------------------------------------------------------*/


static void  DEFUN(doss_svc_tab_09,(m),PC_ENV *m)
 {
   int rv,nw;
   int iscons,ioflags;
   u_int8 chr;

   u_int8 *s = (m->mem_base + ((u_int16) m->R_DS << 4) + m->R_DX);
   iscons = doss_conio_is_conout(m,1) == SYSCALL_SUCCESS;
   
#if defined(DEBUG)
    if (DEBUG_SVC(m))
	{
	   decode_printf(m,"\n---\nDOS SVC 0x9:  STRING OUTPUT\n");
	   decode_printf(m,">>>");
	}
#endif

    while (*s != '$') 
	{
#if defined(DEBUG)
	  if (DEBUG_SVC(m))
	      {
		debug_printf(m,"%c",*s);
	      }
#endif
	  chr = *s;
	  if (iscons)
	      {
		 ioflags = DOSS_CONIO_CTLBRK | DOSS_CONIO_OCOOK ;
		 doss_conio_out_chr(m,chr,ioflags);  
	      }
	  else
	      {
		 rv = doss_fsop_fwrite(m,1,&chr,1,&nw);  /* file handle 1,
							    one character. */
	      }
	  s += 1;
       }
 }  


/*----------------------------------------------------------------------*/
/*----------------------------------------------------------------------*/

static void  DEFUN(doss_svc_tab_0b,(m),PC_ENV *m)
    {
       /* XXX */
       m->R_AL = 0;
       
    }

/*----------------------------------------------------------------------*/
/* SET DEFAULT DRIVE.                                                   */
/*----------------------------------------------------------------------*/

static void  DEFUN(doss_svc_tab_0e,(m),PC_ENV *m)
    {
       
       if (m->R_DL <= m->doss_lastdrive)
	   {
	      /* [JCE] Use drive in DL, not AL... */
	      m->doss_curr_drive = m->R_DL;
	   }
       m->R_AL = m->doss_ndrives;

#if defined(DEBUG)
    if (DEBUG_SVC(m))
	{
	   debug_printf(m,"\n---\nDOS SVC 0x0E:  SET DEFAULT DRIVE\n");    
	   debug_printf(m,"drive=%d ndrives=%d\n", m->R_DL, m->R_AL);
	}
#endif
       
    }


/*----------------------------------------------------------------------*/
/* GET CURRENT DRIVE                                                    */
/*----------------------------------------------------------------------*/


static void  DEFUN(doss_svc_tab_19,(m),PC_ENV *m)
    {
       m->R_AL = m->doss_curr_drive;
#if defined(DEBUG)
    if (DEBUG_SVC(m))
	{
	   debug_printf(m,"\n---\nDOS SVC 0x19: GET DEFAULT DRIVE\n");    
	   debug_printf(m,"drive=%d\n", m->R_AL);
	}
#endif
    }

/*----------------------------------------------------------------------*/
/* SET DTA ADDRESS                                                      */
/*----------------------------------------------------------------------*/


static void  DEFUN(doss_svc_tab_1a,(m),PC_ENV *m)
    {

       m->doss_curr_proc->dta_seg = m->R_DS;
       m->doss_curr_proc->dta_off = m->R_DX;
#if defined(DEBUG)
    if (DEBUG_SVC(m))
	{
	   debug_printf(m,"\n---\nDOS SVC 0x1a: SET DTA ADDR\n");    
	   debug_printf(m,"dta=0x%x:0x%x\n", 
		   m->R_DS, m->R_DX);
	}
#endif

    }


/*----------------------------------------------------------------------*/
/*----------------------------------------------------------------------*/


static void  DEFUN(doss_svc_tab_25,(m),PC_ENV *m)
 {

    unsigned int i;
       
#if defined(DEBUG)
    if (DEBUG_SVC(m))
	{
	   debug_printf(m,"\n---\nDOS SVC 0x25:  SET INTERRUPT VECTOR\n");    
	   debug_printf(m,"Interrupt #%x set to %04x:%04x\n", 
		  m->R_AL, (u_int16) m->R_DS, (u_int16)  m->R_DX);
	}
#endif
    i = m->R_AL;
    *(m->mem_base + i*4) = m->R_DX & 0xff;	/* [JCE] Offset is low word */
    *(m->mem_base + i*4 + 1) = m->R_DX >> 8;
    *(m->mem_base + i*4 + 2) = m->R_DS & 0xff;
    *(m->mem_base + i*4 + 3) = m->R_DS >> 8; 
#ifdef DEBUG
    if (DEBUG_SVC(m))
	{
	   print_int_vect(m,i);
	   decode_printf(m,"\n");
	}
#endif
 }


/*----------------------------------------------------------------------*/
/*----------------------------------------------------------------------*/


static void  DEFUN(doss_svc_tab_2a,(m),PC_ENV *m)
    {

       
#if defined(DEBUG)
    if (DEBUG_SVC(m))
	{
	   decode_printf(m,"\n---\nDOS SVC 0x2a:  Get system date\n");    
	}
#endif
       m->R_AL = 0;
       m->R_CX = 1980;
       m->R_DH = 0;
       m->R_DL = 0;
    }


/*----------------------------------------------------------------------*/
/*  GET SYSTEM TIME                                                     */
/*----------------------------------------------------------------------*/


static void  DEFUN(doss_svc_tab_2c,(m),PC_ENV *m)
    {

       u_int8 hours,minutes,seconds,centisecs;

       doss_get_clock(&hours,&minutes,&seconds,&centisecs);
       m->R_CH = hours;
       m->R_CL = minutes;
       m->R_DH = seconds;
       m->R_DL = centisecs;
       CLEAR_FLAG(m,F_CF);

#if defined(DEBUG)
    if (DEBUG_SVC(m))
	{
	   debug_printf(m,"\n---\nDOS SVC 0x2a:  Get system time\n");    
	   debug_printf(m,"time: %02d:%02d:%02d.%02d\n",
		  hours,minutes,seconds,centisecs);
	}
#endif

    }


/*----------------------------------------------------------------------*/
/*  GET DTA ADDRESS                                                     */
/*----------------------------------------------------------------------*/


static void  DEFUN(doss_svc_tab_2f,(m),PC_ENV *m)
    {
       m->R_ES = m->doss_curr_proc->dta_seg;
       m->R_BX = m->doss_curr_proc->dta_off;
       RESYNCH_SEG_ES(m);

#if defined(DEBUG)
    if (DEBUG_SVC(m))
	{
	   debug_printf(m,"\n---\nDOS SVC 0x2f:  GET DTA\n");    
	   debug_printf(m,"dta=0x%x:0x%x\n", m->R_ES,m->R_BX);
	}
#endif
    }

/*----------------------------------------------------------------------*/
/* GET DOS VERSION NUMBER                                               */
/*----------------------------------------------------------------------*/


static void  DEFUN(doss_svc_tab_30,(m),PC_ENV *m)
 {
    m->R_AL = 3;
    m->R_AH = 30;
    /* so sue me.  What else can be put here??? */

#if defined(DEBUG)
    if (DEBUG_SVC(m))
	{
	   
	   debug_printf(m,"\n---\nDOS SVC 0x30:  GET DOS VERSION\n");    
	   debug_printf(m,"DOS Version: %d.%d\n", 
		  m->R_AL,  m->R_AH);
	}
#endif
 }

/*----------------------------------------------------------------------*/
/*----------------------------------------------------------------------*/


static void  DEFUN(doss_svc_tab_33,(m),PC_ENV *m)
    {
       
       static ctrl_brk=0;
       

       switch(m->R_AL)
	   {
	    case 0:
	      m->R_DL = ctrl_brk;
	      break;
	    case 1:
	      ctrl_brk = m->R_DL;
	      break;
	   }
       
    }


/*----------------------------------------------------------------------*/
/* GET INTERRUPT VECTOR                                                 */
/*----------------------------------------------------------------------*/

static void  DEFUN(doss_svc_tab_35,(m),PC_ENV *m)
 {
       
    unsigned int i;

    i = m->R_AL;
/* [JCE] Offset is the low word */
    m->R_BX  = (u_int8) *(m->mem_base + i*4);	
    m->R_BX |= (u_int8) *(m->mem_base + i*4) << 8;
    m->R_ES  = (u_int8) *(m->mem_base + i*4 + 2);
    m->R_ES |= (u_int8) *(m->mem_base + i*4 + 3) << 8;
    RESYNCH_SEG_ES(m);
       
#if defined(DEBUG)
    if (DEBUG_SVC(m))
	{
	   debug_printf(m,"\n---\nDOS SVC 0x35:  GET INTERRUPT VECTOR\n");    
	   debug_printf(m,"Interrupt #%x is at %04x:%04x\n", 
		  m->R_AL,  (u_int16)m->R_ES, (u_int16)m->R_BX);
	}
#endif
 }

/*----------------------------------------------------------------------*/
/* undocumented.  Get switch char.                                      */
/*----------------------------------------------------------------------*/


static void  DEFUN(doss_svc_tab_37,(m),PC_ENV *m)
    {
       /* based on  _UNDOCUMENTED DOS_ information */

       switch(m->R_AL)
	   {
	    case 0:  /* get? */
	      m->R_DL = '/';
	      break;
	    case 1:
	      m->R_AL = 0xff;
	      break;
	    case 2:
	      m->R_DL = 0xff;
	      break;
	    case 3:
	      /* nada. */
	      break;
	   }
       
       
    }

/*----------------------------------------------------------------------*/
/* get country.                                                         */
/*----------------------------------------------------------------------*/


static void  DEFUN(doss_svc_tab_38,(m),PC_ENV *m)
    {
       
       u_int8 *cntry_buf;

       if ((u_int16)m->R_DX != 0xffff)
	   {
	      /* getting country information */
	      
	      cntry_buf = m->mem_base + 
		((u_int16)m->R_DS<<4) + (u_int16)m->R_DX;
	      
	      /* ignore the BS.  Lets just put in the 
		 info for US -- FIX LATER    XXXX */

	      cntry_buf[0] = 0;
	      cntry_buf[1] = 0;
	      cntry_buf[2] = '$';    /* currency string*/
	      cntry_buf[3] = 0;
	      cntry_buf[4] = 0;
	      cntry_buf[5] = 0;
	      cntry_buf[6] = 0;
	      cntry_buf[7] = ',';    /* thous. sep. */
	      cntry_buf[8] = 0;
	      cntry_buf[9] = '.';    /* decimal sep. */
	      cntry_buf[10] = 0;
	      cntry_buf[11] = '-';   /* date sep. */
	      cntry_buf[12] = 0;
	      cntry_buf[13] = ':';   /* time sep. */
	      cntry_buf[14] = 0;
	      cntry_buf[15] = 0;     /* currency = $x */
	      cntry_buf[16] = 2;     /* digits after . in currency */
	      cntry_buf[17] = 0;     /* 12 hour clock. */
	      cntry_buf[18] = 0xff;
	      cntry_buf[19] = 0xff;
	      cntry_buf[20] = 0x0e;
	      cntry_buf[21] = 0x00;
	      cntry_buf[22] = ',';
	      cntry_buf[23] = 0;
	   }
       CLEAR_FLAG(m,F_CF);
    }
    

/*----------------------------------------------------------------------*/
/* CREATE DIRECTORY                                                     */
/*----------------------------------------------------------------------*/

static void  DEFUN(doss_svc_tab_39,(m),PC_ENV *m)
    {
       int retval;
       u_int8 *path;
      
       path = m->mem_base + ((u_int16)m->R_DS<<4) + (u_int16)m->R_DX;

#if defined(DEBUG)
    if (DEBUG_SVC(m))
	{
	   debug_printf(m,"\n---\nDOS SVC 0x39:  CREATE DIR\n");    
	   debug_printf(m,"dir=%s\n",path);
	}
#endif
       retval = doss_fsop_mkdir(m,path);

#if defined(DEBUG)
    if (DEBUG_SVC(m))
	{
	   if (retval==SYSCALL_SUCCESS)
	     debug_printf(m,"retval=SUCCESS\n---\n");
	   else
	     debug_printf(m,"retval=FAIL errno=%x \n---\n",
		     syscall_errno);
	}
#endif

       
       if (retval == SYSCALL_FAILURE)
	   {
	      SET_FLAG(m,F_CF);
	      m->R_AX = syscall_errno;
	   }
       else
	   {
	      CLEAR_FLAG(m,F_CF);
	   }
    }


/*----------------------------------------------------------------------*/
/* DELETE DIRECTORY                                                     */
/*----------------------------------------------------------------------*/

static void  DEFUN(doss_svc_tab_3a,(m),PC_ENV *m)
    {
       int retval;
       u_int8 *path;

       path = m->mem_base + ((u_int16)m->R_DS<<4) + (u_int16)m->R_DX;

#if defined(DEBUG)
    if (DEBUG_SVC(m))
	{
	   debug_printf(m,"\n---\nDOS SVC 0x3A:  DELETE DIR\n");    
	   debug_printf(m,"dir=%s\n",path);
	}
#endif

       retval = doss_fsop_rmdir(m,path);

#if defined(DEBUG)
    if (DEBUG_SVC(m))
	{
	   if (retval==SYSCALL_SUCCESS)
	     debug_printf(m,"retval=SUCCESS\n---\n");
	   else
	     debug_printf(m,"retval=FAIL errno=%x \n---\n",
		     syscall_errno);
	}
#endif

       
       if (retval == SYSCALL_FAILURE)
	   {
	      SET_FLAG(m,F_CF);
	      m->R_AX = syscall_errno;
	   }
       else
	   {
	      CLEAR_FLAG(m,F_CF);
	   }

    }

/*----------------------------------------------------------------------*/
/* set current directory                                                */
/*----------------------------------------------------------------------*/


static void  DEFUN(doss_svc_tab_3b,(m),PC_ENV *m)
    {
       int retval;
       u_int8 *path;
      
       path = m->mem_base + ((u_int16)m->R_DS<<4) + (u_int16)m->R_DX;

#if defined(DEBUG)
    if (DEBUG_SVC(m))
	{
	   debug_printf(m,"\n---\nDOS SVC 0x3B:  SET CWD\n");    
	   debug_printf(m,"dir=%s\n",path);
	}
#endif

       retval = doss_fsop_setcwd(m,path);


#if defined(DEBUG)
    if (DEBUG_SVC(m))
	{
	   if (retval==SYSCALL_SUCCESS)
	     debug_printf(m,"retval=SUCCESS\n---\n");
	   else
	     debug_printf(m,"retval=FAIL errno=%x \n---\n",
		     syscall_errno);
	}
#endif

       
       if (retval == SYSCALL_FAILURE)
	   {
	      SET_FLAG(m,F_CF);
	      m->R_AX = syscall_errno;
	   }
       else
	   {
	      CLEAR_FLAG(m,F_CF);
	   }
    }


/*----------------------------------------------------------------------*/
/*  CREATE FILE.                                                        */
/*----------------------------------------------------------------------*/


static void  DEFUN(doss_svc_tab_3c,(m),PC_ENV *m)
    {
       int retval;
       int attr;
       u_int8 *path;
       int fhl;
      
       attr = m->R_CX & 0x7;
       path = m->mem_base + ((u_int16)m->R_DS<<4) + (u_int16)m->R_DX;

#if defined(DEBUG)
    if (DEBUG_SVC(m))
	{
	   debug_printf(m,"\n---\nDOS SVC 0x3C:  CREATE FILE\n");    
	   debug_printf(m,"file=%s, attr=%x\n",path,attr);
	}
#endif
       retval = doss_fsop_fcreat(m,path,attr,&fhl);

#if defined(DEBUG)
    if (DEBUG_SVC(m))
	{
	   if (retval==SYSCALL_SUCCESS)
	     debug_printf(m,"retval=SUCCESS handle=%d\n---\n",fhl);
	   else
	     debug_printf(m,"retval=FAIL errno=%x \n---\n",
		     syscall_errno);
	}
#endif

       
       if (retval == SYSCALL_FAILURE)
	   {
	      SET_FLAG(m,F_CF);
	      m->R_AX = syscall_errno;
	   }
       else
	   {
	      CLEAR_FLAG(m,F_CF);
	      m->R_AX = fhl;
	   }
       
       
    }


/*----------------------------------------------------------------------*/
/*  OPEN FILE.                                                          */
/*----------------------------------------------------------------------*/

static void  DEFUN(doss_svc_tab_3d,(m),PC_ENV *m)
    {
       int retval;
       int access;
       u_int8 *path;
       int fhl;
      
       access = m->R_AL;

       path = m->mem_base + ((u_int16)m->R_DS<<4) + (u_int16)m->R_DX;

#if defined(DEBUG)
    if (DEBUG_SVC(m))
	{
	   debug_printf(m,"\n---\nDOS SVC 0x3D:  open FILE\n");    
	   debug_printf(m,"file=%s, access=%x\n",path,access);
	}
#endif

       retval = doss_fsop_fopen(m,path,access,&fhl);

#if defined(DEBUG)
    if (DEBUG_SVC(m))
	{
	   if (retval==SYSCALL_SUCCESS)
	     debug_printf(m,"retval=SUCCESS handle=%d\n---\n",fhl);
	   else
	     debug_printf(m,"retval=FAIL errno=%x \n---\n",
		     syscall_errno);
	}
#endif

       
       if (retval == SYSCALL_FAILURE)
	   {
	      SET_FLAG(m,F_CF);
	      m->R_AX = syscall_errno;
	   }
       else
	   {
	      CLEAR_FLAG(m,F_CF);
	      m->R_AX = fhl;
	   }
    }

/*----------------------------------------------------------------------*/
/*  CLOSE FILE.                                                         */
/*----------------------------------------------------------------------*/


static void  DEFUN(doss_svc_tab_3e,(m),PC_ENV *m)
    {
       int retval;
       int fhl;

       fhl = m->R_BX;

#if defined(DEBUG)
    if (DEBUG_SVC(m))
	{
	   debug_printf(m,"\n---\nDOS SVC 0x3C: CLOSE FILE\n");    
	   debug_printf(m,"filehandle=%d\n",fhl);
	}
#endif

       retval = doss_fsop_fclose(m,fhl);

#if defined(DEBUG)
    if (DEBUG_SVC(m))
	{
	   if (retval==SYSCALL_SUCCESS)
	     debug_printf(m,"retval=SUCCESS\n---\n");
	   else
	     debug_printf(m,"retval=FAIL errno=%x \n---\n",
		     syscall_errno);
	}
#endif

       
       if (retval == SYSCALL_FAILURE)
	   {
	      SET_FLAG(m,F_CF);
	      m->R_AX = syscall_errno;
	   }
       else
	   {
	      CLEAR_FLAG(m,F_CF);
	   }
    }

/*----------------------------------------------------------------------*/
/* READ FILE.                                                           */
/*----------------------------------------------------------------------*/


static void  DEFUN(doss_svc_tab_3f,(m),PC_ENV *m)
    {
       int retval;
       int fhl;
       u_int32 ntoread;
       int nread;	/* [JCE] doss_fsop_fread() takes an int pointer */
       u_int8 *buf;
       
       fhl = m->R_BX;
       ntoread  = (u_int16)m->R_CX;
       buf = m->mem_base + ((u_int16)m->R_DS<<4) + (u_int16)m->R_DX;

#if defined(DEBUG)
    if (DEBUG_SVC(m))
	{
	   debug_printf(m,"\n---\nDOS SVC 0x3F: READ FILE\n");    
	   debug_printf(m,"filehandle=%d ntoread=%d buf=%04x:%04x\n",
		  fhl,ntoread,m->R_DS,m->R_DX);
	}
#endif
       retval = doss_fsop_fread(m,fhl,buf,ntoread,&nread);

#if defined(DEBUG)
    if (DEBUG_SVC(m))
	{
	   if (retval==SYSCALL_SUCCESS)
	     debug_printf(m,"retval=SUCCESS nread=%d\n---\n",nread);
	   else
	     debug_printf(m,"retval=FAIL errno=%x \n---\n",
		     syscall_errno);
	}
#endif

       
       if (retval == SYSCALL_FAILURE)
	   {
	      SET_FLAG(m,F_CF);
	      m->R_AX = syscall_errno;
	   }
       else
	   {
	      CLEAR_FLAG(m,F_CF);
	      m->R_AX = nread;
	   }
    }

/*----------------------------------------------------------------------*/
/* WRITE FILE.                                                           */
/*----------------------------------------------------------------------*/


static void  DEFUN(doss_svc_tab_40,(m),PC_ENV *m)
    {
       int retval;
       int fhl;
       u_int32 ntowrite;
       int nwrite;	/* [JCE] doss_fsop_fwrite takes an int pointer */
       u_int8 *buf;
       
       fhl = m->R_BX;
       ntowrite  = (u_int32)m->R_CX;
       buf = m->mem_base + ((u_int16)m->R_DS<<4) + (u_int16)m->R_DX;

#if defined(DEBUG)
    if (DEBUG_SVC(m))
	{
	   debug_printf(m,"\n---\nDOS SVC 0x3F: WRITE FILE\n");    
	   debug_printf(m,"filehandle=%d ntowrite=%d buf=%04x:%04x\n",
		  fhl,ntowrite,m->R_DS,m->R_DX);
	}
#endif

       retval = doss_fsop_fwrite(m,fhl,buf,ntowrite,&nwrite);

#if defined(DEBUG)
    if (DEBUG_SVC(m))
	{
	   if (retval==SYSCALL_SUCCESS)
	     debug_printf(m,"retval=SUCCESS nwrite=%d\n---\n",nwrite);
	   else
	     debug_printf(m,"retval=FAIL errno=%x \n---\n",
		     syscall_errno);
	}
#endif

       
       if (retval == SYSCALL_FAILURE)
	   {
	      SET_FLAG(m,F_CF);
	      m->R_AX = syscall_errno;
	   }
       else
	   {
	      CLEAR_FLAG(m,F_CF);
	      m->R_AX = nwrite;
	   }
    }

/*----------------------------------------------------------------------*/
/* DELETE FILE.                                                         */
/*----------------------------------------------------------------------*/


static void  DEFUN(doss_svc_tab_41,(m),PC_ENV *m)
    {
       int retval;
       u_int8 *path;
       
       path = m->mem_base + ((u_int16)m->R_DS<<4) + (u_int16)m->R_DX;

#if defined(DEBUG)
    if (DEBUG_SVC(m))
	{
	   debug_printf(m,"\n---\nDOS SVC 0x41: DELETE FILE\n");    
	   debug_printf(m,"filehandle=%s\n",
		  path);
	}
#endif

       retval = doss_fsop_fdelete(m,path);

#if defined(DEBUG)
    if (DEBUG_SVC(m))
	{
	   if (retval==SYSCALL_SUCCESS)
	     debug_printf(m,"retval=SUCCESS\n---\n");
	   else
	     debug_printf(m,"retval=FAIL errno=%x \n---\n",
		     syscall_errno);
	}
#endif

       
       if (retval == SYSCALL_FAILURE)
	   {
	      SET_FLAG(m,F_CF);
	      m->R_AX = syscall_errno;
	   }
       else
	   {
	      CLEAR_FLAG(m,F_CF);
	   }


    }

/*----------------------------------------------------------------------*/
/* SEEK FILE.                                                           */
/*----------------------------------------------------------------------*/


static void  DEFUN(doss_svc_tab_42,(m),PC_ENV *m)
    {
       int retval;
       int fhl;
       int32 whence,offset;
       int32 at;
       
       
       fhl = m->R_BX;
       whence = m->R_AL;
       offset = ((m->R_CX & 0xffff) << 16) | (m->R_DX & 0xffff);

#if defined(DEBUG)
    if (DEBUG_SVC(m))
	{
	   debug_printf(m,"\n---\nDOS SVC 0x42: SEEK FILE\n");    
	   debug_printf(m,"filehandle=%d whence=%s offset=%d\n",
		  fhl,
		  (whence==0?"BEGINNING":(whence==1?"CURRENT":"END")),
		  offset);
	}
#endif

       retval = doss_fsop_fseek(m,fhl,whence,offset,&at);

#if defined(DEBUG)
    if (DEBUG_SVC(m))
	{
	   if (retval==SYSCALL_SUCCESS)
	     debug_printf(m,"retval=SUCCESS at=%d\n---\n",at);
	   else
	     debug_printf(m,"retval=FAIL errno=%x \n---\n",
		     syscall_errno);
	   
	}
#endif

       
       if (retval == SYSCALL_FAILURE)
	   {
	      SET_FLAG(m,F_CF);
	      m->R_AX = syscall_errno;
	   }
       else
	   {
	      CLEAR_FLAG(m,F_CF);
	      m->R_AX = at & 0xffff;
	      m->R_DX = (at>>16)&0xffff;
	   }

    }

/*----------------------------------------------------------------------*/
/*   GET/SET FILE ATTR                                                  */
/*----------------------------------------------------------------------*/


static void  DEFUN(doss_svc_tab_43,(m),PC_ENV *m)
    {

       u_int8 *path;
       u_int16  bits;
       int rval;
       
       
       path = m->mem_base + ((u_int16)m->R_DS<<4) + (u_int16)m->R_DX;
       

       switch (m->R_AL)
	   {
	    case 0:
	      rval = doss_fsop_getfattr(m,path,&bits);
	      
	      if (rval == SYSCALL_SUCCESS)
		  {
		     m->R_CX = bits;
		     CLEAR_FLAG(m,F_CF);
		  }
	      
	      else
		  {
		     m->R_AX = syscall_errno;
		     SET_FLAG(m, F_CF);
		  }
#if defined(DEBUG)
	      if (DEBUG_SVC(m))
		  {
		     debug_printf(m,
			    "\n---\nDOS SVC 0x43: GET ATTR\n");    
		     debug_printf(m,"filehandle=%s\n",
			     path );
		     if (rval == SYSCALL_SUCCESS)
		       debug_printf(m,
			       "SUCCESS: attr=0x%x\n", bits);
		     else
		       debug_printf(m,
			       "FAILED: error=%d\n", syscall_errno);
		  }
#endif

	      break;
	      
	    case 1:
	      bits = m->R_CX;

	      rval = doss_fsop_setfattr(m,path,bits);

#if defined(DEBUG)
	      if (DEBUG_SVC(m))
		  {
		     debug_printf(m,
			     "\n---\nDOS SVC 0x43: SET ATTR\n");    
		     debug_printf(m,"filehandle=%s bits=0x%x\n",
			     path,bits);
		     if (rval == SYSCALL_SUCCESS)
		       debug_printf(m,
			       "SUCCESS:\n");
		     else
		       debug_printf(m,
			       "FAILED: error=%d\n", syscall_errno);
		  }
#endif
	      
	      if (rval == SYSCALL_SUCCESS)
		  {
		     CLEAR_FLAG(m,F_CF);
		  }
	      
	      else
		  {
		     m->R_AX = syscall_errno;
		     SET_FLAG(m, F_CF);
		  }
	
	      break;

	    default:
	      
	      SET_FLAG(m,F_CF);
	      m->R_AX = 1;  /* unknown function */
	   }
       
       
    }

/*----------------------------------------------------------------------*/
/*  IOCTL                                                               */
/*----------------------------------------------------------------------*/


static void  DEFUN(doss_svc_tab_44,(m),PC_ENV *m)
    {

       int retval;
       u_int16 bits;
       int fh;
       
       
       switch (m->R_AL)
	   {
	    case 0:   /* character device attributes. */
	      fh = m->R_BX;
	      
	      retval = doss_fsop_ioctl_sf0(m,fh, &bits);

#ifdef DEBUG
	      if (DEBUG_SYS(m))
		  {
		     debug_printf(m,
			     "\n---\nDOS SVC 0x44: IOCTL 0\n");    
		     debug_printf(m,"filehandle=%d\n",
			     fh);

		     if (retval == SYSCALL_SUCCESS)
		       debug_printf(m,
			       "SUCCESS: attr=0x%x\n", bits);
		     else
		       debug_printf(m,
			       "FAILED: error=%d\n", syscall_errno);
		  }
#endif

	      if (retval == SYSCALL_SUCCESS)
		  {
		     CLEAR_FLAG(m, F_CF);
		     m->R_DX = bits;
		  }
	      else
		  {
		     SET_FLAG(m, F_CF);
		     m->R_AX = syscall_errno;
		  }
	      break;
	    default:   /* [JCE] Return an error for unsupported subfunctions */
	      m->R_AX = 1;
	      SET_FLAG(m, F_CF);
	      break;	      
	   }

    }


/*----------------------------------------------------------------------*/
/*  GET CURRENT DIRECTORY                                               */
/*----------------------------------------------------------------------*/


static void  DEFUN(doss_svc_tab_47,(m),PC_ENV *m)
    {
       int retval;
       u_int8 *path;
       int drive;
       
       path = m->mem_base + ((u_int16)m->R_DS<<4) + (u_int16)m->R_SI;

       drive = m->R_DL;
       
#if defined(DEBUG)
    if (DEBUG_SVC(m))
	{
	   debug_printf(m,"\n---\nDOS SVC 0x47:  GET CWD\n");    
	   debug_printf(m,"drive=%d\n",drive);
	}
#endif

       retval = doss_fsop_getcwd(m,drive,path);

#if defined(DEBUG)
    if (DEBUG_SVC(m))
	{
	   debug_printf(m,"dir=%s\n",path);
	   debug_printf(m,"retval=%s\n---\n",
		   retval==SYSCALL_SUCCESS?SUCCESS:FAIL);
	}
#endif

       
       if (retval == SYSCALL_FAILURE)
	   {
	      SET_FLAG(m,F_CF);
	      m->R_AX = syscall_errno;
	   }
       else
	   {
	      CLEAR_FLAG(m,F_CF);
	   }

    }

/*----------------------------------------------------------------------*/
/* ALLOCATE MEMORY                                                      */
/*----------------------------------------------------------------------*/


static void  DEFUN(doss_svc_tab_48,(m),PC_ENV *m)
    {
       int retval;
       u_int16  npara_wanted;
       u_int16  npara_avail;
       struct doss_mcb *new_mcb;
       

       npara_wanted = m->R_BX;
       

       retval = doss_allocate_memory(m, &new_mcb, (u_int32) npara_wanted,
				     &npara_avail);

       
#if defined(DEBUG)
    if (DEBUG_SVC(m))
	{
	   debug_printf(m,"\n---\nDOS SVC 0x48: ALLOCATE MEMORY\n");    
	   if (retval == SYSCALL_SUCCESS)
	     debug_printf(m,"wanted=0x%x avail=0x%x startseg=0x%x\n",
		    npara_wanted,
		    npara_avail,
		    new_mcb -> start_seg + 1);
	   else
	     debug_printf(m,"wanted=0x%x avail=0x%x: alloc failed\n",
		    npara_wanted,
		    npara_avail);
	}
#endif

       if (retval != SYSCALL_SUCCESS) 
	   {
	      m->R_BX = npara_avail;
	      SET_FLAG(m,F_CF);
	      if (syscall_errno == DOSS_ENOMEM)
		  {
		     m->R_AX = 8; /* insufficient memory */
		  }
	      else
		  {
		     m->R_AX = 7; /* allocator is hosed. */
		  }
	   }
       else
	   {
	      CLEAR_FLAG(m, F_CF);
	      m->R_AX = new_mcb->start_seg + 1;
	      m->R_BX = npara_avail;
	      new_mcb->proc_psp = m->doss_curr_proc -> psp_seg;
	   }
    }

/*----------------------------------------------------------------------*/
/*  DEALLOCATE MEMORY                                                   */
/*----------------------------------------------------------------------*/


static void  DEFUN(doss_svc_tab_49,(m),PC_ENV *m)
    {
       int retval;
       u_int16  mem_segaddr;
       

       mem_segaddr = m->R_ES;
       
       retval = doss_deallocate_memory(m, mem_segaddr);
       
#if defined(DEBUG)
    if (DEBUG_SVC(m))
	{
	   debug_printf(m,"\n---\nDOS SVC 0x49: DEALLOCATE MEMORY\n");    
	   debug_printf(m,"startseg=0x%x\n",mem_segaddr);
	}
#endif

       if (retval != SYSCALL_SUCCESS) 
	   {
	      SET_FLAG(m,F_CF);
	      if (syscall_errno == DOSS_EBADALLOC)
		  {
		     m->R_AX = 9; /* bad segment address*/
		  }
	      else
		  {
		     m->R_AX = 7; /* allocator is hosed. */
		  }
	   }
       else
	   {
	      CLEAR_FLAG(m, F_CF);
	   }

    }

/*----------------------------------------------------------------------*/
/*----------------------------------------------------------------------*/


static void  DEFUN(doss_svc_tab_4a,(m),PC_ENV *m)
    {
       int retval;
       u_int16  npara_wanted;
       u_int16  npara_avail;
       u_int16  mem_segaddr;

       npara_wanted = m->R_BX;

       mem_segaddr = m->R_ES;

       retval = doss_modify_allocation(m, mem_segaddr, npara_wanted,
				     &npara_avail);
      
#ifdef DEBUG
    if (DEBUG_SVC(m))
	{
	   debug_printf(m,
		   "\n---\nDOS SVC 0x4a: MODIFY MEMORY ALLOCATION\n");    
	   debug_printf(m,"wanted=0x%x avail=0x%x startseg=0x%x",
		  npara_wanted,
		  npara_avail,
		  mem_segaddr);
	   if (retval == SYSCALL_SUCCESS)
	     debug_printf(m,"\n");
	   else
	     debug_printf(m," failed error=%d\n",syscall_errno);
	}
#endif

       if (retval != SYSCALL_SUCCESS) 
	   {
	      m->R_BX = npara_avail;
	      SET_FLAG(m,F_CF);
	      if (syscall_errno == DOSS_ENOMEM)
		  {
		     m->R_AX = 8; /* insufficient memory */
		  }
	      else
		  {
		     m->R_AX = 7; /* allocator is hosed. */
		  }
	   }
       else
	   {
	      CLEAR_FLAG(m, F_CF);
	      m->R_BX = npara_avail;
	   }
    }

/*----------------------------------------------------------------------*/
/*    EXEC                                                              */
/*----------------------------------------------------------------------*/


static void  DEFUN(doss_svc_tab_4b,(m),PC_ENV *m)
    {
       u_int8 *pb;
       u_int8 *pname;
       u_int16 eseg;
       u_int16 ctseg,ctoff;
       u_int8 *ctb;
       struct doss_ctail ct;
       int i,rval;

       switch(m->R_AL)
	   {
	    case 0:   /* exec file */
	      
	      /* fetch the address of the program name. */
	      pname = (m->mem_base + 
		       ((u_int16)m->R_DS<<4) + ((u_int16)m->R_DX));
       
	      /* fetch the parameter block */
	      pb = (m->mem_base + 
	     ((u_int16)m->R_ES<<4) + ((u_int16)m->R_BX));

	      /* environment segment*/
	      FETCH_WORD(eseg, pb, 0);

	      /* command arguments. */
	      FETCH_WORD(ctoff, pb, 2);
	      FETCH_WORD(ctseg, pb, 4);
	      ctb = (m->mem_base + 
	      ((u_int16)ctseg<<4) + ((u_int16)ctoff));
       
	      ct.len = *ctb;
	      for(i=0; i<127; i++) 
		ct.cmd[i] = ctb[i+1];

#ifdef DEBUG
	      if (DEBUG_SVC(m))
		  {
		     debug_printf(m,
			     "\n---\nDOS SVC 0x4b: EXEC FILE\n");    
		     debug_printf(m,
			     "cmd='%s %*s\n",
			     pname,
			     ct.len,
			     ct.cmd);
		  }
#endif

	      rval = doss_exec(m,pname,eseg,&ct);
	      
#ifdef DEBUG
	      if (DEBUG_SVC(m))
		  {
		     if (rval == SYSCALL_SUCCESS)
		       debug_printf(m,"\n");
		     else
		       debug_printf(m," failed\n");
		  }

#endif
	    default:
#ifdef DEBUG
	      if (DEBUG_SVC(m))
		  {
		     debug_printf(m,
			     "\n---\nDOS SVC 0x4b: EXEC FILE\n");    
		  }
#endif
	      SET_FLAG(m, F_CF);
	   }
       
    }

/*----------------------------------------------------------------------*/
/*   terminate                                                          */
/*----------------------------------------------------------------------*/


static void  DEFUN(doss_svc_tab_4c,(m),PC_ENV *m)
    {

#ifdef DEBUG
       if (DEBUG_SVC(m))
	   {
	      debug_printf(m,
		      "\n---\nDOS SVC 0x4c: EXIT\n");    
	   }
#endif

       doss_exit(m,m->R_AL);

    }


/*----------------------------------------------------------------------*/
/*--FIND FIRST (DIRREAD)------------------------------------------------*/
/*----------------------------------------------------------------------*/

static void  DEFUN(doss_svc_tab_4e,(m),PC_ENV *m)
    {
       int attr;
       u_int8 *p;
       int retval;
       
       attr = m->R_CX;
       p = m->mem_base + ((u_int16) m->R_DS << 4) + ((u_int16)m->R_DX);

#ifdef DEBUG
       if (DEBUG_SVC(m))
	   {
	      debug_printf(m,
		      "\n---\nDOS SVC 0x4e: FINDFIRST\n");    
	      debug_printf(m,"filename='%s' attr=0x%x\n",p,attr);
	   }
#endif

       retval = doss_fsop_dirread(m, p, attr, 1);
       
#ifdef DEBUG
       if (DEBUG_SVC(m))
	   {
	   if (retval == SYSCALL_SUCCESS)
	     debug_printf(m,"\n");
	   else
	     debug_printf(m," failed with error %d\n",syscall_errno);
	}
#endif

       if (retval == SYSCALL_SUCCESS)
	   {
	      CLEAR_FLAG(m,F_CF);
	   }
       else
	   {
	      m->R_AX  = syscall_errno;
	      SET_FLAG(m,F_CF);
	   }
    }

/*----------------------------------------------------------------------*/
/*--FIND NEXT  (DIRREAD)------------------------------------------------*/
/*----------------------------------------------------------------------*/

static void  DEFUN(doss_svc_tab_4f,(m),PC_ENV *m)
 {

       int retval;
       
       /* note that the second and third args are unused. */
       retval = doss_fsop_dirread(m, (char*)0, 0, 0);

#ifdef DEBUG
    if (DEBUG_SVC(m))
	{
	   debug_printf(m,
		   "\n---\nDOS SVC 0x4f: FINDNEXT\n");    
	   if (retval == SYSCALL_SUCCESS)
	     debug_printf(m,"\n");
	   else
	     debug_printf(m," failed with error %d\n",syscall_errno);
	}
#endif

       if (retval == SYSCALL_SUCCESS)
	   {
	      CLEAR_FLAG(m,F_CF);
	   }
       else
	   {
	      m->R_AX  = syscall_errno;
	      SET_FLAG(m,F_CF);
	   }
       
    }


static void  DEFUN(doss_svc_tab_57,(m),PC_ENV *m)
    {
       u_int16  mtime,mdate;
       int rval;
       int fh;
       


       fh = m->R_BX;

       switch (m->R_AL)
	   {
	    case 0:
#if defined(DEBUG)
	      if (DEBUG_SVC(m))
		  {
		     debug_printf(m,
			     "\n---\nDOS SVC 0x57: GET FILE TIME\n");    
		  }
#endif
	
	      rval = doss_fsop_getftime(m,fh,&mdate,&mtime);
	      
	      if (rval == SYSCALL_SUCCESS)
		  {
		     m->R_CX = mtime;
		     m->R_DX = mdate;
		     CLEAR_FLAG(m,F_CF);
		  }
	      
	      else
		  {
		     m->R_AX = syscall_errno;
		     SET_FLAG(m, F_CF);
		  }
	
	      break;
	      
	    case 1:
#if defined(DEBUG)
	      if (DEBUG_SVC(m))
		  {
		     debug_printf(m,
			     "\n---\nDOS SVC 0x57: SET FILE TIME\n");    
		  }
#endif
	      mdate = m->R_DX;
	      mtime = m->R_CX;

	      rval = doss_fsop_setftime(m,fh,mdate,mtime);
	      
	      if (rval == SYSCALL_SUCCESS)
		  {
		     CLEAR_FLAG(m,F_CF);
		  }
	      
	      else
		  {
		     m->R_AX = syscall_errno;
		     SET_FLAG(m, F_CF);
		  }
	
	      break;

	    default:
	      
	      SET_FLAG(m,F_CF);
	      m->R_AX = 1;  /* unknown function */
	   }
       

    }

/*----------------------------------------------------------------------*/
/*  GET/SET MEMORY ALLOCATION STRATEGY                                  */
/*----------------------------------------------------------------------*/

static void  DEFUN(doss_svc_tab_58,(m),PC_ENV *m)
    {

       switch(m->R_AL)
	   {
	      
	    case 0: /* get allocation strategy */
	      CLEAR_FLAG(m,F_CF);
	      m->R_AX = m->mem_alloc_strategy;
	      break;
	      
	    case 1: /* set allocation strategy. */
	      if (m->R_AX < 3) /* check that it is 0,1,2 */
		  {
		     CLEAR_FLAG(m,F_CF);
		     m->mem_alloc_strategy = m->R_AX;
		  }
	      else                             /* bad strategy */
		  {
		     SET_FLAG(m,F_CF);
		     m->R_AX = 1; /*invalid function, or something like that */
		  }
	      break;
	    default:
	      SET_FLAG(m,F_CF);
	      m->R_AX = 1;
	      break;
	   }
    }

static void  DEFUN(doss_svc_tab_62,(m),PC_ENV *m)
    {

#if defined(DEBUG)
    if (DEBUG_SVC(m))
	{
	   debug_printf(m,"\n---\nDOS SVC 0x62: GET PSP\n");    
	   debug_printf(m,"psp=%x\n", m->doss_curr_proc->psp_seg);
	}
#endif
       m->R_BX = m->doss_curr_proc->psp_seg;
    }



static void  DEFUN(doss_svc_tab_66,(m),PC_ENV *m)
{
    switch (m->R_AL)
    {
	case 1:
#if defined(DEBUG)
	    if (DEBUG_SVC(m))
	    {
		debug_printf(m,"\n---\nDOS SVC 0x66: GET CODEPAGE\n");    
		debug_printf(m,"active page=%d  system page=%d\n",
			m->cpage_active, m->cpage_system);
	    }
#endif
	    m->R_BX = m->cpage_active;
	    m->R_DX = m->cpage_system;
	    CLEAR_FLAG(m, F_CF);
	    break;

	case 2:
#if defined(DEBUG)
	    if (DEBUG_SVC(m))
	    {
		debug_printf(m,"\n---\nDOS SVC 0x66: SET CODEPAGE\n");    
		debug_printf(m,"active page=%d  system page=%d\n",
			m->R_BX, m->R_DX);
	    }
	    m->cpage_active = m->R_BX;
	    m->cpage_system = m->R_DX;
	    CLEAR_FLAG(m, F_CF);
	    break;
#endif
	default:
	    SET_FLAG(m, F_CF);	/* Unsupported subfunction */
	    break;
    }
}

static void  DEFUN(undefined_doss_svc,(m), PC_ENV *m)
 {
   u_int16 svc;
   svc = m->R_AH;
   debug_printf(m,"Undefined dos service 0x%02x\n",svc);
   dump_regs(m);
   halt_sys(m);
 }





/* THE TABLE */

#ifdef __STDC__
void  (*doss_svc[])(PC_ENV *m) =
#else
void  (*doss_svc[])() =
#endif
{
/*00*/	doss_svc_tab_00,    /* exit process       */
/*01*/	doss_svc_tab_01,    /* read character from fh#0 with echo */
/*02*/	doss_svc_tab_02,    /* write character to console or fh#1 */
/*03*/	undefined_doss_svc, 
/*04*/	undefined_doss_svc,
/*05*/	doss_svc_tab_05,    /* write character to printer */
/*06*/	doss_svc_tab_06,    /* console I/O */
/*07*/	doss_svc_tab_07,    /* unfiltered input, no echo */
/*08*/	doss_svc_tab_08,    /* filtered input, no echo */
/*09*/	doss_svc_tab_09,    /* string output to console or fh#1 */
/*0a*/	undefined_doss_svc, 
/*0b*/	doss_svc_tab_0b,    /* hardcoded no-input flag. */
/*0c*/	undefined_doss_svc,
/*0d*/	undefined_doss_svc,
/*0e*/	doss_svc_tab_0e,    /* set default drive (doss_svc_tab)*/
/*0f*/ 	undefined_doss_svc,
/*10*/	undefined_doss_svc,
/*11*/	undefined_doss_svc,
/*12*/	undefined_doss_svc,
/*13*/	undefined_doss_svc,
/*14*/	undefined_doss_svc,
/*15*/	undefined_doss_svc,
/*16*/	undefined_doss_svc,
/*17*/	undefined_doss_svc,
/*18*/	undefined_doss_svc,
/*19*/	doss_svc_tab_19,    /* get default drive (doss_svc_tab) */
/*1a*/	doss_svc_tab_1a,    /* set dta address   (doss_svc_tab) */
/*1b*/	undefined_doss_svc,
/*1c*/	undefined_doss_svc,
/*1d*/	undefined_doss_svc,
/*1e*/	undefined_doss_svc,
/*1f*/	undefined_doss_svc,
/*20*/	undefined_doss_svc,
/*21*/	undefined_doss_svc,
/*22*/	undefined_doss_svc,
/*23*/	undefined_doss_svc,
/*24*/	undefined_doss_svc,
/*25*/	doss_svc_tab_25,  /* set interrupt vect. (doss_svc_tab) */
/*26*/	undefined_doss_svc,
/*27*/	undefined_doss_svc,
/*28*/	undefined_doss_svc,
/*29*/	undefined_doss_svc,
/*2a*/	doss_svc_tab_2a,  /* XXX */
/*2b*/	undefined_doss_svc,  
/*2c*/	doss_svc_tab_2c,  /* get system time (doss_time) */
/*2d*/	undefined_doss_svc,
/*2e*/ 	undefined_doss_svc,
/*2f*/	doss_svc_tab_2f,  /* get dta address (doss_svc_tab) */
/*30*/	doss_svc_tab_30,  /* get version number == 3.3 (doss_svc_tab)*/
/*31*/	undefined_doss_svc,
/*32*/	undefined_doss_svc,
/*33*/	doss_svc_tab_33,  /* get/set cntrl-break flag */
/*34*/	undefined_doss_svc,
/*35*/	doss_svc_tab_35,  /* get interrupt vector (doss_svc_tab)*/
/*36*/	undefined_doss_svc,
/*37*/	doss_svc_tab_37,
/*38*/	doss_svc_tab_38,  /* get/set country      .       */
/*39*/	doss_svc_tab_39,  /* create dir        (doss_dir) */
/*3a*/	doss_svc_tab_3a,  /* delete dir        (doss_dir) */
/*3b*/	doss_svc_tab_3b,  /* set cwd           (doss_dir)*/
/*3c*/	doss_svc_tab_3c,  /* create file       (doss_file)*/
/*3d*/	doss_svc_tab_3d,  /* open file         (doss_file)*/
/*3e*/	doss_svc_tab_3e,  /* close file        (doss_file)*/
/*3f*/	doss_svc_tab_3f,  /* read file         (doss_file)*/
/*40*/	doss_svc_tab_40,  /* write file        (doss_file)*/
/*41*/	doss_svc_tab_41,  /* delete file       (doss_file)*/
/*42*/	doss_svc_tab_42,  /* seek file         (doss_file)*/
/*43*/	doss_svc_tab_43,  /* get/set attr      (doss_file)*/
/*44*/	doss_svc_tab_44,  /* ioctl      (xxx)  (doss_file)*/
/*45*/	undefined_doss_svc,  /* dup */
/*46*/	undefined_doss_svc,  /* dup2 */
/*47*/	doss_svc_tab_47,  /* get cwd           (doss_dir)*/
/*48*/	doss_svc_tab_48,  /* allocate memory   (doss_mem)*/
/*49*/	doss_svc_tab_49,  /* deallocate memory (doss_mem)*/
/*4a*/	doss_svc_tab_4a,  /* reallocate memory (doss_mem)*/
/*4b*/	doss_svc_tab_4b,  /* exec process      (doss_proc) */
/*4c*/	doss_svc_tab_4c,  /* exit process      (doss_proc) */
/*4d*/	undefined_doss_svc,
/*4e*/	doss_svc_tab_4e,  /* findfirst.        (doss_dir)*/
/*4f*/	doss_svc_tab_4f,  /* findnext          (doss_dir)*/
/*50*/	undefined_doss_svc,  /* undefined/undocumented */
/*51*/	undefined_doss_svc,  /* undefined/undocumented */
/*52*/	undefined_doss_svc,  /* undefined/undocumented */
/*53*/	undefined_doss_svc,  /* undefined/undocumented */
/*54*/	undefined_doss_svc,
/*55*/	undefined_doss_svc,
/*56*/	undefined_doss_svc,
/*57*/	doss_svc_tab_57,  /* get/set file time           (doss_file)*/
/*58*/	doss_svc_tab_58,  /* get/set allocation strategy (doss_mem)*/
/*59*/	undefined_doss_svc,
/*5a*/	undefined_doss_svc,
/*5b*/	undefined_doss_svc,
/*5c*/	undefined_doss_svc,
/*5d*/	undefined_doss_svc,
/*5e*/	undefined_doss_svc,
/*5f*/	undefined_doss_svc,
/*60*/	undefined_doss_svc,
/*61*/	undefined_doss_svc,
/*62*/	doss_svc_tab_62,  /* get psp segment (doss_svc_tab)*/
/*63*/	undefined_doss_svc,
/*64*/	undefined_doss_svc,
/*65*/	undefined_doss_svc,
/*66*/	doss_svc_tab_66,  /* get / set system code page */
	};


#endif  /* DOSS_SUPPORT */
