
/* 
 * Dos/PC Emulator
 * Copyright (C) 1991 Jim Hudgens
 * 
 * 
 * The file is part of GDE.
 * 
 * GDE is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 1, or (at your option)
 * any later version.
 * 
 * GDE is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GDE; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.  
 *
 */

#ifdef DOSS_SUPPORT
static char rcsid[]=
  "$Id: doss_proc.c,v 1.4 1991/07/30 01:55:11 hudgens Exp hudgens $";

/* $Log: doss_proc.c,v $
 * Revision 1.4  1991/07/30  01:55:11  hudgens
 * fixed relocation problem.  Added copyright.
 *
 * Revision 1.3  1991/07/21  04:50:25  hudgens
 * attempted to fix relocation problem in the relocation code,
 * which is causing tasm to coredump the emulator.
 *
 * Revision 1.2  1991/07/20  16:49:58  hudgens
 * fixed bug with missing \r on the command tail in PSP.
 *
 * Revision 1.1  1991/07/20  04:07:30  hudgens
 * Initial revision
 *
 *
 */


#include "gde.h"

void DEFUN(doss_proc_init,(m), PC_ENV *m)
 {

    int i;
    u_int16 tot_env_space=0;
    u_int16 master_env_seg;
    u_int16 avail_space;
    int err;
    struct doss_mcb *env_mcb;
    u_int8 *p;
    
    
#ifdef DEBUG
    if (DEBUG_PROC(m))
	{
	   debug_printf(m,"DOSS PROC INITIALIZING\n");
	}
#endif

    m->doss_curr_proc = (struct doss_proc *) 
      malloc(sizeof(struct doss_proc));
    if (m->doss_curr_proc == NULL)
	{
	  sys_fatal(errno,"doss_proc_init: malloc:");
	}
#ifdef DEBUG
    if (DEBUG_PROC(m))
	{
	   debug_printf(m,"Init: doss_curr_proc is now %p\n", m->doss_curr_proc);
	}
#endif
 
    m->doss_curr_proc->parent_proc = NULL;
    m->doss_curr_proc->child_proc = NULL;
    m->doss_curr_proc->saved_CS = 0;
    m->doss_curr_proc->saved_IP = 0;    
    m->doss_curr_proc->psp_seg = 0;

    /*set up the master environment*/
 
    for (i=0; m->doss_envp[i] != NULL; i++)
	{
	   tot_env_space += strlen(m->doss_envp[i]) + sizeof('\0');
	}
    
    tot_env_space += sizeof('\0');  /* final terminating NULL. */
    
    tot_env_space = ((tot_env_space+16)/16);  /* convert to "clicks"*/
    
    err = doss_allocate_memory(m,
			       &env_mcb,
			       tot_env_space,
			       &avail_space);

    env_mcb->proc_psp = 0;

    if (err != SYSCALL_SUCCESS)
	{
	   sys_fatal(0,
		     "memory allocation for master environment failed\n");
	}
    
    
    /*  now have the segment for environment */
    m->doss_curr_proc->env_seg = master_env_seg = env_mcb->start_seg+1;
    
    p = m->mem_base + (master_env_seg<<4);


#ifdef DEBUG
    if (DEBUG_PROC(m))
	{
	   debug_printf(m,"master environment at %04x:%04x, ",
		  master_env_seg,0);
	   debug_printf(m,"length = %x para\n",
		  tot_env_space);
	}
    
#endif
    
    for (i=0; m->doss_envp[i] != NULL; i++)
	{
	   strcat(p,m->doss_envp[i]);
	   p = p + strlen(m->doss_envp[i]) + 1;
	}
    p[0] = '\0';  /* final null. */
    p[1] = '\0';  /* final null. */

#ifdef DEBUG
    /* print out the newly created environment */
    if (DEBUG_PROC(m))
	{
	   p = m->mem_base + (master_env_seg<<4);
	   i=0;
	   debug_printf(m,"ENV: ");
	   for (;;p += 1)
	       {
		  if (p[0] == '\0' && p[1] == '\0') break;
		  else if (p[0] == '\0') debug_printf(m,"\nENV: ");
		  else debug_printf(m,"%c",*p);
	       }
	   debug_printf(m,"\n\n");
	}
    
#endif

 }
    


int DEFUN(doss_env_create,(m,parent_seg, new_env_segaddr),
	  PC_ENV *m AND
	  u_int16 parent_seg AND
	  u_int16 *new_env_segaddr)
 {

    u_int8 *p,*q;
    int env_seg_size;
    u_int16 new_seg;
    int i;
    struct doss_mcb *env_mcb;
    u_int16 avail_space;
    int err;
    

    /* figure out how much memory resides in the parent environment */
    /* terminating condition in MS/PCDOS is apparently the appearance
       of consecutive NULL chars. */

    q = p = m->mem_base + (parent_seg << 4);
    env_seg_size=0;
    for (i=0; env_seg_size < 32767 ; p += 1, env_seg_size += 1)
	{
	   if (p[0] == '\0' && p[1] == '\0') break;
	}
    env_seg_size += 1;  /* catch the final null */
    
    env_seg_size = ((env_seg_size+16)/16);  /* convert to paragraphs*/
    
    err = doss_allocate_memory(m,
			       &env_mcb,
			       env_seg_size,
			       &avail_space);
   
    if (err != SYSCALL_SUCCESS)
	{
	   debug_printf(m,"memory allocation for environment failed\n");
	   return SYSCALL_FAILURE;
	}

    *new_env_segaddr = new_seg =  env_mcb->start_seg + 1;
    
#ifdef DEBUG
    if (DEBUG_PROC(m))
	{
	   debug_printf(m,"new environment at %04x:%04x length %x\n", 
		  new_seg,0,env_seg_size);
	}
#endif

    /* now duplicate the old segment into the new segment.
       q still points to  the parent environment memory */

    p = m->mem_base + (new_seg << 4);
    for (i=0; ; p += 1, q += 1)
	{
	   if (q[0] == '\0' && q[1] == '\0') break;
	   *p = *q;
	}

    p[1] = q[1];  /* final null, again */
    
#ifdef DEBUG
    /* print out the newly created environment */
    if (DEBUG_PROC(m))
	{
	   p = m->mem_base + (new_seg<<4);
	   i=0;
	   debug_printf(m,"ENV: ");
	   for (;;p += 1)
	       {
		  if (p[0] == '\0' && p[1] == '\0') break;
		  else if (p[0] == '\0') debug_printf(m,"\nENV: ");
		  else debug_printf(m,"%c",*p);
	       }
	   debug_printf(m,"\n\n");
	}
#endif
    return SYSCALL_SUCCESS;
    
   }




void DEFUN(doss_psp_create,(m,p,size,ctail),
	  PC_ENV *m AND
	  struct doss_proc *p AND
	  int size AND
	  struct doss_ctail *ctail)
 {
    
    /* all memory for the process has already been allocated.
       and starts at p->psp_seg*16 and goes for p->proc_sz*16
       bytes. */
    
    u_int8 *psp_addr = m->mem_base + (p->psp_seg << 4) ;
    
    int i;

    /* clear the memory allocated to the process; */
    for (i=0; i<256; i++) psp_addr[i] = 0;
    
    psp_addr[0] = 0xcd;
    psp_addr[1] = 0x20;  /* int 20, hard coded in for first two 
			    bytes of the psp. */

    STORE_WORD(psp_addr,2,size);

    /* [JCE] Add the CP/M "call 0x05" (far call to F01D:FEF0 => 000C0) 
     *       and the DOS  "call 0x50" (int 21 and return) entry points */
    psp_addr[5] = 0x9A;
    STORE_WORD(psp_addr,6,0xFEF0);
    STORE_WORD(psp_addr,8,0xF01D);
    
    psp_addr[0x50] = 0xCD;	/* INT 21 */
    psp_addr[0x51] = 0x21;
    psp_addr[0x52] = 0xCB;	/* RET */

    /* skip over the break, terminate, and critical error
       vectors for now... */
    
    /* copy in pointer to the environment segment. */

    psp_addr[0x2c] = p->env_seg & 0xff;
    psp_addr[0x2d] = (p->env_seg >> 8) & 0xff;
   
    /* now copy in the command tail. */

    psp_addr[0x80] = ctail->len;
    for (i=0; i < ctail->len; i++) 
      psp_addr[0x81+i] = ctail->cmd[i];
    psp_addr[0x81+i] = 0xd;
    
    
    /* initialize the psp file table. */
    for (i=0; i<20; i++)
	{
	   psp_addr[0x18+i] = 0xff;
	   p->fh[i] = 0xff;
	}
    
    psp_addr[0x18] = p->fh[0] = 0;  /* con */
    psp_addr[0x19] = p->fh[1] = 0;
    psp_addr[0x1a] = p->fh[2] = 0;
    psp_addr[0x1b] = p->fh[3] = 1;  /* aux */   
    psp_addr[0x1c] = p->fh[4] = 2;  /* prn */
 }
    

int DEFUN(doss_get_next_ufh,(m),
	  PC_ENV *m)
 {
    int i;
    struct doss_proc *p = m->doss_curr_proc;
    u_int8 *psp_addr = m->mem_base + (p->psp_seg << 4);

    /* search the psp file table. */
    for (i=0; i<20; i++)
	{
	   if (psp_addr[0x18+i] == 0xff && p->fh[i] == 0xff)
	     return i;
	}
    return -1;
 }
    
int DEFUN(doss_set_fh,(m,ufh,sfh),
	  PC_ENV *m AND
	  int  ufh  AND
	  int sfh)
 {
    struct doss_proc *p = m->doss_curr_proc;
    u_int8 *psp_addr = m->mem_base + (p->psp_seg << 4) ;

    if (ufh >= 0 && ufh < 20) 
	{
	   m->doss_curr_proc->fh[ufh]=sfh;
	   psp_addr[0x18+ufh] = sfh;
	   return 0;
	}
    else
	{
	   return 1;
	}
 }
    
int DEFUN(doss_ufh_to_sfh,(m,ufh),
	  PC_ENV *m AND
	  int  ufh)
 {
    if (ufh >= 0 && ufh < 20) 
	{
	   return m->doss_curr_proc->fh[ufh];
	}
    else
	{
	   return 0xff;
	}
     
 }

  
int DEFUN(doss_load_com,(m,fh,p),
	  PC_ENV *m AND
	  int fh AND
	  struct doss_proc *p)
 {
    int total_size;
    u_int16 seg;
    u_int8 *load_address;
    int  n;
    int err;
    int nread;
    int available_size;
    
    available_size = p->proc_sz * 16;  /* proc_sz is in paragraphs. */
    total_size=0;
    n = 0;
    seg = p->psp_seg;
    load_address = (u_int8*)( m->mem_base + 0x100 + (p->psp_seg << 4));

/*    decode_printf(m,"base address=%x  load_address=%x\n",
	   m->mem_base,
	   load_address);
*/
    
    
    do
	{
	   err = doss_fsop_fread(m,fh,load_address,1024, &nread);
	   if (err == SYSCALL_FAILURE)
	       {
		  return err;
	       }
	   else if (nread == 0) 
	       {
		  break;
	       }
	   else 
	       {
		  total_size += nread;
		  load_address += nread;
		  n += 1;
		  if (total_size > available_size)
		      {
			 return SYSCALL_FAILURE;
		      }
#ifdef DEBUG
		  if (DEBUG_PROC(m))
		      {
			 decode_printf(m,".");
			 if (n % 70 == 69) decode_printf(m,"\n");
		      }
#endif
	       }
	} while (1);


#ifdef DEBUG
    if (DEBUG_PROC(m)) 
      debug_printf(m,"loaded %x bytes at location %04x:%04x\n", 
	     total_size, seg, 0x100 );
#endif

    /* initialize registers */
    CLEARALL_FLAG(m);
    m->R_AX = 0;    
    m->R_BX = 0;
    m->R_CX = total_size;  /* CX gets size of program  loaded */
    m->R_DX = 0;
    m->R_IP = 0x100;       /* IP gets first address past PSP */
    m->R_CS = seg;         /* seg regs get PSP segment address */
    m->R_DS = seg;
    m->R_SS = seg;
    m->R_ES = seg;
    m->R_SI = 0;           /* these registers get 0 */
    m->R_DI = 0;
    m->R_BP = 0;
    m->R_SP = 0xfffe;      /* set to end of segment, after pushing
			      one word of zeros (below).  */

    
    /* put word of zeros at stack pointer location. */

    load_address = (u_int8*)( m->mem_base + (seg<<4) + 0xfffe);
    load_address[0] = 0;
    load_address[1] = 0;

    return SYSCALL_SUCCESS;
 }
    


/* [JCE] Real DOS does not care about the file extension 

   return 0 if not executable 
   return 1 if a com file;
   return 2 if is an exe file;

int DEFUN(executable_file_type,(s),
	  char *s)
 {

    char *p = rindex(s,'.');
    char ext[4];
    int i;

    // no extension 
    if (p == NULL) return DOSS_OTH_FILE_TYPE;

    p += 1;
    // p now points to a "com", "exe", or other extension 

    strncpy(ext,p,3);
    ext[3]='\0';
    for (i=0; i<3; i++) if (islower(ext[i])) ext[i] = toupper(ext[i]);

    if (strcmp(ext,"EXE") == 0) return DOSS_EXE_FILE_TYPE;
    
    else if (strcmp(ext,"COM") == 0) return DOSS_COM_FILE_TYPE;
    
    else return DOSS_OTH_FILE_TYPE;
 }
*/
    
	 
    
int DEFUN(doss_fetch_exehdr,(m,fh,exehdr),
	  PC_ENV *m AND
	  int fh AND
	  struct doss_exec_header *exehdr)
 {
    u_int8  buffer[512];
    int err;
    int nread;
    
    err = doss_fsop_fread(m,fh,buffer,512, &nread);

    if (err == SYSCALL_FAILURE || nread < 512)
	{
	   /* former could easily happen,
	      latter indicates that the executable is 
	      hosed.  */
	   return SYSCALL_FAILURE;
	}
    
    exehdr->signature[0] = buffer[0];
    exehdr->signature[1] = buffer[1];    
    FETCH_WORD(exehdr->length_rem, buffer, 2);
    FETCH_WORD(exehdr->length_pag, buffer, 4);
    FETCH_WORD(exehdr->n_reloc_entries, buffer, 6);
    FETCH_WORD(exehdr->header_size_pg, buffer,8);
    FETCH_WORD(exehdr->min_pg_needed, buffer, 10);
    FETCH_WORD(exehdr->max_pg_needed, buffer, 12);
    FETCH_WORD(exehdr->initial_ss, buffer, 14);
    FETCH_WORD(exehdr->initial_sp, buffer, 16);
    FETCH_WORD(exehdr->checksum, buffer, 18);
    FETCH_WORD(exehdr->initial_ip, buffer, 20);
    FETCH_WORD(exehdr->initial_cs, buffer, 22);
    FETCH_WORD(exehdr->reloc_offset, buffer, 24);
    FETCH_WORD(exehdr->overlay_num, buffer, 26);

    return SYSCALL_SUCCESS;
    
 }
    
   
int DEFUN(doss_load_exec_body,
	  (m,fh,esize,estart_off,load_addr),
	  PC_ENV *m AND
	  int fh AND
	  int esize AND
	  int estart_off AND
	  u_int8 *load_addr)
 {

    int nread=0;
    int err,n;
    int32 at;	/* [JCE] doss_fsop_fseek takes an int32 pointer */
    
    u_int8 *p = load_addr;

    /* seek to the beginning of the executable code in the 
       opened executable file... */

    err = doss_fsop_fseek(m,fh,0,estart_off, &at);
    if (err == SYSCALL_FAILURE)
	{
	   return SYSCALL_FAILURE;
	}

    /* read it in, block by block... */

    while (nread < esize)
	{
	   err = doss_fsop_fread(m,fh,p,512, &n);
	   if (err == SYSCALL_FAILURE)
	       {
		  return SYSCALL_FAILURE;
	       }

	   if (n < 512 && n + nread < esize)
	       {
		  debug_printf(m,"major botch in doss_load_exe_body\n");
		  debug_printf(m,"short read: esize=%d nread=%d n=%d\n",
			  esize,nread,n);
		  return SYSCALL_FAILURE;
	       }

	   nread += n;
	   p += n;
	}

#ifdef DEBUG
    if (DEBUG_PROC(m))
      debug_printf(m,"exec_load_body: bytes loaded=%d\n",nread);
#endif

    return SYSCALL_SUCCESS;
    
 }   


int DEFUN(doss_reloc_exe,(m,fh,exehdr, loadbase,segbase),
	  PC_ENV *m AND
	  int fh AND
	  struct doss_exec_header *exehdr AND
	  u_int16 loadbase AND
	  u_int16 segbase )
 {
    int n;
    int err;
    u_int8  buff[512];
    u_int16  roff,rseg,rval,oval;
    int   reloc_entry_off;
    int   nreloc;
    u_int32  at;	/* [JCE] int32 may not be the same as int */

    u_int8 *load_addr  =   (m->mem_base + (loadbase<<4) + 0x100);

    /* at this point, the file has already been loaded into memory.
       We now go back and do a linkage edit of the executable, based
       on the exec_header link table *and* the loaded address */

    /* read back in the header and start of reloc table. */   
    err = doss_fsop_fseek(m,fh,0,0, &at);
    if (err == SYSCALL_FAILURE)
	{
	   return SYSCALL_FAILURE;
	}

    /*read in the first block. */
    err = doss_fsop_fread(m,fh,buff,512, &n);
    if (err == SYSCALL_FAILURE)
	{
	   return SYSCALL_FAILURE;
	}
    if (n != 512)
	{
	   return SYSCALL_FAILURE;
	}

    reloc_entry_off = exehdr->reloc_offset;

    for (nreloc = 0; nreloc  < exehdr -> n_reloc_entries; nreloc++)
	{
	   
	   if (reloc_entry_off == 512) 
	       {
		  err = doss_fsop_fread(m,fh,buff,512, &n);
		  if (err == SYSCALL_FAILURE)
		      {
			 return SYSCALL_FAILURE;
		      }
		  if (n != 512)
		      {
			 return SYSCALL_FAILURE;
		      }
		  reloc_entry_off = 0;
	       }

	   /* get the relocation position from the exe header buff q. */
	   FETCH_WORD(roff, buff, (reloc_entry_off));
	   reloc_entry_off += 2;

	   if (reloc_entry_off == 512) 
	       {
		  err = doss_fsop_fread(m,fh,buff,512, &n);
		  if (err == SYSCALL_FAILURE)
		      {
			 return SYSCALL_FAILURE;
		      }
		  if (n != 512)
		      {
			 return SYSCALL_FAILURE;
		      }
		  reloc_entry_off = 0;
	       }
	   
	   FETCH_WORD(rseg, buff, (reloc_entry_off));

	   reloc_entry_off += 2;

	   /* fetch the word to be relocated from the memory image p  */
	   FETCH_WORD(oval, load_addr, (roff + (rseg<<4)));
	   rval = (oval) + segbase;
	   STORE_WORD(load_addr, (roff+(rseg<<4)), rval);

#ifdef DEBUG
 	     if (DEBUG_PROC(m))
               {
	         debug_printf(m,
			 "#%04x: offset=%x relocated %04x:%04x  %x -> %x\n",
			 nreloc,reloc_entry_off,rseg,roff,oval,rval);
               }
#endif

	} /* for */

    return SYSCALL_SUCCESS;
 }
    
    
int DEFUN(doss_load_exe,(m,fh,p),
	  PC_ENV *m AND
	  int fh AND
	  struct doss_proc *p)
 {
    
    struct  doss_exec_header  exec_header;
    int     size;
    int     start;
    u_int16     offset;
    u_int16     seg;
    u_int16     segbase;
    u_int8   *load_addr;
    

    if (doss_fetch_exehdr(m,fh,&exec_header) == SYSCALL_FAILURE)
      {
	 return SYSCALL_FAILURE;
      }

    /* start of the load module */
    start =  exec_header.header_size_pg*16;

    /* size of the load module  XXXXX */
    if (exec_header.length_rem ==  0) 
      size =   exec_header.length_pag*512 - start;
    else
      size =  (exec_header.length_pag-1)*512 + exec_header.length_rem - start;

    seg = p->psp_seg;
    offset = 0x100;  /* at end of PSP */

    /* this is what everything is going to be relocated against */
    segbase = seg + (offset>>4);

    load_addr = (u_int8*)(m->mem_base + (seg<<4) + offset);  
    
#ifdef DEBUG
    if (DEBUG_PROC(m))
	{
	   debug_printf(m,"\nLoading EXE file location %04x:%04x\n",
		   seg,0);
	   debug_printf(m,"SIGNATURE:  %2.2s\n", exec_header.signature);
	   debug_printf(m,"SIZE:   header=%d load=%d total=%d\n",
		  start,size,start+size);
	   debug_printf(m,"INITIAL CS:IP:  %04x:%04x\n",
		  exec_header.initial_cs,
		  exec_header.initial_ip);
	   debug_printf(m,"INITIAL SS:SP:  %04x:%04x\n",
		  exec_header.initial_ss,
		  exec_header.initial_sp);
	   debug_printf(m,"ALLOCATION:  MAX=0x%x MIN=0x%x bytes\n",
		  exec_header.max_pg_needed*16,
		  exec_header.min_pg_needed*16);
	   debug_printf(m,"RELOCATION:  0x%x\n",segbase);
	}
#endif


    if (doss_load_exec_body(m,fh,size,start,load_addr) == SYSCALL_FAILURE)
	{
	   return SYSCALL_FAILURE;
	}
	 
    if (doss_reloc_exe(m, fh, &exec_header,seg, segbase) == SYSCALL_FAILURE)
	{
	   return SYSCALL_FAILURE;
	}
	   
    CLEARALL_FLAG(m);
    m->R_AX = 0;
    m->R_BX = 0;
    m->R_CX = size;
    m->R_DX = 0;
    
    m->R_IP = exec_header.initial_ip;
    m->R_CS = exec_header.initial_cs + segbase;
    m->R_DS = seg;
    m->R_SS = exec_header.initial_ss + segbase;
    m->R_ES = seg;
    m->R_SI = 0;
    m->R_DI = 0;
    m->R_BP = 0;
    m->R_SP = exec_header.initial_sp;
/*    push_word(r,m,0); */
    
    return SYSCALL_SUCCESS;
    
 }
    




int DEFUN(doss_exec,(m,name,p_env_seg,ctail),
	  PC_ENV *m AND
	  char *name AND
	  u_int16 p_env_seg AND
	  struct doss_ctail *ctail)  /* argv, argc, etc. */
 {
    int ftype;
    struct doss_proc *newp,*currp;
    struct doss_mcb *psp_mcb;
    u_int16 parenv,newenv;
    u_int16 avail_pg, avail_pg2;
    u_int16 newpsp;
    int    newfh, nread, at;
    unsigned char magic[32]; 
    
    
    /* determine the type of the file: com, exe, other. Fail
     immediately if type other... */
/* [JCE] DOS doesn't care about the file type. Check for the MZ signature
  somewhere instead. 
    switch(ftype=executable_file_type(name))
	{
	 case DOSS_OTH_FILE_TYPE:
	   return SYSCALL_FAILURE;
	}
   */ 
    /* allocate a new process slot... */
    newp = (struct doss_proc *)malloc(sizeof(struct doss_proc));
    if (newp==NULL)
	{
	  sys_fatal(errno,"doss_exec: malloc:");
	}

    currp  = m->doss_curr_proc;
    currp -> child_proc = newp;
    currp -> saved_CS = m->R_CS;
    currp -> saved_IP = m->R_IP;
    
    newp->parent_proc = currp;
    newp->child_proc = NULL;

    /* handle parent environment... */    
    parenv = p_env_seg;
    if (doss_env_create(m, parenv, &newenv) != SYSCALL_SUCCESS)
	{
	   currp->child_proc = NULL;
	   free(newp);
	   return SYSCALL_FAILURE;
	   
	}
    newp->env_seg = newenv;
    
    /* allocate all remaining memory */
    /* first call guaranteed to fail. */
    /* note that second call attempts to grab remaining 
       memory "avail_pg" */

    if (doss_allocate_memory(m,&psp_mcb,
			     0xffff, &avail_pg) == SYSCALL_FAILURE && 
	doss_allocate_memory(m,&psp_mcb,
			     avail_pg, &avail_pg2 ) == SYSCALL_FAILURE) 
	{
	   /* can this happen ??? */
	   debug_printf(m,"check logic in doss_exec psp allocation... \n");
	   return SYSCALL_FAILURE;
	}

    
    newpsp = psp_mcb->start_seg + 1;

    /* set ownership !! */
    psp_mcb->proc_psp = newpsp;
    doss_modify_mem_owner(m,newenv, newpsp);  /* environment too */


    newp -> psp_seg = newpsp;

    /* this is really, maximum possible proc size at this point. */

    newp -> proc_sz = psp_mcb -> n_para;

    newp -> dta_seg = newpsp + 0x8;
    newp -> dta_off = 0;
    newp -> dtalst = NULL;
    
    /* fill in the PSP for the process.  */
    /* YYY sillyquestion:
       does the parent's open files get inherited by the 
       child proc??? 
     */
    doss_psp_create(m,newp,
		    psp_mcb->start_seg + psp_mcb->n_para,
		    ctail);

#ifdef DEBUG
    if (DEBUG_PROC(m))
      {
	 debug_printf(m,"PSP dump %x:0 to %x:0100\n",newpsp,newpsp);
	 dump_memory(m,newpsp,0,0x100);
	 debug_printf(m,"-----------------------\n");
      }
#endif    
    /* now the fun begins. 
       
       1. the psp and filetable for this proc is already initialized. 
       above.
       2. now we make the change over to the current proc, by resetting
       m->doss_curr_proc.  This allows us to use the current proc's 
       file table to access the executable.
       3. open the executable using the standard doss primitives.
     */
    
    m->doss_curr_proc = newp;
#ifdef DEBUG
    if (DEBUG_PROC(m))
	{
	   debug_printf(m,"Exec: doss_curr_proc is now %p\n", m->doss_curr_proc);
	}
#endif
    
    if (doss_fsop_fopen(m,name,0,&newfh) == SYSCALL_FAILURE)
	{
	   /* XXX deallocate and cleanup */
	   return SYSCALL_FAILURE;
	}
   
    if (doss_fsop_fread(m, newfh, magic, 32, &nread) == SYSCALL_FAILURE ||
        doss_fsop_fseek(m, newfh, 0, 0, &at) == SYSCALL_FAILURE)
    {
           doss_fsop_fclose(m,newfh);
	   /* XXX deallocate and cleanup */
	   return SYSCALL_FAILURE;
    }
    if (nread < 32)	/* EXE files must have at least a 32-byte header */
    {
	 ftype = DOSS_COM_FILE_TYPE;
    }
    else if (magic[0] == 'M' && magic[1] == 'Z')	/* and start MZ */
    {
	 ftype = DOSS_EXE_FILE_TYPE;
    }
    else if (magic[0] == 'Z' && magic[1] == 'M')	/* or ZM */
    {
	 ftype = DOSS_EXE_FILE_TYPE;
    }
    else
    {
	 ftype = DOSS_COM_FILE_TYPE;
    }

   
 
    switch (ftype)
	{
	 case DOSS_EXE_FILE_TYPE:
	   if (doss_load_exe(m,newfh,newp) == SYSCALL_FAILURE)
	       {
		  /* XXX deallocate and cleanup */
		  return SYSCALL_FAILURE;
	       }
	   /* 
	     XXXX 
	      now readjust memory requirements to match that
	      requested by the process. 
	    */
	   break;	
	 case DOSS_COM_FILE_TYPE:
	   if (doss_load_com(m,newfh,newp) == SYSCALL_FAILURE)
	       {
		  /* XXX deallocate and cleanup */
		  return SYSCALL_FAILURE;
	       }
	   break;
	 default:
	   sys_fatal(0,"major logic error in doss_exec\n");
	}
    
    if (doss_fsop_fclose(m,newfh) == SYSCALL_FAILURE)
	       {
		  /* XXX deallocate and cleanup */
		  /* this would be unexpected, methinks, and 
		     representative of a major problem somewhere. */
		  return SYSCALL_FAILURE;
	       }
    
    /* we're done... */
    
    return SYSCALL_SUCCESS;
}
    



int DEFUN(doss_exit,(m,eval),
    PC_ENV *m AND
    int  eval)
 {
    
    struct doss_proc *newp,*currp;
    u_int16 cpspseg;
#ifdef DEBUG
    if (DEBUG_PROC(m))
	{
	   debug_printf(m,"Exit: Process %p terminating\n", m->doss_curr_proc);
	}
#endif
    
    currp  = m->doss_curr_proc;
    newp = currp -> parent_proc;
    if (newp != NULL)
    {
        newp->child_proc = NULL;
        newp->ret_val = eval;
    }
    cpspseg = currp -> psp_seg;


    /* step 1:
       deallocate all memory allocated for this process. */
    
    doss_psp_deallocate_memory(m,cpspseg);

    /* change our current process pointer */
#ifdef DEBUG
    if (DEBUG_PROC(m))
	{
	   debug_printf(m,"Exit: doss_curr_proc is now %p\n", newp);
	}
#endif
    m->doss_curr_proc = newp;
    free(currp);

    if ( newp == NULL )
	{
	  printf("Exiting...\n");
	  exit(0);
	}
    else 
	{
	   m->R_CS = newp -> saved_CS;
	   m->R_IP = newp -> saved_IP;
	   CLEAR_FLAG(m, F_CF);		/* [JCE] Success: clear Carry */	
	}
    
    return SYSCALL_SUCCESS;
}
    
#endif  /* DOSS_SUPPORT */
