
/* 
 * Dos/PC Emulator
 * Copyright (C) 1991 Jim Hudgens
 * 
 * 
 * The file is part of GDE.
 * 
 * GDE is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 1, or (at your option)
 * any later version.
 * 
 * GDE is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GDE; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.  
 *
 */

#ifdef DOSS_SUPPORT

#include "gde.h"

static char rcsid[]=
  "$Id: doss_mem.c,v 1.4 1991/07/30 01:53:43 hudgens Exp hudgens $";

/* $Log: doss_mem.c,v $
 * Revision 1.4  1991/07/30  01:53:43  hudgens
 * added copyright.
 *
 * Revision 1.3  1991/07/20  17:58:02  hudgens
 * changed modify_allocation to always return the available memory.
 *
 * Revision 1.2  1991/07/17  03:52:44  hudgens
 * added function doss_modify_mem_owner for convenience of the
 * exec routines.
 *
 * Revision 1.1  1991/06/18  02:50:42  hudgens
 * Initial revision
 *
 *
 */

/* 
  ENTRY POINTS IN THIS FILE:
  
  doss_memalloc_init:   called at startup to set up the memory allocation
      scheme
  
  doss_allocate_memory: represents the heart of doss interrupt 0x21
      function 0x48.  

      It is called with the pointer of a pointer to struct doss_mcb,
      which gets the address of new node allocated, so that the 
      allocation information can be determined.

      It is also called with the requested size, and a pointer to 
      an integer which returns the maximum available size, if the 
      allocation fails. 

      The final argument is the PC_ENV variable, which holds information
      about the global doss data structures.

      Note that this does not fill in the proc_psp field  of the 
      the allocated mcb structure.  The caller does this.

      Note also, that this is not *exactly* how msdos does this, 
      but it is close enough that only dos-snooping type programs
      would notice.  In particular, nothing is changed in the 
      memory image that doss sees, so that the allocation
      chain cannot be followed by an emulated doss process.
      The extra 16 bytes that msdos apparently uses is allocated
      anyway, in case the emulation of these functions ever reaches
      that level of detail.  Because of this, the caller to this 
      function should return, as the base segment of the allocated
      memory, start_seg+1, since the +1 represents the 16 bytes
      that msdos would use to describe its allocation.

      It returns an indication of success or failure.


  doss_deallocate_memory: represents the basic parts of doss interupt
      0x21 function 0x49: release specific memory allocation.
      It is passed the PC_ENV varible m, and the segment to 
      be released, and if it locates that segment in the 
      doss mcb chain, releases it.  It returns an indication
      of whether the operation was successful or not.

  doss_psp_deallocate_memory: does not represent a doss system call,
      but instead is a function which would be used by the 
      exit system call, to clean up after a process exits.  This
      call deletes all the memory associated with a process.

  doss_get_allocation_strategy: corresponds to doss interrupt 0x21 
      function 0x58, subfunction 0x0.  It returns an integer in
      the range 0..2, corresponding to the current allocation strategy
      in effect.

  doss_set_allocation_strategy: corresponds to doss interrupt 0x21 
      function 0x58, subfunction 0x1.  It is passed an integer in
      the range 0..2, corresponding to the new allocation strategy
      desired.  It returns an indication of success or failure, and
      sets the current allocation strategy.

  doss_modify_allocation: corresponds to doss interrupt 0x21, 
      function 0x4a, modify allocation.  It is passed the 
      base address of the allocation, and the new desired
      size, as well as a pointer to int which returns the 
      maximum available size which can be allocated if the 
      request fails.  This call returns an indication of success
      or failure. 

  Two debugging routines are included and compiled into the system,
  but are not called: 

  doss_check_memalloc: performs minimal checking on the current 
      mcb chain.

  doss_trace_memory_allocation:  prints the current mcb chain, showing
      current allocations.
  
*/
  



/* initialize the memory system. 
 */


   
void   DEFUN(doss_memalloc_init,(m), PC_ENV *m)
 {
    
    m->mcb_root = (struct doss_mcb *)malloc(sizeof (struct doss_mcb));
    m->mem_alloc_strategy = MEMALLOC_FIRST_FIT;
    
    if(!m->mcb_root) 
	{
	  sys_fatal(errno,"DOSS_memalloc_init: malloc");
	}
    m->mcb_root->next = NULL;
    m->mcb_root->prev = NULL;
    m->mcb_root->start_seg = sys_base;
    m->mcb_root->n_para = 0;    /* no allocation.  */
    m->mcb_root->proc_psp = 0;  /* mister root */
 }
    

    
/* doss_allocate_memory

   THis is an implementation of DOSS interrupt 0x21 function
   0x48.  It takes a PC_ENV pointer to get the current 
   allocation chain, a pointer to a struct doss_mcb for
   returning the allocation info, and a requested size.
   It returns either SYSCALL_SUCCESS or SYSCALL_FAILURE.
   In the latter case, it also sets the global variable
   syscall_errno.
   
   This routine does the allocation based on the global
   allocation_strategy found in PC_ENV.  The choices there
   are MEMALLOC_FIRST_FIT (default), MEMALLOC_BEST_FIT, and 
   MEMALLOC_LAST_FIT.  
*/


int DEFUN(doss_allocate_memory,(m,new,size,avail),
	PC_ENV *m AND struct doss_mcb  **new AND
	u_int16  size  AND   u_int16 *avail )
 {
    struct doss_mcb  *p,*q;
    int max_avail,hole_size;   /* gotta be signed. */
    u_int32 best_fit_yet;
    
    p = m->mcb_root;
    *new = q = (struct doss_mcb*) malloc(sizeof (struct doss_mcb));
    if(!(q))
	{
	  sys_fatal(errno,"DOSS_allocate_memory: malloc");
	}
    q->start_seg = 0;
    q->n_para = 0;
    q->next = NULL;
    q->prev = NULL;
    
    max_avail = 0;
    
    switch(m->mem_alloc_strategy)
	{
	 case MEMALLOC_FIRST_FIT:
	   /* strategy:
	      1) first fit goes down the mcb chain, and compares
	      adjacent elements:
	      hole_size == q->start_seg - (p->n_para+p->start_seg+1) - 1
	      special case when we hit the end of the list.
	   */
	   
	   do 
	       {
		  if (p->next)
		      {
			 hole_size =  p->next->start_seg - 
				       (p->start_seg + 1 + p->n_para) - 1;
		      }
		  else 
		      {
			 hole_size = (m->mem_size >> 4) - 
			   (p->start_seg + 1 + p->n_para) - 1;
		      }
		  
		  
		  if (p->next && hole_size >= (int) size)
		    /* insert between two elements of the chain. */
		      {
			 q->start_seg = p->start_seg + 1 + p->n_para;
			 q->n_para = size;
			 q->next = p->next;
			 q->prev = p;
			 p->next = q;
			 q->next->prev = q;
			 return SYSCALL_SUCCESS;
			 /*NOTREACHED*/
		      }
		  else if (p->next == NULL && hole_size >= (int) size)
		    /* alloc from the end of the chain. */
		      {
			 q->start_seg = p->start_seg + 1 + p->n_para;
			 q->n_para = size;
			 q->prev = p;
			 q->next = NULL;
			 p->next = q;
			 return SYSCALL_SUCCESS;
			 /*NOTREACHED*/
		      }
		  else if (hole_size > max_avail) 
		      {
			 max_avail = hole_size;
		      }

		  p = p->next;
	       }
	   while (p != NULL);
	   *avail = max_avail;
	   free(q);
	   *new=NULL;
	   syscall_errno =  DOSS_ENOMEM;
	   return SYSCALL_FAILURE;
	   /*NOTREACHED*/
	   break;

	 case MEMALLOC_LAST_FIT:
	   /* strategy:
	      last fit goes down the mcb chain, 
	      and then does first fit traversing the 
	      chain in reverse.  It compares
	      adjacent elements:
	      hole_size == q->start_seg - (p->n_para+p->start_seg+1) - 1
	      Note, that the special case when we hit the end of the list.
	      I'm guessing that this is how it is done.
	      Why would anyone use this, I wonder...
	   */
	   
	   while(p->next)   p = p->next;

	   if ((m->mem_size >> 4)-
	       (p->start_seg + 1 + p->n_para) - 1 > (int) size)
	       {
		  /* note that p->next == NULL, here */
		  q->start_seg = (p->start_seg+1+p->n_para);
		  q->n_para = size;
		  q->next=NULL;
		  q->prev=p;
		  p->next=q;
		  return SYSCALL_SUCCESS;
		  /*NOTREACHED*/
	       }

	   do 
	       {
		  if (p->prev)
		      {
			 hole_size =  p->start_seg - 
			   (p->prev->start_seg + 1 + p->prev->n_para) - 1;
		      }
		  
		  if (p->prev && hole_size >= (int)size)
		    /* insert between two elements of the chain. */
		      {
			 q->start_seg = p->prev->start_seg + 1 
			   + p->prev->n_para;
			 q->n_para = size;
			 q->next = p;
			 q->prev = p->prev;
			 p->prev = q;
			 q->prev->next = q;
			 return SYSCALL_SUCCESS;
			 /*NOTREACHED*/
		      }
		  else if (hole_size > max_avail) 
		      {
			 max_avail = hole_size;
		      }

		  p = p->prev;
	       }
	   while (p != NULL);
	   *avail = max_avail;
	   free(q);
	   *new=NULL;
	   syscall_errno =  DOSS_ENOMEM;
	   
	   return SYSCALL_FAILURE;
	   /*NOTREACHED*/
	   break;
	   
	   

	 case MEMALLOC_BEST_FIT:
	   /* strategy:
	      sweep the chain, looking for block which best
	      matches the size wanted.  Then allocate that block.
	      As in the previous two, it compares  adjacent elements:

	      hole_size == q->start_seg - (p->n_para+p->start_seg+1) - 1
	      
	      We save all the info in variable "new", but do not insert
	      it into the chain.

	      1) we will stop when we find an exact fit.
	      2) we will save the address of the hole, and set 
	      the prev and next pointer of new to the current value
	      of p and p->next, whenever we find a better match.
	      The size of this hole will be stored in the variable 
	      best_fit_yet, which is initialized to the maximum 
	      possible size.
	      3) when we hit the end of the chain, the decision 
	      becomes clear, and at that point, we insert this 
	      element into the allocation chain.
	      
	   */

	   best_fit_yet = 0xffff;  /* maximum allocable size for doss */
	   do 
	       {
		  if (p->next)
		      {
			 hole_size =  p->next->start_seg - 
				       (p->start_seg + 1 + p->n_para) - 1;
		      }
		  else 
		      {
			 hole_size = (m->mem_size >> 4) - 
			   (p->start_seg + 1 + p->n_para) - 1;
		      }
		  
		  if (hole_size ==  size)
		      {
			 /* allocate it and exit */
			 q->start_seg = p->start_seg + 1 + p->n_para;
			 q->n_para = size;
			 q->next = p->next;
			 q->prev = p;
			 p->next = q;
			 if (q->next) q->next->prev = q;
			 return SYSCALL_SUCCESS;
			 /*NOTREACHED*/
		      }
		  else if (hole_size > (int)size && best_fit_yet > hole_size)
		      {
			 best_fit_yet = hole_size;
			 q->prev = p;
			 q->next = p->next;
			 q->n_para = size;
		      }
		  
		  if (hole_size > max_avail) 
		      {
			 max_avail = hole_size;
		      }

		  p = p->next;
	       }
	   while (p != NULL);
	   if (max_avail > (int)size)
	       {
		  /* we found a block */
		  /* allocate it and exit */
		  q->start_seg = q->prev->start_seg + 1 +
		                   q->prev->n_para;
		  q->n_para = size;
		  q->prev->next = q;
		  if (q->next) q->next->prev = q;
		  return SYSCALL_SUCCESS;
		  /*NOTREACHED*/
	       }
	   else
	       {
		  *avail = max_avail;
		  *new=NULL;
		  free(q);
		  syscall_errno =  DOSS_ENOMEM;
		  return SYSCALL_FAILURE;
		  /*NOTREACHED*/
	       }
	   
	   break;
	   
	}
    return SYSCALL_FAILURE;  /* NOTREACHED? */
 }
    
	   
/* 
  release specific memory allocation
  direct dropin for INT 0x21, Function 0x49. 
  passed the segment address that was **returned to the client**, and 
  deletes the allocation.
  
  If that segment not found, then it returns an error indication.
 */

int  DEFUN(doss_deallocate_memory,(m,seg),
	   PC_ENV *m AND u_int16 seg )
 {
    struct doss_mcb  *p;

    p = m->mcb_root;
    do 
	{
	   if (p->start_seg + 1 == seg) 
	       {
		  /* never deallocate m->mcb_root, anyway */
		  if (p->prev) p->prev->next = p->next;
		  if (p->next) p->next->prev = p->prev;
		  free(p);
		  return SYSCALL_SUCCESS;
	       }
	   p = p->next;
	}
    while (p != NULL);
    
    syscall_errno=DOSS_EBADALLOC;
    return SYSCALL_FAILURE;
    
 }






/* 
  release process's memory allocation
  called by the exit DOSS function 0x4c.

  passed the psp address that was **of the process**, and 
  deletes all the allocations for that process
 */

int DEFUN(doss_psp_deallocate_memory,(m,psp),
	  PC_ENV *m AND    u_int16 psp )
 {
    struct doss_mcb  *p, *nextp;

    p = m->mcb_root;
    do 
    {
	   nextp = p->next;	/* [JCE] Don't access this after it's freed */
	   if (p->proc_psp  == psp) 
	   {
		  /* never deallocate m->mcb_root, anyway */
		  if (p->prev) p->prev->next = p->next;
		  if (p->next) p->next->prev = p->prev;
		  free(p);
	   }
	   p = nextp;
    }
    while (p != NULL);
    return SYSCALL_SUCCESS;
 }

/* modify a given allocation.  Passed the base address of the 
   allocation in variable seg, and the requested size in 
   newsize.  If successful, SYSCALL_SUCCESS will be returned,
   and avail will be unmodified.  Otherwise, SYSCALL_FAILURE
   will be returned, *avail will be set to the amount of 
   available memory for that allocation (i.e. I'm interpreting
   this as the amount of adjacent UNUSED memory which could
   be merged into the current allocation if the request was 
   for an increase in size), and the global var syscall_errno
   set to the reason for the failure. */

int DEFUN(doss_modify_allocation,(m,seg,newsize,avail),
	  PC_ENV *m AND u_int16 seg AND u_int16 newsize 
	  AND u_int16 *avail)
 {
    struct doss_mcb  *p;
    int  holesize;
    
    p = m->mcb_root;
    do 
	{
	   if (p->start_seg + 1 == seg) 
	       {
		  if (newsize > p->n_para)
		      {
			 /* we're looking for memory, 
			    so let's take inventory of what this 
			    allocation contains. */
			 if (p->next)
			     {
				holesize =  p->next->start_seg - 
				       (p->start_seg + 1 + p->n_para) - 1;
			     }
			 else 
			     {
				holesize = (m->mem_size >> 4) - 
				  (p->start_seg + 1 + p->n_para) - 1;
			     }

			 if ( ((int)newsize - (int)p->n_para) < holesize)
			     {
				*avail = holesize;
				p->n_para = newsize;
				return SYSCALL_SUCCESS;
				/*NOTREACHED*/
			     }
			 else
			     {
				*avail = holesize;
				syscall_errno = DOSS_ENOMEM;
				return SYSCALL_FAILURE;
				/*NOTREACHED*/
			     }
		      }
		  else  
		    /* releasing memory. just change the 
		       value of p->n_para, and return */
		      {
			 p->n_para = newsize;
			 return SYSCALL_SUCCESS;
		      }
	       }
	   p = p->next;
	}
    while (p != NULL);
    *avail=0;
    syscall_errno = DOSS_EBADALLOC;    /* book I'm looking at 
					 doesn't list this as an
					 possible return value. Hmmm. */
    return SYSCALL_FAILURE;
 }



void DEFUN(doss_modify_mem_owner,(m,seg,psp),
	  PC_ENV *m AND     u_int16 seg AND u_int16 psp)
 {
    struct doss_mcb  *p;
    
    p = m->mcb_root;
    do 
	{
	   if (p->start_seg + 1 == seg) 
	       {
		  p->proc_psp = psp;
	       }
	   p = p->next;
	}
    while (p != NULL);
 }

/* DOSS get memory allocation strategy:
   int 0x21, function 0x58, subfunction 0x0.
 */

int DEFUN(doss_get_allocation_strategy,(m), PC_ENV *m)
 {
    return m->mem_alloc_strategy;
 }
    

/* DOSS set memory allocation strategy:
   int 0x21, function 0x58, subfunction 0x1.
 */

int DEFUN(doss_set_allocation_strategy,(m,to),
	  PC_ENV *m AND int to)
 {
    switch(to)
	{
	   
	 case MEMALLOC_FIRST_FIT:
	 case MEMALLOC_BEST_FIT:
	 case MEMALLOC_LAST_FIT:
	   m->mem_alloc_strategy = to;
	   return SYSCALL_SUCCESS;
	 default:
	   syscall_errno = DOSS_EINVALFUNC;
	   return SYSCALL_FAILURE;
	}
 }
    

/* doss_check_memalloc
   
   not called except for purposes of debugging.  
   traces the mcb chain looking for anomalies.

*/


int DEFUN(doss_check_memalloc,(m) ,PC_ENV *m)
 {
    
    struct doss_mcb  *p;
    int err;
    
    err = SYSCALL_SUCCESS;
    p = m->mcb_root;
    do
	{
	   /* things to check:
	      1) make sure that all p->start_seg < p->next->start_seg
	      2) make sure that no overlap exists in any 
	      adjacent allocations.
	      3) last block not overallocated.
	      4) all pointers OK.
	      5) memory image OK.
	   */
	   if (p->next && 
	       (p->next->start_seg <= p->start_seg))
	       {
		 debug_printf(m,"Memory allocation out of order\n");
		  err =SYSCALL_FAILURE;
	       }
	   if (p->next && 
	       (int)( p->next->start_seg) - 
	          (int)(p->start_seg + 1 + p->n_para) < 0)
	       {
		  debug_printf(m,"Memory allocation overlap\n");
		  err =SYSCALL_FAILURE;
	       }
	   if (p->next == NULL &&
	       (int)(m->mem_size >> 4) - 
	         (p->start_seg + 1 + p->n_para) < 0)
	       {
		  debug_printf(m,"Overallocated last block\n");
		  err = SYSCALL_FAILURE;
		  
	       }	   
	   if (p->next && p->next->prev != p)
	       {
		  debug_printf(m,
			       "PREV pointer corruption in memalloc chain\n");
		  err=SYSCALL_FAILURE;
	       }		  
	   if (p->prev && p->prev->next != p)
	       {
		  debug_printf(m,
			       "NEXT pointer corruption in memalloc chain\n");
		  err=SYSCALL_FAILURE;
	       }		  

	   p = p->next;
	}
    while (p != NULL);
    return err;
    
   
 }
    

/* doss_trace_memory_allocation

   another debugging routine, 
   which shows the memory allocation by walking the 
   mcb chain.  

*/

void DEFUN(doss_trace_memory_allocation,(m),	   PC_ENV *m)
 {
    
    struct doss_mcb  *p;
    
    p = m->mcb_root;
    
    do
	{
	   debug_printf(m,"Allocated %x para from %04x:0000 to %04x:0000\n",
		  p->n_para,
		  p->start_seg+1,
		  p->start_seg+1+p->n_para);

	   if (p->next && 
	       (p->next->start_seg - (p->start_seg + 1 + p->n_para) > 0))

	   debug_printf(m,"Hole of size %x at base address  %04x:0000\n",
		    p->next->start_seg - (p->start_seg + 1 + p->n_para),
		    p->start_seg + 1 + p->n_para);

	   if (p->next == NULL)
	       {
		 debug_printf(m,"Hole of size %x at base address  %04x:0000\n",
			 (m->mem_size >> 4) -  (p->start_seg + 1 + p->n_para),
			 p->start_seg + 1 + p->n_para);
	       }		  
	   p = p->next;
	}
    while (p != NULL);
    
 }

#ifdef TEST
#include "test/t_doss_mem.c"
#endif

#endif
