
/* 
 * Dos/PC Emulator
 * Copyright (C) 1991 Jim Hudgens
 * 
 * 
 * The file is part of GDE.
 * 
 * GDE is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 1, or (at your option)
 * any later version.
 * 
 * GDE is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GDE; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.  
 *
 */

#ifdef DOSS_SUPPORT
static char rcsid[]=
  "$Id: doss_fs.c,v 1.3 1991/07/30 01:58:52 hudgens Exp hudgens $";

/* DOSS_FS.C

   This module implements a MS-DOS filesystem. 

   These operations assume that the correct drive has already
   been determined, and all operations here are called via
   a switch table, based on the operations contained in the
   drive's fs_ops field.  That is, given a struct drive *d,
   and a request for an open_file, the operation ufs_open_file
   would be invoked as:
   
        (*d->fs_ops->open_file)(...);

   In this way, other mappings of doss filesystems 
   (implemented as either a single unix file, or as a unix
   device) can be added without much trouble.  

 */

#include "gde.h"

/* [JCE] Forward prototype this function to avoid a gcc warning */
static int EXFUN(fs_read_block,( PC_ENV *m AND struct doss_drive *dr AND u_int8 *buf AND u_int  bsize AND u_int  *nread AND u_int  blkno));

/*
  fs_initialize. 

  Set up all the parameters to this filesystem.  Called at 
  system initialization time, the operation of this 
  module depends on whether the object to be initialized
  is a file (ffs) or a device (dfs).

  lots to do in this case.

     make sure that the "mount" point exists and is either a 
     block device or a file.  

     open the device, and save the fd for the device
     
     read in the master boot record, and initialize the 
     fs_attr field of the drive table entry.

     initialize our little buffer cache.  BUFFERS=10!

     read in the fat table.
     
     read in the root directory.

  then return success.  If the path to the directory exists 
  and it is, in fact, a directory, then we are OK.  No other 
  initializations need be done.

 */


static int 
  DEFUN(ffs_initialize,
	(m,dr),
	PC_ENV *m AND struct doss_drive *dr)
 {
    int r,err = 0,retv,nread;
    u_int8 tmp[512];
    struct stat sb;
    u_int16 ts;
    u_int8  tc;
    u_int   size;
    u_int8 *p;
    int i;
    

    retv = FSOP_SUCCESS;
    r = lstat(dr->path,&sb);   
    if (r != 0 || ((sb.st_mode&S_IFMT)!=S_IFREG))
	{
	   if (r != 0) err = 0;
	   else err = errno;
	   sys_warn("fs_init: drive stat fails or indicates non-file: %s\n",
		    dr->path);
	   retv = FSOP_FAILURE;
	   goto errxit;
	}
    sprintf(dr->curr_path,"\\");

    /* now begins the hard part. */
    dr->fd = open(dr->path, O_RDWR );
    if (dr->fd < 0) 
	{
	   sys_warn("fs_init: drive open failed: %s\n",
		    dr->path);
	   err = errno;
	   retv = FSOP_FAILURE;
	   goto errxit;
	}
    
    dr->fs_attr = (struct fs_attr *)malloc(sizeof(struct fs_attr));
    
    if (dr->fs_attr == NULL)
	{
	   sys_fatal(errno, "fs init: malloc: ");
	}
    
    /* read the MBR of the object. */

    r = read(dr->fd,tmp,MBRSIZE);
    if (r != MBRSIZE)
	{
	   sys_warn("fs_init: drive read failed: %s\n",
		    dr->path);
	   retv = FSOP_FAILURE;
	   goto errxit;
	}
    
    /* break out the fields of the MBR into the more useful
       fs_attr struct. 
     */

    
    FETCH_WORD(tc,tmp,0);
    dr->fs_attr->magic = tc;
    FETCH_WORD(ts,tmp,11);
    dr->fs_attr->bytes_per_sector  =  ts;
    FETCH_BYTE(tc,tmp,13);
    dr->fs_attr->sectors_per_cluster = tc;
    FETCH_WORD(ts,tmp,14);
#define FAT_SECT_START  number_reserved_sectors
    dr->fs_attr->number_reserved_sectors = ts;
    FETCH_BYTE(tc,tmp,16);
    dr->fs_attr->number_of_fats = tc;
    FETCH_WORD(ts,tmp,17);
    dr->fs_attr->number_of_root_dir_entries = ts;
    FETCH_WORD(ts,tmp,19);
    dr->fs_attr->number_of_sectors = ts;
    FETCH_BYTE(tc,tmp,21);
    dr->fs_attr->media_descriptor = tc;
    FETCH_WORD(ts,tmp,22);
    dr->fs_attr->sectors_per_fat = ts;
    FETCH_WORD(ts,tmp,24);
    dr->fs_attr->sectors_per_track = ts;
    FETCH_WORD(ts,tmp,26);
    dr->fs_attr->number_of_heads = ts;
    /* calculate some useful quantities */

    dr->fs_attr->dir_sector_start = dr->fs_attr->FAT_SECT_START +
        (dr->fs_attr->number_of_fats * 
	 dr->fs_attr->sectors_per_fat);

    dr->fs_attr->number_of_dir_sectors = 
      (32 * 	 /*sizeof struct msdos_dir */
       dr->fs_attr->number_of_root_dir_entries) / 
       dr->fs_attr->bytes_per_sector;

    dr->fs_attr->data_sector_start = 
      dr->fs_attr->dir_sector_start + 
	dr->fs_attr->number_of_dir_sectors;

    dr->fs_attr->number_data_sectors = 
      dr->fs_attr->number_of_sectors -
	dr->fs_attr->data_sector_start;

    /* sanity checking  on size.  */
    size =     dr->fs_attr->number_of_sectors  * 
      dr->fs_attr->bytes_per_sector ;
    if (sb.st_size != size)
	{
	   sys_warn("fs_init: drive size incorrect: is %d should be %d\n",
		    size, sb.st_size);
	   retv = FSOP_FAILURE;
	   goto errxit;
	}
    /* initialize the buffer cache. */
    dr->fs_attr->nbuf = MAXSECTBUF;
    
    /* this will be larger (entire track) for devices */
    dr->fs_attr->buffers = 
      (struct buf *)malloc(sizeof(struct buf)*dr->fs_attr->nbuf);
    if (dr->fs_attr->buffers == NULL )
	{
	   sys_fatal(errno, "fs init: malloc: buffers");
	}

    for (i=0; i<dr->fs_attr->nbuf; i++)
	{
	   dr->fs_attr->buffers[i].flags = 0;
	   dr->fs_attr->buffers[i].blknum = 0;
	   dr->fs_attr->buffers[i].next = NULL;
	   dr->fs_attr->buffers[i].base_addr = 
	     (u_int8*)malloc(dr->fs_attr->bytes_per_sector);
	   if (	   dr->fs_attr->buffers[i].base_addr == NULL )
	       {
		  sys_fatal(errno, "fs init: malloc: buffers");
	       }
	}

    dr->fs_attr->free = dr->fs_attr->buffers;
    dr->fs_attr->head = NULL;

    /* use only one copy of the mirror */
    dr->fs_attr->fat_size = dr->fs_attr->sectors_per_fat * 
      dr->fs_attr->bytes_per_sector;
    
    dr->fs_attr->fat_table = (u_int8*)malloc(dr->fs_attr->fat_size);
    if (dr->fs_attr->fat_table == NULL )
	{
	   sys_fatal(errno, "fs init: malloc: fat_table");
	}
    
    p = dr->fs_attr->fat_table;
    for (i=0; i< dr->fs_attr->sectors_per_fat; i++)
	{
	   if (fs_read_block(m,dr,
			     p,
			     dr->fs_attr->bytes_per_sector,
			     &nread,
			     i+dr->fs_attr->FAT_SECT_START) != SYSCALL_SUCCESS)
	       {
		  sys_fatal(0,"ffs init: fat read: unknown error");
	       }
	   p += nread;
	}

errxit:    
#ifdef DEBUG
    if (DEBUG_FS(m))
	{
	   debug_printf(m, "FFS %d init\n",dr->unit);
	   debug_printf(m, "Mount point is: %s ..", dr->path);
	   if  (retv == FSOP_SUCCESS) 
	       {
		  debug_printf(m, "success\n\nDisk Parameters=\n");
		  debug_printf(m,"\tBytes per sector=%d\n",
			       dr->fs_attr->bytes_per_sector);

		  debug_printf(m,"\tSectors per cluster=%d\n",
			       dr->fs_attr->sectors_per_cluster);    
		  debug_printf(m,"\tFat from %d to %d\n",
			       dr->fs_attr->FAT_SECT_START, 
			       dr->fs_attr->FAT_SECT_START+
			       (dr->fs_attr->number_of_fats *
				dr->fs_attr->sectors_per_fat - 1));

		  
		  debug_printf(m,"\tDir sectors from");
		  debug_printf(m," %d to %d for %d entries\n",
			       dr->fs_attr->dir_sector_start,
			       (dr->fs_attr->dir_sector_start+
				dr->fs_attr->number_of_dir_sectors - 1),
			       dr->fs_attr->number_of_root_dir_entries);

		  debug_printf(m,"\tData sectors from ");
		  debug_printf(m,"%d to %d on disk of size %d\n",
			       dr->fs_attr->data_sector_start,
			       (dr->fs_attr->data_sector_start+
				dr->fs_attr->number_data_sectors),
			       dr->fs_attr->number_of_sectors);
	       }
	   else
	       {
		  debug_printf(m, "failed\n");
		  if (r == 0)
		      {
			 errno = err;
			 perror("FFS_MOUNT");
		      }
	       }
	}
#endif
    return retv;
 }


static int DEFUN(fs_read_block,(m,dr,buf,bsize,nread,blkno),
		 PC_ENV *m AND
		 struct doss_drive *dr AND
		 u_int8 *buf AND
		 u_int  bsize AND
		 u_int  *nread AND
		 u_int  blkno)
 {
    int sectsize = dr->fs_attr->bytes_per_sector;
    int nheads = dr->fs_attr->number_of_heads;
    int nsecttrk = dr->fs_attr->sectors_per_track;
    int nsect = dr->fs_attr->number_of_sectors;
    int ntrk = nsect / (nsecttrk * nheads);
    int totsize = nsect * sectsize;
    int retv = SYSCALL_SUCCESS;
    int rv;
    int eno;
    long seekval;

    if (blkno >= nsect ||         /* no reads off the end of disk */
	(bsize % sectsize != 0))  /* only read multiples of sectsize */
	{
	   retv = SYSCALL_FAILURE;
	   eno = 0;
	   goto errxit;
	}

    seekval = blkno * sectsize;

    rv = lseek(dr->fd, seekval, SEEK_SET);
    
    if (rv < 0)
	{
	   retv = SYSCALL_FAILURE;
	   eno = errno;
	   goto errxit;
	}
    
    rv = read ( dr->fd , buf, bsize);
    if (rv < 0 )
	{
	   retv = SYSCALL_FAILURE;
	   eno = errno;
	   goto errxit;
	}
    if ( rv % sectsize != 0)
	{
	   retv = SYSCALL_FAILURE;
	   eno = 0;
	   goto errxit;
	}
    *nread = rv;

errxit:    
#ifdef DEBUG
    if (DEBUG_FS(m))
	{
	   debug_printf(m, "FFS %d blkread\n",dr->unit);
	   if  (retv == FSOP_SUCCESS) 
	       {
		  debug_printf(m, "Read successful\n");
		  debug_printf(m, "blkno=%x bsize=%x nread=%x\n",
			       blkno,bsize,*nread);
	       }
	   else
	       {
		  debug_printf(m, "failed: ");
		  if (rv == 0)
		      {
			 errno = eno;
			 debug_printf(m,"errno=<%d> %s\n",
				      errno,
				      STRERROR(errno));
		      }
		  else
		    debug_printf(m,"\n");
	       }
	}
#endif
    return retv;
 }

static int DEFUN(fs_write_block,(m,dr,buf,bsize,nwrite,blkno),
		 PC_ENV *m AND
		 struct doss_drive *dr AND
		 u_int8 *buf AND
		 u_int  bsize AND
		 u_int  *nwrite AND
		 u_int  blkno)
 {
    int sectsize = dr->fs_attr->bytes_per_sector;
    int nheads = dr->fs_attr->number_of_heads;
    int nsecttrk = dr->fs_attr->sectors_per_track;
    int nsect = dr->fs_attr->number_of_sectors;
    int ntrk = nsect / (nsecttrk * nheads);
    int totsize = nsect * sectsize;
    int retv = SYSCALL_SUCCESS;
    int rv;
    int eno;
    long seekval;

    if (blkno >= nsect ||         /* no writes off the end of disk */
	blkno + ((bsize+sectsize)/sectsize) >= nsect || 
	(bsize % sectsize != 0))  /* only write multiples of sectsize */
	{
	   retv = SYSCALL_FAILURE;
	   eno = 0;
	   goto errxit;
	}

    seekval = blkno * sectsize;

    rv = lseek(dr->fd, seekval, SEEK_SET);
    
    if (rv < 0)
	{
	   retv = SYSCALL_FAILURE;
	   eno = errno;
	   goto errxit;
	}
    
    rv = write ( dr->fd , buf, bsize);
    if (rv < 0 )
	{
	   retv = SYSCALL_FAILURE;
	   eno = errno;
	   goto errxit;
	}
    if ( rv % sectsize != 0)
	{
	   retv = SYSCALL_FAILURE;
	   eno = 0;
	   goto errxit;
	}
    *nwrite = rv;

errxit:    
#ifdef DEBUG
    if (DEBUG_FS(m))
	{
	   debug_printf(m, "FFS %d blkwrite\n",dr->unit);
	   if  (retv == FSOP_SUCCESS) 
	       {
		  debug_printf(m, "write successful\n");
		  debug_printf(m, "blkno=%x bsize=%x nwrite=%x\n",
			       blkno,bsize,*nwrite);
	       }
	   else
	       {
		  debug_printf(m, "failed: ");
		  if (rv == 0)
		      {
			 errno = eno;
			 debug_printf(m,"errno=<%d> %s\n",
				      errno,
				      STRERROR(errno));
		      }
		  else
		    debug_printf(m,"\n");
	       }
	}
#endif
    return retv;
 }

#ifdef TEST
static struct buf *DEFUN(buf_read_block,(m,dr,blkno),
		 PC_ENV *m AND
		 struct doss_drive *dr AND
		 u_int  blkno)

#endif


struct doss_fs   gde_ffs = {
    ufile_fs,
    ffs_initialize,     /* DONE   */
    fs_read_block,     /* Unimpl */
    fs_write_block,    /* Unimpl */


#ifdef XXX
    fs_format,         /* Unimpl */
    fs_disk_space,     /* XXXXXX */
    fs_get_fs_attr,    /* XXXXXX */

    fs_create_file,    /* DONE  X */
    fs_open_file,      /* DONE  X */
    fs_close_file,     /* DONE  X */
    fs_seek_file,      /* DONE  X */
    fs_read_file,      /* DONE  X */
    fs_write_file,     /* DONE  X */
    fs_rename_file,
    fs_delete_file,    /* DONE  X */
    fs_get_attr,       /* DONE  X */
    fs_set_attr,       /* DONE  X */

    fs_create_dir,     /* DONE  X */
    fs_delete_dir,     /* DONE  X */
    fs_open_dir,       /* DONE  X */
    fs_read_dir,       /* DONE  X */
    fs_seek_dir,       /* DONE  X */
    fs_tell_dir,       /* DONE  X */
    fs_close_dir,      /* DONE  X */
#endif

 };


#endif   /*DOSS_SUPPORT*/

/******************************************************************/

