/************************************************************************

    SHOWKEYS v1.0.0 - Display keyboard input under Windows
                      (Windows 1.x version)

    Copyright (C) 2011  John Elliott <jce@seasip.demon.co.uk>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*************************************************************************/

/* This was originally written against the Windows 3.1 SDK (based on the 
 * GENERIC sample) and then backported to the Windows 1.0 SDK. */

#define WINVER 0x100
#include "windows.h"            /* required for all Windows applications */
#include "showkeys.h"           /* specific to this program          */

HANDLE hInst;                   /* current instance              */                      
FARPROC defListProc;		/* We will be subclassing the listbox in
				 * the main window. This is its original 
				 * wndproc. */
                      
/****************************************************************************

    FUNCTION: WinMain(HANDLE, HANDLE, LPSTR, int)

    PURPOSE: calls initialization function, processes message loop

    COMMENTS:

        Windows recognizes this function by name as the initial entry point
        for the program.  This function calls the application initialization
        routine, if no other instance of the program is running, and always
        calls the instance initialization routine.  It then executes a message
        retrieval and dispatch loop that is the top-level control structure
        for the remainder of execution.  The loop is terminated when a WM_QUIT
        message is received, at which time this function exits the application
        instance by returning the value passed by PostQuitMessage().

        If this function must abort before entering the message loop, it
        returns the conventional value NULL.

****************************************************************************/

int PASCAL WinMain(hInstance, hPrevInstance, lpCmdLine, nCmdShow)
HANDLE hInstance;                /* current instance         */
HANDLE hPrevInstance;            /* previous instance        */
LPSTR lpCmdLine;                 /* command line             */
int nCmdShow;                    /* show-window type (open/icon) */
{
    MSG msg;                     /* message              */

    if (!hPrevInstance)          /* Other instances of app running? */
    if (!InitApplication(hInstance)) /* Initialize shared things */
        return (FALSE);      /* Exits if unable to initialize     */

    /* Perform initializations that apply to a specific instance */

    if (!InitInstance(hInstance, nCmdShow))
        return (FALSE);

    /* Acquire and dispatch messages until a WM_QUIT message is received. */

    while (GetMessage(&msg,    /* message structure              */
        NULL,          /* handle of window receiving the message */
        NULL,          /* lowest message to examine          */
        NULL))         /* highest message to examine         */
    {
    TranslateMessage(&msg);    /* Translates virtual key codes       */
    DispatchMessage(&msg);     /* Dispatches message to window       */
    }
    return (msg.wParam);       /* Returns the value from PostQuitMessage */
}


/****************************************************************************

    FUNCTION: InitApplication(HANDLE)

    PURPOSE: Initializes window data and registers window class

    COMMENTS:

        This function is called at initialization time only if no other
        instances of the application are running.  This function performs
        initialization tasks that can be done once for any number of running
        instances.

        In this case, we initialize a window class by filling out a data
        structure of type WNDCLASS and calling the Windows RegisterClass()
        function.  Since all instances of this application use the same window
        class, we only need to do this when the first instance is initialized.


****************************************************************************/

BOOL InitApplication(hInstance)
HANDLE hInstance;                  /* current instance       */
{
    WNDCLASS  wc;

    /* Fill in window class structure with parameters that describe the       */
    /* main window.                                                           */

    wc.style = NULL;                    /* Class style(s).                    */
    wc.lpfnWndProc = MainWndProc;       /* Function to retrieve messages for  */
                                        /* windows of this class.             */
    wc.cbClsExtra = 0;                  /* No per-class extra data.           */
    wc.cbWndExtra = 8;                  /* 4 bytes extra per window.          */
    wc.hInstance = hInstance;           /* Application that owns the class.   */
    wc.hIcon = LoadIcon(hInstance, MAKEINTRESOURCE(IDR_MAINFRAME));
    wc.hCursor = LoadCursor(NULL, IDC_ARROW);
    wc.hbrBackground = GetStockObject(WHITE_BRUSH);
    wc.lpszMenuName =  "MainMenu";      /* Name of menu resource in .RC file. */
    wc.lpszClassName = "ShowKeyWClass"; /* Name used in call to CreateWindow. */

    /* Register the window class and return success/failure code. */

    return (RegisterClass(&wc));

}
                                                    
                                                    
                                                    
                                                    

/****************************************************************************

    FUNCTION:  InitInstance(HANDLE, int)

    PURPOSE:  Saves instance handle and creates main window

    COMMENTS:

        This function is called at initialization time for every instance of
        this application.  This function performs initialization tasks that
        cannot be shared by multiple instances.

        In this case, we save the instance handle in a static variable and
        create and display the main program window.

****************************************************************************/

BOOL InitInstance(hInstance, nCmdShow)
    HANDLE          hInstance;          /* Current instance identifier.       */
    int             nCmdShow;           /* Param for first ShowWindow() call. */
{
    HWND            hWnd;               /* Main window handle.                */
    HWND            hWndList;            /* Listbox window handle.             */
    RECT            rc;
        
    /* Save the instance handle in static variable, which will be used in  */
    /* many subsequence calls from this application to Windows.            */

    hInst = hInstance;

    /* Create a main window for this application instance.  */

    hWnd = CreateWindow(
        "ShowKeyWClass",                /* See RegisterClass() call.          */
        "Keyboard events viewer",       /* Text for window title bar.         */
        WS_TILEDWINDOW,                 /* Window style.                      */
        CW_USEDEFAULT,                  /* Default horizontal position.       */
        CW_USEDEFAULT,                  /* Default vertical position.         */
        CW_USEDEFAULT,                  /* Default width.                     */
        CW_USEDEFAULT,                  /* Default height.                    */
        NULL,                           /* Overlapped windows have no parent. */
        NULL,                           /* Use the window class menu.         */
        hInstance,                      /* This instance owns this window.    */
        NULL                            /* Pointer not needed.                */
    );
        
    /* If window could not be created, return "failure" */

    if (!hWnd)
        return (FALSE);
       
    GetClientRect(hWnd, &rc);   
      
    /* Create our child window, a list box */ 
    hWndList = CreateWindow("LISTBOX", /* Control class */
                            "",        /* No title */
                            WS_VISIBLE | WS_CHILD | WS_VSCROLL,  /* Window style  */
                            rc.left, rc.top, (rc.right - rc.left), (rc.bottom - rc.top),
                            hWnd,       /* Parent */
                            NULL,       /* Window class */
                            hInstance,
                            NULL);
    if (!hWndList)
        return FALSE;
                           
    /* Subclass the list box */    
    defListProc = (FARPROC)GetWindowLong(hWndList, GWL_WNDPROC);
    SetWindowLong(hWnd,     0,           (long)hWndList);    
    SetWindowLong(hWndList, GWL_WNDPROC, 
            (LONG)MakeProcInstance((FARPROC)ListWndProc, hInst));

/* If this were to be run on Windows 3.x or later, we would need to force
 * a fixed-pitch system font.             
    hFont = CreateFont(10, 0, 0, 0, FW_DONTCARE, FALSE, FALSE, FALSE, ANSI_CHARSET, 
                        OUT_DEFAULT_PRECIS, CLIP_DEFAULT_PRECIS, PROOF_QUALITY, FIXED_PITCH,
                        "Fixedsys");
    SendMessage(hWndList, WM_SETFONT, (WPARAM)hFont, TRUE);        
*/
    /* Make the window visible; update its client area; and return "success" */

    ShowWindow(hWnd, nCmdShow);  /* Show the window                        */
    UpdateWindow(hWnd);          /* Sends WM_PAINT message                 */
    return (TRUE);               /* Returns the value from PostQuitMessage */

}
               

/* To save my removing the TEXT() decoration from any text pasted in from
 * the MFC version */
#ifndef TEXT
#define TEXT(s) s
#endif
    
/* Convert nChar to a textual VK_ virtual key. Key numbers used are based on */
/* the Windows 2000 DDK; since this has more VK_ macros than the Windows 1.x */
/* SDK headers that we're using here, the other ones are done as constants. */

LPCSTR KeyName(nChar)
UINT nChar;
{    
    static char szBuf[20];

    /* Digits, letters and function keys are all done in a nice neat */
    /* sequence. */    
    if ((nChar >= '0' && nChar <= '9') || (nChar >= 'A' && nChar <= 'Z'))
    {
        sprintf(szBuf, TEXT("VK_%c"), nChar);
        return (LPCSTR)szBuf;
    }          
    if (nChar >= VK_F1 && nChar <= /*VK_F24*/ VK_F16)
    {
        sprintf(szBuf, TEXT("VK_F%d"), (nChar - VK_F1 + 1));
        return (LPCSTR)szBuf;
    }          
    if (nChar >= VK_NUMPAD0 && nChar <= VK_NUMPAD9)
    {
        sprintf(szBuf, TEXT("VK_NUMPAD%d"), (nChar - VK_NUMPAD0));
        return (LPCSTR)szBuf;
    }          

    /* The other ones. */
    switch(nChar)
    {
        case VK_LBUTTON:   return (LPCSTR)("VK_LBUTTON");
        case VK_RBUTTON:   return (LPCSTR)("VK_RBUTTON");
        case VK_CANCEL:    return (LPCSTR)("VK_CANCEL");
        case VK_MBUTTON:   return (LPCSTR)("VK_MBUTTON");
        case 0x05:         return (LPCSTR)("VK_XBUTTON1");
        case 0x06:         return (LPCSTR)("VK_XBUTTON2");
/* 0x07 unassigned */
        case VK_BACK:      return (LPCSTR)("VK_BACK");
        case VK_TAB:       return (LPCSTR)("VK_TAB");
/* 0x0A-0x0B reserved */
        case VK_CLEAR:     return (LPCSTR)("VK_CLEAR");
        case VK_RETURN:    return (LPCSTR)("VK_RETURN");
        case VK_SHIFT:     return (LPCSTR)("VK_SHIFT");
        case VK_CONTROL:   return (LPCSTR)("VK_CONTROL");
        case VK_MENU:      return (LPCSTR)("VK_MENU");
        case VK_PAUSE:     return (LPCSTR)("VK_PAUSE");
        case VK_CAPITAL:   return (LPCSTR)("VK_CAPITAL");
        case 0x15:         return (LPCSTR)("VK_KANA");
        case 0x17:         return (LPCSTR)("VK_JUNJA");
        case 0x18:         return (LPCSTR)("VK_FINAL");
        case 0x19:         return (LPCSTR)("VK_KANJI");
        case VK_ESCAPE:    return (LPCSTR)("VK_ESCAPE");
        case 0x1C:         return (LPCSTR)("VK_CONVERT");
        case 0x1D:         return (LPCSTR)("VK_NONCONVERT");
        case 0x1E:         return (LPCSTR)("VK_ACCEPT");
        case 0x1F:         return (LPCSTR)("VK_MODECHANGE");
        case VK_SPACE:     return (LPCSTR)("VK_SPACE");
        case VK_PRIOR:     return (LPCSTR)("VK_PRIOR");
        case VK_NEXT:      return (LPCSTR)("VK_NEXT");
        case VK_END:       return (LPCSTR)("VK_END");
        case VK_HOME:      return (LPCSTR)("VK_HOME");
        case VK_LEFT:      return (LPCSTR)("VK_LEFT");
        case VK_UP:        return (LPCSTR)("VK_UP");
        case VK_RIGHT:     return (LPCSTR)("VK_RIGHT");
        case VK_DOWN:      return (LPCSTR)("VK_DOWN");
        case VK_SELECT:    return (LPCSTR)("VK_SELECT");
        case VK_PRINT:     return (LPCSTR)("VK_PRINT");
        case VK_EXECUTE:   return (LPCSTR)("VK_EXECUTE");
        case 0x2C:         return (LPCSTR)("VK_SNAPSHOT");
        case VK_INSERT:    return (LPCSTR)("VK_INSERT");
        case VK_DELETE:    return (LPCSTR)("VK_DELETE");
        case VK_HELP:      return (LPCSTR)("VK_HELP");
/* 0x30-0x39 digits */
/* 0x3A-0x40 unassigned */
/* 0x41-0x5A letters */
        case 0x5B:         return (LPCSTR)("VK_LWIN");
        case 0x5C:         return (LPCSTR)("VK_RWIN");
        case 0x5D:         return (LPCSTR)("VK_APPS");        
/* 0x5E reserved */
        case 0x5F:         return (LPCSTR)("VK_SLEEP");        
/* 0x60-0x69 numpad keys */
        case VK_MULTIPLY:  return (LPCSTR)("VK_MULTIPLY");        
        case VK_ADD:       return (LPCSTR)("VK_ADD");        
        case VK_SEPARATOR: return (LPCSTR)("VK_SEPARATOR");    
        case VK_SUBTRACT:  return (LPCSTR)("VK_SUBTRACT");        
        case VK_DECIMAL:   return (LPCSTR)("VK_DECIMAL");        
        case VK_DIVIDE:    return (LPCSTR)("VK_DIVIDE");        
/* 0x70-0x87 function keys */
/* 0x88-0x8F unassigned */
        case 0x90:         return (LPCSTR)("VK_NUMLOCK");        
        case 0x91:         return (LPCSTR)("VK_SCROLL");
        case 0x92:         return (LPCSTR)("VK_OEM_FJ_JISHO");
        case 0x93:         return (LPCSTR)("VK_OEM_FJ_MASSHOU");
        case 0x94:         return (LPCSTR)("VK_OEM_FJ_TOUROKU");
        case 0x95:         return (LPCSTR)("VK_OEM_FJ_LOYA");
        case 0x96:         return (LPCSTR)("VK_OEM_FJ_ROYA");
/* 0x97-0x9F unassigned */
        case 0xA0:         return (LPCSTR)("VK_LSHIFT");     
        case 0xA1:         return (LPCSTR)("VK_RSHIFT");     
        case 0xA2:         return (LPCSTR)("VK_LCONTROL");
        case 0xA3:         return (LPCSTR)("VK_RCONTROL");   
        case 0xA4:         return (LPCSTR)("VK_LMENU"); 
        case 0xA5:         return (LPCSTR)("VK_RMENU");
        case 0xA6:         return (LPCSTR)("VK_BROWSER_BACK");        
        case 0xA7:         return (LPCSTR)("VK_BROWSER_FORWARD");     
        case 0xA8:         return (LPCSTR)("VK_BROWSER_REFRESH");     
        case 0xA9:         return (LPCSTR)("VK_BROWSER_STOP");        
        case 0xAA:         return (LPCSTR)("VK_BROWSER_SEARCH");
        case 0xAB:         return (LPCSTR)("VK_BROWSER_FAVORITES");
        case 0xAC:         return (LPCSTR)("VK_BROWSER_HOME");        
        case 0xAD:         return (LPCSTR)("VK_VOLUME_MUTE");         
        case 0xAE:         return (LPCSTR)("VK_VOLUME_DOWN");         
        case 0xAF:         return (LPCSTR)("VK_VOLUME_UP");           
        case 0xB0:         return (LPCSTR)("VK_MEDIA_NEXT_TRACK");    
        case 0xB1:         return (LPCSTR)("VK_MEDIA_PREV_TRACK");    
        case 0xB2:         return (LPCSTR)("VK_MEDIA_STOP");          
        case 0xB3:         return (LPCSTR)("VK_MEDIA_PLAY_PAUSE");    
        case 0xB4:         return (LPCSTR)("VK_LAUNCH_MAIL");
        case 0xB5:         return (LPCSTR)("VK_LAUNCH_MEDIA_SELECT");
        case 0xB6:         return (LPCSTR)("VK_LAUNCH_APP1");         
        case 0xB7:         return (LPCSTR)("VK_LAUNCH_APP2");
/* 0xB8-0xB9 reserved */
        case 0xBA:         return (LPCSTR)("VK_OEM_1");        
        case 0xBB:         return (LPCSTR)("VK_OEM_PLUS");     
        case 0xBC:         return (LPCSTR)("VK_OEM_COMMA");    
        case 0xBD:         return (LPCSTR)("VK_OEM_MINUS"); 
        case 0xBE:         return (LPCSTR)("VK_OEM_PERIOD");
        case 0xBF:         return (LPCSTR)("VK_OEM_2");
        case 0xC0:         return (LPCSTR)("VK_OEM_3");     
/* 0xC1-0xD7 reserved */
/* 0xD8-0xDA unassigned */
        case 0xDB:         return (LPCSTR)("VK_OEM_4");    
        case 0xDC:         return (LPCSTR)("VK_OEM_5"); 
        case 0xDD:         return (LPCSTR)("VK_OEM_6");
        case 0xDE:         return (LPCSTR)("VK_OEM_7");
        case 0xDF:         return (LPCSTR)("VK_OEM_8");     
        case 0xE1:         return (LPCSTR)("VK_OEM_AX"); 
        case 0xE2:         return (LPCSTR)("VK_OEM_102");
        case 0xE3:         return (LPCSTR)("VK_ICO_HELP");
        case 0xE4:         return (LPCSTR)("VK_ICO_00");     
        case 0xE5:         return (LPCSTR)("VK_PROCESSKEY");     
        case 0xE6:         return (LPCSTR)("VK_ICO_CLEAR");     
        case 0xE7:         return (LPCSTR)("VK_PACKET");     
/* 0xE8 unassigned */
        case 0xE9:         return (LPCSTR)("VK_OEM_RESET");   
        case 0xEA:         return (LPCSTR)("VK_OEM_JUMP");
        case 0xEB:         return (LPCSTR)("VK_OEM_PA1");    
        case 0xEC:         return (LPCSTR)("VK_OEM_PA2");  
        case 0xED:         return (LPCSTR)("VK_OEM_PA3");   
        case 0xEE:         return (LPCSTR)("VK_OEM_WSCTRL");   
        case 0xEF:         return (LPCSTR)("VK_OEM_CUSEL");  
        case 0xF0:         return (LPCSTR)("VK_OEM_ATTN");   
        case 0xF1:         return (LPCSTR)("VK_OEM_FINISH"); 
        case 0xF2:         return (LPCSTR)("VK_OEM_COPY");      
        case 0xF3:         return (LPCSTR)("VK_OEM_AUTO");     
        case 0xF4:         return (LPCSTR)("VK_OEM_ENLW");     
        case 0xF5:         return (LPCSTR)("VK_OEM_BACKTAB");           
        case 0xF6:         return (LPCSTR)("VK_ATTN");
        case 0xF7:         return (LPCSTR)("VK_CRSEL");
        case 0xF8:         return (LPCSTR)("VK_EXSEL");
        case 0xF9:         return (LPCSTR)("VK_EREOF");
        case 0xFA:         return (LPCSTR)("VK_PLAY");
        case 0xFB:         return (LPCSTR)("VK_ZOOM");
        case 0xFC:         return (LPCSTR)("VK_NONAME");
        case 0xFD:         return (LPCSTR)("VK_PA1");
        case 0xFE:         return (LPCSTR)("VK_OEM_CLEAR");
        
        default:           return (LPCSTR)("");
    }
}


/* Add the contents of a WM_KEYDOWN etc. message to the list box */
void ShowKey(hWnd, szMessage, wParam, lParam)
    HWND hWnd;
    char *szMessage;
    WPARAM wParam;
    LPARAM lParam;
{
    char szBuf[200];                               
    LRESULT nItem;

    sprintf(szBuf, TEXT("%-13.13s Char=0x%04x RepCnt=0x%04x Flags=0x%04x %s"), 
        szMessage, (UINT)wParam, (UINT)(lParam & 0xFFFF), 
	(UINT)(lParam >> 16), KeyName(wParam));
    nItem = SendMessage(hWnd, LB_ADDSTRING, (WPARAM)0, (LPARAM)(LPCSTR)szBuf);
    SendMessage(hWnd, LB_SETCURSEL, (WPARAM)nItem, (LPARAM)0);    
}
    
   
/* The list box records keypresses */    
long CALLBACK EXPORT ListWndProc(hWnd, message, wParam, lParam)
HWND hWnd;                      /* window handle                 */
UINT message;                   /* type of message               */
WPARAM wParam;                  /* additional information        */
LPARAM lParam;                  /* additional information        */
{                       
    switch (message)
    {
        case WM_GETDLGCODE:
            return DLGC_WANTALLKEYS;
            
        case WM_KEYDOWN:
            ShowKey(hWnd, "WM_KEYDOWN", wParam, lParam);
            break;

        case WM_SYSKEYDOWN:
            ShowKey(hWnd, "WM_SYSKEYDOWN", wParam, lParam);
            break;

        case WM_KEYUP:
            ShowKey(hWnd, "WM_KEYUP", wParam, lParam);
            break;

        case WM_SYSKEYUP:
            ShowKey(hWnd, "WM_SYSKEYUP", wParam, lParam);
            break;

        default:                  /* Passes it on if unproccessed    */
            return (*defListProc)(hWnd, message, wParam, lParam);
    }
    return (NULL);
}



/****************************************************************************

    FUNCTION: MainWndProc(HWND, UINT, WPARAM, LPARAM)

    PURPOSE:  Processes messages

    MESSAGES:

    WM_COMMAND    - application menu (About dialog box)
    WM_DESTROY    - destroy window

    COMMENTS:

    To process the IDM_ABOUT message, call MakeProcInstance() to get the
    current instance address of the About() function.  Then call Dialog
    box which will create the box according to the information in your
    generic.rc file and turn control over to the About() function.  When
    it returns, free the intance address.

****************************************************************************/

long CALLBACK EXPORT MainWndProc(hWnd, message, wParam, lParam)
HWND hWnd;                      /* window handle                 */
UINT message;                   /* type of message               */
WPARAM wParam;                  /* additional information        */
LPARAM lParam;                  /* additional information        */
{   
    HWND hWndList;
/*    HFONT hFont; */
    char buf[60];
    int nSel;

    switch (message)
    {
        case WM_COMMAND:       /* message: command from application menu */
            switch (wParam)
            {
                case IDM_ABOUT:
                    DialogBox(hInst,        /* current instance          */
                        "AboutBox",         /* resource to use           */
                        hWnd,               /* parent handle             */
                        MakeProcInstance((FARPROC)About, hInst));             /* About() instance address  */
                    break;             
                    
                case ID_KEYBOARD_INFORMATION:
                    hWndList = (HWND)GetWindowLong(hWnd, 0);
/* Function not present in Windows 1.x
                    sprintf(buf, "Keyboard type:    %d", GetKeyboardType(0));
                    nSel = (int)SendMessage(hWndList, LB_ADDSTRING, (WPARAM)0, (LPARAM)(LPCSTR)buf);
                    sprintf(buf, "Keyboard subtype: %d", GetKeyboardType(1));
                    nSel = (int)SendMessage(hWndList, LB_ADDSTRING, (WPARAM)0, (LPARAM)(LPCSTR)buf);
                    sprintf(buf, "Function keys:    %d", GetKeyboardType(2));
                    nSel = (int)SendMessage(hWndList, LB_ADDSTRING, (WPARAM)0, (LPARAM)(LPCSTR)buf);
                    SendMessage(hWndList, LB_SETCURSEL, (WPARAM)nSel, 0);
                    */
                    break;
                                        
                case ID_KEYS_ALLKEYNAMES:
                    {
/* Function not present in Windows 1.x */
                        DWORD dw = 0x10000L;
                        char kname[40];
                        int n;
                                               
                        hWndList = (HWND)GetWindowLong(hWnd, 0);
                        /* The first 256 are normal scancodes. The second 256 are extended. */ 
                        for (n = 0; n < 511; n++)
                        {   
                            /*
                            if (GetKeyNameText(dw, kname, 40))
                            {
                                sprintf(buf, TEXT("%08lx %s"), dw, (LPCSTR)kname);    
                                nSel = (int)SendMessage(hWndList, LB_ADDSTRING, (WPARAM)0, (LPARAM)(LPCSTR)buf);
                            }*/            
                            dw += 0x10000L;
                        }          
                        SendMessage(hWndList, LB_SETCURSEL, (WPARAM)nSel, (LPARAM)0);
                    }
                    break;    
                    
                case ID_FILE_EXIT:       
                    PostMessage(hWnd, WM_CLOSE, (WPARAM)0, (LPARAM)0);
                    break;
                                    
                default:                    /* Lets Windows process it   */
                    return (DefWindowProc(hWnd, message, wParam, lParam));
            }  
            break;
            
        case WM_KEYDOWN:                           
        case WM_KEYUP:
        case WM_SYSKEYDOWN:
        case WM_SYSKEYUP:
            hWndList = (HWND)GetWindowLong(hWnd, 0);
            PostMessage(hWndList, message, wParam, lParam);
            break;    
            
        case WM_SIZE:
            hWndList = (HWND)GetWindowLong(hWnd, 0);
            MoveWindow(hWndList, 0, 0, LOWORD(lParam), HIWORD(lParam), TRUE);
            break;        
                             
        case WM_DESTROY:          /* message: window being destroyed */
/*            hFont = (HFONT)GetWindowLong(hWnd, 1);
            DeleteObject(hFont); */
            PostQuitMessage(0);
            break;

        default:                  /* Passes it on if unproccessed    */
            return (DefWindowProc(hWnd, message, wParam, lParam));
    }
    return (NULL);
}


/****************************************************************************

    FUNCTION: About(HWND, unsigned, WORD, LONG)

    PURPOSE:  Processes messages for "About" dialog box

    MESSAGES:

    WM_INITDIALOG - initialize dialog box
    WM_COMMAND    - Input received

    COMMENTS:

    No initialization is needed for this particular dialog box, but TRUE
    must be returned to Windows.

    Wait for user to click on "Ok" button, then close the dialog box.

****************************************************************************/

static LPCSTR st_szAbout[] =
{
    TEXT("SHOWKEYS v1.0.0 - Display keyboard input under Windows"),
    TEXT(""),
    TEXT("Copyright (C) 2011  John Elliott <jce@seasip.demon.co.uk>"),
    TEXT(""),
    TEXT("This program is free software; you can redistribute it and/or modify"),
    TEXT("it under the terms of the GNU General Public License as published by"),
    TEXT("the Free Software Foundation; either version 2 of the License, or"),
    TEXT("(at your option) any later version."),
    TEXT(""),
    TEXT("This program is distributed in the hope that it will be useful,"),
    TEXT("but WITHOUT ANY WARRANTY; without even the implied warranty of"),
    TEXT("MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the"),
    TEXT("GNU General Public License for more details."),
    TEXT(""),
    TEXT("You should have received a copy of the GNU General Public License"),
    TEXT("along with this program; if not, write to the Free Software"),
    TEXT("Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA."),
    NULL
};                                 


/* About box instance variables. We will have to assume that there aren't 
 * two aboutboxes around at a time. */
static HPEN peBlack;
static HBRUSH brWhite, brGrey, brFace, brHalftone;
static HBITMAP bmHalftone;
static COLORREF crWindow, crWindowText; 

void DeleteColours()
{
    if (peBlack) DeleteObject(peBlack);
    if (brWhite) DeleteObject(brWhite);
    if (brGrey)  DeleteObject(brGrey);
    if (brFace)  DeleteObject(brFace);    
    peBlack = NULL;
    brWhite = NULL;
    brGrey = NULL;                    
    brFace = NULL;
}


void SetupColours()
{                  
    DeleteColours();
    crWindow = GetSysColor(COLOR_WINDOW);
    crWindowText = GetSysColor(COLOR_WINDOWTEXT);
    peBlack = CreatePen(PS_SOLID, 1, crWindowText);
    brWhite = CreateSolidBrush(crWindow);
    brGrey = CreateSolidBrush(0x808080L); /* GetSysColor(COLOR_BTNSHADOW));*/
    brFace = CreateSolidBrush(0xC0C0C0L); /*GetSysColor(COLOR_BTNFACE));*/
}


BOOL EXPORT CALLBACK About(hDlg, message, wParam, lParam)
HWND hDlg;               /* window handle of the dialog box */
unsigned message;        /* type of message                 */
WORD wParam;             /* message-specific information    */
LONG lParam;
{           
    HWND hWndChild, hWndList; 
    HDC hDC;
    int n;
    
    switch (message)
    {
        case WM_INITDIALOG:            /* message: initialize dialog box */
            bmHalftone = LoadBitmap(hInst, "HALFTONE");
            brHalftone = CreatePatternBrush(bmHalftone);
            SetupColours();            

            hWndList = GetDlgItem(hDlg, IDC_GPL);        
            SendMessage(hWndList, LB_RESETCONTENT, (WPARAM)0, (LPARAM)0);
            for (n = 0; st_szAbout[n]; n++)
            {
                SendMessage(hWndList, LB_ADDSTRING, 0, (LPARAM)(LPCSTR)(st_szAbout[n]));
            } 
            return (TRUE);
    
        case WM_DESTROY:
            DeleteColours();
            DeleteObject(brHalftone); brHalftone = NULL;
            DeleteObject(bmHalftone); bmHalftone = NULL;
            break;

        case WM_CTLCOLOR:
            hWndChild = (HWND)LOWORD(lParam);
            hDC = (HDC)wParam;
/*            switch (GetDlgCtrlID(hWndChild))
            {
                case IDC_ST_TITLE:
                case IDC_ST_ICON:
                    SetBkColor(hDC, crWindow);
                    return (BOOL)brWhite;
            } */
            return FALSE;       

        case WM_SYSCOLORCHANGE:
            SetupColours();
            InvalidateRect(hDlg, NULL, TRUE);
            break;
            
        case WM_ERASEBKGND:
            {
                RECT rc, rc2;
                HPEN peOld;
                HBRUSH brOld;
                HDC hDC = (HDC)wParam;
                COLORREF crfg, crbg;
                
                GetClientRect(hDlg, &rc);
                FillRect(hDC, &rc, brFace);
                            
		/* Draw the GPL info and the rectangle around it */	
                hWndChild = GetDlgItem(hDlg, IDC_ST_INFOFRAME);             
                GetWindowRect(hWndChild, &rc);
                ScreenToClient(hDlg, (LPPOINT)&rc.left); 
                ScreenToClient(hDlg, (LPPOINT)&rc.right); 
                peOld = SelectObject(hDC, peBlack);
                brOld = SelectObject(hDC, brWhite);
                Rectangle(hDC, rc.left, rc.top, rc.right, rc.bottom);
                SelectObject(hDC, brGrey);                                      
                rc2.left = rc.right;
                rc2.right = rc.right + 2;
                rc2.top = rc.top + 1;
                rc2.bottom = rc.bottom + 2;
                FillRect(hDC, &rc2, brGrey);
                rc2.left = rc.left + 1;
                rc2.top  = rc.bottom;
                FillRect(hDC, &rc2, brGrey);

		/* Draw the top rectangle. */
                hWndChild = GetDlgItem(hDlg, IDC_ST_FRAME);
                GetWindowRect(hWndChild, &rc);
                ScreenToClient(hDlg, (LPPOINT)&rc.left); 
                ScreenToClient(hDlg, (LPPOINT)&rc.right); 
                crbg = SetBkColor  (hDC, crWindow);
                crfg = SetTextColor(hDC, crWindowText);
                SelectObject(hDC, brHalftone);
                Rectangle(hDC, rc.left, rc.top - 1, rc.right, rc.bottom + 1);
                SetTextColor(hDC, crfg);
                SetBkColor(hDC, crbg);

                hWndChild = GetDlgItem(hDlg, IDC_ST_TITLE);
                GetWindowRect(hWndChild, &rc2);
                ScreenToClient(hDlg, (LPPOINT)&rc2.left); 
                ScreenToClient(hDlg, (LPPOINT)&rc2.right);
                rc2.top = rc.top;
                rc2.bottom = rc.bottom;
                FillRect(hDC, &rc2, brWhite); 

                SelectObject(hDC, brOld);
                SelectObject(hDC, peOld);
                return TRUE;                       
            }
        case WM_COMMAND:               /* message: received a command */
            if (wParam == IDOK         /* "OK" box selected?          */
                || wParam == IDCANCEL) /* System menu close command?  */
            {
                EndDialog(hDlg, TRUE); /* Exits the dialog box        */
                return (TRUE);
            }
            break;
    }
    return (FALSE);               /* Didn't process a message    */
}

