/************************************************************************

    ICOLOAD - load Windows 3.x .ICO and .BMP files as GEM images

    Copyright (C) 1999  John Elliott <jce@seasip.demon.co.uk>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*************************************************************************/


#include "uitest.h"

#include <dos.h>				// for FP_OFF and FP_SEG macros
#include <string.h>				// string functions

#include "win3ico.h"

#ifndef SEEK_SET
#define SEEK_SET 0
#define SEEK_CUR 1
#endif

/* This file loads Windows .BMP files as MFDB objects. If CHEAT has been
 * #defined, it loads them as device-specific images; if not, as 
 * device-independent.
 *
 * If all bitmaps were guaranteed smaller than 64k, we'd have it easy.
 * However, they aren't; 640*480*16 images occupy 150k. Pacific's pointers
 * are "large", not "huge"; they wrap at 64k. malloc() takes a 16-bit
 * parameter, so images bigger than 64k can't be allocated this way.
 *
 * Therefore all allocation is done with lalloc() and lfree(); and
 * the function ptr_add() does huge pointer arithmetic.
 *
 *
 */



/* Debugging code */

#if DEBUG


void dumpMFDB(MFDB *mfb)
	{
		FILE *fp;
		int x, m;
		
		fp = fopen(DEBUGFILE, "a");

		fprintf(fp, "fwp=%d\n"
		            "fh =%d\n"
		            "fww=%d\n"
		            "ff= %d\n"
		            "np= %d\n",
		            mfb->fwp, mfb->fh, mfb->fww, mfb->ff, mfb->np);

		m = mfb->np * mfb->fww * mfb->fh * 2;
		            
		for (x = 0; x < m; x++)
		{
			if (!(x & 3))   fprintf(fp, "\n");
			if (!(x & 127)) fprintf(fp, "\n");
			fprintf(fp, "%02x ", ((unsigned char far *)(mfb->mp))[x]);
		}

		fclose(fp);
	}



#endif	/* DEBUG */


#if (!CHEAT)	/* Convert device-independent to/from VGA16 
	               device-specific format */

static void swapMFDB(MFDB *icon)
{
	unsigned int p, x, y;
	unsigned long pl = 2L * icon->fh * icon->fww;
	UBYTE far *ptr;
	
	for (p = 0; p < icon->np; p++)
	{
		for (y = 0; y < icon->fh; y++)
		{
			ptr = ((UBYTE far *)(icon->mp)) + pl * p + 2*y * icon->fww;

			for (x = 0; x < icon->fww; x++)
			{
				UBYTE b1;

				b1         = ptr[x*2];
				ptr[x*2]   = ptr[x*2+1];
				ptr[x*2+1] = b1;
			}
		}
	}

}

#endif


MLOCAL unsigned char palette[1024]; /* 256 colours -> 1k palette */
MLOCAL int pal_len;					/* Actual size of palette */
MLOCAL WIN31BM bmhdr;				/* .BMP header record */
MLOCAL WORD fwl;					/* Length of a .BMP line */
MLOCAL unsigned long pl;			/* Size of a GEM plane */

MLOCAL int alloc_palette(void)
{	
	switch(bmhdr.bm_cdepth)
	{
		case 1:	/* mono */
		case 4:	/* 16-colour */
		case 8: /* 256-colour */
			if (bmhdr.bm_usedc) pal_len = 4*bmhdr.bm_usedc;
			else pal_len = 4 * (1 << bmhdr.bm_cdepth);
			break;
			
		case 24: /* 24-bit colour */
			pal_len = 0;
			break;

		default:
			TRACE(("Unsupported BMP variation\n"));
			return -5;	/* Unsupported BMP variation*/
	}
	return 0;
}


int load_mask(FILE *fp, MFDB *mask)
{
	UWORD cbpos;
	LONG y, y0;
	int	x,b;
		
	y0  = 0;
	fwl = (bmhdr.bm_width + 7) / 8;

	fwl = ((fwl + 3) / 4) * 4; /* dword-aligned */
	
	for (y = bmhdr.bm_height - 1; y >= 0; y--)
	{
		/* One plane is max. 37k, so "large" pointers are safe */
		LPBYTE dpos = (LPBYTE)mask->mp + (2L * y * mask->fww);

		for (x = 0; x < 2*mask->fww; x++)
		{
			b = fgetc(fp); 
			if (b == EOF)
			{
				TRACE(("Premature EOF reading mask\n"));
				return -2;
			}
			dpos[x] = b;
		}

		cbpos = 2 * mask->fww;
			
		while (cbpos < fwl) 	/* Padding at end of line */
		{
			if (fgetc(fp) == EOF) 
			{
				TRACE(("Premature EOF reading mask\n"));
				return -2;
			}
			++cbpos;
		}
		++y0;
	}
	return 0;
}


LPBYTE ptr_add(LPBYTE ptr, unsigned long pl)
{
	UWORD seg, off, noff;

	seg = FP_SEG(ptr);
	off = FP_OFF(ptr);

	noff = off + (pl & 0xFFFF);
	if (noff < off)	/* Pointer wrap */
	{
		seg += 0x1000;	/* add 64k */
	}

	seg += 0x1000 * (pl >> 16);

	return MK_FP(seg, noff);
}


int load_image(FILE *fp, MFDB *icon)
{
	UWORD cbpos;
	WORD  cbyte, r, g, b, dim, ink;
	LONG  x, y, y0;
	LPBYTE dpos, dbase;
	
	y0 = cbpos = 0;

	TRACE(("w=%ld h=%ld planelen=%ld\n", bmhdr.bm_width, bmhdr.bm_height, pl));
	
	for (y = bmhdr.bm_height - 1; y >= 0; y--)
	{
		dbase = ptr_add((LPBYTE)(icon->mp), 2L * y * icon->fww);

//		TRACE(("icon->mp=%lx dbase=%lx\n", icon->mp, dbase));
		
		for (x = 0; x < bmhdr.bm_width; x++)
		{
			UBYTE  m    = (0x80 >> (x & 7));

			dpos = ptr_add(dbase, (x/8)); 
			
			switch(bmhdr.bm_cdepth)
			{
				case 1: if ((x & 7) == 0) 
						{
							cbyte = fgetc(fp);
							if (cbyte == EOF) return -2;
							++cbpos;
						}

						ink = (cbyte & m) ? 1 : 0; 
						break;
				case 4: 
						if (!(x & 1))
						{
							cbyte = fgetc(fp);
							if (cbyte == EOF) return -2;
							++cbpos;
						    ink = (cbyte >> 4) & 0x0F;
						}
						else ink = cbyte & 0x0F;
						break;

				case 8: ink = fgetc(fp);
						if (ink == EOF) return -2;
						++cbpos;
						break;

				case 24: r = fgetc(fp); ++cbpos; if (r == EOF) return -2;
						 g = fgetc(fp); ++cbpos; if (g == EOF) return -2;
						 b = fgetc(fp);	++cbpos; if (b == EOF) return -2;
						 break;
			}

			if (pal_len)
			{
				r = palette[ink*4]    & 0xFF;
				g = palette[ink*4+1]  & 0xFF;
				b = palette[ink*4+2]  & 0xFF;
			}

			/* Dim colours -> black */

			if (r < 0x80) r = 0;
			if (g < 0x80) g = 0;
			if (b < 0x80) b = 0;
			
			/* Convert black/white/grey/grey to the rather */
			/* odd GEM palette */
			if (r == g && g == b)
			{
				if      (r > 0xE0) { r = g = b = 0;    dim = 0; } /* white */
				else if (r > 0xA0) { r = g = b = 0xFF; dim = 0; } /* dgrey */
				else if (r > 0x40) { r = g = b = 0;    dim = 1; } /* lgrey */
				else               { r = g = b = 0xFF; dim = 1; } /* black */
			}
			else if (r >= 0xC0 || g >= 0xC0 || b >= 0xC0) dim = 0; 
			else dim = 1;

			/* The Pacific C code generator doesn't like dpos getting
			 * past 64k; its pointers aren't "huge". Therefore the 
			 * function ptr_add() is used to implement "huge" pointers
			 * properly 
			 */
			
			if (b)   dpos[0]    |= m; else dpos[0]    &= ~m;
			dpos = ptr_add(dpos, pl);
			if ((unsigned long)dpos < (unsigned long)dbase) TRACE(("y=%ld x=%ld %lx %lx\n",y,x,(long)dpos,(long)dbase));
			
			if (g)   dpos[0]    |= m; else dpos[0]    &= ~m;
			dpos = ptr_add(dpos, pl);
			if ((unsigned long)dpos < (unsigned long)dbase) TRACE(("y=%ld x=%ld %lx %lx\n",y,x,(long)dpos,(long)dbase));
			
			if (r)   dpos[0]    |= m; else dpos[0]    &= ~m;
			dpos = ptr_add(dpos, pl);
			if ((unsigned long)dpos < (unsigned long)dbase) TRACE(("y=%ld x=%ld %lx %lx\n",y,x,(long)dpos,(long)dbase));
			
			if (dim) dpos[0]    |= m; else dpos[0]    &= ~m; 
		}	
		while (cbpos < fwl) 	/* Padding at end of line */
		{
			if (fgetc(fp) == EOF) return -2;
			++cbpos;
		}
		cbpos = 0;
		++y0;
	}
	return 0;
}

/* Load a Windows 3.x DIB-format image. If "mask" is not NULL, then the
 * image is assumed to be in .ico format (height parameter doubled, and a
 * mask following the data).
 * 
 */
int load_win3dib(FILE *fp, MFDB *icon, MFDB *mask)
{
	WORD  step;
	WORD  rv;	
	long  imgbase;
	
	if (fread(&bmhdr, 1, sizeof(bmhdr), fp) < (int)sizeof(bmhdr) 
	||  bmhdr.bm_srecl < 0x28)
	{
		/* Attempt to read the header. The 0x28 check will reject 
		 * OS/2 format DIBs */
		
		TRACE(("EOF reading palette\n"));
		return -2;
	}
	
	/* The BITMAPINFOHEADER might not be exactly 0x28 bytes long. 
	 * Seek to where it really ends. */
	if (bmhdr.bm_srecl > 0x28)
	{
		fseek(fp, bmhdr.bm_srecl - 0x28, SEEK_CUR);
	}
	if (bmhdr.bm_compr) 
	{
		TRACE(("Compressed file, cannot read\n"));
		return -5;	/* Compressed - can't read */
	}

	rv = alloc_palette(); if (rv) return rv;
	
	
	
	/* We are now at the palette. */

	if (pal_len)
	{
		if (fread(palette, 1, pal_len, fp) < pal_len) 
		{
			TRACE(("Premature EOF reading palette\n"));
			return -2;
		}
	}
	
	
	imgbase = ftell(fp);	/* Address of image start */
 	
	if (mask) bmhdr.bm_height /= 2;
	
	/* Now start setting up the MFDBs */

	icon->fwp = bmhdr.bm_width;
	icon->fh  = bmhdr.bm_height;
	icon->fww = (bmhdr.bm_width + 15) / 16;
	icon->ff  = 1;		/* Device-independent colour icons. */

/*
	Always allocate a colour MFDB. Even if the source file _was_ mono,
	it still might be red on blue. Plus, the rest of the code always 
	assumes it has 4 planes to play with; if we only allocate one, then
	memory will get corrupted.
	
	if(bmhdr.bm_cdepth == 1) icon->np = 1;
	else   */                icon->np = 4;
	

	icon->r1 = icon->r2 = icon->r3 = 0L;
	pl = 2L * icon->fww * icon->fh;		/* Length of a plane, bytes */
	

	if (mask)
	{
		memcpy(mask, icon, sizeof(MFDB));
		mask->np = 1; /* Mask is a mono bitmap */
		mask->mp = lalloc(pl * mask->np);
		if (!mask->mp) return -4;
	}
	icon->mp = lalloc(pl * icon->np);

	if (!icon->mp)
	{
		if (mask)    lfree(mask->mp);
		return -4;
	}
	
	/* fwl := Length of a line in the Windows bitmap */

	switch(bmhdr.bm_cdepth)
		{
		case 1:  step = 8; break;
		case 4:  step = 2; break;
		case 8:  step = 1; break;
		case 24: step = 0; break;
		}
	if (step) fwl = (bmhdr.bm_width + (step-1)) / step;
	else      fwl = 3 * bmhdr.bm_width;
	
	fwl = ((fwl + 3) / 4) * 4; /* dword-aligned */

	rv = load_image(fp, icon);

	if (mask && !rv) rv = load_mask(fp, mask);

	if (rv)
	{
		lfree(icon->mp);
		if (mask)    lfree(mask->mp);
		return -2;
	}
	

/* This has loaded the MFDBs in a device-specific format - VGA, 16 
  colours. Transform it to device-independent 4-plane format. 

  If cheating, OTOH, just mark it as device-specific */

#if (!CHEAT)
  
	swapMFDB(icon);
	if (mask) swapMFDB(mask);

#else

	icon->ff = 0;			/* Device-specific format */
	if (mask) mask->ff = 0;
	
#endif

	return 0;
}



/* Convert a Windows .ico file to a GEM MFDB */

int load_ico(FILE *fp, MFDB *icon, MFDB *mask)
{
	WIN31IH header;
	WIN31IR iconrec, bestrec;
	int nic;
	long base;
	
	if (!fp) return -1;
	base = ftell(fp);
	
	if ((fread(&header, 1, sizeof(header), fp) < (int)sizeof(header)) 
	||  header.ih_magic 
	||  header.ih_typ  != 1) return -2;
	memset(&bestrec, 0, sizeof(bestrec));
	for (nic = 0; nic < header.ih_icno; nic++)
	{
		if (fread(&iconrec, 1, sizeof(iconrec), fp) < (int)sizeof(iconrec))
		{
			return -2;
		}
		if (iconrec.ir_cdepth <= 16	/* Search for a 32x32 icon with at */    
		&&  iconrec.ir_width  == 32 /* most 16 colours */
		&&  iconrec.ir_height == 32)
		{
			if (iconrec.ir_cdepth > bestrec.ir_cdepth) 
			{
				memcpy(&bestrec, &iconrec, sizeof(iconrec));
			}
		}
		
	}
	if (!bestrec.ir_cdepth) /* No 32x32x16 icon */
	{
		return -3;
	}
	
	fseek(fp, base + bestrec.ir_offset, SEEK_SET);

	return load_win3dib(fp, icon, mask);
	
}



/* Convert a Windows .ico file to a GEM MFDB */

int load_bmp(FILE *fp, MFDB *bmp)
{
	WIN31BH header;
	
	if (!fp) return -1;
	
	if ((fread(&header, 1, sizeof(header), fp) < (int)sizeof(header)) 
	||  header.bh_type != 0x4D42) return -2;

	return load_win3dib(fp, bmp, NULL);
	
}

