/************************************************************************

    CPPREP v1.00 - Replacement for MODE CON CP PREPARE supporting various
                  file formats.

    Copyright (C) 2006  John Elliott <jce@seasip.demon.co.uk>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*************************************************************************/

/* Load a single codepage from a CPI file. Formats supported are FONT, 
 * FONT.NT and DRFONT. Under DOS, CPX-compressed versions of these are
 * also supported. */
#include "cpprep.h"

static BYTE st_head[23];
static BYTE st_cphead[28];
static WORD glyphs[256];
static BYTE st_intbuf[512];


static FILE *st_fp;
static BYTE far *st_mem;
static BYTE far *st_membase;
static DWORD st_memptr;
static DWORD st_memsize;

static long get_pos(void)
{
	if (st_fp) return ftell(st_fp);
	return st_memptr;
}

static void seekto(long offset, int whence)
{
	if (st_fp) 
	{
		fseek(st_fp, offset, whence);
		return;
	}
	else switch(whence)
	{
		case SEEK_SET: st_memptr  = offset; break;
		case SEEK_CUR: st_memptr += offset; break;
		case SEEK_END: st_memptr  = st_memsize + offset; break;
	}
}


static int readbyte(void)
{
	if (st_fp) return fgetc(st_fp);

	if (st_memptr >= st_memsize) return EOF;
	
	return st_mem[st_memptr++];
}


static int readbytes(void *buf, int size, int count)
{
	int n1, n2, nr;
	BYTE *ptr;

	if (st_fp) return fread(buf, size, count, st_fp);

	ptr = (BYTE *)buf;
	nr = 0;
	for (n1 = 0; n1 < count; n1++)
	{
		for (n2 = 0; n2 < size; n2++)
		{
			int ch = readbyte();
			if (ch == EOF) return nr;
			*ptr++ = ch;
		}
		++nr;
	}
	return nr;
}




/* This CPI parser has to work on CPI files loaded into memory (decompressed 
 * CPXes) and loaded from disc (FONT/FONT.NT/DRFONT). */
static void cpi_parse(char *filename, int page, int *haveFonts, int subtype)
{
	int nsizes;
	int hdrlen;
	long bmpTable;
	WORD cpcount;
	int n, ncp, match, m, o;
	long base;
	long bitBase;	
	int found, wchar, hchar, maxc;
	int ver, subcount, bytes;
	BYTE far *bits;

	match = get_pageno(page);
	/* Nothing in CPI file */
	hdrlen = readbytes(st_head, 1, sizeof(st_head));
       
	if (hdrlen < 18) return;
	if ((st_head[16] == 0 && st_head[17] == 0) || hdrlen < 23)
	{
		fprintf(stderr, "%s: Empty CPI file\n", filename);
		return;
	}
	if (st_head[16] != 1 || st_head[17] != 0 || st_head[18] != 1)
	{
		fprintf(stderr, "%s: Unsupported CPI variant\n", filename);
		return;
	}

	bmpTable = get_pos();
	seekto(PEEK4(st_head + 19), SEEK_SET);
	if (readbytes(st_intbuf, 1, 2) < 2)
	{
		fprintf(stderr, "%s: Empty CPI file\n", filename);
		return;
	}
	cpcount = PEEK2(st_intbuf);
	if (verbose) 
	{
		printf("%s: Loading codepage %d ", filename, match);
		switch(subtype)
		{
			case 0: printf("from standard CPI file\n"); break;
			case 1: printf("from FONT.NT CPI file\n"); break;
			case 2: printf("from DRFONT CPI file\n"); break;
		}
	}
	base = get_pos();
	found = 0;
	for (ncp = 0; ncp < cpcount; ncp++)
	{
		seekto(base, SEEK_SET);
		if (readbytes(&st_cphead, 1, sizeof(st_cphead)) < 
				sizeof(st_cphead))
		{
			fprintf(stderr, "%s: Unexpected EOF\n", filename);
			return;
		}
/* Check that the CPI is a screen CPI for the requested codepage */
		if (PEEK2(st_cphead + 16) != match || PEEK2(st_cphead+6) != 1)
		{
			if (subtype == 1)
				base += PEEK4(st_cphead + 2);
			else	base  = PEEK4(st_cphead + 2);
		}
		else
		{
/* We have a match. */
			found = 1;
			break;
		}
	}
	if (!found)
	{
		fprintf(stderr, "Codepage %d not found in %s\n", 
			match, filename);
		return;
	}
	if (subtype == 1) 
		base += PEEK4(st_cphead + 24);
	else	base  = PEEK4(st_cphead + 24);
	if (readbytes(&st_cphead, 1, 6) < 6)
	{
		fprintf(stderr, "%s: Unexpected EOF\n", filename);
		return;
	}
	ver      = PEEK2(st_cphead);
	subcount = PEEK2(st_cphead+2);
	bytes    = PEEK2(st_cphead+4);
	if (ver == 2) /* DRFONT */
	{
		/* Seek to character bitmaps table */
		long pos = get_pos();
		seekto(bytes, SEEK_CUR);
		if (readbytes(st_intbuf, 1, 512) < 512)
		{
			fprintf(stderr, "%s: Unexpected EOF\n", filename);
			return;
		}
		for (m = 0; m < 256; m++) glyphs[m] = PEEK2(st_intbuf+2*m);
		seekto(pos, SEEK_SET);
	}
	for (ncp = 0; ncp < subcount; ncp++)
	{
		if (readbytes(&st_cphead, 1, 6) < 6)
		{
			fprintf(stderr, "%s: Unexpected EOF\n", filename);
			return;
		}
		hchar = st_cphead[0];
		wchar = st_cphead[1];
		maxc  = PEEK2(st_cphead+4);
		if (wchar <= 8)
			bits = fontbits(page, hchar);
		else	bits = NULL;

		if (bits == NULL)
		{
			if (ver != 2)	/* Skip bitmap if present */
			{
				seekto((((wchar+7)/8)*hchar)*maxc, SEEK_CUR);
			}
			continue;
		}
/* OK, we have a target bitmap */
		if (maxc > 256) maxc = 256;
		if (ver == 2)	/*  DRFONT */
		{
			long save_pos;

			if (subtype != 2)
			{
				fprintf(stderr, "%s: DRFONT record in FONT file\n", filename);
				return;
			}
/* Find the appropriate bitmap table */
			save_pos = get_pos();
			seekto(bmpTable, SEEK_SET);
			nsizes = readbyte();
			for (n = 0; n < nsizes; n++)
			{
				if (readbyte() == hchar) break;
			}
			if (n >= nsizes)
			{
				fprintf(stderr, "%s: Bitmap table for height %d not found\n", filename, hchar);
				return;
			}
			seekto(bmpTable + 1 + nsizes + 4*n, SEEK_SET);
			if (readbytes(&st_intbuf, 1, 4) < 4)
			{
				fprintf(stderr, "%s: Unexpected EOF\n", filename);
				return;
			}
			bitBase = PEEK4(st_intbuf);
/* OK, bitBase is the address of the bitmap table. Now lookup each character
 * shape and copy it across */
			for (m = 0; m < maxc; m++)
			{
				seekto(bitBase + glyphs[m] * hchar, SEEK_SET);
				for (o = 0; o < hchar; o++)
				{
					bits[m*hchar+o] = readbyte();
				}	
			}
			seekto(save_pos, SEEK_SET);
		}
		else
		{
/* Raw codepage is easy: Just read in the bits */
			for (n = 0; n < (hchar * maxc); n++)
			{
				bits[n] = readbyte();
			}
		}
		switch(hchar)
		{
			case  8: *haveFonts |= 1; break;
			case 14: *haveFonts |= 2; break;
			case 16: *haveFonts |= 4; break;
		}
	}

}


void load_cpi (char *filename, FILE *fp, int page, int *haveFonts, int subtype)
{
	st_fp = fp;
	st_mem = NULL;
	cpi_parse(filename, page, haveFonts, subtype);
	st_fp = NULL;
	st_mem = NULL;
}

#ifdef __MSDOS__
#include <dos.h>

void load_cpx(char *filename, FILE *fp, int page, int *haveFonts)
{
	DWORD fsize, n;
	int type, m;
	BYTE magic[8];

	fseek(fp, 0, SEEK_END);
	fsize = ftell(fp);	
	fseek(fp, 0, SEEK_SET);

	st_fp = NULL;
	st_membase = farmalloc(st_memsize = fsize + 65536L);
/* Normalise the base of the memory buffer */
	m = (FP_OFF(st_membase) + 15) / 16;
	m += FP_SEG(st_membase);
	st_mem = MK_FP(m, 0);
	st_memptr = 0;
	if (st_mem == NULL)
	{
		fprintf(stderr, "%s: Could not allocate %ld bytes.\n",
			filename, fsize);
		return;
	}
	for (n = 0; n < fsize; n++)
	{
		st_mem[n] = fgetc(fp);
	}
	if (verbose)
		printf("%s: Decompressing CPX file.\n");
	type = cpx2cpi(st_mem);
	if (type < 0)
	{
		fprintf(stderr, "%s: Could not decompress file.\n",
			filename, fsize);
		farfree(st_membase);
		st_mem = NULL;
		return;
	}
	for (m = 0; m < 8; m++)
		magic[m] = st_mem[m];

	if (!memcmp(magic, "\377FONT   ",      8))
		cpi_parse(filename, page, haveFonts, 0);
	else if (!memcmp(magic, "\377FONT.NT", 8))
		cpi_parse(filename, page, haveFonts, 1);
	else if (!memcmp(magic, "\177DRFONT ", 8))
		cpi_parse(filename, page, haveFonts, 2);
	else
	{
		fprintf(stderr, "%s: CPX file does not contain a recognised "
				"codepage format.\n", filename);
	}
	farfree(st_membase);
	st_fp = NULL;
	st_mem = NULL;
}
#else
void load_cpx(char *filename, FILE *fp, int page, int *haveFonts)
{
	fprintf(stderr, "%s: CPX support is not present\n",
			filename);
}
#endif


