/************************************************************************

    CPPREP v1.00 - Replacement for MODE CON CP PREPARE supporting various
                  file formats.

    Copyright (C) 2006  John Elliott <jce@seasip.demon.co.uk>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*************************************************************************/

/* This file contains the CPI that we are generating and will eventually
 * either write out to disc or inject into DISPLAY.SYS. */

#include "cpprep.h"

/* MSDOS does these with macros */
#ifndef __MSDOS__
unsigned short PEEK2(BYTE *a)
{
	uint16_t b0 = a[1];
	b0 = (b0 << 8) | a[0];
	return b0;
}
unsigned short PEEK4(BYTE *a)
{
	uint32_t b0 = a[3];
	b0 = (b0 << 8) | a[2];
	b0 = (b0 << 8) | a[1];
	b0 = (b0 << 8) | a[0];
	return b0;
}


void POKE2(BYTE *a, WORD b)
{
	a[0] = b & 0xFF;
	a[1] = (b >> 8) & 0xFF;
}


void POKE4(BYTE *a, DWORD b)
{
	a[0] = b & 0xFF;
	a[1] = (b >> 8) & 0xFF;
	a[2] = (b >> 16) & 0xFF;
	a[3] = (b >> 24) & 0xFF;
}


#endif

static WORD st_pagesize;
static WORD st_cpisize;
static BYTE far *st_cpi;
static WORD st_maxf;

/* CPI file header (template) */
static BYTE st_header[] =
{
	0xFF, 'F', 'O', 'N', 'T', ' ', ' ', ' ',
	0,    0,   0,   0,   0,   0,   0,   0,
	1,    0,   1,
	0x17, 0,   0,   0,
	0,    0
};

/* CPI codepage header (template) */
static BYTE st_pagehead[] = 
{
	0x1C, 0x00,					/* 00 */
	0x00, 0x00, 0x00, 0x00,				/* 02 */
	0x01, 0x00,					/* 06 */
	'E',  'G',  'A',  ' ', ' ', ' ', ' ', ' ',	/* 08 */
	0xFF, 0xFF,					/* 10 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00,		/* 12 */
	0x00, 0x00, 0x00, 0x00,				/* 18 */
	0x01, 0x00,					/* 1C */
	0x03, 0x00,					/* 1E */
	0x00, 0x00,					/* 20 */
};

/* Font sizes used in LCD.CPI */
int sizlcd[] = { 8 };
/* Font sizes used in EGA.CPI */
int sizega[] = { 16, 14, 8 };

BYTE far *get_cpi_base()
{
	return st_cpi;
}

WORD get_cpi_len()
{
	return st_cpisize;
}

void free_cpi()
{
	if (st_cpi) farfree(st_cpi);
	st_cpi = NULL;
}

/* Set up an empty CPI file with 'count' pages */
void init_cpi(unsigned count)
{
	unsigned long cpisize;
	unsigned n, m, f;
	unsigned addr, pgaddr;
	int *sizes;

	if (device == 2)	/* LCD */
	{
		st_pagesize = 2048 + 6 + 6 + 28;
		st_maxf = 1;
		sizes = sizlcd;
	}
	else
	{
		st_pagesize = 2048 + 3584 + 4096 + 18 + 6 + 28;
		st_maxf = 3;
		sizes = sizega;
	}
	cpisize = 25L + st_pagesize * (long)count;

	if (cpisize > 0xFFFFL)
	{
		fprintf(stderr, "Cannot load more than 64k of codepage data\n");
		exit(1);
	}
	st_cpi = farmalloc(cpisize);
	if (!st_cpi)
	{
		fprintf(stderr, "Cannot allocate %ld bytes\n", cpisize);
		exit(1);
	}
	st_cpisize = (unsigned)cpisize;
	for (n = 0; n < st_cpisize; n++)
	{
		st_cpi[n] = 0;
	}
	POKE2(st_header + 23, count);
	for (n = 0; n < sizeof(st_header); n++)
	{
		st_cpi[n] = st_header[n];
	}
	for (n = 0; n < count; n++)
	{
		addr = sizeof(st_header) + n * st_pagesize;
		POKE4(st_pagehead +  2, addr + st_pagesize);
		POKE4(st_pagehead + 24, addr + 28);
		if (device == 2) memcpy(st_pagehead + 8, "LCD     ", 8);
		else		 memcpy(st_pagehead + 8, "EGA     ", 8);

		POKE2(st_pagehead + 0x1E, st_maxf);
		POKE2(st_pagehead + 0x20, st_pagesize - 34);
		for (m = 0; m < sizeof(st_pagehead); m++)
			st_cpi[addr + m] = st_pagehead[m];
		pgaddr = addr + 28 + 6;
		for (f = 0; f < st_maxf; f++)
		{
			st_cpi[pgaddr  ] = sizes[f];
			st_cpi[pgaddr+1] = 8;
			st_cpi[pgaddr+2] = 0;
			st_cpi[pgaddr+3] = 0;
			st_cpi[pgaddr+4] = 0;
			st_cpi[pgaddr+5] = 1;
			pgaddr += 6 + 256 * sizes[f];
		}
	}
}

/* Given the index of a codepage (0,1,2...) set its codepage ID. */
void set_pageno(int page, WORD no)
{
	BYTE far *ptr = (&st_cpi[sizeof(st_header) + page * st_pagesize + 16]);
	POKE2(ptr, no);
}

/* Given the index of a codepage (0,1,2...) what is its codepage ID? */
WORD get_pageno(int page)
{
	BYTE far *ptr = (&st_cpi[sizeof(st_header) + page * st_pagesize + 16]);
	return PEEK2(ptr);
}

/* Given the index of a codepage and the subfont height, locate the 
 * bits for that subfont. */
BYTE far *fontbits(int page, int height)
{
	int n;

	unsigned addr = sizeof(st_header) + page * st_pagesize + 28 + 6;

	for (n = 0; n < st_maxf; n++)
	{
		if (st_cpi[addr] == height)
		{
			return &st_cpi[addr + 6];
		}
		addr += 256 * st_cpi[addr] + 6;
	}
	return NULL;
}


/* Write the generated CPI file out to disk. */
void dump_cpi(char *filename)
{
	unsigned n;
	FILE *fp = fopen(filename, "wb");

	if (fp == NULL)
	{
		fprintf(stderr, "Could not open dumpfile: %s\n", filename);
		return;
	}

	for (n = 0; n < st_cpisize; n++)
		fputc(st_cpi[n], fp);
	fclose(fp);
}
