;============================================================================
;
;File:	Z80EM2.ASM - Z80 Emulator
;Date:  February 24, 1987
;
;============================================================================
;
;						      Location
;   Register Usage:	AX = Z80 A-PSW	(AH=F, AL=A)	FF7CH
;			BX = Z80 H-L	(BH=H, BL=L)	FF76H
;			DI = Z80 B-C			FF7AH
;			DX = Z80 D-E	(DH=D, DL=E)	FF78H
;			BP = Z80 SP			
;			SI = Z80 P	(Program Cntr)
;		      	     Z80 IX			FF70H
;			     Z80 IY			FF72H
;
;========================================================================
;
	ORG	80H*PG
ADDB	PROC	NEAR		;80H - ADD B		ADD A,B
;				-----------------------------------------
	XCHG	DI,CX		;Put BC(DI) in CX
	ADD	AL,CH		;Add B(CH) to A(AL)
	LAHF			;Load AH from flags.
	XCHG	DI,CX		;Restore BC(DI) from CX
	JO	ADDB1		;Jump on overflow
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	NXTI
ADDB1	EQU	$
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,004H		;set parity/overflow flag
	NXTI
ADDB	ENDP
;
	ORG	81H*PG
ADDC	PROC	NEAR		;81H - ADD C		ADD A,C
;				-----------------------------------------
	XCHG	DI,CX		;Put BC(DI) in CX
	ADD	AL,CL		;Add C(CL) to A(AL)
	LAHF			;Load AH from flags
	XCHG	DI,CX		;Restore BC(DI) from CX
	JO	ADDC1		;Jump on overflow
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	NXTI
ADDC1	EQU	$
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,004H		;set parity/overflow flag
	NXTI
ADDC	ENDP
;
	ORG	82H*PG
ADDD	PROC	NEAR		;82H - ADD D		ADD A,D
;				-----------------------------------------
	ADD	AL,DH		;Add D(DH) to A(AL)
	LAHF			;Load AH from flags
	JO	ADDD1		;Jump on overflow
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	NXTI
ADDD1	EQU	$
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,004H		;set parity/overflow flag
	NXTI
ADDD	ENDP
;
	ORG	83H*PG
ADDE	PROC	NEAR		;83H - ADD E		ADD A,E
;				------------------------------------------
	ADD	AL,DL		;Add E(DL) to A(AL)
	LAHF			;Load AH from flags
	JO	ADDE1		;Jump on overflow
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	NXTI
ADDE1	EQU	$
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,004H		;set parity/overflow flag
	NXTI
ADDE	ENDP
;
	ORG	84H*PG
ADDH	PROC	NEAR		;84H - ADD H		ADD A,H
;				-----------------------------------------
	ADD	AL,BH		;Add H(BH) to A(AL)
	LAHF			;Load AH from flags
	JO	ADDH1		;Jump on overflow
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	NXTI
ADDH1	EQU	$
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,004H		;set parity/overflow flag
	NXTI
ADDH	ENDP
;
	ORG	85H*PG
ADDL	PROC	NEAR		;85H - ADD L		ADD A,L
;				-----------------------------------------
	ADD	AL,BL		;Add L(BL) to A(AL)
	LAHF			;Load AH from flags
	JO	ADDL1		;Jump on overflow
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	NXTI
ADDL1	EQU	$
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,004H		;set parity/overflow flag
	NXTI
ADDL	ENDP
;
	ORG	86H*PG
ADDM	PROC	NEAR		;86H - ADD M		ADD A,(HL)
;				------------------------------------------
	ADD	AL,[BX]		;Add byte pointed to by HL(BX) to A(AL)
	LAHF			;Load AH from flags
	JO	ADDM1		;Jump on overflow
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	NXTI
ADDM1	EQU	$
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,004H		;set parity/overflow flag
	NXTI
ADDM	ENDP
;
	ORG	87H*PG
ADDA	PROC	NEAR		;87H - ADD A		ADD A,A
;				-----------------------------------------
	ADD	AL,AL		;Add A(AL) to A(AL)
	LAHF			;Load AH from flags
	JO	ADDA1		;Jump on overflow
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	NXTI
ADDA1	EQU	$
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,004H		;set parity/overflow flag
	NXTI
ADDA	ENDP
;
	ORG	88H*PG
ADCB	PROC	NEAR		;88H - ADC B		ADC A,B
;				------------------------------------------
	SAHF			;Store AH in flags
	XCHG	DI,CX		;Put BC(DI) in CX
	ADC	AL,CH		;Add with carry B(CH) to A(AL)
	LAHF			;Load AH from flags
	XCHG	DI,CX		;Restore BC(DI) from CX
	JO	ADCB1		;Jump on overflow
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	NXTI
ADCB1	EQU	$
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,004H		;set parity/overflow flag
	NXTI
ADCB	ENDP
;
	ORG	89H*PG
ADCC	PROC	NEAR		;89H - ADC C		ADC A,C
;				----------------------------------------
	SAHF			;Store AH in flags
	XCHG	DI,CX		;Put BC(DI) in CX
	ADC	AL,CL		;Add with carry C(CL) to A(AL)
	LAHF			;Load AH from flags
	XCHG	DI,CX		;Restore BC(DI) from CX
	JO	ADCC1		;Jump on overflow
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	NXTI
ADCC1	EQU	$
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,004H		;set parity/overflow flag
	NXTI
ADCC	ENDP
;
	ORG	8AH*PG
ADCD	PROC	NEAR		;8AH - ADC D		ADC A,D
;				----------------------------------------
	SAHF			;Store AH in flags
	ADC	AL,DH		;Add with carry D(DH) to A(AL)
	LAHF			;Load AH from flags
	JO	ADCD1		;Jump on overflow
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	NXTI
ADCD1	EQU	$
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,004H		;set parity/overflow flag
	NXTI
ADCD	ENDP
;
	ORG	8BH*PG
ADCE	PROC	NEAR		;8BH - ADC E		ADC A,E
;				----------------------------------------
	SAHF			;Store AH in flags
	ADC	AL,DL		;Add with carry E(DL) to A(AL)
	LAHF			;Load AH from flags
	JO	ADCE1		;Jump on overflow
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	NXTI
ADCE1	EQU	$
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,004H		;set parity/overflow flag
	NXTI
ADCE	ENDP
;
	ORG	8CH*PG
ADCH	PROC	NEAR		;8CH - ADC H		ADC A,H
;				------------------------------------------
	SAHF			;Store AH in flags
	ADC	AL,BH		;Add with carry H(BH) to A(AL)
	LAHF			;Load AH from flags
	JO	ADCH1		;jump on overflow
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	NXTI
ADCH1	EQU	$
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,004H		;set overflow flag
	NXTI
ADCH	ENDP
;
	ORG	8DH*PG
ADCL	PROC	NEAR		;8DH - ADC L		ADC A,L
;				-----------------------------------------
	SAHF			;Store AH in flags
	ADC	AL,BL		;Add with carry L(BL) to A(AL)
	LAHF			;Load AH from flags
	JO	ADCL1		;Jump on overflow
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	NXTI
ADCL1	EQU	$
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,004H		;set parity/overflow flag
	NXTI
ADCL	ENDP
;
	ORG	8EH*PG
ADCM	PROC	NEAR		;8EH - ADC M		ADC A,(HL)
;				----------------------------------------
	SAHF			;Save AH in flags
	ADC	AL,[BX]		;Add with carry byte via HL(BX) to A(AL)
	LAHF			;Load AH from flags
	JO	ADCM1		;Jump on overflow
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	NXTI
ADCM1	EQU	$
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,004H		;set parity/overflow flag
	NXTI
ADCM	ENDP
;
	ORG 	8FH*PG
ADCA	PROC	NEAR		;8FH - ADC A		ADC A,A
;				-----------------------------------------
	SAHF			;Store AH in flags
	ADC	AL,AL		;Add with carry A(AL) to A(AL)
	LAHF			;Load AH from flags
	JO	ADCA1		;Jump on overflow
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	NXTI
ADCA1	EQU	$
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,004H		;set parity/overflow flag
	NXTI
ADCA	ENDP
;
	ORG	90H*PG
SUBB	PROC	NEAR		;90H - SUB B		SUB B
;				-----------------------------------------
	XCHG	DI,CX		;Put BC(DI) in CX
	SUB	AL,CH		;Subtract B(CH) from A(AL)
	LAHF			;Load AH from flags
	XCHG	DI,CX		;Restore BC(DI) from CX
	JO	SUBB1		;Jump on overflow
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,002H		;set subtract flag
	NXTI
SUBB1	EQU	$
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,006H		;set parity/overflow and subtract flags
	NXTI
SUBB	ENDP
;
	ORG	91H*PG
SUBC	PROC	NEAR		;91H - SUB C		SUB C
;				------------------------------------------
	XCHG	DI,CX		;Put BC(DI) in CX
	SUB	AL,CL		;Subtract C(CL) from A(AL)
	LAHF			;Load AH from flags
	XCHG	DI,CX		;Restore BC(DI) from CX
	JO	SUBC1		;Jump on overflow
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,002H		;set subtract flag
	NXTI
SUBC1	EQU	$
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,006H		;set parity/overflow and subtract flags
	NXTI
SUBC	ENDP
;
	ORG	92H*PG
SUBD	PROC	NEAR		;92H - SUB D		SUB D
;				-----------------------------------------
	SUB	AL,DH		;Subtract D(DH) from A(AL)
	LAHF			;Load AH from flags
	JO	SUBD1		;Jump on overflow
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,002H		;set subtract flag
	NXTI
SUBD1	EQU	$
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,006H		;set parity/overflow and subtract flags
	NXTI
SUBD	ENDP
;
	ORG	93H*PG
SUBE	PROC	NEAR		;93H - SUB E		SUB E
;				-------------------------------------------
	SUB	AL,DL		;Subtract E(DL) from A(AL)
	LAHF			;Load AH from flags
	JO	SUBE1		;Jump on overflow
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,002H		;set subtract flag
	NXTI
SUBE1	EQU	$
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,006H		;set parity/overflow and subtract flags
	NXTI
SUBE	ENDP
;
	ORG	94H*PG
SUBH	PROC	NEAR		;94H - SUB H		SUB H
;				------------------------------------------
	SUB	AL,BH		;Subtract H(BH) from A(AL)
	LAHF			;Load AH from flags
	JO	SUBH1		;Jump on overflow
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,002H		;set subtract flag
	NXTI
SUBH1	EQU	$
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,006H		;set parity/overflow and subtract flags
	NXTI
SUBH	ENDP
;
	ORG	95H*PG
SUBL	PROC	NEAR		;95H - SUB L		SUB L
;				-----------------------------------------
	SUB	AL,BL		;Subtract L(BL) from A(AL)
	LAHF			;Load AH from flags
	JO	SUBL1		;Jump on overflow
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,002H		;set subtract flag
	NXTI
SUBL1	EQU	$
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,006H		;set parity/operflow and subtract flags
	NXTI
SUBL	ENDP
;
	ORG	96H*PG
SUBM	PROC	NEAR		;96H - SUB M		SUB (HL)
;				----------------------------------------
	SUB	AL,[BX]		;Subtract byte ptd to by HL(BX) from A(AL)
	LAHF			;Load AH from flags
	JO	SUBM1		;Jump on overflow
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,002H		;set subtract flag
	NXTI
SUBM1	EQU	$
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,006H		;set parity/overflow and subtract flags
	NXTI
SUBM	ENDP
;
	ORG	97H*PG
SUBA	PROC	NEAR		;97H - SUB A		SUB A
;				-----------------------------------------
	SUB	AL,AL		;Subtract A(AL) from A(AL)
	LAHF			;Load AH from flags
	JO	SUBA1		;Jump on overflow
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,002H		;set subtract flag
	NXTI
SUBA1	EQU	$
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,006H		;set parity/overflow and subtract flags
	NXTI
SUBA	ENDP
;
	ORG	98H*PG
SBBB	PROC	NEAR		;98H - SBB B		SBC A,B
;				-----------------------------------------
	SAHF			;Store AH in flags
	XCHG	DI,CX		;Put BC(DI) in CX
	SBB	AL,CH		;Subtract w/borrow B(CH) from A(AL)
	LAHF			;Load AH from flags
	XCHG	DI,CX		;Restore BC(DI) from CX
	JO	SBBB1		;Jump on overflow
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,002H		;set subtract flag
	NXTI
SBBB1	EQU	$
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,006H		;set parity/overflow and subtract flags
	NXTI
SBBB	ENDP
;
	ORG	99H*PG
SBBC	PROC	NEAR		;99H - SBB C		SBC A,C
;				------------------------------------------
	SAHF			;Store AH in flags
	XCHG	DI,CX		;Put BC(DI) in CX
	SBB	AL,CL		;Subtract w/borrow C(CL)from A(AL)
	LAHF			;Load AH from flags
	XCHG	DI,CX		;Restore BC(DI) from CX
	JO	SBBC1		;Jump on overflow
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,002H		;set subtract flag
	NXTI
SBBC1	EQU	$
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,006H		;set parity/overflow and subtract flags
	NXTI
SBBC	ENDP
;
	ORG	9AH*PG
SBBD	PROC	NEAR		;9AH - SBB D		SBC A,D
;				------------------------------------------
	SAHF			;Store AH in flags
	SBB	AL,DH		;Subtract w/borrow D(DH) from A(AL)
	LAHF			;Load AH from flags
	JO	SBBD1		;Jump on overflow
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,002H		;set subtract flag
	NXTI
SBBD1	EQU	$
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,006H		;set parity/overflow and subtract flags
	NXTI
SBBD	ENDP
;
	ORG	9BH*PG
SBBE	PROC	NEAR		;9BH - SBB E		SBC A,E
;				-------------------------------------------
	SAHF			;Store AH in flags
	SBB	AL,DL		;Subtract w/borrow E(DL) from A(AL)
	LAHF			;Load AH from flags
	JO	SBBE1		;Jump on overflow
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,002H		;set subtract flag
	NXTI
SBBE1	EQU	$
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,006H		;set parity/overflow and subtract flags
	NXTI
SBBE	ENDP
;
	ORG	9CH*PG
SBBH	PROC	NEAR		;9CH - SBB H		SBC A,H
;				-------------------------------------------
	SAHF			;Store AH in flags
	SBB	AL,BH		;Subtract w/borrow H(BH) from A(AL)
	LAHF			;Load AH from flags
	JO	SBBH1		;Jump on overflow
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,002H		;set subtract flag
	NXTI
SBBH1	EQU	$
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,006H		;set parity/overflow and subtract flags
	NXTI
SBBH	ENDP
;
	ORG	9DH*PG
SBBL	PROC	NEAR		;9DH - SBB L		SBC A,L
;				--------------------------------------------
	SAHF			;Store AH in flags
	SBB	AL,BL		;Subtract w/borrow L(BL) from A(AL)
	LAHF			;Load AH from flags
	JO	SBBL1		;Jump on overflow
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,002H		;set subtract flag
	NXTI
SBBL1	EQU	$
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,006H		;set parity/overflow and subtract flags
	NXTI
SBBL	ENDP
;
	ORG	9EH*PG
SBBM	PROC	NEAR		;9EH - SBB M		SBC A,(HL)
;				-------------------------------------------
	SAHF			;Store AH in flags
	SBB	AL,[BX]		;Subtract w/borrow byte via HL(BX) fr A(AL)
	LAHF			;Load AH from flags
	JO	SBBM1		;Jump on overflow
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,002H		;set subtract flag
	NXTI
SBBM1	EQU	$
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,006H		;set parity/overflow and subtract flags
	NXTI
SBBM	ENDP
;
	ORG	9FH*PG
SBBA	PROC	NEAR		;9FH - SBB A		SBC A,A
;				-------------------------------------------
	SAHF			;Store AH in flags
	SBB	AL,AL		;Subtract w/borrow A(AL) from A(AL)
	LAHF			;Load AH from flags
	JO	SBBA1		;Jump on overflow
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,002H		;set subtract flag
	NXTI
SBBA1	EQU	$
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,006H		;set parity/overflow and subtract flags
	NXTI
SBBA	ENDP
;	
	ORG	0A0H*PG
ANAB	PROC	NEAR		;A0H - ANA B		AND B
;				-----------------------------------------
	XCHG	DI,CX		;Put BC(DI) in CX
	AND	AL,CH		;AND B(CH) to A(AL)
	LAHF			;Load AH from flags
	XCHG	DI,CX		;Restore BC(DI) from CX
	AND	AH,0ECH		;clear aux carry, subtract and carry flags
	OR	AH,010H		;set aux carry flag
	NXTI
ANAB	ENDP
;
	ORG	0A1H*PG
ANAC	PROC	NEAR		;A1H - ANA C		AND C
;				------------------------------------------
	XCHG	DI,CX		;Put BC(DI) in CX
	AND	AL,CL		;AND C(CL) to A(AL)
	LAHF			;Load AH from flags
	XCHG	DI,CX		;Restore BC(DI) from CX
	AND	AH,0ECH		;clear aux carry, subtract and carry flags
	OR	AH,010H		;set aux carry flag
	NXTI
ANAC	ENDP
;
	ORG	0A2H*PG
ANAD	PROC	NEAR		;A2H - ANA D		AND D
;				------------------------------------------
	AND	AL,DH		;AND D(DH) to A(AL)
	LAHF			;Load AH from flags
	AND	AH,0ECH		;clear aux carry, subtract and carry flags
	OR	AH,010H		;set aux carry flag
	NXTI
ANAD	ENDP
;
	ORG	0A3H*PG
ANAE	PROC	NEAR		;A3H - ANA E		AND E
;				------------------------------------------
	AND	AL,DL		;AND E(DL) to A(AL)
	LAHF			;Load AH from flags
	AND	AH,0ECH		;clear aux carry, subtract and carry flags
	OR	AH,010H		;set aux carry flag
	NXTI
ANAE	ENDP
;
	ORG	0A4H*PG
ANAH	PROC	NEAR		;A4H - ANA H		AND H
;				-----------------------------------------
	AND	AL,BH		;AND H(BH) to A(AL)
	LAHF			;Load AH from flags
	AND 	AH,0ECH		;clear aux carry, subtract and carry flags
	OR	AH,010H		;set aux carry flag
	NXTI
ANAH	ENDP
;
	ORG	0A5H*PG
ANAL	PROC	NEAR		;A5H - ANA L		AND L
;				-----------------------------------------
	AND	AL,BL		;AND L(BL) to A(AL)
	LAHF			;Load AH from flags
	AND	AH,0ECH		;clear aux carry, subtract and carry flags
	OR	AH,010H		;set aux carry flag
	NXTI
ANAL	ENDP
;
	ORG	0A6H*PG
ANAM	PROC	NEAR		;A6H - ANA M		AND (HL)
;				------------------------------------------
	AND	AL,[BX]		;AND byte via HL(BX) to A(AL)
	LAHF			;Load AH from flags
	AND	AH,0ECH		;clear aux carry, subtract and carry flags
	OR	AH,010H		;set aux carry flag
	NXTI
ANAM	ENDP
;
	ORG 	0A7H*PG
ANAA	PROC	NEAR		;A7H - ANA A		AND A
;				------------------------------------------
	AND	AL,AL		;AND A(AL) to A(AL)
	LAHF			;Load AH from flags
	AND	AH,0ECH		;clear aux carry, subtract and carry flags
	OR	AH,010H		;set aux carry flag
	NXTI
ANAA	ENDP
;
	ORG	0A8H*PG
XRAB	PROC	NEAR		;A8H - XRA B		XOR B
;				------------------------------------------
	XCHG	DI,CX		;Put BC(DI) in CX
	XOR	AL,CH		;XOR B(CH) to A(AL)
	LAHF			;Load AH from flags
	XCHG	DI,CX		;Restore BC(DI) from CX
	AND	AH,0ECH		;clear aux carry, subtract and carry flags
	NXTI
XRAB	ENDP
;
	ORG	0A9H*PG
XRAC	PROC	NEAR		;A9H - XRA C		XOR C
;				------------------------------------------
	XCHG	DI,CX		;Put BC(DI) in CX
	XOR	AL,CL		;XOR C(CL) to A(AL)
	LAHF			;Load AH from flags
	XCHG	DI,CX		;Restore BC(DI) from CX
	AND	AH,0ECH		;clear aux carry, subtract and carry flags
	NXTI
XRAC	ENDP
;
	ORG	0AAH*PG
XRAD	PROC	NEAR		;AAH - XRA D		XOR D
;				-----------------------------------------
	XOR	AL,DH		;XOR D(DH) to A(AL)
	LAHF			;Load AH from flags
	AND	AH,0ECH		;clear aux carry, subtract and carry flags
	NXTI
XRAD	ENDP
;
	ORG	0ABH*PG
XRAE	PROC	NEAR		;ABH - XRA E		XOR E
;				------------------------------------------
	XOR	AL,DL		;XOR E(DL) to A(AL)
	LAHF			;Load AH from flags
	AND	AH,0ECH		;clear aux carry, subtract and carry flags
	NXTI
XRAE	ENDP
;
	ORG	0ACH*PG
XRAH	PROC	NEAR		;ACH - XRA H		XOR H
;				------------------------------------------
	XOR	AL,BH		;XOR H(BH) to A(AL)
	LAHF			;Load AH from flags
	AND	AH,0ECH		;clear aux carry, subtract and carry flags
	NXTI
XRAH	ENDP
;
	ORG	0ADH*PG
XRAL	PROC	NEAR		;ADH - XRA L		XOR L
;				-----------------------------------------
	XOR	AL,BL		;XOR L(BL) to A(AL)
	LAHF			;Load AH from flags
	AND	AH,0ECH		;clear aux carry, subtract and carry flags
	NXTI
XRAL	ENDP
;
	ORG	0AEH*PG
XRAM	PROC	NEAR		;AEH - XRA M		XOR (HL)
;				------------------------------------------
	XOR	AL,[BX]		;XOR from byte via HL(BX) to A(AL)
	LAHF			;Load AH from flags
	AND	AH,0ECH		;clear aux carry, subtract and carry flags
	NXTI
XRAM	ENDP
;
	ORG	0AFH*PG
XRAA	PROC	NEAR		;AFH - XRA A		XOR A
;				------------------------------------------
	XOR	AL,AL		;XOR A(AL) to A(AL)
	LAHF			;Load AH from flags
	AND	AH,0ECH		;clear aux carry, subtract and carry flags
	NXTI
XRAA	ENDP
;
	ORG	0B0H*PG
ORAB	PROC	NEAR		;B0H - ORA B		OR B
;				------------------------------------------
	XCHG	DI,CX		;Put BC(DI) in CX
	OR	AL,CH		;OR B(CH) to A(AL)
	LAHF			;Load AH from flags
	XCHG	DI,CX		;Restore BC(DI) from CX
	AND	AH,0ECH		;clear aux carry, subtract and carry flags
	NXTI
ORAB	ENDP
;
	ORG	0B1H*PG
ORAC	PROC	NEAR		;B1H - ORA C		OR C
;				------------------------------------------
	XCHG	DI,CX		;Put BC(DI) in CX
	OR	AL,CL		;OR C(CL) to A(AL)
	LAHF			;Load AH from flags
	XCHG	DI,CX		;Restore BC(DI) from CX
	AND	AH,0ECH		;clear aux carry, subtract and carry flags
	NXTI
ORAC	ENDP
;
	ORG	0B2H*PG
ORAD	PROC	NEAR		;B2H - ORA D		OR D
;				------------------------------------------
	OR	AL,DH		;OR D(DH) to A(AL)
	LAHF			;Load AH from flags
	AND	AH,0ECH		;clear aux carry, subtract and carry flags
	NXTI
ORAD	ENDP
;
	ORG	0B3H*PG
ORAE	PROC	NEAR		;B3H - ORA E		OR E
;				------------------------------------------
	OR	AL,DL		;OR E(DL) to A(AL)
	LAHF			;Load AH from flags
	AND	AH,0ECH		;clear aux carry, subtract and carry flags
	NXTI
ORAE	ENDP
;
	ORG	0B4H*PG
ORAH	PROC	NEAR		;B4H - ORA H		OR H
;				------------------------------------------
	OR	AL,BH		;OR H(BH) to A(AL)
	LAHF			;Load AH from flags
	AND	AH,0ECH		;clear aux carry, subtract and carry flags
	NXTI
ORAH	ENDP
;
	ORG	0B5H*PG
ORAL	PROC	NEAR		;B5H - ORA L		OR L
;				------------------------------------------
	OR	AL,BL		;OR L(BL) to A(AL)
	LAHF			;Load AH from flags
	AND	AH,0ECH		;clear aux carry, subtract and carry flags
	NXTI
ORAL	ENDP
;
	ORG	0B6H*PG
ORAM	PROC	NEAR		;B6H - ORA M		OR (HL)
;				-----------------------------------------
	OR	AL,[BX]		;OR byte via HL(BX) to A(AL)
	LAHF			;Load AH from flags
	AND	AH,0ECH		;clear aux carry, subtract and carry flags
	NXTI
ORAM	ENDP
;
	ORG	0B7H*PG
ORAA	PROC	NEAR		;B7H - ORA A		OR A
;				-----------------------------------------
	OR	AL,AL		;OR A(AL) to A(AL)
	LAHF			;Load AH from flags
	AND	AH,0ECH		;clear aux carry, subtract and carry flags
	NXTI
ORAA	ENDP
;
	ORG	0B8H*PG
CMPB	PROC	NEAR		;B8H - CMP B		CP B
;				-----------------------------------------
	XCHG	DI,CX		;Put BC(DI) in CX
	CMP	AL,CH		;Compare B(CH) with A(AL)
	LAHF			;Load AH from flags
	XCHG	DI,CX		;Restore BC(DI) from CX
	JO	CMPB1		;Jump on overflow
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,002H		;set subtract flag
	NXTI
CMPB1	EQU	$
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,006H		;set parity/overflow and subtract flags
	NXTI
CMPB	ENDP
;
	ORG	0B9H*PG
CMPC	PROC	NEAR		;B9H - CMP C		CP C
;				-----------------------------------------
	XCHG	DI,CX		;Put BC(DI) in CX
	CMP	AL,CL		;Compare C(CL) with A(AL)
	LAHF			;Load AH from flags
	XCHG	DI,CX		;Restore BC(DI) from CX
	JO	CMPC1		;Jump on overflow
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,002H		;set subtract flag
	NXTI
CMPC1	EQU	$
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,006H		;set parity/overflow and subtract flags
	NXTI
CMPC	ENDP
;
	ORG	0BAH*PG
CMPD	PROC	NEAR		;BAH - CMP D		CP D
;				-----------------------------------------
	CMP	AL,DH		;Compare D(DH) with A(AL)
	LAHF			;Load AH from flags
	JO	CMPD1		;Jump on overflow
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,002H		;set subtract flag
	NXTI
CMPD1	EQU	$
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,006H		;set parity/overflow and subtract flags
	NXTI
CMPD	ENDP
;
	ORG	0BBH*PG
CMPE	PROC	NEAR		;BBH - CMP E		CP E
;				----------------------------------------
	CMP	AL,DL		;Compare E(DL) with A(AL)
	LAHF			;Load AH from flags
	JO	CMPE1		;Jump on overflow
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,002H		;set subtract flag
	NXTI
CMPE1	EQU	$
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,006H		;set parity/overflow and subtract flags
	NXTI
CMPE	ENDP
;
	ORG	0BCH*PG
CMPH	PROC	NEAR		;BCH - CMP H		CP H
;				-----------------------------------------
	CMP	AL,BH		;Compare H(BH) with A(AL)
	LAHF			;Load AH from flags
	JO	CMPH1		;Jump on overflow
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,002H		;set subtract flag
	NXTI
CMPH1	EQU	$
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,006H		;set parity/overflow and subtract flags
	NXTI
CMPH	ENDP
;
	ORG	0BDH*PG
CMPL	PROC	NEAR		;BDH - CMP L		CP L
;				-----------------------------------------
	CMP	AL,BL		;Compare L(BL) with A(AL)
	LAHF			;Load AH from flags
	JO	CMPL1		;Jump on overflow
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,002H		;set subtract flag
	NXTI
CMPL1	EQU	$
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,006H		;set parity/overflow and subtract flags
	NXTI
CMPL	ENDP
;
	ORG	0BEH*PG
CMPM	PROC	NEAR		;BEH - CMP M		CP (HL)
;				-----------------------------------------
	CMP	AL,[BX]		;Compare byte ptd to by HL(BX) with A(AL)
	LAHF			;Load AH from flags
	JO	CMPM1		;Jump on overflow
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,002H		;set subtract flag
	NXTI
CMPM1	EQU	$
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,006H		;set parity/overflow and subtract flags
	NXTI
CMPM	ENDP
;
	ORG	0BFH*PG
CMPA	PROC	NEAR		;BFH - CMP A		CP A
;				------------------------------------------
	CMP	AL,AL		;Compare A(AL) with A(AL)
	LAHF			;Load AH from flags
	JO	CMPA1		;Jump on overflow
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,002H		;set subtract flag
	NXTI
CMPA1	EQU	$
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,006H		;set parity/overflow and subtract flags
	NXTI
CMPA	ENDP
; 
	ORG	0C0H*PG
RNZ	PROC	NEAR		;C0H - RNZ		RET NZ
;				-----------------------------------------
	SAHF			;Store AH in flags
	JNZ	RNZ1		;Jump if non-zero
	NXTI			;Otherwise, fall thru to next instruction
RNZ1	EQU	$
	MOV	SI,DS:[BP+00H]	;Get return address in P(SI) from stack.
	INC	BP		;to pop the stack
	INC	BP
	NXTI			;return and get next instruction
RNZ	ENDP
;
	ORG	0C1H*PG
POPB	PROC	NEAR		;C1H - POP B		POP BC
;				----------------------------------------
	MOV	DI,DS:[BP+00H]	;Load DI(BC) from the stack.
	INC	BP		;pop the stack
	INC	BP
	NXTI
POPB	ENDP
;
	ORG	0C2H*PG
JNZ80	PROC	NEAR		;C2H - JNZ Addr		JP NZ Addr
;				----------------------------------------
	SAHF			;Store AH in flags
	JZ	JNZ1		;jump if zero (fall thru to next instr)
	MOV	SI,[SI]		;otherwise set P(SI) to JNZ address
	NXTI			;and get next instruction there
JNZ1	EQU	$
	INC	SI		;bump P(SI) over jump address
	INC	SI
	NXTI			;and fall through to next instruction
JNZ80	ENDP
;
	ORG	0C3H*PG
JMP80	PROC	NEAR		;C3H - JMP Addr		JP Addr
;				-----------------------------------------
	MOV	SI,[SI]		;Set P(SI) to JMP address
	NXTI			;and get next instruction there
JMP80	ENDP
;
	ORG	0C4H*PG
CNZ	PROC	NEAR		;C4H - CNZ Addr		CALL NZ,Addr
;				-----------------------------------------
	SAHF			;Store AH in flags
	JZ	CNZ1		;Jump on zero (fall thru to next instr)
	XCHG	CX,AX		;Save PSW(AX) in CX
	LODSW			;Get the Addr to call
	DEC	BP		;Push the stack
	DEC	BP
	MOV	DS:[BP+00H],SI	;save return address on the stack
	MOV	SI,AX		;Set P(SI) to the call address
	XCHG	CX,AX		;Restore PSW(AX) from CX
	XOR 	CL,CL
	NXTI			;get next instruction from called address
CNZ1	EQU	$
	INC	SI		;increment P(SI) past address bytes
	INC	SI
	NXTI			;fall through to get next instruction
CNZ	ENDP
;
	ORG	0C5H*PG
PUSHB	PROC	NEAR		;C5H - PUSH B		PUSH BC
;				-----------------------------------------
	DEC	BP		;Push the stack
	DEC	BP
	MOV	DS:[BP+00H],DI	;put BC(DI) on the stack
	NXTI
PUSHB	ENDP
;
	ORG	0C6H*PG
ADI	PROC	NEAR		;C6H - ADI D8		ADD A,D8
;				-----------------------------------------
	ADD	AL,[SI]		;Add immediate data byte to A(AL)
	LAHF			;Load AH from flags
	JO	ADI1		;Jump on overflow
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	INC	SI		;bump P past immediate data byte
	NXTI
ADI1	EQU	$
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,004H		;set parity/overflow flag
	INC	SI		;bump P past immediate data byte
	NXTI
ADI	ENDP
;
	ORG	0C7H*PG
RST0	PROC	NEAR		;C7H - RST 0		RST 0
;				-----------------------------------------
	DEC	BP		;Push the stack
	DEC	BP
	MOV	DS:[BP+00H],SI	;put P(SI) return address on the stack
	MOV	SI,0000H	;set P(SI) to 0000H
	NXTI			;get next instruction from 0000H
RST0	ENDP
;
	ORG	0C8H*PG
RZ	PROC	NEAR		;C8H - RZ		RET Z
;				-----------------------------------------
	SAHF			;Store AH in flags
	JZ	RZ1		;Jump if zero (do return on zero)
	NXTI			;Otherwise, fall thru to next instruction
RZ1	EQU	$
	MOV	SI,DS:[BP+00H]	;Get return address in P(SI) from stack.
	INC	BP		;Pop the stack
	INC	BP
	NXTI			;get next instruction from return location
RZ	ENDP
;
	ORG	0C9H*PG
RET80	PROC	NEAR		;C9H - RET		RET
;				-----------------------------------------
	MOV	SI,DS:[BP+00H]	;Put return address in P(SI) from stack.
	INC	BP		;pop the stack
	INC	BP
	NXTI			;get next instruction from return location
RET80	ENDP
;
	ORG	0CAH*PG
JZ80	PROC	NEAR		;CAH - JZ,Addr		JP Z,Addr
;				------------------------------------------
	SAHF			;Store AH in flags
	JNZ	JZ1		;Jump on non-zero (fall thru to next instr)
	MOV	SI,[SI]		;Put jump address in P(SI)
	NXTI			;Get next instruction from jump address.
JZ1	EQU	$
	INC	SI		;bump P(SI) over the jump address
	INC	SI
	NXTI			;fall through to next instruction
JZ80	ENDP
;
	ORG	0CBH*PG
CBESC	PROC	NEAR		;CBH - (none)		Z80 Escape
;				-------------------------------------------
	MOV	CH,[SI]		;Get next byte of Op Code
	MOV	CL,CBOP		;CB operation offset
	INC	SI		;Bump P(SI) to next instruction
	JMP	CX		;Go process CB operation.
CBESC	ENDP
;
	ORG	0CCH*PG
CZ	PROC	NEAR		;CCH - CZ Addr		CALL Z,Addr
;				------------------------------------------				
	SAHF			;Store AH in flags
	JNZ	CZ1		;Jump on non-zero to fall thru to next instr.
	XCHG	CX,AX		;Save PSW(AX) in CX
	LODSW			;Get the call address
	DEC	BP		;push the stack
	DEC	BP
	MOV	DS:[BP+00H],SI	;put P(SI) return address on the stack
	MOV	SI,AX		;get call address in P(SI)
	XCHG	CX,AX		;Restore PSW(AX) from CX
	XOR	CL,CL
	NXTI			;get next instruction from call address
CZ1	EQU	$
	INC	SI		;bump P(SI) over the call address
	INC	SI
	NXTI			;fall through to the next instruction
CZ	ENDP
;
	ORG	0CDH*PG
CALL80	PROC	NEAR		;CDH - CALL Addr	CALL Addr
;				--------------------------------------------
	XCHG	CX,AX		;Save PSW(AX) in CX
	LODSW			;Get the call address
	DEC	BP		;push the stack
	DEC	BP
	MOV	DS:[BP+00H],SI	;put P(SI) return address on the stack
	MOV	SI,AX		;get call address in P(SI)
	XCHG	CX,AX		;Restore PSW(AX) from CX
	XOR	CL,CL
	NXTI			;get next instruction from call address
CALL80	ENDP
;
	ORG	0CEH*PG
ACI	PROC	NEAR		;CEH - ACI D8		ADC A,D8
;				--------------------------------------------
	SAHF			;Store AH in flags
	ADC	AL,[SI]		;Add with carry immediate byte to A(AL)
	LAHF			;Load AH from flags
	JO	ACI1		;Jump on overflow
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	INC	SI		;bump P(SI) over the immediate data byte
	NXTI			;get next instruction
ACI1	EQU	$
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,004H		;set the parity/overflow flag
	INC	SI		;bump P(SI) over the immediate data byte
	NXTI			;get next instruction
ACI	ENDP
;
	ORG	0CFH*PG
RST1	PROC	NEAR		;CFH - RST 1		RST 8
;				-------------------------------------------
	DEC	BP		;push the stack pointer
	DEC	BP
	MOV	DS:[BP+00H],SI	;put P(SI) return address on the stack
	MOV	SI,0008H	;put 0008H location in P(SI)
	NXTI			;get next instruction from 0008H in TPA
RST1	ENDP
;
	ORG	0D0H*PG
RNC	PROC	NEAR		;D0H - RNC		RET NC
;				-----------------------------------------
	SAHF			;Store AH in flags
	JNB	RNC1		;Jump on no carry to return
	NXTI			;Otherwise fall thru to next instruction
RNC1	EQU	$
	MOV	SI,DS:[BP+00H]	;Get return address from stack in P(SI)
	INC	BP		;pop the stack
	INC	BP
	NXTI			;get next instruction from return location
RNC	ENDP
;
	ORG	0D1H*PG
POPD	PROC	NEAR		;D1H - POP D		POP DE
;				------------------------------------------
	MOV	DX,DS:[BP+00H]	;Get DE(DX) from the stack.
	INC	BP		;pop the stack
	INC	BP
	NXTI
POPD	ENDP
;
	ORG	0D2H*PG
JNC80	PROC	NEAR		;D2H - JNC Addr		JP NC,Addr
;				------------------------------------------
	SAHF			;Store AH in flags
	JB	JNC1		;Jump on carry - fall thru to next instruction
	MOV	SI,[SI]		;Get jump address in P(SI)
	NXTI			;Jump on no carry
JNC1	EQU	$
	INC	SI		;Bump P(SI) over the address
	INC	SI
	NXTI			;Fall thru to next instruction
JNC80	ENDP
;
	ORG	0D3H*PG
OUT80	PROC	NEAR		;D3H - OUT D8		OUT (D8),A
;				------------------------------------------
	PUSH	DX		;Save DE(DX) on 8088 stack
;	XOR	DH,DH		;Clear high 8 bits of port
	MOV	DH,BYTE PTR R	;Get high port byte from Refresh Register
	MOV	DL,[SI]		;Get low port byte from immediate byte
	INC	SI		;Bump P(SI) to next instruction
	OUT	DX,AL		;Output A(AL) to the port
	POP	DX		;Restore DE(DX) from 8088 stack
	XOR	CL,CL
	NXTI
OUT80	ENDP
;
	ORG	0D4H*PG
CNC	PROC	NEAR		;D4H - CNC Addr		CALL NC,Addr
;				------------------------------------------
	SAHF			;Store AH in flags
	JB	CNC1		;Jump on carry (fall thru to next instr)
	XCHG	CX,AX		;Save PSW(AX) in CX
	LODSW			;Get call address
	DEC	BP		;Push the stack
	DEC	BP
	MOV	DS:[BP+00H],SI	;Put return address P(SI) on the stack.
	MOV	SI,AX		;Get call address in P(SI)
	XCHG	CX,AX		;Restore PSW(AX) from CX
	XOR	CL,CL
	NXTI			;do the call on no carry
CNC1	EQU	$
	INC	SI		;bump P(SI) over the call address
	INC	SI
	NXTI			;fall thru to next instruction
CNC	ENDP
;
	ORG	0D5H*PG
PUSHD	PROC	NEAR		;D5H - PUSH D		PUSH DE
;				---------------------------------------
	DEC	BP		;Push the stack
	DEC	BP
	MOV	DS:[BP+00H],DX	;Put DE(DX) on the stack.
	NXTI
PUSHD	ENDP
;
	ORG	0D6H*PG
SUI	PROC	NEAR		;D6H - SUI D8		SUB D8
;				----------------------------------------
	SUB	AL,[SI]		;Subtract immediate byte from A(AL)
	LAHF			;Load AH from flags
	JO	SUI1		;Jump on overflow
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,002H		;set subtract flag
	INC	SI		;bump P(SI) over immediate byte
	NXTI
SUI1	EQU	$
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,006H		;set parity/overflow and subtract flags
	INC	SI		;bump P(SI) over immediate byte
	NXTI
SUI	ENDP
;
	ORG	0D7H*PG
RST2	PROC	NEAR		;D7H - RST 2		RST 10H
;				-----------------------------------------
	DEC	BP		;push stack
	DEC	BP
	MOV	DS:[BP+00H],SI	;put return address P(SI) on the stack
	MOV	SI,0010H	;get 0010H in P(SI)
	NXTI			;get next instruction from 0010H
RST2	ENDP
;
	ORG	0D8H*PG
RC	PROC	NEAR		;D8H - RC		RET C
;				-----------------------------------------
	SAHF			;Store AH in flags
	JB	RC1		;Jump on carry to do return on carry
	NXTI			;otherwise fall thru to next instruction
RC1	EQU	$
	MOV	SI,DS:[BP+00H]	;Get return address from stack in P(SI)
	INC	BP		;pop the stack
	INC	BP
	NXTI			;get next instruction from return address
RC	ENDP
;
	ORG	0D9H*PG
EXX	PROC	NEAR		;D9H - (none)		EXX
;				-----------------------------------------
	XCHG	BX,HL		;Exchange HL(BX) with Alternate HL
	XCHG	DI,BC		;Exchange BC(DI) with Alternate BC
	XCHG	DX,DE		;Exchange DE(DX) with Alternate DE
	NXTI
EXX	ENDP
;
	ORG	0DAH*PG
JC80	PROC	NEAR		;DAH - JC Addr		JP C,Addr
;				-----------------------------------------
	SAHF			;Store AH in flags
	JNB	JC1		;Jump on no carry (fall thru to next instr)
	MOV	SI,[SI]		;Get jump address in P(SI)
	NXTI			;do jump on carry
JC1	EQU	$
	INC	SI		;Bump P(SI) over the jump address
	INC	SI
	NXTI			;fall thru to next instruction
JC80	ENDP
;
	ORG	0DBH*PG
IN80	PROC	NEAR		;DBH - IN D8		IN A,D8
;				-----------------------------------------
	PUSH	DX		;save DE(DX) on 8088 stack
;	XOR	DH,DH		;clear high 8 bits of port
	MOV	DH,BYTE PTR R	;Get high port byte from Refresh register
	MOV	DL,[SI]		;Get low port byte from immed operand
	INC	SI		;bump P(SI) around port byte
	IN	AL,DX		;Do the IN, return byte in A(AL)
	POP	DX		;restore DE(DX) from 8088 stack
	XOR	CL,CL
	NXTI
IN80	ENDP
;
	ORG	0DCH*PG
CC	PROC	NEAR		;DCH - CC Addr		CALL C,Addr
;				-----------------------------------------
	SAHF			;Save AH in flags
	JNB	CC1		;Jump on no carry (fall thru to next instr)
	XCHG	CX,AX		;Save PSW(AX) in CX
	LODSW			;Get call address
	DEC	BP		;push the stack
	DEC	BP
	MOV	DS:[BP+00H],SI	;Put return address P(SI) on the stack
	MOV	SI,AX		;Get call address in P(SI)
	XCHG	CX,AX		;Restore PSW(AX) from CX
	XOR	CL,CL
	NXTI			;do the call on carry
CC1	EQU	$
	INC	SI		;bump P(SI) around the call address
	INC	SI
	NXTI			;fall thru to next instruction
CC	ENDP
;
	ORG	0DDH*PG
DDESC	PROC	NEAR		;DDH - (none)		DD Escape
;				------------------------------------------
	MOV	CH,[SI]		;Get DD operation byte
	CMP	CH,0CBH		;see if DDCB escape sequence
	JZ	DDCB1		;yes
	MOV	CL,DDOP		;Get DD operation offset address
	INC	SI		;Bump P(SI) around DD op byte
	JMP	CX		;Jump to DD operation requested
DDCB1	EQU	$		;process DDCB escape sequence
	INC	SI		;Bump P(SI) around CB escape byte
	MOV	CL,[SI]		;Get the displacement byte
	XCHG	CX,AX
	CBW			;convert byte to word
	XCHG	CX,AX
	ADD	CX,IX		;add register IX to displacement in CX
	MOV	IZ,CX		;save result in alternate set at IZ
	INC	SI		;Bump P(SI) around displacement byte
	MOV	CH,[SI]		;Get the DDCB instruction op code
	INC	SI		;Bump P(SI) to next instruction
	MOV	CL,0CEH		;Get the DDCB offset address
	JMP	CX		;Go process the DDCB instruction
DDESC	ENDP
;
	ORG	0DEH*PG	
SBI	PROC	NEAR		;DEH - SBI D8		SBC A,D8
;				---------------------------------------
	SAHF			;Store AH in flags
	SBB	AL,[SI]		;Subtract with borrow immed byte from A(AL)
	LAHF			;Load AH from flags
	JO	SBI1		;Jump on overflow
	AND	AH,0F9H		;clear parity/overflow and carry flags
	OR	AH,002H		;set subtract flag
	INC	SI		;bump P(SI) over immediate byte
	NXTI
SBI1	EQU	$
	AND	AH,0F9H		;clear parity/overflow and carry flags
	OR	AH,006H		;set parity/overflow and carry flags
	INC	SI		;bump P(SI) over immediate byte
	NXTI
SBI	ENDP
;
	ORG	0DFH*PG
RST3	PROC	NEAR		;DFH - RST 3		RST 18H
;				-----------------------------------------
	DEC	BP		;push the stack
	DEC	BP
	MOV	DS:[BP+00H],SI	;put P(SI) return address on the stack
	MOV	SI,0018H	;put 0018H in P(SI)
	NXTI			;get next instruction at 0018H
RST3	ENDP
;		
	ORG	0E0H*PG
RPO	PROC	NEAR		;E0H - RPO		RET PO
;				-----------------------------------------
	SAHF			;Store AH in flags
	JPO	RPO1		;Jump on parity odd to return on parity odd
	NXTI			;Otherwise, fall thru to next instruction.
RPO1	EQU	$
	MOV	SI,DS:[BP+00H]	;Get return address from stack in P(SI)
	INC	BP		;pop the stack
	INC	BP
	NXTI			;get next instruction from return location
RPO	ENDP
;
	ORG	0E1H*PG
POPH	PROC	NEAR		;E1H - POP H		POP HL
;				------------------------------------------
	MOV	BX,DS:[BP+00H]	;Get HL(BX) from the stack.
	INC	BP		;pop the stack
	INC	BP
	NXTI
POPH	ENDP
;
	ORG	0E2H*PG
JPO80	PROC	NEAR		;E2H - JPO Addr		JP PO,Addr
;				-----------------------------------------
	SAHF			;Save AH in the flags
	JPE JPO1		;Jump on parity even (fall thru to next instr)
	MOV	SI,[SI]		;Set P(SI) to jump location
	NXTI			;jump on parity
JPO1	EQU	$
	INC	SI		;bump P(SI) over address
	INC	SI
	NXTI			;fall thru to next instruction
JPO80	ENDP
;
	ORG	0E3H*PG
XTHL	PROC	NEAR		;E3H - XTHL		EX (SP),HL
;				----------------------------------------
	XCHG	BX,DS:[BP+00H]	;Exchange HL(BX) with what is on stack.
	NXTI
XTHL	ENDP
;
	ORG	0E4H*PG
CPO	PROC	NEAR		;E4H - CPO Addr		CALL PO,Addr
;				----------------------------------------
	SAHF			;Save AH in flags
	JPE	CPO1		;Jump on parity even (fall thru to next instr)
	XCHG	CX,AX		;Save PSW(AX) in CX
	LODSW			;Get the call address
	DEC	BP		;push the stack
	DEC	BP
	MOV	DS:[BP+00H],SI	;put return address P(SI) on the stack
	MOV	SI,AX		;get call address in P(SI)
	XCHG	CX,AX		;Restore PSW(AX) from CX
	XOR	CL,CL
	NXTI			;do the call on parity odd
CPO1	EQU	$
	INC	SI		;bump P(SI) over the call address
	INC	SI
	NXTI			;fall thru to next instruction
CPO	ENDP
;
	ORG	0E5H*PG
PUSHH	PROC	NEAR		;E5H - PUSH H		PUSH HL
;				-----------------------------------------
	DEC	BP		;push the stack
	DEC	BP
	MOV	DS:[BP+00H],BX	;push HL(BX) on the stack
	NXTI
PUSHH	ENDP
;
	ORG	0E6H*PG
ANI	PROC	NEAR		;E6H - ANI D8		AND D8
;				-----------------------------------------
	AND	AL,[SI]		;AND immediate byte to A(AL)
	LAHF			;Load AH from flags
	AND	AH,0ECH		;Clear aux carry, subtract and carry flags
	OR	AH,010H		;set aux carry flag
	INC	SI		;bump P(SI) around immediate byte
	NXTI
ANI	ENDP
;
	ORG	0E7H*PG
RST4	PROC	NEAR		;E7H - RST 4		RST 20H
;				----------------------------------------
	DEC	BP		;push the stack
	DEC	BP
	MOV	DS:[BP+00H],SI	;push P(SI) return address on the stack
	MOV	SI,0020H	;put 0020H in P(SI)
	NXTI			;get next instruction from 0020H
RST4	ENDP
;
	ORG	0E8H*PG
RPE	PROC	NEAR		;E8H - RPE		RET PE
;				----------------------------------------
	SAHF			;Store AH in flags
	JPE	RPE1		;Jump on parity even to call on parity even
	NXTI			;otherwise, fall thru to next instruction
RPE1	EQU	$
	MOV	SI,DS:[BP+00H]	;Get return address in P(SI) from stack
	INC	BP		;pop the stack
	INC	BP
	NXTI			;do the return on parity even
RPE	ENDP
;
	ORG	0E9H*PG
PCHL	PROC	NEAR		;E9H - PCHL		JP (HL)
;				-----------------------------------------
	MOV	SI,BX		;Put HL(BX) in P(SI) program counter.
	NXTI
PCHL	ENDP
;
	ORG	0EAH*PG
JPE80	PROC	NEAR		;EAH - JPE Addr		JP PE,Addr
;				-----------------------------------------
	SAHF			;Store AH in flags
	JPO	JPE1		;Jump on parity odd (fall thru to next instr)
	MOV	SI,[SI]		;Set P(SI) to the jump address
	NXTI			;jump on parity even
JPE1	EQU	$
	INC	SI		;bump P(SI) over the address
	INC	SI
	NXTI			;fall thru to next instruction
JPE80	ENDP
;
	ORG	0EBH*PG
XCHG80	PROC	NEAR		;EBH - XCHG		EX DE,HL
;				------------------------------------------
	XCHG	DX,BX		;Exchange DE(DX) and HL(BX)
	NXTI
XCHG80	ENDP
;
	ORG	0ECH*PG
CPE	PROC	NEAR		;ECH - CPE Addr		CALL PE,Addr
;				------------------------------------------
	SAHF			;Store AH in flags
	JPO	CPE1		;Jump on parity odd (fall thru to next instr)
	XCHG	CX,AX		;Save PSW(AX) in CX
	LODSW			;Get call address
	DEC	BP		;push the stack
	DEC	BP
	MOV	DS:[BP+00H],SI	;push return P(SI) on the stack
	MOV	SI,AX		;Get call address in P(SI)
	XCHG	CX,AX		;Restore PSW(AX) from CX
	XOR	CL,CL
	NXTI			;do the call on parity even
CPE1	EQU	$
	INC	SI		;bump P(SI) over the address
	INC	SI
	NXTI			;fall thru to the next instruction
CPE	ENDP
;
	ORG	0EDH*PG
EDESC	PROC	NEAR		;EDH - (none)		ED Escape
;				------------------------------------------
	MOV	CH,[SI]		;Get ED operation code byte
	INC	SI		;Bump P(SI) over the operation code byte
	MOV	CL,EDOP		;Join with ED operation offset
	JMP	CX		;Go process the ED operation
EDESC	ENDP
;
	ORG	0EEH*PG
XRI	PROC	NEAR		;EEH - XRI D8		XOR D8
;				-----------------------------------------
	XOR	AL,[SI]		;XOR immediate byte to A(AL)
	LAHF			;Load AH from flags
	AND	AH,0ECH		;Clear aux carry, subtract and carry flags
	INC	SI		;bump P(SI) over the immediate data byte
	NXTI
XRI	ENDP
;
	ORG	0EFH*PG
RST5	PROC	NEAR		;EFH - RST 5		RST 28H
;				-----------------------------------------
	DEC	BP		;push the stack
	DEC	BP
	MOV	DS:[BP+00H],SI	;save return address P(SI) on the stack
	MOV	SI,0028H	;put 0028H in P(SI)
	NXTI			;get next instruction from 0028H
RST5	ENDP		
;
	ORG	0F0H*PG
RP	PROC	NEAR		;F0H - RP		RET P
;				-----------------------------------------
	SAHF			;Store AH in flags
	JNS	RP1		;Jump if not sign to return on positive
	NXTI			;Otherwise, fall thru to next instruction
RP1	EQU	$
	MOV	SI,DS:[BP+00H]	;Get return address in P(SI) from stack.
	INC	BP		;pop the stack
	INC	BP
	NXTI			;return on positive
RP	ENDP
;
	ORG	0F1H*PG
POPPSW	PROC	NEAR		;F1H - POP PSW		POP AF
;				----------------------------------------
	MOV	AX,DS:[BP+00H]	;Load PSW(AX) from the stack.
	XCHG	AL,AH		;position bytes for Z80
	INC	BP		;pop the stack
	INC	BP
	NXTI
POPPSW	ENDP
;
	ORG	0F2H*PG
JP80	PROC	NEAR		;F2H - JP Addr		JP P,Addr
;				----------------------------------------
	SAHF			;Store AH in flags
	JS	JP1		;Jump on sign (fall thru to next instr)
	MOV	SI,[SI]		;Get jump address in P(SI)
	NXTI			;jump on positive
JP1	EQU	$
	INC	SI		;bump P(SI) over the address
	INC	SI
	NXTI			;fall through to next instruction
JP80	ENDP
;
	ORG	0F3H*PG
DI80	PROC	NEAR		;F3H - DI		DI
;				----------------------------------------
	MOV	BYTE PTR INTFLG,00H	;Disable Z80 interrupts.
	NXTI
DI80	ENDP
;
	ORG	0F4H*PG
CP	PROC	NEAR		;F4H - CP Addr		CALL P,Addr
;				----------------------------------------
	SAHF			;Store AH in flags
	JS	CP1		;Jump on sign (fall thru to next instr)
	XCHG	CX,AX		;Save PSW(AX) in CX
	LODSW			;Get call address
	DEC	BP		;push the stack
	DEC	BP
	MOV	DS:[BP+00H],SI	;put P(SI) return address on the stack
	MOV	SI,AX		;Get call address in P(SI)
	XCHG	CX,AX		;Restore PSW(AX) from CX
	XOR	CL,CL
	NXTI			;get next instruction from call address
CP1	EQU	$
	INC	SI		;bump P(SI) over the address
	INC	SI
	NXTI			;fall through to next instruction
CP	ENDP
;
	ORG	0F5H*PG
PSHPSW	PROC	NEAR		;F5H - PUSH PSW		PUSH AF
;				---------------------------------------
	DEC	BP		;push the stack
	DEC	BP
	XCHG	AL,AH		;adjust PSW bytes
	MOV	DS:[BP+00H],AX	;put PSW(AX) on the stack
	XCHG	AL,AH		;adjust PSW bytes back
	NXTI
PSHPSW	ENDP
;
	ORG	0F6H*PG
ORI	PROC	NEAR		;F6H - ORI D8		OR D8
;				---------------------------------------
	OR	AL,[SI]		;OR the immediate byte to A(AL)
	LAHF			;Load AH from flags
	AND	AH,0ECH		;clear aux carry, subtract and carry flags
	INC	SI		;bump P(SI) over the immediate byte
	NXTI
ORI	ENDP
;
	ORG	0F7H*PG
RST6	PROC	NEAR		;F7H - RST 6		RST 30H
;				---------------------------------------
	DEC	BP		;push the stack
	DEC	BP
	MOV	DS:[BP+00H],SI	;put return address P(SI) on the stack
	MOV	SI,0030H	;get 0030H in P(SI)
	NXTI			;get next instruction from 0030H
RST6	ENDP
;
	ORG	0F8H*PG
RM	PROC	NEAR		;F8H - RM		RET M
;				-----------------------------------------
	SAHF			;Store AH in flags
	JS	RM1		;Jump on sign to return on minus
	NXTI			;Otherwise, fall thru to next instruction.
RM1	EQU	$
	MOV	SI,DS:[BP+00H]	;Get return address in P(SI) from the stack.
	INC	BP		;pop the stack
	INC	BP
	NXTI			;get next instruction from the return address
RM	ENDP
;
	ORG	0F9H*PG
SPHL	PROC	NEAR		;F9H - SPHL		LD SP,HL
;				------------------------------------------
	MOV	BP,BX		;Load SP(BP) from HL(BX)
	NXTI
SPHL	ENDP
;
	ORG	0FAH*PG
JM80	PROC	NEAR		;FAH - JM Addr		JP M,Addr
;				------------------------------------------
	SAHF			;Store AH in flags
	JNS	JM1		;Jump on not sign (fall thru to next instr)
	MOV	SI,[SI]		;get the jump address in P(SI)
	NXTI			;jump on minus
JM1	EQU	$
	INC	SI		;bump P(SI) over the address
	INC	SI
	NXTI			;fall through to the next instruction
JM80	ENDP
;
	ORG	0FBH*PG
EI	PROC	NEAR		;FBH - EI		EI
;				------------------------------------------
	MOV	BYTE PTR INTFLG,03H	;set Z80 interrupts enabled
	NXTI
EI	ENDP
;
	ORG	0FCH*PG
CM	PROC	NEAR		;FCH - CM Addr		CALL M,Addr
;				------------------------------------------
	SAHF			;Store AH in flags
	JNS	CM1		;Jump on not sign (fall thru to next instr)
	XCHG	CX,AX		;Save PSW(AX) in CX
	LODSW			;Get call address
	DEC	BP		;push the stack
	DEC	BP
	MOV	DS:[BP+00H],SI	;save return address P(SI) on the stack
	MOV	SI,AX		;get call address in P(SI)
	XCHG	CX,AX		;Restore PSW(AX) from CX
	XOR	CL,CL
	NXTI			;do call on minus
CM1	EQU	$
	INC	SI		;bump P(SI) over the address
	INC	SI
	NXTI			;fall through to the next instruction
CM	ENDP
;
	ORG	0FDH*PG
FDESC	PROC	NEAR		;FDH - (none)		FD Escape
;				------------------------------------------
	MOV	CH,[SI]		;Get FD operation byte
	CMP	CH,0CBH		;see if FDCB escape sequence
	JZ	FDCB1		;yes
	MOV	CL,FDOP		;Get FD operation offset address
	INC	SI		;Bump P(SI) around FD op byte
	JMP	CX		;Jump to FD operation requested
FDCB1	EQU	$		;process FDCB escape sequence
	INC	SI		;Bump P(SI) around CB escape byte
	MOV	CL,[SI]		;Get the displacement byte
	XCHG	CX,AX
	CBW			;convert byte to word
	XCHG	CX,AX
	ADD	CX,IY		;add register IY to displacement in CX
	MOV	IZ,CX		;save result in IZ
	INC	SI		;Bump P(SI) around displacement byte
	MOV	CH,[SI]		;Get the DDCB instruction op code
	INC	SI		;Bump P(SI) to next instruction
	MOV	CL,0CEH		;Get the FDCB offset address
	JMP	CX		;Go process the FDCB instruction
FDESC	ENDP
;
	ORG	0FEH*PG
CPI	PROC	NEAR		;FEH - CPI D8		CP D8
;				----------------------------------------
	CMP	AL,[SI]		;Compare immediate byte with A(AL)
	LAHF			;Load AH from flags
	JO	CPI1		;Jump on overflow
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,002H		;set subtract flag
	INC	SI		;bump P(SI) over immediate byte
	NXTI
CPI1	EQU	$
	AND	AH,0F9H		;clear parity/overflow and subtract flags
	OR	AH,006H		;set parity/overflow and subtract flags
	INC	SI		;bump P(SI) over immediate byte
	NXTI
CPI	ENDP
;
	ORG	0FFH*PG
RST7	PROC	NEAR		;FFH - RST 7		RST 38H
;				-----------------------------------------
	DEC	BP		;push the stack
	DEC	BP
	MOV	DS:[BP+00H],SI	;put return pointer P(SI) on the stack
	MOV	SI,0038H	;put 0038H in P(SI)
	NXTI			;get next instruction from 0038H
RST7	ENDP
;
;
;========================================================================
;
