/* 
 * plg_lib.c - Profiling Log library for the prf device
 * MV, 28.4.1997
 *
 * 01.07.2000 MV  some more optional debug output
 *
 */

/*
 * To compile:  cc -W2 -O2 plg_lib.c -c plg_lib.o
 */

#include <stdio.h>              /* stderr */

#include <fcntl.h>              /* O_CREAT */

#include <limits.h>             /* ULONG_MAX */
#include <unistd.h>             /* ioctl() */
#include <sys/ioctl.h>		/* ioctl() for LINUX */

#include "memutil.h"
#include "plg_lib.h"             /* the same include for all plg_lib's */
#include "cm_lib.h"		/* the same include for all cm_lib's */


/* #define DEBUG */


#ifdef DEBUG
#include <string.h>	/* strlen */
static int plg_info_flg = 0;
ErrCode plg_show_info(void);
#endif /* DEBUG */


#define MONITOR_DEVICE PRF_DEVICE     /* defined in ?? */


/*
 * plg: profiling log Interface 
 *
 * Functions provided by this library:
 * plg_get_info
 * plg_get_eventtxt
 * (plg_test_cnt_event)
 * plg_open:    	open log, set events, start log
 * plg_close:   	close log
 * plg_reset: 		(stop &) reset log (not needed)
 * plg_read_1log:	read current log from one CPU
 * plg_read_log:	read current log from all CPUs
 * plg_ctart:        start the CPU monitor (counters remain its values)
 * plg_ctop:         stop the CPU monitor
 */

/* ****************************************************************************
 * plg_get_info - profiling log get information
 *
 * Description:
 *   Get information about the profiling log library, e.g. number of events,
 *   number of physical counters.
 *   The profiling log need not be open to get these parameters.
 *
 * Parameters:
 *   struct plg_info_s *plgi      cm info struct
 *
 * Return values:
 *   ErrCode
 *
 * ****************************************************************************
 */
ErrCode plg_get_info(struct plg_info_s *plgi) {	/* see plg_get_info */
  plgi->type = CM_TYPE;
  plgi->version = CM_VERSION;
  plgi->event_num = CM_EVENT_NUM;
  plgi->p_cnt_num = PRF_P_CNT_NUM;
  plgi->p_cnt_bits = 32;	/* always 32, not CM_P_CNT_BITS */
  plgi->user_mode_f = CM_USER_MODE_F;
#ifdef DEBUG
  if (plg_info_flg == 0)  plg_show_info();
#endif /* DEBUG */
  return OK;
}


/* ****************************************************************************
 * plg_get_eventtxt - profiling log get event text
 *
 * Description:
 *   Get text description for a event (counter).
 *   The profiling log needs not be open to get the text.
 *
 * Parameters:
 *  const int event             event number
 *
 * Return values:
 *   char *
 *
 * ****************************************************************************
 */
char *plg_get_eventtxt(const int event) {
  static char *event_txt[CM_EVENT_NUM] = {
    CM_EVENT_TXT
  };
  static char *event_null = "";
  if (event >= CM_EVENT_NUM) {
    return (event_null);        /* wrong index */
  }
  return (event_txt[event]);
}

/* ****************************************************************************
 * plg_get_eventdescr - profiling log get event description
 *
 * Description:
 *   Get long text description for a event (counter).
 *   The profiling log needs not be open to get the text.
 *
 * Parameters:
 *  const int event             event number
 *
 * Return values:
 *   char *
 *
 * ****************************************************************************
 */
char *plg_get_eventdescr(const int event) {
  static char *event_descr[CM_EVENT_NUM] = {
    CM_EVENT_DESCR
  };
  static char *event_null = "";
  if (event >= CM_EVENT_NUM) {
    return (event_null);        /* wrong index */
  }
  return (event_descr[event]);
}


ErrCode plg_test_cnt_event(const int p_cnt, const int event) {
  if (p_cnt != CM_GET_CNT(event, p_cnt)) {
    return ERROR;       /* event impossible on counter */
  }
  return OK;    /* event possible on counter */
}



#define EVENT_UNUSED (-1)       /* an unused event number */

static ErrCode create_ev2tab(struct plg_lib_c *plgl, const int write_flg, int *sample_num) {
  int cnt;
  int i;
  int event_cnt = 0;
  int *ev_mark = memutil_calloc((size_t)plgl->event_num * sizeof(int));
  int ana_md = -1;
  *sample_num = 0;
  while (event_cnt < plgl->event_num) {
    for (cnt = 0; cnt < PRF_P_CNT_NUM; cnt++) {
      for (i = 0; i < plgl->event_num; i++) { /* find possible event */
        if ((ev_mark[i] == 0) && (plg_test_cnt_event(cnt, plgl->events[i]) == OK)) {
          if (ana_md == -1) {
            ana_md = CM_GET_MODE(plgl->events[i]);
          }
          if (ana_md == CM_GET_MODE(plgl->events[i])) {
            ev_mark[i] = 1;
            event_cnt++;
            break;  /* event is possible with counter */
          }
        }
      }
      if (write_flg > 0) {
        plgl->samples[(*sample_num) * PRF_P_CNT_NUM + cnt] =
         (i < plgl->event_num) ? plgl->events[i] : EVENT_UNUSED;
           /* maybe inactive */
#ifdef DEBUG
        fprintf(stderr, "sample %d, cnt %d: event_num=%d, ana_md=%d\n",
          *sample_num, cnt, plgl->samples[(*sample_num) * PRF_P_CNT_NUM + cnt],
          ana_md);
#endif /* DEBUG */
      }
    }
    (*sample_num)++;
    ana_md = -1;
  }
  memutil_free(ev_mark);
  return OK;
}


#ifdef DEBUG
static ErrCode show_stat(struct plg_lib_c *plgl) {
  int h;
  printf("# status:\n");
  h = ioctl(plgl->fd, PRF_RD_STATE, 0);
  printf("# PRF_RD_STATE: %d\n", h);

  h = ioctl(plgl->fd, PRF_RD_BUFNUM, 0);
  printf("# PRF_RD_BUFNUM: %d\n", h);

  h = ioctl(plgl->fd, PRF_RD_CPUCNT, 0);
  printf("# PRF_RD_CPUCNT: %d\n", h);

  /* h = ioctl(plgl->fd, PRF_IO_MCONTROL, 0); */
  /* printf("# PRF_IO_MCONTROL: %d\n", h); */

  {
    const int cpu = 0;
    h = ioctl(plgl->fd, PRF_RD_IDX, cpu);
    printf("# PRF_RD_IDX: %d\n", h);
  }
  return OK;
}
#endif	/* DEBUG */


/* ****************************************************************************
 * plg_close - profiling log close
 *
 * Description:
 *   Close the profiling log. 
 *
 * Parameters:
 *   struct plg_lib_c *plgl	context structure
 *
 * Return values:
 *   ErrCode
 *
 * ****************************************************************************
 */
ErrCode plg_close(struct plg_lib_c *plgl) {
  if (plgl->fd == 0) {
    fprintf(stderr, "plg_close(): device not open.\n");
    return ERROR;
  }

  if (plg_stop(plgl) != OK) {	/* stop it, if still running */
  }

  if (ioctl(plgl->fd, PRF_FREE, 0) < 0) {
    perror("ioctl(PRF_FREE)");
  }

  memutil_free(plgl->pmd);
  plgl->pmd = NULL;

  memutil_free(plgl->events);
  plgl->events = NULL;
  memutil_free(plgl->samples);
  plgl->samples = NULL;

  if (close(plgl->fd) < 0) {
    perror("plg_close:close()");
    plgl->fd = 0;
    return ERROR;
  }
  plgl->fd = 0;
  return OK;
}


#ifdef DEBUG
ErrCode plg_show_info(void) {
  struct plg_info_s plgi;
  int i;
  int max_len = 0;

  plg_info_flg = 1;
  if (plg_get_info(&plgi) != OK)  return ERROR;
  plg_info_flg = 0;
  printf("CPU Monitor Library Information:\n");
  printf("type %d, version %d, %d events, %d phys. counters (%d bits), usr_md_f=%d\n", plgi.type, plgi.version, plgi.event_num, plgi.p_cnt_num, plgi.p_cnt_bits,
   plgi.user_mode_f);

  for (i = 0; i < plgi.event_num; i++) {
    if ((int)strlen(plg_get_eventtxt(i)) > max_len) {
      max_len = strlen(plg_get_eventtxt(i));
    }
  }

  for (i = 0; i < plgi.event_num; i++) {
    int j;
    printf("event %2d: %*s  on cnt ", i, -max_len, plg_get_eventtxt(i));
    for (j = 0; j < plgi.p_cnt_num; j++) {
      if (plg_test_cnt_event(j, i) == OK) {
        printf("%d ", j);
      }
    }
#ifdef I686_EVCNT
    printf(" - phys. reg=%d (%03x)", event_reg[i], (size_t)event_reg[i]); 
#endif /* I686_EVCNT */
    printf("\n");
  }
  return OK;
}
#endif /* DEBUG */



/* ****************************************************************************
 * plg_get_sample_num - profiling log get sample number
 *
 * Description:
 *   Return the number of samples the plg_lib will use to simulate access to
 *   an arbitrary number of event counters.
 *   The value is valid after an open.
 *
 * Parameters:
 *   struct plg_lib_c *plgl	context structure
 *   int *sample_num		number of samples plg_read0 will use
 *
 * Return values:
 *   ErrCode
 *
 * ****************************************************************************
 */
ErrCode plg_get_sample_num(struct plg_lib_c *plgl, int *sample_num) {
  *sample_num = plgl->sample_num;
  /* if (plgl->sample_num == 0)  return ERROR; */
  return OK;
}

ErrCode plg_get_sample(struct plg_lib_c *plgl, const int sample_pos, int *sample) {
  register int i;
  register int *sptr;
  if (sample_pos >= plgl->sample_num) {
    return ERROR;
  }
  sptr = &plgl->samples[sample_pos * PRF_P_CNT_NUM + 0];
  for (i = 0; i < PRF_P_CNT_NUM; i++) {
    sample[i] = sptr[i];
  }
  return OK;
}



static int plg_range_flg = 0;   /* range, where to count */

#ifdef CM_EVENT_REG	/* some translation? */
/* yes, use event register mapping */
static int cm_event_reg[CM_EVENT_NUM] = {
  CM_EVENT_REG
};
#endif	/* CM_EVENT_REG */

static ErrCode plg_sample2mode(const int *sample, const int range_flg, unsigned int *mode) {
  /* First, get the first two event numbers of the current sample. */
  /* If one event is unused, replace it with number 0 */
  register unsigned int ev0 = (sample[0] == EVENT_UNUSED) ? 0 : sample[0];
  register unsigned int ev1 = (sample[1] == EVENT_UNUSED) ? 0 : sample[1];
  /* convert ev0, ev1 to mode values for the device driver ... */
  /* It can use range_flg */
  CM_SAMPLE2MODE
  /* and save the mode values */
  mode[0] = ev0;
  mode[1] = ev1;
  return OK;
}


/* ****************************************************************************
 * plg_open - profiling log open
 *
 * Description:
 *   Open the profiling log.
 *
 * Parameters:
 *   struct plg_lib_c *plgl	context structure
 *   const int event_num	number of events/counters
 *   const int *events		array with event numbers to measure
 *   const int user_mode_f	counting in user mode only, if supported
 *   ...
 *
 * Return values:
 *   ErrCode
 *
 * ****************************************************************************
 */
ErrCode plg_open(struct plg_lib_c *plgl, const int event_num, const int *events, const int measure_mask, const int buf_num, const int col_pid, int *cpu_num) {
  plgl->event_num = event_num;
  plgl->col_pid = col_pid;	/* collect pid >= 0, -1 = all */
  plgl->fd = 0;
  /* plgl->sample_pos = 0; */
  plgl->events = NULL;
  plgl->samples = NULL;
  plgl->sample_num = 0;
  plgl->cpu_num = 0;
  plgl->pmd = NULL;

#ifdef MUELL
  if (measure_mask > CM_USER_MODE_F) {
    fprintf(stderr, "Warning: User mode flag unsupported. (%d).\n", measure_mask);
  }
#endif	/* MUELL */
  plg_range_flg = CM_GET_RANGE(measure_mask);

  if (event_num < 0) { /* 0 is possible! */
    fprintf(stderr, "Error: No events specified.\n");
    return ERROR;
  }

  if (event_num > CM_EVENT_NUM) {
    fprintf(stderr, "plg_open: More counters than available events (%d).\n", event_num);
    return ERROR;
  }

  if (event_num > 0) {
    plgl->events = memutil_malloc((size_t)plgl->event_num * sizeof(int));
    {
      int i;
      for (i = 0; i < event_num; i++) {
        plgl->events[i] = events[i];
        if (events[i] >= CM_EVENT_NUM) {
          fprintf(stderr, "plg_open: Event number out of range: %d\n", events[i]);
          return ERROR;
        }
      }
    }
  
    if (create_ev2tab(plgl, 0, &plgl->sample_num) != OK) return ERROR; /* get size */
    plgl->samples = memutil_malloc((size_t)plgl->sample_num * PRF_P_CNT_NUM * sizeof(int));
    if (create_ev2tab(plgl, 1, &plgl->sample_num) != OK) return ERROR;
  }


#ifdef DEBUG
  fprintf(stderr, "plg_open: sample_num=%d\n", plgl->sample_num);
#endif /* DEBUG */

  /* open CPU prf device */
  {
    const int open_mode = O_RDWR;
    if ((plgl->fd = open(MONITOR_DEVICE, open_mode)) < 0) {
      plgl->fd = 0;
      perror(MONITOR_DEVICE);
      return ERROR;
    }
  }
#ifdef DEBUG
  fprintf(stderr, "plg_open: device opened.\n");
#endif /* DEBUG */
  {
    int version;
    if ((version = ioctl(plgl->fd, PRF_VERSION, 0)) < 0) {
      perror("ioctl(PRF_VERSION)");
      return ERROR;
    }
    version = version;	/* to avoid warning */
#ifdef DEBUG
  fprintf(stderr, "plg_open: version=%d\n\n", version);
#endif /* DEBUG */ 
  }

  if ((plgl->cpu_num = ioctl(plgl->fd, PRF_RD_CPUCNT, 0)) < 1) {
    perror("ioctl(PRF_RD_CPUCNT)");
    return ERROR;
  }
  *cpu_num = plgl->cpu_num;
#ifdef DEBUG
  fprintf(stderr, "plg_open: number of CPUs=%d\n\n",  plgl->cpu_num);
#endif /* DEBUG */

  {
    int state;
    if ((state = ioctl(plgl->fd, PRF_RD_STATE, 0)) < 0) {
      perror("ioctl(PRF_RD_STATE)");
      return ERROR;
    }
    if (state > 0) {
      if (buf_num > 0) {
        fprintf(stderr, "Warning: device state = %d. Trying to reset...\n", state);
        if (state & PRF_ON) {
          if (ioctl(plgl->fd, PRF_IO_MCONTROL, 0) < 0) {
            perror("ioctl(PRF_IO_MCONTROL)");
            return ERROR;
          }
        }
        if (state & PRF_ALLOC) {
          if (ioctl(plgl->fd, PRF_FREE, 0) < 0) {
            perror("ioctl(PRF_FREE)");
            return ERROR;
          }
        }
      }
    }
  }

  {
    int pbuf_num = buf_num;
    if (pbuf_num > 0) {
      if (ioctl(plgl->fd, PRF_INIT, pbuf_num) < 0) {
        perror("ioctl(PRF_INIT)");
        fprintf(stderr, "PRF_INIT: %d\n", pbuf_num);
        return ERROR;
      }
    } else {
      if ((pbuf_num = ioctl(plgl->fd, PRF_RD_BUFNUM, 0)) < 0) {
        perror("ioctl(PRF_RD_BUFNUM)");
        return ERROR;
      }
    }
  }

  if (plgl->sample_num > 0) {
    int i;
    size_t pmd_size = sizeof(struct prf_mode_s) * (size_t)plgl->sample_num;
    plgl->pmd = memutil_malloc(pmd_size);
    for (i = 0; i < plgl->sample_num; i++) {
      register int *sample = &plgl->samples[i * PRF_P_CNT_NUM + 0];
      register unsigned int *mode = &plgl->pmd[i].ev_mode[0];
      if (plg_sample2mode(sample, plg_range_flg, mode) != OK)  return ERROR;
#ifdef DEBUG
      {
        int j;
        fprintf(stderr, "plg_sample2mode: sample (");
        for (j = 0; j < PRF_P_CNT_NUM; j++) {
          fprintf(stderr, "%d%s", sample[j], (j < (PRF_P_CNT_NUM - 1)) ? ", " : "");
        }
        fprintf(stderr, ") -> mode (%u, %u)\n", mode[0], mode[1]);
      }
#endif /* DEBUG */
    }

    if (ioctl(plgl->fd, PRF_SET_EVENTS, plgl->sample_num) < 0) {
      perror("ioctl(PRF_SET_EVENTS)");
      return ERROR;
    }
    if (write(plgl->fd, plgl->pmd, pmd_size) != (ssize_t)pmd_size) {
      perror("write(EVENT_MODES)");
      return ERROR;
    }
  }

  if (plgl->col_pid >= 0) {
    if (ioctl(plgl->fd, PRF_SET_PID, plgl->col_pid) < 0) {
      perror("ioctl(PRF_SET_PID)");
      return ERROR;
    }
  }

  if (plg_start(plgl) != OK)  return ERROR;
    /* start the measurement... */
  return OK;
}


/* ****************************************************************************
 * plg_reset - profiling log reset (& stop)
 *
 * Description:
 *   Reset the profiling log.
 *
 * Parameters:
 *   struct plg_lib_c *plgl	context structure
 *
 * Return values:
 *   ErrCode
 *
 * ****************************************************************************
 */
ErrCode plg_reset(struct plg_lib_c *plgl) {
  if (plg_stop(plgl) != OK) {	/* stop it, if still running */
  }
  if (ioctl(plgl->fd, PRF_RESET, 0) < 0) {
    perror("ioctl(PRF_RESET)");
    return ERROR;
  }
  if (plgl->col_pid >= 0) {
    if (ioctl(plgl->fd, PRF_SET_PID, plgl->col_pid) < 0) {
      perror("ioctl(PRF_SET_PID)");
      return ERROR;
    }
  }
  return OK;
}


/* ****************************************************************************
 * plg_read_1log - profiling log read one log
 *
 * Description:
 *   Read the profiling log of one CPU.
 *
 * Parameters:
 *   struct plg_lib_c *plgl	context structure
 *
 * Return values:
 *   ErrCode
 *
 * ****************************************************************************
 */
ErrCode plg_read_1log(struct plg_lib_c *plgl, const int cpu, struct prfctr_s *buf, int *buf_num) {
  int read_cnt;
  if ((read_cnt = ioctl(plgl->fd, PRF_SET_RDCPU, cpu)) < 0) {
    perror("ioctl(PRF_SET_RDCPU)");
    return ERROR;
  }
#ifdef DEBUG
  fprintf(stderr, "# PRF_SET_RDCPU(%d): read_cnt=%d\n", cpu, read_cnt);
#endif
  {
    size_t remain = (size_t)read_cnt * sizeof(struct prfctr_s);
    ssize_t actual;
    char *plglc = (char *)buf;
    while ((actual = read(plgl->fd, plglc, remain)) < (ssize_t)remain) {
#ifdef DEBUG
      fprintf(stderr, "# read: actual=%d, remain=%u\n", actual, remain);
#endif
      if (actual <= 0) {
        perror("read");
        return ERROR;
      }
      plglc += actual;
      remain -= actual;
    }
#ifdef DEBUG
    fprintf(stderr, "# read(test): actual=%d, remain=%u\n", actual, remain);
#endif
  }
  *buf_num = read_cnt;
  return OK;
}


/* ****************************************************************************
 * plg_read_log - profiling log read log
 *
 * Description:
 *   Read the profiling log.
 *
 * Parameters:
 *   struct plg_lib_c *plgl	context structure
 *
 * Return values:
 *   ErrCode
 *
 * ****************************************************************************
 */
ErrCode plg_read_log(struct plg_lib_c *plgl, struct prfctr_s *buf, int *buf_num) {
  int i;
  int curr_buf_num;
  *buf_num = 0;
  for (i = 0; i < plgl->cpu_num; i++) {
    if (plg_read_1log(plgl, i, buf, &curr_buf_num) != OK)  return ERROR;
#ifdef DEBUG
    fprintf(stderr, "plg_read_log: cpu %d: entries: %d\n", i, curr_buf_num);
#endif
    buf += curr_buf_num;
    *buf_num += curr_buf_num;
  }
  {
    register int prfstat = ioctl(plgl->fd, PRF_RD_STATE, 0);
    if (prfstat & PRF_OFLOW) {
      fprintf(stderr, "WARNING: (plg_read_log) kernel buffer overflow!\n");
    }
  }
#ifdef DEBUG
  (void)show_stat(plgl);
/*
  {
    struct prfctr_s *prfc = buf - *buf_num;
    int j;
    for (j = 0; j < *buf_num; j++) {
      fprintf(stderr, "   %2u %5u  %08x  %8u   %8u %8u\n",
        prfc->cpu, prfc->pid, (size_t)prfc->vaddr, prfc->lbolt,
        prfc->ev_cnt[0], prfc->ev_cnt[1]);
      prfc++;
    }
  }
*/
#endif
  return OK;
}


static ErrCode plg_mcontrol(const int plg_fd, const int mode) {
#ifdef DEBUG
    fprintf(stderr, "plg_mcontrol: mode=%d\n", mode);
#endif
  if (ioctl(plg_fd, PRF_IO_MCONTROL, mode) < 0) {
    perror("ioctl(PRF_IO_MCONTROL)");
    return ERROR;
  }
  return OK;
}


/* ****************************************************************************
 * plg_start - profiling log
 *
 * Description:
 *  Start the profiling log.
 *
 * Parameters:
 *   struct plg_lib_c *plgl	context structure
 *
 * Return values:
 *   ErrCode
 *
 * ****************************************************************************
 */
ErrCode plg_start(struct plg_lib_c *plgl) {
  if (plg_mcontrol(plgl->fd, 1) < 0) {
    return ERROR;
  }
#ifdef DEBUG
  (void)show_stat(plgl);
#endif
  return OK;
}

/* ****************************************************************************
 * plg_stop - profiling log stop
 *
 * Description:
 *   Stop the profiling log.
 *
 * Parameters:
 *   struct plg_lib_c *plgl	context structure
 *
 * Return values:
 *   ErrCode
 *
 * ****************************************************************************
 */
ErrCode plg_stop(struct plg_lib_c *plgl) {
  if (plg_mcontrol(plgl->fd, 0) < 0) {
    return ERROR;
  }
#ifdef DEBUG
  (void)show_stat(plgl);
#endif
  return OK;
}

/*
 * plg: CPU Monitor Interface (End)
 */
