/* 
 * cm_lib.c - CPU monitor library for the evcnt device
 * MV, 7.10.1996 (28.8.1996)
 *
 */

/*
 * To compile:  cc -W2 -O2 cm_lib.c -c cm_lib.o
 */

#include <stdio.h>      	/* stderr */
#include <fcntl.h>		/* O_CREAT */

#include <limits.h>		/* ULONG_MAX */
#include <unistd.h>		/* ioctl() */
#include <sys/ioctl.h>		/* ioctl() for LINUX */

#include "memutil.h"
#include "cm_lib.h"


/* #define DEBUG */

#ifdef DEBUG
static int cm_info_flg = 0;
ErrCode cm_show_info(void);
#endif /* DEBUG */


#define MONITOR_DEVICE EVCNT_DEVICE	/* defined in cm_lib.h */


struct cpum_erg {
  hint h[CM_P_CNT_NUM];
};


/*
 * cm: CPU Monitor Interface 
 *
 * Functions provided by this library:
 * cm_get_info
 * cm_get_eventtxt
 * (cm_test_cnt_event)
 * cm_open:		open CPU monitor, set measurement mode, init and start
 * cm_close:		close CPU monitor
 * cm_init:		initialize the counters with 0
 * cm_read0_diff:	read the counter differences (of CPU 0)
 * cm_lib_ctart:	start the CPU monitor (counters remain its values)
 * cm_lib_ctop:		stop the CPU monitor
 */



/* ****************************************************************************
 * cm_get_info - cpu monitor get information
 *
 * Description:
 *   Get information about the CPU monitor library, e.g. number of events,
 *   number of physical counters.
 *   The CPU monitor need not be open to get these parameters.
 *
 * Parameters:
 *   struct cm_info_s *cmi	cm info struct
 *
 * Return values:
 *   ErrCode
 *
 * ****************************************************************************
 */
ErrCode cm_get_info(struct cm_info_s *cmi) {
  cmi->type = CM_TYPE;
  cmi->version = CM_VERSION;
  cmi->event_num = CM_EVENT_NUM;
  cmi->p_cnt_num = CM_P_CNT_NUM;
  cmi->p_cnt_bits = CM_P_CNT_BITS;
  cmi->user_mode_f = CM_USER_MODE_F;
#ifdef DEBUG
  if (cm_info_flg == 0)  cm_show_info();
#endif /* DEBUG */
  return OK;
}


/* ****************************************************************************
 * cm_get_eventtxt - cpu monitor get event text
 *
 * Description:
 *   Get text description for a event (counter).
 *   The CPU monitor needs not be open to get the text.
 *
 * Parameters:
 *  const int event		event number
 *
 * Return values:
 *   char *
 *
 * ****************************************************************************
 */
char *cm_get_eventtxt(const int event) {
  static char *event_txt[CM_EVENT_NUM] = {
    CM_EVENT_TXT
  };
  static char *event_null = "";
  if (event >= CM_EVENT_NUM) {
    return (event_null);	/* wrong index */
  }
  return (event_txt[event]);
}

/* ****************************************************************************
 * cm_get_eventdescr - cpu monitor get event description
 *
 * Description:
 *   Get detailed text description for a event (counter).
 *   The CPU monitor needs not be open to get the text.
 *
 * Parameters:
 *  const int event		event number
 *
 * Return values:
 *   char *
 *
 * ****************************************************************************
 */
char *cm_get_eventdescr(const int event) {
  static char *event_descr[CM_EVENT_NUM] = {
    CM_EVENT_DESCR
  };
  static char *event_null = "";
  if (event >= CM_EVENT_NUM) {
    return (event_null);	/* wrong index */
  }
  return (event_descr[event]);
}


ErrCode cm_test_cnt_event(const int p_cnt, const int event) {
  if (p_cnt != CM_GET_CNT(event, p_cnt)) {
    return ERROR;	/* event impossible on counter */
  }
  return OK;	/* event possible on counter */
}


#define EVENT_UNUSED (-1)	/* an unused event number */

static ErrCode create_ev2tab(struct cm_lib_c *cml, const int write_flg, int *sample_num) {
  int cnt;
  int i;
  int event_cnt = 0;
  int *ev_mark = memutil_calloc((size_t)cml->event_num * sizeof(int));
  int ana_md = -1;
  *sample_num = 0;
  while (event_cnt < cml->event_num) {
    for (cnt = 0; cnt < CM_P_CNT_NUM; cnt++) {
      for (i = 0; i < cml->event_num; i++) { /* find possible event */
        if ((ev_mark[i] == 0) && (cm_test_cnt_event(cnt, cml->events[i]) == OK)) {
          if (ana_md == -1) {
            ana_md = CM_GET_MODE(cml->events[i]);
          }
          if (ana_md == CM_GET_MODE(cml->events[i])) {
            ev_mark[i] = 1;
            event_cnt++;
            break;  /* event is possible with counter */
          }
        }
      }
      if (write_flg > 0) {
        cml->samples[(*sample_num) * CM_P_CNT_NUM + cnt] =
         (i < cml->event_num) ? cml->events[i] : EVENT_UNUSED;
           /* maybe inactive */
#ifdef DEBUG
        fprintf(stderr, "sample %d, cnt %d: event_num=%d, ana_md=%d\n",
          *sample_num, cnt, cml->samples[(*sample_num) * CM_P_CNT_NUM + cnt],
          ana_md);
#endif /* DEBUG */
      }
    }
    (*sample_num)++;
    ana_md = -1;
  }
  memutil_free(ev_mark);
  return OK;
}


/* ****************************************************************************
 * cm_close - cpu monitor close
 *
 * Description:
 *   Close the CPU monitor. 
 *
 * Parameters:
 *   struct cm_lib_c *cml	context structure
 *
 * Return values:
 *   ErrCode
 *
 * ****************************************************************************
 */
ErrCode cm_close(struct cm_lib_c *cml) {
  if (cml->fd == 0) {
    fprintf(stderr, "cm_close(): device not open.\n");
    return ERROR;
  }

  memutil_free(cml->cmd);
  cml->cmd = NULL;
  memutil_free(cml->c_result);
  cml->c_result = NULL;
  memutil_free(cml->events);
  cml->events = NULL;
  memutil_free(cml->samples);
  cml->samples = NULL;

  if (close(cml->fd) < 0) {
    perror("cm_close:close()");
    cml->fd = 0;
    return ERROR;
  }
  cml->fd = 0;
  return OK;
}


#ifdef DEBUG
ErrCode cm_show_info(void) {
  struct cm_info_s cmi;
  int i;
  int max_len = 0;

  cm_info_flg = 1;
  if (cm_get_info(&cmi) != OK)  return ERROR;
  cm_info_flg = 0;
  printf("CPU Monitor Library Information:\n");
  printf("type %d, version %d, %d events, %d phys. counters (%d bits), usr_md_f=%d\n", cmi.type, cmi.version, cmi.event_num, cmi.p_cnt_num, cmi.p_cnt_bits,
   cmi.user_mode_f);

  for (i = 0; i < cmi.event_num; i++) {
    if ((int)strlen(cm_get_eventtxt(i)) > max_len) {
      max_len = strlen(cm_get_eventtxt(i));
    }
  }

  for (i = 0; i < cmi.event_num; i++) {
    int j;
    printf("event %2d: %*s  on cnt ", i, -max_len, cm_get_eventtxt(i));
    for (j = 0; j < cmi.p_cnt_num; j++) {
      if (cm_test_cnt_event(j, i) == OK) {
        printf("%d ", j);
      }
    }
#ifdef I686_EVCNT
    printf(" - phys. reg=%d (%03x)", event_reg[i], event_reg[i]); 
#endif /* I686_EVCNT */
    printf("\n");
  }
  return OK;
}
#endif /* DEBUG */




static ErrCode cm_set_null(struct cm_lib_c *cml) {
  hint h[CM_P_CNT_NUM];
#ifdef CM_NEED_SETCNTS0
  register int i;
  for (i = 0; i < CM_P_CNT_NUM; i++) {
    h[i].lo = 0;
    h[i].hi = 0;
  }
#endif /* CM_NEED_SETCNTS0 */
  if (ioctl(cml->fd, EV_SETCNTS, h) < 0) {	/* set two counters */
    perror("ioctl(EV_SETCNTS)");
    return ERROR;
  }
  return OK;
}



/* ****************************************************************************
 * cm_get_sample_num - cpu monitor get sample number
 *
 * Description:
 *   Return the number of samples the cm_lib will use to simulate access to
 *   an arbitrary number of event counters.
 *   The value is valid after an open.
 *
 * Parameters:
 *   struct cm_lib_c *cml	context structure
 *   int *sample_num		number of samples cm_read0 will use
 *
 * Return values:
 *   ErrCode
 *
 * ****************************************************************************
 */
ErrCode cm_get_sample_num(struct cm_lib_c *cml, int *sample_num) {
  *sample_num = cml->sample_num;
  if (cml->sample_num == 0)  return ERROR;
  return OK;
}


static int cm_range_flg = 0;   /* range, where to count */


#ifdef CM_EVENT_REG	/* some translation? */
/* yes, use event register mapping */
static int cm_event_reg[CM_EVENT_NUM] = {
  CM_EVENT_REG
};
#endif	/* CM_EVENT_REG */

static ErrCode cm_sample2mode(const int *sample, const int range_flg, unsigned int *mode) {
  /* First, get the first two event numbers of the current sample. */
  /* If one event is unused, replace it with number 0 */
  register unsigned int ev0 = (sample[0] == EVENT_UNUSED) ? 0 : sample[0];
  register unsigned int ev1 = (sample[1] == EVENT_UNUSED) ? 0 : sample[1];
  /* convert ev0, ev1 to mode values for the device driver ... */
  /* It can use range_flg */
  CM_SAMPLE2MODE
  /* and save the mode values */
  mode[0] = ev0;
  mode[1] = ev1;
  return OK;
}



/* ****************************************************************************
 * cm_open - cpu monitor open
 *
 * Description:
 *   Open the CPU monitor.
 *   First open the CPU monitor kernel device, get number of CPU monitors,
 *   allocate memory to save the counters for every cpu.
 *   Finally, set the status (analyze mode) and start
 *   the CPU monitors.
 *
 * Parameters:
 *   struct cm_lib_c *cml	context structure
 *   const int event_num	number of events/counters
 *   const int *events		array with event numbers to measure
 *   int *sample_num		number of samples cm_read0 will use
 *   const int user_mode_f	counting in user mode only, if supported
 *
 * Return values:
 *   ErrCode
 *
 * ****************************************************************************
 */
ErrCode cm_open(struct cm_lib_c *cml, const int event_num, const int *events, const int measure_mask) {
  cml->event_num = event_num;
  cml->fd = 0;
  cml->sample_pos = 0;
  cml->events = NULL;
  cml->samples = NULL;
  cml->c_result = NULL;
  cml->cmd = NULL;

#ifdef MUELL
  if (measure_mask > CM_USER_MODE_F) {
    fprintf(stderr, "Warning: User mode flag unsupported. (%d).\n", measure_mask);
  }
#endif /* MUELL */
  cm_range_flg = CM_GET_RANGE(measure_mask);

  if (event_num <= 0) {
    fprintf(stderr, "Error: No events specified.\n");
    return ERROR;
  }

  if (event_num > CM_EVENT_NUM) {
    fprintf(stderr, "cm_open: More counters than available events (%d).\n", event_num);
    return ERROR;
  }

  cml->events = memutil_malloc((size_t)cml->event_num * sizeof(int));
  {
    int i;
    for (i = 0; i < event_num; i++) {
      cml->events[i] = events[i];
      if (events[i] >= CM_EVENT_NUM) {
        fprintf(stderr, "cm_open: Event number out of range: %d\n", events[i]);
        return ERROR;
      }
    }
  }

  if (create_ev2tab(cml, 0, &cml->sample_num) != OK) return ERROR; /* get size */
  cml->samples = memutil_malloc((size_t)cml->sample_num * CM_P_CNT_NUM * sizeof(int));
  if (create_ev2tab(cml, 1, &cml->sample_num) != OK) return ERROR;

#ifdef DEBUG
  fprintf(stderr, "cm_open: sample_num=%d\n", cml->sample_num);
#endif /* DEBUG */

  /* open CPU event device */
  {
    const int open_mode = O_RDONLY;
    if ((cml->fd = open(MONITOR_DEVICE, open_mode)) < 0) {
      cml->fd = 0;
      perror(MONITOR_DEVICE);
      return ERROR;
    }
  }

  {
    int cpu_num = 1;
    cml->c_result = memutil_calloc((size_t)cpu_num * sizeof(struct cpum_erg));
  }

  /* We need the array for all possible events, not only event_num */
  cml->old_event = memutil_malloc((size_t)CM_EVENT_NUM * sizeof(cm_counter_t));

  if (cml->sample_num > 0) {
    int i;
    size_t cmd_size = sizeof(struct cm_mode_s) * (size_t)cml->sample_num;
    cml->cmd = memutil_malloc(cmd_size);
    for (i = 0; i < cml->sample_num; i++) {
      register int *sample = &cml->samples[i * CM_P_CNT_NUM + 0];
      register unsigned int *mode = &cml->cmd[i].ev_mode[0];
      if (cm_sample2mode(sample, cm_range_flg, mode) != OK)  return ERROR;
#ifdef DEBUG
      {
        int j;
        fprintf(stderr, "cm_sample2mode: sample (");
        for (j = 0; j < CM_P_CNT_NUM; j++) {
          fprintf(stderr, "%d%s", sample[j], (j < (CM_P_CNT_NUM - 1)) ? ", " : "");
        }
        fprintf(stderr, ") -> mode (%08x, %08x)\n", mode[0], mode[1]);
      }
#endif /* DEBUG */
    }
  }
  if (cm_init(cml) != OK)  return ERROR;
    /* initialize counters and start the CPU monitor... */
  return OK;
}




static ErrCode cm_set_events(struct cm_lib_c *cml) {
  hint h;
  h.lo = cml->cmd[cml->sample_pos].ev_mode[0];
  h.hi = cml->cmd[cml->sample_pos].ev_mode[1];
#ifdef DEBUG
    fprintf(stderr, "EV_SETMODE: %08lx, %08lx\n", h.lo, h.hi);
#endif
  if (ioctl(cml->fd, EV_SETMODE, &h) < 0) {
    perror("ioctl(EV_SETMODE)");
    return ERROR;
  }
  return OK;
}


/* ****************************************************************************
 * cm_init - cpu monitor init
 *
 * Description:
 *   Initialize the CPU monitor (Set the counters to 0).
 *
 * Parameters:
 *   struct cm_lib_c *cml	context structure
 *
 * Return values:
 *   ErrCode
 *
 * ****************************************************************************
 */
ErrCode cm_init(struct cm_lib_c *cml) {
  register int i;
  for (i = 0; i < CM_EVENT_NUM /* cml->event_num */; i++) {
    cml->old_event[i] = 0;
  }
  cml->sample_pos = 0;	/* start with sample 0 */
  cml->stop_flg = 0;	/* not stopped */
  if (cm_set_events(cml) != OK)  return ERROR;
#ifndef CM_SETEV_RESET
  if (cm_set_null(cml) != OK)  return ERROR;	/* explicit reset to 0 */
#endif	/* CM_SETEV_RESET */
  return OK;
}



/* ****************************************************************************
 * cm_read0_diff - cpu monitor read counter difference
 *
 * Description:
 *   Read the CPU monitor counters and write them in the counter array.
 *   First, read the counters of all CPU monitors and check the return codes.
 *   The difference to the last cm_read0_diff()
 *   is returned.
 *
 * Parameters:
 *   struct cm_lib_c *cml	context structure
 *   cm_counter_t *cnt		pointer to an array of CM_CNT_NUM double values,
 *    				where the counters are saved.
 * Return values:
 *   ErrCode
 *
 * ****************************************************************************
 */
#define HINT2DOUBLE(d, h) ( (d) = ((cm_counter_t) (h).lo) + \
   (((cm_counter_t) (h).hi) * ((cm_counter_t)ULONG_MAX) ))

ErrCode cm_read0_diff(struct cm_lib_c *cml, cm_counter_t *cnt) {
#ifdef DEBUG
  fprintf(stderr, "cm_read0_diff: sample_pos=%d\n", cml->sample_pos);
#endif /* DEBUG */

  if (cml->stop_flg == 0) { /* not stopped -> get counters */
    if (ioctl(cml->fd, EV_GETCNTS, cml->c_result) < 0) {
      perror("ioctl(EV_GETCNTS)");
      return ERROR;
    }
  }

  {
    register const int cpu = 0;	/* use CPU 0 only */
    register int i;
    register const hint *resptr = &cml->c_result[cpu].h[0];
    register cm_counter_t cnt1;
    register int event_num;

    for (i = 0; i < CM_P_CNT_NUM; i++) {
      event_num = cml->samples[cml->sample_pos * CM_P_CNT_NUM + i];
      if (event_num != EVENT_UNUSED) {	/* event active? */
        HINT2DOUBLE(cnt1, resptr[i]);	/* convert counter to double */
        cml->old_event[event_num] = cnt1;
      }
    }

    /* get the result */
    for (i = 0; i < cml->event_num; i++) {
      cnt[i] = cml->old_event[cml->events[i]];

    }

#ifdef DEBUG
    for (i = 0; i < cml->event_num; i++) {
      fprintf(stderr, "%10.4e ", cnt[i]);
    }
    fprintf(stderr, "\n");
#endif /* DEBUG */

  }

  if (cml->sample_num > 1) {
    if ((++cml->sample_pos) >= cml->sample_num) {
      cml->sample_pos = 0;
    }
    if (cm_set_events(cml) != OK)  return ERROR;
#ifndef CM_SETEV_RESET
      if (cml->stop_flg == 0) {
        if (cm_set_null(cml) != OK)  return ERROR; /* explicit reset to 0 */
      }
#endif	/* CM_SETEV_RESET */
  } else if (cml->stop_flg == 0) {
    if (cm_set_null(cml) != OK)  return ERROR; /* not stopped -> init cnts */
  }
  return OK;
}




/* ****************************************************************************
 * cm_start - cpu monitor start
 *
 * Description:
 *  Start the CPU monitor. This is done by initializing the counter with 0.
 *
 * Parameters:
 *   struct cm_lib_c *cml	context structure
 *
 * Return values:
 *   ErrCode
 *
 * ****************************************************************************
 */
ErrCode cm_start(struct cm_lib_c *cml) {
  if (cm_set_null(cml) != OK)  return ERROR;
  cml->stop_flg = 0;
  return OK;
}

/* ****************************************************************************
 * cm_stop - cpu monitor stop
 *
 * Description:
 *   Stop the CPU monitor. This is done by reading the counter values into an
 *   internal structure. cm_read0_diff will return these values.
 *
 * Parameters:
 *   struct cm_lib_c *cml	context structure
 *
 * Return values:
 *   ErrCode
 *
 * ****************************************************************************
 */
ErrCode cm_stop(struct cm_lib_c *cml) {
  /* read results */
  if (ioctl(cml->fd, EV_GETCNTS, cml->c_result) < 0) {	/* get counters */
    perror("ioctl(EV_GETCNTS)");
    return ERROR;
  }
  cml->stop_flg = 1;
  return OK;
}

/*
 * cm: CPU Monitor Interface (End)
 */
