/* read_tsc_ix86.c */


#include <stdio.h>  
#include <limits.h>             /* ULONG_MAX */
#include <stdlib.h>  		/* atoi() */
#include <unistd.h>             /* getopt(), ioctl() */
#include <sys/ioctl.h>          /* ioctl */
#include <fcntl.h>              /* O_RDONLY */

#include "../../../include/general.h"   /* ad_t, OK, ERROR */

#include "evcnt_ix86.h"	/* hint */
#include "msr_ix86.h"	/* read_tsc */

#include "evcnt_feat_ix86.h"

static void get_tsc_1(hint *h) {
  read_tsc(h->lo, h->hi);	/* get TSC directly */
}


struct ev_s {
  int fd;
};
struct ev_s evs;

/* read_msr(h->lo, h->hi, 0x10); get TSC as MSR -> impossible in user mode! */

static void get_tsc_2(hint *h) {
  if (ioctl(evs.fd, EV_GETTSC, h) < 0) {   /* get TSC */
    perror("ioctl(EV_GETTSC)");
    return;
  }
}

static void get_tsc_3(hint *h) {
  hint h_tmp[2];
  h_tmp[0].lo = 0x10;	/* TSC as MSR */
  h_tmp[0].hi = 0;
  if (ioctl(evs.fd, EV_GETMSR, h_tmp) < 0) {   /* get msr */
    perror("ioctl(EV_GETMSR)");
    return;
  }
  h->lo = h_tmp[1].lo;
  h->hi = h_tmp[1].hi;
}


#ifdef EVCNT_CYCNT
static void get_tsc_4(hint *h) {
  hint h_tmp[EVCNT_CNUM];
  if (ioctl(evs.fd, EV_GETCNTS, h_tmp) < 0) {   /* get msr */
    perror("ioctl(EV_GETMSR)");
    return;
  }
  h->lo = h_tmp[EVCNT_CNUM - 1].lo;
  h->hi = h_tmp[EVCNT_CNUM - 1].hi;
}
#endif /* EVCNT_CYCNT */


static void (*get_tsc_func)(hint *h) = get_tsc_1;


static double get_tsc_d(void) {
  hint h;
  get_tsc_func(&h);
  return((double)h.lo + h.hi * (double)ULONG_MAX);
}


static ErrCode set_tsc_mode(const int test_mode) {
  switch(test_mode) {
    case 1:
      get_tsc_func = get_tsc_1;
      printf("get_tsc: function in user mode\n");
    break;

    case 2:
      get_tsc_func = get_tsc_2;
      printf("get_tsc: ioctl(EV_GETTSC)\n");
    break;

    case 3:
      get_tsc_func = get_tsc_3;
      printf("get_tsc: ioctl(EV_GETMSR, 0x10)\n");
    break;

#ifdef EVCNT_CYCNT
    case 4:
      get_tsc_func = get_tsc_4;
      printf("get_tsc: ioctl(EV_GETCNTS)\n");
    break;
#endif /* EVCNT_CYCNT */

    default:
      fprintf(stderr, "Warning: Unknown test mode %d\n", test_mode);
      return ERROR;
    break;
  }
  if (test_mode >= 2) {
    if ((evs.fd = open(EVCNT_DEVICE, O_RDONLY)) < 0) {
      perror(EVCNT_DEVICE);
      return ERROR;
    }
  }
  return OK;
}

/* ****************************
 *  read options, usage
 * **************************** */

#define DEF_TEST_MODE 1
#define DEF_LOOPS 1000000

static void usage(const char *pname) {
  fprintf(stderr, "tsc_test - Time Stamp Counter (TSC) test\n");
  fprintf(stderr, "Marco Vieth, 17.7.1997\n");
  fprintf(stderr, "Usage: %s [-t n]\n", pname);
  fprintf(stderr, "\t-t num\t test mode (def. %d)\n", DEF_TEST_MODE);
  fprintf(stderr, "\t-l num\t loops (def. %d)\n", DEF_LOOPS);
  fprintf(stderr, "\t-h\t help\n");
  fprintf(stderr, "\t-d\t debug\n");
  fprintf(stderr, "\n");
  exit(1);  
}


struct option_s {
  int loops;
  int test_mode;
  int debug_f;
};


static int read_opt(int argc, char **argv, struct option_s *opts) {
  int n;
  while ((n = getopt(argc, argv, "dhl:t:")) != EOF) {
    switch (n) {
      case 'd':         /* debug output */
        opts->debug_f = 1;
      break;

      case 'l':         /* loops */
        opts->loops = atoi(optarg);
      break;

      case 't':         /* test mode */
        opts->test_mode = atoi(optarg);
      break;

      case 'h':  
      case '?':
        usage(argv[0]);
      break;
    }
  }
  return OK;
}


/* ****************************
 *  main
 * **************************** */


int main(const int argc, char **argv) {
  struct option_s opts = { DEF_LOOPS, DEF_TEST_MODE, 0};
  ErrCode rc = OK;
  register double d1, d2;
  register double cpu_freq_hz;
  register int loops;

  if (read_opt(argc, argv, &opts) != OK)  { }
  loops = opts.loops;

  printf("tsc_test - Time Stamp Counter (TSC) test\n");

  if (set_tsc_mode(opts.test_mode) != OK)  return ERROR;

  printf("Measuring CPU speed...\n");
  sleep(1);
  d1 = get_tsc_d();
  sleep(1);
  d2 = get_tsc_d();
  cpu_freq_hz = d2 - d1;
  if (cpu_freq_hz < 1)  cpu_freq_hz = 1;
  printf("CPU speed = %.0f MHz (%.0f Hz)\n", cpu_freq_hz / 1000000, cpu_freq_hz);
  while (loops-- > 0) {
    d1 = get_tsc_d();
    d2 = get_tsc_d();
    printf("%.0f %.0f (%f s)\r", d1, d2, d1 / cpu_freq_hz);
  }
  printf("\n");
  if (opts.test_mode >= 2) {
    if (close(evs.fd) < 0) {
      perror("close()");
      return ERROR;
    }
  }
  return (rc);
}
