/*
 * prf.c - Profiling Device Driver for Linux (sprof) 
 * Marco Vieth, 30.4.1997
 *
 * 01.07.2000 0.21 MV  adapted for kernel 2.2x: file_operations, memcpy_tofs -> copy_to_user,
 *                     memcpy_fromfs -> copy_from_user; prf_ioctl: *rvalp -> rval (no Compiler warning!);
 *                     SMP: smp_message_pass -> smp_call_function (how to get regs in prf_interrupt_other?).
 * 19.01.2001 0.22 MV  adapted for AMD Athlon
 *
 */

/*
 * This device contains the evcnt device plus the prf device.
 *
 *
 * 1. Installing the device:
 *      mknod /dev/evcnt1 c 63 1
 *      chmod a+rw /dev/evcnt1
 *
 * 1.1 Installing as module (for a kernel which supports modules, e.g. 0.99p4):
 * Compile with -DMODULE
 * Manual installation:
 * 	insmod prf_ix86.o
 * At this state an error message may appear in /var/spool/warn
 * (IRQ 0 not free).
 * At this stage, make the following change to
 * /usr/src/linux/arch/i386/kernel/time.c:
 *   static struct irqaction irq0  = { timer_interrupt, 0, 0, "timer", NULL, NULL};
 *                                                      ^-- SA_SHIRQ
 * (If the second parameter is SA_INTERRUPT, replace it by SA_SHIRQ).
 * Replace the second parameter (0) by SA_SHIRQ to allow IRQ sharing.
 *   static struct irqaction irq0  = { timer_interrupt, SA_SHIRQ, 0, "timer", NULL, NULL};
 * Now you must recompile the kernel and reboot it.
 * 	insmod prf_ix86.o   should work now.
 * 
 * To remove the module, use
 *	rmmod prf_ix86
 *
 * You can also make the module auto loadable with kerneld, as described
 * in evcnt_ix86.c.
 *
 * Note for SMP:
 * 
 * Make the following change to
 * /usr/src/linux/arch/i386/kernel/irq.c:
 * static struct irqaction irq13 = { smp_message_irq, SA_INTERRUPT, 0, "IPI", NULL, NULL };
 * Replace the second parameter SA_INTERRUPT by SA_SHIRQ to disable fast IRQs
 * and allow a module to share IRQ13.
 * static struct irqaction irq13 = { smp_message_irq, SA_SHIRQ, 0, "IPI", NULL, NULL };
 *
 *
 * OLD notes for SMP:
 * On SMP machines you have to compile with -D__SMP__ -UMODULE and link the
 * device to the kernel because some variables are not accessible from
 * modules, e.g. smp_num_cpus.
 * You must ALSO do some changes to /usr/src/linux/arch/i386/kernel/smp.c:
 * 1. Add the following lines after the #incudes:
 *   extern void prf_interrupt_other(struct pt_regs *regs);
 *   #define MSG_SPROF_TICK (MSG_RESCHEDULE + 1)
 * 2. Extend the function smp_message_irq (after MSG_STOP_CPU):
 *    case MSG_SPROF_TICK:
 *      prf_interrupt_other(regs);
 *    break;
 *
 *  To do for SMP:
 *  - PC of other CPUs than CPU 3 is wrong: always f000ff47.   corrected.
 *  - CPU numbers are physical instead of logical.  fixed on 14-08-97.
 */

#include <linux/module.h>

#include <linux/types.h>
/* #include <sys/errno.h> */		/* EINVAL, ... */


/* #define CONFIG_X86_LOCAL_APIC */  /* test */
#include <linux/malloc.h>	/* kmalloc, kfree */
#include <linux/smp.h>		/* smp_num_cpus,  smp_processor_id */
#include <asm/ptrace.h>		/* instruction_pointer() */

#include "prf_ix86.h"
#include "msr_ix86.h"		/* macros to access the ix86 MSRs */

#define WITH_IRQ		/* define this! Otherwise it does not work. */


#ifdef WITH_IRQ
/* #include <linux/interrupt.h> */
#define PRF_IRQ 0  /* 0 for system timer or 8 for rtc */
#ifdef __SMP__
  #if LINUX_VERSION_CODE >= 0x020104
  #else
    #define MSG_IRQ 13 /* 13 for message pass interrupt */
  #endif
#endif /* __SMP__ */
#else
#endif


#ifdef __SMP__

#if LINUX_VERSION_CODE >= 0x020104
  extern volatile int cpu_logical_map[NR_CPUS]; /* from smp.c */
  #define prf_cpu_logical(n)  cpu_logical_map[n]
#else
  extern void prf_interrupt_other(struct pt_regs *regs); /* MV, 18.7.1997 */
  #define MSG_SPROF_TICK (MSG_RESCHEDULE + 1)
  /* or define this message in linux/smp.h */
  #define prf_cpu_logical(n)  cpu_logical_map(n)
#endif /* LINUX_VERSION_CODE */

#endif  /* __SMP__ */

#include "evcnt_ioctl.c"        /* include evcnt_ioctl command */


/* extern int maxprf; */
static unsigned prfstat;	/* state of profiler */



/* variables for every CPU */
struct prfc_percpu_s {
  struct prfctr_s *ctr;	/* counter array */
  int wr_ctr_idx;	/* current (write) counter index (for every CPU) */
  int rd_ctr_idx;	/* current read counter idx */

  /* for event counter sampling */
  int evc_idx;		/* event counter index in pev */
#ifdef EVCNT_CYCNT
  hint tsc_base;  /* time stamp counter base for every CPU! */
#endif /* EVCNT_CYCNT */
};

struct prfc_s {
  int num;		/* number of entries (same as maxprf, if initialized) */
  int rd_cpu;		/* CPU number to read data from */
  int rd_cnt;		/* next read count (can be split into two parts) */
  struct prfc_percpu_s cpu[PRF_NCPU];

  int collect_pid;      /* collect data for special PID >= 0, -1 = for all */
  /* for event counter sampling */
  int prf_ev_num;	/* number of events */
  struct prf_mode_s *pev;	/* event number array */
};


static int prfc_free_events(struct prfc_s *prfc) {
  if (prfc->prf_ev_num > 0) {
    /* register unsigned pevsize = sizeof(struct prf_mode_s) * (unsigned)prfc->prf_ev_num; */
    if (prfc->pev != NULL) {
      kfree(prfc->pev);
      /* kfree_s(prfc->pev, pevsize); */
    }
    prfc->prf_ev_num = 0;
    prfstat &= ~PRF_EVWRITE;	/* test */
  }
  return(0);
}

static int prfc_reset(struct prfc_s *prfc) {
  int i;
  prfstat &= PRF_ALLOC;		/* only hold alloc flag */
  for (i = 0; i < smp_num_cpus; i++) {
    prfc->cpu[i].wr_ctr_idx = 0;
    prfc->cpu[i].rd_ctr_idx = 0;
    prfc->cpu[i].evc_idx = 0;
#ifdef EVCNT_CYCNT
    prfc->cpu[i].tsc_base.lo = 0;
    prfc->cpu[i].tsc_base.hi = 0;
#endif /* EVCNT_CYCNT */
  }
  prfc->rd_cpu = 0;
  prfc->rd_cnt = 0;
  prfc->collect_pid = -1;
  return(0);
}

static int prfc_free(struct prfc_s *prfc) {
  register int rc = 0;

  if (prfstat & PRF_ON) {	/* profiling on -> do not free */
    rc = -EINVAL;
    return(rc);
  }

  rc = prfc_free_events(prfc);

  if (prfstat & PRF_ALLOC) {
    register int i;
    for (i = 0; i < smp_num_cpus; i++) {
      if (prfc->cpu[i].ctr != NULL) {
        kfree(prfc->cpu[i].ctr);
   	prfc->cpu[i].ctr = NULL;
      }
    }
    prfstat &= ~PRF_ALLOC;	/* reset flag: memory freed */
    prfc->num = 0;
  }
  return(rc);
}

static int prfc_init(struct prfc_s *prfc, const int prfnum) {
  register int rc = 0;
  register unsigned prfsize = sizeof(struct prfctr_s) * (unsigned)prfnum;

  if (prfnum == 0) {
    return(prfc_free(prfc));	/* number==0 -> free */
  }

  if ((prfstat & PRF_ALLOC) == 0) {
    register int i;
    for (i = 0; i < smp_num_cpus; i++) {
      prfc->cpu[i].ctr = (struct prfctr_s *)kmalloc(prfsize, GFP_KERNEL);
      if (prfc->cpu[i].ctr == NULL) { /* no mem -> free memory already alloc. */
        for (i -= 1; i >= 0; i--) {
          kfree_s(prfc->cpu[i].ctr, prfsize);
  	  prfc->cpu[i].ctr = NULL;
        }
        return -ENOMEM;
      }
    }
    prfc->num = prfnum;
    prfstat |= PRF_ALLOC;	/* set flag: memory allocated */
    (void)prfc_reset(prfc);
  }
  return(rc);
}

static int prfc_set_events(struct prfc_s *prfc, const int event_num) {
  register int rc = 0;

  if (event_num == 0) {
    return(prfc_free_events(prfc));
  }

  if (prfc->prf_ev_num == 0) {
    prfc->prf_ev_num = event_num;
    {
      register unsigned pevsize = sizeof(struct prf_mode_s) * (unsigned)prfc->prf_ev_num;
      prfc->pev = (struct prf_mode_s *)kmalloc(pevsize, GFP_KERNEL);
      if (prfc->pev == NULL) { /* no mem */
        prfc->prf_ev_num = 0;
        return -ENOMEM;
      }
    }
    prfstat |= PRF_EVWRITE;
  }
  return(rc);
}


static struct prfc_s prfc = {0};

static int prf_active = 0;

/* ****************************************************************************
 * prfopen() - prf open
 *
 * Description:
 *
 * Parameters:
 *
 * Return values:
 *   int
 *
 * ****************************************************************************
 */
static int prf_open(struct inode *inode, struct file *file) {
  register int rc = 0;
  if (prf_active > 0) {
    return -EBUSY;
  }
  if (!(prfstat & PRF_ALLOC)) {
    MOD_INC_USE_COUNT;
  }
  prf_active = 1;
  return(rc);
}


/* ****************************************************************************
 * prfclose() - prf close
 *
 * Description:
 *
 * Parameters:
 *
 * Return values:
 *   int
 *
 * ****************************************************************************
 */
#if LINUX_VERSION_CODE >= LINUX_V_NEWFS
static int prf_close(struct file *file) {
#else
static int prf_close(struct inode *inode, struct file *file) {
#endif
  prf_active = 0;
  if (!(prfstat & PRF_ALLOC)) {
    MOD_DEC_USE_COUNT;
  }
  return 0;  /* 30.6.2000: void->int; 0=no errors */
}

/* ****************************************************************************
 * prfread() - prfread
 *
 * Description:
 *
 * Parameters:
 *
 * Return values:
 *   int
 *
 * ****************************************************************************
 */
#if LINUX_VERSION_CODE >= LINUX_V_NEWFS
static ssize_t prf_read(struct file *file, char *buf, size_t count, loff_t *ppos) {
#else
static int prf_read(struct inode *inode, struct file *file, char *buf, int count) {
#endif 
  register int rc = 0;
  register int cpu = prfc.rd_cpu;
  register int cnt = prfc.rd_cnt;
  register int rd_ctr;

  rd_ctr = prfc.cpu[cpu].rd_ctr_idx;
  if (prfc.cpu[cpu].ctr == NULL) {
    return -ENXIO;	/* rc: counter does not exist */
  }

  if (rd_ctr + cnt > prfc.num) {	/* read in 2 parts? */
    cnt = prfc.num - rd_ctr;
  }

  {
    caddr_t data = (caddr_t)(prfc.cpu[cpu].ctr + rd_ctr);
    int data_len = (unsigned)cnt * sizeof(struct prfctr_s);

    if (count < data_len) {
      return -EINVAL;
    }

#if LINUX_VERSION_CODE >= 0x020104
    if (copy_to_user(buf, data, data_len)) {
      return -EFAULT;
    }
#else
    rc = verify_area(VERIFY_WRITE, buf, data_len);
    if (rc) {
      return(rc);
    }
    memcpy_tofs(buf, data, data_len);
#endif
    rc = data_len;
  }

  rd_ctr += cnt;
  if (rd_ctr >= prfc.num) {	/* modulo prfc.num */
    rd_ctr = 0;
  }
  prfc.cpu[cpu].rd_ctr_idx = rd_ctr;	/* set new read index */

  prfc.rd_cnt -= cnt;	/* maybe two parts */

  if (prfc.rd_cnt == 0) {
    prfstat &= ~PRF_RESULT;
  }
  return(rc);
}



/* ****************************************************************************
 * prfwrite() - prf write
 *
 * Description:
 *
 * Parameters:
 *
 * Return values:
 *   int
 *
 * ****************************************************************************
 */
#if LINUX_VERSION_CODE >= LINUX_V_NEWFS
static ssize_t prf_write(struct file *file, const char *buf, size_t count, loff_t *ppos) {
#else
static int prf_write(struct inode *inode, struct file *file, const char *buf, int count) {
#endif
  register int rc = 0;
  if ((prfstat & PRF_EVWRITE) == 0) {
    return -EINVAL;
  }
  if (count < 0) {
    return -EINVAL;
  }

#if LINUX_VERSION_CODE >= 0x020104
#else
  rc = verify_area(VERIFY_READ, (void *)buf, count);
  if (rc) {
    return(rc);
  }
#endif

  {
    register unsigned pevsize = sizeof(struct prf_mode_s) * (unsigned)prfc.prf_ev_num;
    if (prfc.pev == NULL) { /* no mem */
      return -EINVAL;
    }
    if (count != (int)pevsize) {
      return -ENOSPC;
    }

#if LINUX_VERSION_CODE >= 0x020104
    if (copy_from_user((caddr_t)prfc.pev, buf, count)) {
      return -EFAULT;
    }
#else
    memcpy_fromfs((caddr_t)prfc.pev, buf, count);
#endif

  }
  if (rc == 0) {
    rc = count;
    prfstat &= ~PRF_EVWRITE;
  }
  return(rc);
}


/* ****************************************************************************
 * prfioctl() - prf ioctl
 *
 * Description:
 *
 * Parameters:
 *
 * Return values:
 *   int
 *
 * ****************************************************************************
 */
static int prf_ioctl(struct inode *inode, struct file *file, unsigned int cmd, unsigned long arg) {
  register int rc = 0;
  register int rval = 0;

  if (MINOR(inode->i_rdev) == EVCNT_MINOR) {
    return(evcnt_ioctl(inode, file, cmd, arg));
  } else if (MINOR(inode->i_rdev) != PRF_MINOR) {
    return -ENODEV;     /* wrong minor number */
  }

  switch (cmd) {
    /* Change the default commands 1..3 to return an error */
    case PRF_GETCNTS: /* unused */
    case PRF_SETCNTS: /* unused */
    case  PRF_SETMODE: /* unused: */
      rc = -EINVAL;
    break;

    case PRF_VERSION:
      rc = evcnt_ioctl(inode, file, cmd, arg) + (1 << 8);
        /* get evcnt version and add 1``8=256 */
    break;

    case PRF_INIT:	/* initialize counter array, arg = #entries, 0=free */
      rc = prfc_init(&prfc, arg);
    break;

    case PRF_FREE:
      rc = prfc_free(&prfc);
    break;    

    case PRF_RD_STATE: /* read state */
      rval = prfstat;
    break;

    case PRF_RD_BUFNUM: /* read buffer num */
      rval = prfc.num;
    break;

    case PRF_IO_MCONTROL:
      if (!(prfstat & PRF_ON)) {	/* will be switched on? */
        register int i;
        for (i = 0; i < smp_num_cpus; i++) {
          prfc.cpu[i].evc_idx = -1;	/* first sample is invalid */
        }
      }
      prfstat = (prfstat & ~(unsigned)PRF_ON) | (unsigned)(arg & PRF_ON);
      rval = arg & PRF_ON;
#ifndef WITH_IRQ
#endif
    break;

    case PRF_RD_IDX: /* read cnt ctr_idx of CPU "arg" */
      if (arg >= smp_num_cpus) {
        rc = -ENXIO;
      } else {
        rval = prfc.cpu[arg].wr_ctr_idx; /* reading is maybe unsave !! */
      }
    break;

    case PRF_RD_CPUCNT: /* read number of CPUs */
      rval = smp_num_cpus;
    break;

    case PRF_RESET:
      rc = prfc_reset(&prfc);
    break;

    case PRF_SET_RDCPU: /* set read CPU number, return number of read bytes */
      if (arg >= smp_num_cpus) {
        rc = -ENXIO;
      } else {
        prfc.rd_cpu = arg;
        prfc.rd_cnt = prfc.cpu[arg].wr_ctr_idx - prfc.cpu[arg].rd_ctr_idx;
        if (prfc.rd_cnt < 0) {
          prfc.rd_cnt += prfc.num;
        }
        prfstat |= PRF_RESULT;
        rval = prfc.rd_cnt;
      }
    break;  

    case PRF_SET_EVENTS: /* set event numbers, arg=#samples, use write to set! */
      if (prfstat & PRF_ON) {
        rc = -EBUSY;
      } else {
        rc = prfc_set_events(&prfc, arg);
      }
    break;

    case PRF_SET_PID:   /* set PID >=0 to measure, -1 = all (default) */
      if (prfstat & PRF_ON) {
        rc = -EBUSY;
      } else {
        prfc.collect_pid = arg;
      }
    break;

    default:
      rc = -EINVAL;
    break;
  }
  if (rc) {
    return(rc);
  } else return (rval);
}


/* ... */

#if defined(I586_EVCNT) || defined(I586MMX_EVCNT)
static void ev_ix86_set_mode(register unsigned e0, register unsigned e1) {
  register hint h;
  register hint tmp;
  /* set mode for counter 0 ... */
  read_msr(h.lo, h.hi, MSR_EVCTL_R);
  tmp.lo = e0;	/* mode cnt 0 */
  tmp.hi = (tmp.lo >> 16) & 0xffff;	/* mode control cnt0 */
  tmp.lo &= 0xffff;
  h.lo &= ~(EVCTL_TYPE0 | EVCTL_CPL0 | EVCTL_ES0);
  /* clear counter 0: type, cpl, event (b0..5) */
  h.lo |= (tmp.lo & EVCTL_ES0) | ((tmp.hi << 6) & (EVCTL_TYPE0 | EVCTL_CPL0));
  /* set event, cpl,type */

  /* and for counter 1 ... */
  tmp.lo = e1;	/* mode cnt 1 */
  tmp.hi = (tmp.lo >> 16) & 0xffff;	/* mode control cnt1 */
  tmp.lo &= 0xffff;
  h.lo &= ~(EVCTL_TYPE1 | EVCTL_CPL1 | EVCTL_ES1);
  /* clear counter 1: type, cpl, event */
  h.lo |= ((tmp.lo << 16) & EVCTL_ES1) | ((tmp.hi << 22) & 
    (EVCTL_TYPE1 | EVCTL_CPL1));  /* set event, cpl, type */
  write_msr(h.lo, h.hi, MSR_EVCTL_R);	/* write back */

  /* And initialize the counters with 0... */
  write_msr(0, 0, MSR_CNT0_R);
  write_msr(0, 0, MSR_CNT1_R);
}
#endif /* I586_EVCNT */

#if defined(I686_EVCNT) || defined(I686MMX_EVCNT) || defined(ATHLON_EVCNT)
static void ev_ix86_set_mode(register unsigned e0, register unsigned e1) {
  register hint h;
  register hint tmp;
  /* set mode for counter 0 ... */
  /* read_msr(h.lo, h.hi, MSR_EVCTL0_R); */
  h.hi = 0;	/* clear high dword */
  tmp.lo = e0;  /* mode cnt 0 */
  tmp.hi = (tmp.lo >> 16) & 0xffff;  /* mode control cnt0 */
  tmp.lo &= 0xffff;
  h.lo = 0;  /* clear event select */
  h.lo |= (tmp.lo & (EVCTL_ES | EVCTL_US)) | ((tmp.hi << 16) & (EVCTL_TYPE | EVCTL_CPL)) | EVCTL_EI;
  /* set event+unit mask, cpl, type, enable counting for both counters */
  write_msr(h.lo, h.hi, MSR_EVCTL0_R);	/* write event select 0 */

  /* and for counter 1 ... */
  /* read_msr(h.lo, h.hi, MSR_EVCTL1_R); */
  tmp.lo = e1;		/* mode cnt 1 */
  tmp.hi = (tmp.lo >> 16) & 0xffff;	/* mode control cnt1 */
  tmp.lo &= 0xffff;
  h.lo = 0;  /* clear event select */
#if defined(ATHLON_EVCNT)
  h.lo |= (tmp.lo & (EVCTL_ES | EVCTL_US)) | ((tmp.hi << 16) & (EVCTL_TYPE | EVCTL_CPL)) | EVCTL_EI;
#else
  h.lo |= (tmp.lo & (EVCTL_ES | EVCTL_US)) | ((tmp.hi << 16) & (EVCTL_TYPE | EVCTL_CPL));
#endif
  /* set event, cpl, type */
  write_msr(h.lo, h.hi, MSR_EVCTL1_R);	/* write back */

  /* And initialize the counters with 0... */
  write_msr(0, 0, MSR_CNT0_R);
  write_msr(0, 0, MSR_CNT1_R);
}
#endif /* I686_EVCNT */


static int ev_get_cnt(unsigned *cnts, struct prfc_percpu_s *prfc_cpu) {
  register unsigned long c0, c1, c_high;
  read_msr(c0, c_high, MSR_CNT0_R);
  read_msr(c1, c_high, MSR_CNT1_R);
  cnts[0] = c0;	/* save only low 32 bit! */
  cnts[1] = c1;
#ifdef EVCNT_CYCNT
  {
    register hint *tsc_b = &prfc_cpu->tsc_base;
      /* = &prfc.cpu[cpu_id].tsc_base; */
    read_tsc(c0, c_high);
    if (c0 >= tsc_b->lo) {
      c0 -= tsc_b->lo;
    } else {
      c0 -= tsc_b->lo - ULONG_MAX;
      /* c_high--; */
    }
    /* c_high -= tsc_b->hi; */
    cnts[2] = c0;	/* only low 32 bit */
  }
#endif /* EVCNT_CYCNT */
  return(0);
}

static void ev_set_mode_cnt(unsigned *ev_nums, struct prfc_percpu_s *prfc_cpu) {
  register unsigned e0 = ev_nums[0];
  register unsigned e1 = ev_nums[1];
  ev_ix86_set_mode(e0, e1);
#ifdef EVCNT_CYCNT
  {
    register hint *tsc_b = &prfc_cpu->tsc_base;
      /* = &prfc.cpu[cpu_id].tsc_base; */
    
    read_tsc(tsc_b->lo, tsc_b->hi);	/* set new base */
  }
#endif /* EVCNT_CYCNT */
}



/* ****************************************************************************
 * prfintr() - prf interrupt handler
 *
 * Description:
 * Interrupt function called by the clock handler every 10ms in clock.c,
 * if (prfstat & PRF_ON).
 *
 * Parameters:
 *   struct pt_regs * regs	PC/IP during exception in instruction_pointer()
 *
 * Return values:
 *   ErrCode
 *
 * ****************************************************************************
 */
#define my_min(a,b) ((a) <= (b) ? (a) : (b))

void prfintr(struct pt_regs *regs) {
#ifdef __SMP__
  const int cpu_id = prf_cpu_logical(smp_processor_id());
#else /* !__SMP__ */
  const int cpu_id = smp_processor_id();
#endif /* __SMP__ */
  register struct prfc_percpu_s *prfc_cpu = &prfc.cpu[cpu_id];
  register struct prfctr_s *ip = prfc_cpu->ctr;
  register int idx = prfc_cpu->wr_ctr_idx; /* write counter index */
  if (ip == NULL) {
    return;	/* error: counter array not initialized */
  }

  ip += idx;	/* point to ip[idx] */
  ip->vaddr = (ad_t)instruction_pointer(regs);	/* pc or ip */
  ip->pid = current->pid;

  if (prfc.collect_pid >= 0) {
    if (prfc.collect_pid != ip->pid) {
      return;   /* PID's different -> do not collect */
    }
  }

  ip->ppid = current->p_pptr->pid;

  ip->ticks = jiffies;
  ip->cpu = cpu_id;

#ifdef WITH_CMDSTR
  {
    register unsigned slen = my_min(16, PRF_CMDLEN); /* 16=const in sched.h */
    strncpy(ip->cmdstr, current->comm, slen);
    ip->cmdstr[slen - 1] = '\0';
  }
#endif

  /* prf_ev_get_counters... */
  /* prf_mode_set_mode... */
  if (prfc.prf_ev_num > 0) { /* event sampling active? */
    register int evcidx = prfc_cpu->evc_idx;
    register int rc = 0;
    ip->sample_pos = evcidx;
    rc = ev_get_cnt(ip->ev_cnt, prfc_cpu);
    if ((evcidx == -1) || (rc != 0)) {	/* this sample is invalid? */
      register int i;
      for (i = 0; i < PRF_P_CNT_NUM; i++) {	/* yes -> clear counters */
        ip->ev_cnt[i] = 0;
      }
    }
    evcidx++;
    if (evcidx >= prfc.prf_ev_num) {
      evcidx = 0;
    }
    prfc_cpu->evc_idx = evcidx;
    ev_set_mode_cnt(prfc.pev[evcidx].ev_mode, prfc_cpu);
  } else {
    register int i;
    for (i = 0; i < PRF_P_CNT_NUM; i++) {
      ip->ev_cnt[i] = 0;
    }
    ip->sample_pos = -1;
  }

  idx++; /* increment counter index */
  if (idx >= prfc.num) {	/* modulo buffer size */
    idx = 0;
  }
  /* overflow, if (wr_idx+1) = rd_idx */
  if (idx != prfc_cpu->rd_ctr_idx) {	/* no overflow? */
    prfc_cpu->wr_ctr_idx = idx;	/* write ctr_idx back */
  } else {
    prfstat |= PRF_OFLOW;	/* set overflow flag */
  }
}


#ifdef WITH_IRQ
static void prf_interrupt(int irq, void *dev_id, struct pt_regs *regs) {
  if (prfstat & PRF_ON) {
#ifdef __SMP__
#if LINUX_VERSION_CODE >= 0x020104
 /* extern int smp_call_function (void (*func) (void *info), void *info, int retry, int wait); */
  smp_call_function(prf_interrupt_other, NULL, 0, 0);
#else
    smp_message_pass(MSG_ALL_BUT_SELF, MSG_SPROF_TICK, 0L, 0);
#endif
#endif /* __SMP__ */
    prfintr(regs);
  }
}

#ifdef __SMP__

#ifdef MODULE
void prf_interrupt_other(int irq, void *dev_id, struct pt_regs *regs) {  
  if (smp_msg_id == xx) {	/* does not work! */
    if (prfstat & PRF_ON) {   /* this check is not really needed */
      prfintr(regs); 
    }                                                                             }
}  
#else	/* !MODULE */
void prf_interrupt_other(struct pt_regs *regs) {        
  if (prfstat & PRF_ON) {   /* this check is not really needed */
      prfintr(regs); 
  }                                                                             
}  
#endif	/* MODULE */

#endif	/* __SMP__ */


#else	/* !WITH_IRQ */

void prf_interrupt(struct pt_regs *regs) {        
  if (prfstat & PRF_ON) {                                                       
      prfintr(regs); 
  }                                                                             
}  
#endif	/* WITH_IRQ */


static struct file_operations prf_fops = {
  NULL,         /* prf_lseek */
  prf_read,
  prf_write,
  NULL,         /* prf_readdir */
  NULL,         /* prf_select */
  prf_ioctl,
  NULL,         /* prf_mmap */
  prf_open,
  prf_close	/* this is flush! (30.6.2000) */
};


#define DEF_ID (void *)1	/* do not use NULL for shared timer interrupt */

static void free_it(void) {
#ifdef WITH_IRQ
  free_irq(PRF_IRQ, DEF_ID);

#ifdef __SMP__
#ifdef MODULE
  #if LINUX_VERSION_CODE >= 0x020104
  #else
    free_irq(MSG_IRQ, DEF_ID);
  #endif
#endif
#endif

#else
#endif
}


static int prf_major = PRF_MAJOR;	/* we may get something else */

#ifdef MODULE

#define prf_init init_module

void cleanup_module(void) {
  free_it();
  unregister_chrdev(prf_major, PRF_NAME);
}

#endif  /* MODULE */


int prf_init(void) {
#ifdef CHECK_MSR
  if ((x86_capability & MSR_PRESENT) != MSR_PRESENT) {
    printk("evcnt: Pentium MSR not present.\n");
    return -EIO;
  }
#endif /* CHECK_MSR */

#ifdef WITH_IRQ
  if (request_irq(PRF_IRQ, prf_interrupt, /* SA_INTERRUPT | */ SA_SHIRQ, PRF_NAME, DEF_ID)) {
    printk("prf: IRQ %d is not free.\n", PRF_IRQ);
    return -EIO;
  }

#ifdef __SMP__
#ifdef MODULE
  #if LINUX_VERSION_CODE >= 0x020104
  #else   
  if (request_irq(MSG_IRQ, prf_interrupt_other, SA_SHIRQ, PRF_NAME, DEF_ID)) {
    printk("prf: IRQ %d is not free.\n", MSG_IRQ);
    return -EIO;
  }
  #endif
#endif
#endif

#else /* !WITH_IRQ */
#endif /* WITH_IRQ */

  if (register_chrdev(PRF_MAJOR, PRF_NAME, &prf_fops) < 0) {
    int major = 0;      /* get free number */
    if ((major = register_chrdev(major, PRF_NAME, &prf_fops)) <= 0) {
      printk("prf: cannot get major.\n");
      free_it();
      return -EIO;
    } else {
      printk("prf: device registered using major %d\n", major);
      prf_major = major;
    }
  }

  return(0); 
}

/* end of prf.c */
