/*
 * evcnt.c -- Evcnt Device for Pentium (i586) and Pentium Pro (i686)
 *
 * Version 0.2
 *
 * Copyright (C) 1996, 07/1997 by Marco Vieth
 *
 * 01.07.2000 0.21 MV  adapted for kernel 2.2x: file_operations
 *
 */

/*
 * The Pentium has two built-in event counter register (40 bit each) which
 * can count several events (cache misses, pipeline stalls...).
 * MSR (model specific register) 0x11 specifies the event types and
 * MSRs 0x12, 0x13 are the counters.
 *
 * ---
 * The Pentium Pro has two built-in event counter register (40 bit each) which
 * can count several events (cache misses, pipeline stalls...).
 * MSRs (model specific register) 0x186 and 0x187 specifiy the event types and
 * MSRs 0xC1, 0xC2 are the counters.
 * ---
 *
 * In general, the MSRs can only be accessed from CPL 0, so
 * this device driver is needed.
 *
 * This device supplies the following commands:
 * typedef struct { unsigned long lo; unsigned long hi; } hint;
 * hint h[EVCNT_CNUM], one_h;
 * 
 * - ioctl(fd, EV_SETMODE, &one_h)
 *   Set the measurement modes: h[0] for cnt0, h[1] for cnt1.
 *   On i586 they range from 0 to 0x29, on i686 up to 0xff.
 * - ioctl(fd, EV_SETCNTS, &h)
 *   Set (initialize) the counters cnt0, cnt1 with h[0], h[1] respectively.
 *   The TSC_base is set to h[2]   (if EVCNT_CYCNT is defined).
 * - ioctl(fd, EV_GETCNTS, &h)
 *   Get the counter values cnt0, cnt1 to h[0], h[1] respectively.
 *   h[2] is loaded from the TSC and subtracted by.
 *
 *
 * - ioctl(fd, EV_GETTSC, &one_h)	(additional)
 *   Get the TSC (Time Stamp Counter).
 *
 *
 * 1. Installing the device:
 *	mknod /dev/evcnt c 63 0
 *
 * 1.1 Installing as module (for a kernel which supports modules, e.g. 0.99p4):
 * Compile with -DMODULE:
 * gcc -D__KERNEL__ -I/usr/src/linux/include -Wall -Wstrict-prototypes -O2 -fomit-frame-pointer -fno-strength-reduce -pipe -m486 -malign-loops=2 -malign-jumps=2 -malign-functions=2 -DCPU=586 -DMODULE  -c -o evcnt_ix86.o evcnt_ix86.c
 * There are two possibilities:
 * 1.1.1 Install the module permanently (fast test):
 * Install the module permanently:
 *      /sbin/insmod evcnt_ix86.o
 * To remove the module, use:
 *      /sbin/rmmod evcnt_ix86
 *
 * 1.1.2 Activate autoloading of the module by kerneld:
 * Copy evcnt_ix86.o into the module directory:
 * 	cp -p evcnt_ix86.o /lib/modules/`uname -r`/misc
 * Create new module dependency file:
 *	depmod -a
 * Insert device number in /etc/conf.modules (for kerneld):
 *	alias char-major-63  evcnt_ix86
 * Without kerneld, use /sbin/modprobe to load the module permanently:
 *	modprobe evcnt_ix86
 * ("/sbin/rmmod evcnt_ix86" removes it, /sbin/lsmod lists loaded modules.)
 *
 *
 * 1.2 Installing as regular device (in the kernel):
 * Copy the driver into the linux source directory:
 *	cp -p evcnt_ix86.c evcnt_ix86.o /usr/src/linux/drivers/char
 * Edit /usr/src/linux/drivers/char/Config.in and insert (before endmenu):
 *		tristate 'Event Counter Support' CONFIG_EVCNT
 * Edit /usr/src/linux/drivers/char/misc.c and insert (in misc_init):
 * 	#ifdef CONFIG_EVCNT
 *        evcnt_init();
 *	#endif
 * 
 * And add a section to /usr/src/linux/drivers/char/Makefile:
 *      ifeq ($(CONFIG_EVCNT),y)
 *        M = y
 *        L_OBJS += evcnt.o
 *      else
 *        ifeq ($(CONFIG_EVCNT),m)
 *          M_OBJS += evcnt.o
 *          MM = m
 *        endif
 *      endif
 *
 *
 * Change config and recompile the kernel:
 *	make menuconfig  -> activate "Event Counter Support" (Character Devices)
 *	make dep ; make clean
 *	make zImage
 *	make modules ; make modules_install  (for modules, maybe also depmod -a)
 * And boot with the new zImage.
 *
 * Remark for booting a new kernel:
 * For Lilo you may have to copy the new kernel image to the boot kernel
 * (see /etc/lilo.conf) ...
 *      cp -p /usr/src/linux/arch/i386/boot/zImage  /boot/vmlinux.???
 *  and reconfigure lilo:
 *      /sbin/lilo
 * 
 *
 */


#include <linux/module.h>

#include <linux/ioctl.h>	/* _IOR, _IOW */
#include <linux/errno.h>
#include <linux/kernel.h>
#include <linux/major.h>
#include <linux/sched.h>
#include <linux/lp.h>
#include <linux/malloc.h>
#include <linux/ioport.h>
#include <linux/fcntl.h>

#include <asm/io.h>
#include <asm/segment.h>
#include <asm/system.h>

/* #define EVCNT_DEBUG */

#include "evcnt_ix86.h"
/* #include "evcnt_feat_ix86.h" */


#include "msr_ix86.h"

/* 
 * We can only check for MSR present, if evcnt is not compiled as a module
 * since we need the global variable x86_capability from setup.c.
 * Another possibility would be to use get_cpuinfo() from setup.c but it
 * copies all sort of data into an "unlimited" buffer.
 */

#ifndef MODULE
#define CHECK_MSR
#define MSR_PRESENT 0x20	/* b5 of x86_capability (EDX from CPUID) */
  /* x86_capability is defined in asm/processor.h */
#endif	/* MODULE */




#include "evcnt_ioctl.c"	/* include evcnt_ioctl command */


static int evcnt_active = 0;

static int evcnt_open(struct inode *inode, struct file *file) {
  if (evcnt_active > 0) {
    return -EBUSY;
  }
#ifdef EVCNT_DEBUG
  printk("evcnt_open\n");
#endif /* EVCNT_DEBUG */
  MOD_INC_USE_COUNT;
  evcnt_active = 1;
  return 0;
}

#if LINUX_VERSION_CODE >= LINUX_V_NEWFS
static int evcnt_release(struct file *file) {
#else
static int evcnt_release(struct inode *inode, struct file *file) {
#endif
#ifdef EVCNT_DEBUG
  printk("evcnt_release\n");
#endif /* EVCNT_DEBUG */
  MOD_DEC_USE_COUNT;
  evcnt_active = 0;
  return 0;  /* 30.6.2000: void->int; 0=no errors */
}


/* some undefined commands: */


#if LINUX_VERSION_CODE >= LINUX_V_NEWFS
static ssize_t evcnt_read(struct file *file, char *buf, size_t count, loff_t *ppos) {
#else
static int evcnt_read(struct inode *inode, struct file *file, char *buf, int count) {
#endif
  return -EINVAL;
}

#if LINUX_VERSION_CODE >= LINUX_V_NEWFS
static ssize_t evcnt_write(struct file *file, const char *buf, size_t count, loff_t *ppos) {
#else                      
static int evcnt_write(struct inode *inode, struct file *file, const char *buf, int count) {
#endif
  return -EINVAL;
}


static struct file_operations evcnt_fops = {
  NULL,		/* evcnt_lseek */
  evcnt_read,	/* evcnt_read */
  evcnt_write,	/* evcnt_write */
  NULL,		/* evcnt_readdir */
  NULL,		/* evcnt_select */
  evcnt_ioctl,
  NULL,		/* evcnt_mmap */
  evcnt_open,
  evcnt_release /* this is flush! (30.6.2000) */
};


static int evcnt_major = EVCNT_MAJOR;       /* we may get something else */

#ifdef MODULE

#define evcnt_init init_module

void cleanup_module(void) {
  unregister_chrdev(evcnt_major, EVCNT_NAME);
}

#endif	/* MODULE */


int evcnt_init(void) {
#ifdef CHECK_MSR
  if ((x86_capability & MSR_PRESENT) != MSR_PRESENT) {
    printk("evcnt: Pentium MSR not present.\n");
    return -EIO;
  }
#endif /* CHECK_MSR */
  if (register_chrdev(EVCNT_MAJOR, EVCNT_NAME, &evcnt_fops) < 0) {
    int major = 0;	/* get free number */
    if ((major = register_chrdev(major, EVCNT_NAME, &evcnt_fops)) <= 0) {
      printk("evcnt: cannot get major.\n");
      return -EIO;
    } else {
      printk("evcnt: device registered using major %d\n", major);
      evcnt_major = major;
    }
  }
  return(0); 
}

/* end */
