//: K_FMAP.CPP - Kohonen's Feature Map Algorithm
// 2.1.1995
//

#include <iostream.h>
#include <fstream.h>
#include <conio.h>	// kbhit()
#include <stdlib.h>	// random
#include <math.h>	// sqrt
#include <string.h>	//


#include "k_matrix.h"
#include "k_fmap.h"
#include "bgi.h"


void featuremap::error(char *msg1, char *msg2) {
  cerr << "matrix error: " << msg1 << " " << msg2 << endl;
  exit(1);
}

featuremap::featuremap(int n_input, int n_x1, int n_x2) {
  strcpy(print_file, "EPS_0000.EPS");
  f_map = matrix::matrix(n_input, n_x1 * n_x2, 0);
  ini_mode = RANDOM;
  ini_distrib = 0.25;	// init distribution
  shw_points = 1;
  displ_adapt = 100;
  inp_shape = RECTANGLE;
  inp_s1 = 1;		// parameter: size of rectangle, diagonal of circle

  init_parameter();

  num_x1 = n_x1;
  num_x2 = n_x2;

  x_width = Graphics::xmax()/2;	// display: x_width
  y_width = Graphics::ymax()/2;	// display: y_width
}


void featuremap::adapt(matrix& ivec) {
  if (ivec.rows() != rows())
    error("number of rows must be identical in featuremap()");
  if (ivec.cols() != 1)
    error("number of cols must be 1 in featuremap()");

  if (shw_points) { 	// display inputs as points?
    Graphics::plot(ivec.val(0,0), ivec.val(1,0));
  }

  matrix y(cols());   		// cols() = num_x1 * num_x2
  scalar sum, temp, min_sum = 1.1;	// 1.1 = impossible maximum
  int m_x1 = 0;			// position of the minumum output neuron
  int m_x2 = 0;

  for (int x1 = 0; x1 < num_x1; x1++) 	// for every output neuron ...
    for (int x2 = 0; x2 < num_x2; x2++) {
      sum = 0.0;
      for (int j = 0; j < rows(); j++) {	// for every input
	temp = (ivec.val(j, 0) - val(j, x1+num_x1*x2));
	sum += temp*temp;
      }
      y.val(x1+num_x1*x2, 0) = sum;	// save distances
      if (sum < min_sum) {
	min_sum = sum;	// save minimal activation
	m_x1 = x1; m_x2 = x2;
      }
    }
  // now we've found a neuron at position m_x1,m_x2 with a minimal activation
  // (distance to the input vector ivec)

  // adaption:


  double neighbor = 1;
  double diff1, diff2;

  k_step++;	// timestep (now at least 1)
//  double sigma = 1/((double)k_step/3000);//2; // 1/((double)k_step*0.01);

  for (x1 = 0; x1 < num_x1; x1++)
    for (int x2 = 0; x2 < num_x2; x2++) {
      diff1 = ((double)x1-m_x1); // /num_x1;
      diff2 = ((double)x2-m_x2); // /num_x2;
      neighbor = exp( -( diff1*diff1 + diff2*diff2) / (2 * sigma*sigma));
// optional: show the neighbourhood function:
//Graphics::plot((double)(k_step % 3000)/3000,neighbor);
      for (int i = 0; i < rows(); i++) {
	val(i, x1+num_x1*x2) += alpha * neighbor*(ivec.val(i, 0) -
	  val(i, x1+num_x1*x2));
      }
    }

  // decrease the learning factor:
  if ((alpha-alpha_dec) >= alpha_min) {
    alpha -= alpha_dec;
  }

  // decrease the neighbourhood factor:
  if ((sigma-sigma_dec) >= sigma_min) {
    sigma -= sigma_dec;
  }
}

void featuremap::show() {
  Graphics::clearwindow();
//print();		// test

  // draw border:
//  Graphics::move(0, 0); Graphics::draw(1, 0); Graphics::draw(1, 1); Graphics::draw(0, 1); Graphics::draw(0, 0);

  // connect neurons with "horizontal" lines
  for (int x2 = 0; x2 < num_x2; x2++) {
    Graphics::move(val(0, 0+x2*num_x1), val(1, 0+x2*num_x1));
    for (int x1 = 1; x1 < num_x1; x1++) {
      Graphics::draw(val(0, x1+x2*num_x1), val(1, x1+x2*num_x1));
    }
  }

  // connect neurons with "vertical" lines
  for (int x1 = 0; x1 < num_x1; x1++) {
    Graphics::move(val(0, x1+0*num_x1), val(1, x1+0*num_x1));
    for (int x2 = 1; x2 < num_x2; x2++) {
      Graphics::draw(val(0, x1+x2*num_x1), val(1, x1+x2*num_x1));
    }
  }
}





void featuremap::simulate(int steps) {

  // use a big bigger window to draw the border ...
  Graphics::window(x_width+2, y_width+2);
  Graphics::move(0, 0); Graphics::draw(1, 0); Graphics::draw(1, 1); Graphics::draw(0, 1); Graphics::draw(0, 0);

  Graphics::window(x_width, y_width);	// define our window for drawing
  Graphics::clearwindow();	// and clear it (the border remains)

  show();
  matrix in_vec(rows());

  scalar x1, x2;

  const scalar center = 0.5;
  while (!kbhit() && steps--) {
    for (int i = 0; i < displ_adapt; i++) {

      switch(inp_shape) {
	case RECTANGLE:
	    x1 = (double)rand()/(RAND_MAX+1) * inp_s1 - inp_s1/2 + center;
	    x2 = (double)rand()/(RAND_MAX+1) * inp_s1 - inp_s1/2 + center;
	break;

	case TRIANGLE:
	  do {
	    x1 = (double)rand()/(RAND_MAX+1) * inp_s1 - inp_s1/2 + center;
	    x2 = (double)rand()/(RAND_MAX+1) * inp_s1 - inp_s1/2 + center;
	  } while (x2 > -x1+1);
	break;

	case CIRCLE:
	  do {
	    x1 = (double)rand()/(RAND_MAX+1) * inp_s1 - inp_s1/2;
	    x2 = (double)rand()/(RAND_MAX+1) * inp_s1 - inp_s1/2;
	  } while (x1*x1 + x2*x2 > inp_s1/2*inp_s1/2);
	  x1 += center;
	  x2 += center;
	break;

	case L_SHAPE:
	  do {
	    x1 = (double)rand()/(RAND_MAX+1) * inp_s1;
	    x2 = (double)rand()/(RAND_MAX+1) * inp_s1;
	  } while ((x1 > inp_s1/2) && (x2 > inp_s1/2));
	  x1 +=  - inp_s1/2 + center;
	  x2 +=  - inp_s1/2 + center;
	break;

	default:
	  cerr << "no shape !";
	break;
      }

      in_vec.val(0,0) = x1;
      in_vec.val(1,0) = x2;
      adapt(in_vec);
    }
    show();
  }
//  if (inp_s1 < 1) {	// draw a smaller box
    if (inp_shape == CIRCLE) {
      Graphics::circle(0.5, 0.5, inp_s1/2);
    }
    else {
      Graphics::move(0.5-inp_s1/2, 0.5-inp_s1/2);
      Graphics::draw(0.5+inp_s1/2, 0.5-inp_s1/2);
      if (inp_shape == L_SHAPE) {
	Graphics::draw(0.5+inp_s1/2, 0.5);
	Graphics::draw(0.5, 0.5);
	Graphics::draw(0.5, 0.5+inp_s1/2);
      }
      else if (inp_shape != TRIANGLE) {
	Graphics::draw(0.5+inp_s1/2, 0.5+inp_s1/2);
      }
      Graphics::draw(0.5-inp_s1/2, 0.5+inp_s1/2);
      Graphics::draw(0.5-inp_s1/2, 0.5-inp_s1/2);
    }
//  }
  while (kbhit()) { getch(); }
}




int featuremap::input_yn(char *msg, int oldpara) {
  cout << msg << " (Y/N) (" << ((oldpara) ? "Y" : "N") << "): ";
  char in;
  cin.get(in);
  //cout << "in=" <<  (int)in << endl;
  if (in >= ' ') {
    oldpara = ((in == 'Y') || (in == 'y')) ? 1 : 0;
    cin.get();
  }
  return oldpara;
}

double featuremap::input_double(char *msg, const double oldpara,
  const double min, const double max) {

  double para;
  int ok;
  do {
    para = oldpara;
    cout << msg << " (" << oldpara << "): ";
    char in;
    cin.get(in);
    //cout << "in=" << (int)in << endl;
    if (in >= ' ') {
      cin.putback(in);
      cin >> para;
      cin.get();
    }
    if ((para < min) || (para > max)) {
      cout << "Valid values range from " << min << " to " << max << ". ";
      cout << "Please try again ..." << endl;
      ok = 0;
    }
    else ok = 1;
  } while (!ok);
  return para;
}

void featuremap::displ_options(void) {
  shw_points = input_yn("Display input vectors as points", shw_points);
  displ_adapt = (int)input_double("Number of input vectors to learn"
  " before display", displ_adapt, 1, 32000);
  x_width = (int)input_double("Display window: x-width", x_width, 10,
    Graphics::xmax()-2);
  y_width = (int)input_double("Display window: y-width", y_width, 10,
    Graphics::ymax()-2);
}


void featuremap::parameters(void) {
  alpha = input_double("Learning factor alpha", alpha, 0, 1);
  alpha_dec = input_double("  Decrease amount for learning factor alpha",
    alpha_dec, 0, 1);
  alpha_min = input_double("  Minimum alpha", alpha_min, 0, 1);

  sigma = input_double("Neighbourhood factor sigma", sigma, 0, 100);
  sigma_dec = input_double("  Decrease amount for factor sigma",
    sigma_dec, 0, 1);
  sigma_min = input_double("  Minimum sigma", sigma_min, 0, 1);

}

void featuremap::inp_option(void) {
  char select;
  cout << "Shapes: R)ectangle, T)riangle, C)ircle, L)-Shape\n";
  cout << "Your choice: ";
  cin >> select;
  cin.get();
  cout << endl;
  if ((select >= 'a') && (select <= 'z')) select -= 'a' - 'A';
  switch(select) {
    case 'R' :
      inp_shape = RECTANGLE;
      inp_s1 = input_double("Size of the rectangle", inp_s1, 0, 1);
    break;

    case 'T' :
      inp_shape = TRIANGLE;
      inp_s1 = input_double("Size of the triangle", inp_s1, 0, 1);
    break;

    case 'C' :
      inp_shape = CIRCLE;
      inp_s1 = input_double("Length of diagonal of the circle:", inp_s1, 0, 1);
    break;

    case 'L' :
      inp_shape = L_SHAPE;
      inp_s1 = input_double("Size of L-shape:", inp_s1, 0, 1);
    break;

    default:
      cout <<" no change.\n";
    break;
  }
}



void featuremap::init_parameter(void) {
  k_step = 0;		// timestep 0
  alpha = 0.065;
  alpha_dec = 0.00001;	// alpha decrease
  alpha_min = 0.0001;
  sigma = 2.4;
  sigma_dec = 0.0005;
  sigma_min = 0.6;

  if (ini_mode == RANDOM) {
    // init the weights randomly distributed around (0.5, 0.5)
    const scalar center = 0.5;
    for (int i = 0; i < rows(); i++)
      for (int j = 0; j < cols(); j++) {
	val(i, j) = ((double)rand()/(RAND_MAX+1))*ini_distrib
	  + center - ini_distrib/2;
      }
  }
  else {
    // init the weights to an ideal grid, only for rows()==2 !!
    double x1_step = ini_distrib / ((num_x1 > 1) ? num_x1 - 1 : 1);
    double x2_step = ini_distrib / ((num_x2 > 1) ? num_x2 - 1 : 1);

    for (int x1 = 0; x1 < num_x1; x1++)
      for (int x2 = 0; x2 < num_x2; x2++) {
	val(0, x1+x2*num_x1) = x1_step * x1 + ((1 - ini_distrib) / 2);
	val(1, x1+x2*num_x1) = x2_step * x2 + ((1 - ini_distrib) / 2);
	// values between 0..1
      }
  }
}



void featuremap::init_map() {
  int mode = (ini_mode == RANDOM);
  mode = input_yn("Init with random values", mode);
  if (mode) {
    ini_mode = RANDOM;
  }
  else {
    cout << "So init as a grid.\n";
    ini_mode = GRID;
  }
  ini_distrib = input_double("Size of region/grid: ", ini_distrib, 0.2, 1);
  init_parameter();
}


void featuremap::resize_map() {
  num_x1 = input_double("Map size: Neurons in x-direction", num_x1, 1, 500);
  num_x2 = input_double("Map size: Neurons in y-direction", num_x2, 1, 500);
  resize_matrix(rows(), num_x1 * num_x2, 0);
  init_parameter();
}


void featuremap::print_state() {
  cout << "Parameter: " << num_x1 << "x" << num_x2 << "-map; step=" <<
    k_step << ", alpha=" << alpha << ", sigma=" << sigma << endl;
}


void featuremap::ps_print() {
  ifstream infile("koho_eps.eps");
  if (!infile) {
    cerr << "couldn't open koho_eps.eps!\n";
    return;
  }
  ofstream outfile(print_file);
  if (!infile) {
    cerr << "couldn't open " << print_file << "!\n";
    return;
  }
  // copy template file ...
  char buf[100];
  while (infile.getline(buf, 100) && (strcmp(buf, "$F2psEnd") != 0)) {
    outfile.write(buf, strlen(buf));
    outfile.put('\n');
  }
  // show in postscript ...
  const double xm = 1000.0, ym = 1000.0;

  outfile << "% draw frame:\n";
  outfile << "n " << (int)(0*xm) << " " << (int)(0*ym) << " m\n";
  outfile << (int)(1*xm) << " " << (int)(0*ym) << " l\n";
  outfile << (int)(1*xm) << " " << (int)(1*ym) << " l\n";
  outfile << (int)(0*xm) << " " << (int)(1*ym) << " l\n";
  outfile << (int)(0*xm) << " " << (int)(0*ym) << " l s\n";


  outfile << "% draw shape:\n";
    if (inp_shape == CIRCLE) {
//      Graphics::circle(0.5, 0.5, inp_s1/2);
    }
    else {
      outfile << "n " << (int)((0.5-inp_s1/2)*xm) << " " <<
	(int)((0.5-inp_s1/2)*ym) << " m\n";
//      Graphics::move(0.5-inp_s1/2, 0.5-inp_s1/2);
      outfile << (int)((0.5+inp_s1/2)*xm) << " " <<
	(int)((0.5-inp_s1/2)*ym) << " l\n";
//      Graphics::draw(0.5+inp_s1/2, 0.5-inp_s1/2);
      if (inp_shape == L_SHAPE) {
	outfile << (int)((0.5+inp_s1/2)*xm) << " " <<
	  (int)((0.5)*ym) << " l\n";
//	Graphics::draw(0.5+inp_s1/2, 0.5);
	outfile << (int)((0.5)*xm) << " " <<
	  (int)((0.5)*ym) << " l\n";
//	Graphics::draw(0.5, 0.5);
	outfile << (int)((0.5)*xm) << " " <<
	  (int)((0.5+inp_s1/2)*ym) << " l\n";
//	Graphics::draw(0.5, 0.5+inp_s1/2);
      }
      else if (inp_shape != TRIANGLE) {
      outfile << (int)((0.5+inp_s1/2)*xm) << " " <<
	(int)((0.5+inp_s1/2)*ym) << " l\n";
//	Graphics::draw(0.5+inp_s1/2, 0.5+inp_s1/2);
      }
      outfile << (int)((0.5-inp_s1/2)*xm) << " " <<
	(int)((0.5+inp_s1/2)*ym) << " l\n";
//      Graphics::draw(0.5-inp_s1/2, 0.5+inp_s1/2);
      outfile << (int)((0.5-inp_s1/2)*xm) << " " <<
	(int)((0.5-inp_s1/2)*ym) << " l\n";
//      Graphics::draw(0.5-inp_s1/2, 0.5-inp_s1/2);
    }
  outfile << "s\n";


//  Graphics::move(0, 0); Graphics::draw(1, 0); Graphics::draw(1, 1); Graphics::draw(0, 1); Graphics::draw(0, 0);


  // connect neurons with "horizontal" lines
  outfile <<"% horizontal:\n";
  for (int x2 = 0; x2 < num_x2; x2++) {
    outfile << "n " << (int)(val(0, 0+x2*num_x1)*xm) << " " <<
      (int)(val(1, 0+x2*num_x1)*ym) << " m\n";
//    Graphics::move(val(0, 0+x2*num_x1), val(1, 0+x2*num_x1));
    for (int x1 = 1; x1 < num_x1; x1++) {
      outfile << (int)(val(0, x1+x2*num_x1)*xm) << " " <<
	(int)(val(1, x1+x2*num_x1)*ym) << " l\n";
//      Graphics::draw(val(0, x1+x2*num_x1), val(1, x1+x2*num_x1));
    }
    outfile << "s\n";
  }

  // connect neurons with "vertical" lines
  outfile <<"% vertical:\n";
  for (int x1 = 0; x1 < num_x1; x1++) {
    outfile << "n " << (int)(val(0, x1+0*num_x1)*xm) << " " <<
      (int)(val(1, x1+0*num_x1)*ym) << " m\n";
//    Graphics::move(val(0, x1+0*num_x1), val(1, x1+0*num_x1));
    for (int x2 = 1; x2 < num_x2; x2++) {
      outfile << (int)(val(0, x1+x2*num_x1)*xm) << " " <<
	(int)(val(1, x1+x2*num_x1)*ym) << " l\n";
//      Graphics::draw(val(0, x1+x2*num_x1), val(1, x1+x2*num_x1));
    }
    outfile << "s\n";
  }





  outfile.write(buf, strlen(buf));	// write last line

  cout << "File " << print_file << " written.\n";

  print_file[7]++;	// increase ascii-counter
}


//end
