//
// advent1.js - adventure interpreter
// (c) Marco Vieth, 2003
// http://www.benchmarko.de/
//
// 11.01.2003 0.8  OO design of AdvPlayer, AdvRoom, AdvObject
// 13.01.2003 0.9
// 14.01.2003 1.0  new action S (show images), some bugfixes
// 17.01.2003 1.1  event playback mode
// 20.01.2003 1.2  some corrections for StarOffice 5.2: cannot set option length; myparseint: remove blanks before numbers
// 21.01.2003 1.3  separate module debug1.js, advent1_main.js (for NGS js)
// 31.12.2003 1.31 adapted for new sshow1.js
// 02.01.2004 1.32 action 'D' corrected
// 12.01.2004 1.33 Netscape 3.x compatibility: don't use do..while, switch..case, substr
// 18.01.2004 1.34 some actions extended (Sx.y.z, Dx.y.z, Mx.y)
//
//

//
// Notes:
// - For debugging load debug1.js (this makes class Debug and object g_debug available)
// - Checked with NGS 0.2.5 (see advent1_main.js)
//

var g_debug = null; // maybe changed later...

var myint = Math.floor; // cast to integer for NGS, floor for others


// Verbs

// Messages


// Rooms (First numbers are room connections N, S, E, W, U, D; then a room image number)

// Action codes (new):
// 0. General:
// - I (if)
// - T (then)
// - E (else)
// - ; (separator, finish if/then/else; only needed if some other actions follow...)
//
// 1. Conditions:
// - ! (Not)
// - Fx,y (Flag x of object y is set)
// - Ox.y (Object x is in room y; if x or y are missing, current object or room is assumed)
//   - Ox.0 (Object x is hidden)         / O.0 (current object is hidden)
//   - Ox.1 (Object x is in inventory)   / O.1 (current object is in inventory)
//   - Ox. (Object x is in current room) / O.  (current object is in current room)
// - Px.y (Player x is in room y; if x or y are missing, current player or room is assumed)
//   - P.y (current player is in room y)
//
// 2. Actions:
// - Cx.y (Clear flag x of object y)
// - Fx.y (Set flag x of object y)
// - Mx.y (Print message x, with sex y)   / Mx (print message x with sex of object)
// - Ox.y (Move object x into room y; if x or y are missing, current object or room is assumed)
//   - Ox.0 (Hide object x)                  / O.0 (hide current object)
//   - Ox.1 (move object x into inventory)   / O.1 (move current object into inventory)
//   - Ox. (move object x into current room) / O.  (move current object into current room)
// - Px.y (Move player x to room y; if x or y are missing, current player or room is assumed)
// - Dx.y.z (Create passage with direction x from room z to room y and back) / Dx.y (from current room)
// - Sx.y.z (Set image x, depending on set type z: z=0: for object y, z=1: for room y, z=2: show temporary)
// - A    (Continue with action of inherited object)
// - Q    (Quit: win or loose)
//


// Objects (first number is initial room number of object, 0=hidden)
// The object name is marked with ^
// Components: isa, room, flag, sex, name, actions
// - isa: is-a: object hierarcy
// - room: room where object can be found
// - flag: object flag/status
// - sex: object sex: 0=masculin, 1=femin, 2=neutrum
// - name: (get obj name after ^)
// - actions: for different verbs


// Players


// --------------------------------------------------------

// format a string to clen characters with leading spaces
// in:  str = string, clen = column width
// out: str = formatted string
function mynumformat1(num, clen) {
  var str = String(num);
  for (var i = str.length; i < clen; i++) {
    str = " " + str;
  }
  return(str);
}


function myparseint(str) {
  var pos = 0;
  while (str.charAt(pos) == ' ') { // overread previous spaces (StarOffice 5.x does not like it)
    pos++;
  }
  return parseInt(str.substring(pos));
}


// --------------------------------------------------------
// --------------------------------------------------------

// Constructor for class AdvMaps
function AdvMaps(maps_str) {
  this.level = 0;
  this.x = 0;
  this.y = 0;
  this.flag = 0;

  if (maps_str) {
    var max1 = 3;
    var ms = maps_str.split(",");
    if (ms.length != (max1 + 1)) {
      window.alert("WARNING: Maps: "+ maps_str);
    }
    this.level = myparseint(ms[0]);
    this.x = myparseint(ms[1]);
    this.y = myparseint(ms[2]);
    this.flag = myparseint(ms[3]);
  }

  if (!AdvMaps.prototype) { // old browsers < JS 1.2 get the methods directly into the object...
    this.getlevel = AdvMaps_getlevel;
    this.getx = AdvMaps_getx;
    this.gety = AdvMaps_gety;
    this.getflag = AdvMaps_getflag;
    this.setflag = AdvMaps_setflag;
    this.dump = AdvMaps_dump;
  }
}

// For new browsers (JS 1.2) set the prototypes...
new AdvMaps(); // force old browsers to create prototype
if (AdvMaps.prototype) {
  AdvMaps.prototype.getlevel = AdvMaps_getlevel;
  AdvMaps.prototype.getx = AdvMaps_getx;
  AdvMaps.prototype.gety = AdvMaps_gety;
  AdvMaps.prototype.getflag = AdvMaps_getflag;
  AdvMaps.prototype.setflag = AdvMaps_setflag;
  AdvMaps.prototype.dump = AdvMaps_dump;
}


function AdvMaps_getlevel() {
  return this.level;
}

function AdvMaps_getx() {
  return this.x;
}

function AdvMaps_gety() {
  return this.y;
}

function AdvMaps_getflag() {
  return this.flag;
}

function AdvMaps_setflag(flag) {
  this.flag = flag;
}

function AdvMaps_dump() {
  var txt = "";
  txt += mynumformat1(this.level, 2) +","+ mynumformat1(this.x, 3) +","+ mynumformat1(this.y, 3) +","+ mynumformat1(this.flag, 1);
  return txt;
}

// --------------------------------------------------------
// --------------------------------------------------------

// Constructor for class AdvMsg
function AdvMsg(messages) {
  this.messages = messages;
  if (!AdvMsg.prototype) { // old browsers < JS 1.2 get the methods directly into the object...
    this.get = AdvMsg_get;
    this.print = AdvMsg_print;
    this.len = AdvMsg_len;
    this.dump = AdvMsg_dump;
  }
}

// For new browsers (JS 1.2) set the prototypes...
new AdvMsg(); // force old browsers to create prototype
if (AdvMsg.prototype) {
  AdvMsg.prototype.get = AdvMsg_get;
  AdvMsg.prototype.print = AdvMsg_print;
  AdvMsg.prototype.len = AdvMsg_len;
  AdvMsg.prototype.dump = AdvMsg_dump;
}


// normally msg_arg is the sex of the object (0,1,2)
function AdvMsg_get(idx, msg_arg) {
  var str = this.messages[idx];
  //if (g_debug) { g_debug.writeln("AdvMsg_get: idx='"+ idx +"', str='"+ str +"'", 1); }
  if (str == null) {
    window.alert("WARNING: Unknown message index: "+ idx);
    return "";
  }
  var pos1;
  if (msg_arg == null) {
    msg_arg = 2; // TEST
  }
  while ((pos1 = str.indexOf('{')) >= 0) {
    var pos2 = str.indexOf('}');
    if (pos2 < 0) {
      window.alert("WARNING: Wrong message syntax: "+ str);
      last;
    }
    var part1 = str.substring(pos1 + 1, pos2); // get part inside brackets {x}
    var parts1 = part1.split(',');
    str = str.substring(0, pos1) + parts1[msg_arg] + str.substring(pos2 + 1, str.length);
    if (g_debug) { g_debug.writeln("AdvMsg_get: part1='"+ part1 +"', str='"+ str +"'", 0); }
  }
  return str;
}


// unused
function AdvMsg_print(str) {
}

function AdvMsg_len() {
  return this.messages.length;
}

function AdvMsg_dump(idx) {
  var txt = this.messages[idx]; // don't convert!
  return txt;
}

// --------------------------------------------------------
// --------------------------------------------------------


// Constructor for class AdvPlayer
function AdvPlayer(player_str) {
  this.startroom = 0; // start room
  this.room = 0; // current room
  this.step = 0; // current step
  this.name = ""; // player name (unused)
  this.steplist = ""; // recorded steps

  if (player_str) {
    var rs = player_str.split(",");
    if (rs.length != 4) {
      window.alert("WARNING: Player: "+ player_str);
    }
    this.startroom = myparseint(rs[0]);
    var dummy1 = myparseint(rs[1]);
    this.name = rs[2];
    this.steplist = rs[3];
    //window.alert("steplist='"+ this.steplist +"'");
  }

  this.room = this.startroom;

  if (!AdvPlayer.prototype) { // old browsers < JS 1.2 get the methods directly into the object...
    this.setroom = AdvPlayer_setroom;
    this.getroom = AdvPlayer_getroom;
    this.setstep = AdvPlayer_setstep;
    this.getstep = AdvPlayer_getstep;
    this.setsteplist = AdvPlayer_setsteplist;
    this.addsteplist = AdvPlayer_addsteplist;
    this.getsteplist = AdvPlayer_getsteplist;
    this.setname = AdvPlayer_setname;
    this.getname = AdvPlayer_getname;
    this.dump = AdvPlayer_dump;
  }
  if (player_str) { // don't call debug for prototype creation!
    if (g_debug) { g_debug.writeln("Player created: "+ '"' + this.dump() +'",', 2); }
  }
}

// normally we would create unnamed functions here but for old browsers...

// For new browsers (JS 1.2) set the prototypes...
new AdvPlayer(); // force old browsers to create prototype
if (AdvPlayer.prototype) {
  AdvPlayer.prototype.setroom = AdvPlayer_setroom;
  AdvPlayer.prototype.getroom = AdvPlayer_getroom;
  AdvPlayer.prototype.setstep = AdvPlayer_setstep;
  AdvPlayer.prototype.getstep = AdvPlayer_getstep;
  AdvPlayer.prototype.setsteplist = AdvPlayer_setsteplist;
  AdvPlayer.prototype.addsteplist = AdvPlayer_addsteplist;
  AdvPlayer.prototype.getsteplist = AdvPlayer_getsteplist;
  AdvPlayer.prototype.setname = AdvPlayer_setname;
  AdvPlayer.prototype.getname = AdvPlayer_getname;
  AdvPlayer.prototype.dump = AdvPlayer_dump;
}

function AdvPlayer_setroom(room) {
  this.room = room;
}

function AdvPlayer_getroom() {
  return this.room;
}

function AdvPlayer_setstep(step) {
  this.step = step;
}

function AdvPlayer_getstep() {
  return this.step;
}

function AdvPlayer_setsteplist(steplist) {
  return this.steplist = steplist;
}

function AdvPlayer_addsteplist(steplist) {
  if (this.steplist != "") {
    this.steplist += "/"+ steplist;
  } else {
    this.steplist = steplist;
  }
}

function AdvPlayer_getsteplist() {
  return this.steplist;
}

function AdvPlayer_setname(name) {
  this.name = name;
}

function AdvPlayer_getname() {
  return this.name;
}

function AdvPlayer_dump() {
  var txt = mynumformat1(this.startroom, 3) + "," + mynumformat1(this.room, 3) + "," + this.name + "," + this.steplist;
  return txt;
}

// --------------------------------------------------------
// --------------------------------------------------------

// Constructor for class AdvRoom
function AdvRoom(room_str) {
  this.name = "";
  var max_dir = 6;
  this.direction = new Array(max_dir);
  for (var i = 0; i < this.direction.length; i++) {
    this.direction[i] = 0;
  }
  this.image = 0; // image number

  if (room_str) {
    var rs = room_str.split(",");
    if (rs.length != (max_dir + 2)) {
      window.alert("WARNING: Room: "+ room_str);
    }
    this.name = rs[rs.length - 1]; // last element is name
    for (var i = 0; i < this.direction.length; i++) {
      this.direction[i] = myparseint(rs[i]);
    }
    this.image = myparseint(rs[max_dir + 1]);
  }

  if (!AdvRoom.prototype) { // old browsers < JS 1.2 get the methods directly into the object...
    this.setname = AdvRoom_setname;
    this.getname = AdvRoom_getname;
    this.setimage = AdvRoom_setimage;
    this.getimage = AdvRoom_getimage;
    this.setdirection = AdvRoom_setdirection;
    this.getdirection = AdvRoom_getdirection;
    this.getdirections = AdvRoom_getdirections;
    this.dump = AdvRoom_dump;
  }
  if (room_str) { // don't call debug for prototype creation!
    if (g_debug) { g_debug.writeln("Room created: "+ '"' + this.dump() +'",', 2); }
  }
}

// normally we would create unnamed functions here but for old browsers...

// For new browsers (JS 1.2) set the prototypes...
new AdvRoom(); // force old browsers to create prototype
if (AdvRoom.prototype) {
  AdvRoom.prototype.setname = AdvRoom_setname;
  AdvRoom.prototype.getname = AdvRoom_getname;
  AdvRoom.prototype.setimage = AdvRoom_setimage;
  AdvRoom.prototype.getimage = AdvRoom_getimage;
  AdvRoom.prototype.setdirection = AdvRoom_setdirection;
  AdvRoom.prototype.getdirection = AdvRoom_getdirection;
  AdvRoom.prototype.getdirections = AdvRoom_getdirections;
  AdvRoom.prototype.dump = AdvRoom_dump;
}


function AdvRoom_setname(name) {
  this.name = name;
}

function AdvRoom_getname() {
  return this.name;
}

function AdvRoom_setimage(image) {
  this.image = image;
}

function AdvRoom_getimage() {
  return this.image;
}

function AdvRoom_setdirection(dir_num, room_num) {
  this.direction[dir_num] = room_num;
  if (g_debug) { g_debug.writeln("AdvRoom_setdirection: name="+ this.getname() +", dir_num="+ dir_num +", room_num="+ room_num, 0); }
}

function AdvRoom_getdirection(dir_num) {
  //window.alert("AdvRoom_getdirection: dir="+ dir_num);
  return this.direction[dir_num];
}

function AdvRoom_getdirections() {
  return this.direction;
}

function AdvRoom_dump() {
  var txt = "";
  if (!this.name) {
    return txt; // empty name
  }
  for (var i = 0; i < this.direction.length; i++) {
    txt += mynumformat1(this.direction[i], 3) + ",";
  }
  txt += mynumformat1(this.image, 3) + ",";
  txt += this.name;
  return txt;
}

// --------------------------------------------------------
// --------------------------------------------------------

// Constructor for class AdvObject
function AdvObject(object_str, max_action) {
  this.isa = 0; // is a (other object or 0)
  this.room = 0; // room number
  this.flag = 0; // flag
  this.sex = 0;
  this.image = 0;
  this.name_comp = ""; // name component
  this.name = ""; // name

  this.action = new Array(max_action);
  //for (var i = 0; i < this.action.length; i++) {
    //?? this.action[i] = null;
  //}

  if (object_str) {
    var action_idx = 6;
    var os = object_str.split(",");
    if (os.length != (action_idx + max_action)) { //5 = #components in string before action
      window.alert("WARNING: Object: "+ object_str);
    }
    this.isa = myparseint(os[0]); // is-a: object hierarcy
    this.room = myparseint(os[1]); // room where object can be found
    this.flag = myparseint(os[2]); // object flag/status
    this.sex = myparseint(os[3]); // object sex: 0=masculin, 1=feminin, 2=neutrum
    this.image = myparseint(os[4]);
    var name_c = os[5].split('^'); // get obj name after ^
    this.name_comp = name_c[0];
    this.name = name_c[1];
    for (var i = action_idx; i < os.length; i++) {
      this.action[i - action_idx] = os[i];
    }
  }

  if (!AdvObject.prototype) { // old browsers < JS 1.2 get the methods directly into the AdvObject...
    this.setname = AdvObject_setname;
    this.getname = AdvObject_getname;
    this.getfullname = AdvObject_getfullname;
    this.setimage = AdvObject_setimage;
    this.getimage = AdvObject_getimage;
    this.setflag = AdvObject_setflag;
    this.getflag = AdvObject_getflag;
    this.setsex = AdvObject_setsex;
    this.getsex = AdvObject_getsex;
    this.setroom = AdvObject_setroom;
    this.getroom = AdvObject_getroom;
    this.getisa = AdvObject_getisa;
    this.getaction = AdvObject_getaction;
    this.dump = AdvObject_dump;
  }
  if (object_str) {
    if (g_debug) { g_debug.writeln("Object created: "+ '"' + this.dump() +'",', 2); }
  }
}

// normally we would create unnamed functions here but for old browsers...

// For new browsers (JS 1.2) set the prototypes...
new AdvObject(); // force old browsers to create prototype
if (AdvObject.prototype) {
  AdvObject.prototype.setname = AdvObject_setname;
  AdvObject.prototype.getname = AdvObject_getname;
  AdvObject.prototype.getfullname = AdvObject_getfullname;
  AdvObject.prototype.setimage = AdvObject_setimage;
  AdvObject.prototype.getimage = AdvObject_getimage;
  AdvObject.prototype.setflag = AdvObject_setflag;
  AdvObject.prototype.getflag = AdvObject_getflag;
  AdvObject.prototype.setsex = AdvObject_setsex;
  AdvObject.prototype.getsex = AdvObject_getsex;
  AdvObject.prototype.setroom = AdvObject_setroom;
  AdvObject.prototype.getroom = AdvObject_getroom;
  AdvObject.prototype.getisa = AdvObject_getisa;
  AdvObject.prototype.getaction = AdvObject_getaction;
  AdvObject.prototype.dump = AdvObject_dump;
}


function AdvObject_setname(name) {
  this.name = name;
}

function AdvObject_getname() {
  return this.name;
}

function AdvObject_getfullname() {
  return this.name_comp + this.name;
}

function AdvObject_setimage(image) {
  this.image = image;
}

function AdvObject_getimage() {
  return this.image;
}

function AdvObject_setflag(value) {
  this.flag = value;
  if (g_debug) { g_debug.writeln("AdvObject_setflag="+ this.flag, 0); }
}

function AdvObject_getflag() {
  if (g_debug) { g_debug.writeln("AdvObject_getflag="+ this.flag, 0); }
  return this.flag;
}

function AdvObject_setsex(value) {
  this.sex = value;
  if (g_debug) { g_debug.writeln("AdvObject_setsex="+ this.sex, 0); }
}

function AdvObject_getsex() {
  if (g_debug) { g_debug.writeln("AdvObject_getsex="+ this.sex, 0); }
  return this.sex;
}

function AdvObject_setroom(room) {
  this.room = room;
  if (g_debug) { g_debug.writeln("AdvObject_setroom="+ this.room, 0); }
}

function AdvObject_getroom() {
  return this.room;
}

function AdvObject_getisa() {
  return this.isa;
}

function AdvObject_getaction(action) {
  return this.action[action];
}

function AdvObject_dump() {
  var txt = "";
  if (!this.name) {
    return txt;
  }
  txt = mynumformat1(this.isa, 3) +","+ mynumformat1(this.room, 3) +","+ mynumformat1(this.flag, 2) +","+ mynumformat1(this.sex, 1) +","+ mynumformat1(this.image, 3) +","+ this.name_comp +"^"+ this.name;
  //var amap = [ 0, 3, 5, 1, 6, 2, 4 ];
  for (var i = 0; i < this.action.length; i++) {
    txt += ",";
    if (this.action[i]) { // action defined?
      txt += this.action[i];
    }
  }
  return txt;
}

// --------------------------------------------------------
// --------------------------------------------------------



// Constructor for class Adv(enture) (images are optional)
function Adv(verbs, messages, rooms, objects, images, players, maps) {
  // attributes
  this.verbs = verbs;
  this.messages = new AdvMsg(messages);

  if (rooms) {
    this.rooms = new Array(rooms.length);
    for (var i = 0; i < rooms.length; i++) {
      this.rooms[i] = new AdvRoom(rooms[i]);
      this.rooms[i].setimage(i); // TEST!
    }
    if (g_debug) {
      for (var i = 0; i < this.rooms.length; i++) {
        if (g_debug) { g_debug.writeln("room: "+ '"' + this.rooms[i].dump() + '",', 2); }
      }
    }
  }

  if (objects) {
    this.objects = new Array(objects.length); // objects in room
    for (var i = 0; i < objects.length; i++) {
      this.objects[i] = new AdvObject(objects[i], this.verbs.length);
    }
    if (g_debug) {
      for (var i = 0; i < this.objects.length; i++) {
        if (g_debug) { g_debug.writeln("object: "+ '"' + this.objects[i].dump() + '",', 2); }
      }
    }
  }

  if (images) {
    this.sshow = new Sshow(images, null, null, 0); // no precaching
  }

  if (players) {
    this.players = new Array(players.length);
    for (var i = 0; i < players.length; i++) {
      this.players[i] = new AdvPlayer(players[i]);
    }
    this.player_idx = 0;
    this.player = this.players[this.player_idx]; // current player (normally 1) [this.set_player(0);]
  }

  if (maps) {
    this.maps = new Array(maps.length);
    for (var i = 0; i < maps.length; i++) {
      this.maps[i] = new AdvMaps(maps[i]);
    }
  }

  // methods
  if (!Adv.prototype) { // old browsers get the methods directly into the object...
    this.dump_data = Adv_dump_data;
    this.init_frm = Adv_init_frm;
    this.get_objects = Adv_get_objects;
    this.get_direction_txt = Adv_get_direction_txt;
    this.show_room = Adv_show_room;
    this.go_direction = Adv_go_direction;
    this.do_action2 = Adv_do_action2;
    this.do_action = Adv_do_action;
    this.play_event = Adv_play_event;
    this.play_events = Adv_play_events;
    this.play_events_auto = Adv_play_events_auto;
    this.getsteplist = Adv_getsteplist;
    this.set_player = Adv_set_player;
    this.refresh_view = Adv_refresh_view;
    this.show_maps = Adv_show_maps;
  }
}


//
// normally we would create unnamed functions here but for old browsers...
// Adv.prototype.get_rnd_index1 = function(list) { ... }
//

// For new browsers (JS 1.2) set the prototypes...
new Adv(); // force old browsers to create prototype
if (Adv.prototype) {
  Adv.prototype.dump_data = Adv_dump_data;
  Adv.prototype.init_frm = Adv_init_frm;
  Adv.prototype.get_objects = Adv_get_objects;
  Adv.prototype.get_direction_txt = Adv_get_direction_txt;
  Adv.prototype.show_room = Adv_show_room;
  Adv.prototype.go_direction = Adv_go_direction;
  Adv.prototype.do_action2 = Adv_do_action2;
  Adv.prototype.do_action = Adv_do_action;
  Adv.prototype.play_event = Adv_play_event;
  Adv.prototype.play_events = Adv_play_events;
  Adv.prototype.play_events_auto = Adv_play_events_auto;
  Adv.prototype.getsteplist = Adv_getsteplist;
  Adv.prototype.set_player = Adv_set_player;
  Adv.prototype.refresh_view = Adv_refresh_view;
  Adv.prototype.show_maps = Adv_show_maps;
}





// for debugging...
function Adv_dump_data() {
  var txt = "";
  var nl = "\n";
  var comma = "";

  var win = window.open("", "Data", "width=800,height=600,resizable=yes,scrollbars=yes,dependent=yes");
  win.document.open("text/html", "reuse");
  txt += '// Adventure Ressources' + nl;

  txt += nl + "// Verbs" + nl + 'var g_verbs1 = new Array(' + nl;
  for (var i = 0; i < this.verbs.length; i++) {
    comma = ((i + 1) < this.verbs.length) ? "," : "";
    txt += '  "' + this.verbs[i] + '"' + comma +" //V" + i + nl;
  }
  txt += ');' + nl + nl;

  txt += "// Messages" + nl + 'var g_messages1 = new Array(' + nl;
  var mlen1 = this.messages.len();
  for (var i = 0; i < mlen1; i++) {
    comma = ((i + 1) < mlen1) ? "," : "";
    txt += '  "' + this.messages.dump(i) + '"' + comma +" //M" + i + nl;
  }
  txt += ');' + nl + nl;

  txt += "// Rooms" + nl + 'var g_rooms1 = new Array(' + nl;
  for (var i = 0; i < this.rooms.length; i++) {
    comma = ((i + 1) < this.rooms.length) ? "," : "";
    txt += '  "' + this.rooms[i].dump() + '"' + comma +" //R" + i + nl;
  }
  txt += ');' + nl + nl;

  txt += "// Objects, Actions" + nl + 'var g_objects1 = new Array(' + nl;
  for (var i = 0; i < this.objects.length; i++) {
    comma = ((i + 1) < this.objects.length) ? "," : "";
    txt += '  "' + this.objects[i].dump() + '"' + comma +" //O" + i + nl;
  }
  txt += ');' + nl + nl;

  txt += "// Images" + nl + 'var g_images1 = new Array(' + nl;
  for (var i = 0; i < this.sshow.inames.length; i++) { // fast hack to access image names...
    comma = ((i + 1) < this.sshow.inames.length) ? "," : "";
    txt += '  "' + this.sshow.inames[i] + '"' + comma +" //S" + i + nl;
  }
  txt += ');' + nl + nl;

  txt += "// Labyrinth Map" + nl + 'var g_lmaps1 = new Array(' + nl;
  for (var i = 0; i < this.maps.length; i++) { // fast hack to access image names...
    comma = ((i + 1) < this.maps.length) ? "," : "";
    txt += '  "' + this.maps[i].dump() + '"' + comma +" //L" + i + nl;
  }
  txt += ');' + nl + nl;

  txt += "// Players" + nl + 'var g_players1 = new Array(' + nl;
  for (var i = 0; i < this.players.length; i++) {
    comma = ((i + 1) < this.players.length) ? "," : "";
    txt += '  "' + this.players[i].dump() + '"' + comma +" //P" + i + nl;
  }
  txt += ');' + nl + nl;

  txt += "// Current Player" + nl + 'var g_curr_player1 = ';
  txt += this.player_idx + ';' + nl;

  win.document.writeln("<pre>"+ txt +"</pre>");
  win.document.close(); // important to flush window contents
}


function Adv_set_option1_priv(frm_sel, idx, name, value) {
  if (frm_sel.options[idx]) { // reuse option if it does exist
    frm_sel.options[idx].text = name;
    frm_sel.options[idx].value = value;
  } else {
    frm_sel.options[idx] = new Option(name, value);
  }
}


function Adv_set_options_priv(frm_sel, first_idx, objects, list, last_selected_o) {
  for (var i = 0; i < list.length; i++) {
    Adv_set_option1_priv(frm_sel, first_idx + i, objects[list[i]].getname(), list[i]);
    if (list[i] == last_selected_o) {
      frm_sel.options[first_idx + i].selected = true;
    }
  }
}


function Adv_init_frm(frm, img_idx, delay_sec) {
  if (frm) {
    this.adv_frm = frm;
  }

  var room = this.player.getroom();
  if (this.maps) {
    this.maps[room].setflag(1);
  }

  if (img_idx) {
    if (this.sshow) {
      this.sshow.change_image_abs(this.rooms[room].getimage(), img_idx);
    }
  }

  if (this.adv_frm && this.adv_frm.verb) {
    for (var i = 0; i < this.verbs.length; i++) {
      Adv_set_option1_priv(this.adv_frm.verb, i, this.verbs[i], i);
    }
    this.adv_frm.verb.options[0].selected = true;
  }

  if (this.adv_frm && this.adv_frm.player) {
    for (var i = 0; i < this.players.length; i++) {
      Adv_set_option1_priv(this.adv_frm.player, i, this.players[i].getname(), i);
    }
    this.adv_frm.player.options[0].selected = true;
  }

  // does not work:
  //if (this.adv_frm && this.adv_frm.object) {
  //  var frm_sel = this.adv_frm.object;
  //  this.use_setlength = eval("frm_sel.options.length = frm_sel.options.length");
  //}

  this.show_room(this.player, "");
  var steplist = this.player.getsteplist();
  if (steplist && steplist != "") { // we need the second condition for NGS js, if steplist is ""
    this.player.setsteplist(null); // initialize
    if (!delay_sec) {
      this.play_events(steplist); // start playback
    } else {
      this.play_events_auto(steplist);
    }
  }
}


// get objects in room
function Adv_get_objects(room) {
  var olist = new Array();
  for (var i = 0; i < this.objects.length; i++) {
    if (this.objects[i].getroom() == room) {
      olist[olist.length] = i; // push object to list
    }
  }
  if (g_debug) { g_debug.writeln("get_objects: room="+ room +", olist="+ olist, 1); }
  return olist;
}


// get directions in room
function Adv_get_direction_txt(room) {
  var txt = "";
  if (g_debug) { g_debug.writeln("get_direction_txt: room="+ room, 1); }
  var ds = this.rooms[room].getdirections();
  for (var i = 0; i < ds.length; i++) {
    if (ds[i] > 0) {
      if (txt.length > 0) {
        txt += ", ";
      }
      txt += this.messages.get(6 + i); //M6-M11
    }
  }
  return txt;
}


//
// Adv_show_room - change text (of e.g. input field)
// IN:  [<txtval> = text field]
// OUT: ...
//
function Adv_show_room(player, msg_txt) {
  var room = player.getroom();
  var txt = "";
  {
    var stxt = this.messages.get(16).split(","); //M16
    txt += stxt[0] +": "+ player.getname() +", "+ stxt[1] +": "+ room +", "+ stxt[2] +": "+ player.getstep() + "\n" + "\n";
  }
  txt += msg_txt + "\n" + "\n" + this.messages.get(1) +" " + this.rooms[room].getname() + ".\n" +
    this.messages.get(2) +" "; //  I can see
  var olist = this.get_objects(room);
  if (olist.length <= 0) {
   txt += this.messages.get(3); // nothing special
  } else {
    for (var i = 0; i < olist.length; i++) {
      txt += this.objects[olist[i]].getfullname();
      if ((i + 1) < olist.length) {
        txt += ", ";
      }
    }
  }

  txt += ".\n" +
    this.messages.get(4) +" "+ this.get_direction_txt(room) +".";

  if (this.adv_frm && this.adv_frm.status) {
    this.adv_frm.status.value = txt;
    //e.g. document.Situation.description.value = txt;
  } else {
    window.document.writeln(txt); // output for NGS
  }

  if (this.adv_frm && this.adv_frm.object) {
    var frm_sel = this.adv_frm.object;
    var ilist = this.get_objects(1); // get inventar
    var last_selected_o = (frm_sel.selectedIndex >= 0) ? frm_sel[frm_sel.selectedIndex].value : -1;
    frm_sel.selectedIndex = -1;
    Adv_set_options_priv(frm_sel, 0, this.objects, olist, last_selected_o);
    Adv_set_option1_priv(frm_sel, olist.length, this.objects[0].getname(), -1); // Inventar
    Adv_set_options_priv(frm_sel, olist.length + 1, this.objects, ilist, last_selected_o);
    //frm_sel.options.length = olist.length + 1 + ilist.length; // cannot set length on old browsers (StarOffice),
    //so we keep all the objects and overwite those which are not needed...
    for (var i = olist.length + 1 + ilist.length; i < frm_sel.options.length; i++) {
      Adv_set_option1_priv(frm_sel, i, "", -1); // clear options which are not needed
    }
    if (frm_sel.selectedIndex < 0) {
      frm_sel.selectedIndex = 0;
    }
  }

  //return txt;
}


function Adv_go_direction(dir) {
  if (g_debug) { g_debug.writeln("go_direction: dir="+ dir, 0); }
  //window.alert("DEBUG: go_direction: dir="+ dir +", room="+ this.player.getroom() +", rooms_len="+ this.rooms.length);
  this.player.addsteplist(dir);
  var newroom = this.rooms[this.player.getroom()].getdirection(dir);
  var txt = "";
  if (newroom > 0) {
    this.player.setroom(newroom);
    this.maps[newroom].setflag(1);
    this.player.setstep(this.player.getstep() + 1);
    //txt += this.messages.get(13); //M13: ok
    txt += this.messages.get(19) +' '+ this.messages.get(6 + dir) +'.'; //M19, M6-M11
    if (this.sshow) {
      this.sshow.change_image_abs(this.rooms[newroom].getimage());
    }
    if (this.maps && this.maps_window && !this.maps_window.closed) {
      this.show_maps();
    }
  } else {
    txt += this.messages.get(6 + dir) +': '+ this.messages.get(12); //M6-M11, M12: no way
  }
  this.show_room(this.player, txt);
}


function parse_number(str, start_pos) {
  var pos = start_pos;
  while ((str.charAt(pos) >= '0') && (str.charAt(pos) <= '9')) {
    pos++;
  }
  return str.substring(start_pos, pos);
}


function Adv_do_action2(v1, curr_obj, actstr) {
  var img_idx1 = 0;
  var pos = 0;
  var txt = "";
  var if_started_f = 0; // if started flag
  var cond_f = 0; // condition flag
  var num = new Array(3); // up to 3 parameters
  var curr_room = this.player.getroom(); // current room of current player
  if (g_debug) { g_debug.writeln("do_action2: action string="+ actstr +", verb="+ v1 +", obj=" + curr_obj, 0); }
  while (pos < actstr.length) {
    cond_f = 0;
    var neg_f = 0; // negation flag (for next code)
    var act_code = actstr.charAt(pos); // action code
    if (act_code == "!") { // neg?
      neg_f = 1;
      pos++;
      act_code = actstr.charAt(pos); // action code
    }
    var num_cnt = 0; // number count
    num[0] = -1;
    num[1] = -1;
    num[2] = -1;
    // get optional numbers separated by dots (if present) (avoid do..while because of old Netscape 3)
    var tmp_char1 = ".";
    while (tmp_char1 == ".") {
      pos++;
      var nstr = parse_number(actstr, pos); // number following?
      if (nstr.length > 0) {
        num[num_cnt] = parseInt(nstr); pos += nstr.length;
      }
      num_cnt++;
      tmp_char1 = actstr.charAt(pos); // dot "."?
    }

    if (g_debug) { g_debug.writeln("do_action2: action code="+ act_code +", n0="+ num[0] +", n1="+ num[1] +", n2="+ num[2] +", if="+ if_started_f + ", cond="+ cond_f +", neg="+ neg_f, 0); }


    // (avoid switch..case because of old Netscape 3)
    if (act_code == 'I') { // if
      if_started_f = 1; // if started
      cond_f = 1; // set condition to true

    } else if (act_code == 'T') { // then
      if_started_f = 0;

    } else if (act_code == 'E') { // else
      while ((pos < actstr.length) && (actstr.charAt(pos) != ';')) { // run into else (we processed then) -> find end of if...
        pos++;
      }

    } else if (act_code == ';') { // finish if/then/else
      if_started_f = 0;
      //cond_f = 0;
      //neg_f = 0;

    } else if (act_code == 'F') { // flag F<flag>.<obj>
      num[0] = (num[0] >= 0) ? num[0] : 1;
      num[1] = (num[1] >= 0) ? num[1] : curr_obj;
      if (if_started_f > 0) { // condition?
        if ((this.objects[num[1]].getflag() & num[0]) > 0) {
          cond_f = 1;
        }
        if (g_debug) { g_debug.writeln("do_action2: action test 'F': flag="+ num[0] +", obj="+ num[1] +": "+ cond_f, 0); }
      } else { // set flag F<flag>.<obj>
        this.objects[num[1]].setflag(this.objects[num[1]].getflag() | num[0]);
        if (g_debug) { g_debug.writeln("do_action2: action set 'F': flag "+ num[0] +", set for obj "+ num[1], 0); }
      }

    } else if (act_code == 'C') { // clear flag F<flag>.<obj>
      num[0] = (num[0] >= 0) ? num[0] : 1;
      num[1] = (num[1] >= 0) ? num[1] : curr_obj;
      this.objects[num[1]].setflag(this.objects[num[1]].getflag() & ~num[0]);
      if (g_debug) { g_debug.writeln("do_action2: action clear 'F': flag "+ num[0] +", clear for obj "+ num[1], 0); }

    } else if (act_code == 'O') { // object O<obj>.<room>
      num[0] = (num[0] >= 0) ? num[0] : curr_obj;
      num[1] = (num[1] >= 0) ? num[1] : curr_room;
      if (if_started_f > 0) { // condition?
        if (this.objects[num[0]].getroom() == num[1]) {
          cond_f = 1;
        }
      } else { // move object O<obj>.<room>
        this.objects[num[0]].setroom(num[1]);
      }

    } else if (act_code == 'P') { // player P<player>.<room>
      num[0] = (num[0] >= 0) ? num[0] : 0; // player??
      num[1] = (num[1] >= 0) ? num[1] : curr_room;
      if (if_started_f > 0) { // condition?
        if (this.players[num[0]].getroom() == num[1]) {  //if (curr_room == num[1]) {
          cond_f = 1;
        }
      } else { // move player P<player>.<room>
        this.players[num[0]].setroom(num[1]);
        curr_room = num[1];
        if (this.sshow) {
          this.sshow.change_image_abs(this.rooms[num[1]].getimage());
        }
        if (this.maps) {
          this.maps[num[1]].setflag(1); // room visited
        }
        if (this.maps && this.maps_window && !this.maps_window.closed) {
          this.show_maps();
        }
      }

    } else if (act_code == 'M') { // print message M<message>.<sex>
      num[1] = (num[1] >= 0) ? num[1] : this.objects[curr_obj].getsex();
      if (txt.length > 0) {
        txt += ' '; // append space
      }
      txt += this.messages.get(num[0], num[1]);

    } else if (act_code == 'D') { // create passage (direction) D<dir>.<room>.<startroom>
      num[0] = (num[0] >= 0) ? num[0] : 0; //direction
      num[1] = (num[1] >= 0) ? num[1] : curr_room; //destination room
      num[2] = (num[2] >= 0) ? num[2] : curr_room; //starting room
      var inv_room = this.rooms[num[2]].getdirection(num[0]) || num[1]; // get neighbour room
      this.rooms[num[2]].setdirection(num[0], num[1]);
      num[0] = (num[0] == 0) ? 1 : (num[0] == 1) ? 0 : (num[0] == 2) ? 3 : (num[0] == 3) ? 2 : (num[0] == 4) ? 5 : (num[0] == 5) ? 4 : -1; // inverse direction
      this.rooms[inv_room].setdirection(num[0], (num[1]) ? num[2] : 0); // create/remove inverse direction

    } else if (act_code == 'Q') { // quit
        txt += "** QUIT **";

    } else if (act_code == 'A') { // continue with action of inherited object
      if (this.objects[curr_obj].getisa() > 0) {
        actstr = this.objects[this.objects[curr_obj].getisa()].getaction(v1);
        pos = 0;
        if_started_f = 0;
      } else {
        pos = actstr.length;
      }

    } else if (act_code == 'S') { // set image S<img>.<set type>.<idx>
      num[0] = (num[0] >= 0) ? num[0] : this.rooms[curr_room].getimage(); // show image
      num[2] = (num[2] >= 0) ? num[2] : 0; // set type 0=object, 1=room, 2=show temporary

      if (num[2] == 0) { // set object image
        num[1] = (num[1] >= 0) ? num[1] : curr_obj; // idx
        this.objects[num[1]].setimage(num[0]); // set image for object (don't show!)

      } else if (num[2] == 1) { // set room image
        num[1] = (num[1] >= 0) ? num[1] : curr_room; // idx
        this.rooms[num[1]].setimage(num[0]); // set image for room (don't show!)

      } else if (num[2] == 2) { // show image temporary
        img_idx1 = num[0]; // sshow: see below

      } else {
        window.alert("Warning: unknown image type='"+ num[1] + "'");
      }

    } else {
      window.alert("Warning: unknown action: '"+ actstr +"', code=" + act_code);
    }

    if ((act_code != '!') && (if_started_f > 0) && (cond_f == neg_f)) { // "if" started, condition false => find "else"...
      while ((pos < actstr.length) && (actstr.charAt(pos) != 'E') && (actstr.charAt(pos) != ';')) {
        pos++;
      }
      if ((actstr.charAt(pos) == 'E') || (actstr.charAt(pos) == ';')) {
        pos++; // overread
      }
      if (pos >= actstr.length) {
        if (txt == "") { // no message yet?
          txt += this.messages.get(15); //M15: Currently not possible
        }
      }
      if_started_f = 0;
    }
  }
  if (img_idx1 == 0) {
    img_idx1 = this.objects[curr_obj].getimage(); // get object image
  }
  if (img_idx1) {
    if (this.sshow) {
      this.sshow.change_image_abs(img_idx1); // show image
    }
  }
  if (g_debug) {
    g_debug.writeln("do_action2: action result: ("+ actstr +"): "+ txt, 0);
    return "("+ actstr +"): "+ txt;
  } else {
    return txt;
  }
}


// do action depending on verb v1 and object o1...
function Adv_do_action(v1, o1) {
  if (g_debug) { g_debug.writeln("do_action: verb="+ v1 +", obj=" + o1, 0); }
  this.player.addsteplist(v1 +"."+ o1);

  var msgtxt = "";
  if ((v1 < 0) || (o1 < 0)) {
    msgtxt = this.messages.get(18); //M18: Please select something!
  } else {

    var action = this.objects[o1].getaction(v1);
    if (action) {
      msgtxt = this.do_action2(v1, o1, action);
    } else {
      var tmp_o1 = o1;
      while (!action && (this.objects[tmp_o1].getisa() > 0)) {
        tmp_o1 = this.objects[tmp_o1].getisa(); // get parent object, if available
        action = this.objects[tmp_o1].getaction(v1);
      }
      if (action) {
        msgtxt = this.do_action2(v1, o1, action);
      } else {
        msgtxt = this.messages.get(14); //M14: how to do that?
        var img_idx1 = this.objects[o1].getimage(); // get object image
        if (img_idx1) {
          if (this.sshow) {
            this.sshow.change_image_abs(img_idx1); // show image
          }
        }
      }
    }
  }
  this.player.setstep(this.player.getstep() + 1);
  this.show_room(this.player, msgtxt);
}


function Adv_play_event(event) {
  if (g_debug) { g_debug.writeln("play_event: event="+ event); }
  var seppos = event.indexOf(".");
  if (seppos >= 0) { // action?
    this.do_action(parseInt(event.substring(0, seppos)), parseInt(event.substring(seppos + 1)));
  } else { // go
    this.go_direction(parseInt(event));
  }
}

function Adv_play_events(eventstr) {
  var events = eventstr.split("/");
  //window.alert("play_events: eventstr.length="+eventstr.length+", events.length="+events.length);
  for (var i = 0; i < events.length; i++) {
    this.play_event(events[i]);
  }
}

// unused:
function Adv_play_events_auto(eventstr, delay_sec) {
  if (g_debug) { g_debug.writeln("play_events_auto: eventstr="+ eventstr +", delay_sec="+ delay_sec); }

  if (eventstr) { // new event string?
    if (this.itimer) {
      this.itimer.close(); // stop & close
      this.itimer = 0;
    }
    this.eventstr = eventstr; //memorize

    if (delay_sec == null) {
      delay_sec = 1;
    }

    var obj = this;
    var method = 'play_events_auto';
    var arg0 = null; // arg0 not needed.

    if ((typeof Timer != "undefined") && (delay_sec)) {
      this.itimer = new Timer(obj, method, arg0);
      this.itimer.setInterval(delay_sec * 1000);
    }

  } else { // simulate...

    var pos = this.eventstr.indexOf("/");
    if (pos < 0) {
      pos = this.eventstr.length;
    }
    this.play_event(this.eventstr.substring(0, pos)); //substr?
    this.eventstr = this.eventstr.substring(pos + 1); //substr?
    if (this.eventstr.length <= 0) {
      if (this.itimer) {
        this.itimer.close(); // stop & close
        this.itimer = 0;
      }
    }
  }
}


function Adv_getsteplist() {
  return this.player.getsteplist();
}


function Adv_set_player(idx) {
  if (this.players) {
    this.player_idx = idx;
    this.player = this.players[this.player_idx];
  }
}


function Adv_refresh_view(txt) {
  if (!txt) { // txt = old message would be nice...
    txt = "";
  }
  if (this.sshow) {
    this.sshow.change_image_abs(this.rooms[this.player.getroom()].getimage());
  }
  if (this.maps && this.maps_window && !this.maps_window.closed) {
    this.show_maps();
  }
  this.show_room(this.player, txt);
}


function Adv_show_maps() {
  var win = window.open("", "Map", "width=355,height=260,resizable=yes,scrollbars=yes,status=yes,dependent=yes");
  this.maps_window = win; // memorize for automatic update
  if (win && !win.closed && this.maps) {
    win.document.open("text/html", "reuse");
    var nl = "\n";
    var txt = '';
    txt += "<html><head><title>Map</title></head>" + nl;
    txt += '<body text="#FFFFFF" bgcolor="#000000"><table cellspacing="0" cellpadding="0" border="1">' + nl;

    var room = this.player.getroom();
    var level = this.maps[room].getlevel();
    var max_x = 7;
    var max_y = 7;
    var maps2 = new Array(max_x * max_y);
    for (var i = 0; i < this.maps.length; i++) {
      if ((this.maps[i].getlevel() == level) && this.maps[i].getflag()) {
        maps2[this.maps[i].gety() * max_x + this.maps[i].getx()] = i;
        //if (g_debug) { g_debug.writeln("Adv_show_maps: room="+ i +", level="+ level +", x="+ this.maps[i].getx() +", y="+ this.maps[i].gety() +", flag="+ this.maps[i].getflag()); }
      }
    }
    for (var y = 0; y < max_y; y++) {
      txt += '<tr>' + nl;
      for (var x = 0; x < max_x; x++) {
        var td_style1 = "";
        var td_str1 = '&nbsp;';
        if (maps2[y * max_x + x]) {
          td_style1 = ' bgcolor="#00FFFF"';
          var room1 = maps2[y * max_x + x];
          if (room1 == room) {
            var img_name1 = this.sshow.inames[this.rooms[room1].getimage()];
            var room_str1 = this.rooms[room1].getname();
            td_str1 = '<img src="'+ img_name1 + '" border="1" width="40" height="30" alt="'+ room_str1 +'" />';
          }
        }
        txt += '<td width="44" height="34"'+ td_style1 +'>' + td_str1 + '</td>' + nl;
      }
      txt += '</tr>' + nl;
    }
    txt += "</table></body></html>";
    win.document.writeln(txt);
    win.document.close();
    if (win.focus) { win.focus(); }
  }
}

// end
