/*
 * s2g.c - Creates G64 image from a Zipcollection V.2 "Sixpack" archive
 *
 * Written by
 *  Markus Brenner (markus@brenner.de)
 * Based on d64tog64.c code by
 *  Andreas Boose  (boose@unixserv.rz.fh-hannover.de)
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 *  02111-1307  USA.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>

#define VERSION 0.12

#define BYTE unsigned char
#define DWORD unsigned int
#define MAX_TRACKS_1541 42
#define MAXSTRLEN 80
#define HEADLONG 8


static char sector_map_1541[43] =
{
    0,
    21, 21, 21, 21, 21, 21, 21, 21, 21, 21,     /*  1 - 10 */
    21, 21, 21, 21, 21, 21, 21, 19, 19, 19,     /* 11 - 20 */
    19, 19, 19, 19, 18, 18, 18, 18, 18, 18,     /* 21 - 30 */
    17, 17, 17, 17, 17,                         /* 31 - 35 */
    17, 17, 17, 17, 17, 17, 17          /* Tracks 36 - 42 are non-standard. */
};


static int speed_map_1541[42] = { 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3,
                                  3, 3, 3, 3, 2, 2, 2, 2, 2, 2, 2, 1, 1,
                                  1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                                  0, 0, 0 };


static BYTE GCR_conv_data[16] = { 0x0a, 0x0b, 0x12, 0x13,
                                  0x0e, 0x0f, 0x16, 0x17,
                                  0x09, 0x19, 0x1a, 0x1b,
                                  0x0d, 0x1d, 0x1e, 0x15 };

static int GCR_decode[32] = { -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,
                              -1, 0x8, 0x0, 0x1,  -1, 0xc, 0x4, 0x5,
                              -1,  -1, 0x2, 0x3,  -1, 0xf, 0x6, 0x7,
                              -1, 0x9, 0xa, 0xb,  -1, 0xd, 0xe,  -1 };
                             

static void convert_4bytes_to_GCR(BYTE *buffer, BYTE *ptr)
{
    *ptr = GCR_conv_data[(*buffer) >> 4] << 3;
    *ptr |= GCR_conv_data[(*buffer) & 0x0f] >> 2;
    ptr++;

    *ptr = GCR_conv_data[(*buffer) & 0x0f] << 6;
    buffer++;
    *ptr |= GCR_conv_data[(*buffer) >> 4] << 1;
    *ptr |= GCR_conv_data[(*buffer) & 0x0f] >> 4;
    ptr++;

    *ptr = GCR_conv_data[(*buffer) & 0x0f] << 4;
    buffer++;
    *ptr |= GCR_conv_data[(*buffer) >> 4] >> 1;
    ptr++;

    *ptr = GCR_conv_data[(*buffer) >> 4] << 7;
    *ptr |= GCR_conv_data[(*buffer) & 0x0f] << 2;
    buffer++;
    *ptr |= GCR_conv_data[(*buffer) >> 4] >> 3;
    ptr++;

    *ptr = GCR_conv_data[(*buffer) >> 4] << 5;
    *ptr |= GCR_conv_data[(*buffer) & 0x0f];
}


int convert_4bytes_from_GCR(BYTE *gcr, BYTE *plain)
{
    int hnibble, lnibble;

    if((hnibble = GCR_decode[gcr[0] >> 3]) < 0) return -1;
    if((lnibble = GCR_decode[((gcr[0] << 2) | (gcr[1] >> 6)) & 0x1f]) < 0) return -1;
    *plain++ = hnibble << 4 | lnibble;
    if((hnibble = GCR_decode[(gcr[1] >> 1) & 0x1f]) < 0) return -1;
    if((lnibble = GCR_decode[((gcr[1] << 4) | (gcr[2] >> 4)) & 0x1f]) < 0) return -1;
    *plain++ = hnibble << 4 | lnibble;
    if((hnibble = GCR_decode[((gcr[2] << 1) | (gcr[3] >> 7)) & 0x1f]) < 0) return -1;
    if((lnibble = GCR_decode[(gcr[3] >> 2) & 0x1f]) < 0) return -1;
    *plain++ = hnibble << 4 | lnibble;
    if((hnibble = GCR_decode[((gcr[3] << 3) | (gcr[4] >> 5)) & 0x1f]) < 0) return -1;
    if((lnibble = GCR_decode[gcr[4] & 0x1f]) < 0) return -1;
    *plain++ = hnibble << 4 | lnibble;

    return(1);
}


static void convert_sector_to_GCR(BYTE *buffer, BYTE *ptr, int track,
                                  int sector, BYTE diskID1, BYTE diskID2)
{
    int i;
    BYTE buf[4];

    memset(ptr, 0xff, 5);       /* Sync */
    ptr += 5;

    buf[0] = 0x08;              /* Header identifier */
    buf[1] = sector ^ track ^ diskID2 ^ diskID1;
    buf[2] = sector;
    buf[3] = track;
    convert_4bytes_to_GCR(buf, ptr);
    ptr += 5;

    buf[0] = diskID2;
    buf[1] = diskID1;
    buf[2] = buf[3] = 0x0f;
    convert_4bytes_to_GCR(buf, ptr);
    ptr += 5;

    memset(ptr, 0x55, 9);       /* Header Gap */
    ptr += 9;

    memset(ptr, 0xff, 5);       /* Sync */
    ptr += 5;

    for (i = 0; i < 65; i++) {
        convert_4bytes_to_GCR(buffer, ptr);
        buffer += 4;
        ptr += 5;
    }

    /* FIXME: This is approximated.  */
    memset(ptr, 0x55, 6);       /* Gap before next sector.  */
    ptr += 6;

}


int read_sixpack_id(BYTE *sixpack, BYTE *id)
{
    BYTE buf[4];

    if (convert_4bytes_from_GCR(sixpack+5,buf))
    {
        id[0]=buf[1];
        id[1]=buf[0];
        id[2]='\0';
        printf("Disk ID: %s (0x%02x 0x%02x)\n", id, id[0], id[1]);
        return(1);
    }
    else
    {
        printf("WARNING: Couldn't read disk ID!\n");
        id[0]=id[1]=0;
        id[2]='\0';
        return(0);
    }
}


int convert_sixpack_sector(int track, int sector, int sectors_on_track)
{
 /* this braindead function is necessary because the Zip Archiver reads
    and stores sectors in this order, probably for speed reasons */

    int i;
    int six_sector;
    int used[21];
    
    for (i=0; i < sectors_on_track; i++) used[i] = 0;

    for (i=six_sector=0; i < sectors_on_track; i++)
    {
        if (six_sector == sector) return(i);        
        used[six_sector] = 1;

        if (sectors_on_track <= 8)
        {
            six_sector++;
            continue;
        }
        else
        {
            six_sector += 8;
            if (six_sector >= sectors_on_track)
                six_sector -= sectors_on_track;
            while (used[six_sector])
                if((++six_sector) >= sectors_on_track) exit (-1);
        }
    }
}


static int convert_GCR_sector(BYTE *sixpack, BYTE *ptr, int track,
                                  int sector, BYTE diskID1, BYTE diskID2)
{
    int i;
    int sixp_sector, data_sector;
    BYTE buf[4];
    BYTE *sixp_track;
    BYTE header[10];


    if (track > 40) return (0); /* only 40 tracks stored in sixpack */

    for (sixp_track = sixpack, i = 1; i < track; i++)
        sixp_track += (0x100 + sixp_track[0xff] * 326);

    if (sector == 0) printf("(%02d) ", sixp_track[0xff]);

    memset(ptr, 0xff, 5);       /* Sync */
    ptr += 5;


    for (sixp_sector = 0; sixp_sector < sixp_track[0xff]; sixp_sector++)
    {
        convert_4bytes_from_GCR(sixp_track+sixp_sector*10, header);
        convert_4bytes_from_GCR(sixp_track+sixp_sector*10+5, header+4);
        if ((header[0]==0x08) && (header[2]==sector) && (header[3]==track))
            break;
    }

    if (sixp_sector == sixp_track[0xff]) return (0);

    if ((header[5]==diskID1) || (header[4]==diskID2))
        printf(".");
    else
        printf("-");
    

    for(i=0; i < 10; i++)
        *ptr++ = sixp_track[sixp_sector*10+i];

    memset(ptr, 0x55, 9);       /* Header Gap */
    ptr += 9;

    memset(ptr, 0xff, 5);       /* Sync */
    ptr += 5;

    data_sector = convert_sixpack_sector(track, sixp_sector, sixp_track[0xff]);

    sixpack = sixp_track+0x100+data_sector*326+0x46;
    for (i = 0; i < 0x100; i++)
    {
        *ptr++ = *sixpack++;
    }

    sixpack = sixp_track+256+data_sector*326;
    for (i = 0; i < 0x46; i++)
    {
        *ptr++ = *sixpack++;
    }

    /* FIXME: This is approximated.  */
    memset(ptr, 0x55, 5);       /* Gap before next sector.  */
    ptr += 5;

    return (1);
}


static int write_dword(FILE *fd, DWORD *buf, int num)
{
    int i;
    BYTE *tmpbuf;

    tmpbuf = malloc(num);

    for (i = 0; i < (num / 4); i++) {
        tmpbuf[i * 4] = buf[i] & 0xff;
        tmpbuf[i * 4 + 1] = (buf[i] >> 8) & 0xff;
        tmpbuf[i * 4 + 2] = (buf[i] >> 16) & 0xff;
        tmpbuf[i * 4 + 3] = (buf[i] >> 24) & 0xff;
    }

    if (fwrite((char *)tmpbuf, num, 1, fd) < 1) {
        free(tmpbuf);
        return -1;
    }
    free(tmpbuf);
    return 0;
}


BYTE *load_sixpack(char *sixpackname)
{
    int part;
    unsigned long file_len, archive_len;    
    FILE *fpsix;
    BYTE *sixpack_data, *current_data;
    char fullname[MAXSTRLEN];

 /* Bytes 0 and 1 in each sixpack file are the start address and will
    be ignored. Byte 2 needs to be ignored, because due to a bug in
    the Zipcode Archiver saving starts one byte too early */


    archive_len = 0;
    for (part = 0; part < 6; part++)
    {
        sprintf(fullname, "%d!!%s.prg", part+1, sixpackname);
        fpsix = fopen(fullname, "rb");
        if (fpsix == NULL)
        {
            fprintf(stderr, "Cannot open Sixpack file %s.\n", fullname);
            exit (-1);
        }

	fseek(fpsix, 0, SEEK_END);
	file_len = ftell(fpsix);
	archive_len += (file_len - 3);

        printf("%s: %5u bytes\n",fullname, file_len);

        close(fullname);
    }
    printf("%s: %u bytes\n",sixpackname, archive_len);
    current_data = sixpack_data = malloc(sizeof(BYTE) * archive_len);
    if (sixpack_data == NULL)
    {
        fprintf(stderr, "Cannot allocate memory.\n");
        exit (-1);
    }


    for (part = 0; part < 6; part++)
    {
        sprintf(fullname, "%d!!%s.prg", part+1, sixpackname);
        fpsix = fopen(fullname, "rb");
        if (fpsix == NULL)
        {
            fprintf(stderr, "Cannot open Sixpack file %s.\n", fullname);
            exit (-1);
        }

        fseek(fpsix, 0, SEEK_END);
        file_len = ftell(fpsix) - 3;
        if (file_len <= 0)
        {
            fprintf(stderr, "Error: File too short!\n");
            exit (-1);
        }
        fseek(fpsix, 3, SEEK_SET);
	
        if (fread(current_data, sizeof(BYTE), file_len, fpsix) < file_len)
        {
            fprintf(stderr, "Error: File ended too soon!\n");
            exit (-1);
        }
	current_data += file_len;
        close(fullname);
    }
    return (sixpack_data);
}


void usage(void)
{
    fprintf(stderr, "Wrong number of arguments.\n"
    "Usage: s2g Sixpack [g64image]\n\n");
    exit (-1);
}


int main(int argc, char **argv)
{
    FILE *fdd64, *fdg64;
    char sixpackname[1024], g64name[1024];
    BYTE *sixpack_data;
    int track, sector;
    BYTE gcr_header[12], id[3];
    DWORD gcr_track_p[MAX_TRACKS_1541 * 2];
    DWORD gcr_speed_p[MAX_TRACKS_1541 * 2];
    BYTE gcr_track[7930], rawdata[260];
    BYTE *gcrptr;

#if defined DJGPP
    _fmode = O_BINARY;
#endif


    fprintf(stdout,
"\ns2g is a small stand-alone converter to create a G64 GCR image out\n"
"of a Sixpack 1541 disk archive.  Copyright 1999 Markus Brenner.\n"
"This is free software, covered by the GNU General Public License.\n"
"Version %.2f\n\n", VERSION);

    id[0]=id[1]=id[2] = '\0';

    while (--argc && (*(++argv)[0] == '-'))
    {
        if ((*argv)[1] == 'i')
        {
            id[0] = strtol(*(++argv), (char**)NULL, 0);
            id[1] = strtol(*(++argv), (char**)NULL, 0);
            id[2] = '\0';
            printf("ID: %s\n",id);
            argc--;
            argc--;
        }
        else usage();
    }

    if (argc == 1)
    {
        char *dot;
        strcpy(sixpackname, argv[0]);
        strcpy(g64name, sixpackname);
        dot = strrchr(g64name, '.');
        if (dot != NULL)
            strcpy(dot, ".g64");
        else
            strcat(g64name, ".g64");
    }
    else if (argc == 2)
    {
        strcpy(sixpackname, argv[0]);
        strcpy(g64name, argv[1]);
    }
    else usage();

    sixpack_data = load_sixpack(sixpackname);

    read_sixpack_id(sixpack_data, id);


    fdg64 = fopen(g64name, "wb");
    if (fdg64 == NULL) {
        fprintf(stderr, "Cannot open G64 image %s.\n", g64name);
        exit (-1);
    }


    strcpy((char *) gcr_header, "GCR-1541");

    gcr_header[8] = 0;
    gcr_header[9] = MAX_TRACKS_1541 * 2;
    gcr_header[10] = 7928 % 256;
    gcr_header[11] = 7928 / 256;

    if (fwrite((char *)gcr_header, sizeof(gcr_header), 1, fdg64) != 1) {
        fprintf(stderr, "Cannot write G64 header.\n");
        goto fail;
    }

    for (track = 0; track < MAX_TRACKS_1541; track++) {
        gcr_track_p[track * 2] = 12 + MAX_TRACKS_1541 * 16 + track * 7930;
        gcr_track_p[track * 2 + 1] = 0;
        gcr_speed_p[track * 2] = speed_map_1541[track];
        gcr_speed_p[track * 2 + 1] = 0;
    }

    if (write_dword(fdg64, gcr_track_p, sizeof(gcr_track_p)) < 0) {
        fprintf(stderr, "Cannot write track header.\n");
        goto fail;
    }

    if (write_dword(fdg64, gcr_speed_p, sizeof(gcr_speed_p)) < 0) {
        fprintf(stderr, "Cannot write speed header.\n");
        goto fail;
    }

    for (track = 0; track < MAX_TRACKS_1541; track++) {
        int raw_track_size[4] = { 6250, 6666, 7142, 7692 };

        printf("\nTrack: %2d - Sector: ",track+1);

        memset(&gcr_track[2], 0xff, 7928);
        gcr_track[0] = raw_track_size[speed_map_1541[track]] % 256;
        gcr_track[1] = raw_track_size[speed_map_1541[track]] / 256;
        gcrptr = &gcr_track[2];

        for (sector = 0; sector < sector_map_1541[track + 1]; sector++) {
            BYTE chksum;
            int i;

            printf("%d",sector);

            memset(rawdata, 0, 260);
            rawdata[0] = 7;
            chksum = rawdata[1];
            for (i = 1; i < 256; i++)
                chksum ^= rawdata[i + 1];
            rawdata[257] = chksum;

            if (!convert_GCR_sector(sixpack_data, gcrptr, track + 1, sector, id[0], id[1]))
                convert_sector_to_GCR(rawdata, gcrptr, track + 1, sector, id[0], id[1]);
            gcrptr += 360;
        }

        if (fwrite((char *) gcr_track, sizeof(gcr_track), 1, fdg64) != 1) {
            fprintf(stderr, "Cannot write track data.\n");
            goto fail;
        }
    }


fail:
    fclose(fdg64);
    return -1;
}
