// Verwaltung des seriellen IEC-Bus

#include "pc64.h"

extern "C" void cdecl ResetIEC();
extern "C" void cdecl StartIEC();
extern "C" void cdecl StopIEC();
extern "C" word cdecl Listen(word wDevice);
extern "C" word cdecl Talk(word wDevice);
extern "C" word cdecl ChListen(word wChannel);
extern "C" word cdecl ChTalk(word wChannel);
extern "C" word cdecl IECIn();
extern "C" word cdecl IECOut(byte bValue);
extern "C" word cdecl Unlisten();
extern "C" word cdecl Untalk();

static flag IsDevicePresent(CDevice* pDevice) {
  if (!pDevice) {
    return FALSE;
  }
  if (pDevice->GetType() != DEVLINK) {
    return TRUE;
  }
  if (fWindowsNT) {
    return FALSE;
  }
  char acDir[10];
  pDevice->GetDir(acDir, 9);
  assert(strlen(acDir) > 0);
  assert(strlen(acDir) < 10);
  word wLpt = acDir[0] - '1';
  assert(wLpt < 4);
  acDir[0] = (char)(0x20 + def.fScreenOn);
  if (L64Send(wLpt, acDir, 1) == 1) {
    return TRUE;
  }
  return FALSE;
}

void cdecl ResetIEC() {
  for (int i = 0; i < 16; i++) {
    if (apDevice[i]) {
      apDevice[i]->Reset();
    }
  }
  iec.eMode = IEC_FREE;
}

void cdecl StartIEC() {
  for (int i = 0; i < 16; i++) {
    if (apDevice[i]) {
      apDevice[i]->Start();
    }
  }
}

void cdecl StopIEC() {
  for (int i = 0; i < 16; i++) {
    if (apDevice[i]) {
      apDevice[i]->Stop();
    }
  }
}

static void freeBus() {
  switch (iec.eMode) {
  case IEC_OPEN:
    #if GERMAN
      INFO (DEV) "Datei %s ffnen Gert %02X Kanal %02X", MakeStr(iec.acName, iec.wName, 45), iec.bDevice, iec.bChannel);
    #else
      INFO (DEV) "Open file %s device %02X channel %02X", MakeStr(iec.acName, iec.wName, 45), iec.bDevice, iec.bChannel);
    #endif
    iec.acName[iec.wName] = 0;
    apDevice[iec.bDevice]->Open(iec.bChannel, iec.acName, iec.wName);
    break;
  case IEC_CLOSE:
    #if GERMAN
      INFO (DEV) "Datei schlieen Gert %02X Kanal %02X", iec.bDevice, iec.bChannel);
    #else
      INFO (DEV) "Close file device %02X channel %02X", iec.bDevice, iec.bChannel);
    #endif
    apDevice[iec.bDevice]->Close(iec.bChannel);
    break;
  case IEC_READ:
    if (iec.bChannel == 15) {
      #if GERMAN
        INFO (DEV) "Fehlerkanal lesen Gert %02X", iec.bDevice);
      #else
        INFO (DEV) "Read error on device %02X", iec.bDevice);
      #endif
    }
    break;
  case IEC_WRITE:
    if (iec.bChannel == 15 && iec.wCommand) {
      #if GERMAN
        INFO (DEV) "Befehl %s ausfhren Gert %02X", MakeStr(iec.acCommand, iec.wCommand, 46), iec.bDevice);
      #else
        INFO (DEV) "Execute command %s on device %02X", MakeStr(iec.acCommand, iec.wCommand, 43), iec.bDevice);
      #endif
      apDevice[iec.bDevice]->Command(iec.acCommand, iec.wCommand);
    }
    break;
  }
  iec.eMode = IEC_FREE;
}

word cdecl Listen(word wDev) {
  #if GERMAN
    INFO (IEC) "Listen Gert %02X", wDev);
  #else
    INFO (IEC) "Listen device %02X", wDev);
  #endif
  if (iec.eMode != IEC_FREE) {
    #if GERMAN
      WARNING (IEC) "Der Bus war noch von Gert %02X Kanal %02X belegt!", iec.bDevice, iec.bChannel);
    #else
      WARNING (IEC) "Device %02X channel %02X did not free the bus!", iec.bDevice, iec.bChannel);
    #endif
    freeBus();
  }
  if (wDev >= 16 || !IsDevicePresent(apDevice[wDev])) {
    WARNING (IEC) "Device not present!");
    return ENODEVICE;
  }
  iec.eMode = IEC_LISTEN;
  iec.bDevice = (byte)wDev;
  iec.bChannel = 0;
  return 0;
}

word cdecl Talk(word wDev) {
  #if GERMAN
    INFO (IEC) "Talk Gert %02X", wDev);
  #else
    INFO (IEC) "Talk device %02X", wDev);
  #endif
  if (iec.eMode != IEC_FREE) {
    #if GERMAN
      WARNING (IEC) "Der Bus war noch von Gert %02X Kanal %02X belegt!", iec.bDevice, iec.bChannel);
    #else
      WARNING (IEC) "Device %02X channel %02X did not free the bus!", iec.bDevice, iec.bChannel);
    #endif
    freeBus();
  }
  if (wDev >= 16 || !IsDevicePresent(apDevice[wDev])) {
    WARNING (IEC) "Device not present!");
    return ENODEVICE;
  }
  iec.eMode = IEC_TALK;
  iec.bDevice = (byte)wDev;
  iec.bChannel = 0;
  return 0;
}

word cdecl ChListen(word wCh) {
  INFO (IEC) "Listen Kanal %02X", wCh);
  if (iec.eMode != IEC_LISTEN) {
    #if GERMAN
      WARNING (IEC) "Der Bus ist nicht im Listen-Modus!");
    #else
      WARNING (IEC) "Bus is not in listen mode!");
    #endif
    freeBus();
    return ENODEVICE | ENDOFFILE;
  }
  iec.wName = 0;
  iec.wCommand = 0;
  iec.bChannel = (byte)(wCh & 0x0F);
  switch (wCh & 0xF0) {
  case 0xF0:
    iec.eMode = (word)iec.bChannel == 15 ? IEC_WRITE : IEC_OPEN;
    break;
  case 0xE0:
    iec.eMode = (word)iec.bChannel == 15 ? IEC_WRITE : IEC_CLOSE;
    break;
  default:
    #if GERMAN
      WARNING (IEC) "Unbekannter Kanalmodus wurde auf 60 (Ausgabe) gendert!");
    #else
      WARNING (IEC) "Unknown channel mode was changed to 60 (output)!");
    #endif
  case 0x60:
    iec.eMode = IEC_WRITE;
  }
  return 0;
}

word cdecl ChTalk(word wCh) {
  #if GERMAN
    INFO (IEC) "Talk Kanal %02X", wCh);
  #else
    INFO (IEC) "Talk channel %02X", wCh);
  #endif
  if (iec.eMode != IEC_TALK) {
    #if GERMAN
      WARNING (IEC) "Der Bus ist nicht im Talk-Modus!");
    #else
      WARNING (IEC) "The bus is not in talk mode!");
    #endif
    freeBus();
    return ENODEVICE;
  }
  if ((wCh & 0xF0) != 0x60) {
    #if GERMAN
      WARNING (IEC) "Unbekannter Kanalmodus wurde auf 60 (Eingabe) gendert!");
    #else
      WARNING (IEC) "Unknown channel mode was changed to 60 (input)!");
    #endif
  }
  iec.eMode = IEC_READ;
  iec.bChannel = (byte)(wCh & 0x0F);
  if (iec.bChannel == 15 && apDevice[iec.bDevice]->GetType() != DEVIMAGE) {
    apDevice[iec.bDevice]->GetError(iec.acError, 62);
    strcat(iec.acError, "\r");
    iec.wError = 0;
  }
  return 0;
}

word cdecl IECIn() {
  if (iec.eMode != IEC_READ) {
    INFO (IEC) "Eingabe ??", 0);
    #if GERMAN
      WARNING (IEC) "Der Bus ist nicht im Eingabe-Modus!");
    #else
      WARNING (IEC) "The bus is not in input mode!");
    #endif
    return EINPUT;
  }
  word wValue;
  if (iec.bChannel == 15 && apDevice[iec.bDevice]->GetType() != DEVIMAGE) {
    wValue = (byte)iec.acError[iec.wError];
    if (!wValue) {
      wValue = EINPUT;
    } else {
      if (!iec.acError[++iec.wError]) {
        wValue |= ENDOFFILE;
      }
    }
  } else {
    wValue = apDevice[iec.bDevice]->Get(iec.bChannel);
  }
  #if GERMAN
    INFO (IEC) "Eingabe %02X", (byte)wValue);
  #else
    INFO (IEC) "Input %02X", (byte)wValue);
  #endif
  return wValue;
}

word cdecl IECOut(byte bVal) {
  #if GERMAN
    INFO (IEC) "Ausgabe %02X", bVal);
  #else
    INFO (IEC) "Output %02X", bVal);
  #endif
  switch (iec.eMode) {
  case IEC_OPEN:
    if (iec.wName >= 63) {
      return EOUTPUT;
    }
    iec.acName[iec.wName++] = (char)bVal;
    return 0;
  case IEC_LISTEN:
    #if GERMAN
      INFO (IEC) "Fehlende Sekundradresse wird durch 0 ersetzt");
    #else
      INFO (IEC) "Missing channel number defaults to 0");
    #endif
    iec.eMode = IEC_WRITE;
  case IEC_WRITE:
    if (iec.bChannel == 15) {
      if (iec.wCommand >= 64) {
        return EOUTPUT;
      }
      iec.acCommand[iec.wCommand++] = (char)bVal;
      return 0;
    } else {
      return apDevice[iec.bDevice]->Put(iec.bChannel, bVal);
    }
  }
  #if GERMAN
    WARNING (IEC) "Der Bus ist nicht im Ausgabe-Modus!");
  #else
    WARNING (IEC) "The bus is not in output mode!");
  #endif
  return ENODEVICE;
}

word cdecl Unlisten() {
  INFO (IEC) "Unlisten");
  switch (iec.eMode) {
  case IEC_WRITE:
    apDevice[iec.bDevice]->Unlisten();
  case IEC_OPEN:
  case IEC_CLOSE:
    freeBus();
    return 0;
  }
  #if GERMAN
    WARNING (IEC) "Der Bus ist nicht im Ausgabe-Modus!");
  #else
    WARNING (IEC) "The bus is not in output mode!");
  #endif
  freeBus();
  return 0;
}

word cdecl Untalk() {
  INFO (IEC) "Untalk");
  if (iec.eMode != IEC_READ) {
    #if GERMAN
      WARNING (IEC) "Der Bus ist nicht im Eingabe-Modus!");
    #else
      WARNING (IEC) "The bus is not in input mode!");
    #endif
  }
  freeBus();
  return 0;
}
