type
{RGB color record}
  RGBColor = record
    Red,
    Green,
    Blue: Byte;
  end;

const
{Maximum value for a byte}
  MaxByte = $FF;
{Maximum value for a word}
  MaxWord = $FFFF;
{Hexa digits}
  HexaNum: string[16] = '0123456789ABCDEF';
{Base value for EGA RGB values: when high color bit set, use this value; when
  low color bit set, use twice this value; when both color bits set, use three
  times this value}
  EGAColorRGBBase = $54;
{Number of Commodore colors}
  CBMColorNum = $10;
{Number of EGA colors}
  EGAColorNum = $40;
{CBM color RGB values; from VICE palette, measured by Pepto}
  CBMColors: array [$00..CBMColorNum - 1] of RGBColor =
    ((Red:$00; Green: $00; Blue: $00),
     (Red:$FF; Green: $FF; Blue: $FF),
     (Red:$68; Green: $37; Blue: $2B),
     (Red:$70; Green: $A4; Blue: $B2),
     (Red:$6F; Green: $3D; Blue: $86),
     (Red:$58; Green: $8D; Blue: $43),
     (Red:$35; Green: $28; Blue: $79),
     (Red:$B8; Green: $C7; Blue: $6F),
     (Red:$6F; Green: $4F; Blue: $25),
     (Red:$43; Green: $39; Blue: $00),
     (Red:$9A; Green: $67; Blue: $59),
     (Red:$44; Green: $44; Blue: $44),
     (Red:$6C; Green: $6C; Blue: $6C),
     (Red:$9A; Green: $D2; Blue: $84),
     (Red:$6C; Green: $5E; Blue: $B5),
     (Red:$95; Green: $95; Blue: $95));

var
  CBMColor,
  EGAColor: Byte;
  RedDist,
  GreenDist,
  BlueDist: Longint;
  EGAColors: array [$00..EGAColorNum - 1] of RGBColor;
  EGADists: array [$00..CBMColorNum - 1, $00..EGAColorNum - 1] of Word;

{Convert longint into hexadecimal representation}
function HexaStr(D: Longint; L: Byte): string;
var
  I             : Byte;
  S             : string;
begin
  S := '';
  for I := L - 1 downto 0 do S := S + HexaNum[(D shr (I * 4) and $0F) + 1];
  HexaStr := S;
end;

{Compute square root with Newton-iteration: x[0] = whatever; x[k+1] = (x[k] +
  Number / x[k]) / 2}
function SquareRoot(Num: Longint): Longint;
var
  Temp,
  Prev,
  Result: Longint;
begin
  if Num = 0 then
  begin
    Result := 0;
  end
  else
  begin
    Prev := Num shr 1;
    Result := Longint(MaxWord);
    while Prev <> Result do
    begin
      Temp := Prev;
      Prev := Result;
      Result := (Temp + Num div Temp) shr 1;
    end;
  end;
  SquareRoot := Result;
end;

begin
{Fill EGA color table with RGB values}
  for EGAColor := $00 to EGAColorNum - 1 do
  begin
    EGAColors[EGAColor].Red := (((EGAColor and $04) shr 1) or ((EGAColor and $20) shr 5)) * EGAColorRGBBase;
    EGAColors[EGAColor].Green := ((EGAColor and $02) or ((EGAColor and $10) shr 4)) * EGAColorRGBBase;
    EGAColors[EGAColor].Blue := (((EGAColor and $01) shl 1) or ((EGAColor and $08) shr 3)) * EGAColorRGBBase;
  end;
{Compute difference between each Commodore and EGA color as distance of two
  points in three-dimensional space: Distance = Square_Root(Red_Distance ^ 2 +
  Green_Distance ^ 2 + Blue_Distance ^ 2)}
  for CBMColor := $00 to CBMColorNum - 1 do
  begin
    for EGAColor := $00 to EGAColorNum - 1 do
    begin
      RedDist := Abs(Integer(CBMColors[CBMColor].Red) - Integer(EGAColors[EGAColor].Red));
      GreenDist := Abs(Integer(CBMColors[CBMColor].Green) - Integer(EGAColors[EGAColor].Green));
      BlueDist := Abs(Integer(CBMColors[CBMColor].Blue) - Integer(EGAColors[EGAColor].Blue));
      EGADists[CBMColor, EGAColor] := SquareRoot(Sqr(RedDist) + Sqr(GreenDist) + Sqr(BlueDist));
    end;
  end;
{Display table header: Commodore colors}
  Write(',');
  for CBMColor := $00 to CBMColorNum - 1 do
  begin
    Write('$', HexaStr(CBMColor, 2));
    if CBMColor < CBMColorNum - 1 then Write(',');
  end;
  WriteLn;
{Display data lines: first column is EGA color, the others are differences}
  for EGAColor := $00 to EGAColorNum - 1 do
  begin
    Write('$', HexaStr(EGAColor, 2), ',');
    for CBMColor := $00 to CBMColorNum - 1 do
    begin
      Write('$', HexaStr(EGADists[CBMColor, EGAColor], 4));
      if CBMColor < CBMColorNum - 1 then Write(',');
    end;
    WriteLn;
  end;
end.
