/*
 * Extension: Zmodem with FTP asynhronous background transfer (ZmFTP).
 * Copyright (c) 1995 CAD lab.
 *
 * v1.0 17/11/95 bob@turbo.nsk.su
 */
/******************************************************************************/
/* Project : Unite!       File : zmodem transmit       Version : 1.02         */
/*                                                                            */
/* (C) Mattheij Computer Service 1994                                         */
/*                                                                            */
/* contact us through (in order of preference)                                */
/*                                                                            */
/*   email:          jacquesm@hacktic.nl                                      */
/*   mail:           MCS                                                      */
/*                   Prinses Beatrixlaan 535                                  */
/*                   2284 AT  RIJSWIJK                                        */
/*                   The Netherlands                                          */
/*   voice phone:    31+070-3936926                                           */
/******************************************************************************/

#include <sys/types.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <signal.h>
#include <ctype.h>
#include <stdio.h>
#include <fcntl.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include "version.h"

#include "zmodem.h"
#include "zmdm.h"
#include "ftp.h"

extern int errno;

#define MAX_SUBPACKETSIZE 1024

int opt_v = FALSE;												/* show progress output */
int opt_d = FALSE;												/* show debug output */
int subpacket_size = MAX_SUBPACKETSIZE;							/* data subpacket size. may be modified during a session */
int n_files_remaining;
unsigned char tx_data_subpacket[1024];

long current_file_size;
time_t transfer_start;

/* 
 * show the progress of the transfer like this:
 * zmtx: sending file "garbage" 4096 bytes ( 20%)
 */

void
show_progress(char *name, FILE *fp)

{
	time_t duration;
	int cps;
	int percentage;

	if (current_file_size > 0)
		percentage = (ftell(fp) * 100) / current_file_size;
	else 	percentage = 100;

	duration = time(NULL) - transfer_start;

	if (duration == 0l) duration = 1l;

	cps = (ftell(fp) - restart_point) / duration;

	fprintf(stderr, "zmtx: sending \"%s\" %8ld bytes (%3d%% %5dcps)\n",
		name,ftell(fp),percentage,cps);
	fflush(stderr);
}

/*
 * send from the current position in the file
 * all the way to end of file or until something goes wrong.
 * (ZNAK or ZRPOS received)
 * the name is only used to show progress
 */

int
send_from(char * name,FILE * fp)

{
	int n;
	int type = ZCRCG;
	char zdata_frame[] = { ZDATA, 0, 0, 0, 0 };

	/*
 	 * put the file position in the ZDATA frame
	 */

	zdata_frame[ZP0] =  ftell(fp)        & 0xff;
	zdata_frame[ZP1] = (ftell(fp) >> 8)  & 0xff;
	zdata_frame[ZP2] = (ftell(fp) >> 16) & 0xff;
	zdata_frame[ZP3] = (ftell(fp) >> 24) & 0xff;

	tx_header(zdata_frame);
	/*
	 * send the data in the file
	 */

	while (ftell(fp) < current_file_size) {
		if (opt_v) {
			show_progress(name, fp);
		}

		/*
		 * read a block from the file
		 */
		if (hostname == NULL) {	/* Read local file */
			n = fread(tx_data_subpacket,1,subpacket_size,fp);
			if (n == 0) {
				/*
				* nothing to send ?
				*/
				break;
			}
		} else {	/* Read remote file */
			do {
				clearerr(fp);
				n = fread(tx_data_subpacket,1,subpacket_size,fp);
				if (n > 0) break;
				if (ferror(fp))	{
					if (opt_v) {
						fprintf(stderr, "zmtx: fread \"%s\": %s\n", name, strerror(errno));
					}
					return ZFERR;
				}
			} while (flag_ftp_done == FALSE);

			if (n < 1) {
				if (feof(fp) &&	ftell(fp) == current_file_size) break;
				if (opt_v) {
					fprintf(stderr, "zmtx: fread \"%s\": unexpected error, ftp died?\n", name);
				}
				return ZABORT;
			}
		}
		/*
		 * at end of file wait for an ACK
		 */
		if (ftell(fp) == current_file_size) {
			type = ZCRCW;
		}

		tx_data(type,tx_data_subpacket,n);

		if (type == ZCRCW) {
			int type;
			do {
				type = rx_header(10000);
				if (type == ZNAK || type == ZRPOS) {
					return type;
				}
			} while (type != ZACK);

			if (ftell(fp) == current_file_size) {
				if (opt_d) {
					fprintf(stderr,"end of file\n");
				}
				return ZACK;
			}
		}

		/* 
		 * characters from the other side
		 * check out that header
		 */

		while (rx_poll()) {
			int type;
			int c;
			c = rx_raw();
			if (c == ZPAD) {
				type = rx_header(1000);
				if (type != TIMEOUT && type != ACK) {
					return type;
				}
			}
		}
	}

	/*
	 * end of file reached.
	 * should receive something... so fake ZACK
	 */

	return ZACK;
}

/*
 * send a file; returns true when session is aborted.
 * (using ZABORT frame)
 */
int
send_file(char * name)

{
	long pos, size;
	struct stat s;
	FILE *fp = NULL;
	unsigned char *p;
	char zfile_frame[] = { ZFILE, 0, 0, 0, 0 };
	char zeof_frame[] = { ZEOF, 0, 0, 0, 0 };
	int type;
	char *n;

	/*
	 * before doing a lot of unnecessary work check if the file exists
	 */
	if (hostname == NULL) {	/* Open local file */
		if ((fp = fopen(name,"r")) == NULL) {
			fprintf(stderr,"zmtx: can't open file \"%s\"\n",name);
			return FALSE;
		}
		fstat(fileno(fp),&s);
	} else {	/* Find remote file */
		if (ftp_stat(name, &s) < 0) {
			fprintf(stderr, "ftp: can't find file \"%s\"\n", name);
			return FALSE;
		}
	}
	size = s.st_size;
	current_file_size = size;

	/*
	 * the file exists. now build the ZFILE frame
	 */

	/*
	 * set conversion option
	 * (not used; always binary)
	 */

	zfile_frame[ZF0] = ZF0_ZCBIN;

	/*
	 * management option
	 */

	if (management_protect) {
		zfile_frame[ZF1] = ZF1_ZMPROT;		
		if (opt_d) {
			fprintf(stderr,"zmtx: protecting destination\n");
		}
	}

	if (management_clobber) {
		zfile_frame[ZF1] = ZF1_ZMCLOB;
		if (opt_d) {
			fprintf(stderr,"zmtx: overwriting destination\n");
		}
	}

	if (management_newer) {
		zfile_frame[ZF1] = ZF1_ZMNEW;
		if (opt_d) {
			fprintf(stderr,"zmtx: overwriting destination if newer\n");
		}
	}

	/*
	 * transport options
	 * (just plain normal transfer)
	 */

	zfile_frame[ZF2] = ZF2_ZTNOR;

	/*
	 * extended options
	 */

	zfile_frame[ZF3] = 0;

	/*
 	 * now build the data subpacket with the file name and lots of other
	 * useful information.
	 */

	/*
	 * first enter the name and a 0
	 */

	p = tx_data_subpacket;

	/*
	 * strip the path name from the filename
	 */

	n = strrchr(name,'/');
	if (n == NULL) {
		n = name;
	}
	else {
		n++;
	}

	strcpy(p,n);

	p += strlen(p) + 1;

	/*
	 * next the file size
	 */

	sprintf(p,"%ld ",size);

	p += strlen(p);

	/*
 	 * modification date
	 */

	sprintf(p,"%lo ",s.st_mtime);

	p += strlen(p);

	/*
	 * file mode
	 */

	sprintf(p,"0 ");

	p += strlen(p);

	/*
	 * serial number (??)
	 */

	sprintf(p,"0 ");

	p += strlen(p);

	/*
	 * number of files remaining
	 */

	sprintf(p,"%d ",n_files_remaining);

	p += strlen(p);

	/*
	 * file type
	 */

	sprintf(p,"0");

	p += strlen(p) + 1;

	do {
		/*
	 	 * send the header and the data
	 	 */

		tx_header(zfile_frame);
		tx_data(ZCRCW,tx_data_subpacket,p - tx_data_subpacket);
	
		/*
		 * wait for anything but an ZACK packet
		 */

		do {
			type = rx_header(10000);
		} while (type == ZACK);

		if (opt_d) {
			fprintf(stderr,"type: %d\n",type);
		}

		if (type == ZSKIP) {
			if (fp != NULL) fclose(fp);
			if (opt_v) {
				fprintf(stderr,"zmtx: skipped file \"%s\"\n",name);
			}
			return FALSE;
		}

	} while (type != ZRPOS);

	transfer_start = 0;
	do {
		/*
		 * fetch pos from the ZRPOS header
		 */

		if (type == ZRPOS) {
			pos = rxd_header[ZP0] | (rxd_header[ZP1] << 8) | (rxd_header[ZP2] << 16) | (rxd_header[ZP3] << 24);
		}

		if (!transfer_start) {
			if (pos >= current_file_size) break;
			if (hostname != NULL) {	/* Open remote file */
				if ((fp = ftp_getbackground(name, pos)) == NULL) {
					if (opt_v) {
						fprintf(stderr, "zmtx: can't open temp file\n");
					}
					return TRUE;
				}
			}
			if (opt_v) {
				if (pos)
					fprintf(stderr,"zmtx: restarting \"%s\" from offset %ld\n", name, pos);
				else	fprintf(stderr,"zmtx: start sending \"%s\"\n",name);
			}
			transfer_start = time(NULL);
		}

		/*
 		 * seek to the right place in the file
		 */
		fseek(fp,pos,0);

		/*
		 * and start sending
		 */

		type = send_from(n,fp);

		if (type == ZFERR || type == ZABORT) {
 			fclose(fp);
			return TRUE;
		}

	} while (type == ZRPOS || type == ZNAK);

	/*
	 * file sent. send end of file frame
	 * and wait for zrinit. if it doesnt come then try again
	 */
	if (fp != NULL) fclose(fp);

	zeof_frame[ZP0] =  s.st_size        & 0xff;
	zeof_frame[ZP1] = (s.st_size >> 8)  & 0xff;
	zeof_frame[ZP2] = (s.st_size >> 16) & 0xff;
	zeof_frame[ZP3] = (s.st_size >> 24) & 0xff;

	do {
		tx_hex_header(zeof_frame);
		type = rx_header(10000);
	} while (type != ZRINIT);

	if (opt_v) {
		fprintf(stderr,"zmtx: \"%s\" sending done\n", name);
	}
	return FALSE;
}

void
usage(void)

{
	printf("zmtx %s (C) Mattheij Computer Service 1994\n",VERSION);
	printf("ZmFTP %s (c) 1995 CAD lab.\n", ZMFTPVER);
	printf("usage: zmftp [-h host [-u user] [-p pswd]] [-n|o|s] [-v] files\n");
	printf("	-h host     ftp host\n");
	printf("	-u user     ftp login\n");
	printf("	-p pswd     ftp password\n");
	printf("	-n          transfer if source is newer\n");
	printf("	-o    	    overwrite if exists\n");
	printf("	-s          skip; don't overwrite if exists\n");
	printf("	-v          verbose output; use redirect stderr to file\n");
	exit(1);
}

static void
onchld()
{
	flag_ftp_done = TRUE;
	pid_ftp = 0;
	if (wait(0) == -1) {
		if (opt_d) {
			fprintf(stderr, "ftp: wait: %s\n", strerror(errno));
		}
		cleanup();
		exit(2);
	}
	if (opt_d) {
		fprintf(stderr, "ftp: receiving done\n");
	}
}

void
cleanup()
{
	if (pid_ftp) {
		if (kill(pid_ftp, SIGINT) < 0) {
			if (opt_d) {
				fprintf(stderr, "ftp: kill: %s\n", strerror(errno));
			}
		} else	onchld();			
	}
	fflush(stderr);
	fd_exit();
}

void
onabort()
{
	cleanup();
	exit(1);
}

int
main(int argc,char ** argv)

{
	int i;
	char *host, *user, *pswd;
	extern char *optarg;
	extern int optind, opterr;

	hostname = NULL;
	host = NULL;
	user = NULL;
	pswd = NULL;
	pid_ftp = 0;
	opterr = 0;

	while ((i = getopt(argc, argv, "h:u:p:nosdv")) != EOF)
		switch (i) {
		case 'h': host = optarg; break;
		case 'u': user = optarg; break;
		case 'p': pswd = optarg; break;
		case 'n': management_newer++; break;
		case 'o': management_clobber++; break;
		case 's': management_protect++; break;
		case 'd': opt_d++; break;
		case 'v': opt_v++; break;
		default: usage();
		}

	if ((host == NULL && (user != NULL || pswd != NULL)) ||
	    (management_newer + management_clobber + management_protect) > 1 ||
	    argv[optind] == NULL) {
		usage();
	}
	if (opt_d) {
		opt_v = TRUE;
	}
	if (host != NULL) {
		char buf[80];
		if (user == NULL) user = "anonymous";
		if (pswd == NULL) {
			pswd = strcpy(buf, "Zmodem@");
			gethostname(&buf[7], sizeof(buf)-7);
		}
		if (ftp_connect(host, 21) == NULL) exit(2);
		if (ftp_login(user, pswd) < 0) exit(2);
		(void) signal(SIGCHLD, onchld);
	}
	(void) signal(SIGINT, SIG_IGN);
	(void) signal(SIGTSTP, SIG_IGN);
	(void) signal(SIGQUIT, onabort);
	(void) signal(SIGHUP, onabort);

	/*
	 * set the io device to transparent
	 */

	fd_init();	

	/*
	 * clear the input queue from any possible garbage
	 * this also clears a possible ZRINIT from an already started
	 * zmodem receiver. this doesn't harm because we reinvite to
	 * receive again below and it may be that the receiver whose
	 * ZRINIT we are about to wipe has already died.
	 */

	rx_purge();

	/*
	 * establish contact with the receiver
	 */

	if (opt_v) {
		fprintf(stderr,"zmtx: establishing contact with receiver\n");
	}

	i = 0;
	do {
		unsigned char zrqinit_header[] = { ZRQINIT, 0, 0, 0, 0 };
		i++;
		if (i > 10) {
			fprintf(stderr,"zmtx: can't establish contact with receiver\n");
			cleanup();
			exit(3);
		}

		tx_raw('z');
		tx_raw('m');
		tx_raw(13);
		tx_hex_header(zrqinit_header);
	} while (rx_header(7000) != ZRINIT);

	if (opt_v) {
		fprintf(stderr,"zmtx: contact established\n");
	}

	/*
	 * decode receiver capability flags
	 * forget about encryption and compression.
	 */

	can_full_duplex					= (rxd_header[ZF0] & ZF0_CANFDX)  != 0;
	can_overlap_io					= (rxd_header[ZF0] & ZF0_CANOVIO) != 0;
	can_break						= (rxd_header[ZF0] & ZF0_CANBRK)  != 0;
	can_fcs_32						= (rxd_header[ZF0] & ZF0_CANFC32) != 0;
	escape_all_control_characters	= (rxd_header[ZF0] & ZF0_ESCCTL)  != 0;
	escape_8th_bit					= (rxd_header[ZF0] & ZF0_ESC8)    != 0;

	use_variable_headers			= (rxd_header[ZF1] & ZF1_CANVHDR) != 0;

	if (opt_d) {
		fprintf(stderr,"receiver %s full duplex\n"          ,can_full_duplex               ? "can"      : "can't");
		fprintf(stderr,"receiver %s overlap io\n"           ,can_overlap_io                ? "can"      : "can't");
		fprintf(stderr,"receiver %s break\n"                ,can_break                     ? "can"      : "can't");
		fprintf(stderr,"receiver %s fcs 32\n"               ,can_fcs_32                    ? "can"      : "can't");
		fprintf(stderr,"receiver %s escaped control chars\n",escape_all_control_characters ? "requests" : "doesn't request");
		fprintf(stderr,"receiver %s escaped 8th bit\n"      ,escape_8th_bit                ? "requests" : "doesn't request");
		fprintf(stderr,"receiver %s use variable headers\n" ,use_variable_headers          ? "can"      : "can't");
	}

	/* 
	 * and send each file in turn
	 */

	n_files_remaining = argc - optind;

	while (argv[optind] != NULL) {
		if (send_file(argv[optind++])) {
			if (opt_v) {
				fprintf(stderr,"zmtx: remote aborted\n");
			}
			break;
		}
		n_files_remaining--;
	}

	/*
	 * close the session
	 */

	if (opt_v) {
		fprintf(stderr,"zmtx: closing the session\n");
	}

	{
		int type;
		unsigned char zfin_header[] = { ZFIN, 0, 0, 0, 0 };

		tx_hex_header(zfin_header);
		do {
			type = rx_header(10000);
		} while (type != ZFIN && type != TIMEOUT);
		
		/*
		 * these Os are formally required; but they don't do a thing
		 * unfortunately many programs require them to exit 
		 * (both programs already sent a ZFIN so why bother ?)
		 */

		if (type != TIMEOUT) {
			tx_raw('O');
			tx_raw('O');
		}
	}

	/*
	 * c'est fini
	 */

	if (opt_d) {
		fprintf(stderr,"zmtx: cleanup and exit\n");
	}
	sleep(1);
	cleanup();
	exit(0);
}
