/*
 *	Copyright (c) 1994 The CAD lab of the
 *	Novosibirsk Institute of Broadcasting and Telecommunication
 *
 *	TNSDrive $Id$
 *
 *	$Log$
 *
 * Redistribution and use in source forms, with and without modification,
 * are permitted provided that this entire comment appears intact.
 *
 * THIS SOURCE CODE IS PROVIDED ``AS IS'' WITHOUT ANY WARRANTIES OF ANY KIND.
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <pwd.h>
#include <unistd.h>
#include <signal.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>

#include "drive.h"
#include "callsign.h"
#include "loginfo.h"
#include "getconf.h"
#include "utmpent.h"

#define	OB	"OnBoard v1.2"

/* ipc.c */
extern void go_conf();
extern void go_talk();

/* userlist.c */
extern void who_is_there();
extern void who_in_conf();
extern void userlist(char *);
extern void callsignlist(char *);
extern void whodo(int);

struct loginfo process;

char tmpbuf[1024], homedir[1024];
char callsign[LOGINFO_NAMELEN], fntty[LOGINFO_TTYLEN];
char *orgdir;
int termflags = 0, globalflags = 0, sigusr1 = 0, pager = 0;

int cflag = 0, dflag = 0, gflag = 0, tflag = 0, uflag = 0, sflag = 0, wflag = 0;

uid_t
findorgdir()
{
	struct passwd *pwd;

	if ((pwd = getpwnam(BBS)) == NULL) {
		if ((pwd = getpwnam("tns")) == NULL) {
			fprintf(stderr, "No %s in this system\n", BBS);
			exit(1);
		}
	}
	orgdir = strdup(pwd->pw_dir);
	return pwd->pw_uid;
}

main(argc, argv)
	int argc;
	char **argv;
{
	int op;
	FILE *fp;
	char *resptty = NULL, *user_name = NULL;
	struct stat st;
	void usage();
	extern char *optarg;
	extern int optind, opterr;

	opterr = 0;
	while ((op = getopt(argc, argv, "cdgtu:s:w:h?")) != EOF)
		switch (op) {
		case 'c':
			++cflag;
			break;
		case 'd':
			++dflag;
			break;
		case 'g':
			++gflag;
			break;
		case 't':
			++tflag;
			break;
		case 'u':
			++uflag;
			if (strcasecmp(optarg, "all"))
				user_name = optarg;
			break;
		case 's':
			++sflag;
			if (strcasecmp(optarg, "all"))
				user_name = optarg;
			break;
		case 'w':
			++wflag;
			user_name = optarg;
			break;
		case 'h':
		case '?':
		default:
			usage();
		}
	op = cflag + dflag + gflag + tflag + uflag + sflag + wflag;
	resptty = argv[optind];
	if (op > 1 || (op && resptty)) {
		fprintf(stderr, "Use only one action per time\n");
		exit(1);
	}

	if (findorgdir() != geteuid()) {
		fprintf(stderr, "This program must have set-uid to %s\n", BBS);
		exit(1);
	}
	initcallsign(orgdir);
	initgetconf(orgdir);

	if (wflag) {
		getuserinfo(printf, user_name);
		exit(0);
	}
	if (uflag) {
		userlist(user_name);
		exit(0);
	}
	if (sflag) {
		callsignlist(user_name);
		exit(0);
	}

	if (cflag) who_in_conf();
	else if (dflag) whodo(FALSE);
	else if (gflag) {
		initprocess();
		dropusr1();
		go_conf();
		quit(0);
	} else if (tflag) {
		initprocess();
		dropusr1();
		go_talk();
		quit(0);
	} else if (resptty) {
		if (resptalk(resptty)) quit(0);
	} else who_is_there();

	exit(0);
}

int
resptalk(tty)
	char *tty;
{
	struct loginfo *info;
	extern int talkportid;

	talkportid = 0;
	while ((info = getinfo(FALSE)) != NULL) if (!strcmp(info->tty, tty)) {
		talkportid = info->port;
		closeinfo();
		break;
	}
	if (!talkportid) {
		fprintf(stderr, "No user on `%s'\n", tty);
		return 0;
	}
	if (info->flags & INTALK) {
		globalflags |= BEGINCHAT;
		initprocess();
		dropusr1();
		talk(-2);
		return 1;
	}
	fprintf(stderr, "There is no talk requested user on `%s'\n", tty);
	return 0;
}

static void
onterm()
{
	fprintf(stderr, "*** interrupted\n");
	quit(1);
}

void
quit(rval)
	int rval;
{
	purgeinfo(&process);
	ttyrestore(1);
	if (sigusr1) dropusr1();
	exit(rval);
}

initprocess()
{
	int i;
	char *ptr;
	struct passwd *pwd;
	struct utmpent *ut;

	if (!isatty(0)) {
		fprintf(stderr, "No control terminal\n");
		exit(1);
	}
	memset(&process, 0, sizeof(struct loginfo));

	if ((pwd = getpwuid(getuid())) == NULL) {
		fprintf(stderr, "Can't find your passwd entry\n");
		exit(1);
	}
	i = 0;
	if (pwd->pw_gecos != NULL) {
		for (ptr = pwd->pw_gecos; *ptr && *ptr != ',' &&
		     i < LOGINFO_NAMELEN-1; i++, ptr++)
			process.name[i] = *ptr;
	}
	process.name[i] = '\0';
	if (!i) {
		fprintf(stderr, "Can't find your real name; please check passwd database\n");
		exit(1);
	}
	(void)strcpy(callsign, pwd->pw_name);
	(void)strcpy(homedir, pwd->pw_dir);
	(void)strcpy(process.tty, ttyname(0) + 5);
	for (i = 0, ptr = process.tty; *ptr; ptr++)
		if (*ptr != '/') fntty[i++] = *ptr;
	fntty[i] = '\0';

	while ((ut = getutmpent(pwd->pw_name)) != NULL) {
		if (!strcmp(process.tty, ut->line)) {
			process.ltime = ut->time;
			break;
		}
	}
	endutmpent();
	if (!process.ltime) process.ltime = time(NULL);

	process.pid = getpid();
	process.baud = ttysave();
	process.flags = (MESGNO|TERMINATOR|INFINITYCONF|INFINITYUSER);
	if (initIPC() < 0) {
		fprintf(stderr, "Can't init IPC\n");
		exit(1);
	}
	ttysetraw();

	signal(SIGHUP, onterm);
	signal(SIGINT, onterm);
	signal(SIGQUIT, SIG_IGN);
	signal(SIGABRT, onterm);
	signal(SIGTERM, onterm);
	signal(SIGTSTP, SIG_IGN);
	signal(SIGUSR1, SIG_IGN);
	signal(SIGUSR2, SIG_IGN);

	pager = 1;
}

void
hidestr(str)
	char *str;
{
	register i, len = strlen(str);
	for (i = 0; i < len; i++) putchr('\b');
	for (i = 0; i < len; i++) putchr(' ');
	putchr('\r');
	return;
}

int
more(curline)
	int *curline;
{
	if (!pager) return 1;

	if (*curline >= DEFSCRLINES) {
		int ch;
		char *prompt = "More? [Y/n]";

		*curline = 0;
		putstr("%s", prompt);
		ch = toupper(getchr(1));
		hidestr(prompt);
		if (ch == 'N' || ch == 'Q') return 0;
	}
	(*curline)++;
	return 1;
}

dropusr1()
{
	pid_t monpid;
	FILE *fp;
	char buf[40];

	if ((fp = fopen(TNSMONPID, "r")) != NULL) {
		if (fgets(buf, sizeof(buf), fp) != NULL) {
			monpid = atoi(buf);
			if (monpid && monpid != process.pid)
				if (kill(monpid, SIGUSR1) == 0) sigusr1++;
		}
		fclose(fp);
	}
	return;
}

void
usage()
{
	fprintf(stderr, "%s\n", OB);
	fprintf(stderr, "usage: onboard [-cdgt | -u substr | -s substr | -w username | tty]\n");
	fprintf(stderr, "where:\n\
\t-c\twho in Conference\n\
\t-d\twho Do\n\
\t-g\tGo into conference\n\
\t-t\tTalk request\n\
\t-u\tUser list, use All or substring(s) for search\n\
\t-s\tcallSign list, use All or substring(s) for search\n\
\t-w\tWho is \"User Name\"\n\
\ttty\tRespond on talk requested by user on ttyXX\n\
\tdefault\tWho on BBS\n");
	exit(1);
}
