/*
 *	Copyright (c) 1996 The CAD lab of the
 *	Siberian State Academy of Telecommunication
 *
 * Redistribution and use in source forms, with and without modification,
 * are permitted provided that this entire comment appears intact.
 *
 * THIS SOURCE CODE IS PROVIDED ``AS IS'' WITHOUT ANY WARRANTIES OF ANY KIND.
 */

#include <sys/types.h>
#include <sys/utsname.h>
#include <stdio.h>
#include <signal.h>
#include <string.h>
#include <unistd.h>
#include <time.h>
#include "netchat.h"
#include "callsign.h"
#include "userbase.h"

void
cmd_help(from, to, arg)
	int from, to;
	struct args *arg;
{
	FILE *fp_in, *fp_out;
	char buf[256];

	if (!from) {
		if (to)	/* Display remote data to user */
			remotedata(to, arg);
		else	/* Display local data to user */
			msgreply("%s DISPLAY=%s", CMDSERVMARKER, helpfile);
		return;
	}
	/* Assume data connection */
	if ((fp_out = opendata(0, NULL)) == NULL) return;
	if ((fp_in = fopen(helpfile, "r")) == NULL) {
		fputs("\n<No help available>", fp_out);
		fclose(fp_out);
		return;
	}
	while (fgets(buf, sizeof(buf), fp_in) != NULL) fputs(buf, fp_out);
	fclose(fp_out);
	fclose(fp_in);
}

void
cmd_ping(from, to, arg)
	int from, to;
	struct args *arg;
{
	if (!to) {	/* Reply to user */
		if (arg->user) {
			struct loginfo *info;
			arg->user = getlogname(arg->user);
			while ((info = getinfo()) != NULL)
				if (!strcasecmp(info->name, arg->user)) {
					closeinfo();
					break;
				}
			if (!info)
				msgreply("<%s:%s is unreachable>",
					 arg->host, arg->user);
			else	msgreply("<%s:%s %s>", arg->host, arg->user,
					 flag2str(info->flags));
		} else	msgreply("<%s is alive>", arg->host);
	} else	/* Forward to remote host */
		msgremote(to, "> %s %s:%s", arg->cmd, arg->host,
			  arg->user ? arg->user : "");
}

void
cmd_conf(from, to, arg)
	int from, to;
	struct args *arg;
{
	int cnt;
	FILE *fp;
	char *ptr, temp[80];
	struct loginfo *info;

	if (!from) {
		if (to)	{
			remotedata(to, arg);
			return;
		}
		if (mktemp(strcpy(temp, TMPFILEMASK)) == NULL ||
		    (fp = fopen(temp, "w")) == NULL) return;
		fprintf(fp, "\nHost         State\n");
		fprintf(fp, "=======================\n");
		for (cnt = 0; cnt < hn; cnt++)
			fprintf(fp, "%-12.12s %s\n", cf[cnt].nick,
				cf[cnt].alive ? "active" : "no answer");
		fclose(fp);
		msgreply("%s DISPLAY=%s", CMDSERVMARKER, temp);
		sleep(1);
		unlink(temp);
		return;
	}
	/* Assume data connection */
	if ((fp = opendata(0, NULL)) == NULL) return;
	if (arg->user) arg->user = getlogname(arg->user);
	cnt = 0;
	fprintf(fp, "\n");
	while ((info = getinfo()) != NULL) if (info->flags & INCONF) {
		if (arg->user && strcasecmp(info->name, arg->user)) continue;
		if (!arg->user && !cnt) {
			fprintf(fp, "The following users are presently in conference on %s:\n", arg->host);
			fprintf(fp, "TTY    LoginTime        UserName                 CallSign\n");
			fprintf(fp, "===============================================================================\n");
		}
		if ((ptr = getsignbyname(info->name)) == NULL) ptr = "Not selected";
		fprintf(fp, "%-6.6s %-.15s  %-24.24s %s", info->tty,
			ctime(&info->ltime)+4, info->name, ptr);
		if (info->flags & TERMINATOR) fprintf(fp, " \t*terminator*");
		else if (info->flags & KICKER) fprintf(fp, " \t*kicker*");
		fprintf(fp, "\n");
		cnt++;
		if (arg->user) {
			closeinfo();
			break;
		}
	}
	if (!cnt) {
		if (arg->user)
			fprintf(fp, "<No user %s in conference on %s>\n", arg->user, arg->host);
		else	fprintf(fp, "<No users in conference on %s>\n",	arg->host);
	}
	fclose(fp);
}

void
cmd_do(from, to, arg)
	int from, to;
	struct args *arg;
{
	int cnt;
	FILE *fp;
	struct loginfo *info;

	if (to)	{	/* Display remote data to user */
		remotedata(to, arg);
		return;
	}
	/* Assume data connection */
	if ((fp = opendata(0, NULL)) == NULL) return;
	if (arg->user) arg->user = getlogname(arg->user);
	cnt = 0;
	fprintf(fp, "\n");
	while ((info = getinfo()) != NULL) {
		if (arg->user && strcasecmp(info->name, arg->user)) continue;
		if (!arg->user && !cnt) {
			fprintf(fp, "TTY    UserName                 Status\n");
			fprintf(fp, "===============================================================================\n");
		}
		fprintf(fp, "%-6.6s %-24.24s %s\n", info->tty, info->name,
			flag2str(info->flags));
		cnt++;
		if (arg->user) {
			closeinfo();
			break;
		}
	}
	if (!cnt) {
		if (arg->user)
			fprintf(fp, "<No user %s on %s>\n", arg->user, arg->host);
		else	fprintf(fp, "<No users on %s>\n", arg->host);
	}
	fclose(fp);
}

void
cmd_who(from, to, arg)
	int from, to;
	struct args *arg;
{
	int cnt;
	FILE *fp;
	char *ptr;
	struct loginfo *info;
	char city[25];

	if (to)	{	/* Display remote data to user */
		remotedata(to, arg);
		return;
	}
	/* Assume data connection */
	if ((fp = opendata(0, NULL)) == NULL) return;
	if (arg->user) arg->user = getlogname(arg->user);
	cnt = 0;
	fprintf(fp, "\n");
	while ((info = getinfo()) != NULL) {
		if (arg->user && strcasecmp(info->name, arg->user)) continue;
		if ((ptr = getuserinfo(info->name, USERLOCATION, "Unknown")) == NULL)
			continue;
		strncpy(city, ptr, sizeof(city)-1);
		city[sizeof(city)-1] = '\0';
		if (!arg->user && !cnt) {
			fprintf(fp, "The following users are presently on %s:\n", arg->host);
			fprintf(fp, "TTY    LoginTime        UserName                 City/State           BaudRate\n");
			fprintf(fp, "==============================================================================\n");
		}
		fprintf(fp, "%-6.6s %.15s  %-24.24s %-20.20s %d\n", info->tty,
			ctime(&info->ltime)+4, info->name, city, info->baud);
		cnt++;
		if (arg->user) {
			closeinfo();
			break;
		}
	}
	if (!cnt) {
		if (arg->user)
			fprintf(fp, "<No user %s on %s>\n", arg->user, arg->host);
		else	fprintf(fp, "<No users on %s>\n", arg->host);
	}
	fclose(fp);
}

void
cmd_info(from, to, arg)
	int from, to;
	struct args *arg;
{
	FILE *fp;

	if (to)	{	/* Display remote data to user */
		remotedata(to, arg);
		return;
	}
	/* Assume data connection */
	if ((fp = opendata(0, NULL)) == NULL) return;
	fprintf(fp, "\n");
	if (arg->user) {
		arg->user = getlogname(arg->user);
		who_is_user(arg->user, fp);
	} else {
		struct utsname info;

		if (uname(&info) == 0) {
			fprintf(fp, "%16s: %s\n", "Operating system", info.sysname);
			fprintf(fp, "%16s: %s\n", "Network node", info.nodename);
			fprintf(fp, "%16s: %s\n", "Release level", info.release);
			fprintf(fp, "%16s: %s\n", "Version level", info.version);
			fprintf(fp, "%16s: %s\n", "Hardware type", info.machine);
		}
		fprintf(fp, "%16s: %s\n", "Netchat daemon", NC);
	}
	fclose(fp);
}

void
cmd_kick(from, to, arg)
	int from, to;
	struct args *arg;
{
	pid_t pid;
	char *ptr, user[40], kicker[40];
	struct loginfo *info;

	if (from || to) return;
	pid = 0;
	*kicker = 0;
	while ((info = getinfo()) != NULL) if (info->flags & INCONF) {
		if (info->port == userport) {
			pid = info->pid;
			(void)strcpy(user, info->name);
		}
		if (info->flags & KICKER) (void)strcpy(kicker, info->name);
	}
	if (!pid) return;
	if (*kicker) {
		msgreply("<%s already kicker>", kicker);
		kill(pid, SIGQUIT);
		return;
	}
	if ((ptr = getuserinfo(user, USERPRIVLEVEL, "0")) == NULL) return;
	srandom((int)time(NULL));
	if (random() % (MAXPRIVLEVEL+1) > atoi(ptr)) kill(pid, SIGQUIT);
	else kill(pid, SIGPIPE);
}

void
cmd_term(from, to, arg)
	int from, to;
	struct args *arg;
{
	pid_t pid;
	char *ptr, user[40], term[40];
	struct loginfo *info;

	if (from || to) return;
	pid = 0;
	*term = 0;
	while ((info = getinfo()) != NULL) if (info->flags & INCONF) {
		if (info->port == userport) {
			pid = info->pid;
			(void)strcpy(user, info->name);
		}
		if (info->flags & TERMINATOR) (void)strcpy(term, info->name);
	}
	if (!pid) return;
	if (*term) {
		msgreply("<%s already terminator>", term);
		kill(pid, SIGHUP);
		return;
	}
	if ((ptr = getuserinfo(user, USERPRIVLEVEL, "0")) == NULL) return;
	srandom((int)time(NULL));
	if ((random() % (MAXPRIVLEVEL+1)) > (atoi(ptr) / 2)) kill(pid, SIGHUP);
	else msgreply("%s KICKER=T", CMDSERVMARKER);
}


struct cmd cmdtab[] = {
	/* "str"	host	user	text	cmd */

	{ "?",		0,	0,	0,	cmd_help },
	{ "help",	0,	0,	0,	cmd_help },
	{ "ping",	1,	0,	0,	cmd_ping },
	{ "conf",	0,	0,	0,	cmd_conf },
	{ "do",		1,	0,	0,	cmd_do   },
	{ "who",	1,	0,	0,	cmd_who  },
	{ "info",	1,	0,	0,	cmd_info },
	{ "kick",	0,	0,	0,	cmd_kick },
	{ "term",	0,	0,	0,	cmd_term },
	{ 0 },
};
