/*
 *	Copyright (c) 1994,1995 The CAD lab of the
 *	Novosibirsk Institute of Broadcasting and Telecommunication
 *
 *	TNSDrive $Id$
 *
 *	$Log$
 *
 * Redistribution and use in source forms, with and without modification,
 * are permitted provided that this entire comment appears intact.
 *
 * THIS SOURCE CODE IS PROVIDED ``AS IS'' WITHOUT ANY WARRANTIES OF ANY KIND.
 */

#include <curses.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <pwd.h>
#include <string.h>
#include <signal.h>
#include <stdlib.h>
#include <unistd.h>
#include <setjmp.h>
#include <errno.h>

#include "drive.h"
#include "loginfo.h"
#include "variables.h"
#include "monitor.h"

static char *header = "TNSDrive BBS on-line Monitor v1.2";
static char *copy  = "Copyright (c) 1994-1997 The CAD lab of the";
static char *right = "Siberian State Academy of Telecommunication, RU";

extern int errno;
void onterm(), onusr1();

char *orgdir, *hname;

struct loginfo *topinfo = NULL;
struct loginfo *usrlist[MAXBBSUSERS];
int nusers = 0;
int redraw = 1;

int usridx, first_line, scr_line, showmode;
int pagesize;

void
rescan()
{
	register i, k;
	struct stat st;

	if (topinfo != NULL) {
		free(topinfo);
		topinfo = NULL;
		nusers = 0;
	}
	if ((i = open(LOGINFO_FILE, O_RDONLY)) < 0) return;
	if (fstat(i, &st) < 0) return;
	if ((topinfo = malloc((int)st.st_size)) == NULL) return;
	while ((k = read(i, topinfo, (int)st.st_size)) < 0 && errno == EINTR);
	close(i);
	if (k != (int)st.st_size || (int)st.st_size % sizeof(struct loginfo)) {
		free(topinfo);
		topinfo = NULL;
		return;
	}
	for (i = 0; i < (int)st.st_size/sizeof(struct loginfo); i++) {
		if (!topinfo[i].pid || kill(topinfo[i].pid, 0)) continue;
		usrlist[nusers++] = &topinfo[i];
		if (nusers >= MAXBBSUSERS) break;
	}

}

main(argc, argv)
	int argc;
	char **argv;
{
	initbbsenv();
	initcallsign(orgdir);
	initgetconf(orgdir);
	inithostname();
	droppidfile();
	initterm();
	signal(SIGHUP, onterm);
	signal(SIGINT, onterm);
	signal(SIGQUIT, onterm);
	signal(SIGTERM, onterm);
	signal(SIGTSTP, SIG_IGN);
	signal(SIGUSR1, onusr1);
	mainloop();
	onterm();
}

inithostname()
{
	int fd;
	char *ptr, buf[21];

	buf[0] = 0;
	gethostname(buf, sizeof(buf));
	buf[sizeof(buf)-1] = 0;
	if ((hname = strdup(buf)) == NULL) exit(1);
	if ((fd = open(".", O_RDONLY)) < 0) exit(1);
	chdir(orgdir);
	fchdir(fd);
	close(fd);
	return;
}

initterm()
{
	puts(NICECOLOR);
	COLS = 80;
	if (initscr() == NULL) exit(1);
	if (LINES < 20) {
		addstr("Must more LINES on term");
		panic(1);
	}
	if (COLS < 80) {
		addstr("Must more COLS on term");
		panic(1);
	}
	pagesize = LINES - MINPAGESIZE;

	cbreak();
	noecho();
	nonl();
/*	leaveok(stdscr, FALSE);
	scrollok(stdscr, FALSE);
*/
	addstr(hname);
	mvaddstr(0, COLS/2 - strlen(header)/2, header);
	mvaddstr(1, 0, "\
    #     TTY     LoginTime        UserName                 BaudRate   State");
	mvaddstr(2, 0, "\
================================================================================");

}

droppidfile()
{
	FILE *fp;
	char buf[40];

	if ((fp = fopen(TNSMONPID, "r")) != NULL) {
		if (fgets(buf, sizeof(buf), fp) != NULL)
			if (kill(atoi(buf), 0) == 0) {
				fprintf(stderr, "Monitor already running\n");
				exit(1);
			}
		fclose(fp);
	}
	if (mkdir(MONITORDIR, 0700) < 0 && errno != EEXIST) {
		perror("mkdir");
		exit(1);
	}
	if ((fp = fopen(TNSMONPID, "w")) == NULL) {
		fprintf(stderr, "Can't write to \"%s\": %s\n", TNSMONPID,
			strerror(errno));
		exit(1);
	}
	fprintf(fp, "%d\n", getpid());
	fclose(fp);
}

void
onterm()
{
	signal(SIGINT, SIG_IGN);
	signal(SIGUSR1, SIG_IGN);
	mvcur(0, COLS - 1, LINES - 1, 0);
	endwin();
	printf(DEFCOLOR);
	unlink(TNSMONPID);
	exit(0);
}

void
onusr1()
{
	redraw++;
	signal(SIGUSR1, onusr1);
}

panic(rval)
	int rval;
{
	refresh();
	signal(SIGINT, SIG_IGN);
	signal(SIGUSR1, SIG_IGN);
	mvcur(0, COLS - 1, LINES - 1, 0);
	endwin();
	printf("\n");
	unlink(TNSMONPID);
	exit(rval);
}

initbbsenv()
{
	struct passwd *pwd;

	/*
	* Get the bbs home directory and bbs user ID.
	*/
	if ((pwd = getpwnam(BBS)) == NULL)
		if ((pwd = getpwnam("tns")) == NULL) {
			fprintf(stderr, "No %s in this system\n", BBS);
			exit(1);
		}
	if (geteuid() != pwd->pw_uid) {
		fprintf(stderr, "This programm must have set-user-ID to %s\n", BBS);
		exit(1);
	}
	orgdir = strdup(pwd->pw_dir);
	return;
}

static int
inputchar()
{
	return getch();
}

static jmp_buf escapepressed;

static void
waitforescape()
{
	alarm(0);
	signal(SIGALRM, SIG_DFL);
	longjmp(escapepressed, 1);
}

int
get_arrow_key(get1ch)
	int (*get1ch)();
{
	int ch, ch1;

	if (setjmp(escapepressed)) return KEYMAP_ESCAPE;
	signal(SIGALRM, waitforescape);
	alarm(1);
	ch = (*get1ch)();
	alarm(0);
	signal(SIGALRM, SIG_DFL);

	if (ch == '[' || ch == 'O')
		ch = (*get1ch)();
	switch (ch) {
		case '\0':	/* xterm */
			return KEYMAP_HOME;
		case 'A':
		case 'i':
			return KEYMAP_UP;

		case 'B':
			return KEYMAP_DOWN;

		case 'D':
			return KEYMAP_LEFT;

		case 'C':
			return KEYMAP_RIGHT;

		case 'I':		/* ansi  PgUp */
		case 'V':		/* at386 PgUp */
		case 'S':		/* 97801 PgUp */
		case 'v':		/* emacs style */
			return KEYMAP_PAGE_UP;

		case 'G':		/* ansi  PgDn */
		case 'U':		/* at386 PgDn */
		case 'T':		/* 97801 PgDn */
			return KEYMAP_PAGE_DOWN;

		case 'H':		/* at386  Home */
			return KEYMAP_HOME;
					
		case 'F':		/* ansi   End */
		case 'Y':		/* at386  End */
			return KEYMAP_END;

		case '5':		/* vt200 PgUp */
			ch = (*get1ch)();	/* eat the ~ (interesting use of words :) */
			return KEYMAP_PAGE_UP;

		case '6':		/* vt200 PgUp */
			ch = (*get1ch)();	/* eat the ~  */
			return KEYMAP_PAGE_DOWN;

		case '1':		/* vt200 PgUp */
			ch = (*get1ch)();
			switch(ch) {	/* xterm */
				case '1':
					ch1 = (*get1ch)();	/* eat the ~ */
					return KEYMAP_F1;
				case '2':
					ch1 = (*get1ch)();	/* eat the ~ */
					return KEYMAP_F2;
				case '3':
					ch1 = (*get1ch)();	/* eat the ~ */
					return KEYMAP_F3;
				case '4':
					ch1 = (*get1ch)();	/* eat the ~ */
					return KEYMAP_F4;
				case '5':	/* RS/6000 PgUp is 150g, PgDn is 154g */
					ch = (*get1ch)();
					ch1 = (*get1ch)();	/* eat the ~ */
					if (ch == '0')
						return KEYMAP_PAGE_UP;
					if (ch == '4') 
						return KEYMAP_PAGE_DOWN;
			}
			return KEYMAP_HOME;

		case '4':		/* vt200 PgUp */
			ch = (*get1ch)();	/* eat the ~  */
			return KEYMAP_END;

		case '2':	/* xterm */
			ch = (*get1ch)();	/* eat the ~  */
		case 'L':
			return KEYMAP_INS;
		case 'M':
			return KEYMAP_F1;
		case 'N':
			return KEYMAP_F2;
		case 'O':
			return KEYMAP_F3;
		case 'P':
			return KEYMAP_F4;

		default:
			return KEYMAP_UNKNOWN;
	}
}

showtime()
{
	register char *ptr;
	time_t ct;

	ct = time(NULL);
	ptr = (char *)ctime(&ct);
	ptr[19] = 0;
	mvaddstr(0, COLS - strlen(ptr) - 1, ptr);
	refresh();
}

char *
userstate(uf)
	int uf;
{
	char *ptr;
	static char buf[10];

	if (uf & INCONF) {
		if (uf & TERMINATOR)  ptr = "Conf-T";
		else if (uf & KICKER) ptr = "Conf-K";
		else ptr = "Conf";
	} else if (uf & INTALK)	ptr = "Talk";
	else if (uf & INRECVF)	ptr = "Recv";
	else if (uf & INSENDF)	ptr = "Send";
	else if (uf & EXECEXT)	ptr = "Extern";
	else			ptr = "BBSing";
	return strcpy(buf, ptr);
}

showlist(first)
	int first;
{
	register i, l, y;
	static unsigned int gluk;

	move(Y_FIRST, X_FIRST);
	if (showmode) {
		clrtobot();
		if (usridx >= 0) getuserinfo(printw, usrlist[usridx]->name);
	} else {
		for (i = first, l = 0; i < nusers && l < pagesize; i++, l++) {
			y = l + Y_FIRST;
			move(y, X_FIRST);
			printw("%5d", i+1);
			move(y, X_CURSOR);
			if (i == usridx) addstr("->");
			else addstr("  ");
			move(y, X_NAME);
			printw("%-6.6s  %.15s  %-24.24s %-6d     %-6.6s",
			       usrlist[i]->tty,
			       ctime(&usrlist[i]->ltime) + 4,
			       usrlist[i]->name,
			       usrlist[i]->baud,
			       userstate(usrlist[i]->flags));
		}
		clrtobot();
		if (nusers <= 0) {
			switch(++gluk % 6) {
			case 0:
			case 1:	mvaddstr(Y_FIRST, COLS/2-strlen(copy)/2, copy);
				break;
			case 2:
			case 3: mvaddstr(Y_FIRST, COLS/2-strlen(copy)/2, copy);
				mvaddstr(Y_FIRST+1, COLS/2-strlen(right)/2, right);
				break;
			}
		}
	}
	if (nusers <= 0 || usridx < 0)
		mvaddstr(LINES-1, COLS/2 - 11, "No bbs users on-line");
	else {
		mvaddstr(LINES-1, COLS/2 - strlen(usrlist[usridx]->name)/2,
		      usrlist[usridx]->name);
		if (showmode) {
			move(LINES-1, COLS/2+12);
			printw("- %d -", usridx + 1);
		}
		mvaddstr(LINES-1, COLS-14, "h - Help");
	}
	refresh();
}

mainloop()
{
	int ch, sec;

home_list:
	showmode = usridx = first_line = scr_line = 0;

rescan_list:
	redraw = sec = 0;
	rescan();
	if (usridx >= nusers) showmode = usridx = first_line = scr_line = 0;
	move(Y_FIRST, X_FIRST);
	clrtobot();
	showlist(first_line);
	while (1) {
		if (!inqueue(0)) {
			showtime();
			if (redraw) goto rescan_list;
			if (nusers <= 0) showlist(0);
			else if (++sec > 5) goto rescan_list;
			continue;
		}
		if ((ch = getch()) == ERR) panic(1);
		switch(ch) {
			case ESC:
				switch(get_arrow_key(inputchar)) {
					case KEYMAP_UP:
						goto line_up;
					case KEYMAP_DOWN:
						goto line_down;
					case KEYMAP_LEFT:
					case KEYMAP_PAGE_UP:
						goto page_up;
					case KEYMAP_RIGHT:
					case KEYMAP_PAGE_DOWN:
						goto page_down;
					case KEYMAP_HOME:
						goto home_list;
					case KEYMAP_END:
						goto end_list;
					default:
						mvaddstr(LINES-1, COLS-14, "Bad command\007");
						refresh();
						continue;
				}
				break;

			case 'k':	/* line up */
			case ctrl('P'):
line_up:
				if (nusers <= 0 || usridx - 1 < 0) break; /* was continue */
				usridx--;
				if (--scr_line < 0 && first_line) {
					scr_line = 0;
					if (--first_line < 0) first_line = 0;
				}
				break;

			case 'j':
			case ctrl('N'):	/* line down */
line_down:
				if (nusers <= 0 || usridx + 1 >= nusers) break; /* was continue */
				usridx++;
				if (++scr_line >= pagesize) {
					scr_line--;
					first_line = usridx - scr_line;
				}
				break;

			case '\b':	/* page up */
			case 'b':
			case ctrl('U'):
			case ctrl('B'):
page_up:	
				if (nusers <= 0) break;
				if (scr_line > 0) usridx -= scr_line;
				else {
					usridx -= pagesize;
					first_line -= pagesize;
					if (usridx < 0 || first_line < 0)
						usridx = first_line = 0;
				}
				scr_line = 0;
				break;

			case ' ':	/* page down */
			case ctrl('D'):
			case ctrl('F'):
page_down:
				if (nusers <= 0) break;
				if (nusers <= pagesize)
					usridx = scr_line = nusers - 1;
				else {
					if (scr_line < pagesize - 1)
						usridx += pagesize - scr_line - 1;
					else {
						usridx += pagesize;
						first_line = usridx - pagesize + 1;
					}
					if (usridx >= nusers) {
						usridx = nusers - 1;
						first_line = nusers - pagesize;
					}
					scr_line = pagesize - 1;
				}
				break;

			case ctrl('A'):		/* home */
				goto home_list;

			case ctrl('E'):		/* end */
end_list:
				if (nusers <= 0) break;
				if (nusers <= pagesize)
					usridx = scr_line = nusers - 1;
				else {
					usridx = nusers - 1;
					first_line = nusers - pagesize;
					scr_line = pagesize - 1;
				}
				break;

			case '\r':
			case '\n':
				if (nusers <= 0 || usridx < 0) break;
				endwin();
				printf("%sWait for %s...\n",
				       DEFCOLOR, usrlist[usridx]->tty);
				if (grabtty(usrlist[usridx]) < 0) {
					printf("Press ENTER to continue");
					if (getchar() == EOF) onterm();
				}
				initterm();
				goto rescan_list;

			case '\t':
				if (nusers > 0) {
					showmode ^= 1;
					move(Y_FIRST, X_FIRST);
					clrtobot();
				}
				break;

			case 'd':
			case 0x7f:		/* delete user */
				if (nusers <= 0) break;
				move(LINES-1, 0);
				addstr("Terminate this user?\007");
				refresh();
				if ((ch = getch()) == ERR) panic(1);
				if (ch == 'y' || ch == 'Y' ||
				    ch == '\r' || ch == '\n')
					kill(usrlist[usridx]->pid, SIGTERM);
				break;

			case 'q':		/* quit */
				onterm();

			case ctrl('L'):		/* refresh screen */
				wrefresh(curscr);
				continue;

			case 'h':		/* help page */
				help_page();
				break;

			default:
				mvaddstr(LINES-1, COLS-14, "Bad command\007");
				refresh();
				continue;
		}
		showlist(first_line);
	}
}

help_page()
{
	move(Y_FIRST, X_FIRST);
	clrtobot();
	mvaddstr(Y_FIRST+1, COLS/2-strlen(copy)/2, copy);
	mvaddstr(Y_FIRST+2, COLS/2-strlen(right)/2, right);
	mvaddstr(Y_FIRST+4, COLS/2-5, "Help Page");
	move(Y_FIRST+6, X_FIRST);
	addstr("\
 k, ^P, UP          line Up             j, ^N, DOWN        line Down\n\
 b, BS, LEFT, PgUp  page Up             SPACE, RIGHT, PgDn page Down\n\
 ^A, HOME           home list           ^E, END            end list\n\
 ENTER              user assist         d, DEL             terminate user\n\
 TAB                toggle show mode    ^L                 refresh screen\n\
 q, ^C              quit\n");
	mvaddstr(LINES-1, COLS/2-7, "Press any key");
	refresh();
	if (getch() == ERR) panic(1);
	move(Y_FIRST, X_FIRST);
	clrtobot();
	return;
}
