/*
 *	Copyright (c) 1997 Rinet Corp., Novosibirsk, Russia
 *
 * Redistribution and use in source forms, with and without modification,
 * are permitted provided that this entire comment appears intact.
 *
 * THIS SOURCE CODE IS PROVIDED ``AS IS'' WITHOUT ANY WARRANTIES OF ANY KIND.
 */

#ifdef	HAVE_CONFIG_H
#include <config.h>
#endif

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/file.h>
#include <netinet/in.h>
#ifdef	HAVE_STROPTS_H
#include <stropts.h>
#else
#include <fcntl.h>
#endif
#include <signal.h>
#include <unistd.h>
#include <errno.h>
#include <syslog.h>

#include "udpserv.h"

extern int errno;
extern void (*error)();

u_long localhost;

static int sock = -1;
static void (*func)(u_long, char *, int);
static struct sockaddr_in from;
static void udprecv();
static void setsigio(void);


/*
 * Return udp port number (network bytes order) we are listen for or -1.
 */
int
udpserv(f)
	void (*f)();
{
	int s, on = 1;
	struct sockaddr_in sin;

	if ((s = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
		(*error)(LOG_ERR, "udpserv: socket: %s", strerror(errno));
		return -1;
	}
	if (setsockopt(s, SOL_SOCKET, SO_REUSEADDR, (void *)&on, sizeof(on)) < 0) {
		(*error)(LOG_ERR, "udpserv: setsockopt: %s", strerror(errno));
		return -1;
	}
	sin.sin_family = AF_INET;
	sin.sin_addr.s_addr = htonl(INADDR_ANY);
	sin.sin_port = 0;
	if (bind(s, (struct sockaddr *)&sin, sizeof(sin)) < 0) {
		(*error)(LOG_ERR, "udpserv: bind: %s", strerror(errno));
		return -1;
	}
	on = sizeof(sin);
	if (getsockname(s, (struct sockaddr *)&sin, &on) < 0) {
		(*error)(LOG_ERR, "udpserv: getsockname: %s", strerror(errno));
		return -1;
	}

	/* set async I/O notification on socket */
#ifdef	HAVE_STROPTS_H
	if (ioctl(s, I_SETSIG, S_INPUT) < 0) {
		(*error)(LOG_ERR, "udpserv: ioctl: %s", strerror(errno));
		return -1;
	}
#else
	if (fcntl(s, F_SETOWN, getpid()) < 0) {
		(*error)(LOG_ERR, "udpserv: fcntl: %s", strerror(errno));
		return -1;
	}
	if (fcntl(s, F_SETFL, FASYNC | O_NONBLOCK) < 0) {
		(*error)(LOG_ERR, "udpserv: fcntl: %s", strerror(errno));
		return -1;
	}
#endif
	/* set async I/O signal handler for udprecv() */
	setsigio();

	sock = s;
	func = f;
	localhost = inet_addr(LOOPBACK);

	return (int)sin.sin_port;
}

int
udpsend(port, data, len)
	int port;
	char *data;
	int len;
{
	int nb;
	struct sockaddr_in sin;

	if (sock < 0) return 0;

	sin.sin_family = AF_INET;
	sin.sin_addr.s_addr = localhost;
	sin.sin_port = port;

	if ((nb = sendto(sock, (char *)data, len, 0,
			 (struct sockaddr *)&sin, sizeof(sin))) < 0) {
		(*error)(LOG_ERR, "udpsend: sendto: %s", strerror(errno));
		return -1;
	}

	return nb;
}

int
udpreply(data, len)
	char *data;
	int len;
{
	int nb;

	if (sock < 0) return 0;

	if ((nb = sendto(sock, (char *)data, len, 0,
			 (struct sockaddr *)&from, sizeof(from))) < 0) {
		(*error)(LOG_ERR, "udpreply: sendto: %s", strerror(errno));
		return -1;
	}

	return nb;
}

/* SIGIO handler */
static void
udprecv()
{
	register nb = 0;
	int len = sizeof(from); 
	static char buf[512];

	while ((nb = recvfrom(sock, buf, sizeof(buf)-1, 0, 
			      (struct sockaddr *)&from, &len)) > 0) {
		buf[nb] = '\0';
	        (*func)(from.sin_addr.s_addr, buf, 0);
	}
	buf[0]='\0';
//	setsigio();

	(*func)(from.sin_addr.s_addr, buf, 1);
}

static void
setsigio()
{
    	struct sigaction act;

	(void) signal(SIGIO, udprecv);
	act.sa_handler = udprecv;
	sigemptyset(&act.sa_mask);
#ifdef	SA_NODEFER
	act.sa_flags = SA_RESTART | SA_NODEFER;
#else
	act.sa_flags = SA_RESTART;
#endif
	(void) sigaction(SIGIO, &act, NULL);
}
