/*
 *	Copyright (c) 1997 Rinet Corp., Novosibirsk, Russia
 *
 * Redistribution and use in source forms, with and without modification,
 * are permitted provided that this entire comment appears intact.
 *
 * THIS SOURCE CODE IS PROVIDED ``AS IS'' WITHOUT ANY WARRANTIES OF ANY KIND.
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <stdlib.h>
#include <string.h>

#include "compat.h"
#include "scandir.h"

static int
sortbyname(e1, e2)
	struct dir_ent *e1, *e2;
{
	return strcmp(e1->name, e2->name);
}

static int
sortbydate(e1, e2)
	struct dir_ent *e1, *e2;
{
	if (e1->date < e2->date) return 1;
	if (e1->date > e2->date) return -1;
	return 0;
}

static int
sortbysize(e1, e2)
	struct dir_ent *e1, *e2;
{
	if (e1->size < e2->size) return 1;
	if (e1->size > e2->size) return -1;
	return 0;
}

int
scan_dir(path, files, select, sort)
	char *path;
	struct dir_ent **files;
	int (*select) (char *, struct dir_ent *);
	int sort;
{
	DIR *dirp;
	struct dirent *dp;
	struct dir_ent file, *array;
	int arraysz, nitems;
	struct stat st;
	char *filename, filepath[1024];

	if ((dirp = opendir(path)) == NULL) return -1;

	/* first time allocation */
	arraysz = 100;
	if ((array = malloc(arraysz * sizeof(struct dir_ent))) == NULL)
		return -1;

	filename = &filepath[strlen(strcpy(filepath, path))-1];
	if (*filename != '/') *++filename = '/';
	filename++;

	nitems = 0;
	while ((dp = readdir(dirp)) != NULL) {

		/* skip files with leading dot */
		if (*dp->d_name == '.' || (unsigned char)*dp->d_name < 0x20)
			continue;

		/* obtain a file status (by full pathname) */
		(void)strcpy(filename, dp->d_name);
		if (stat(filepath, &st) < 0) continue;
		file.mode = st.st_mode;		/* file mode */
		file.date = st.st_mtime;	/* last modification */
		file.size = st.st_size;		/* file size */
		file.uid = st.st_uid;		/* user-id of file owner */
		file.gid = st.st_gid;		/* group-id of file owner */
		file.priv = 0;	/* init value */
		file.key = 0;	/* init value */

		/* make temporary file name entry just for select */
		file.name = filename;
		file.len = strlen(filename);
		if (select != NULL && !(*select)(filepath, &file))
			continue;	/* just selected files */

		/* make actually file name entry */
		if ((file.name = (char *)malloc(file.len+1)) == NULL)
			return -1;
		(void)strcpy(file.name, filename);

		/* make sure the array has space left */
		if (++nitems >= arraysz) {
			arraysz += 50;
			array = realloc(array, arraysz * sizeof(struct dir_ent));
			if (array == NULL) return -1;
		}
		array[nitems-1] = file;
	}
	closedir(dirp);

	if (nitems > 0) {
		switch (sort) {
		case SCAN_SORTBYNAME:
			qsort(array, nitems, sizeof(struct dir_ent),
			      sortbyname);
			break;
		case SCAN_SORTBYDATE:
			qsort(array, nitems, sizeof(struct dir_ent),
			      sortbydate);
			break;
		case SCAN_SORTBYSIZE:
			qsort(array, nitems, sizeof(struct dir_ent),
			      sortbysize);
			break;
		}
		*files = array;
	} else {
		free(array);
		*files = NULL;
	}

	return nitems;
}

/*
 * Free memory allocated by scandir()
 */
void
free_dir(files, num)
	struct dir_ent **files;
	int num;
{
	if (*files != NULL) {
		register i;
		register struct dir_ent *f = *files;
		for (i = 0; i < num; i++) free(f[i].name);
		free(f);
		*files = NULL;
	}
}

int
files_only(path, file)
	char *path;
	register struct dir_ent *file;
{
	return ((file->mode & S_IFMT) == S_IFREG);
}

int
dirs_only(path, file)
	char *path;
	register struct dir_ent *file;
{
	return ((file->mode & S_IFMT) == S_IFDIR);
}
