/*
 *	Copyright (c) 1995 The CAD lab of the
 *	Sibirian State Academy of Telecommunication, RU
 *
 *	TNSDrive $Id$
 *
 *	$Log$
 *
 * Redistribution and use in source forms, with and without modification,
 * are permitted provided that this entire comment appears intact.
 *
 * THIS SOURCE CODE IS PROVIDED ``AS IS'' WITHOUT ANY WARRANTIES OF ANY KIND.
 */

#ifdef	HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <string.h>
#include "group.h"
#include "rematch.h"

struct group_keyword group_keywords[] = {
	{ "GROUP",	GSID_GROUP },
	{ "INCLUDE",	GSID_INCLUDE },
	{ "MENU",	GSID_MENU },
	{ "FILE",	GSID_FILE },
	{ "USENET",	GSID_USENET },
	{ "MESSAGE",	GSID_MESSAGE },
	{ "EMAIL",	GSID_EMAIL },
	{ "FTP",	GSID_FTP },
	{ NULL,		GSID_NONE },
};

grp_ent grp_first = NULL;
grp_ent grp_top = NULL;

static int cfline;

static int
getkeyword(word)
	char *word;
{
	register struct group_keyword *p;

	for (p = group_keywords; p->word; ++p)
		if (!strncasecmp(word, p->word, strlen(p->word)))
			return p->key;
	return GSID_NONE;
}

static int
gflags(s)
	char *s;
{
	int flags;

	for (flags = 0; *s; s++)
		switch(*s) {
		case 'S': flags |= GRF_S; break;
		case 'R': flags |= GRF_R; break;
		case 'W': flags |= GRF_W; break;
		case 'D': flags |= GRF_D; break;
		default:
			fprintf(stderr, "Error (line %d): unknown flag '%c'\n", cfline, *s);
			return -1;
		}
	return flags;
}

/* Initialize groups. Return -1 on error, 0 no groups or 1 if sucess. */
int
initgrpbase(file)
	char *file;
{
	int gsid, n, argused, include;
	char *ptr, *line, *word;
	grp_ent curgrp, tmpgrp;
	grp_desc curdesc;

	if (file != NULL) {
		if ((ptr = (char *)loadfile(file)) == NULL) return 0;
	} else {
		if ((ptr = (char *)loadfile(MAINGROUPCONF)) == NULL &&
		    (ptr = (char *)loadfile(GROUPCONF)) == NULL) return 0;
	}

	for (gsid = GSID_NONE, cfline = 0, argused = 1;
	     (line = strchr(ptr, '\n')) != NULL; ptr = line) {
		*line++ = '\0';
		cfline++;
next_args:
		while ((unsigned char)*ptr <= 0x20 && *ptr) ptr++;
		if (!*ptr || *ptr == '#') continue;
		if (*ptr == '%') {
			if (!argused) {
				fprintf(stderr, "Error (line %d): argument expected\n", cfline);
				return -1;
			}
			argused = 0;
			word = ++ptr;
			while ((unsigned char)*ptr > 0x20) ptr++;
			if ((gsid = getkeyword(word)) == GSID_NONE) {
				fprintf(stderr, "Error (line %d): unknown keyword\n", cfline);
				return -1;
			}
			goto next_args;
		}
		argused++;
		for (word = line - 2; *word && (unsigned char)*word <= 0x20; --word);
		word[1] = '\0';
		switch(gsid) {
		case GSID_GROUP:
			if (grp_first == NULL) {
				grp_first = (grp_ent)calloc(1, sizeof(struct grp_ent__));
				curgrp = grp_first;
			} else	{
				for (tmpgrp = grp_first; tmpgrp; tmpgrp = tmpgrp->next)
					if (!strcasecmp(ptr, tmpgrp->name)) break;
				if (tmpgrp != NULL) {
					fprintf(stderr, "Error (line %d): group \"%s\" already exist\n", cfline, ptr);
					return -1;
				}
				curgrp->next = (grp_ent)calloc(1, sizeof(struct grp_ent__));
				curgrp = curgrp->next;
			}
			if (curgrp == NULL) {
				fprintf(stderr, "Can't allocate memory\n");
				return -1;
			}
			curgrp->name = strdup(ptr);
			include = 0;
			break;
		case GSID_INCLUDE:
			if (grp_first == NULL) {
				fprintf(stderr, "Error (line %d): group declaration expected\n", cfline);
				return -1;
			}
			if (include) {
				fprintf(stderr, "Error (line %d): include statement duplicated\n", cfline);
				return -1;
			}
			if (!strcasecmp(ptr, curgrp->name)) {
				fprintf(stderr, "Error (line %d): illegal recursion\n", cfline);
				return -1;
			}
			for (tmpgrp = grp_first; tmpgrp; tmpgrp = tmpgrp->next)
				if (!strcasecmp(ptr, tmpgrp->name)) break;
			if (tmpgrp == NULL) {
				fprintf(stderr, "Error (line %d): group \"%s\" not found\n", cfline, ptr);
				return -1;
			}
			if (curgrp->entry == NULL)
				curgrp->entry = tmpgrp->entry;
			else 	curdesc->next = tmpgrp->entry;
			include++;
			break;
		case GSID_MENU:
		case GSID_FILE:
		case GSID_USENET:
		case GSID_MESSAGE:
		case GSID_EMAIL:
		case GSID_FTP:
			if (grp_first == NULL) {
				fprintf(stderr, "Error (line %d): group declaration expected\n", cfline);
				return -1;
			}
			if (include) {
				fprintf(stderr, "Error (line %d): include must be last statement in group\n", cfline);
				return -1;
			}
			if (curgrp->entry == NULL) {
				curgrp->entry = (grp_desc)calloc(1, sizeof(struct grp_desc__));
				curdesc = curgrp->entry;
			} else	{
				curdesc->next = (grp_desc)calloc(1, sizeof(struct grp_desc__));
				curdesc = curdesc->next;
			}
			if (curdesc == NULL) {
				fprintf(stderr, "Can't allocate memory\n");
				return -1;
			}
			ptr = strtok(ptr, " \t");
			if ((curdesc->rgc = recomp(ptr)) == NULL) {
				fprintf(stderr, "Error (line %d): bad regular expression \"%s\"\n", cfline, ptr);
				return -1;
			}
			curdesc->expr = strdup(ptr);
			curdesc->grf = GRF_DEFAULT;
			curdesc->gsid = gsid;
			for (n = 0, ptr = NULL; (ptr = strtok(ptr, " \t")) != NULL; ptr = NULL) {
				if (*ptr == '-' || *ptr == '+')
					n = gflags(&ptr[1]);
				else 	n = gflags(ptr);
				if (n < 0) return -1;
				if (!n) n = GRF_S | GRF_R | GRF_W | GRF_D;
				switch(*ptr) {
					case '-': curdesc->grf &= ~n; break;
					case '+': curdesc->grf |= n; break;
					default:  curdesc->grf = n;
				}
			}
			if (!n) {
				fprintf(stderr, "Error (line %d): argument expected\n", cfline);
				return -1;
			}
			break;
		default:
			fprintf(stderr, "Error (line %d): syntax error\n", cfline);
			return -1;
		}
	}
	if (!argused) {
		fprintf(stderr, "Error (line %d): argument expected\n", cfline);
		return -1;
	}
	grp_top = grp_first;
	return grp_first != NULL;
}

#define	MAX_USER_GROUPS		100

static grp_ent groups[MAX_USER_GROUPS];
static int ngroups = 0;

/* Setup group name of current user */
int
setgrpname(name)
	char *name;
{
	grp_ent tmpgrp;
	char *p, buf[1024];

	ngroups = 0;
	if (grp_top == NULL) return 0;
	for (p = strcpy(buf, name); (p = strtok(p, " \t,;")) != NULL; p = NULL) {
		for (tmpgrp = grp_top; tmpgrp; tmpgrp = tmpgrp->next)
			if (!strcasecmp(p, tmpgrp->name)) break;
		if (tmpgrp) {
			if (ngroups < MAX_USER_GROUPS)
				groups[ngroups++] = tmpgrp;
			else	break;
		}
	}
	return ngroups;
}

int
getgrpright(gsid, s)
	int gsid;
	char *s;
{
	register i;
	grp_desc grpdesc;

	for (i = 0; i < ngroups; i++) {
		grp_first = groups[i];
		for (grpdesc = grp_first->entry; grpdesc; grpdesc = grpdesc->next)
			if (grpdesc->gsid == gsid && reexec(grpdesc->rgc, s))
				return grpdesc->grf;
	}
	return GRF_DEFAULT;
}

int
isgrpright(gsid, s, grf)
	int gsid, grf;
	char *s;
{
	register i;
	grp_desc grpdesc;

	for (i = 0; i < ngroups; i++) {
		grp_first = groups[i];
		for (grpdesc = grp_first->entry; grpdesc; grpdesc = grpdesc->next)
			if (grpdesc->gsid == gsid && reexec(grpdesc->rgc, s))
				return (grpdesc->grf & grf) == grf;
	}
	return (GRF_DEFAULT & grf) == grf;
}
