/*
 *	Copyright (c) 1997 Rinet Corp., Novosibirsk, Russia
 *
 * Redistribution and use in source forms, with and without modification,
 * are permitted provided that this entire comment appears intact.
 *
 * THIS SOURCE CODE IS PROVIDED ``AS IS'' WITHOUT ANY WARRANTIES OF ANY KIND.
 */

/* Internal Variables */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>

#include "drive.h"
#include "loginfo.h"
#include "sysmsg.h"
#include "usenet.h"
#include "ftp.h"
#include "scandir.h"
#include "variables.h"
#include "version.h"

extern struct loginfo process;

static char makestrbuf[1024];

char *
makestrempty()
{
	makestrbuf[0] = '\0';
	return makestrbuf;
}

char *
makeshowpath(path)
	char *path;
{
	int i, j, l;
	register char *ptr, *ptr2;
	char buf[1024];
	char *makestrpath();
	extern char *hidedirs[];

	makestrbuf[0] = '\0';
	for (i = j = 0; hidedirs[i] != NULL; i++) {
		l = strlen(hidedirs[i]);
		if (!strncmp(hidedirs[i], path, l) && l > j) j = l;
	}
	(void)strcpy(buf, &path[j]);
	for (ptr = buf; *ptr != '\0'; ptr++) if (*ptr == '_') *ptr = ' ';
	ptr2 = buf;
	while ((ptr = strtok(ptr2, "/")) != NULL) {
		if ((ptr2 = strchr(ptr, '.')) != NULL) i = ptr2-ptr;
		else i = strlen(ptr);
		if (i) {
			strcat(makestrbuf, "<");
			strncat(makestrbuf, ptr, i);
			strcat(makestrbuf, ">-");
		}
		ptr2 = NULL;
	}
	i = strlen(makestrbuf);
	if (i && makestrbuf[i-1] == '-') makestrbuf[--i] = '\0';
	if (i) return makestrbuf;
	return makestrpath();
}

char *
makestrpath()
{
	if (!strcmp(dirpath, ".")) return makestrempty();
	return makeshowpath(dirpath);
}

char *
makestrmenudir(value)
	char *value;
{
	if (value) {
		if (!strcmp(value, HOMEDIR)) value = homedir;
		if (isaccess(value)) strcpy(dirpath, value);
	}
	return dirpath;
}

char *
makestrfpath()
{
	char *ptr = getarea();
	if (*ptr) return makeshowpath(ptr);
	return sysmsg(MSG_NONE);
}

char *
makestrfdir(value)
	char *value;
{
	if (value) {
		if (!strcmp(value, HOMEDIR)) value = homedir;
		if (isaccess(value)) return setarea(value);
	}
	return getarea();
}

char *
makestrrfdir(value)
	char *value;
{
	extern char readfilesdir[];
	if (value) {
		if (!strcmp(value, HOMEDIR)) value = homedir;
		if (isaccess(value)) strcpy(readfilesdir, value);
	}
	return readfilesdir;
}

char *
makestrulpath()
{
	extern char *uploadpath, *defuploadpath;
	if (uploadpath != NULL) return makeshowpath(uploadpath);
	return makeshowpath(defuploadpath);
}

char *
makestruldir(value)
	char *value;
{
	extern char *uploadpath, *defuploadpath;
	if (value) {
		if (!strcmp(value, HOMEDIR)) value = homedir;
		if (isaccess(value)) {
			if (uploadpath) free(uploadpath);
			uploadpath = strdup(value);
		}
	}
	if (uploadpath != NULL) return uploadpath;
	return defuploadpath;
}

char *
makestrmpath()
{
	extern char folder_path[];
	if (folder_path[0] == '\0') return (sysmsg(MSG_NONE));
	return makeshowpath(folder_path);
}

char *
makestrmdir(value)
	char *value;
{
	extern char folder_path[];

	if (value) {
		if (*value == '$') value++;
		(void) change_folder(value);
	}
	return folder_path;
}

char *
makestrmsgn()
{
	extern char folder_path[];
	return itoa(getmsgnum(folder_path));
}

char *
makestrupath()
{
	extern int curgrpidx, *my_group;
	extern struct group_ent *active;
	if (curgrpidx == -1) return (sysmsg(MSG_NONE));
	return active[my_group[curgrpidx]].name;
}

char *
makestrartn()
{
	extern int top_base;
	return itoa(top_base);
}

char *
makestrkey()
{
	int i, j;
	extern struct dir_ent *menu_item;
	extern int menu_num;

	if (menu_item == NULL) return makestrempty();

	for (i = j = 0; i < menu_num; i++) {
		if (menu_item[i].key) {
			makestrbuf[j++] = (char)menu_item[i].key;
			makestrbuf[j++] = ',';
		}
	}
	makestrbuf[j++] = '?';
	makestrbuf[j++] = ',';
	makestrbuf[j++] = '.';
	makestrbuf[j] = 0;
	return makestrbuf;
}

char *
makestrctime()
{
	time_t tmnow;
	char *ptr;

	tmnow = time(NULL);
	ptr = ctime(&tmnow);
	ptr[24] = 0;
	return ptr;
}

char *
makestrlastcall()
{
	char *ptr;
	ptr = ctime(&lastcalltime);
	ptr[24] = 0;
	return ptr;
}

char *
makestrotime()
{
	return sec2str(time(NULL) - process.ltime);
}

char *
makestrltime(value)
	char *value;
{
	int tl;

	if (value && (tl = atoi(value)) > 2) settimer(tl);
	return (sec2str(gettimer()));
}

char *
makestrstime()
{
	extern int maxsessiontime;
	return sec2str(maxsessiontime * 60);
}

char *
makestruser()
{
	return process.name;
}

char *
makestrgroup(value)
	char *value;
{
	if (value && setgrpname(value)) {
		if (usergroup) free(usergroup);
		usergroup = strdup(value);
	}
	return usergroup;
}

char *
makestrname()
{
	return getuserconf(REALNAME, DEFREALNAME);
}

char *
makestremail()
{
	extern char *fromfield();
	return fromfield(0);
}

char *
makestrcallsign(value)
	char *value;
{
	if (value) {
		if (*value == '.' || *value == '#' ||
		    *value == ' ' || *value == '\t') callsign[0] = '\0';
		else {
			(void)strncpy(callsign, value, LOGINFO_NAMELEN);
			callsign[LOGINFO_NAMELEN-1] = '\0';
		}
	}
	return callsign;
}

char *
makestrsign()
{
	return getuserconf(SIGNATURE, sysmsg(MSG_UNKNOWN));
}

char *
makestrcity()
{
	return getuserconf(USERLOCATION, sysmsg(MSG_UNKNOWN));
}

char *
makestrphone()
{
	return getuserconf(PHONE, sysmsg(MSG_UNKNOWN));
}

char *
makestrpasswd()
{
	return getuserconf(PASSWORD, sysmsg(MSG_UNKNOWN));
}

char *
makestrorg()
{
	return getuserconf(USERORG, DEFORG);
}

char *
makestrscrlines()
{
	return itoa(scrlines+1);
}

char *
makestrlevel(value)
	char *value;
{
	extern int privlevel;

	if (value) privlevel = atoi(value);
	return itoa(privlevel);
}

char *
makestrproto()
{
	return getuserconf(PROTOCOL, sysmsg(MSG_NONE));
}

char *
makestrlang()
{
	extern char *langname;
	return langname;
}

char *
makestrlnum()
{
	extern int langnumber;
	return itoa(langnumber);
}

char *
makestrcpage()
{
	extern int codepage;
	return itoa(codepage);
}

char *
makestrdllimit(value)
	char *value;
{
	extern int dailydllimit;

	if (value) dailydllimit = atoi(value);
	sprintf(makestrbuf, "%dkb", dailydllimit/1024);
	return makestrbuf;
}

char *
makestrdailydl()
{
	extern int dailydlsize;
	sprintf(makestrbuf, "%dkb", dailydlsize/1024);
	return makestrbuf;
}

char *
makestrdailyul()
{
	extern int dailyulsize;
	sprintf(makestrbuf, "%dkb", dailyulsize/1024);
	return makestrbuf;
}

char *
makestrdlall()
{
	extern int userdlsize;
	sprintf(makestrbuf, "%dkb", userdlsize/1024);
	return makestrbuf;
}

char *
makestrulall()
{
	extern int userulsize;
	sprintf(makestrbuf, "%dkb", userulsize/1024);
	return makestrbuf;
}

char *
makestrsratio(value)
	char *value;
{
	extern int dlulratio;
	if (value) dlulratio = atoi(value);
	return itoa(dlulratio);
}

char *
makestruratio()
{
	extern int userdlsize, userulsize;
	return itoa(userdlsize/userulsize);
}

char *
makestrulmulti(value)
	char *value;
{
	extern int ulmultiplier;
	if (value) ulmultiplier = atoi(value);
	return itoa(ulmultiplier);
}

char *
makestrncalls()
{
	extern int ncalls;
	return itoa(ncalls);
}

char *
makestrsysname()
{
	extern char *systemname;
	return systemname;
}

char *
makestrsyshost()
{
	extern char *systemhost;
	return systemhost;
}

char *
makestrsysmail()
{
	return sysmailto;
}

char *
makestrsysadm()
{
	return sysadmname;
}

char *
makestrstatus()
{
	if (process.flags & CHARGEUSER) strcpy(makestrbuf, "Charge");
	else strcpy(makestrbuf, "Regular");
	if (process.flags & INFINITYUSER) strcat(makestrbuf, ",Infinity");
	if (process.flags & TERMINATOR) strcat(makestrbuf, ",Terminator");
	else if (process.flags & KICKER) strcat(makestrbuf, ",Kicker");
	return makestrbuf;
}

char *
makestrstate()
{
	static char state[256];
	extern char inrunning[];
	extern char *xferproto;

	if (process.flags & EXECEXT) return inrunning;
	if (process.flags & INCONF) return strcpy(state, "Conference");
	if (process.flags & INTALK) return strcpy(state, "Talk");
	if (process.flags & INRECVF) {
		snprintf(state, sizeof(state), "Recv %s %s",
			xferproto ? xferproto : "",
			TransferFile);
		return state;
	}
	if (process.flags & INSENDF) {
		snprintf(state, sizeof(state), "Send %s %s",
			xferproto ? xferproto : "",
			TransferFile);
		return state;
	}
	return strcpy(makestrbuf, "BBSing");
}

char *
makestrnewssort()
{
	if (newssort != NULL) strcpy(makestrbuf, newssort);
	else strcpy(makestrbuf, sysmsg(MSG_NONE));
	return makestrbuf;
}

char *
makestrlartread()
{
	extern int ArtReadLimit;
	return itoa(ArtReadLimit);
}

char *
makestrlmsgread()
{
	extern int MsgReadLimit;
	return itoa(MsgReadLimit);
}

char *
makestrlartsize()
{
	extern int ArtSizeLimit;
	sprintf(makestrbuf, "%dkb", ArtSizeLimit/1024);
	return makestrbuf;
}

char *
makestrlartpost()
{
	extern int ArtPostLimit;
	return itoa(ArtPostLimit);
}

char *
makestrlmsgpost()
{
	extern int MsgPostLimit;
	return itoa(MsgPostLimit);
}

char *
makestruartread()
{
	extern int UserArtRead;
	return itoa(UserArtRead);
}

char *
makestrumsgread()
{
	extern int UserMsgRead;
	return itoa(UserMsgRead);
}

char *
makestruartsize()
{
	extern int UserArtSize;
	sprintf(makestrbuf, "%dkb", UserArtSize/1024);
	return makestrbuf;
}

char *
makestruartpost()
{
	extern int UserArtPost;
	return itoa(UserArtPost);
}

char *
makestrumsgpost()
{
	extern int UserMsgPost;
	return itoa(UserMsgPost);
}

char *
makestrdotname()
{
	register char *p;
	for (p = strcpy(makestrbuf, process.name); *p; p++)
		if (*p == ' ' || *p == '\t') *p = '.';
	return makestrbuf;
}

char *
makestrtermtype()
{
	extern char termtype[];
	return termtype;
}

char *
makestrmypid()
{
	return itoa(process.pid);
}

char *
makestrmytty()
{
	return process.tty;
}

char *
makestrtimerate(value)
	char *value;
{
	extern int timerate;
	if (value) timerate = atoi(value);
	return itoa(timerate);
}

char *
makestrintrrate(value)
	char *value;
{
	int rate;
	extern int intrrate;

	if (value && (rate = atoi(value)) > 0) {
		intrrate = rate;
		settimer(gettimer()/60);
	}
	return itoa(intrrate);
}

char *
makestrhotkey()
{
	if (termflags & HOTKEYON) strcpy(makestrbuf, "Yes");
	else strcpy(makestrbuf, "No");
	return makestrbuf;
}

char *
makestrshortmenu()
{
	if (termflags & SHORTMENUON) strcpy(makestrbuf, "Yes");
	else strcpy(makestrbuf, "No");
	return makestrbuf;
}

char *
makestrmesg(value)
	char *value;
{
	if (!value) value = " ";
	return strcpy(makestrbuf, mesgtty(value));
}

char *
makestrhomedir()
{
	return homedir;
}

char *
makestrorgdir()
{
	return orgdir;
}

char*
makestrconnect(value)
	char *value;
{
	int sp;

	if (value && (sp = atoi(value)) >= 1200) {
		process.baud = sp;
		saveinfo(&process);
	}
	return itoa(process.baud);
}

char *
makedisplay(value)
	char *value;
{
	static char nostr[] = "";
	if (value) display(0, value);
	return nostr;
}

char *
makekicker(value)
	char *value;
{
	if (value) {
		if (*value == 'T') process.flags |= (TERMINATOR|INFINITYCONF);
		else if (*value == 'K') process.flags |= KICKER;
		else process.flags &= ~(TERMINATOR|KICKER);
	}
	if (process.flags & TERMINATOR) strcpy(makestrbuf, "*terminator*");
	else if (process.flags & KICKER) strcpy(makestrbuf, "*kicker*");
	else strcpy(makestrbuf, "*nobody*");
	return makestrbuf;
}

char *
makefingerprog(value)
	char *value;
{
	if (value) {
		if (fingerprog != NULL) free(fingerprog);
		fingerprog = strdup(value);
	}
	if (fingerprog != NULL) return fingerprog;
	return makestrempty();
}

char *
maketalkprog(value)
	char *value;
{
	if (value) {
		if (talkprog != NULL) free(talkprog);
		talkprog = strdup(value);
	}
	if (talkprog != NULL) return talkprog;
	return makestrempty();
}

char *
makestrftpserv()
{
	if (FTPserver != NULL) return FTPserver;
	return sysmsg(MSG_NONE);
}

char *
makestrftppwd()
{
	return FTPpwd;
}

char *
makestrID()
{
	return package;
}

char *
makestrVER()
{
	return version;
}

char *
makestrOS()
{
	return target;
}

char *
makestrCC()
{
	return compiled;
}

char *
makestrline(value)
	char *value;
{
	if (value) {
		if (loginline != NULL) free(loginline);
		loginline = strdup(value);
	}
	if (loginline != NULL) return loginline;
	return makestrempty();
}

char *
makestraddr()
{
	if (remote_addr != NULL) return remote_addr;
	return sysmsg(MSG_NONE);
}

char *
makestrhost()
{
	if (remote_host != NULL) return remote_host;
	return sysmsg(MSG_NONE);
}

static struct dispfunc {
	char *(*f)(void *);
	char *key;
} dispfuncs[] = {
	{ makestrctime,	   SYSTEMTIME	},	/* Current system time	*/
	{ makestrotime,	   ONLINETIME	},	/* On-line time		*/
	{ makestrfpath,	   FILEAREAPATH	},	/* FileArea path	*/
	{ makestrfdir,	   FILEAREADIR	},	/* FileArea directory	*/
	{ makestrulpath,   UPLOADPATH	},	/* Upload Area path	*/
	{ makestruldir,	   UPLOADDIR	},	/* Upload directory	*/
	{ makestrmpath,    MSGAREAPATH	},	/* MsgArea path		*/
	{ makestrmdir,	   MSGAREADIR	},	/* MsgArea directory	*/
	{ makestrmsgn,	   MSGNUMMSG	},	/* Number of messages	*/
	{ makestrupath,    USENETGROUP	},	/* Usenet group name	*/
	{ makestrkey,	   SHORTKEYS	},	/* Menu short keys	*/
	{ makestrlevel,	   USERPRIVLEVEL},	/* User Privlevel	*/
	{ makestrpath,	   MENUPATH	},	/* Menu path		*/
	{ makestrmenudir,  MENUDIR	},	/* Menu directory	*/
	{ makestrltime,	   USERTIMELEFT	},	/* Time left		*/
	{ makestrstime,	   SESSIONTIME	},	/* Max session time	*/
	{ makestruser,	   USER		},	/* User name		*/
	{ makestrgroup,	   USERGROUP	},	/* Group of user	*/
	{ makestrname,	   REALNAME	},	/* User real name	*/
	{ makestremail,	   EMAIL	},	/* E-mail of user	*/
	{ makestrcallsign, CALLSIGN	},	/* User call signature	*/
	{ makestrsign,	   SIGNATURE	},	/* User signature	*/
	{ makestrcity,	   USERLOCATION	},	/* User Sity/State	*/
	{ makestrpasswd,   PASSWORD	},	/* Encrypted password	*/
	{ makestrorg,	   USERORG	},	/* User Organization	*/
	{ makestrproto,	   PROTOCOL	},	/* Default protocol	*/
	{ makestrlang,	   LANGUAGE	},	/* Language		*/
	{ makestrlnum,	   LANGNUMBER	},	/* Language number	*/
	{ makestrcpage,	   CODEPAGE	},	/* CodePage number	*/
	{ makestrdailydl,  DAILYDLSIZE	},	/* User daily Download	*/
	{ makestrdailyul,  DAILYULSIZE	},	/* User daily Upload	*/
	{ makestrdlall,	   USERDLSIZE	},	/* User Download	*/
	{ makestrulall,	   USERULSIZE	},	/* User Upload	*/
	{ makestrdllimit,  DAILYDLLIMIT	},	/* Daily Download limit	*/
	{ makestrsratio,   SYSDLULRATIO	},	/* System DL/UL ratio	*/
	{ makestruratio,   USERDLULRATIO},	/* User DL/UL ratio	*/
	{ makestrulmulti,  ULMULTIPLIER	},	/* UL multiplier	*/
	{ makestrncalls,   CALLNUMBER	},	/* Number of calls	*/
	{ makestrscrlines, SCRLINES	},	/* Lines on screen	*/
	{ makestrsysname,  SYSTEMNAME	},	/* Name of this system	*/
	{ makestrsyshost,  MYHOSTNAME	},	/* Host of this system	*/
	{ makestrsysmail,  SYSMAILTO	},	/* Name of this system	*/
	{ makestrsysadm,   SYSADMNAME	},	/* Name of this system	*/
	{ makestrstatus,   USERSTATUS	},	/* User status		*/
	{ makestrstate,	   USERSTATE	},	/* User state		*/
	{ makestrlastcall, LASTCALLTIME	},	/* User Last Call Time	*/
	{ makestrtermtype, TERM		},	/* User terminal type	*/
	{ makestrphone,	   PHONE	},	/* User phone number	*/
	{ makestrnewssort, NEWSSORT	},	/* NewsSort string	*/
	{ makestrlartread, ARTREADLIMIT	},	/* ArtRead limit	*/
	{ makestrlartsize, ARTSIZELIMIT	},	/* ArtSize limit	*/
	{ makestrlartpost, ARTPOSTLIMIT	},	/* ArtPost limit	*/
	{ makestrlmsgread, MSGREADLIMIT	},	/* MsgRead limit	*/
	{ makestrlmsgpost, MSGPOSTLIMIT	},	/* MsgPost limit	*/
	{ makestruartread, USERARTREAD	},	/* How many Art readed	*/
	{ makestruartsize, USERARTSIZE	},	/* How much Art posted	*/
	{ makestruartpost, USERARTPOST	},	/* How many Art posted	*/
	{ makestrumsgread, USERMSGREAD	},	/* How many Msg readed	*/
	{ makestrumsgpost, USERMSGPOST	},	/* How many Msg posted	*/
	{ makestrdotname,  DOTTEDNAME	},	/* DottedName John.Doe	*/
	{ makestrmypid,	   PID		},	/* My PID		*/
	{ makestrmytty,	   TTY		},	/* My tty		*/
	{ makestrtimerate, TIMERATE	},	/* timerate divisor	*/
	{ makestrintrrate, INTERRUPT	},	/* interrupt rate	*/
	{ makestrhotkey,   HOTKEY	},	/* hotkey yes/no	*/
	{ makestrshortmenu,SHORTMENU	},	/* shortmenu yes/no	*/
	{ makestrmesg,	   MESG		},	/* mesg status yes/no	*/
	{ makestrhomedir,  HOMEDIR	},	/* bbs user home dir	*/
	{ makestrorgdir,   ORGDIR	},	/* bbs system org dir	*/
	{ makestrconnect,  CONNECT	},	/* connect speed	*/
	{ makedisplay,	   DISPLAY	},	/* exec display() func	*/
	{ makekicker,	   USERKICKER	},	/* set/unset kicker	*/
	{ makefingerprog,  FINGER	},	/* set/show finger prgm	*/
	{ maketalkprog,	   TALK		},	/* set/show talk prgm	*/
	{ makestrftpserv,  FTPSERVER	},	/* connected FTP server	*/
	{ makestrftppwd,   FTPPWD	},	/* current work dir	*/
	{ makestrrfdir,	   READFILESDIR	},	/* readfiles dir name	*/
	{ makestrline,	   LINE		},	/* login line description */
	{ makestraddr,	   REMOTEADDR	},	/* remote address */
	{ makestrhost,	   REMOTEHOST	},	/* remote host */

	{ makestrID,	   PACKAGE	},
	{ makestrVER,	   VERSION	},
	{ makestrOS,	   TARGET	},
	{ makestrCC,	   COMPILED	},

	{ 0, 0 },
};

char *
(*do_disp(key))()
	char *key;
{
	struct dispfunc *p;
	extern int errno;

	for (p = dispfuncs; p->f; ++p)
		if (!strcasecmp(key, p->key)) return p->f;

	LOGIT(LOG_ERR, "do_disp: \"%s\": Unknown variable", key);
	return makestrempty;
}

char *
getvariables(var)
	char *var;
{
	char *ptr;
	static char buf[1024];

	for (buf[0] = 0; (var = strtok(var, " \t")) != NULL; var = NULL) {
		ptr = (*do_disp(var))(NULL);
		if (*ptr == '\0') ptr = "-";
		if (buf[0]) strcat(buf, "\t");
		strcat(buf, ptr);
	}
	if (buf[0] == '\0') strcat(buf, "-");
	return buf;
}

void
insvariables(d)
	char *d;
{
	register i;
	register char *s, *p;
	char var[80], buf[1024];

	s = strcpy(buf, d);
	while (*s) {
		if (*s == '$') {
			for (i = 0, p = s + sizeof(char); i < sizeof(var)-1 &&
			     isupper(*p); p++, i++) var[i] = *p;
			var[i] = '\0';
			if (i) {
				s = p;
				p = (*do_disp(var))(NULL);
				strcpy(d, p);
				d += strlen(p);
				continue;
			}
		}
		*d++ = *s++;
	}
	*d = '\0';
}

void
setvariables(line)
	char *line;
{
	register char *p;
	char key[1024];

	for (p = line; (p = strtok(p, " \t")) != NULL; p = NULL) {
		strncpy(key, p, sizeof(key)-1);
		key[sizeof(key)-1] = '\0';
		if ((p = strchr(key, '=')) == NULL) continue;
		*p++ = '\0';
		insvariables(p);
		(void)(*do_disp(key))(p);
	}
}
