/*
 *	Copyright (c) 1997 Rinet Corp., Novosibirsk, Russia
 *
 * Redistribution and use in source forms, with and without modification,
 * are permitted provided that this entire comment appears intact.
 *
 * THIS SOURCE CODE IS PROVIDED ``AS IS'' WITHOUT ANY WARRANTIES OF ANY KIND.
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <errno.h>

#include "drive.h"
#include "callsign.h"
#include "scandir.h"
#include "loginfo.h"
#include "getconf.h"
#include "variables.h"
#ifndef	EXTERN_APPS
#include "sysmsg.h"
#endif

extern int errno;
extern struct loginfo process;

/*
 * Display list of users currently on-line.
 */
void
who_is_there()
{
	int lncnt;
	char *ptr;
	struct loginfo *info;

#ifndef	EXTERN_APPS
	LOGIT(LOG_INFO, "Who on BBS");
#endif
	lncnt = 0;
	while ((info = getinfo(TRUE)) != NULL && (info->flags & HIDDENUSER) == 0) {
		if ((ptr = getuserfile(info->name)) == NULL) continue;
		if (!lncnt) {
#ifdef	EXTERN_APPS
			putstr("The following users are presently on BBS:\n");
			putstr("TTY    LoginTime        UserName                 City/State           BaudRate\n");
			putstr("==============================================================================\n");
#else
			putstr("%s\n", sysmsg(MSG_WHOISTHEREHEADER));
			putstr("TTY    LoginTime        UserName                 City/State           BaudRate\n");
			putstr("%s\n", sysmsg(MSG_DELIMITER));
			if (termflags & (ANSITERM | RIPTERM))
				putstr("\033[1;37m");
#endif
			lncnt = 3;
		}
#ifdef	EXTERN_APPS
		ptr = getfileconf(ptr, USERLOCATION, "Unknown");
#else
		ptr = getfileconf(ptr, USERLOCATION, sysmsg(MSG_UNKNOWN));
#endif
		putstr("%-6.6s %.15s  %-24.24s %-20.20s %d\n", info->tty,
		       ctime(&info->ltime)+4, info->name, ptr, info->baud);

		if (!more(&lncnt)) {
			closeinfo();
			break;
		}
	}
#ifndef	EXTERN_APPS
	if (termflags & (ANSITERM | RIPTERM)) putstr("\033[0;37m");
#endif
}

/*
 * Display list of users currently in conference.
 */
void
who_in_conf()
{
	int lncnt;
	char *ptr;
	struct loginfo *info;

#ifndef	EXTERN_APPS
	LOGIT(LOG_INFO, "Who in conference");
#endif
	lncnt = 0;
	while ((info = getinfo(TRUE)) != NULL && (info->flags & HIDDENUSER) == 0) {
		if (info->flags & INCONF) {
			if (!lncnt) {
#ifdef	EXTERN_APPS
				putstr("The following users are presently in conference:\n");
				putstr("TTY    LoginTime        UserName                 CallSign\n");
				putstr("===============================================================================\n");
#else
				putstr("%s\n", sysmsg(MSG_INCONFHEADER));
				putstr("TTY    LoginTime        UserName                 CallSign\n");
				putstr("%s\n", sysmsg(MSG_DELIMITER));
				if (termflags & (ANSITERM | RIPTERM))
					putstr("\033[1;37m");
#endif
				lncnt = 3;
			}
			if ((ptr = getsignbyname(info->name)) == NULL)
#ifndef	EXTERN_APPS
				ptr = sysmsg(MSG_NONE);
#else
				ptr = "Not selected";
#endif
			putstr("%-6.6s %-.15s  %-24.24s %s", info->tty,
			       ctime(&info->ltime)+4, info->name, ptr);
			if (info->flags & TERMINATOR) putstr(" \t*terminator*");
			else if (info->flags & KICKER) putstr(" \t*kicker*");
			putstr("\n");
			if (!more(&lncnt)) {
				closeinfo();
				break;
			}
		}
	}
#ifdef	EXTERN_APPS
	if (!lncnt) putstr("No users in conference\n");
#else
	if (termflags & (ANSITERM | RIPTERM)) putstr("\033[0;37m");
	if (!lncnt) putstr("%s\n", sysmsg(MSG_NOUSERSINCONF));
#endif
}

int
whodo(myself)
	int myself;
{
	struct loginfo *info;
	int i = 0, lncnt = 0;

#ifndef	EXTERN_APPS
	if (myself) LOGIT(LOG_INFO, "Who do");
#endif
	while ((info = getinfo(myself)) != NULL && (info->flags & HIDDENUSER) == 0) {
		if (!lncnt) {
			if (myself) lncnt = 2;
			else {
#ifdef	EXTERN_APPS
				putstr("You can talk only available for talk users:\n\n");
#else
				putstr("%s\n", sysmsg(MSG_INTALKHEADER));
#endif
				lncnt = 3;
			}
			putstr(" #   TTY    UserName                 Status\n");
#ifdef	EXTERN_APPS
			putstr("===============================================================================\n");
#else
			putstr("%s\n", sysmsg(MSG_DELIMITER));
			if (termflags & (ANSITERM | RIPTERM)) putstr("\033[1;37m");
#endif
		}
		putstr("%2d)  %-6.6s %-24.24s ", ++i, info->tty, info->name);
#ifdef	EXTERN_APPS
		if (info->flags & INCONF)	putstr("In conference");
		else if (info->flags & INTALK)	putstr("In talk");
		else if (info->flags & INRECVF)	putstr("Receiving file");
		else if (info->flags & INSENDF)	putstr("Sending file");
		else if (info->flags & EXECEXT)	putstr("Running ext apps");
		else if (info->flags & MESGNO)	putstr("Have mesg no");
		else putstr("Available for talk");
#else
		if (info->flags & INCONF)	psysmsg(MSG_USERINCONF);
		else if (info->flags & INTALK)	psysmsg(MSG_USERINTALK);
		else if (info->flags & INRECVF)	psysmsg(MSG_USERRECVFILE);
		else if (info->flags & INSENDF)	psysmsg(MSG_USERSENDFILE);
		else if (info->flags & EXECEXT)	psysmsg(MSG_USEREXECEXT);
		else if (info->flags & MESGNO)	psysmsg(MSG_USERMESGNO);
		else psysmsg(MSG_USERTALKAVAIL);
#endif
		putstr("\n");
		if (!more(&lncnt)) {
			closeinfo();
			break;
		}
	}
#ifndef	EXTERN_APPS
	if (termflags & (ANSITERM | RIPTERM)) putstr("\033[0;37m");
#endif
	return i;
}

/*
 * Display information about user in short form (by line).
 */
int
user_info(user)
	char *user;
{
	time_t lc_time;
	char *unknown, *file, org[25], city[25];

#ifndef	EXTERN_APPS
	unknown = sysmsg(MSG_UNKNOWN);
#else
	unknown = "Unknown";
#endif
	if ((file = getuserfile(user)) == NULL) return 0;

	(void)strncpy(org, getfileconf(file, USERORG, unknown), sizeof(org)-1);
	org[sizeof(org)-1] = '\0';

	(void)strncpy(city, getfileconf(file, USERLOCATION, unknown), sizeof(city)-1);
	city[sizeof(city)-1] = '\0';

	lc_time = atoi(getfileconf(file, LASTCALLTIME, "0"));

	putstr("%-20.20s %-20.20s %-20.20s %-.15s\n", user, org, city,
	       lc_time ? ctime(&lc_time)+4 : unknown);

	return 1;
}

static char *search_str;

static int
search_user(path, file)
	register char *path;
	struct dir_ent *file;
{
	register char *first, *last;
	int found;

	if ((file->mode & S_IFMT) != S_IFDIR) return 0;

	for (last = &path[strlen(path)-1]; last > path; last--)
		if (*last == '/') break;
	if (last == path) return 0;
	for (first = last-1; first > path; first--)
		if (*first == '/') break;
	if (first == path) return 0;
	first++;
	*last = ' ';
	found = (__strcasestr(first, search_str) != NULL);
	*last = '/';

	return found;
}

/*
 * Display list of users from userbase.
 */
void
userlist(substr)
	char *substr;
{
	int i, j, lncnt = 3;
	char path[1024], *first, *last;
	struct dir_ent *fn, *ln;
	int nf, nl;
	int (*func)();

#ifndef	EXTERN_APPS
	if (substr == NULL) {
		LOGIT(LOG_INFO, "User List");
		putstr("%s\n", sysmsg(MSG_USERLISTHEADER));
		func = dirs_only;
	} else {
		LOGIT(LOG_INFO, "User List for \"%s\"", substr);
		search_str = substr;
		func = search_user;
	}
	putstr("UserName             Organization         City/State           LastCall       \n");
	putstr("%s\n", sysmsg(MSG_DELIMITER));
	if (termflags & (ANSITERM | RIPTERM)) putstr("\033[1;37m");
#else
	if (substr == NULL) {
		putstr("List of all users...\n");
		func = dirs_only;
	} else {
		putstr("Search for \"%s\"...\n", substr);
		search_str = substr;
		func = search_user;
	}
	putstr("UserName             Organization         City/State           LastCall       \n");
	putstr("==============================================================================\n");
#endif

	if (USRHOMEDIR[0] != '/') {
		if (bbsdir == NULL && getbbsdir() == NULL) return;
		(void)strcpy(path, bbsdir);
		(void)strcat(path, "/");
	} else *path = '\0';
	(void)strcat(path, USRHOMEDIR);

	if ((nf = scan_dir(path, &fn, dirs_only, 0)) < 0) {
#ifdef	EXTERN_APPS
		fprintf(stderr, "can't scan_dir \"%s\": %s\n", path, strerror(errno));
#else
		LOGIT(LOG_ERR, "can't scan_dir \"%s\": %m", path);
#endif
		return;
	}

	first = &path[strlen(path)];
	*first++ = '/';
	for (i = 0; i < nf; i++) {
		(void)strcpy(first, fn[i].name);
		if ((nl = scan_dir(path, &ln, func, 0)) < 0) {
			free_dir(&fn, nf);
#ifdef	EXTERN_APPS
			fprintf(stderr, "can't scan_dir \"%s\": %s\n", path,
				strerror(errno));
#else
			LOGIT(LOG_ERR, "can't scan_dir \"%s\": %m", path);
#endif
			return;
		}
		last = &first[strlen(first)];
		*last++ = ' ';
		for (j = 0; j < nl; j++) {
			(void)strcpy(last, ln[j].name);
			if (user_info(first) == 0) continue;
			if (!more(&lncnt)) {
				i = nf;
				break;
			}
		}
		free_dir(&ln, nl);
	}
	free_dir(&fn, nf);
#ifndef	EXTERN_APPS
	if (termflags & (ANSITERM | RIPTERM)) putstr("\033[0;37m");
#endif
}

/*
 * Display list of callsigns from callsignbase.
 */
void
callsignlist(substr)
	char *substr;
{
	int lncnt = 3;
	char *p;

#ifndef	EXTERN_APPS
	if (substr == NULL) LOGIT(LOG_INFO, "CallSign List");
	else LOGIT(LOG_INFO, "CallSign List for \"%s\"", substr);

	putstr("CallSign         UserName\n");
	putstr("%s\n", sysmsg(MSG_DELIMITER));
	if (termflags & (ANSITERM | RIPTERM)) putstr("\033[1;37m");
#else
	putstr("CallSign         UserName\n");
	putstr("==============================================================================\n");
#endif
	closesignent();
	while ((p = getsignent()) != NULL) {
		if (substr != NULL && __strcasestr(p, substr) == NULL)
			continue;
		putstr("%s\n", p);
		if (!more(&lncnt)) break;
	}
	closesignent();
#ifndef	EXTERN_APPS
	if (termflags & (ANSITERM | RIPTERM)) putstr("\033[0;37m");
#endif
}

/*
 * Display list of baned users.
 */
void
banlist(substr)
	char *substr;
{
	FILE *fp;
	int lncnt = 3;
	char *p, *p1, *p2, buf[256];

	if (substr != NULL) {
		while (*substr == ' ') substr++;
		if (!*substr) substr = NULL;
	}
#ifndef	EXTERN_APPS
	if (substr == NULL) LOGIT(LOG_INFO, "Ban List");
	else LOGIT(LOG_INFO, "Ban List for \"%s\"", substr);

	putstr("UserName              Baned by             Reason\n");
	putstr("%s\n", sysmsg(MSG_DELIMITER));
	if (termflags & (ANSITERM | RIPTERM)) putstr("\033[1;37m");
#else
	putstr("UserName              Baned by             Reason\n");
	putstr("==============================================================================\n");
#endif
	if ((fp = fopen(BANLIST, "r")) == NULL) return;
	while (fgets(buf, sizeof(buf), fp) != NULL) {
		buf[sizeof(buf)-1] = '\0';
		if (buf[0] == '#' || buf[0] == '\n') continue;
		if ((p = strchr(buf, '\n')) != NULL) *p = '\0';
		if (substr != NULL && __strcasestr(buf, substr) == NULL)
			continue;
		if ((p = strchr(buf, ':')) == NULL) continue;
		*p++ = '\0';
		if ((p1 = strchr(p, ':')) == NULL) continue;
		*p1++ = '\0';
		if ((p2 = getsignbyname(buf)) == NULL) p2 = buf;
		putstr("%-20.20s %-20.20s %-.36s\n", p2, p, p1);
		if (!more(&lncnt)) break;
	}
	fclose(fp);
#ifndef	EXTERN_APPS
	if (termflags & (ANSITERM | RIPTERM)) putstr("\033[0;37m");
#endif
}

#ifndef	EXTERN_APPS

void
user_list()
{
	char *p;

	p = prompt_str(MSG_ENTERSUBSTR, NULL, LOGINFO_NAMELEN);
	putchr('\n');
	userlist(p);
}

void
callsign_list()
{
	char *p;

	p = prompt_str(MSG_ENTERSUBSTR, NULL, LOGINFO_NAMELEN);
	putchr('\n');
	callsignlist(p);
}

void
ban_list()
{
	char *p;

	p = prompt_str(MSG_ENTERSUBSTR, NULL, LOGINFO_NAMELEN);
	putchr('\n');
	banlist(p);
}

void
who_do()
{
	whodo(TRUE);
}

void
who_is_who()
{
	char *ptr;
	int rcode;

	ptr = prompt_str(MSG_ENTERUSERNAME, process.name, LOGINFO_NAMELEN);
	if (ptr == NULL) return;
	putchr('\n');
	LOGIT(LOG_INFO, "Who is \"%s\"", ptr);

	if (termflags & (ANSITERM | RIPTERM)) putstr("\033[1;37m");
	rcode = getuserinfo(putstr, ptr);
	if (termflags & (ANSITERM | RIPTERM)) putstr("\033[0;37m");

	if (rcode < 0) warning(MSG_NOSUCHUSER, 1);
}
#endif
