/*
 *	Copyright (c) 1997 Rinet Corp., Novosibirsk, Russia
 *
 * Redistribution and use in source forms, with and without modification,
 * are permitted provided that this entire comment appears intact.
 *
 * THIS SOURCE CODE IS PROVIDED ``AS IS'' WITHOUT ANY WARRANTIES OF ANY KIND.
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>

#include "drive.h"
#include "getconf.h"

extern int errno;

int confcached = FALSE;
static char cache[USRBUFSIZE * 2];

/*
 * Get value of variable from current user config using cache when available.
 * If no such variable found function return default value.
 */
char *
getuserconf(var, def)
	char *var, *def;
{
	register len;
	register char *p, *eol;
	static char buf[1024];

	if (confcached == FALSE) {
		struct stat st;
		sprintf(buf, "%s/%s", homedir, userconf);
		if (stat(buf, &st) < 0) return def;
		if (!st.st_size) {
			char bak[512];
			(void)strcpy(bak, buf);
			(void)strcat(bak, ".bak");
			unlink(buf);
			if (link(bak, buf) < 0) {
				LOGIT(LOG_ERR, "restore \"%s\" failed", buf);
				return def;
			}
			LOGIT(LOG_WARNING, "\"%s\" restored from backup", buf);
		}
		return (getfileconf(buf, var, def));
	}
	len = strlen(var);
	for (p = cache; (eol = strchr(p, '\n')) != NULL; p = ++eol) {
		if (p[len] == '=' && !strncmp(p, var, len)) {
			p += len + 1;
			len = eol - p;
			if (len > 0) {
				memcpy(buf, p, len);
				buf[len] = '\0';
				return buf;
			}
			break;
		}
	}
	return def;
}

/*
 * Put value of variable to current user config creating cache.
 * Sync cache to file and make it backup copy.
 */
void
putuserconf(var, val)
	char *var, *val;
{
	register char *p1, *p2;
	register fd, len;
	int flag;
	char *line, fileconf[1024], buf[USRBUFSIZE * 2];

	(void)sprintf(fileconf, "%s/%s", homedir, userconf);
	if (confcached == FALSE) {
		while ((fd = open(fileconf, O_RDONLY)) < 0 && errno == EINTR);
		if (fd < 0 && errno != ENOENT) {
			LOGIT(LOG_CRIT, "can't open \"%s\": %m", fileconf);
			quit(1);
		}
		if (fd > 0) {
			while ((len = read(fd, cache, USRBUFSIZE)) < 0 && errno == EINTR);
			close(fd);
			if (len < 0) {
				LOGIT(LOG_CRIT, "can't read \"%s\": %m", fileconf);
				quit(1);
			}
		} else	len = 0;
		if (len) {
			if (cache[len-1] != '\n') cache[len++] = '\n';
		}
		cache[len] = '\0';
		confcached = TRUE;
	}
	buf[0] = '\0';
	flag = 0;
	len = strlen(var);
	for (p1 = cache, p2 = buf; (line = strchr(p1, '\n')) != NULL; p1 = ++line) {
		fd = line - p1;
		memcpy(p2, p1, fd);
		p2[fd] = '\0';
		if (p2[len] == '=' && !strncmp(p2, var, len)) {
			/* Is there the same value, not need to be rewriting */
			if (!strcmp(&p2[len+1], val)) return;
			flag++;
			(void)strcpy(&p2[len+1], val);
		}
		p2 += strlen(strcat(p2, "\n"));
	}
	if (!flag) {	/* append the new variable */
		/* attention: config may rapidly grow, how about cache size? */
		(void)strcat(buf, var);
		(void)strcat(buf, "=");
		(void)strcat(buf, val);
		(void)strcat(buf, "\n");
	}
	len = strlen(strcpy(cache, buf));

	/* Make backup copy (hard link) of .user.conf */
	strcpy(buf, fileconf);
	strcat(buf, ".bak");
	unlink(buf);
	if (link(fileconf, buf) < 0 && errno != ENOENT)
		LOGIT(LOG_ERR, "backup \"%s\" failed", buf);

	/* Temporary write userinfo in .user.conf.b */
	strcpy(buf, fileconf);
	strcat(buf, ".b");
	unlink(buf);
	if ((fd = open(buf, O_WRONLY|O_CREAT, 0640)) < 0 && errno != EINTR)
		LOGIT(LOG_ERR, "can't write \"%s\": %m", fileconf);
	if (fd >= 0) {
		while ((flag = write(fd, cache, len)) < 0 && errno == EINTR);
		fsync(fd);
		close(fd);
	} else	flag = 0;

	/* Actually make file .user.conf from .user.conf.b */
	if (flag != len || rename(buf, fileconf) < 0)
		LOGIT(LOG_ERR, "write \"%s\" failed", fileconf);
}

char *
getrcent(rc, var)
	char *rc, *var;
{
	FILE *fp;
	int len;
	char buf[1024];
	static char val[256];

	sprintf(buf, "%s/%s", homedir, rc);
	if ((fp = fopen(buf, "r")) == NULL) return NULL;
	(void) strcpy(val, var);
	(void) strcat(val, "=");
	len = strlen(val);
	while (fgets(buf, sizeof(buf), fp) != NULL) {
		buf[sizeof(buf)-1] = '\0';
		if (!strncmp(buf, val, len)) {
			char *p = val;
			while ((u_char)buf[len] > 0x20)	*p++ = buf[len++];
			*p = '\0';
			fclose(fp);
			return val;
		}
	}
	fclose(fp);
	return NULL;
}

void
putrcent(rc, var, val)
	char *rc, *var, *val;
{
	FILE *ifp, *ofp;
	int len;
	char *p, buf[1024], tmp[1024], token[256];

	sprintf(buf, "%s/%s", homedir, rc);
	if ((ifp = fopen(buf, "r")) == NULL) {
		if ((ofp = fopen(buf, "w")) != NULL) {	/* make it */
			fprintf(ofp, "%s=%s\n", var, val);
			fclose(ofp);
		}
		return;
	}
	(void) strcpy(tmp, buf);
	(void) strcat(tmp, "~tmp");
	if ((ofp = fopen(tmp, "w")) == NULL) {
		fclose(ifp);
		return;
	}
	(void) strcpy(token, var);
	(void) strcat(token, "=");
	len = strlen(token);
	while (fgets(buf, sizeof(buf), ifp) != NULL) {
		buf[sizeof(buf)-1] = '\0';
		if (len && !strncmp(buf, token, len)) {
			if ((p = strchr(&buf[len], '\n')) != NULL) *p = '\0';
			if (!strcmp(&buf[len], val)) {	/* the same value */
				len = -1;
				break;
			}
			fprintf(ofp, "%s%s\n", token, val);
			len = 0;
			continue;
		}
		fputs(buf, ofp);
	}
	if (len > 0) fprintf(ofp, "%s%s\n", token, val);
	fclose(ofp);
	fclose(ifp);
	if (len < 0) unlink(tmp);
	else {
		sprintf(buf, "%s/%s", homedir, rc);
		rename(tmp, buf);
	}
}
