/*
 *	Copyright (c) 1994 The CAD lab of the
 *	Novosibirsk Institute of Broadcasting and Telecommunication
 *
 *	TNSDrive $Id$
 *
 *	$Log$
 *
 * Redistribution and use in source forms, with and without modification,
 * are permitted provided that this entire comment appears intact.
 *
 * THIS SOURCE CODE IS PROVIDED ``AS IS'' WITHOUT ANY WARRANTIES OF ANY KIND.
 */

#include <sys/types.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <sys/signal.h>
#include <sys/file.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <time.h>
#include <ctype.h>
#include <unistd.h>

#include "compat.h"
#include "drive.h"
#include "sysmsg.h"
#include "usenet.h"
#include "nntp.h"
#include "group.h"
#include "rematch.h"
#include "variables.h"

static FILE *active_fp = NULL;

int max_active = 0;
struct group_ent *active = NULL;	/* active file */
int group_hash[TABLE_SIZE];		/* group name --> active[] */
int *my_group = NULL;			/* .newsrc --> active[] */
int *unread = NULL;			/* highest art read in group */
int num_active;				/* one past top of active */
int local_top;				/* one past top of my_group */
long note_size;				/* stat size in bytes of article */

struct header *arts = NULL;
long *base = NULL;
int max_art = 0;
int top = 0;
int top_base = 0;

static char tmpbuf[1024];
int usenetinuse = 0;
static char usenetpathconf[MAXPATHLEN+1];
static char usenetspooldir[MAXPATHLEN+1];
static char active_file[MAXPATHLEN+1];
static char newsgroups_file[MAXPATHLEN+1];
static char usenetnntpserv[MAXPATHLEN+1];
int curgrpidx = -1;				/* current group index */
int canpost = FALSE;

extern int errno;

int ArtReadLimit = 0, ArtPostLimit = 0, ArtSizeLimit = 0;
int UserArtRead = 0, UserArtPost = 0, UserArtSize = 0;


initusenetarea()
{
	char *ptr;
	extern char subject_search_string[], author_search_string[], body_search_string[];

	usenetinuse = 0;
	if ((ptr = getuserconf(NEWSSORT, NULL)) != NULL)
		newssort = strdup(ptr);
	else	newssort = NULL;
	if ((ptr = getuserconf(ARTREADLIMIT, NULL)) != NULL)
		ArtReadLimit = atoi(ptr);
	if ((ptr = getuserconf(ARTSIZELIMIT, NULL)) != NULL)
		ArtSizeLimit = atoi(ptr) * 1024;
	if ((ptr = getuserconf(ARTPOSTLIMIT, NULL)) != NULL)
		ArtPostLimit = atoi(ptr);
	if ((ptr = getuserconf(USERARTREAD, NULL)) != NULL)
		UserArtRead = atoi(ptr);
	if ((ptr = getuserconf(USERARTSIZE, NULL)) != NULL)
		UserArtSize = atoi(ptr) * 1024;
	if ((ptr = getuserconf(USERARTPOST, NULL)) != NULL)
		UserArtPost = atoi(ptr);
	subject_search_string[0] = '\0';
	author_search_string[0] = '\0';
	body_search_string[0] = '\0';
	return;
}

asfail(file, line, cond)
	int line;
	char *file, *cond;
{
	extern void onint();
	UsenetClose();
	LOGIT(LOG_ERR, "BUG: assertion failure: %s (%d): %s", file, line, cond);
	sleep(1);
	onint();
}

void
connectbroken()
{
	extern void onint();
	UsenetClose();
	putstr("%s\n", sysmsg(MSG_CONNECTBROKEN));
	onint();
}

int
UsenetOpen()
{
	int fd, n;
	char *line, *ptr, *arg1, *arg2 = "", *arg3 = "";

	pleasewait(0);
	if (usenetinuse && !strcmp(dirpath, usenetpathconf)) return 0;
	sprintf(tmpbuf, "%s/%s", dirpath, USENETPATHCONF);
	if ((fd = open(tmpbuf, O_RDONLY)) < 0) return -1;
	while ((n = read(fd, tmpbuf, sizeof(tmpbuf)-1)) < 0 && errno == EINTR);
	close(fd);
	if (n < 3) return -1;
	if (tmpbuf[n-1] != '\n') tmpbuf[n++] = '\n';
	tmpbuf[n] = '\0';
	for (ptr = tmpbuf, arg1 = NULL; (line = strchr(ptr, '\n')) != NULL; ptr = line) {
		*line++ = '\0';
		while ((u_char)*ptr <= 0x20 && *ptr) ptr++;
		if (!*ptr || *ptr == '#') continue;
		arg1 = ptr;
		while ((u_char)*ptr > 0x20) ptr++;
		if (*ptr) for (*ptr++ = 0; (u_char)*ptr <= 0x20 && *ptr; ptr++);
		arg2 = ptr;
		if (*ptr) {
			while ((u_char)*ptr > 0x20) ptr++;
			if (*ptr) for (*ptr++ = 0; (u_char)*ptr <= 0x20 && *ptr; ptr++);
			arg3 = ptr;
		}
		break;
	}
	if (arg1 == NULL) return -1;
	if (!*arg2 && strchr(arg1, '/')) return -1;
	switch (usenetinuse & READ_NEWS_MODE) {
		case USESPOOL:
			if (!strcmp(arg1, usenetspooldir) &&
			    !strcmp(arg2, active_file) &&
			    (!*arg3 || !strcmp(arg3, newsgroups_file)))
				return 0;
			UsenetClose();
			break;
		case USENNTP:
			if (!strcmp(arg1, usenetnntpserv)) return 0;
			UsenetClose();
	}
	max_active = 1000;	/* initial alloc */
	active = (struct group_ent *)malloc(sizeof(*active) * max_active);
	my_group = (int *)malloc(sizeof(int) * max_active);
	unread = (int *)malloc(sizeof(int) * max_active);
	max_art = 100;		/* initial alloc */
	arts = (struct header *)malloc(sizeof(*arts) * max_art);
	base = (long *)malloc(sizeof(long) * max_art);
	if (active == NULL || my_group == NULL || unread == NULL ||
	    arts == NULL || base == NULL) {
		warning(MSG_OUTOFMEMORY, 1);
		return -1;
	}
	if (!*arg2) {	/* nntpserv */
		putstr("%s %s... ", sysmsg(MSG_CONNECTING), arg1);
		switch (server_init(arg1, NNTP_TCP_NAME, NNTP_TCP_PORT)) {
			case OK_CANPOST:
				break;
			case OK_NOPOST:
				putstr("%s ", sysmsg(MSG_NNTPREADONLY));
				break;
			case -1:
				UsenetClose();
				putstr("%s\n", sysmsg(MSG_NNTPFAILED));
				return -1;
			default:
				UsenetClose();
				putstr("%s\n", sysmsg(MSG_NNTPREJECTED));
				return -1;
		}
		usenetinuse = USENNTP;
		strcpy(usenetnntpserv, arg1);

		/* if INN nnrp switch to mode reader */
		put_server("mode reader");
		if (get_respcode() != ERR_COMMAND) usenetinuse |= INN_NNTP_SERVER;

		/* if INN nntp supports XOVER command */
		put_server("xover");
		if (get_respcode() != ERR_COMMAND) usenetinuse |= XOVER_SUPPORTED;

	} else {	/* spooldir */
		struct stat st;
		if (stat(arg1, &st) < 0 || (st.st_mode & S_IFMT) != S_IFDIR ||
		    !(st.st_mode & S_IRUSR) || !(st.st_mode & S_IXUSR)) {
			UsenetClose();
			putstr("%s\n", sysmsg(MSG_CANTSPOOLACCESS));
			return -1;
		}
		if (stat(arg2, &st) < 0 || (st.st_mode & S_IFMT) != S_IFREG ||
		    !(st.st_mode & S_IRUSR)) {
			UsenetClose();
			putstr("%s\n", sysmsg(MSG_CANTACTIVEACCESS));
			return -1;
		}
		if (*arg3 &&
		    (stat(arg3, &st) < 0 || (st.st_mode & S_IFMT) != S_IFREG ||
		    !(st.st_mode & S_IRUSR))) arg3 = "";
		strcpy(usenetspooldir, arg1);
		strcpy(active_file, arg2);
		strcpy(newsgroups_file, arg3);
		usenetinuse = USESPOOL;
	}
	putstr("%s ", sysmsg(MSG_READGROUPLIST));
	if (!read_active()) {
		UsenetClose();
		putstr("%s\n", sysmsg(MSG_CANTACTIVEACCESS));
		return -1;
	}
	read_descriptions();
	strcpy(usenetpathconf, dirpath);
	curgrpidx = -1;
	putstr(" Ok\n\n");
	return 0;
}

int
get_respcode()
{
	char line[NNTP_STRLEN];

	if (get_server(line, NNTP_STRLEN) == -1) connectbroken();
	return atoi(line);
}

FILE *
nntp_to_fp()
{
	FILE *fp;
	char *p, line[NNTP_STRLEN];

	note_size = 0;
	mktemp(strcpy(line, TMPFILEMASK));
	if ((fp = fopen(line, "w+")) == NULL) return NULL;
	unlink(line);
	while (1) {
		if (get_server(line, NNTP_STRLEN) == -1) {
			fclose(fp);
			connectbroken();
		}
		if (!strcmp(line, ".")) break;	/* end of text */
		p = line;
		if (*p == '.') p++;	/* reduce leading .'s */
		note_size += strlen(p) + 1;
		if (*p) fputs(p, fp);
		fputc('\n', fp);
	}
	if (note_size) {
		rewind(fp);
		return fp;
	}
	fclose(fp);
	return NULL;
}

FILE *
open_active_fp()
{
	switch (usenetinuse & READ_NEWS_MODE) {
		case USESPOOL:
			return fopen(active_file, "r");
		case USENNTP:
			put_server("list");
			if (get_respcode() != OK_GROUPS) return NULL;
			return nntp_to_fp();
	}
	return NULL;
}

FILE *
open_newsgroups_fp()
{
	switch (usenetinuse & READ_NEWS_MODE) {
		case USESPOOL:
			if (newsgroups_file[0] == '\0') return NULL;
			return fopen(newsgroups_file, "r");
		case USENNTP:
			put_server("list newsgroups");
			if (get_respcode() != OK_GROUPS) return NULL;
			return nntp_to_fp();
	}
	return NULL;
}

/*
 * Open a group XOVER file
 */ 
FILE *
open_xover_fp(group_name, group_path, min, max)
	char *group_name, *group_path;
	long min, max;
{
	char xover_file[LEN];
	char line[NNTP_STRLEN];

	if (usenetinuse & XOVER_SUPPORTED) {
		sprintf(line, "xover %ld-%ld", min, max);
		put_server(line);
		if (get_respcode() != OK_XOVER) return NULL;
		return nntp_to_fp();
	} else {
		sprintf(xover_file, "%s/%s/.overview", usenetspooldir, group_path);
		return fopen(xover_file, "r");
	}
}

FILE *
open_header_fp(group_path, art)
	char *group_path;
	long art;
{
	switch (usenetinuse & READ_NEWS_MODE) {
		case USESPOOL:
			sprintf(tmpbuf, "%s/%s/%ld", usenetspooldir,
				group_path, art);
			return fopen(tmpbuf, "r");
		case USENNTP:
			sprintf(tmpbuf, "head %ld", art);
			put_server(tmpbuf);
			if (get_respcode() != OK_HEAD) return NULL;
			return nntp_to_fp();
	}
	return NULL;
}

FILE *
open_art_fp(group_path, art)
	char *group_path;
	long art;
{
	struct stat st;

	switch (usenetinuse & READ_NEWS_MODE) {
		case USESPOOL:
			sprintf(tmpbuf, "%s/%s/%ld", usenetspooldir,
				group_path, art);
			if (stat(tmpbuf, &st) < 0) note_size = 0;
			else note_size = (int)st.st_size;
			return fopen(tmpbuf, "r");
		case USENNTP:
			sprintf(tmpbuf, "article %ld", art);
			put_server(tmpbuf);
			if (get_respcode() != OK_ARTICLE) return NULL;
			return nntp_to_fp();
	}
	return NULL;
}

long
hash_groupname(buf)
	char *buf;
{
	char *t = buf;
	unsigned long h = *t++;
	while (*t) h = ((h << 1) ^ *t++) % TABLE_SIZE;
	return h;
}

int
find_group_index(group)
	char *group;
{
	int i = group_hash[hash_groupname(group)];

	while (i >= 0) {
		if (!strncmp(group, active[i].name, strlen(active[i].name)))
			return i;
		i = active[i].next;
	}
	return -1;
}

read_descriptions()
{
	FILE *fp = NULL;
	FILE *fp_save = NULL;
	char buf[LEN];
	char group[MAXPATHLEN+1];
	register char *p, *q;
	register i;
	int l = 0;
	extern char working[];

	if (usenetinuse & USENNTP) {
		if ((fp = fopen(NEWSGROUPS, "r")) == NULL)
			fp_save = sfopen(NEWSGROUPS, "w");
	}
	if (fp == NULL && (fp = open_newsgroups_fp()) == NULL) {
		if (fp_save != NULL) {
			sfclose(fp_save);
			unlink(NEWSGROUPS);
		}
		return;
	}
	while (fgets(buf, sizeof(buf), fp) != NULL) {
		buf[sizeof(buf)-1] = '\0';
		if ((p = strchr(buf, '\n')) != NULL) *p = '\0';
		if (buf[0] == '\0' || buf[0] == '#') continue;
		for (p = buf, q = group; *p && *p != ' ' && *p != '\t' ; p++, q++)
			*q = *p;
		*q = '\0';
		while (*p == '\t' || *p == ' ' || *p == '-') p++;
		if (strlen(p) < 2) continue;
		i = find_group_index(group);
		if (i >= 0 && active[i].desc == NULL) {
			active[i].desc = strdup(p);
			if (fp_save != NULL) fprintf(fp_save, "%s\n", buf);
		}
		if (!(i % 100)) putstr("\b%c", working[++l & 3]);
	}
	fclose(fp);
	if (fp_save != NULL) sfclose(fp_save);
	return;
}

static int
sortgrpbyname(g1, g2)
	register struct group_ent *g1;
	register struct group_ent *g2;
{
	return (strcmp(g1->name, g2->name));
}

/*
 * parse line from news active files
 */ 
int
parse_active_line(line, max, min, moderated)
	char *line;
	long *max;
	long *min;
	char *moderated;
{
	register char *p, *q, *r;
	
	if (line[0] == '#' || line[0] == '\n') return FALSE;
	for (p = line; *p && *p != ' ' && *p != '\n'; p++);
	if (*p != ' ') return FALSE;
	*p++ = '\0';
	for (q = p; *q && *q != ' '; q++);
	if (*q != ' ') return FALSE;;
	*q++ = '\0';
	for (r = q; *r && *r != ' '; r++);
	if (*r != ' ') return FALSE;;
	*r++ = '\0';
	if (*r) {
		strcpy(moderated, r);
		if ((r = strchr(moderated, '\n')) != NULL) *r = '\0';
	}
	*max = (long)atol(p);
	*min = (long)atol(q);
	return TRUE;
}

int
read_active()
{
	register i, j;
	char buf[LEN], moderated[LEN];
	long h, min, max;

	num_active = 0;
	hash_init();
	for (i = 0; i < TABLE_SIZE; i++) group_hash[i] = -1;
	if ((active_fp = open_active_fp()) == NULL) return FALSE;
	putstr("    ");
	strcpy(moderated, "y");
	while (fgets(buf, sizeof(buf), active_fp) != NULL) {
		buf[sizeof(buf)-1] = '\0';
		if (!parse_active_line(buf, &max, &min, moderated)) continue;
		/*
		 * Don't load group into active[] from active file if 
		 * 'x'  junked group
		 * '='  aliased group
		 */
		if (moderated[0] == 'x' || moderated[0] == '=') continue;
		if (num_active >= max_active && !expand_active()) continue;
		h = hash_groupname(buf);
		if (group_hash[h] == -1) group_hash[h] = num_active;
		else {	/* hash linked list chaining */
			for (i = group_hash[h]; active[i].next >= 0;
			     i = active[i].next)
				if (!strcmp(active[i].name, buf)) break;
			if (!strcmp(active[i].name, buf)) continue;
			active[i].next = num_active;
		}
		/* Load group info */
		active[num_active].name = strdup(buf);
		active[num_active].desc = NULL;
		active[num_active].max = max;
		active[num_active].min = min;
		active[num_active].next = -1;
		active[num_active].flag = NOTGOT;
		if (!(++num_active % 100)) putstr("\b\b\b\b%4d", num_active);
	}
	fclose(active_fp);
	active_fp = NULL;
	putstr("\b\b\b\b%4d ", num_active);
	qsort(active, num_active, sizeof(struct group_ent), sortgrpbyname);
	for (i = 0; i < TABLE_SIZE; i++) group_hash[i] = -1;
	for (j = 0; j < num_active; j++) {
		h = hash_groupname(active[j].name);
		if (group_hash[h] == -1) group_hash[h] = j;
		else {	/* hash linked list chaining */
			for (i = group_hash[h]; active[i].next >= 0;
			     i = active[i].next)
				if (!strcmp(active[i].name, active[j].name))
					break;
			if (!strcmp(active[i].name, active[j].name)) continue;
			active[i].next = j;
		}
		active[j].next = -1;
	}
	return TRUE;
}

/*
 *  Read the article numbers existing in a group's spool directory
 *  into base[] and sort them.  top_base is one past top.
 */
setup_base(group, group_path)
	char *group, *group_path;
{
	top_base = 0;
	switch (usenetinuse & READ_NEWS_MODE) {
		case USESPOOL:
			return spool_setup_base(group_path);
		case USENNTP:
			return nntp_setup_base(group);
	}
}

base_comp(a, b)
	register long *a, *b;
{
	if (*a < *b) return -1;
	if (*a > *b) return 1;
	return 0;
}

spool_setup_base(group_path)
	char *group_path;
{
	DIR *d;
	struct dirent *e;
	long art;

	sprintf(tmpbuf, "%s/%s", usenetspooldir, group_path);
	if (access(tmpbuf, R_OK|X_OK) < 0) return;
	if ((d = opendir(tmpbuf)) != NULL) {
		while ((e = readdir(d)) != NULL) {
			art = atol(e->d_name);
			if (art >= 0) {
				if (top_base >= max_art) expand_art();
				base[top_base++] = art;
			}
		}
		closedir(d);
		qsort(base, top_base, sizeof(long), base_comp);
	}
}

nntp_stat(artnum)
	long artnum;
{
	sprintf(tmpbuf, "stat %ld", artnum);
	put_server(tmpbuf);
	if (get_respcode() != OK_NOTEXT) return FALSE;
	return TRUE;
}

nntp_setup_base(group)
	char *group;
{
	long dummy, count, start, last;

	sprintf(tmpbuf, "group %s", group);
	put_server(tmpbuf);
	if (get_server(tmpbuf, NNTP_STRLEN) == -1) connectbroken();
	if (atoi(tmpbuf) != OK_GROUP) return;
	sscanf(tmpbuf,"%ld %ld %ld %ld", &dummy, &count, &start, &last);
	if (last - count > start) start = last - count;
	while (start < last && !nntp_stat(start)) start++;
	while (start <= last) {
		if (top_base >= max_art) expand_art();
		base[top_base++] = start++;
	}
}

int
UsenetClose()
{
	extern FILE *note_fp;

	if (note_fp) fclose(note_fp);
	if (active_fp) fclose(active_fp);
	if (usenetinuse & USENNTP) close_server();
	usenetinuse = 0;
	max_active = 0;
	max_art = 0;
	curgrpidx = -1;
	if (active) {
		register i;
		for (i = 0; i < num_active; i++) {
			free(active[i].name);
			if (active[i].desc != NULL) free(active[i].desc);
		}
		free(active);
		active = NULL;
	}
	if (my_group) {
		free(my_group);
		my_group = NULL;
	}
	if (unread) {
		free(unread);
		unread = NULL;
	}
	if (arts) {
		free(arts);
		arts = NULL;
	}
	if (base) {
		free(base);
		base = NULL;
	}
	return 0;
}

int
expand_art()
{
	max_art += max_art / 2;		/* increase by 50% */
	arts = (struct header *)realloc(arts, sizeof(*arts) * max_art);
	base = (long *)realloc(base, sizeof(long) * max_art);
	return (arts != NULL && base != NULL);
}

int
expand_active()
{
	max_active += max_active / 2;		/* increase by 50% */
	active = (struct group_ent *)realloc(active,
					     sizeof(*active) * max_active);
	my_group = (int *)realloc(my_group, sizeof(int) * max_active);
	unread = (int *)realloc(unread, sizeof(int) * max_active);
	return (active != NULL && my_group != NULL && unread != NULL);
}

static void
setupnewsgrp(idx)
	register idx;
{
	register i;

	curgrpidx = idx;
	canpost = unread[my_group[idx]] <= privlevel;
	for (i = 0; i < local_top; i++) unread[i] = -2;
	add_newsrc_group(active[my_group[idx]].name);
	LOGIT(LOG_INFO, "Change UsenetGroup to \"%s\"", active[my_group[idx]].name);
	return;
}

void
UsenetBrowseAreas(flag)
	int flag;
{
	register i, j;
	register char *ptr;
	int lncnt, rval, fidx[MAXSCRLINES*2];
	int rdl[MAXCONFLINES], wrl[MAXCONFLINES], *rd_level;
	char *line, *mask[MAXCONFLINES];
	regex *rgc;
	extern char working[];

	sprintf(tmpbuf, "%s/%s", dirpath, USENETAREACONF);
	if ((ptr = loadfile(tmpbuf)) == NULL) return;
	if (UsenetOpen() < 0) return;

	pleasewait(1);
	for (i = 0; i < MAXCONFLINES &&
	     (line = strchr(ptr, '\n')) != NULL; ptr = line) {
		*line++ = '\0';
		while ((u_char)*ptr <= 0x20 && *ptr) ptr++;
		if (!*ptr || *ptr == '#') continue;
		mask[i] = ptr;
		while ((u_char)*ptr > 0x20) ptr++;
		if (*ptr) for (*ptr++ = 0; (u_char)*ptr <= 0x20 && *ptr; ptr++);
		if (*ptr) {
			if (!parse_access(ptr, &rdl[i], &wrl[i])) continue;
		} else	rdl[i] = 0, wrl[i] = MAXPRIVLEVEL;
		i++;
	}
	if ((rd_level = (int *)malloc(sizeof(int) * num_active)) == NULL) {
		warning(MSG_OUTOFMEMORY, 1);
		return;
	}
	lncnt = i;
	for (i = 0; i < num_active; i++)
		rd_level[i] = -1, unread[i] = MAXPRIVLEVEL;
	for (j = 0; j < lncnt; j++) {
		if ((rgc = recomp(mask[j])) == NULL) {
			badline(USENETAREACONF, mask[j]);
			continue;
		}
		for (i = rval = 0; i < num_active; i++) {
			if (!(termflags & RIPTERM) && !(i % 100))
				putstr("\b%c", working[++rval & 3]);
			if (!reexec(rgc, active[i].name)) continue;
			if (rd_level[i] < rdl[j]) {
				rd_level[i] = rdl[j];
				unread[i] = wrl[j];
			}
		}
		refree(rgc);
	}
	for (local_top = i = 0; i < num_active; i++) {
		if (rd_level[i] >= 0 && rd_level[i] <= privlevel &&
		    isgrpright(GSID_USENET, active[i].name, GRF_S)) {
			active[i].flag &= ~NOTGOT;
			my_group[local_top++] = i;
		}
	}
	free(rd_level);
	if (!local_top) {
		warning(MSG_NOGROUPAVAIL, 1);
		return;
	}
	if (nchr) {
		ptr = getstr(0, 1, ECHODISABLE);
		for (i = 0; i < local_top; i++)
			if (!strcmp(active[my_group[i]].name, ptr)) {
				setupnewsgrp(i);
				return;
			}
		warning(MSG_NOSUCHAREA, 1);
		return;
	}
	if (flag) {
		if ((i = gonewstree("")) > 0) setupnewsgrp(--i);
		return;
	}
	lncnt = 1;
	for (i = 0; i < local_top; i++) {
		if (lncnt == 1) {
			lncnt += 2;
			j = 4;
			memset(fidx, 0, sizeof(fidx));
			fidx[0] = -1;
			snprintf(tmpbuf, MAXSTRINGLENGTH-8, "%s %s",
				 sysmsg(MSG_AFILEAREAHEAD), makeshowpath(dirpath));
			if (termflags & (ANSITERM | RIPTERM))
				putstr("\033[H\033[J\033[1;37;44m %-72.72s %5d\033[K\033[40m\n\n", tmpbuf, local_top);
			else {
				clearline();
				putstr("\n %s (%5d)\n\n", tmpbuf, local_top);
			}
		}
		fidx[j++] = i+1;
		if (active[my_group[i]].desc != NULL)
			ptr = active[my_group[i]].desc;
		else	ptr = active[my_group[i]].name;
		if (termflags & (ANSITERM | RIPTERM))
			putstr("\033[1;33m%4d \033[0;36m%-34.34s", i+1, ptr);
		else	putstr("%4d) %-33.33s", i+1, ptr);
		if (!(i & 1)) {
			putchr(' ');
			continue;
		}
		putchr('\n');
more_again:
		rval = morenum(MSG_SELECTFAREA, &lncnt, local_top);
		if (rval == -1) return;
		if (rval > 0) {
			if (rval > local_top) {
				rval = fidx[rval-local_top-1];
				if (rval < 0) return;
				if (!rval) {
					lncnt = scrlines;
					goto more_again;
				}
			}
			setupnewsgrp(rval-1);
			return;
		}
		if (rval == -2)	i -= (scrlines - 2) * 4;
		else if (rval == -3) i = local_top - ((scrlines - 2) * 2) - 1;
		else if (rval == -4) i = -1;
		if (i < -1) i = -1;
	}
	if (local_top & 1) putchr('\n');
	lncnt = scrlines;
	goto more_again;
}

void
UsenetAreaTree()
{
	UsenetBrowseAreas(TRUE);
}

void
UsenetAreaChange()
{
	UsenetBrowseAreas(FALSE);
}

static char *
newssubdir(buf, group, newstop)
	char *buf;
	register char *group, *newstop;
{
	register n = 0, i = 0;
	char *av[20], buf2[256];

	group = strcpy(buf, group);
	newstop = strcpy(buf2, newstop);

	while ((newstop = strtok(newstop, "./")) != NULL && n < 20) {
		av[n++] = newstop;
		newstop = NULL;
	}
	while ((group = strtok(group, "./")) != NULL && i < n) {
		if (strcmp(group, av[i])) break;
		group = NULL;
		i++;
	}
	if (i < n) return NULL;
	return group;
}

int
isitgroup(grp, subgrp)
	char *grp, *subgrp;
{
	register i;

	if ((i = strlen(grp) - strlen(subgrp)) < 0) return FALSE;
	return (!strcmp(&grp[i], subgrp));
}

int
gonewstree(newstop)
	register char *newstop;
{
	register i, n;
	register char *ptr, *ptr2;
	int lncnt, rval, fidx[MAXSCRLINES];
	struct {
		int idx;
		int desc;
	} grptop[MAXCONFLINES];
	char buf[256], buf2[256];

	for (i = n = 0; n < MAXCONFLINES && i < local_top; i++) {
		ptr = active[my_group[i]].name;
		grptop[n].desc = FALSE;
		if (!n) {
			if ((ptr = newssubdir(buf, ptr, newstop)) != NULL) {
				if (isitgroup(active[my_group[i]].name, ptr))
					grptop[n].desc = TRUE;
				grptop[n++].idx = my_group[i];
			}
		} else {
			if ((ptr = newssubdir(buf, ptr, newstop)) != NULL) {
				ptr2 = active[grptop[n-1].idx].name;
				if ((ptr2 = newssubdir(buf2, ptr2, newstop)) != NULL) {
					if (strcmp(ptr, ptr2)) {
						if (isitgroup(active[my_group[i]].name, ptr))
							grptop[n].desc = TRUE;
						grptop[n++].idx = my_group[i];
					}
				}
			}
		}
	}
	n++;
select_again:
	lncnt = 1;
	for (i = 0; i < n; i++) {
		if (lncnt == 1) {
			lncnt += 2;
			memset(fidx, 0, sizeof(fidx));
			fidx[0] = -1;
			if (*newstop) ptr = newstop;
			else ptr = makeshowpath(newstop);
			snprintf(buf, MAXSTRINGLENGTH-8, "%s %s",
				 sysmsg(MSG_AFILEAREAHEAD), ptr);
			if (termflags & (ANSITERM | RIPTERM))
				putstr("\033[H\033[J\033[1;37;44m %-72.72s %4d\033[K\033[40m\n\n", buf, n);
			else {
				clearline();
				putstr("\n %s %4d\n\n", buf, n);
			}
		}
		fidx[lncnt-1] = i+1;
		if (i > 0) {
			ptr = newssubdir(buf2, active[grptop[i-1].idx].name, newstop);
			assert(ptr != NULL);
			if (grptop[i-1].desc) {
				if ((ptr2 = active[grptop[i-1].idx].desc) == NULL)
					ptr2 = "";
			} else	ptr2 = sysmsg(MSG_CONTINUEGROUP);
		} else {
			ptr = "..";
			ptr2 = "";
		}
		if (termflags & (ANSITERM | RIPTERM))
			putstr("\033[1;37m%4d \033[33m%-16.16s  \033[0;36m%-.55s\n",
			       i+1, ptr, ptr2);
		else	putstr("%4d) %-16.16s  %-.55s\n", i+1, ptr, ptr2);
more_again:
		rval = morenum(MSG_SELECTFAREA, &lncnt, n);
		if (rval == -1) goto quit_list;
		if (rval > 0) {
			if (rval > n) {
				rval = fidx[(rval-n-1)/2];
				if (rval < 0) goto quit_list;
				if (!rval) {
					lncnt = scrlines;
					goto more_again;
				}
			}
			--rval;
			if (!rval) return 0;
			ptr2 = newssubdir(buf2, active[grptop[rval-1].idx].name, newstop);
			assert(ptr2 != NULL);
			if (*newstop) {
				sprintf(buf, "%s.%s", newstop, ptr2);
				ptr2 = buf;
			}
			if ((i = gonewstree(ptr2)) < 0)
				LOGIT(LOG_ERR, "can't newstree \"%s\": %m", ptr2);
			if (i <= 0) goto select_again;
			return i;
		}
		if (rval == -2)	i -= (scrlines - 2) * 2;
		else if (rval == -3) i = n - (scrlines - 2) - 1;
		else if (rval == -4) i = -1;
		if (i < -1) i = -1;
	}
	lncnt = scrlines;
	goto more_again;

quit_list:
	for (i = 0; i < local_top; i++)
		if (!strcmp(active[my_group[i]].name, newstop)) return i+1;
	return 0;
}

/* Return Group Directory Path or NULL */
char *
begin_usenet(postflag)
	int postflag;
{
	static char group_path[LEN];

	if (!postflag && ArtReadLimit && UserArtRead >= ArtReadLimit) {
		if (termflags & (ANSITERM | RIPTERM)) putstr("\033[1;31m");
		putstr("%s\n", sysmsg(MSG_ARTREADLIMIT));
		anykey();
		return NULL;
	}
	if (curgrpidx == -1) {
		warning(MSG_SELECTAREAFIRST, 1);
		return NULL;
	}
	if (UsenetOpen() < 0) return NULL;

	if (curgrpidx == -1) {
		warning(MSG_SELECTAREAFIRST, 1);
		return NULL;
	}
	if (!isgrpright(GSID_USENET, active[my_group[curgrpidx]].name, GRF_R)) {
		nopermission(GRF_R);
		return NULL;
	}

	make_group_path(active[my_group[curgrpidx]].name, group_path);

	/* update index file */
	index_group(active[my_group[curgrpidx]].name, group_path);
	read_newsrc_line(active[my_group[curgrpidx]].name); /* NEWSRC */
	if (top_base <= 0) warning(MSG_NOARTICLES, 1);
	return group_path;
}

/* Show Article in line */
void
showartline(i)
	int i;	/* accept only base article! */
{
	int n;
	char resps[6], new_resps, *p;
	char from_name[LEN], from_addr[LEN];

	if (new_responses(i))	new_resps = '+';
	else			new_resps = ' ';
	/* number of responses */
	if (n = nresp(i)) sprintf(resps, "%-4d", n);
	else	strcpy(resps, "    ");
	n = base[i];
	parsefrom(arts[n].from, from_addr, from_name);
	if (*from_name) p = from_name;
	else		p = from_addr;
	if (termflags & (ANSITERM | RIPTERM))
		putstr("\033[1;33m%-4d\033[31m%c%s \033[32m%-18.18s \033[0;36m%-50.50s\n",
		       i+1, new_resps, resps, p, arts[n].subject);
	else	putstr("%-4d%c%s %-18.18s %-50.50s\n",
		       i+1, new_resps, resps, p, arts[n].subject);
}

void
art_search(sfunc, pnum, sbuf)
	int (*sfunc)(), pnum;
	char *sbuf;
{
	register i, n;
	int lncnt, s_art, prevbase, begin = 2;
	char *group_path;

	if ((group_path = begin_usenet(FALSE)) == NULL || top_base <= 0)
		return;

	if (nchr) n = atoi(getstr(0, 1, ECHODISABLE));
	else n = 0;
	if (n) n--;
	if (n >= top_base) return;
	n = base[n];
	putchr('\n');
search_again:
	s_art = prevbase = -1;
	lncnt = 3;
	while ((i = (*sfunc)(n, begin)) >= 0) {
		begin = -1;
		if (s_art == -1) {
			putstr("\n\n");
			s_art = i;
		} else	if (i <= s_art) break;
		n = which_base(i);
		if (n == prevbase) {
			n = i;
			continue;
		}
		prevbase = n;
		showartline(n);
		if ((n = morenum(MSG_SELECTMESSAGE, &lncnt, top_base)) < 0) {
			if (n == -1) return;
			n = 0;
			goto search_again;
		}
		if (!n) n = i;
		else {
			n = show_page((int)base[n-1],
				      active[my_group[curgrpidx]].name, group_path) - 1;
			if (termflags & (ANSITERM | RIPTERM))
				putstr("\033[H\033[J");
			else	putchr('\n');
			lncnt = 1;
		}
	}
	if (i < 0) return;
	lncnt = scrlines;
	if ((n = morenum(MSG_SELECTMESSAGE, &lncnt, top_base)) >= 0 || n == -2) {
		if (n > 0) {
			show_page((int)base[n-1],
				  active[my_group[curgrpidx]].name, group_path);
			putchr('\n');
		}
		n = 0;
		if (termflags & (ANSITERM | RIPTERM))
			putstr("\033[H\033[J\033[1;36m%s\033[5;37m:\033[0m %s",
			       sysmsg(pnum), sbuf);
		else	putstr("\n%s: %s", sysmsg(pnum), sbuf);
		goto search_again;
	}
}

void
UsenetArtList()
{
	register i;
	int lncnt, startmsg, rval, tmp, fidx[MAXSCRLINES];
	char *group_path, line[MAXSTRINGLENGTH];

	if ((group_path = begin_usenet(FALSE)) == NULL || top_base < 1)
		return;

	if (nchr) {
		startmsg = atoi(getstr(0, 1, ECHODISABLE));
		if (startmsg < 1) startmsg = 1;
	} else	startmsg = 1;

	lncnt = 1;
	for (i = startmsg; i <= top_base; i++) {
		startmsg = 0;
		if (lncnt == 1) {
			lncnt += 4;
			memset(fidx, 0, sizeof(fidx));
			fidx[0] = -1;
			snprintf(line, MAXSTRINGLENGTH-8, "%s %s",
				 sysmsg(MSG_USENETAREAHEAD), active[my_group[curgrpidx]].name);
			if (termflags & (ANSITERM | RIPTERM))
				putstr("\033[H\033[J\033[1;37;44m %-72.72s %5d\033[K\033[40m\n\n", line, top_base);
			else {
				clearline();
				putstr("\n %s (%5d)\n\n", line, top_base);
			}
			putstr(" #        %-18.18s ", sysmsg(MSG_MSGFROM));
			putstr("%s\n", sysmsg(MSG_MSGSUBJECT));
			putstr("%s\n", sysmsg(MSG_DELIMITER));
			tmp = i;
		}
		fidx[lncnt-1] = i;
		showartline(i-1);
more_again:
		rval = morenum(MSG_SELECTMESSAGE, &lncnt, top_base);
		if (rval == -1) return;
		if (rval > 0) {
			if (rval > top_base) {
				rval = fidx[(rval-top_base-1)/2];
				if (rval < 0) return;
				if (!rval) {
					lncnt = scrlines;
					goto more_again;
				}
			}
			savescr();
			rval = show_page((int)base[rval-1],
				      active[my_group[curgrpidx]].name, group_path);
			if ((termflags & RIPTERM) &&
			    rval >= tmp && rval < tmp + (scrlines - 4)) {
				restorescr();
				lncnt = scrlines;
				goto more_again;
			}
			i = rval - (scrlines - 4) / 2;
		} else if (rval == -2) i -= (scrlines - 4) * 2;
		else if (rval == -3) i = top_base - (scrlines - 4);
		else if (rval == -4) i = 0;
		if (i < 0) i = 0;
	}
	if (!startmsg) {
		lncnt = scrlines;
		goto more_again;
	}
}

void
UsenetReader()
{
	int n;
	char *group_path;

	if ((group_path = begin_usenet(FALSE)) == NULL || top_base <= 0)
		return;

	if (nchr) n = atoi(getstr(0, 1, ECHODISABLE));
	else n = 0;
	if (n) n--;
	if (n < top_base)
		show_page((int)base[n], active[my_group[curgrpidx]].name,
			  group_path);
	if (termflags & (ANSITERM | RIPTERM)) putstr("\033[0;37m");
}

void
UsenetPost()
{
	char *group_path;
	struct stat st;

	if (ArtSizeLimit && UserArtSize >= ArtSizeLimit) {
		if (termflags & (ANSITERM | RIPTERM)) putstr("\033[1;31m");
		putstr("%s\n\n", sysmsg(MSG_ARTSIZELIMIT));
		return;
	}
	if (ArtPostLimit && UserArtPost >= ArtPostLimit) {
		if (termflags & (ANSITERM | RIPTERM)) putstr("\033[1;31m");
		putstr("%s\n\n", sysmsg(MSG_ARTPOSTLIMIT));
		return;
	}
	if ((group_path = begin_usenet(TRUE)) == NULL) return;
	if (!isgrpright(GSID_USENET, active[my_group[curgrpidx]].name, GRF_W)) {
		nopermission(GRF_W);
		return;
	}
	if (post_base(active[my_group[curgrpidx]].name)) {
		UserArtPost++;
		if (stat(writearticle, &st) == 0)
			UserArtSize += (int)st.st_size;
	}
}

void
UsenetAuthorSearch()
{
	extern int search_author();
	extern char author_search_string[];

	art_search(search_author, MSG_AUTHORSEARCHF, author_search_string);
}

void
UsenetSubjectSearch()
{
	extern int search_subject();
	extern char subject_search_string[];

	art_search(search_subject, MSG_SUBJSEARCHF, subject_search_string);
}
