/*
 * rip_* functions correspond to commands described in _RIPscrip
 * Graphics Protocol Specification_, Revision 2.A1, by TeleGrafix
 * Communications, Inc.
 *
 * ripv_* functions are equivalent to their rip_* counterparts,
 * but take an argument vector in place of an inline list.
 *
 * ripio_* functions are specific to this library.
 */
 
#ifdef NX_COMPILER_RELEASE_3_0
#include <ansi/stdio.h>
#ifdef __STRICT_BSD__
#include <bsd/strings.h>
#else
#include <ansi/string.h>
#endif
#ifdef __STRICT_ANSI__
#include <ansi/stdarg.h>
#define STDARG_RELIGION 1
#else
#include <bsd/varargs.h>
#define VARARGS_RELIGION 1
#endif
#else	/* NX_COMPILER_RELEASE_3_0 */
#include <stdio.h>
#ifdef __STRICT_BSD__
#include <strings.h>
#else
#include <string.h>
#endif
#ifdef __STRICT_ANSI__
#include <stdarg.h>
#define STDARG_RELIGION 1
#else
#include <stdarg.h>
#define VARARGS_RELIGION 1
#endif
#endif	/* NX_COMPILER_RELEASE_3_0 */


/* ----- infrastructure ----- */

static unsigned int ripio_cmd_chars;
static unsigned int ripio_width = 80;

/* [internal] output a single character */
/* #define ripio_putchar(c)	fputc(c, stdout) */
void ripio_putchar(unsigned char c)
{
	write(1, &c, 1);
}

/* [internal] output a C string; no newline added */
/* #define ripio_putstr(s)		fputs(s, stdout) */
void ripio_putstr(const unsigned char *s)
{
	write(1, s, strlen(s));
}

/* set screen width (for command wrapping) */
void ripio_set_width(unsigned int width)
{
	if (width < ripio_width && ripio_cmd_chars + 2 >= width) {
		(void)ripio_putchar('\n');
		ripio_cmd_chars = 0;
	}
	ripio_width = width;
}

/* obtain current screen width setting */
unsigned int ripio_get_width(void)
{
	return ripio_width;
}

/* [internal] begin a RIPscrip command */
void ripio_command(int baggage)
{
	if (ripio_cmd_chars == 0) {
		(void)ripio_putstr("!|");
		ripio_cmd_chars = 2;
	} else if (ripio_cmd_chars + 4 + baggage >= ripio_width) {
		(void)ripio_putstr("\n!|");
		ripio_cmd_chars = 2;
	} else {
		(void)ripio_putchar('|');
		ripio_cmd_chars++;
	}
}

/* end RIPscrip command line */
void ripio_flush(void)
{
	if (ripio_cmd_chars > 0) {
		(void)ripio_putchar('\n');
		ripio_cmd_chars = 0;
	}
}

static void ripio_megadig(unsigned int value, int digits)
{
	if (value == 0) {
		while (digits--) (void)ripio_putchar('0');
	} else {
		if (digits > 1)
			ripio_megadig(value/36, digits-1); /* recursive! */
		(void)ripio_putchar("0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ"[value%36]);
	}
}

/* [internal] output an unsigned int in base 36 */
void ripio_meganum(unsigned int value, int digits)
{
	if (digits > 0) {
		if (ripio_cmd_chars + 2 + digits >= ripio_width) {
			(void)ripio_putstr("\\\n");
			ripio_cmd_chars = 0;
		}
		ripio_megadig(value, digits);
		ripio_cmd_chars += digits;
	}
}

#ifdef	RIP_ULTRANUM	/* not tested yet! */
/* [internal] output an unsigned int in base 64 */
void ripio_ultranum(unsigned int value, int digits)
{
	int part, pos;
	char buf[8];
	static char ultra_num_table[64] = {
		'0', '1', '2', '3', '4', '5', '6', '7',
		'8', '9', 'A', 'B', 'C', 'D', 'E', 'F',
		'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N',
		'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V',
		'W', 'X', 'Y', 'Z', 'a', 'b', 'c', 'd',
		'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l',
		'm', 'n', 'o', 'p', 'q', 'r', 's', 't',
		'u', 'v', 'w', 'x', 'y', 'z', '#', '@'
	};

	if (digits < 1) return;
	if (ripio_cmd_chars + 2 + digits >= ripio_width) {
		(void)ripio_putstr("\\\n");
		ripio_cmd_chars = 0;
	}
	pos = 0;
	while (value >= 64L) {
		part = value & 0x003Fl;
		value = value >> 6;
		buf[pos++] = ultra_num_table[part];
	}
	if (value) buf[pos++] = ultra_num_table[value];
	if (pos) {
		buf[pos] = '\0';
		(void)ripio_putstr(buf);
		ripio_cmd_chars += pos;
	} else {
		(void)ripio_putchar('0');
		ripio_cmd_chars++;
	}
	/* ...and what about width of UltraNum digit? It is not documented! */
}
#endif	/* RIP_ULTRANUM */

/* [internal] output a text string */
void ripio_text(const char *string)
{
	register const char *s;
	register int quotable, length;

	if (!string || !*string) return;
	quotable = 0, length = 0;
	for (s = string; *s; length++) {
		if (*s == '!' || *s == '|' || *s == '\\') quotable++;
		s++;
	}
	if (ripio_cmd_chars+2+length+quotable >= ripio_width) {
		(void)ripio_putstr("\\\n");
		ripio_cmd_chars = 0;
	}
	if (quotable == 0 && length+2 < ripio_width) {
		(void)ripio_putstr(string);
		ripio_cmd_chars += length;
	} else {
		for (s = string; *s; s++) {
			if (ripio_cmd_chars+4 >= ripio_width) {
				(void)ripio_putstr("\\\n");
				ripio_cmd_chars = 0;
			}
			if (*s == '!' || *s == '|' || *s == '\\') {
				(void)ripio_putchar('\\');
				ripio_cmd_chars++;
			}
			(void)ripio_putchar((int)*s);
			ripio_cmd_chars++;
		}
	}
}

/* Query RIPscrip version number */
void ripio_query(void)
{
	(void)ripio_putstr("\033[!");
}

/* Disables all RIPscrip processing */
void ripio_disable(void)
{
	(void)ripio_putstr("\033[1!");
}

/* Enables RIPscrip processing */
void ripio_enable(void)
{
	(void)ripio_putstr("\033[2!");
}


/* ----- RIPscrip V2.A1 functions ----- */

/* Header command for subsequent RIPscrip sequence */
void rip_header(unsigned int flags, unsigned int revision, unsigned int res)
{
	ripio_command(8);
	(void)ripio_putchar('h');
	ripio_cmd_chars++;
	ripio_meganum(flags, 4);
	ripio_meganum(revision, 2);
	ripio_meganum(res, 2);
}

/* Sets the base math for most RIPscrip parameters */
void rip_set_base_math(unsigned int base_math)
{
	ripio_command(2);
	(void)ripio_putchar('b');
	ripio_cmd_chars++;
	ripio_meganum(base_math, 2);
}

/* Sets the base math for most RIPscrip parameters */
void rip_set_coordinate_size(unsigned int byte_size, unsigned int res)
{
	ripio_command(4);
	(void)ripio_putchar('n');
	ripio_cmd_chars++;
	ripio_meganum(byte_size, 1);
	ripio_meganum(res, 3);
}

/* Sets the coordinate for the World Frame */
void rip_set_world_frame(unsigned int x_dim, unsigned int y_dim)
{
	ripio_command(4);
	(void)ripio_putchar('r');
	ripio_cmd_chars++;
	ripio_meganum(x_dim, 2);
	ripio_meganum(y_dim, 2);
}

/* Define the size and location of the Text Window */
/* cf: Borland window(), _wscroll */
void rip_text_window(unsigned int x0, unsigned int y0, unsigned int x1,
		     unsigned int y1, unsigned int wrap, unsigned int size)
{
	ripio_command(10);
	(void)ripio_putchar('w');
	ripio_cmd_chars++;
	ripio_meganum(x0, 2);
	ripio_meganum(y0, 2);
	ripio_meganum(x1, 2);
	ripio_meganum(y1, 2);
	ripio_meganum(wrap, 1);
	ripio_meganum(size, 1);
}

/* Define the size & location of the Graphics Window */
/* cf: Borland setviewport() */
void rip_viewport(unsigned int x0, unsigned int y0, unsigned int x1,
		  unsigned int y1)
{
	ripio_command(8);
	(void)ripio_putchar('v');
	ripio_cmd_chars++;
	ripio_meganum(x0, 2);
	ripio_meganum(y0, 2);
	ripio_meganum(x1, 2);
	ripio_meganum(y1, 2);
}

/* Sets the coordinates for the Drawing Frame */
void rip_set_drawing_frame(unsigned int x_dim, unsigned int y_dim)
{
	ripio_command(4);
	(void)ripio_putchar('f');
	ripio_cmd_chars++;
	ripio_meganum(x_dim, 2);
	ripio_meganum(y_dim, 2);
}

/* Clear Graphics/Text Windows & reset to full screen */
/* cf: Borland setgraphmode() */
void rip_reset_windows(void)
{
	ripio_command(0);
	(void)ripio_putchar('*');
	ripio_cmd_chars++;
}

/* Clears Text Window to current background color */
/* cf: Borland clrscr() */
void rip_erase_window(void)
{
	ripio_command(0);
	(void)ripio_putchar('e');
	ripio_cmd_chars++;
}

/* Clear Graphics Window to current background color */
/* cf: Borland clearviewport() */
void rip_erase_view(void)
{
	ripio_command(0);
	(void)ripio_putchar('E');
	ripio_cmd_chars++;
}

/* Move text cursor to row & column in Text Window */
/* cf: Borland gotoxy() */
void rip_gotoxy(unsigned int x, unsigned int y)
{
	ripio_command(4);
	(void)ripio_putchar('g');
	ripio_cmd_chars++;
	ripio_meganum(x, 2);
	ripio_meganum(y, 2);
}

/* Move cursor to upper-left corner of Text Window */
void rip_home(void)
{
	ripio_command(0);
	(void)ripio_putchar('H');
	ripio_cmd_chars++;
}

/* Erase current line from cursor to end of line */
/* cf: Borland clreol() */
void rip_erase_eol(void)
{
	ripio_command(0);
	(void)ripio_putchar('>');
	ripio_cmd_chars++;
}

/* Set current Drawing Color for graphics */
/* cf: Borland setcolor() */
void rip_color(unsigned int color)
{
/*	ripio_command(4); */
	ripio_command(2);
	(void)ripio_putchar('c');
	ripio_cmd_chars++;
/*	ripio_meganum(color, 4); */
	ripio_meganum(color, 2);
}

/* Set background Drawing Color for graphics */
void rip_back_color(unsigned int color)
{
/*	ripio_command(4); */
	ripio_command(2);
	(void)ripio_putchar('k');
	ripio_cmd_chars++;
/*	ripio_meganum(color, 4); */
	ripio_meganum(color, 2);
}

/* Set 16-color Palette from master 64-color palette */
/* cf: Borland setallpalette() */
void rip_set_palette(unsigned int c1, unsigned int c2, unsigned int c3,
		     unsigned int c4, unsigned int c5, unsigned int c6,
		     unsigned int c7, unsigned int c8, unsigned int c9,
		     unsigned int c10, unsigned int c11, unsigned int c12,
		     unsigned int c13, unsigned int c14,unsigned int c15,
		     unsigned int c16)
{
	ripio_command(32);
	(void)ripio_putchar('Q');
	ripio_cmd_chars++;
	ripio_meganum(c1, 2);
	ripio_meganum(c2, 2);
	ripio_meganum(c3, 2);
	ripio_meganum(c4, 2);
	ripio_meganum(c5, 2);
	ripio_meganum(c6, 2);
	ripio_meganum(c7, 2);
	ripio_meganum(c8, 2);
	ripio_meganum(c9, 2);
	ripio_meganum(c10, 2);
	ripio_meganum(c11, 2);
	ripio_meganum(c12, 2);
	ripio_meganum(c13, 2);
	ripio_meganum(c14, 2);
	ripio_meganum(c15, 2);
	ripio_meganum(c16, 2);
}

/* Set color of 16-color Palette from Master Palette */
/* cf: Borland setpalette() */
void rip_one_palette(unsigned int color, unsigned int value)
{
	ripio_command(4);
	(void)ripio_putchar('a');
	ripio_cmd_chars++;
	ripio_meganum(color, 2);
	ripio_meganum(value, 2);
}

#ifdef	RIP_ULTRANUM	/* not tested yet! */

/* Set Drawing Palette entries to RGB colors */
#ifdef STDARG_RELIGION
void rip_set_drawing_palette(unsigned int num, unsigned int start,
			     unsigned int bits, ...)
#else
void rip_set_drawing_palette(va_alist)
	va_dcl
#endif
{
	va_list ap;
#ifndef STDARG_RELIGION
	unsigned int num, start, bits;

	va_start(ap);
	num = va_arg(ap, unsigned int);
	start = va_arg(ap, unsigned int);
	bits = va_arg(ap, unsigned int);
#else
	va_start(ap, bits);
#endif
	if (num > 0) {
		ripio_command((int)num * 4 + 5);
		(void)ripio_putchar('D');
		ripio_cmd_chars++;
		ripio_meganum(num, 2);
		ripio_meganum(start, 2);
		ripio_meganum(bits, 1);
		/* rgb_color */
		while (num--) ripio_ultranum(va_arg(ap, unsigned int), 4);
	}
	va_end(ap);
}

/* Set Drawing Palette entries to RGB colors */
void rip_one_drawing_palette(unsigned int entry, unsigned int bits,
			     unsigned int rgb_color)
{
	ripio_command(7);
	(void)ripio_putchar('d');
	ripio_cmd_chars++;
	ripio_meganum(entry, 2);
	ripio_meganum(bits, 1);
	ripio_ultranum(rgb_color, 4);
}

#endif	/* RIP_ULTRANUM */

/* Set the Color Drawing Mode (MAP or DIRECT RGB) */
void rip_set_color_mode(unsigned int mode, unsigned int bits)
{
	ripio_command(2);
	(void)ripio_putchar('M');
	ripio_cmd_chars++;
	ripio_meganum(mode, 1);
	ripio_meganum(bits, 1);
}

/* Set drawing mode for graphics primitives */
/* cf: Borland setwritemode() */
void rip_write_mode(unsigned int mode)
{
	ripio_command(2);
	(void)ripio_putchar('W');
	ripio_cmd_chars++;
	ripio_meganum(mode, 2);
}

/* Move the current drawing position to (X,Y) */
/* cf: Borland moveto() */
void rip_move(unsigned int x, unsigned int y)
{
	ripio_command(4);
	(void)ripio_putchar('m');
	ripio_cmd_chars++;
	ripio_meganum(x, 2);
	ripio_meganum(y, 2);
}

/* Draw text in current font/color at current spot */
/* cf: Borland outtext() */
void rip_text(const char *text_string)
{
	ripio_command(strlen(text_string));
	(void)ripio_putchar('T');
	ripio_cmd_chars++;
	ripio_text(text_string);
}

/* Draw text in current font/color at specific spot */
/* cf: Borland outtextxy() */
void rip_text_xy(unsigned int x, unsigned int y, const char *text_string)
{
	ripio_command(4 + strlen(text_string));
	(void)ripio_putchar('@');
	ripio_cmd_chars++;
	ripio_meganum(x, 2);
	ripio_meganum(y, 2);
	ripio_text(text_string);
}

/* Select current font style, orientation and size */
/* cf: Borland settextstyle() */
void rip_font_style(unsigned int font, unsigned int direction, unsigned int size,
		    unsigned int h_align, unsigned int v_align)
{
	ripio_command(8);
	(void)ripio_putchar('Y');
	ripio_cmd_chars++;
	ripio_meganum(font, 2);
	ripio_meganum(direction, 2);
	ripio_meganum(size, 2);
	ripio_meganum(h_align, 1);
	ripio_meganum(v_align, 1);
}

/* Select current font style, orientation and size */
void rip_extended_font(unsigned int direction, unsigned int size,
		       unsigned int style, unsigned int h_align,
		       unsigned int v_align, const char *font_name_string)
{
	ripio_command(8 + strlen(font_name_string));
	(void)ripio_putchar('y');
	ripio_cmd_chars++;
	ripio_meganum(direction, 2);
	ripio_meganum(size, 2);
	ripio_meganum(style, 2);
	ripio_meganum(h_align, 1);
	ripio_meganum(v_align, 1);
	ripio_text(font_name_string);	
}

/* Draws one pixel using current drawing color */
/* cf: Borland putpixel() */
void rip_pixel(unsigned int x, unsigned int y)
{
	ripio_command(4);
	(void)ripio_putchar('X');
	ripio_cmd_chars++;
	ripio_meganum(x, 2);
	ripio_meganum(y, 2);
}

/* Draw a line in the current color/line style */
/* cf: Borland line() */
void rip_line(unsigned int x0, unsigned int y0, unsigned int x1, unsigned int y1)
{
	ripio_command(8);
	(void)ripio_putchar('L');
	ripio_cmd_chars++;
	ripio_meganum(x0, 2);
	ripio_meganum(y0, 2);
	ripio_meganum(x1, 2);
	ripio_meganum(y1, 2);
}

/* Draw a rectangle in current color/line style */
/* cf: Borland rectangle() */
void rip_rectangle(unsigned int x0, unsigned int y0, unsigned int x1, unsigned int y1)
{
	ripio_command(8);
	(void)ripio_putchar('R');
	ripio_cmd_chars++;
	ripio_meganum(x0, 2);
	ripio_meganum(y0, 2);
	ripio_meganum(x1, 2);
	ripio_meganum(y1, 2);
}

/* Draw filled rectangle with fill color/pattern */
/* cf: Borland bar() */
void rip_bar(unsigned int x0, unsigned int y0, unsigned int x1, unsigned int y1)
{
	ripio_command(8);
	(void)ripio_putchar('B');
	ripio_cmd_chars++;
	ripio_meganum(x0, 2);
	ripio_meganum(y0, 2);
	ripio_meganum(x1, 2);
	ripio_meganum(y1, 2);
}

/* Draw circle in current color and line thickness */
/* cf: Borland circle() */
void rip_circle(unsigned int x_center, unsigned int y_center, unsigned int radius)
{
	ripio_command(6);
	(void)ripio_putchar('C');
	ripio_cmd_chars++;
	ripio_meganum(x_center, 2);
	ripio_meganum(y_center, 2);
	ripio_meganum(radius, 2);
}

/* Draw elliptical arc in current color/line style */
/* cf: Borland ellipse() */
void rip_oval(unsigned int x, unsigned int y, unsigned int st_ang,
	      unsigned int end_ang, unsigned int x_rad, unsigned int y_rad)
{
	ripio_command(12);
	(void)ripio_putchar('O');
	ripio_cmd_chars++;
	ripio_meganum(x, 2);
	ripio_meganum(y, 2);
	ripio_meganum(st_ang, 2);
	ripio_meganum(end_ang, 2);
	ripio_meganum(x_rad, 2);
	ripio_meganum(y_rad, 2);
}

/* Draw filled ellipse using current color/pattern */
/* cf: Borland fillellipse() */
void rip_filled_oval(unsigned int x_center, unsigned int y_center,
		     unsigned int x_rad, unsigned int y_rad)
{
	ripio_command(8);
	(void)ripio_putchar('o');
	ripio_cmd_chars++;
	ripio_meganum(x_center, 2);
	ripio_meganum(y_center, 2);
	ripio_meganum(x_rad, 2);
	ripio_meganum(y_rad, 2);
}

/* Draw circular arc in current color/line thickness */
/* cf: Borland arc() */
void rip_arc(unsigned int x, unsigned int y, unsigned int st_ang,
	     unsigned int end_ang, unsigned int radius)
{
	ripio_command(10);
	(void)ripio_putchar('A');
	ripio_cmd_chars++;
	ripio_meganum(x, 2);
	ripio_meganum(y, 2);
	ripio_meganum(st_ang, 2);
	ripio_meganum(end_ang, 2);
	ripio_meganum(radius, 2);
}

/* Draw an elliptical arc */
/* (what's the difference between this and rip_oval()?) */
void rip_oval_arc(unsigned int x, unsigned int y, unsigned int st_ang,
		  unsigned int e_ang, unsigned int radx, unsigned int rady)
{
	ripio_command(12);
	(void)ripio_putchar('V');
	ripio_cmd_chars++;
	ripio_meganum(x, 2);
	ripio_meganum(y, 2);
	ripio_meganum(st_ang, 2);
	ripio_meganum(e_ang, 2);
	ripio_meganum(radx, 2);
	ripio_meganum(rady, 2);
}

/* Draws a circular pie slice */
/* cf: Borland pieslice() */
void rip_pie_slice(unsigned int x, unsigned int y, unsigned int start_ang,
		   unsigned int end_ang, unsigned int radius)
{
	ripio_command(10);
	(void)ripio_putchar('I');
	ripio_cmd_chars++;
	ripio_meganum(x, 2);
	ripio_meganum(y, 2);
	ripio_meganum(start_ang, 2);
	ripio_meganum(end_ang, 2);
	ripio_meganum(radius, 2);
}

/* Draws an elliptical pie slice */
/* cf: Borland sector() */
void rip_oval_pie_slice(unsigned int x, unsigned int y, unsigned int st_ang,
			unsigned int e_ang, unsigned int radx, unsigned int rady)
{
	ripio_command(12);
	(void)ripio_putchar('i');
	ripio_cmd_chars++;
	ripio_meganum(x, 2);
	ripio_meganum(y, 2);
	ripio_meganum(st_ang, 2);
	ripio_meganum(e_ang, 2);
	ripio_meganum(radx, 2);
	ripio_meganum(rady, 2);
}

/* Draw a bezier curve */
void rip_bezier(unsigned int x1, unsigned int y1, unsigned int x2,
		unsigned int y2, unsigned int x3, unsigned int y3,
		unsigned int x4, unsigned int y4, unsigned int cnt)
{
	ripio_command(18);
	(void)ripio_putchar('Z');
	ripio_cmd_chars++;
	ripio_meganum(x1, 2);
	ripio_meganum(y1, 2);
	ripio_meganum(x2, 2);
	ripio_meganum(y2, 2);
	ripio_meganum(x3, 2);
	ripio_meganum(y3, 2);
	ripio_meganum(x4, 2);
	ripio_meganum(y4, 2);
	ripio_meganum(cnt, 2);
}

/* Draw poly-bizier curve (multi-segmented) */
void rip_poly_bizier(unsigned int num, ...)
{
	va_list ap;
	va_start(ap, num);
	if (num > 0) {
		ripio_command((int)num * 6 + 2);
		(void)ripio_putchar('z');
		ripio_cmd_chars++;
		ripio_meganum(num, 2);
		while (num--) {
			ripio_meganum(va_arg(ap, unsigned int), 2); /* count */
			ripio_meganum(va_arg(ap, unsigned int), 2); /* x_base */
			ripio_meganum(va_arg(ap, unsigned int), 2); /* y_base */
		}
	}
	va_end(ap);
}

/* Draw polygon in current color/line-style */
void rip_polygon(unsigned int npoints, ...)
{
	va_list ap;
	va_start(ap, npoints);
	if (npoints > 1) {	/* min 2 points */
		ripio_command((int)npoints * 4 + 2);
		(void)ripio_putchar('P');
		ripio_cmd_chars++;
		ripio_meganum(npoints, 2);
		while (npoints--) {
			ripio_meganum(va_arg(ap, unsigned int), 2); /* xn */
			ripio_meganum(va_arg(ap, unsigned int), 2); /* yn */
		}
	}
	va_end(ap);
}

/* Draw filled polygon in current color/fill pattern */
/* cf: Borland fillpoly() */
void rip_fill_polygon(unsigned int npoints, ...)
{
	va_list ap;
	va_start(ap, npoints);
	if (npoints > 1) {	/* min 2 points */
		ripio_command((int)npoints * 4 + 2);
		(void)ripio_putchar('p');
		ripio_cmd_chars++;
		ripio_meganum(npoints, 2);
		while (npoints--) {
			ripio_meganum(va_arg(ap, unsigned int), 2); /* xn */
			ripio_meganum(va_arg(ap, unsigned int), 2); /* yn */
		}
	}
	va_end(ap);
}

/* Draw a Poly-Line (multi-faceted line) */
/* cf: Borland drawpoly() */
void rip_polyline(unsigned int npoints, ...)
{
	va_list ap;
	va_start(ap, npoints);
	if (npoints > 1) {	/* min 2 points */
		ripio_command((int)npoints * 4 + 2);
		(void)ripio_putchar('l');
		ripio_cmd_chars++;
		ripio_meganum(npoints, 2);
		while (npoints--) {
			ripio_meganum(va_arg(ap, unsigned int), 2); /* xn */
			ripio_meganum(va_arg(ap, unsigned int), 2); /* yn */
		}
	}
	va_end(ap);
}

/* Flood fill screen area with current fill settings */
/* cf: Borland floodfill() */
void rip_fill(unsigned int x, unsigned int y, unsigned int border)
{
	ripio_command(8);
	(void)ripio_putchar('F');
	ripio_cmd_chars++;
	ripio_meganum(x, 2);
	ripio_meganum(y, 2);
	ripio_meganum(border, 4);
}

/* Defines a line style and thickness */
/* cf: Borland setlinestyle() */
void rip_line_style(unsigned int odd_draw, unsigned int style,
		    unsigned int user_pat, unsigned int thick)
{
	ripio_command(8);
	(void)ripio_putchar('=');
	ripio_cmd_chars++;
	ripio_meganum(odd_draw, 1);
	ripio_meganum(style, 1);
	ripio_meganum(user_pat, 4);
	ripio_meganum(thick, 2);
}

/* Set current fill style (predefined) & fill color */
/* cf: Borland setfillstyle() */
void rip_fill_style(unsigned int pattern, unsigned int color)
{
/*	ripio_command(6); */
	ripio_command(4);
	(void)ripio_putchar('S');
	ripio_cmd_chars++;
	ripio_meganum(pattern, 2);
/*	ripio_meganum(color, 4); */
	ripio_meganum(color, 2);
}

/* Set user-definable (custom) fill pattern/color */
/* cf: Borland setfillpattern() */
void rip_fill_pattern(unsigned int c1, unsigned int c2, unsigned int c3,
		      unsigned int c4, unsigned int c5, unsigned int c6,
		      unsigned int c7, unsigned int c8, unsigned int col)
{
	ripio_command(20);
	(void)ripio_putchar('s');
	ripio_cmd_chars++;
	ripio_meganum(c1, 2);
	ripio_meganum(c2, 2);
	ripio_meganum(c3, 2);
	ripio_meganum(c4, 2);
	ripio_meganum(c5, 2);
	ripio_meganum(c6, 2);
	ripio_meganum(c7, 2);
	ripio_meganum(c8, 2);
	ripio_meganum(col, 4);
}

/* End of RIPscrip Scene */
void rip_no_more(void)
{
	ripio_command(0);
	(void)ripio_putchar('#');
	ripio_cmd_chars++;
}

/* Defines a rectangular hot mouse region */
void rip_mouse(unsigned int num, unsigned int x0, unsigned int y0,
	       unsigned int x1, unsigned int y1, unsigned int clk,
	       unsigned int clr, unsigned int res, const char *text)
{
	ripio_command(18 + strlen(text));
	(void)ripio_putstr("1M");
	ripio_cmd_chars += 2;
	ripio_meganum(num, 2);
	ripio_meganum(x0, 2);
	ripio_meganum(y0, 2);
	ripio_meganum(x1, 2);
	ripio_meganum(y1, 2);
	ripio_meganum(clk, 1);
	ripio_meganum(clr, 1);
	ripio_meganum(res, 5);
	ripio_text(text);
}

/* Destroys all previously defined hot mouse regions */
void rip_kill_mouse_fields(void)
{
	ripio_command(1);
	(void)ripio_putstr("1K");
	ripio_cmd_chars += 2;
}

/* Destroys any Mouse Fields inside a region */
void rip_kill_enclosed_mouse_fields(unsigned int x0, unsigned int y0,
				    unsigned int x1, unsigned int y1,
				    unsigned int flags)
{
	ripio_command(13);
	(void)ripio_putstr("1k");
	ripio_cmd_chars += 2;
	ripio_meganum(x0, 2);
	ripio_meganum(y0, 2);
	ripio_meganum(x1, 2);
	ripio_meganum(y1, 2);
	ripio_meganum(flags, 4);
}

/* Define a rectangular text region */
void rip_begin_text(unsigned int x1, unsigned int y1,
		    unsigned int x2, unsigned int y2, unsigned int res)
{
	ripio_command(11);
	(void)ripio_putstr("1T");
	ripio_cmd_chars += 2;
	ripio_meganum(x1, 2);
	ripio_meganum(y1, 2);
	ripio_meganum(x2, 2);
	ripio_meganum(y2, 2);
	ripio_meganum(res, 2);
}

/* Display a line of text in rectangular text region */
void rip_region_text(unsigned int justify, const char *text_string)
{
	ripio_command(2 + strlen(text_string));
	(void)ripio_putstr("1t");
	ripio_cmd_chars += 2;
	ripio_meganum(justify, 1);
	ripio_text(text_string);
}

/* End a rectangular text region */
void rip_end_text(void)
{
	ripio_command(1);
	(void)ripio_putstr("1E");
	ripio_cmd_chars += 2;
}

/* Copy rectangular image to clipboard (as icon) */
void rip_get_image(unsigned int x0, unsigned int y0, unsigned int x1,
		   unsigned int y1, unsigned int clipboard_no)
{
	ripio_command(10);
	(void)ripio_putstr("1C");
	ripio_cmd_chars += 2;
	ripio_meganum(x0, 2);
	ripio_meganum(y0, 2);
	ripio_meganum(x1, 2);
	ripio_meganum(y1, 2);
	ripio_meganum(clipboard_no, 1);
}

/* Pastes the clipboard contents onto the screen */
void rip_put_image(unsigned int x, unsigned int y, unsigned int clipboard_no,
		   unsigned int mode, unsigned int res)
{
	ripio_command(8);
	(void)ripio_putstr("1P");
	ripio_cmd_chars += 2;
	ripio_meganum(x, 2);
	ripio_meganum(y, 2);
	ripio_meganum(clipboard_no, 1);
	ripio_meganum(mode, 1);
	ripio_meganum(res, 1);
}

/* Write contents of the clipboard (icon) to disk */
void rip_write_icon(unsigned int clipboard_no, const char *filename)
{
	ripio_command(2 + strlen(filename));
	(void)ripio_putstr("1W");
	ripio_cmd_chars += 2;
	ripio_meganum(clipboard_no, 1);
	ripio_text(filename);
}

/* Loads and displays a disk-based icon to screen */
void rip_load_icon(unsigned int x, unsigned int y, unsigned int clipboard_no,
		   unsigned int mode, unsigned int clipboard, 
		   unsigned int res, const char *filename)
{
	ripio_command(10 + strlen(filename));
	(void)ripio_putstr("1I");
	ripio_cmd_chars += 2;
	ripio_meganum(x, 2);
	ripio_meganum(y, 2);
	ripio_meganum(clipboard_no, 1);
	ripio_meganum(mode, 1);
	ripio_meganum(clipboard, 1);
	ripio_meganum(res, 2);	/* should be set to 10 */
	ripio_text(filename);
}

/* Loads and displays a disk-based bitmap to screen */
void rip_load_bitmap(unsigned int x0, unsigned int y0, unsigned int x1,
		     unsigned int y1, unsigned int clipboard_no,
		     unsigned int mode, unsigned int palette,
		     unsigned int dither, unsigned int clipboard,
		     unsigned int res, const char *filename)
{
	ripio_command(16 + strlen(filename));
	(void)ripio_putstr("1b");
	ripio_cmd_chars += 2;
	ripio_meganum(x0, 2);
	ripio_meganum(y0, 2);
	ripio_meganum(x1, 2);
	ripio_meganum(y1, 2);
	ripio_meganum(clipboard_no, 1);
	ripio_meganum(mode, 1);
	ripio_meganum(palette, 1);
	ripio_meganum(dither, 1);
	ripio_meganum(clipboard, 1);
	ripio_meganum(res, 2);
	ripio_text(filename);
}

/* Display a scalable photo type image */
void rip_image(unsigned int res, const char *filename)
{
	ripio_command(5 + strlen(filename));
	(void)ripio_putstr("1p");
	ripio_cmd_chars += 2;
	ripio_meganum(res, 4);
	ripio_text(filename);
}

/* Alter subsequent displayed image settings */
void rip_image_style(unsigned int x0, unsigned int y0, unsigned int x1,
		     unsigned int y1, unsigned int flags, unsigned int res)
{
	ripio_command(25);
	(void)ripio_putstr("1i");
	ripio_cmd_chars += 2;
	ripio_meganum(x0, 2);
	ripio_meganum(y0, 2);
	ripio_meganum(x1, 2);
	ripio_meganum(y1, 2);
	ripio_meganum(flags, 4);
	ripio_meganum(res, 12);
}

/* Button style definition */
void rip_button_style(unsigned int wid, unsigned int hgt, unsigned int orient,
		      unsigned int flags, unsigned int size, unsigned int dfore,
		      unsigned int dback, unsigned int bright, unsigned int dark,
		      unsigned int surface, unsigned int grp_no,
		      unsigned int flags2, unsigned int uline_col,
		      unsigned int corner_col, unsigned int clipboard_no,
		      unsigned int slot, unsigned int res)
{
	ripio_command(37);
	(void)ripio_putstr("1B");
	ripio_cmd_chars += 2;
	ripio_meganum(wid, 2);
	ripio_meganum(hgt, 2);
	ripio_meganum(orient, 2);
	ripio_meganum(flags, 4);
	ripio_meganum(size, 2);
	ripio_meganum(dfore, 2);
	ripio_meganum(dback, 2);
	ripio_meganum(bright, 2);
	ripio_meganum(dark, 2);
	ripio_meganum(surface, 2);
	ripio_meganum(grp_no, 2);
	ripio_meganum(flags2, 2);
	ripio_meganum(uline_col, 2);
	ripio_meganum(corner_col, 2);
	ripio_meganum(clipboard_no, 1);
	ripio_meganum(slot, 1);
	ripio_meganum(res, 4);
}

/* Define a Mouse Button */
void rip_button(unsigned int x0, unsigned int y0, unsigned int x1,
		unsigned int y1, unsigned int hotkey, unsigned int flags,
		unsigned int slot, const char *text)
{
	ripio_command(13 + strlen(text));
	(void)ripio_putstr("1U");
	ripio_cmd_chars += 2;
	ripio_meganum(x0, 2);
	ripio_meganum(y0, 2);
	ripio_meganum(x1, 2);
	ripio_meganum(y1, 2);
	ripio_meganum(hotkey, 2);
	ripio_meganum(flags, 1);
	ripio_meganum(slot, 1);
	ripio_text(text);
}

/* Define a text variable */
void rip_define(unsigned int flags, unsigned int res, const char *text)
{
	ripio_command(6 + strlen(text));
	(void)ripio_putstr("1D");
	ripio_cmd_chars += 2;
	ripio_meganum(flags, 3);
	ripio_meganum(res, 2);
	ripio_text(text);
}

/* Defines a scroller interface object */
void rip_scroller(unsigned int id, unsigned int x0, unsigned int y0,
		  unsigned int x1, unsigned int y1, unsigned int entries,
		  unsigned int cur_ent, unsigned int page_size,
		  unsigned int flags, unsigned int size, unsigned int bright,
		  unsigned int dark, unsigned int surface, unsigned int corner,
		  unsigned int res, const char *host_string)
{
	ripio_command(40 + strlen(host_string));
	(void)ripio_putstr("1S");
	ripio_cmd_chars += 2;
	ripio_meganum(id, 1);
	ripio_meganum(x0, 2);
	ripio_meganum(y0, 2);
	ripio_meganum(x1, 2);
	ripio_meganum(y1, 2);
	ripio_meganum(entries, 4);
	ripio_meganum(cur_ent, 4);
	ripio_meganum(page_size, 4);
	ripio_meganum(flags, 4);
	ripio_meganum(size, 2);
	ripio_meganum(bright, 2);
	ripio_meganum(dark, 2);
	ripio_meganum(surface, 2);
	ripio_meganum(corner, 2);
	ripio_meganum(res, 4);
	ripio_text(host_string);
}

/* Query the contents of a text variable */
void rip_query(unsigned int mode, unsigned int window_num, unsigned int res,
	       const char *text)
{
	ripio_command(5 + strlen(text));
	(void)ripio_putstr("1\033");
	ripio_cmd_chars += 2;
	ripio_meganum(mode, 1);
	ripio_meganum(window_num, 1);
	ripio_meganum(res, 2);
	ripio_text(text);
}

/* Copy (scroll) screen region up/down */
void rip_scroll(unsigned int x0, unsigned int y0, unsigned int x1,
		     unsigned int y1, unsigned int mode, unsigned int res,
		     unsigned int dest_line)
{
	ripio_command(13);
	(void)ripio_putstr("1G");
	ripio_cmd_chars += 2;
	ripio_meganum(x0, 2);	/* must be divisible by 8 */
	ripio_meganum(y0, 2);
	ripio_meganum(x1, 2);	/* must be divisible by 8 */
	ripio_meganum(y1, 2);
	ripio_meganum(mode, 1);
	ripio_meganum(res, 1);
	ripio_meganum(dest_line, 2);
}

/* Copy a screen area to a new location (bit blit) */
void rip_copy_blit(unsigned int x0, unsigned int y0, unsigned int x1,
		   unsigned int y1, unsigned int dx0, unsigned int dy0,
		   unsigned int mode, unsigned int res)
{
	ripio_command(15);
	(void)ripio_putstr("1g");
	ripio_cmd_chars += 2;
	ripio_meganum(x0, 2);
	ripio_meganum(y0, 2);
	ripio_meganum(x1, 2);
	ripio_meganum(y1, 2);
	ripio_meganum(dx0, 2);
	ripio_meganum(dy0, 2);
	ripio_meganum(mode, 1);
	ripio_meganum(res, 1);
}

/* Playback local .RIP file */
void rip_read_scene(unsigned int res, const char *filename)
{
	ripio_command(9 + strlen(filename));
	(void)ripio_putstr("1R");
	ripio_cmd_chars += 2;
	ripio_meganum(res, 8);
	ripio_text(filename);
	ripio_flush();	/* mandatory */
}

/* Query existing information on a particular file */
void rip_file_query(unsigned int mode, unsigned int res, const char *filename)
{
	ripio_command(7 + strlen(filename));
	(void)ripio_putstr("1F");
	ripio_cmd_chars += 2;
	ripio_meganum(mode, 2);
	ripio_meganum(res, 4);
	ripio_text(filename);
}

/* Alter the current mouse cursor shape that is shown on the screen */
void rip_set_mouse_cursor(unsigned int cursor_style, unsigned int res)
{
	ripio_command(7);
	(void)ripio_putstr("1c");
	ripio_cmd_chars += 2;
	ripio_meganum(cursor_style, 2);
	ripio_meganum(res, 4);
}

/* Switch to another Text Window (activate) */
void rip_switch_text_window(unsigned int window_num, unsigned int res)
{
	ripio_command(3);
	(void)ripio_putstr("2S");
	ripio_cmd_chars += 2;
	ripio_meganum(window_num, 1);
	ripio_meganum(res, 1);
}

/* Switch to another Graphics Viewport (activate) */
void rip_switch_viewport(unsigned int viewport_num, unsigned int res)
{
	ripio_command(3);
	(void)ripio_putstr("2s");
	ripio_cmd_chars += 2;
	ripio_meganum(viewport_num, 1);
	ripio_meganum(res, 1);
}

/* Switches to a new Drawing Style Context */
void rip_switch_style(unsigned int style_num, unsigned int res)
{
	ripio_command(3);
	(void)ripio_putstr("2Y");
	ripio_cmd_chars += 2;
	ripio_meganum(style_num, 1);
	ripio_meganum(res, 1);
}

/* Sets a sequence to send host to refresh display */
void rip_set_refresh(unsigned int res, const char *refresh_string)
{
	ripio_command(5 + strlen(refresh_string));
	(void)ripio_putstr("2R");
	ripio_cmd_chars += 2;
	ripio_meganum(res, 4);
	ripio_text(refresh_string);
}

/* Causes the Client Terminal to Pause */
void rip_delay(unsigned int delay_period)
{
	ripio_command(5);
	(void)ripio_putstr("3D");
	ripio_cmd_chars += 2;
	ripio_meganum(delay_period, 4);
}

/* Set Baud Rate Emulation for Local RIP Playback */
void rip_baud_emulation(unsigned int rate_val)
{
	ripio_command(3);
	(void)ripio_putstr("3e");
	ripio_cmd_chars += 2;
	ripio_meganum(rate_val, 4);
}

/* Begins a bitmap animated motion sequence */
void rip_animate(unsigned int res, unsigned int clipboard_no, unsigned int pages,
		 unsigned int page_inc, unsigned int flags, unsigned int invis_color,
		 unsigned int res2, unsigned int x0, unsigned int y0,
		 unsigned int segments, ...)
{
	va_list ap;
	va_start(ap, segments);
	if (segments > 0) {
/*		ripio_command((int)segments * 8 + 24); */
		ripio_command((int)segments * 8 + 22);
		(void)ripio_putstr("3A");
		ripio_cmd_chars += 2;
		ripio_meganum(res, 1);
		ripio_meganum(clipboard_no, 1);
		ripio_meganum(pages, 1);
		ripio_meganum(page_inc, 2);
		ripio_meganum(flags, 4);
/*		ripio_meganum(invis_color, 4); */
		ripio_meganum(invis_color, 2);
		ripio_meganum(res2, 4);
		ripio_meganum(x0, 2);
		ripio_meganum(y0, 2);
		ripio_meganum(segments, 2);
		while (segments--) {
			ripio_meganum(va_arg(ap, unsigned int), 2); /* xn */
			ripio_meganum(va_arg(ap, unsigned int), 2); /* yn */
			ripio_meganum(va_arg(ap, unsigned int), 2); /* frames */
			ripio_meganum(va_arg(ap, unsigned int), 2); /* delay_rate */
		}
	}
	va_end(ap);
}

/* Controls animation sequences */
void rip_animation_control(unsigned int mode, unsigned int res)
{
	ripio_command(5);
	(void)ripio_putstr("3a");
	ripio_cmd_chars += 2;
	ripio_meganum(mode, 1);
	ripio_meganum(res, 3);
}

/* Enter block transfer mode with host */
void rip_enter_block_mode(unsigned int mode, unsigned int proto,
			  unsigned int file_type, unsigned int res,
			  const char *filename)
{
	ripio_command(9 + strlen(filename));
	(void)ripio_putstr("9\033");
	ripio_cmd_chars += 2;
	ripio_meganum(mode, 1);
	ripio_meganum(proto, 1);
	ripio_meganum(file_type, 2);
	ripio_meganum(res, 4);
	ripio_text(filename);
	ripio_flush();	/* mandatory */
}

/* Begin a generic UU-Encoded data block */
void rip_begin_uuencode_block(unsigned int file_type, unsigned int res)
{
	ripio_command(11);
	(void)ripio_putstr("9U");
	ripio_cmd_chars += 2;
	ripio_meganum(file_type, 2);
	ripio_meganum(res, 8);
	ripio_flush();	/* mandatory */
}

/* Begin RIF streamable image block */
void rip_display_rif_image()
{
	ripio_command(1);
	(void)ripio_putstr("9I");
	ripio_cmd_chars += 2;
	ripio_flush();	/* mandatory */
}


/* ----- vector interfaces ----- */

/* Set 16-color Palette from master 64-color palette */
/* cf: Borland setallpalette() */
void ripv_set_palette(unsigned char colors[16])
{
    register unsigned char *c;
    register int i;

    ripio_command(32);
    (void)ripio_putchar('Q');
    ripio_cmd_chars++;
    c=colors; i=15; do ripio_meganum((unsigned int)*c++, 2); while (--i>=0);
}

/* Draw polygon in current color/line-style */
void ripv_polygon(unsigned int npoints, unsigned short *polypoints)
{
    register unsigned short *p;
    register unsigned int n;
    
    ripio_command((int)npoints*4+2);
    (void)ripio_putchar('P');
    ripio_cmd_chars++;
    ripio_meganum(n=npoints, 2);
    for (p=polypoints;n>0;--n) {
	ripio_meganum(*p++, 2);	/* xn */
	ripio_meganum(*p++, 2);	/* yn */
    }
}

/* Draw filled polygon in current color/fill pattern */
/* cf: Borland fillpoly() */
void ripv_fill_polygon(unsigned int npoints, unsigned short *polypoints)
{
    register unsigned short *p;
    register unsigned int n;
    
    ripio_command((int)npoints*4+2);
    (void)ripio_putchar('p');
    ripio_cmd_chars++;
    ripio_meganum(n=npoints, 2);
    for (p=polypoints;n>0;--n) {
	ripio_meganum(*p++, 2);	/* xn */
	ripio_meganum(*p++, 2);	/* yn */
    }
}

/* Draw a Poly-Line (multi-faceted line) */
/* cf: Borland drawpoly() */
void ripv_polyline(unsigned int npoints, unsigned short *polypoints)
{
    register unsigned short *p;
    register unsigned int n;

    ripio_command((int)npoints*4+2);
    (void)ripio_putchar('l');
    ripio_cmd_chars++;
    ripio_meganum(n=npoints, 2);
    for (p=polypoints;n>0;--n) {
	ripio_meganum(*p++, 2);	/* xn */
	ripio_meganum(*p++, 2);	/* yn */
    }
}

/* Set user-definable (custom) fill pattern/color */
/* cf: Borland setfillpattern() */
void ripv_fill_pattern(unsigned char upattern[8], unsigned int color)
{
    register unsigned char *u;
    register int i;

    ripio_command(18);
    (void)ripio_putchar('s');
    ripio_cmd_chars++;
    u=upattern; i=7; do ripio_meganum((unsigned int)*u++, 2); while (--i>=0);
    ripio_meganum(color, 2);
}
