/*
 *	Copyright (c) 1994 The CAD lab of the
 *	Novosibirsk Institute of Broadcasting and Telecommunication
 *
 *	TNSDrive $Id$
 *
 *	$Log$
 *
 * Redistribution and use in source forms, with and without modification,
 * are permitted provided that this entire comment appears intact.
 *
 * THIS SOURCE CODE IS PROVIDED ``AS IS'' WITHOUT ANY WARRANTIES OF ANY KIND.
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <pwd.h>
#include <unistd.h>
#include <errno.h>
#include <time.h>
#include <ctype.h>

#include "compat.h"
#include "drive.h"
#include "getconf.h"
#include "loginfo.h"
#include "sysmsg.h"
#include "callsign.h"
#include "rematch.h"
#include "variables.h"
#include "utmpent.h"

extern struct loginfo process;

char *orgdir = NULL;
extern int errno;


int
tnslogin()
{
	int rval;
	char *ptr;
	extern char *makestrctime();

	orgdir = strdup(homedir);
	if (strcmp(process.name, BBS) != 0 && strcmp(process.name, "tns") != 0) {
		if (checktwice(0) != 0) quit(12);
		if (ischargeuser() < 0) quit(10);
		return 0;
	}
	strupr(process.name);
	if (showfile(NOLOGIN)) {
		LOGIT(LOG_WARNING, "Login into %s disabled", process.name);
		putstr("Login into %s disabled\n", process.name);
		quit(2);
	}
	if (!checknusr(&rval)) {
		LOGIT(LOG_WARNING, "Too many (%d) users", rval);
		putstr("Sorry, there are too many (%d) %s users using the system at this time\n",
		       rval, process.name);
		quit(3);
	}
	if (!checkload()) {
		LOGIT(LOG_WARNING, "Too heavy loaded system");
		putstr("Sorry, there is too heavy loaded system to use %s at this time\n",
		       process.name);
		quit(4);
	}
	switch(checktty()) {
		case 0:
			LOGIT(LOG_WARNING, "No tty conf, login anyway");
			putstr("No tty conf, login anyway\n");
			break;
		case -1:
			LOGIT(LOG_WARNING, "Restricted line `%s'", process.tty);
			putstr("You called %s in restricted line `%s'\n",
				process.name, process.tty);
			showfile(TTYALLOWED);
			quit(5);
		case -2:
			LOGIT(LOG_WARNING, "Restricted time");
			putstr("You called %s in restricted time\n",
				process.name);
			putstr("The current time is %s\n", makestrctime());
			showfile(TTYALLOWED);
			quit(6);
	}
	putstr("\n%s\n", systemid);
	initcallsign(orgdir);

	display(0, BANNER);
	proctitle("AUTH");
	if (!(rval = userauth())) {
		LOGIT(LOG_WARNING, "User name failed");
		putstr("\n\007%s\n", sysmsg(MSG_USERNAMEFAILED));
		quit(7);
	}
	if (rval < 0) {		/* New User */
		if (display(0, NEWUSERSDENY)) {
			LOGIT(LOG_WARNING, "New Users deny");
			putstr("\n\007%s\n", sysmsg(MSG_NEWUSERDENY));
			quit(8);
		}
		LOGIT(LOG_NOTICE, "Recognize as new user");
		if (!display(0, NEWUSERSWELCOME))
			putstr("\n%s\n", sysmsg(MSG_YOUARENEWUSER));
		if (!makenewuser(process.name)) {
			LOGIT(LOG_WARNING, "New User failed");
			putstr("\n\007%s\n", sysmsg(MSG_NEWUSERFAILED));
			quit(9);
		}
		putstr("\n%s\n", sysmsg(MSG_SELECTTERMINAL));
	} else {
		if ((ptr = getuserconf("WARNING", NULL)) != NULL) {
			if (*ptr == '#') ptr = sysmsg(atoi(++ptr));
			putstr("\n%s\n\n", ptr);
			putuserconf("WARNING", "");
		}
	}
	settermtype();
	if ((ptr = getsignbyname(process.name)) == NULL) ptr = process.name;
	for (rval = 0; *ptr && rval < LOGINFO_NAMELEN; rval++, ptr++) {
		if (*ptr == ' ') callsign[rval] = '.';
		else callsign[rval] = *ptr;
	}
	callsign[rval] = '\0';
	putchr('\n');
	return 1;
}

void
badpassword()
{
	char *ptr, buf[256];
	int todo;

	if ((ptr = getsysconf("BADPASSWORD", NULL)) != NULL) {
		for (todo = 0; (ptr = strtok(ptr, " ,;/-\t")) != NULL;
		     ptr = NULL)
			switch(*ptr) {
				case 'B':
				case 'b': todo |= 1; break;
				case 'N':
				case 'n': todo |= 2; break;
				case 'L':
				case 'l': todo |= 4; break;
				case 'M':
				case 'm': todo |= 8; break;
			}
	} else todo = 15;

	if (todo & 1) putuserconf(PASSWORD, "");
	else {
		sprintf(buf, "#%d", MSG_LASTLOGBAD);
		putuserconf("WARNING", buf);
	}
	if ((todo & 4) || (todo & 8)) {
		sprintf(buf, "%s: bad password", process.name);
		if (todo & 1) strcat(buf, ", account blocked");
		if (todo & 4) LOGIT(LOG_WARNING, buf);
		if (todo & 8) LOGIT(LOG_ERR, buf);
	}
	if (todo & 2) {
		if (!display(0, BADPASSWORD)) {
			putchr('\n');
			putstr(sysmsg(MSG_FORGOTPASSWORD), process.name);
			putchr('\n');
		}
	}
}

int
userauth()
{
	int retry, rval, wrong;
	char *ptr, *ptr2;

	process.name[0] = '\0';
	wrong = 0;
	retry = 6;
	while (--retry) {
		if (!askusername()) return 0;
		if ((ptr = getuserdir(process.name)) == NULL) { /* New user */
			display(0, USERNOTFOUND);
			putchr('\n');
			rval = prompt_yn(MSG_YOURNAMEIS, process.name, 'y');
			putchr('\n');
			if (!rval) {
				wrong = 0;
				continue;
			}
			return -1;
		}
		(void)strcpy(homedir, ptr);
		if (checktwice(1) != 0) quit(12);
		if (ischargeuser() < 0 ||
		    (ptr = getuserconf(PASSWORD, NULL)) == NULL) {
			LOGIT(LOG_WARNING, "Account for %s disabled", process.name);
			if (!display(0, USERDENY))
				putstr("\n%s\n", sysmsg(MSG_ACCOUNTDISABLED));
			quit(10);
		}
		rval = 3;
		do {
			if (rval != 3) putchr('\n');
			putstr("%s:", sysmsg(MSG_ENTERPASSWORD));
			ptr2 = getstr(25, 0, ECHODISABLE);
		} while (--rval && (ptr2 == NULL || *ptr2 == '\0'));
		if (strcmp((char *)crypt(ptr2, ptr), ptr) != 0) {
			wrong++;
			putstr("\007  ");
			putstr(sysmsg(MSG_WRONGPASSWORD), retry-1);
			putchr('\n');
			if (retry == 2)
				putstr("\n%s\n", sysmsg(MSG_WARNINGPASSWORD));
		} else break;
	}
	putchr('\n');
	if (!retry && wrong) {
		badpassword();
		quit(11);
	}
	if (checktwice(1) != 0) quit(12);
	saveinfo(&process);	/* process.flags MUST have HIDDENUSER */
	return retry;
}

int
makenewuser(user_name)
	char *user_name;
{
	int fd;
	struct stat st;
	char *ptr, *ptr2, tmp[1024];
	char *assignpasswd();

	if ((ptr2 = assignpasswd(0)) == NULL) return 0;

	if (USRHOMEDIR[0] != '/') {
		(void)strcpy(homedir, orgdir);
		(void)strcat(homedir, "/");
	} else	*homedir = '\0';
	(void)strcat(homedir, USRHOMEDIR);
	(void)sprintf(tmp, "%s/%s", homedir, USERLISTFILE);

	if (!(stat(homedir, &st) == 0 && (st.st_mode & S_IFMT) == S_IFDIR &&
	    (st.st_mode & S_IXUSR) && (st.st_mode & S_IWUSR) && (st.st_mode & S_IRUSR))) {
		unlink(homedir);
		if (mkdir(homedir, 0751) < 0) {
			LOGIT(LOG_ERR, "can't mkdir \"%s\": %m", homedir);
			return 0;
		}
	}
	if ((fd = open(tmp, O_WRONLY|O_APPEND, 0640)) > 0) {
		sprintf(tmp, "%s\n", user_name);
		while (write(fd, tmp, strlen(tmp)) < 0 && errno == EINTR);
		close(fd);
	}
	ptr = strcpy(tmp, user_name);
	while ((ptr = strtok(ptr, " .")) != NULL) {
		strcat(homedir, "/");
		strcat(homedir, ptr);
		if (mkdir(homedir, 0751) < 0 && errno != EEXIST) {
			LOGIT(LOG_ERR, "can't mkdir \"%s\": %m", homedir);
			return 0;
		}
		ptr = NULL;
	}
	chmod(homedir, 0750);
	putuserconf(PASSWORD, ptr2);
	return 1;
}

void
reporttwice(tty, flag)
	char *tty;
	int flag;
{
	if (!flag) LOGIT(LOG_WARNING, "%s already on %s", process.name, tty);
	else LOGIT(LOG_WARNING, "%s stay on %s after killprocs", process.name, tty);
	putstr(sysmsg(MSG_ALREADYONLINE), tty);
	putchr('\n');
}

char *
findtwice(where)
	int where;
{
	static char tty[UTMPENT_LINESIZE];

	*tty = '\0';
	if (!where) {	/* utmp */
		struct utmpent *ut;

		while ((ut = getutmpent(process.name)) != NULL) {
			if (strcmp(ut->line, process.tty)) {
				(void)strcpy(tty, ut->line);
				break;
			}
		}
		endutmpent();
	} else {	/* loginfo */
		struct loginfo *info;

		while ((info = getinfo(FALSE)) != NULL) {
			if (!strcmp(info->name, process.name) && strcmp(info->tty, process.tty)) {
				(void) strcpy(tty, info->tty);
				break;
			}
		}
		(void)closeinfo();
	}
	return (*tty ? tty : NULL);
}

int
checktwice(where)
	int where;	/* 0 - utmp; 1 - loginfo */
{
	char *p, *tty, buf[1024];
	int rep;

	if ((tty = findtwice(where)) == NULL) return 0;

	sprintf(buf, "%s/%s", homedir, CHARGEUSERCONF);
	if ((p = getfileconf(buf, "RESETLINE", NULL)) == NULL) {
		sprintf(buf, "%s/%s", homedir, USERCONF);
		p = getfileconf(buf, "RESETLINE", NULL);
	}
	if (p != NULL) {
		if (!strcasecmp(p, "No")) {
			reporttwice(tty, 0);
			return 1;
		}
		if (!strcasecmp(p, "Yes")) {
			if (killprocs(tty, 1) == 0) {	/* HangUp this line */
				LOGIT(LOG_NOTICE, "killprocs %s on %s", process.name, tty);
				sleep(2);
				if ((tty = findtwice(where)) == NULL) return 0;
				reporttwice(tty, 1);
			} else LOGIT(LOG_ERR, "can't killprocs on %s: %m", tty);
			return 1;
		}
	}
	rep = 0;
	if (!strcasecmp(getsysconf("RESETLINE", "No"), "Yes")) {
		if (killprocs(tty, 1) == 0) {	/* HangUp this line */
			LOGIT(LOG_NOTICE, "killprocs %s on %s", process.name, tty);
			sleep(2);
			if ((tty = findtwice(where)) == NULL) return 0;
			rep = 1;
		} else LOGIT(LOG_ERR, "can't killprocs on %s: %m", tty);
	}
	reporttwice(tty, rep);
	return 1;
}

int
checknusr(rval)
	int *rval;
{
	int n;
	char *ptr;

	*rval = 0;
	while (getinfo(TRUE) != NULL) (*rval)++;
	if ((ptr = getsysconf(MAXUSERS, NULL)) == NULL) return 1;
	if ((n = atoi(ptr)) == 0) return 1;
	return (*rval <= n);
}

int
checkload()
{
	char *ptr; double loadavg;

	if ((ptr = getsysconf("MAXLOADAVG", NULL)) != NULL) {
		if (getloadavg(&loadavg, 1) == 1 && (loadavg > atof(ptr))) 
			return 0;
	}
	return 1;
}

int
checktty()
{
	int flag, mnd, mn, d1, d2, hr1, hr2, mn1, mn2;
	char *ptr, *line;
	struct tm *tm;

	if ((ptr = loadfile(MAINTTYCONF)) == NULL &&
	    (ptr = loadfile(TTYCONF)) == NULL) return 0;

	tm = localtime(&process.ltime);

	mn = tm->tm_hour * 60 + tm->tm_min;
	mnd = tm->tm_wday * 10080 + tm->tm_hour * 60 + tm->tm_min;

	for (; (line = strchr(ptr, '\n')) != NULL; ptr = line) {
		*line++ = '\0';
		while ((u_char)*ptr <= 0x20 && *ptr) ptr++;
		if (!*ptr || *ptr == '#' ||
		    (ptr = strtok(ptr, " \t")) == NULL ||
		    !rematch(ptr, process.tty)) continue;
		for (ptr = NULL, flag = FALSE;
		     (ptr = strtok(ptr, " \t")) != NULL; ptr = NULL) {
			if (!strcasecmp(ptr, INFINITY))
				process.flags |= INFINITYUSER;
			else {
				if (sscanf(ptr, "%1d=%2d:%2d-%1d=%2d:%2d", &d1, &hr1, &mn1, &d2, &hr2, &mn2) == 6) {
					mn1 = d1 * 10080 + hr1 * 60 + mn1;
					mn2 = d2 * 10080 + hr2 * 60 + mn2;
					if (mnd >= mn1 && mnd <= mn2) break;
				} else
				if (sscanf(ptr, "%2d:%2d-%2d:%2d", &hr1, &mn1, &hr2, &mn2) == 4) {
					mn1 = hr1 * 60 + mn1;
					mn2 = hr2 * 60 + mn2;
					if (mn >= mn1 && mn <= mn2) break;
				}
				flag = TRUE;
			}
		}
		if (flag == TRUE && ptr == NULL) return -2;
		else return 1;
	}
	return -1;
}

int
askusername()
{
	int retry, i;
	char *ptr, *av[10], *ptr2;
	char first[LOGINFO_NAMELEN], last[LOGINFO_NAMELEN], fullname[100];
	char *ismaskedname();

	retry = 4;
next_retry:
	while (--retry) {
		putstr("\n%s: ", sysmsg(MSG_ENTERYOURNAME));
		ptr = editstr(process.name, LOGINFO_NAMELEN-1, 1);
		if (*ptr == '\0') continue;
		if ((ptr2 = getnamebysign(ptr)) != NULL) {
			ptr = ptr2;
			putstr(" (%s)", ptr);
		}
		if (!istruestr(ptr)) continue;
		ptr = strcpy(first, ptr);
		i = 0;
		while ((ptr = strtok(ptr, " \t")) != NULL) {
			if (!istruestr(ptr)) goto next_retry;
			if (!istruename(ptr)) goto next_retry;
			av[i++] = ptr;
			ptr = NULL;
		}
		if (i > 2) {
			putstr("  %s", sysmsg(MSG_USEFIRSTLAST));
			continue;
		}
		if (i < 2) {
			putstr("\n%s: ", sysmsg(MSG_ENTERLASTNAME));
			ptr = getstr(LOGINFO_NAMELEN-1, 1, ECHOENABLE);
			if (!istruestr(ptr)) continue;
			if (!istruename(ptr)) continue;
			av[i++] = strcpy(last, ptr);
		}
		if (!strcmp(av[0], av[1])) {
			putstr("  %s", sysmsg(MSG_FIRSTEQUIVLAST));
			continue;
		}
		sprintf(fullname, "%s %s", av[0], av[1]);
		if (istruestr(fullname)) {
			ptr = ismaskedname(fullname);
			if (ptr != NULL) {
				putstr("  ");
				putstr(sysmsg(MSG_ILLEGALNAME), ptr);
				putchr('\n');
				return 0;
			}
			(void)strncpy(process.name, fullname, LOGINFO_NAMELEN);
			process.name[LOGINFO_NAMELEN-1] = '\0';
			break;
		}
	}
	putchr('\n');
	return retry;
}

int
istruestr(str)
	char *str;
{
	int len;
	char *ptr = str + strlen(str) - 1;

	while (ptr >= str) {
		if (isupper(*ptr) || islower(*ptr) || isdigit(*ptr)) break;
		*ptr-- = '\0';
	}
	if (!(len = strlen(str))) return 0;
	if (len > 24) {
		putstr("  %s", sysmsg(MSG_VERYLONGNAME));
		return 0;
	}
	if (len < 3) {
		putstr("  %s", sysmsg(MSG_VERYSHORTNAME));
		return 0;
	}
	return 1;
}

int
istruename(str)
	char *str;
{
	int i;
	char *ptr;

	if (*str == '\0') return 0;
	ptr = str;
	while (*ptr) {
		if (!(isupper(*ptr) || islower(*ptr) || isdigit(*ptr))) break;
		ptr++;
	}
	if (*ptr != '\0') {
		putstr("  ");
		putstr(sysmsg(MSG_ILLEGALCHAR), *ptr);
		return 0;
	}
	strlwr(str);
	for (ptr = str, i = 0; *ptr != '\0' && i < 2; ptr++)
		if (*ptr == *(ptr+1)) i++;
		else i = 0;
	if (i == 2) {
		*(ptr+1) = '\0';
		putstr("  ");
		putstr(sysmsg(MSG_FALSESTRING), ptr-2);
		return 0;
	}
	*str = chrupr(*str);
	return 1;
}

char *
ismaskedname(name)
	char *name;
{
	char *ptr, *line;
	static char namemask[80];

	if ((ptr = loadfile(BADNAMESLIST)) == NULL) return NULL;

	for (; (line = strchr(ptr, '\n')) != NULL; ptr = line) {
		*line++ = '\0';
		while ((u_char)*ptr <= 0x20 && *ptr) ptr++;
		if (!*ptr || *ptr == '#') continue;
		if (rematch(ptr, name)) return strcpy(namemask, ptr);
	}
	return NULL;
}

settermtype()
{
	int retry;
	char *ptr, *def;

	def = getuserconf(TERM, ANSI);
	retry = 5;
	while (--retry) {
		putstr("%s: ", sysmsg(MSG_PROMPTTERMINAL));
		ptr = editstr(def, 10, 1);
		strlwr(ptr);
		if (!strcmp(ptr, RIP)) {
			strcpy(term, ptr);
			if (isripterm()) break;
			putstr("  %s\n", sysmsg(MSG_NOTRIPTERMINAL));
			continue;
		} else if (!strcmp(ptr, DUMP) || !strcmp(ptr, ANSI)) {
			strcpy(term, ptr);
			break;
		}
		putstr("  %s\n", sysmsg(MSG_BADTERMINALTYPE));
	}
	if (!retry) {
		strcpy(term, DUMP);
		putstr("%s: %s", sysmsg(MSG_PROMPTTERMINAL), term);
	}
	putchr('\n');
	putuserconf(TERM, term);
	return;
}

void
promptpasswd(pnum)
	int pnum;
{
	char *p = sysmsg(pnum);
	if (termflags & (ANSITERM | RIPTERM))
		putstr("\033[0m\033[1;36m%s\033[5;37m:\033[0m", p);
	else	putstr("%s:", p);
}

char *
assignpasswd(graph)
	int graph;
{
	register char *p;
	int tries, retry, len;
	char buf[25], salt[9];

	for (buf[0] = '\0', tries = retry = 0;;) {
		putchr('\n');
		if (graph) promptpasswd(MSG_ASSIGNPASSWD);
		else putstr("%s:", sysmsg(MSG_ASSIGNYOURPASSWORD));
		p = getstr(25, 0, ECHODISABLE);
		if (!*p) {
			if (graph) return NULL;
			if (++retry > 5) return NULL;
			warning(MSG_PASSWORDSHOULDBE, 1);
			continue;
		}
		len = strlen(p);
		if (len > 24) {
			if (++retry > 5) return NULL;
			warning(MSG_VERYLONGPASSWORD, 1);
			continue;
		}
		if (len <= 5 && ++tries < 2) {
			if (++retry > 5) return NULL;
			warning(MSG_ENTERLONGERPASSWORD, 1);
			continue;
		}
		strcpy(buf, p);
		memset(p, 0, len);
		for (p = buf; *p && islower(*p); ++p);
		if (!*p && ++tries < 2) {
			if (++retry > 5) return NULL;
			warning(MSG_LOWERCASEPASSWORD, 1);
			continue;
		}
		putchr('\n');
		if (graph) promptpasswd(MSG_RETYPEPASSWD);
		else putstr("%s:", sysmsg(MSG_RETYPEYOURPASSWORD));
		p = getstr(25, 0, ECHODISABLE);
		len = strcmp(buf, p);
		memset(p, 0, strlen(p));
		if (!len) break;
		if (++retry > 5) return NULL;
		warning(MSG_MISMATCHPASSWORD, 1);
	}
	putchr('\n');
	/* grab a random printable character that isn't a colon */
	srandom((int)time((time_t *)NULL));
	to64(&salt[0], random(), 2);
	salt[2] = '\0';
	return ((char *)crypt(buf, salt));
}

static unsigned char itoa64[] =		/* 0 ... 63 => ascii - 64 */
	"./0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz";

to64(s, v, n)
	register char *s;
	register long v;
	register int n;
{
	while (--n >= 0) {
		*s++ = itoa64[v&0x3f];
		v >>= 6;
	}
}

static FILE *tgfp = NULL;

static int
gettgent(var, length)
	char *var;
	size_t *length;
{
	int len;
	char buf[1024];
	static char val[256];

	if (tgfp == NULL) {
		sprintf(buf, "%s/%s", homedir, TIMEGETCONF);
		if ((tgfp = fopen(buf, "r+")) == NULL) return 0;
	} else rewind(tgfp);

	(void) strcpy(val, var);
	(void) strcat(val, "=");
	len = strlen(val);
	while (fgets(buf, sizeof(buf), tgfp) != NULL) {
		buf[sizeof(buf)-1] = '\0';
		if (!strncmp(buf, val, len)) {
			char *p = val;
			*length = strlen(&buf[len]);
			while ((u_char)buf[len] > 0x20)	*p++ = buf[len++];
			*p = '\0';
			return atoi(val);
		}
	}
	return 0;
}

ischargeuser()
{
	int regular, charge;
	struct stat st;
	char buf[1024];
	extern int confcached;

	sprintf(buf, "%s/%s", homedir, USERCONF);
	regular = (stat(buf, &st) == 0 && st.st_size > sizeof(PASSWORD)-1 &&
		   (st.st_mode & S_IFMT) == S_IFREG &&
		   (st.st_mode & S_IRUSR) && (st.st_mode & S_IWUSR));

	sprintf(buf, "%s/%s", homedir, CHARGEUSERCONF);
	charge = (stat(buf, &st) == 0 && st.st_size > sizeof(PASSWORD)-1 &&
		  (st.st_mode & S_IFMT) == S_IFREG &&
		  (st.st_mode & S_IRUSR) && (st.st_mode & S_IWUSR));

	if (!regular && !charge) return 0;

	if (charge) {
		char *ptr;
		int timeleft = 0;
		size_t len;

		if ((ptr = getfileconf(buf, USERTIMELEFT, NULL)) != NULL)
			timeleft = atoi(ptr);
		if (timeleft < 2) {	/* fake: try grouptime file */
			int timeget = gettgent(TIMEGET, &len);
			if (timeget < 1) timeget = DEFTIMELEFT;

			if (tgfp != NULL &&
			    (timeleft = gettgent(USERTIMELEFT, &len)) > 1) {
				flock(fileno(tgfp), LOCK_EX);
				if (timeleft > timeget) timeleft -= timeget;
				else {
					timeget = timeleft;
					timeleft = 0;
				}
				fseek(tgfp, (long) -len, SEEK_CUR);
				len--;
				sprintf(buf, "%-*d\n", len, timeleft);
				fputs(buf, tgfp);
				fflush(tgfp);
				flock(fileno(tgfp), LOCK_UN);
				fclose(tgfp);

				confcached = FALSE;
				strcpy(userconf, CHARGEUSERCONF);
				putuserconf(USERTIMELEFT, itoa(timeget));

				confcached = FALSE;
				strcpy(userconf, USERCONF);
				goto timeget_done;
			}
			if (tgfp != NULL) fclose(tgfp);

			if (!display(0, CHARGELIMIT))
				putstr("%s\n", sysmsg(MSG_CHARGEEXHAUSTED));
			return (regular ? 0 : -1);
		}
timeget_done:
		if (regular && !prompt_yn(MSG_USECHARGEACCOUNT, NULL, 'y'))
			return 0;
		strcpy(userconf, CHARGEUSERCONF);
		process.flags |= CHARGEUSER;
	}
	return 0;
}

void
make_callsign()
{
	char *ptr, buf[16];

	buf[0] = '\0';
	ptr = prompt_str(MSG_ENTERNEWCALLSIGN,
			 getsignbyname(process.name), sizeof(buf)-1);
	if (ptr != NULL) {
		strncpy(buf, ptr, sizeof(buf));
		buf[sizeof(buf)-1] = '\0';
		for (ptr = buf; *ptr; ptr++)
			if (!isalnum(*ptr)) break;
		if (*ptr != '\0' || strlen(buf) < 2 ||
		    !strcasecmp(buf, "All") || ismaskedname(buf)) {
			warning(MSG_BADNEWCALLSIGN, 1);
			return;
		}
		if ((ptr = getnamebysign(buf)) != NULL) {
			if (strcasecmp(ptr, process.name))
				warning(MSG_CALLSIGNINUSE, 1);
			return;
		}
	} else {
	    warning(MSG_BADNEWCALLSIGN, 1);
	    return;
	}
	if (addsignent(buf, process.name) < 0) warning(MSG_NOPERMISSION, 1);
	else {
		LOGIT(LOG_INFO, "Make CallSign \"%s\"", buf);
		strcpy(callsign, buf);
		putstr(" Ok\n");
	}
}
