/*
 *	Copyright (c) 1994 The CAD lab of the
 *	Novosibirsk Institute of Broadcasting and Telecommunication
 *
 *	TNSDrive $Id$
 *
 *	$Log$
 *
 * Redistribution and use in source forms, with and without modification,
 * are permitted provided that this entire comment appears intact.
 *
 * THIS SOURCE CODE IS PROVIDED ``AS IS'' WITHOUT ANY WARRANTIES OF ANY KIND.
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <ctype.h>
#include <errno.h>

#include "drive.h"
#include "getconf.h"
#include "variables.h"
#include "scandir.h"
#include "loginfo.h"
#include "utmpent.h"
#include "admin.h"

extern int errno;
char *usrdir;
char tmphome[1024];
char gfile[1024];

findusrdir()
{
	char buf[512];

	strcpy(buf, USRHOMEDIR);
	if (buf[0] != '/') sprintf(buf, "%s/%s", orgdir, USRHOMEDIR);
	usrdir = strdup(buf);
}

int
isuserexist(user)
	char *user;
{
	int flag;
	char *ptr, dirname[MAXUSRNAMELEN+1];
	struct stat st;

	strcpy(dirname, user);
	if ((ptr = strchr(dirname, ' ')) != NULL) *ptr = '/';

	flag = UF_UNKNOWN;
	(void) sprintf(tmphome, "%s/%s", usrdir, dirname);
	if (stat(tmphome, &st) == 0) {
		if ((st.st_mode & S_IFMT) == S_IFDIR) {
			(void) sprintf(tmphome, "%s/%s/%s", usrdir, dirname, CHARGEUSERCONF);
			if (stat(tmphome, &st) == 0) flag |= UF_CHARGE;

			(void) sprintf(tmphome, "%s/%s/%s", usrdir, dirname, USERCONF);
			if (stat(tmphome, &st) == 0) flag |= UF_REGULAR;

			if (flag & UF_CHARGE) {
				(void) sprintf(gfile, "%s/%s/%s", usrdir, dirname, ".guser.conf");
				if (stat(gfile, &st) == 0) flag |= UF_GROUP;
				(void) sprintf(tmphome, "%s/%s/%s", usrdir, dirname, CHARGEUSERCONF);
			}
		} else if ((st.st_mode & S_IFMT) == S_IFREG) flag = UF_GBANK;
	}
	return flag;
}

int
isuseronline(name)
	char *name;
{
	int fd;
	char *cp;
	struct loginfo *info;
	struct utmpent *ut;

	while ((info = getinfo(0)) != NULL) {
		if (!strcmp(info->name, name)) {
			closeinfo();
			return 1;
		}
	}
	if ((cp = strchr(name, ' ')) != NULL) name = ++cp;
	ut = getutmpent(name);
	endutmpent();

	return (ut != NULL);
}

void
freelist(usrlist, nusers)
	struct usrent **usrlist;
	int nusers;
{
	register i;
	if (usrlist != NULL) {
		for (i = 0; i < nusers; i++) free(usrlist[i]);
		free(usrlist);
	}
}

static char *usrbuf;

/*
 * Fast find needed values to fill user entry structure.
 */
void
scanuser(u)
	register struct usrent *u;
{
	register char *p, *t;
	register n;
	int fd, max;

	u->u_time = u->u_call = u->u_tleft = u->u_priv = 0;
	u->u_flag = isuserexist(u->u_name);
	if (u->u_flag == UF_UNKNOWN) return;

	if ((fd = open(tmphome, O_RDONLY)) < 0) return;
	while ((n = read(fd, usrbuf, USRBUFSIZE-1)) < 0 && errno == EINTR);
	close(fd);
	usrbuf[n] = '\0';

	if (u->u_flag == UF_GBANK) max = 1;
	else max = 7;
	for (p = usrbuf, n = 0; n < max && (t = strchr(p, '\n')) != NULL; p = t) {
		*t++ = '\0';
		if (*p == '\0' || *p == '#') continue;
		if (!strncmp(p, PASSWORD, sizeof(PASSWORD)-1)) {
			p += sizeof(PASSWORD)-1;
			if (*p == '=') {
				n++;
				if (*++p == '\0') u->u_flag |= UF_BLOCKED;
			}
		} else if (!strncmp(p, LASTCALLTIME, sizeof(LASTCALLTIME)-1)) {
			p += sizeof(LASTCALLTIME)-1;
			if (*p == '=') u->u_time = atoi(++p), n++;
		} else if (!strncmp(p, CALLNUMBER, sizeof(CALLNUMBER)-1)) {
			p += sizeof(CALLNUMBER)-1;
			if (*p == '=') u->u_call = atoi(++p), n++;
		} else if (!strncmp(p, USERTIMELEFT, sizeof(USERTIMELEFT)-1)) {
			p += sizeof(USERTIMELEFT)-1;
			if (*p == '=') u->u_tleft = atoi(++p), n++;
		} else if (!strncmp(p, USERPRIVLEVEL, sizeof(USERPRIVLEVEL)-1)) {
			p += sizeof(USERPRIVLEVEL)-1;
			if (*p == '=') u->u_priv = atoi(++p), n++;
		} else if (!strncmp(p, DAILYTIMELEFT, sizeof(DAILYTIMELEFT)-1)) {
			p += sizeof(DAILYTIMELEFT)-1;
			if (*p == '=') {
				n++;
				if (!atoi(++p)) u->u_flag |= UF_INFINITY;
			}
		} else if (!strncmp(p, "TERMINATOR=", 11)) {
			p += 11;
			n++;
			if (*p == '1' ||
			    !strcasecmp(p, "Yes") || !strcasecmp(p, "On"))
				u->u_flag |= UF_TERMINATOR;
		}
	}
	return;
}

int
scanusers(usrlist)
	struct usrent ***usrlist;
{
	register i, j;
	int dnf, dnl, nitems = 0, msize = FIRSTALLOC;
	char buf[1024], *ptr;
	struct dir_ent *fname, *lname;
	struct usrent u, *p, **users;

	if ((dnf = scan_dir(usrdir, &fname, dirs_only, 0)) < 0)
		return -1;

	/* allocation for scanuser() */
	if ((usrbuf = (char *)malloc(USRBUFSIZE)) == NULL) return -1;

	/* initial allocation */
	users = (struct usrent **)malloc(msize * sizeof(struct usrent *));
	if (users == NULL) return -1;

	printf("     ");
	for (i = 0; i < dnf; i++) {
		sprintf(buf, "%s/%s", usrdir, fname[i].name);
		if ((dnl = scan_dir(buf, &lname, dirs_only, 0)) < 0)
			continue;
		for (j = 0; j < dnl; j++) {
			u.u_nlen = snprintf(u.u_name, MAXUSRNAMELEN, "%s %s",
					    fname[i].name, lname[j].name);
			scanuser(&u);
			if (++nitems > msize) {
				msize += FIRSTALLOC;
				users = (struct usrent **)realloc((char *)users,
					msize * sizeof(struct usrent *));
			}
			p = (struct usrent *)malloc(USRSIZ(&u));
			if (users == NULL || p == NULL) return -1;
			p->u_time = u.u_time;
			p->u_tleft = u.u_tleft;
			p->u_call = u.u_call;
			p->u_priv = u.u_priv;
			p->u_flag = u.u_flag;
			p->u_nlen = u.u_nlen;
			memcpy(p->u_name, u.u_name, p->u_nlen + 1);
			users[nitems-1] = p;
			if (nitems % 10 == 0) {
				printf("\b\b\b\b\b%-5d", nitems);
				fflush(stdout);
			}
		}
		free_dir(&lname, dnl);
	}
	free_dir(&fname, dnf);
	free(usrbuf);
	*usrlist = users;
	return nitems;
}

int
edituser(name)
	char *name;
{
	int flag;
	char *p, buf[1024];

	if ((flag = isuserexist(name)) == UF_UNKNOWN) return -1;
	if (flag != UF_GBANK && isuseronline(name)) return -1;
	if ((p = getenv("EDITOR")) == NULL) p = DEF_USEREDITOR;
	sprintf(buf, "%s %s", p, tmphome);
	return system(buf);
}

/*
 * Delete user by name. If this user currently on-line it will be killed also.
 * Return: -1 on error else user successful deleted and killed.
 */
int
delete_user(name)
	char *name;
{
	int flag;
	char *ptr;

	if ((flag = isuserexist(name)) == UF_UNKNOWN) return -1;
	if (flag == UF_GBANK) return unlink(tmphome);
	if (isuseronline(name)) return -1;
	if ((ptr = strrchr(tmphome, '/')) == NULL) return -1;
	*ptr = '\0';
	if (deldirtree(tmphome) != 0) return -1;
	if ((ptr = strrchr(tmphome, '/')) == NULL) return -1;
	*ptr = '\0';
	return ((rmdir(tmphome) < 0 && errno != ENOTEMPTY) ? -1 : 0);
}

purge_users(days, level)
	int days, level;
{
	register i, j;
	int dnf, dnl, n = 0, d = 0;
	time_t purgetime = time(NULL) - days * 86400;
	char buf[1024], user_name[MAXUSRNAMELEN+1], *ptr;
	struct dir_ent *fname, *lname;
	extern int yflag;
	
	if ((dnf = scan_dir(usrdir, &fname, dirs_only, 0)) < 0) {
		fprintf(stderr, "can't scandir \"%s\"\n", usrdir);
		exit(1);
	}
	for (i = 0; i < dnf; i++) {
		sprintf(buf, "%s/%s", usrdir, fname[i].name);
		if ((dnl = scan_dir(buf, &lname, dirs_only, 0)) < 0)
			continue;
		for (j = 0; j < dnl; j++) {
			n++;
			snprintf(user_name, sizeof(user_name), "%s %s", fname[i].name, lname[j].name);
			if ((isuserexist(user_name) & (UF_REGULAR | UF_CHARGE)) == 0)
				continue;
			if (!yflag) {
				printf("\r* %05d: %-24.24s", n, user_name);
				fflush(stdout);
			}
			if (level >= 0) {
				if ((ptr = getfileconf(tmphome, USERPRIVLEVEL, NULL)) == NULL) {
					if (yflag)
						fprintf(stderr, "* %05d: %-24.24s Warning: %s was not found!\n",
							n, user_name, USERPRIVLEVEL);
					else	printf(" Warning: %s was not found!\n",
						       USERPRIVLEVEL);
					continue;
				}
				if (atoi(ptr) > level) continue;
			}
			if ((ptr = getfileconf(tmphome, LASTCALLTIME, NULL)) == NULL) {
				if (yflag)
					fprintf(stderr, "* %05d: %-24.24s Warning: %s was not found!\n",
						n, user_name, LASTCALLTIME);
				else	printf(" Warning: %s was not found!\n",
					       LASTCALLTIME);
				continue;
			}
			if (atoi(ptr) > purgetime) continue;
			if (!yflag) {
				printf("\n");
				getuserinfo(printf, user_name);
				printf("> Purge? ");
				if (fgets(tmphome, sizeof(tmphome)-1, stdin) == NULL) {
					free_dir(&lname, dnl);
					goto break_list;
				}
				printf("\n");
				if (tmphome[0] != 'y' && tmphome[0] != 'Y')
					continue;
			}
			sprintf(tmphome, "%s/%s", buf, lname[j].name);
			if (deldirtree(tmphome) == 0) d++;
			else	fprintf(stderr, "Can't delete dirtree \"%s\": %s\n",
					tmphome, strerror(errno));
		}
		free_dir(&lname, dnl);
		rmdir(buf);
	}
break_list:
	free_dir(&fname, dnf);
	printf("\r* %05d Done. Purge %d users. Rest %d users\n", n, d, n-d);
	fflush(stderr);
	fflush(stdout);
	return;
}
