/* Subroutines for assembly code output for the 6502.
	Copyright (C) 1987, 88, 89, 92, 93, 1994 Free Software Foundation, Inc.
	Contributed by Dave McWherter at Franklin Electronic Publishers
	(Dave_Mcwherter@franklin.com).

This file is part of GNU CC.

GNU CC is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

GNU CC is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU CC; see the file COPYING.  If not, write to
the Free Software Foundation, 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include <stdio.h>
#include "config.h"
#include "tree.h"
#include "rtl.h"
#include "regs.h"
#include "hard-reg-set.h"
#include "real.h"
#include "insn-config.h"
#include "conditions.h"
#include "insn-flags.h"
#include "output.h"
#include "insn-attr.h"
#include "flags.h"
#include "expr.h"
#include "recog.h"
#include "function.h"
#include <ctype.h>

/* The __MWERKS__ conditional is used to compile on a Macintosh using the
	Metrowerks CodeWarrior compiler. */

#ifdef	__MWERKS__
#include "jump.h"
#include "rtlanal.h"
#include "mac.h"
#endif	/* __MWERKS__ */
#include "tm-protos.h"
#ifdef	__MWERKS__
#include "print-rtl.h"
#endif	/* __MWERKS__ */

#ifndef	LOCAL_LABELS
extern	int	label_num;	/* in emit-rtl.c */
#endif	/* LOCAL_LABELS */

/******************************************************************************/
/* A list of symbols which are located in zero page.
Obtained from file 'gcc02.config' which is read in md_file_start(). */

#define	MAX_ZP	256
char	zp_symbols[MAX_ZP][100];
int	num_zp_syms;

/******************************************************************************/
/* A list of functions which do not use our registers r0-r15.
Obtained from file 'gcc02.config' which is read in md_file_start(). */

#define	MAX_NOREGS	500
char	noregs_funcs[MAX_NOREGS][100];
int	num_noregs;

/******************************************************************************/
/* A list of functions which hand-coded and which do not need a wrapper (they
take no arguments and return no results).
Obtained from file 'gcc02.config' which is read in md_file_start(). */

#define	MAX_NOWRAP	500
char	nowrap_funcs[MAX_NOWRAP][100];
int	num_nowrap;

/******************************************************************************/
/* Local prototypes: */

void	read_config_file(void);
enum machine_mode	get_reg_max_mode(int regno);
void	md_save_registers (FILE *file, unsigned long gmask);
int	md_restore_registers (FILE *file, unsigned long gmask);
int	is_leaf_function(void);

/******************************************************************************/
/* Structure to be filled in by md_compute_frame_size with register
	save masks, and offsets for the current function. */

/* (Putting this stuff into a structure seems a little silly, but I'll leave
it as it is (inherited from Sparc) since it works. DMMcW ) */

struct md_frame_info
{
	int		var_size;	/* # bytes that variables take up. */
	int		saved_regs;	/* # saved gp regs. */
	unsigned long	frame_size;	/* # bytes locals + saved registers */
	unsigned long	gmask;		/* Mask of saved gp registers. */
	int		initialized;	/* Nonzero if frame size already calculated. */
};

unsigned int	saved_regs;	/* number of saved gp regs. */

/* For each register we save, this is the widest mode used by that register. */
enum machine_mode	saved_reg_modes[N_GENERAL_REGS];

/******************************************************************************/
/* Current frame information calculated by md_compute_frame_size. */
struct md_frame_info current_frame_info;

/* Zero structure to initialize current_frame_info. */
struct md_frame_info zero_frame_info;

/******************************************************************************/
/* The current section that we're in. */

typedef enum {
	CSEG,	/* code */
	TSEG,	/* data */
	DSEG,	/* variables */
	ZSEG,	/* zp variables */
	TEXT_SEG,	/* TEXT statements */
	NO_SEG
} SEG_TYPE;

SEG_TYPE	curseg;

/******************************************************************************/

/* Name of a called function. */
char	md_call_name[256];

/* Name of current zero page symbol. */
char	md_zp_name[256];

/* Comment string to tack on to asm output. */
char	md_comment[256];

/* Label string made by md_make_label(). */
char	md_label[256];

/* Stack pointer label - either ".sp" or "sp_nnn". */
char	md_sp_label[256];
int	md_sp_labelnum;

/* Filename of file we're compiling. No paths, no suffix. */
char	md_filename[256];

/* String (asm opcode) to use to start a CSEG section. */
char	md_cseg_name[256];

/* String (asm opcode) to use to start a TSEG section. */
char	md_tseg_name[256];

/* String (asm opcode) to use to start a TEXT section. */
char	md_text_name[256];

/* Name of current function being compiled. */
char	md_func_name[256];

/* TRUE if current function is public. */
int	md_func_public;

/* Name of configuration file - obtained from option:
	-mconfig-filename
   where filename is the full name of the configuration file. */
char	*md_config_file = 0;

/* TRUE if our 'return' define_insn is used in the current function.
(GCC does NOT put a return in empty functions.) */
int	md_saw_return;

/* Current 'local' label number. */
int	md_labelnum;

/* Current contents of 6502 registers A, X, and Y. Zero if unknown. */
rtx	axy_rtx;

/* Stack offset for addresses of OP_TYPE OP_SP_OFF. Computed in md_get_op_type(). */
int	sp_off;

/* Number of arguments to current function. */
int	md_n_args;

/******************************************************************************/
/* Macros used in md_address_cost and md_go_if_legitimate_address to determine
if a register is OK as a base or index register. */

#define RTX_OK_FOR_BASE_P(X) \
(	(GET_CODE (X) == REG && REG_OK_FOR_BASE_P (X)) || \
	(	GET_CODE (X) == SUBREG && \
		GET_CODE (SUBREG_REG (X)) == REG && \
		REG_OK_FOR_BASE_P (SUBREG_REG (X)) \
	) \
)

#define RTX_OK_FOR_INDEX_P(X) \
(	(GET_CODE (X) == REG && REG_OK_FOR_INDEX_P (X)) || \
	(	GET_CODE (X) == SUBREG && \
		GET_CODE (SUBREG_REG (X)) == REG && \
		REG_OK_FOR_INDEX_P (SUBREG_REG (X)) \
	) \
)

/******************************************************************************/

#ifdef	__MWERKS__
extern	char	*asm_file_name;
#endif	/* __MWERKS__ */

void
md_init(void)
{

		/* No frame info yet. */
	current_frame_info = zero_frame_info;
		/* Init local label numbers. */
	md_labelnum = md_sp_labelnum = 0;

#ifdef	__MWERKS__
	{
		static char	cfgname[256];
		char	line[256];
		char	*p, *q;
	
		strcpy(cfgname, asm_file_name);
	
		p = cfgname;
		while (q = strchr(p, ':')) {
			p = q + 1;
		}
		
		*p = 0;
		strcat(cfgname, "gcc02.config");
		md_config_file = cfgname;
	}
#endif	/* __MWERKS__ */
	
}

/******************************************************************************/
/* TRUE if OP is a general or immediate operand:
	rn
	(rn)
	(zp symbol)
	symbol
	symbol+const
	label
	label+const
	(sp)
	(sp+n)
	#immediate
*/

int
md_gen_imm_op(rtx op, enum machine_mode mode)
{

	if (md_gen_op(op, mode)) {
		return 1;
	}

	return md_imm_op(op, mode);

}

/******************************************************************************/
/* TRUE if OP is a general operand:
	rn
	(rn)
	(zp symbol)
	symbol
	symbol+const
	label
	label+const
	(sp)
	(sp+n)
*/

int
md_gen_op(rtx op, enum machine_mode mode)
{

	if (md_gpr_op(op, mode)) {
		return 1;
	}

	if (md_mem_op(op, mode)) {
		return 1;
	}

	if (md_sp_ind_op(op, mode)) {
		return 1;
	}

	return (md_sp_off_op(op, mode));

}

/******************************************************************************/
/* TRUE if OP is a general, memory, or sp indirect operand:
	rn
	(rn)
	(zp symbol)
	symbol
	symbol+const
	label
	label+const
	(sp)
*/

int
md_gen_sp_op(rtx op, enum machine_mode mode)
{

	if (md_gpr_op(op, mode)) {
		return 1;
	}

	if (md_mem_op(op, mode)) {
		return 1;
	}

	if (md_sp_op(op, mode)) {
		return 1;
	}

	return md_imm_op(op, mode);

}

/******************************************************************************/
/* TRUE if OP is a general purpose register. */

int
md_gpr_op(rtx op, enum machine_mode mode)
{
	rtx	regop;

	if (GET_CODE(op) == SUBREG) {
		if (GET_CODE(SUBREG_REG(op)) == MEM) {
			return 0;
		}
	}

	if (GET_CODE(op) == MEM) {
		return 0;
	}

	if (!register_operand(op, mode)) {
		return 0;
	}

	if (GET_CODE(op) == SUBREG) {
		regop = SUBREG_REG(op);
	}
	else {
		regop = op;
	}

	return REG_OK_FOR_GPR(regop);

}

/******************************************************************************/
/* TRUE if OP is an address operand:
	symbol
	symbol+const
	label
	label+const
*/

int
md_addr_op(rtx op, enum machine_mode mode)
{
	rtx	op0;

	switch (GET_CODE(op)) {

	case SYMBOL_REF:
	case LABEL_REF:
		return 1;

	case CONST:
		op0 = XEXP(op, 0);
		if ((GET_CODE(op0) == PLUS) || (GET_CODE(op0) == MINUS)) {
			return (	(	GET_CODE(XEXP(op0, 0)) == SYMBOL_REF ||
						GET_CODE(XEXP(op0, 0)) == LABEL_REF
					) &&
					GET_CODE(XEXP(op0, 1)) == CONST_INT
				);
		}

	default:
		return 0;
	}

}

/******************************************************************************/
/* TRUE if OP is a symbol. */

int
md_sym_op(rtx op, enum machine_mode mode)
{

	return (GET_CODE(op) == SYMBOL_REF);

}

/******************************************************************************/
/* TRUE if OP is a memory operand:
	(rn)
	(zp symbol)
	symbol
	symbol+const
	label
	label+const
*/

int
md_mem_op(rtx op, enum machine_mode mode)
{
	rtx	op0;

	if ((GET_CODE(op) != MEM) || (GET_MODE(op) != mode)) {
		return 0;
	}

		/* Get first operand of MEM. */
	op0 = XEXP(op, 0);

	switch (GET_CODE(op0)) {

	case REG:	/* (rn) */
		return REG_OK_FOR_GPR(op0);

	case SYMBOL_REF:	/* symbol */
	case LABEL_REF:		/* label */
		return 1;

	case CONST:
			/* Get first operand of CONST. */
		op0 = XEXP(op0, 0);
		if ((GET_CODE(op0) == PLUS) || (GET_CODE(op0) == MINUS)) {
				/* symbol or label + const */
			return (	(	GET_CODE(XEXP(op0, 0)) == SYMBOL_REF ||
						GET_CODE(XEXP(op0, 0)) == LABEL_REF
					) &&
					GET_CODE(XEXP(op0, 1)) == CONST_INT
				);
		}
		return 0;

	case MEM:	/* (mem) is allowed if operand is a zp op. */
		return md_is_zp_op(op0);

	default:
		return 0;
	}

}

/******************************************************************************/
/* TRUE if OP is an immediate operand. */

int
md_imm_op(rtx op, enum machine_mode mode)
{

		/* Standard gcc code for this one. */
	return immediate_operand(op, mode);

}

/******************************************************************************/
/* TRUE if OP is a register or memory operand:
	rn
	(zp symbol)
	symbol
	symbol+const
	label
	label+const
*/

int
md_gpr_mem_op(rtx op, enum machine_mode mode)
{

	if (md_gpr_op(op, mode)) {
		return 1;
	}

	return md_mem_op(op, mode);

}

/******************************************************************************/
/* TRUE if OP is the stack pointer. */
/* Also allows frame pointer, but we don't support that. Need to allow it in
early stages of the compiler though or it will abort. It will be optimized
away later. */

int
md_sp_op(rtx op, enum machine_mode mode)
{

	if (!register_operand(op, mode)) {
		return 0;
	}

	if (GET_CODE(op) == SUBREG) {
		op = SUBREG_REG(op);
	}

	return (REG_OK_FOR_FP(op) || REG_OK_FOR_SP(op));

}

/******************************************************************************/
/* TRUE if OP is a general purpose register or immediate operand. */

int
md_gpr_imm_op(rtx op, enum machine_mode mode)
{

	if (immediate_operand(op, mode)) {
		return 1;
	}
	
	return md_gpr_op(op, mode);

}

/******************************************************************************/
/* TRUE if OP is a general purpose register, stack pointer, or frame pointer. */

int
md_gpr_sp_op(rtx op, enum machine_mode mode)
{

	if (!register_operand(op, mode)) {
		return 0;
	}

	return (REG_OK_FOR_GPR(op) || REG_OK_FOR_FP(op) || REG_OK_FOR_SP(op));

}

/******************************************************************************/
/* TRUE if OP is stack pointer indirect:
	(sp)
*/

int
md_sp_ind_op(rtx op, enum machine_mode mode)
{
	rtx	op0;

	if (GET_MODE(op) != mode) {
		return 0;
	}

	if (GET_CODE(op) == MEM) {
		op0 = XEXP(op, 0);
		if (GET_CODE(op0) == REG) {
			return (REG_OK_FOR_FP(op0) || REG_OK_FOR_SP(op0));
		}
	}
	
	return 0;

}

/******************************************************************************/
/* TRUE if OP is stack pointer indirect with an offset:
	(sp+n)
*/

int
md_sp_off_op(rtx op, enum machine_mode mode)
{
	rtx	opm0, op0, op1;

	if (GET_MODE(op) != mode) {
		return 0;
	}

	if (GET_CODE(op) == MEM) {
		opm0 = XEXP(op, 0);
		if (GET_CODE(opm0) == PLUS) {
			op0 = XEXP(opm0, 0);
			op1 = XEXP(opm0, 1);
			if (	(GET_CODE(op0) == REG) &&
				(REG_OK_FOR_FP(op0) || REG_OK_FOR_SP(op0)) &&
				(GET_CODE(op1) == CONST_INT)
			   ) {
				return 1;
			}
		}
	}

	return 0;

}

/******************************************************************************/
/* TRUE if OP is a constant integer with a value of zero. */

int
md_is_zero(rtx op)
{
	int	val;
	
	if (GET_CODE(op) != CONST_INT) {
		return 0;
	}
	
	val = INTVAL(op);

	return (val == 0);
	
}

/******************************************************************************/
/* TRUE if OP is a constant integer with a value of zero. */
/* Same as md_is_zero except has mode arg because it's used in predicates in
	the md file. */

int
md_int_0_op(rtx op, enum machine_mode mode)
{
	int	val;
	
	if (!const_int_operand(op, mode)) {
		return 0;
	}
	
	val = INTVAL(op);

	return (val == 0);
	
}

/******************************************************************************/
/* TRUE if OP is a constant integer with a value of one. */

int
md_int_1_op(rtx op, enum machine_mode mode)
{
	int	val;
	
	if (!const_int_operand(op, mode)) {
		return 0;
	}
	
	val = INTVAL(op);

	return (val == 1);
	
}

/******************************************************************************/
/* TRUE if OP is a constant integer with a value of three. */

int
md_int_3_op(rtx op, enum machine_mode mode)
{
	int	val;
	
	if (!const_int_operand(op, mode)) {
		return 0;
	}
	
	val = INTVAL(op);

	return (val == 3);
	
}

/******************************************************************************/
/* TRUE if OP is (rn). */

int
md_gpr_ind_op(rtx op)
{
	
	return (md_get_op_type(op) == OP_GPR_IND);
	
}

/******************************************************************************/
/* TRUE if OP is a constant integer and it's low 8 bits are equal to zero. */

int
md_is_low_zero(rtx op)
{

	if (	(GET_CODE(op) == CONST_INT) &&
		((INTVAL(op) & 0xFF) == 0)
	   ) {
		return 1;
	}

	return 0;

}

/* TRUE if OP is a constant integer and it's middle 8 bits are equal to zero. */

int
md_is_high_zero(rtx op)
{

	if (	(GET_CODE(op) == CONST_INT) &&
		((INTVAL(op) & 0xFF00) == 0)
	   ) {
		return 1;
	}

	return 0;

}

/* TRUE if OP is a constant integer and it's high 8 bits are equal to zero. */

int
md_is_bank_zero(rtx op)
{

	if (	(GET_CODE(op) == CONST_INT) &&
		((INTVAL(op) & 0xFF0000) == 0)
	   ) {
		return 1;
	}

	return 0;

}

/******************************************************************************/
/* TRUE if high and low bytes of OP are equal. */

int
md_are_high_low_same(rtx op)
{

	if (GET_CODE(op) != CONST_INT) {
		return 0;
	}

	return (	(INTVAL(op) & 0x00FF) ==
			((INTVAL(op) & 0xFF00) >> 8)
		);

}

/* TRUE if bank and high bytes of OP are equal. */

int
md_are_bank_high_same(rtx op)
{

	if (GET_CODE(op) != CONST_INT) {
		return 0;
	}

	return (	(INTVAL(op) & 0x00FF00) ==
			((INTVAL(op) & 0xFF0000) >> 8)
		);

}

/******************************************************************************/
/* TRUE if OP needs the 6502 Y register for addressing it's high and bank bytes:
	(rn)
	(rn+off)
	(zp symbol)
*/

int
md_need_y(rtx op)
{
	rtx	op0;

	if (GET_CODE(op) == MEM) {
		op0 = XEXP(op, 0);
		if (	(GET_CODE(op0) == REG) &&
			(REG_OK_FOR_GPR(op0))
		   ) {
			return 1;
		}
		return md_is_zp_op(op0);
	}

	return 0;

}

/******************************************************************************/
/* TRUE if OP is a valid call target. */

int
md_call_op(rtx op, enum machine_mode mode)
{

	return md_mem_op(XEXP(op, 0), mode);

}

/******************************************************************************/
/* Emit insns to move operands[1] into operands[0].

Return 1 if we have written out everything that needs to be done to
do the move. Otherwise, return 0 and the caller will try to emit the move
some other way. */

int
md_movmm(rtx *operands, enum machine_mode mode)
{
	rtx	dest = operands[0];
	rtx	src = operands[1];
	rtx	plus0, plus1, plus2, plus3, plusop;

		/* We can handle any general or immediate operand as the
		source and any general operand as the destination. */

	if (md_gen_imm_op(src, mode) || md_gen_op(dest, mode)) {
			switch (mode) {
			case QImode:
				emit_insn(gen_mv_8(dest, src));
				break;
			case HImode:
				emit_insn(gen_mv_16(dest, src));
				break;
			case SImode:
				emit_insn(gen_mv_24(dest, src));
				break;
			}
			return 1;
	}
	
		/* We can also move (rn + op) to (rn + op) in QImode or
		HImode. */
	
	if ((GET_CODE(src) == MEM) && (GET_CODE(dest) == MEM)) {
		if (	(GET_CODE(XEXP(src, 0)) == PLUS) &&
			(GET_CODE(XEXP(dest, 0)) == PLUS)
		   ) {
			plus0 = XEXP(XEXP(dest, 0), 0);
			plus2 = XEXP(XEXP(src, 0), 0);
			if ((GET_CODE(plus0) == REG) && (GET_CODE(plus2) == REG)) {
				plus1 = XEXP(XEXP(dest, 0), 1);
				plus3 = XEXP(XEXP(src, 0), 1);
				switch (GET_MODE(dest)) {
				case QImode:
					emit_insn(gen_mv_indrxr_indrxr_8(
							plus0, plus1, plus2, plus3
						));
					return 1;
				
				case HImode:
					emit_insn(gen_mv_indrxr_indrxr_16(
							plus0, plus1, plus2, plus3
						));
					return 1;
				}
			}
		}
	}

		/* If dest is memory, simplify the source by putting it
		into a register. */

	if (GET_CODE(dest) == MEM) {
		if (!reload_in_progress) {
			operands[0] = validize_mem(dest);
			operands[1] = force_reg(mode, src);
		}
	}

	/* Otherwise have caller try some other way. */

	return 0;

}

/******************************************************************************/
/* Do what is necessary for `va_start'. */

rtx
md_builtin_saveregs(tree arglist)
{

	/* Not implemented. */

}

/******************************************************************************/
/* Output 8-bit data pointed to by P for LEN bytes as asm constant bytes. */

void
md_output_ascii(FILE *file, char *p, int len)
{
	int	count, i;
	char	ch;
	
	if (curseg != TEXT_SEG) {
		fprintf(file, "\n\t%s\n", md_text_name);
		curseg = TEXT_SEG;
	}

	count = 16;
	for (i = 0; i < len; i++) {
		if (count == 16) {
			fputs("\ttext\t\"", file);
		}
		ch = *p++ & 0xFF;
		if (	((ch >= ' ') && (ch <= 0x7F)) &&
			(ch != '"') &&
			(ch != '/')
		   ) {
			putc(ch, file);
		}
		else {
			switch (ch) {
			case TARGET_CR:
				fputs("/r", file);
				break;
			
			case TARGET_NEWLINE:
				fputs("/n", file);
				break;
			
			case TARGET_TAB:
				fputs("/t", file);
				break;
			
			case TARGET_FF:
				fputs("/f", file);
				break;
			
			case 0:
				fputs("/0", file);
				break;
			
			case '/':
				fputs("//", file);
				break;
			
			default:
				fprintf(file, "/$%.2X", (ch & 0xFF));
				break;
			
			}
		}
		count--;
		if (count == 0) {
			fputs("\"\n", file);
			count = 16;
		}
	}
	
	if (count != 16) {
		fputs("\"\n", file);
		count = 16;
	}

}

/******************************************************************************/
/* Print operand X (an rtx) in assembler syntax to file FILE.
CODE is a letter or dot (`z' in `%z0') or 0 if no letter was specified.
For `%' followed by punctuation, CODE is the punctuation and X is null. */

/* Special codes for 6502:
	@	print current md_label string
	;	print current md_comment string
	#	print current md_call_name string
	$	print current md_sp_label string
	L	(LOW) print address of low 8 bits of an operand
	H	(HIGH) print address of middle 8 bits of an operand
	B	(BANK) print address of high 8 bits of an operand
	V	print address of an operand with no LOW/HIGH/BANK info
*/

void
md_print_operand(FILE *file, rtx x, int code)
{
	rtx	offset, plus;
	int	val;
#ifndef	LOCAL_LABELS
	char	label[256], *p;
#endif	/* LOCAL_LABELS */

	if (code == '@') {
#ifdef	LOCAL_LABELS
		fprintf(file, "%s", md_label);
#else
		strcpy(label, md_label);
		p = strchr(label, ':');
		if (p) {
			*p = 0;
		}
		fprintf(file, "%s", label);
#endif	/* LOCAL_LABELS */
	}
	else if (code == ';') {
		if (md_comment[0]) {
			fprintf(file, "\t;; %s", md_comment);
		}
	}
	else if (code == '#') {
		if (md_call_name[0] == '*') {
			fprintf(file, "%s", (md_call_name + 1));
		}
		else {
			fprintf(file, "%s", md_call_name);
		}
	}
	else if (code == '$') {
		fprintf(file, "%s", md_sp_label);
	}
	else if (code == 'L') {
		switch (md_get_op_type(x)) {
		case OP_GPR:
			fputs(reg_names[REGNO(x)], file);
			break;

		case OP_MEM:
			if (GET_CODE(x) == MEM) {
				output_address(XEXP(x, 0));
			}
			else {
				output_addr_const(file, x);
			}
			break;

		case OP_GPR_IND:
			fprintf(file, "(%s),0", reg_names[REGNO(XEXP(x, 0))]);
			break;
		
		case OP_ZP_IND:
			fprintf(file, "(%s),0", md_zp_name);
			break;

		case OP_SP:
			fprintf(file, "%s+0", md_sp_label);
			break;

		case OP_SP_OFF:
			fprintf(file, "%s+%d", md_sp_label, sp_off);
			break;

		case OP_IMM:
		case OP_ADDR:
			fprintf(file, "#low ");
			output_addr_const(file, x);
			break;
		}
	}
	else if (code == 'H') {
		switch (md_get_op_type(x)) {
		case OP_GPR:
			fprintf(file, "%s+1", reg_names[REGNO(x)]);
			break;

		case OP_MEM:
			if (GET_CODE(x) == MEM) {
				output_address(XEXP(x, 0));
			}
			else {
				output_addr_const(file, x);
			}
			fprintf(file, "+1");
			break;

		case OP_GPR_IND:
			fprintf(file, "(%s),y", reg_names[REGNO(XEXP(x, 0))]);
			break;

		case OP_ZP_IND:
			fprintf(file, "(%s),0", md_zp_name);
			break;

		case OP_SP:
			fprintf(file, "%s+0+1", md_sp_label);
			break;

		case OP_SP_OFF:
			fprintf(file, "%s+%d+1", md_sp_label, sp_off);
			break;

		case OP_IMM:
		case OP_ADDR:
			fprintf(file, "#high ");
			output_addr_const(file, x);
			break;
		}
	}
	else if (code == 'B') {
		switch (md_get_op_type(x)) {
		case OP_GPR:
			fprintf(file, "%s+2", reg_names[REGNO(x)]);
			break;

		case OP_MEM:
			if (GET_CODE(x) == MEM) {
				output_address(XEXP(x, 0));
			}
			else {
				output_addr_const(file, x);
			}
			fprintf(file, "+2");
			break;

		case OP_GPR_IND:
			fprintf(file, "(%s),y", reg_names[REGNO(XEXP(x, 0))]);
			break;

		case OP_ZP_IND:
			fprintf(file, "(%s),0", md_zp_name);
			break;

		case OP_SP:
			fprintf(file, "%s+0+2, md_sp_label");
			break;

		case OP_SP_OFF:
			fprintf(file, "%s+%d+2", md_sp_label, sp_off);
			break;

		case OP_IMM:
		case OP_ADDR:
			fprintf(file, "#bank ");
			output_addr_const(file, x);
			break;
		}
	}
	else if (code == 'V') {
		switch (md_get_op_type(x)) {
		case OP_GPR:
			fputs(reg_names[REGNO(x)], file);
			break;

		case OP_MEM:
			if (GET_CODE(x) == MEM) {
				output_address(XEXP(x, 0));
			}
			else {
				output_addr_const(file, x);
			}
			break;

		case OP_GPR_IND:
			fprintf(file, "(%s),0", reg_names[REGNO(XEXP(x, 0))]);
			break;

		case OP_SP:
			fprintf(file, "%s+0", md_sp_label);
			break;

		case OP_SP_OFF:
			fprintf(file, "%s+%d", md_sp_label, sp_off);
			break;

		case OP_IMM:
		case OP_ADDR:
			output_addr_const(file, x);
			break;
		}
	}
	else if (code != 0) {
		/* Undocumented flag. */
		output_operand_lossage("invalid operand output code");
	}
	else if (GET_CODE(x) == REG) {
		fputs(reg_names[REGNO(x)], file);
	}
	else if (GET_CODE(x) == MEM) {
		if (CONSTANT_P(XEXP(x, 0)))
			output_address(XEXP(x, 0));
	}
	else if (GET_CODE(x) == CONST_INT) {
		offset = x;
		val = INTVAL(offset);
		if (val & 0x00800000L) {
			if (val > 0) {
				INTVAL(offset) = val - 16777216;
			}
		}
		else {
			INTVAL(offset) &= 0x00FFFFFFL;
		}
		output_addr_const(file, offset);
	}
	else {
		if (GET_CODE(x) == CONST) {
			if (GET_CODE(XEXP(x, 0)) == PLUS) {
				plus = XEXP(x, 0);
				if (GET_CODE(XEXP(plus, 1)) == CONST_INT) {
					output_addr_const(file, XEXP(plus, 0));
					offset = XEXP(plus, 1);
					val = INTVAL(offset);
					if (val & 0x00800000L) {
						if (val > 0) {
							INTVAL(offset) = val - 16777216;
						}
					}
					else {
						INTVAL(offset) &= 0x00FFFFFFL;
						if (INTVAL(offset) != 0) {
							fprintf(file, "+");
						}
					}
					output_addr_const(file, offset);
					return;
				}
			}
		}
		output_addr_const(file, x);
	}
	
}

/******************************************************************************/
/* Used by output_address in final.c. Address is either a register, a symbol,
a label, or a symbol or label plus/minus a constant. */

void
md_print_operand_address(FILE *file, rtx addr)
{
	rtx	offset;

	if (GET_CODE(addr) == REG) {
		fputs(reg_names[REGNO(addr)], file);
	}
	else {
		output_addr_const(file, addr);
	}

}

/******************************************************************************/
/* Nested function support. */

/* Emit RTL insns to initialize the variable parts of a trampoline.
FNADDR is an RTX for the address of the function's pure code.
CXT is an RTX for the static chain value for the function. */

void
md_initialize_trampoline(rtx tramp, rtx fnaddr, rtx cxt)
{
	/* Not implemented. */
}

/******************************************************************************/
/* Output assembler code to FILE to increment profiler label # LABELNO
	for profiling a function entry. */

void
md_increment_profiler_label(FILE *file, int labelno)
{
	/* Not implemented. */
}

/******************************************************************************/
/* Output assembler code to FILE to initialize this source file's
	basic block profiling info, if that has not already been done. */

void
md_block_profiler_init(FILE *file, int labelno)
{
	/* Not implemented. */
}

/******************************************************************************/
/* Output assembler code to FILE to increment the entry-count for
	the BLOCKNO'th basic block in this source file. */

void
md_block_profiler_increment(FILE * file, int blockno)
{
	/* Not implemented. */
}

/******************************************************************************/
/* Output assembler code for a block containing the constant parts
	of a trampoline, leaving space for the variable parts. */

void
md_output_trampoline_template(FILE *file)
{
	/* Not implemented. */
}

/******************************************************************************/
/* Called at the start of every file to be compiled. We use it to:
	- read the configuration file.
	- build the cseg and tseg section name ops based on the input filename.
	- output our standard setup stuff at the start of the output file.
*/

extern	char	*main_input_filename;

void
md_file_start(FILE * file)
{
	char	*p, *q;
	char	aname[256];

		/* Read the configuration file. */
	read_config_file();

		/* Get input filename (stripping any path stuff in it) */
	p = main_input_filename;
#ifdef	__MWERKS__
	while (q = strchr(p, ':')) {
#else
	while (q = strchr(p, '/')) {
#endif	/* __MWERKS__ */
		p = q + 1;
	}
	
	strcpy(md_filename, p);
	p = strstr(md_filename, ".c");
	if (p) {
		*p = 0;
	}

		/* Build the section op strings. */
	strcpy(md_cseg_name, "CSEG_");
	strcat(md_cseg_name, md_filename);
	strcpy(md_tseg_name, "\n\tTSEG_");
	strcat(md_tseg_name, md_filename);
	strcpy(md_text_name, "\n\tTEXT_");
	strcat(md_text_name, md_filename);
	
	strcpy(aname, md_filename);
	p = aname;
	while (*p) {
		if (islower(*p)) {
			*p = toupper(*p);
		}
		p++;
	}
	strcat(aname, "_A");

		/* Put standard stuff at start of output file. */
	fprintf(file, "; begin_header\n\n");

	fprintf(file, "\tgcc\t\t/* 6502 */\n\n");
	
	fprintf(file, "%s\tequ\t1\n\n", aname);

	if (TARGET_FARCALL) {
		fprintf(file, "\tincl\t\"franklin.h65\"\n");
	}

	fprintf(file, "\tincl\t\"gcc_app.h65\"\n\n");

	if (TARGET_FARCALL) {
		fprintf(file, "\tincl\t\"%s.f65\"\n\n", md_filename);
	}

	fprintf(file, "\tifndef\tCSEG_%s\n", md_filename);
	fprintf(file, "CSEG_%s\tmacro\n", md_filename);
	fprintf(file, "\tcseg\t1\n");
	fprintf(file, "\tendm\n");
	fprintf(file, "\tendi\n\n");

	fprintf(file, "\tifndef\tTSEG_%s\n", md_filename);
	fprintf(file, "TSEG_%s\tmacro\n", md_filename);
	fprintf(file, "\t%s\n", md_cseg_name);
	fprintf(file, "\tendm\n");
	fprintf(file, "\tendi\n\n");

	fprintf(file, "\tifndef\tTEXT_%s\n", md_filename);
	fprintf(file, "TEXT_%s\tmacro\n", md_filename);
	fprintf(file, "\t%s\n", md_cseg_name);
	fprintf(file, "\tendm\n");
	fprintf(file, "\tendi\n\n");

	fprintf(file, "\tincl\t\"gcc_lib02.h65\"\n");
	fprintf(file, "\tincl\t\"gcc_macros02.h65\"\n");
	
	fprintf(file, "\n; end_header\n");

		/* Not in any particular type of section yet. */
	curseg = NO_SEG;
	
}

/******************************************************************************/

void
md_identify_gcc(FILE * file)
{
	/* Done in md_file_start. */
}

/******************************************************************************/
/* Output debugging info about a source line number. We DO NOT output the info
in standard DBX format. The format is a simple one which our post-processor
can handle. */

void
md_output_source_line(FILE *file, int lineno)
{

	fprintf(file, "< %d\n", lineno);

}

/******************************************************************************/
/* Output a label. */

void
md_output_label(FILE *file, char *name)
{

	assemble_name(file, name);
	fputs("\n", file);
	md_forget_axy();

}

/******************************************************************************/
/* Output a reference to a label. */

void
md_output_labelref(FILE *file, char *name)
{

	fprintf(file, "%s", name);

}

/******************************************************************************/
/* This is used to 'build' internal labels before they are eventually output
by md_output_internal_label below. */

void
md_generate_internal_label(char *label, char *prefix, int num)
{

	sprintf(label, "*%s%d", prefix, num);

}

/******************************************************************************/
/* These labels come out through here: (there may be others)
	Lnnn	code label
	LBBnnn	block begin
	LBEnnn	block end
	LCnnn	constant
	Ltext	??? at start of every file
*/

void
md_output_internal_label(FILE *file, char *prefix, int num)
{

	if (!strcmp(prefix, "LC")) {
		if (curseg != TEXT_SEG) {
			fprintf(file, "\n\t%s\n", md_text_name);
			curseg = TEXT_SEG;
		}
	}

	fprintf(file, "%s%d:\n", prefix, num);
	md_forget_axy();

}

/******************************************************************************/
/* Output a public label. */

void
md_globalize_label(FILE *file, char *name)
{

	fputs("\n/******************************************************************************/\n\n", file);
	fputs("\tdef\t", file);
	assemble_name(file, name);
	fputs("\n", file);

}

/******************************************************************************/
/* Output a reference to an external symbol. */

void
md_output_external(FILE *file, tree decl, char *name)
{
	char	c2cname[256];

	if (md_is_symbol_zp(name)) {
		fputs("\tzref\t", file);
		assemble_name(file, name);
	}
	else {
		fputs("\tref\t", file);
		strcpy(c2cname, name);
		strcat(c2cname, "_c2c");
		assemble_name(file, name);
		if (!md_is_func_nowrap(name)) {
			fputs(", ", file);
			assemble_name(file, c2cname);
		}
	}
	fputs("\n", file);

}

/******************************************************************************/
/* This is confusing - our int's are 16 bits, but this is called to output a
constant of size 24 bits. (With our C6502 changes in varasm.c. Without our
C6502 changes, this would be called to output 32 bits.) Seems like the name
should be changed. */

void
md_output_int(FILE *file, rtx value)
{

	fprintf(file, "\t%s\t", ASM_LONG);	/* ASM_LONG = d24 */
	output_addr_const(file, value);
	fprintf(file, "\n");

}

/******************************************************************************/
/* Output a 16 bit constant. */

void
md_output_short(FILE *file, rtx value)
{
	char	name[256];

	fprintf(file, "\t%s\t", ASM_SHORT);	/* ASM_SHORT = d16 */
	output_addr_const(file, value);
	fprintf(file, "\n");

}

/******************************************************************************/
/* Output an 8 bit constant. */

void
md_output_char(FILE *file, rtx value)
{

	fprintf(file, "\t%s\t", ASM_BYTE_OP);	/* ASM_BYTE_OP = d8 */
	output_addr_const(file, value);
	fprintf(file, "\n");

}

/******************************************************************************/
/* I'm not sure when this is used (if ever). */

void
md_output_byte(FILE *file, int value)
{

	fprintf(file, "\t%s\t0x%x\n", ASM_BYTE_OP, value);

}

/******************************************************************************/
/* Output an element of a jump table. */

void
md_output_addr_vec_elt(FILE *file, int value)
{
	char label[30];

	md_generate_internal_label(label, "L", value);
	fprintf(file, "\td16\t");
	assemble_name(file, label);
	fprintf(file, "\n");

}

/******************************************************************************/
/* Output an asm opcode to 'skip' size bytes. */

void
md_output_skip(FILE *file, int size)
{

	fprintf(file, "\tskip\t%u\n", size);

}

/******************************************************************************/
/* Output a public RAM variable. */

void
md_output_common(FILE *file, char *name, int size, int rounded)
{

	if (md_is_symbol_zp(name)) {
		if (curseg != ZSEG) {
			fputs("\n\tzseg\n", file);
			curseg = ZSEG;
		}
	}
	else {
		if (curseg != DSEG) {
			fputs("\n\tdseg\n", file);
			curseg = DSEG;
		}
	}

	fputs("\tdef\t", file);
	assemble_name(file, name);
	fputs("\n", file);
	assemble_name(file, name);
	fprintf(file, "\tds\t%u\n", size);

}

/******************************************************************************/
/* Output a local RAM variable. */

void
md_output_local(FILE *file, char *name, int size, int rounded)
{

	if (md_is_symbol_zp(name)) {
		if (curseg != ZSEG) {
			fputs("\n\tzseg\n", file);
			curseg = ZSEG;
		}
	}
	else {
		if (curseg != DSEG) {
			fputs("\n\tdseg\n", file);
			curseg = DSEG;
		}
	}

	assemble_name(file, name);
	fprintf(file, "\tds\t%u\n", size);

}

/******************************************************************************/
/* Return the string of the asm opcode to start a text section. */

char *
md_text_section_asm_op(void)
{

	curseg = TSEG;

	return md_tseg_name;

}

/******************************************************************************/
/* Return the string of the asm opcode to start a data section. */

char *
md_data_section_asm_op(void)
{

	curseg = TSEG;

	return md_tseg_name;

}

/******************************************************************************/
/* There are things you can do with this to store one bit of information in
a variable DECL. I played with it a bit in the Sneak compiler, but never
really got anything useful out of it. This may be the right place for a ZP
flag. */

void
md_encode_section_info(tree decl)
{

#if	0
	if (TREE_CODE(decl) == VAR_DECL) {
		SYMBOL_REF_FLAG(XEXP(DECL_RTL(decl), 0)) = 1;
	}
	else {
		SYMBOL_REF_FLAG(XEXP(DECL_RTL(decl), 0)) = 0;
	}
#endif

}

/******************************************************************************/
/* We use a callee-saves model. The first 8 registers are 'volatile'. The
second 8 are 'call-used'. This means that we only need to save/restore 
registers r8-r15. We save/restore a register if it is ever 'live' during the
life of the function. */

#define	FIRST_SAVE_REG	8

#define MUST_SAVE_REGISTER(regno)	(regs_ever_live[regno])

/******************************************************************************/
/* Return the bytes needed to compute the frame pointer from the current
	stack pointer. */

unsigned long
md_compute_frame_size(size)
	int	size;	/* # of var. bytes allocated. */
{
	int		regno;
	unsigned long	gmask;		/* Mask of saved gp registers. */

	saved_regs = gmask = 0;

		/* Determine which registers must be saved. */

	for (regno = FIRST_SAVE_REG; regno < N_GENERAL_REGS; regno++) {
		if (MUST_SAVE_REGISTER(regno)) {
			saved_regs++;
			gmask |= 1 << regno;
		}
	}
	
	/* Save other computed information. */
	current_frame_info.var_size	= size;
	current_frame_info.saved_regs = saved_regs;
	current_frame_info.frame_size	= (saved_regs * UNITS_PER_WORD) + size;
	current_frame_info.gmask	= gmask;
	current_frame_info.initialized	= reload_completed;

	return current_frame_info.frame_size;

}

/******************************************************************************/
/* We don't support a frame pointer, but we need to pretend we do during the
early stages of compilation. It will be optimized away later. That's why you
MUST compile with the -O2 option. */

unsigned long
md_compute_frame_pointer_offset (void)
{

	md_compute_frame_size(get_frame_size());
	
	if (frame_pointer_needed) {
		return current_frame_info.frame_size;
	}
	else {
		return current_frame_info.var_size;
	}

}

/******************************************************************************/

int
md_first_parm_offset(tree fundecl)
{

		/* Careful - frame_pointer_needed might not be setup yet! */
	if (frame_pointer_needed || (current_function_args_info > N_ARG_REGS(VOIDmode))) {
	
		/* stacked fp */
		return 2;
	}
	else {
		/* frame size */
		return (md_compute_frame_size(get_frame_size()));
	}

}

/******************************************************************************/
/* Determine the widest mode that a register is ever used for in the
	function. */

enum machine_mode
get_reg_max_mode(int regno)
{
	enum machine_mode	mode, maxmode;
	rtx	insn, dest;
	
	maxmode = VOIDmode;

	for (insn = get_insns(); insn; insn = NEXT_INSN(insn)) {
		mode = VOIDmode;
		if (GET_CODE(insn) == INSN) {
			if (GET_CODE(PATTERN(insn)) == SET) {
				dest = SET_DEST(PATTERN(insn));
				if (GET_CODE(dest) == SUBREG) {
					dest = SUBREG_REG(dest);
				}
				if (GET_CODE(dest) == REG) {
					if (REGNO(dest) == regno) {
						mode = GET_MODE(dest);
						if (mode > maxmode) {
							maxmode = mode;
						}
					}
				}
			}
		}
	}
	
	return maxmode;

}

/******************************************************************************/
/* Save the registers which we have determined need to be saved in
md_compute_frame_size(). We only need to save the number of bytes of the
register which are actually used in the function. (For example, if the widest
mode of a register's usage is QImode, then we only need to save one byte.) */

void
md_save_registers(FILE *file, unsigned long gmask)
{
	int	regno;
	int	npush;

	if (gmask) {

		/* First, let post-processor know what we'll be saving */
		fprintf(file, "\n; begin_save_regs:");

		npush = 0;
		for (regno = FIRST_SAVE_REG; 
					regno < N_GENERAL_REGS; regno++) {
			if (gmask & (1L << regno)) {
				switch(get_reg_max_mode(regno)) {
				case QImode:
					fprintf(file,
						" %s.8",
						reg_names[regno]);
					++npush;
					break;
				case HImode:
					fprintf(file,
						" %s.16",
						reg_names[regno]);
					++npush;
					break;
				case SImode:
					fprintf(file,
						" %s.24",
						reg_names[regno]);
					++npush;
					break;
				}
			}
		}

		fprintf(file, "\n");

		if (npush) {
			fprintf(file, "\t_begin_pushregs\n");
		}

		/* Next, push the registers */

		for (regno = FIRST_SAVE_REG; regno < N_GENERAL_REGS; regno++) {
			if (gmask & (1L << regno)) {
				saved_reg_modes[regno] = get_reg_max_mode(regno);
				switch (saved_reg_modes[regno]) {
				case QImode:
					fprintf(file, "\tpush_8\t%s\n", reg_names[regno]);
					break;
				case HImode:
					fprintf(file, "\tpush_16\t%s\n", reg_names[regno]);
					break;
				case SImode:
					fprintf(file, "\tpush_24\t%s\n", reg_names[regno]);
					break;
				}
			}
		}

		/* Mark the end of the save register block */

		if (npush) {
			fprintf(file, "\t_end_pushregs\n");
		}

		fprintf(file, "; end_save_regs\n");
	}

}

/******************************************************************************/
/* Inverse of md_save_registers. If the last pop is a 16-bit pop of
FIRST_SAVE_REG (r8), then we return TRUE so that the caller knows to jump to
the library routine 'lib_exit_pop16_r8'. This save a few bytes of exit code. */

int
md_restore_registers(FILE *file, unsigned long gmask)
{
	int	regno;

	if (gmask) {

		/* Mark the start of the restore register block */

		fprintf(file, "; begin_restore_regs\n");
		fprintf(file, "\t_begin_popregs\n");

		for (regno = N_GENERAL_REGS; regno >= FIRST_SAVE_REG; regno--) {
			if (gmask & (1L << regno)) {
				switch (saved_reg_modes[regno]) {
				case QImode:
					fprintf(file, "\tpop_8\t%s\n", reg_names[regno]);
					break;
				case HImode:
					if (regno == FIRST_SAVE_REG) {
						/* Mark special case */
						fprintf(file, 
							"; jmp_restore_regs\n");
						return 1;
					}
					fprintf(file, "\tpop_16\t%s\n", reg_names[regno]);
					break;
				case SImode:
					fprintf(file, "\tpop_24\t%s\n", reg_names[regno]);
					break;
				}
			}
		}

			/* Mark the end of the restore register block */

		fprintf(file, "\t_end_popregs\n");
		fprintf(file, "; end_restore_regs\n");

}
	return 0;

}

/******************************************************************************/

void
md_declare_function_name(FILE *file, char *name, tree decl)
{

	strcpy(md_func_name, name);

	md_func_public = TREE_PUBLIC(decl);

}

/******************************************************************************/
/* Set up the stack and frame (if desired) for the function. */

void
md_output_function_prologue(FILE *file, int size)
{
	char	c2cname[256];

		/* Forget the contents of the 6502 A,X,Y registers. */
	md_forget_axy();

		/* Compute frame and register save info if we haven't already
		done so. */
	if (!current_frame_info.initialized) {
		size = md_compute_frame_size(size);
	}

	fputs("\n/******************************************************************************/\n\n", file);

	fprintf(file, "\tfbeg\n");

#ifdef	LOCAL_LABELS
	fprintf(file, "\tloc\n");
#endif	/* LOCAL_LABELS */

		/* Allocate storage for any locals. On the 6502, this is
		STATIC RAM - we do not use a stack. */
	if (current_frame_info.var_size > 0) {

			/* If it's a leaf function, then any locals can
			be placed in a SHARE section with all other leaf
			functions. */
		if (is_leaf_function()) {
			if (current_frame_info.var_size < 10) {
				fprintf(file, "\tzshare\tLEAF_LOCALS_ZP\n");
			}
			else {
				fprintf(file, "\tshare\tLEAF_LOCALS\n");
			}
		}
		else {
			fprintf(file, "\tdseg\n");
		}

#ifdef	LOCAL_LABELS
		strcpy(md_sp_label, ".sp");
#else
		sprintf(md_sp_label, "sp_%d", md_sp_labelnum++);
#endif	/* LOCAL_LABELS */
		fprintf(file, "%s\tds\t%d\n", md_sp_label, current_frame_info.var_size);

		if (!TARGET_FARCALL) {
			fprintf(file, "\t%s\n", md_cseg_name);
		}
	}

	strcpy(c2cname, md_func_name);
	strcat(c2cname, "_c2c");

	if (TARGET_FARCALL) {
		fprintf(file, "\tFUNCTION_ENTRY\t%s\n", md_func_name);
		fprintf(file, "\tFUNCTION_ENTRY\t%s\n", c2cname);
		curseg = CSEG;
	}
	else {
			/* Switch to the code section if we're not already in it. */
		if (curseg != CSEG) {
			fprintf(file, "\n\t%s\n", md_cseg_name);
			curseg = CSEG;
		}
		
		if (md_func_public) {
#ifdef	FRANKLIN
			/* DEF the function as funcname and funcname_c2c.
			(md_globalize_label will not be called for function names
			because of a change to varasm.c.) */
			fprintf(file, "\tdef\t%s, %s\n", md_func_name, c2cname);
#else
			/* DEF the function as funcname_c2c (funcname has already
			been DEF'ed by md_globalize_label.) */
			fprintf(file, "\tdef\t%s\n", c2cname);
#endif	/* FRANKLIN */
		}
		md_output_label(file, md_func_name);
		md_output_label(file, c2cname);
	}

	if (frame_pointer_needed) {
		fprintf(stderr, "Abort - frame pointer not supported!\n");
		abort();
	}

		/* Save any non-volatile (r8-r15) registers we're going to use. */

	md_save_registers(file, current_frame_info.gmask);
	
	md_saw_return = 0;

}

/******************************************************************************/
/* Do any necessary cleanup after a function to restore stack, frame,
and regs. */

void
md_output_function_epilogue(FILE *file, int size)
{

	fprintf(file, "\n");

	size = (!current_frame_info.initialized
		? md_compute_frame_size(size)
		: current_frame_info.var_size);

	if (frame_pointer_needed) {
		fprintf(stderr, "Abort - frame pointer not supported!\n");
		abort();
	}

	if (md_restore_registers(file, current_frame_info.gmask)) {
/*		fprintf(file, "\tjmp\tlib_exit_pop16_r8\n"); */
		fprintf(file, "\t_pop16_r8_end_popregs_exit\n");
	}
	
		/* If we saved any registers, then our 'return' insn in the
		md couldn't be used. So do the RTS now. Also, gcc does not
		put a return in an empty function, so handle that case too. */

	else if (saved_regs || !md_saw_return) {
		fprintf(file, "\trts\n");
	}

	fprintf(file, "\tfend\n\n");

		/* Reset state info for each function. */
	current_frame_info = zero_frame_info;

}

/******************************************************************************/
/* Not used anymore since we switched to the 'callee-save' model. */

int
md_caller_save_profitable(int refs, int calls)
{

	/* We get slightly smaller code if we don't save/restore registers.
		Just reload them as needed. */

	return 0;

#if	0
	/* Magic formula - based on default ((4 * calls) < refs). I tried
	a more 'scientific' formula, but it lost miserably. */

	return ((7 * calls) < refs);
#endif

}

/******************************************************************************/
/* Compute the relative cost of an address. */

int
md_address_cost(rtx addr)
{
	rtx	op0, op1;

	if (GET_CODE(addr) == REG) {
		if (REG_OK_FOR_SP(addr) || REG_OK_FOR_FP(addr)) {
			return 2;
		}
		return 3;
	}
	
	if (md_is_zp_op(addr)) {
		return 3;
	}
	
	switch (GET_CODE(addr)) {

	case SYMBOL_REF:
	case LABEL_REF:
		return 2;

	case CONST:
	case CONST_INT:
		return 1;
	
	case PLUS:
		op0 = XEXP(addr, 0);
		op1 = XEXP(addr, 1);

		if (REG_OK_FOR_SP(op0) || REG_OK_FOR_FP(op0)) {
			if (GET_CODE(op1) == CONST_INT) {
				return 2;
			}
		}

		if (REG_OK_FOR_SP(op1) || REG_OK_FOR_FP(op1)) {
			if (GET_CODE(op1) == CONST_INT) {
				return 2;
			}
		}

		if (RTX_OK_FOR_BASE_P(op0)) {
			if (GET_CODE(op1) == CONST_INT) {
				if (INTVAL(op1) < 256) {
					return 3;
				}
				else {
					return 6;
				}
			}
			else if (RTX_OK_FOR_INDEX_P(op1)) {
				return 6;
			}
		}
		else if (RTX_OK_FOR_BASE_P(op1)) {
			if (GET_CODE(op0) == CONST_INT) {
				if (INTVAL(op0) < 256) {
					return 3;
				}
				else {
					return 6;
				}
			}
			else if (RTX_OK_FOR_INDEX_P(op0)) {
				return 6;
			}
		}

	}

	return 100;

}

/******************************************************************************/
/* More black magic. See md_address_cost, above. */

int
md_rtx_costs(rtx x, enum rtx_code code, enum rtx_code outer_code)
{

	switch (code) {
	case MEM:
		return md_address_cost(XEXP(x, 0));

	case PLUS:
	case MINUS:
		return 2;

	case ASHIFT:
	case ASHIFTRT:
	case LSHIFTRT:
	case ROTATERT:
	case ROTATE:
		return 2;

	case MULT:
	case UDIV:
	case UMOD:
	case DIV:
	case MOD:
		return 4;

	case NEG:
	case NOT:
		return 3;

	case ZERO_EXTEND:
		return 2;

	case SIGN_EXTEND:
		return 10;

	default:
		break;
	}

	return 5;

}

/******************************************************************************/
/* You can change this to anything you want, and I don't think it will effect
anything. */

int
md_register_move_cost(int class1, int class2)
{

	return 1;

}

/******************************************************************************/
/* Ditto for this. */

int
md_memory_move_cost(enum machine_mode mode)
{

	switch (mode) {
	case QImode:
		return 2;
	
	case HImode:
		return 3;
	
	case SImode:
		return 4;
	}
	
	return 10;

}

/******************************************************************************/
/* Determines when structure moves will be done with in-line move insns or
a call to memcpy. Current value (3) was picked so that DATE structures in SDK
(five bytes) will get in-line moves. Structures six bytes or greater will get
memcpy's. (I think.) */

int
md_move_ratio(void)
{

	return 3;

}

/******************************************************************************/
/* Return TRUE if the INSN (a CALL_INSN) is a 'tail call' - the next active
insn is a RETURN. */

int
md_is_tail_call(rtx insn)
{

	if (saved_regs) {
		return 0;
	}

	while (insn = next_insn(insn)) {
		if (GET_CODE(insn) == CODE_LABEL) {
			continue;
		}
		if (GET_CODE(insn) == JUMP_INSN) {
			if (GET_CODE(PATTERN(insn)) == RETURN) {
				return 1;
			}
			return 0;
		}
		if (GET_CODE(insn) == CALL_INSN) {
			return 0;
		}
		if (GET_CODE(insn) == INSN) {
			if (GET_CODE(PATTERN(insn)) == USE) {
				continue;
			}
			return 0;
		}
	}
	
	return 0;

}

/******************************************************************************/
/* Return TRUE if INSN (a RETURN insn) is immediately after a tail-call CALL
insn. If it is, then the tail-call CALL will have been changed to a JUMP and
we don't need this RETURN. */

int
md_was_tail_call(rtx insn)
{

	if (saved_regs) {
		return 0;
	}

	while (insn = previous_insn(insn)) {
			/* Careful - if we see a label, then somebody's jumping
			to this return and we better keep it. */
		if (GET_CODE(insn) == CODE_LABEL) {
			return 0;
		}
		if (GET_CODE(insn) == JUMP_INSN) {
			return 0;
		}
		if (GET_CODE(insn) == CALL_INSN) {
			return 1;
		}
		if (GET_CODE(insn) == INSN) {
			if (GET_CODE(PATTERN(insn)) == USE) {
				continue;
			}
			return 0;
		}
	}
	
	return 0;

}

/******************************************************************************/
/* This is the condition part of our 'return' insn in the md. We can only
 return with one insn if we didn't save any registers. */

int
md_use_return(void)
{

		/* The gcc manual says we can only use a return afer reload
		is completed. */
	if (!reload_completed)  {
		return 0;
	}

	if (saved_regs) {
		return 0;
	}

	return 1;

}

/******************************************************************************/
/* Return TRUE if a register is dead. We begin at INSN and go forward until we
either know that the register is dead (it's set) or live (it's referenced).
If we see any jumps, we don't follow the trail and just return FALSE.
(I tried following the jumps, but at this point of the compilation, peepholes
have been combined into parallels, and it is very difficult. I couldn't figure
out how to do it.) */

int
md_is_reg_dead(rtx insn, rtx reg)
{
	rtx	src, dest, rtnreg;
	int	regno = REGNO(reg);

	while (insn) {
		switch (GET_CODE(insn)) {

		case JUMP_INSN:
			if (GET_CODE(PATTERN(insn)) == RETURN) {
				/* This is a RETURN insn. If we're not dealing
				with r0, then continue forward. (This might be
				an in-line return, so we can't stop yet.) */
				
				if (regno != 0) {
					break;
				}
				
				/* If it is r0, see if it's the return value of
				the function. If it isn't, then the reg is dead.
				This depends on gcc always putting out a USE
				insn before the return for the return value
				reg - I hope that's always true. */
				
				insn = prev_nonnote_insn(insn);
				if (!insn) {
					return 0;
				}
				if (	(GET_CODE(insn) == INSN) &&
					(GET_CODE(PATTERN(insn)) == USE)
				   ) {
					return 0;
				}
				return 1;
			}
			return 0;
		
		case CALL_INSN:
		
			/* If this is a call-used reg, then the call does
			not blow it away so continue on. */
			
			if (regno >= FIRST_SAVE_REG) {
				break;
			}
			
			/* If the reg is an arg to the call, then it's not
			dead. */
			
			if (find_regno_fusage(insn, USE, regno)) {
				return 0;
			}
	
			/* Careful - we might call indirect through
			the register! */
			if (reg_referenced_p(reg, PATTERN(insn))) {
				return 0;
			}

			/* If the called function uses no registers, then the
			reg isn't dead yet. */
			
			if (md_is_func_noregs(insn)) {
				break;
			}
			
			return 1;

		case INSN:
			if (GET_CODE(PATTERN(insn)) == SET) {
			
				/* Must check src first - there could be
				a register dead note on the src but the
				dest could be equal to the src! In this
				case the register is not really dead. */

				/* If it's referenced, it's not dead. */
				
				if (reg_referenced_p(reg, PATTERN(insn))) {
					return 0;
				}

				/* If it's SET, then it's dead. */
				
				dest = SET_DEST(PATTERN(insn));
				if (GET_CODE(dest) == SUBREG) {
					dest = SUBREG_REG(dest);
				}
				if (GET_CODE(dest) == REG) {
					if (REGNO(dest) == regno) {
						return 1;
					}
				}
			}
			break;
		}
	
		insn = next_insn(insn);
	}
	
	if (regno == 0) {
		return 0;
	}
	
	return 1;

}

/******************************************************************************/
/* Used in md file to call dead_or_set_p in rtlanal.c because dead_or_set_p
will abort if it's arg is not a register. For us, if it's not a register, then
it's not dead. */

int
md_dead_or_set_p(rtx insn, rtx reg)
{

	if (GET_CODE(reg) != REG) {
		return 0;
	}

	return dead_or_set_p(insn, reg);

}

/******************************************************************************/
/* Called from md insns mv_8, mv_16, and mv_24 to move SRC to DEST. */

void
md_mv_m_m(rtx src, rtx dest, int size, char *cmt)
{
	rtx	ops[2];
	int	remember, low_in_x, high_on_stack;
	OP_TYPE	src_op, dest_op;

	ops[0] = src;
	ops[1] = dest;
	strcpy(md_comment, cmt);
	remember = 1;
	low_in_x = high_on_stack = 0;
	src_op = md_get_op_type(src);
	dest_op = md_get_op_type(dest);

	if (!md_is_axy(src)) {
		output_asm_insn("lda	%L0%;", ops);
		if (	(src_op == OP_GPR_IND) &&
			(dest_op == OP_GPR) &&
			(REGNO(XEXP(src, 0)) == REGNO(dest)) &&
			(size > 8)
		   ) {
			output_asm_insn("tax", ops);
			low_in_x = 1;
		}
		else {
			output_asm_insn("sta	%L1", ops);
		}
	}
	else {
		output_asm_insn("sta	%L1%;", ops);
	}

	if (size >= 16) {
		if (md_need_y(src) && md_need_y(dest)) {
			output_asm_insn("ldy	#1", ops);
			output_asm_insn("lda	%H0", ops);
			output_asm_insn("sta	%H1", ops);
			remember = 0;
		}
		else if (md_need_y(src)) {
			output_asm_insn("ldy	#1", ops);
			output_asm_insn("lda	%H0", ops);
			if (	(src_op == OP_GPR_IND) &&
				(dest_op == OP_GPR) &&
				(REGNO(XEXP(src, 0)) == REGNO(dest)) &&
				(size > 16)
			   ) {
				output_asm_insn("pha", ops);
				high_on_stack = 1;
			}
			else {
				output_asm_insn("sta	%H1", ops);
			}
			remember = 0;
		}
		else if (md_need_y(dest)) {
			if (md_is_axy(src)) {
				output_asm_insn("txa", ops);
			}
			else {
				output_asm_insn("lda	%H0", ops);
			}
			output_asm_insn("ldy	#1", ops);
			output_asm_insn("sta	%H1", ops);
			remember = 0;
		}
		else {
			if (!md_is_axy(src)) {
				output_asm_insn("ldx	%H0", ops);
			}
			output_asm_insn("stx	%H1", ops);
		}
	}
	
	if (size >= 24) {
		if (md_need_y(src) && md_need_y(dest)) {
			output_asm_insn("iny", ops);
			output_asm_insn("lda	%B0", ops);
			output_asm_insn("sta	%B1", ops);
			remember = 0;
		}
		else if (md_need_y(src)) {
			output_asm_insn("iny", ops);
			output_asm_insn("lda	%B0", ops);
			output_asm_insn("sta	%B1", ops);
			remember = 0;
		}
		else if (md_need_y(dest)) {
			output_asm_insn("lda	%B0", ops);
			output_asm_insn("iny", ops);
			output_asm_insn("sta	%B1", ops);
			remember = 0;
		}
		else {
			if (!md_is_axy(src)) {
				output_asm_insn("ldy	%B0", ops);
			}
			output_asm_insn("sty	%B1", ops);
		}
	}

	if (low_in_x) {
		output_asm_insn("stx	%L1", ops);
	}
	if (high_on_stack) {
		output_asm_insn("pla", ops);
		output_asm_insn("sta	%H1", ops);
	}

	if (!remember) {
		md_forget_axy();
	}
	else if (GET_CODE(dest) == REG) {
		md_remember_axy(dest);
	}
	else {
		md_remember_axy(src);
	}

}

/******************************************************************************/
/* Called from md insns mv_8, mv_16, and mv_24 to move an immediate SRC to
DEST. */

void
md_mv_imm_m(rtx src, rtx dest, int size, char *cmt)
{
	rtx	ops[2];
	int	remember;

	ops[0] = src;
	ops[1] = dest;
	strcpy(md_comment, cmt);

	remember = 1;

	if (	(GET_CODE(src) == CONST_INT) &&
		((INTVAL(src) & 0x0000FF) == 0) &&
		(!md_need_y(dest))
	   ) {
		output_asm_insn("stz	%L1%;", ops);
		remember = 0;
	}
	else {
		if (!md_is_axy(src)) {
			output_asm_insn("lda	%L0%;", ops);
			output_asm_insn("sta	%L1", ops);
		}
		else {
			output_asm_insn("sta	%L1%;", ops);
		}
	}

	if (size >= 16) {
		if (	(GET_CODE(src) == CONST_INT) &&
			((INTVAL(src) & 0x00FF00) == 0) &&
			(!md_need_y(dest))
		   ) {
			output_asm_insn("stz	%H1", ops);
			remember = 0;
		}
		else {
			if (md_need_y(dest)) {
				if (!md_is_axy(src)) {
					if (!md_are_high_low_same(src)) {
						output_asm_insn("lda	%H0", ops);
					}
				}
				else {
					output_asm_insn("txa", ops);
				}
				output_asm_insn("ldy	#1", ops);
				output_asm_insn("sta	%H1", ops);
				remember = 0;
			}
			else {
				if (!md_is_axy(src)) {
					output_asm_insn("ldx	%H0", ops);
				}
				output_asm_insn("stx	%H1", ops);
			}
		}

	}

	if (size >= 24) {
		if (	(GET_CODE(src) == CONST_INT) &&
			((INTVAL(src) & 0xFF0000) == 0) &&
			(!md_need_y(dest))
		   ) {
			output_asm_insn("stz	%B1", ops);
			remember = 0;
		}
		else {
			if (md_need_y(dest)) {
				if (!md_are_bank_high_same(src)) {
					output_asm_insn("lda	%B0", ops);
				}
				output_asm_insn("iny", ops);
				output_asm_insn("sta	%H1", ops);
				remember = 0;
			}
			else {
				if (!md_is_axy(src)) {
					output_asm_insn("ldy	%B0", ops);
				}
				output_asm_insn("sty	%B1", ops);
			}
		}
	}

	if (remember) {
		if (GET_CODE(dest) == REG) {
			md_remember_axy(dest);
		}
		else {
			md_remember_axy(src);
		}
	}
	else {
		md_forget_axy();
	}

}

/******************************************************************************/
/* Used by some md insns to move a GPR to reg 't' (temporary). */

void
md_mv_r_t_16(rtx src, char *cmt)
{
	rtx	ops[1];

	ops[0] = src;
	strcpy(md_comment, cmt);

	if (!md_is_axy(src)) {
		output_asm_insn("lda	%L0%;", ops);
		output_asm_insn("sta	t", ops);
		output_asm_insn("ldx	%H0", ops);
		output_asm_insn("stx	t+1", ops);
	}
	else {
		output_asm_insn("sta	t%;", ops);
		output_asm_insn("stx	t+1", ops);
	}

	if (GET_CODE(src) == REG) {
		axy_rtx = src;
	}
	else {
		md_forget_axy();
	}

}

/******************************************************************************/
/* Used by some md insns to move a GPR to reg 't' (temporary). */

void
md_mv_r_t_24(rtx src, char *cmt)
{
	rtx	ops[1];

	ops[0] = src;
	strcpy(md_comment, cmt);

	if (!md_is_axy(src)) {
		output_asm_insn("lda	%L0%;", ops);
		output_asm_insn("sta	t", ops);
		output_asm_insn("ldx	%H0", ops);
		output_asm_insn("stx	t+1", ops);
		output_asm_insn("ldy	%B0", ops);
		output_asm_insn("sty	t+2", ops);
	}
	else {
		output_asm_insn("sta	t%;", ops);
		output_asm_insn("stx	t+1", ops);
		output_asm_insn("sty	t+2", ops);
	}

	if (GET_CODE(src) == REG) {
		axy_rtx = src;
	}
	else {
		md_forget_axy();
	}

}

/******************************************************************************/
/* Create a new asm6 local label and return it's 'number part'. */

int
md_new_label(void)
{

#ifdef	LOCAL_LABELS
	md_make_label(++md_labelnum);
#else
	md_make_label(label_num);
	md_labelnum = label_num++;
#endif	/* LOCAL_LABELS */

	return md_labelnum;

}

/* Make an asm6 local label having a 'number part' of LABELNUM. */

void
md_make_label(int labelnum)
{

#ifdef	LOCAL_LABELS
	sprintf(md_label, ".l%d", labelnum);
#else
	sprintf(md_label, "L%d:", labelnum);
#endif	/* LOCAL_LABELS */
	
}

/* Output an asm6 local label having a 'number part' of LABELNUM. */

void
md_output_md_label(int labelnum)
{

	md_make_label(labelnum);
	fprintf(asm_out_file, "%s\n", md_label);

}

/******************************************************************************/
/* When we do things we maybe shouldn't, at least give them a warning in the
output file. */

void
md_output_warning(char *warning)
{

	fprintf(asm_out_file, ";; WARNING: %s\n", warning);

}

/******************************************************************************/
/* Remember OP as the current contents of the 6502's AXY registers. */

void
md_remember_axy(rtx op)
{

	axy_rtx = op;

}

/******************************************************************************/
/* Say that contents of 6502's AXY registers are unknown. */

void
md_forget_axy(void)
{

	axy_rtx = 0;

}

/******************************************************************************/
/* TRUE if OP is equal to the contents of the 6502's AXY registers. */

int
md_is_axy(rtx op)
{

	if (!axy_rtx) {
		return 0;
	}

	return rtx_equal_p(op, axy_rtx);

}

/******************************************************************************/
/* Analyze the current function to see if it's a leaf function - it calls no
routines that use our registers r0-r15. */

int
is_leaf_function(void)
{
	rtx	insn;;

	for (insn = get_insns(); insn; insn = NEXT_INSN(insn)) {

		if (GET_CODE(insn) == CALL_INSN) {

			/* It's a CALL - if it's a NOREGS routine (defined in
			the configuration file), then it doesn't use our
			register set. Otherwise, this isn't a leaf function. */

			if (!md_is_func_noregs(insn)) {
				return 0;
			}
		}
		else if (	(GET_CODE(insn) == INSN) &&
				(GET_CODE(PATTERN(insn)) == SEQUENCE) &&
				(GET_CODE(XVECEXP(PATTERN(insn), 0, 0)) == CALL_INSN)
		   ) {
			/* Should probably check for NOREGS here too, but I'm
			not sure what's going on in this case (code stolen from
			leaf_function_p() in final.c). */

			return 0;
		}
	}
	
	return 1;

}

/******************************************************************************/
/* Determine the OP_TYPE of OP:
	OP_GPR		rn
	OP_GPR_IND	(rn)
	OP_ZP_IND	(zp symbol)
	OP_MEM		symbol or label, maybe plus/minus a const
	OP_SP		(sp)
	OP_SP_OFF	(sp+off)
	OP_ADDR		#immediate symbol or label
	OP_IMM		#immediate integer
*/

OP_TYPE
md_get_op_type(rtx op)
{
	rtx	opm0, op0, op1;

	if (register_operand(op, GET_MODE(op))) {
		if (GET_CODE(op) == SUBREG) {
			op0 = SUBREG_REG(op);
		}
		else {
			op0 = op;
		}
		if (REG_OK_FOR_GPR(op0)) {
			return OP_GPR;
		}
	}

	switch (GET_CODE(op)) {
	
	case SYMBOL_REF:
	case LABEL_REF:
		return OP_ADDR;

	case CONST:
		op = XEXP(op, 0);
		if ((GET_CODE(op) == PLUS) || (GET_CODE(op) == MINUS)) {
			if (	(	(GET_CODE(XEXP(op, 0)) == SYMBOL_REF) ||
					(GET_CODE(XEXP(op, 0)) == LABEL_REF)
				) &&
				(GET_CODE(XEXP(op, 1)) == CONST_INT)
			   ) {
				return OP_ADDR;
			}
		}
		break;

	case MEM:
		opm0 = XEXP(op, 0);
		if (GET_CODE(opm0) == SYMBOL_REF) {
			return OP_MEM;
		}

		if ((GET_CODE(opm0) == REG) && (REG_OK_FOR_GPR(opm0))) {
			return OP_GPR_IND;
		}
		
		if (md_is_zp_op(opm0)) {
			return OP_ZP_IND;
		}

		if (GET_CODE(opm0) == CONST) {
			op0 = XEXP(opm0, 0);
			if (GET_CODE(op0) == PLUS) {
				if (	(	GET_CODE(XEXP(op0, 0)) == SYMBOL_REF ||
						GET_CODE(XEXP(op0, 0)) == LABEL_REF
					) &&
						GET_CODE(XEXP(op0, 1)) == CONST_INT
				   ) {
					return OP_MEM;
				}
			}
		}


		if (register_operand(opm0, GET_MODE(opm0))) {
			if (REG_OK_FOR_FP(opm0) || REG_OK_FOR_SP(opm0)) {
				return OP_SP;
			}
		}
		if (GET_CODE(opm0) == PLUS) {
			op0 = XEXP(opm0, 0);
			op1 = XEXP(opm0, 1);
			if (	(register_operand(op0, GET_MODE(op0))) &&
				(REG_OK_FOR_FP(op0) || REG_OK_FOR_SP(op0)) &&
				(GET_CODE(op1) == CONST_INT)
			   ) {
			   	sp_off = INTVAL(op1);
				return OP_SP_OFF;
			}
		}
		break;
	
	case CONST_INT:
		return OP_IMM;

	}

	return OP_NONE;

}

/******************************************************************************/
/* Return TRUE if ADDR is a valid address for mode MODE. */

int
md_go_if_legitimate_address(rtx addr, enum machine_mode mode)
{
	rtx	op0, op1;

	if (RTX_OK_FOR_BASE_P(addr)) {
		return 1;
	}
	
	if (md_is_zp_op(addr)) {
		return 1;
	}
	
	switch (GET_CODE(addr)) {

	case SYMBOL_REF:
	case LABEL_REF:
	case CONST:
	case CONST_INT:
		return 1;
	
	case PLUS:
		op0 = XEXP(addr, 0);
		op1 = XEXP(addr, 1);

		if (RTX_OK_FOR_BASE_P(op0)) {
			if (	RTX_OK_FOR_INDEX_P(op1) ||
				(GET_CODE(op1) == CONST_INT)
			  ) {
				return 1;
			}
		}
		else if (RTX_OK_FOR_BASE_P(op1)) {
			if (	RTX_OK_FOR_INDEX_P(op0) ||
				(GET_CODE(op0) == CONST_INT)
			  ) {
				return 1;
			}
		}

	}

	return 0;

}

/******************************************************************************/
/* Not used - here as a 'hook' in case we ever need to use it. */

rtx
md_legitimize_address(rtx x, rtx oldx, enum machine_mode mode)
{

	return (rtx) 0;

}

/******************************************************************************/

int
md_is_reg_return_value(rtx reg, rtx jmp_label)
{
	rtx	x;
	int	regno = REGNO(reg);

return 1;

/* FIXME - this stuff crashes on Sun Unix!!! */

	if (GET_CODE(reg) != REG) {
		return 0;
	}

	if (regno != 0) {
		return 0;
	}

	for (x = get_insns(); x; x = NEXT_INSN(x)) {
		if (	(GET_CODE(x) == CODE_LABEL) &&
			(rtx_equal_p(jmp_label, x))
		   ) {
			while (x = NEXT_INSN(x)) {
				if (GET_CODE(x) == INSN) {
					x = PATTERN(x);
					if (GET_CODE(x) == USE) {
						if (rtx_equal_p(XEXP(x, 0), reg)) {
							return 1;
						}
					}
					break;
				}
			}
			break;
		}
	}
	
	return 0;
	
}

/******************************************************************************/
/* TRUE if op1 is a register and op2 is using the same register as a pointer.
Or vice versa. We have to be careful in these cases that we don't blow away the
low byte of the ptr reg before we read the high or low bytes. */

int
md_are_ops_same_reg(rtx op1, rtx op2)
{
	OP_TYPE	ot1, ot2;

	ot1 = md_get_op_type(op1);
	ot2 = md_get_op_type(op2);

	if (ot1 == OP_GPR) {
		if (ot2 == OP_GPR_IND) {
			return ((REGNO(op1) == REGNO(XEXP(op2, 0))));
		}
	}

	if (ot1 == OP_GPR_IND) {
		if (ot2 == OP_GPR) {
			return (REGNO(XEXP(op1, 0)) == REGNO(op2));
		}
	}

	return 0;

}

/******************************************************************************/
/* Read the configuration file, if specified. The config file can contain
these definitions:

zp	symbol_name
noregs	function_name
nowrap	function_name

'zp' says that the symbol will live in zero page.

'noregs' says that the function will not use any of our registers r0-r15.

'nowrap' says that the function is not a compiled function and does not need
	a wrapper routine (it takes no arguments and returns no results).

*/

void
read_config_file(void)
{
	char	line[256], *lp;
	FILE	*cfgfile;

	num_zp_syms = num_noregs = 0;

	if (!md_config_file) {
		return;
	}
	cfgfile = fopen(md_config_file, "r");
	if (!cfgfile) {
		return;
	}

	while (TRUE) {
		if (!fgets(line, 255, cfgfile)) {
			break;
		}
		line[strlen(line) - 1] = 0;
		if (!strncmp(line, "zp", 2)) {
			if (num_zp_syms < MAX_ZP) {
				lp = line + 2;
				while ((*lp == 0x09) || (*lp == 0x20)) {
					lp++;
				}
				if (*lp) {
					strcpy(zp_symbols[num_zp_syms++], lp);
				}
			}
		}
		else if (!strncmp(line, "noregs", 6)) {
			if (num_noregs < MAX_NOREGS) {
				lp = line + 6;
				while ((*lp == 0x09) || (*lp == 0x20)) {
					lp++;
				}
				if (*lp) {
					strcpy(noregs_funcs[num_noregs++], lp);
				}
			}
		}
		else if (!strncmp(line, "nowrap", 6)) {
			if (num_nowrap < MAX_NOWRAP) {
				lp = line + 6;
				while ((*lp == 0x09) || (*lp == 0x20)) {
					lp++;
				}
				if (*lp) {
					strcpy(nowrap_funcs[num_nowrap++], lp);
				}
			}
		}
	}
	
	fclose(cfgfile);

}

/******************************************************************************/
/* TRUE if NAME is a zero page symbol. */

int
md_is_symbol_zp(char *name)
{
	int	i;
	
	for (i = 0; i < num_zp_syms; i++) {
		if (!strcmp(zp_symbols[i], name)) {
			strcpy(md_zp_name, name);
			return 1;
		}
	}
	
	return 0;

}

/******************************************************************************/
/* TRUE if OP is a zero page symbol. */

int
md_is_zp_op(rtx op)
{
	rtx	sym;

	if (GET_CODE(op) == MEM) {
		sym = XEXP(op, 0);
		if (GET_CODE(sym) == SYMBOL_REF) {
			if (md_is_symbol_zp(XSTR(sym, 0))) {
				return 1;
			}
		}
	}

	return 0;

}

/******************************************************************************/
/* TRUE if CALL_INSN is calling a 'noregs' function. */

int
md_is_func_noregs(rtx call_insn)
{
	rtx	target;
	char	name[100];
	int	i;

	name[0] = 0;
	if (recog_memoized(call_insn) == CODE_FOR_call_symbol) {
		target = XEXP(PATTERN(call_insn), 0);
		strcpy(name, XSTR(XEXP(target, 0), 0));
	}
	else if (recog_memoized(call_insn) == CODE_FOR_call_value_symbol) {
		target = XEXP(PATTERN(call_insn), 1);
		strcpy(name, XSTR(XEXP(target, 0), 0));
	}

	if (name[0]) {
		for (i = 0; i < num_noregs; i++) {
			if (!strcmp(noregs_funcs[i], name)) {
				return 1;
			}
		}
	}
	
	return 0;

}

/******************************************************************************/
/* TRUE if NAME is a nowrap function. */

int
md_is_func_nowrap(char *name)
{
	int	i;
	
	for (i = 0; i < num_nowrap; i++) {
		if (!strcmp(nowrap_funcs[i], name)) {
			return 1;
		}
	}
	
	return 0;

}

/******************************************************************************/

