/*
 * ACME - a crossassembler for producing 6502/65c02/65816 code.
 * Copyright (C) 1998 Marco Baye
 * Have a look at "acme.c" for further info
 */

/*
 * Tree stuff
 */

#include "tree.h"

/*
 * Constants
 */

/*
 * Keyword tables:
 *
 * These tables hold the name and byte values for several special keywords
 * the assembler must know about, e.g. pseudo opcodes, arithmetic and logic
 * operators, processor names and assembler mnemonics.
 *
 */

/* these are used more than once */
char  s_65816[] = "65816";
char  s_cbm[] = "cbm";
char  s_endoffile[] = "endoffile";
char  s_pet[] = "pet";
char  s_raw[] = "raw";
char  s_scr[] = "scr";
char  s_subzone[] = "subzone";
char  s_XOR[] = "XOR";
char  s_08[] = "08";

/* Yes, I know I'm sick */
#define s_16   (&s_65816[3])
#define s_8    (&s_08[1])
#define s_file (&s_endoffile[5])
#define s_OR   (&s_XOR[1])
#define s_zone (&s_subzone[3])

/*
 * Hmm, if I changed the case switching around, I could omit four of these:
 * "and", "AND",
 * "asl", "ASL",
 * "eor", "EOR",
 * "lsr", "LSR",
 */

/* the predefined stuff using byte values*/
treeItem PredefinedB[] = {
  {NULL, NULL, 1, HTYPE_6502MNEMO,  "adc",   {{  96, G_ACCM}}},
  {NULL, NULL, 1, HTYPE_6502MNEMO,  "and",   {{  32, G_ACCM}}},
  {NULL, NULL, 1, HTYPE_6502MNEMO,  "asl",   {{   0, G_MISCO}}},
  {NULL, NULL, 1, HTYPE_6502MNEMO,  "bcc",   {{ 144, G_RELS}}},
  {NULL, NULL, 1, HTYPE_6502MNEMO,  "bcs",   {{ 176, G_RELS}}},
  {NULL, NULL, 1, HTYPE_6502MNEMO,  "beq",   {{ 240, G_RELS}}},
  {NULL, NULL, 1, HTYPE_6502MNEMO,  "bit",   {{   1, G_MISCA}}},
  {NULL, NULL, 1, HTYPE_6502MNEMO,  "bmi",   {{  48, G_RELS}}},
  {NULL, NULL, 1, HTYPE_6502MNEMO,  "bne",   {{ 208, G_RELS}}},
  {NULL, NULL, 1, HTYPE_6502MNEMO,  "bpl",   {{  16, G_RELS}}},
  {NULL, NULL, 1, HTYPE_6502MNEMO,  "brk",   {{   0, G_IMP}}},
  {NULL, NULL, 1, HTYPE_6502MNEMO,  "bvc",   {{  80, G_RELS}}},
  {NULL, NULL, 1, HTYPE_6502MNEMO,  "bvs",   {{ 112, G_RELS}}},
  {NULL, NULL, 1, HTYPE_6502MNEMO,  "clc",   {{  24, G_IMP}}},
  {NULL, NULL, 1, HTYPE_6502MNEMO,  "cld",   {{ 216, G_IMP}}},
  {NULL, NULL, 1, HTYPE_6502MNEMO,  "cli",   {{  88, G_IMP}}},
  {NULL, NULL, 1, HTYPE_6502MNEMO,  "clv",   {{ 184, G_IMP}}},
  {NULL, NULL, 1, HTYPE_6502MNEMO,  "cmp",   {{ 192, G_ACCM}}},
  {NULL, NULL, 1, HTYPE_6502MNEMO,  "cpx",   {{   2, G_MISCR}}},
  {NULL, NULL, 1, HTYPE_6502MNEMO,  "cpy",   {{   3, G_MISCR}}},
  {NULL, NULL, 1, HTYPE_6502MNEMO,  "dec",   {{   4, G_MISCO}}},
  {NULL, NULL, 1, HTYPE_6502MNEMO,  "dex",   {{ 202, G_IMP}}},
  {NULL, NULL, 1, HTYPE_6502MNEMO,  "dey",   {{ 136, G_IMP}}},
  {NULL, NULL, 1, HTYPE_6502MNEMO,  "eor",   {{  64, G_ACCM}}},
  {NULL, NULL, 1, HTYPE_6502MNEMO,  "inc",   {{   5, G_MISCO}}},
  {NULL, NULL, 1, HTYPE_6502MNEMO,  "inx",   {{ 232, G_IMP}}},
  {NULL, NULL, 1, HTYPE_6502MNEMO,  "iny",   {{ 200, G_IMP}}},
  {NULL, NULL, 1, HTYPE_6502MNEMO,  "jmp",   {{   0, G_JUMP}}},
  {NULL, NULL, 1, HTYPE_6502MNEMO,  "jsr",   {{   1, G_JUMP}}},
  {NULL, NULL, 1, HTYPE_6502MNEMO,  "lda",   {{ 160, G_ACCM}}},
  {NULL, NULL, 1, HTYPE_6502MNEMO,  "ldx",   {{   6, G_MISCR}}},
  {NULL, NULL, 1, HTYPE_6502MNEMO,  "ldy",   {{   7, G_MISCR}}},
  {NULL, NULL, 1, HTYPE_6502MNEMO,  "lsr",   {{   8, G_MISCO}}},
  {NULL, NULL, 1, HTYPE_6502MNEMO,  "nop",   {{ 234, G_IMP}}},
  {NULL, NULL, 1, HTYPE_6502MNEMO,  "ora",   {{   0, G_ACCM}}},
  {NULL, NULL, 1, HTYPE_6502MNEMO,  "pha",   {{  72, G_IMP}}},
  {NULL, NULL, 1, HTYPE_6502MNEMO,  "php",   {{   8, G_IMP}}},
  {NULL, NULL, 1, HTYPE_6502MNEMO,  "pla",   {{ 104, G_IMP}}},
  {NULL, NULL, 1, HTYPE_6502MNEMO,  "plp",   {{  40, G_IMP}}},
  {NULL, NULL, 1, HTYPE_6502MNEMO,  "rol",   {{   9, G_MISCO}}},
  {NULL, NULL, 1, HTYPE_6502MNEMO,  "ror",   {{  10, G_MISCO}}},
  {NULL, NULL, 1, HTYPE_6502MNEMO,  "rti",   {{  64, G_IMP}}},
  {NULL, NULL, 1, HTYPE_6502MNEMO,  "rts",   {{  96, G_IMP}}},
  {NULL, NULL, 1, HTYPE_6502MNEMO,  "sbc",   {{ 224, G_ACCM}}},
  {NULL, NULL, 1, HTYPE_6502MNEMO,  "sec",   {{  56, G_IMP}}},
  {NULL, NULL, 1, HTYPE_6502MNEMO,  "sed",   {{ 248, G_IMP}}},
  {NULL, NULL, 1, HTYPE_6502MNEMO,  "sei",   {{ 120, G_IMP}}},
  {NULL, NULL, 1, HTYPE_6502MNEMO,  "sta",   {{ 128, G_ACCS}}},
  {NULL, NULL, 1, HTYPE_6502MNEMO,  "stx",   {{  11, G_MISCO}}},
  {NULL, NULL, 1, HTYPE_6502MNEMO,  "sty",   {{  12, G_MISCO}}},
  {NULL, NULL, 1, HTYPE_6502MNEMO,  "tax",   {{ 170, G_IMP}}},
  {NULL, NULL, 1, HTYPE_6502MNEMO,  "tay",   {{ 168, G_IMP}}},
  {NULL, NULL, 1, HTYPE_6502MNEMO,  "tsx",   {{ 186, G_IMP}}},
  {NULL, NULL, 1, HTYPE_6502MNEMO,  "txa",   {{ 138, G_IMP}}},
  {NULL, NULL, 1, HTYPE_6502MNEMO,  "txs",   {{ 154, G_IMP}}},
  {NULL, NULL, 1, HTYPE_6502MNEMO,  "tya",   {{ 152, G_IMP}}},
  {NULL, NULL, 1, HTYPE_65816MNEMO, "brl",   {{ 130, G_RELL}}},
  {NULL, NULL, 1, HTYPE_65816MNEMO, "cop",   {{  16, G_MISCO}}},
  {NULL, NULL, 1, HTYPE_65816MNEMO, "jml",   {{   2, G_JUMP}}},
  {NULL, NULL, 1, HTYPE_65816MNEMO, "jsl",   {{   3, G_JUMP}}},
  {NULL, NULL, 1, HTYPE_65816MNEMO, "mvn",   {{0x54, G_MOVE}}},
  {NULL, NULL, 1, HTYPE_65816MNEMO, "mvp",   {{0x44, G_MOVE}}},
  {NULL, NULL, 1, HTYPE_65816MNEMO, "pea",   {{  17, G_MISCO}}},
  {NULL, NULL, 1, HTYPE_65816MNEMO, "pei",   {{0xd4, G_PEI}}},
  {NULL, NULL, 1, HTYPE_65816MNEMO, "per",   {{  98, G_RELL}}},
  {NULL, NULL, 1, HTYPE_65816MNEMO, "phb",   {{ 139, G_IMP}}},
  {NULL, NULL, 1, HTYPE_65816MNEMO, "phd",   {{  11, G_IMP}}},
  {NULL, NULL, 1, HTYPE_65816MNEMO, "phk",   {{  75, G_IMP}}},
  {NULL, NULL, 1, HTYPE_65816MNEMO, "plb",   {{ 171, G_IMP}}},
  {NULL, NULL, 1, HTYPE_65816MNEMO, "pld",   {{  43, G_IMP}}},
  {NULL, NULL, 1, HTYPE_65816MNEMO, "rep",   {{  18, G_MISCO}}},
  {NULL, NULL, 1, HTYPE_65816MNEMO, "rtl",   {{ 107, G_IMP}}},
  {NULL, NULL, 1, HTYPE_65816MNEMO, "sep",   {{  19, G_MISCO}}},
  {NULL, NULL, 1, HTYPE_65816MNEMO, "stp",   {{ 219, G_IMP}}},
  {NULL, NULL, 1, HTYPE_65816MNEMO, "tcd",   {{  91, G_IMP}}},
  {NULL, NULL, 1, HTYPE_65816MNEMO, "tcs",   {{  27, G_IMP}}},
  {NULL, NULL, 1, HTYPE_65816MNEMO, "tdc",   {{ 123, G_IMP}}},
  {NULL, NULL, 1, HTYPE_65816MNEMO, "tsc",   {{  59, G_IMP}}},
  {NULL, NULL, 1, HTYPE_65816MNEMO, "txy",   {{ 155, G_IMP}}},
  {NULL, NULL, 1, HTYPE_65816MNEMO, "tyx",   {{ 187, G_IMP}}},
  {NULL, NULL, 1, HTYPE_65816MNEMO, "wai",   {{ 203, G_IMP}}},
  {NULL, NULL, 1, HTYPE_65816MNEMO, "wdm",   {{  66, G_IMP}}},
  {NULL, NULL, 1, HTYPE_65816MNEMO, "xba",   {{ 235, G_IMP}}},
  {NULL, NULL, 1, HTYPE_65816MNEMO, "xce",   {{ 251, G_IMP}}},
  {NULL, NULL, 1, HTYPE_65C02MNEMO, "bra",   {{ 128, G_RELS}}},
  {NULL, NULL, 1, HTYPE_65C02MNEMO, "phx",   {{ 218, G_IMP}}},
  {NULL, NULL, 1, HTYPE_65C02MNEMO, "phy",   {{  90, G_IMP}}},
  {NULL, NULL, 1, HTYPE_65C02MNEMO, "plx",   {{ 250, G_IMP}}},
  {NULL, NULL, 1, HTYPE_65C02MNEMO, "ply",   {{ 122, G_IMP}}},
  {NULL, NULL, 1, HTYPE_65C02MNEMO, "stz",   {{  13, G_MISCO}}},
  {NULL, NULL, 1, HTYPE_65C02MNEMO, "trb",   {{  14, G_MISCO}}},
  {NULL, NULL, 1, HTYPE_65C02MNEMO, "tsb",   {{  15, G_MISCO}}},
  {NULL, NULL, 1, HTYPE_CONV,       s_file,  {{CODETABLE_FILE}}},
  {NULL, NULL, 1, HTYPE_CONV,       s_pet,   {{CODETABLE_PET }}},
  {NULL, NULL, 1, HTYPE_CONV,       s_raw,   {{CODETABLE_RAW }}},
  {NULL, NULL, 1, HTYPE_CONV,       s_scr,   {{CODETABLE_SCR }}},
  {NULL, NULL, 1, HTYPE_CPU,        "6502",  {{CPUBIT_6502 }}},
  {NULL, NULL, 1, HTYPE_CPU,        "6510",  {{CPUBIT_6510 }}},
  {NULL, NULL, 1, HTYPE_CPU,        s_65816, {{CPUBIT_65816}}},
  {NULL, NULL, 1, HTYPE_CPU,        "65c02", {{CPUBIT_65C02}}},
/*{NULL, NULL, 1, HTYPE_CPU,        "z80",   {{CPUBIT_Z80  }}},*/
  {NULL, NULL, 1, HTYPE_UNTILWHILE, "until", {{ID_UNTIL}}},
  {NULL, NULL, 1, HTYPE_UNTILWHILE, "while", {{ID_WHILE}}},
  {NULL, NULL, 1, HTYPE_ELSE,       "else",  {{ID_ELSE}}},
  {NULL, NULL, 1, HTYPE_OPERATOR,   "AND",   {{HOP_AND,    PRIO_AND   }}},
  {NULL, NULL, 1, HTYPE_OPERATOR,   "ASL",   {{HOP_SL,     PRIO_SL    }}},
  {NULL, NULL, 1, HTYPE_OPERATOR,   "DIV",   {{HOP_DIVIDE, PRIO_DIVIDE}}},
  {NULL, NULL, 1, HTYPE_OPERATOR,   "EOR",   {{HOP_EXOR,   PRIO_EXOR  }}},
  {NULL, NULL, 1, HTYPE_OPERATOR,   "LSL",   {{HOP_SL,     PRIO_SL    }}},
  {NULL, NULL, 1, HTYPE_OPERATOR,   "LSR",   {{HOP_LSR,    PRIO_LSR   }}},
  {NULL, NULL, 1, HTYPE_OPERATOR,   "MOD",   {{HOP_MODULO, PRIO_MODULO}}},
  {NULL, NULL, 1, HTYPE_OPERATOR,   s_OR,    {{HOP_OR,     PRIO_OR    }}},
  {NULL, NULL, 1, HTYPE_OPERATOR,   s_XOR,   {{HOP_EXOR,   PRIO_EXOR  }}},
  {NULL, NULL, 1, HTYPE_OUTFORMAT,  s_cbm,   {{OUTFORMAT_CBM  }}},
  {NULL, NULL, 1, HTYPE_OUTFORMAT,  "o65",   {{OUTFORMAT_O65  }}},
  {NULL, NULL, 0, HTYPE_OUTFORMAT,  "plain", {{OUTFORMAT_PLAIN}}},
  /*           ^ this marks the last element */
};

/* the predefined stuff using pointers */
treeItemP PredefinedP[] = {
  {NULL, NULL, 1, HTYPE_PSEUDO, s_08,       {(void *) PO_08}},
  {NULL, NULL, 1, HTYPE_PSEUDO, s_16,       {(void *) PO_16}},
  {NULL, NULL, 1, HTYPE_PSEUDO, "24",       {(void *) PO_24}},
  {NULL, NULL, 1, HTYPE_PSEUDO, "32",       {(void *) PO_32}},
  {NULL, NULL, 1, HTYPE_PSEUDO, s_8,        {(void *) PO_08}},
  {NULL, NULL, 1, HTYPE_PSEUDO, "al",       {(void *) CPU_SetLongA}},
  {NULL, NULL, 1, HTYPE_PSEUDO, "as",       {(void *) CPU_SetShortA}},
  {NULL, NULL, 1, HTYPE_PSEUDO, "by",       {(void *) PO_08}},
  {NULL, NULL, 1, HTYPE_PSEUDO, "ct",       {(void *) PO_convtab}},
  {NULL, NULL, 1, HTYPE_PSEUDO, "do",       {(void *) FlowPO_do}},
  {NULL, NULL, 1, HTYPE_PSEUDO, "fi",       {(void *) PO_fill}},
  {NULL, NULL, 1, HTYPE_PSEUDO, "if",       {(void *) FlowPO_if}},
  {NULL, NULL, 1, HTYPE_PSEUDO, "rl",       {(void *) CPU_SetLongR}},
  {NULL, NULL, 1, HTYPE_PSEUDO, "rs",       {(void *) CPU_SetShortR}},
  {NULL, NULL, 1, HTYPE_PSEUDO, "sl",       {(void *) Label_PO_sl}},
  {NULL, NULL, 1, HTYPE_PSEUDO, "sz",       {(void *) FlowPO_subzone}},
  {NULL, NULL, 1, HTYPE_PSEUDO, "to",       {(void *) PO_to}},
  {NULL, NULL, 1, HTYPE_PSEUDO, "tx",       {(void *) PO_text}},
  {NULL, NULL, 1, HTYPE_PSEUDO, "wo",       {(void *) PO_16}},
  {NULL, NULL, 1, HTYPE_PSEUDO, "zn",       {(void *) FlowPO_zone}},
  {NULL, NULL, 1, HTYPE_PSEUDO, "eof",      {(void *) FlowPO_eof}},
  {NULL, NULL, 1, HTYPE_PSEUDO, "for",      {(void *) FlowPO_for}},
  {NULL, NULL, 1, HTYPE_PSEUDO, "src",      {(void *) FlowPO_source}},
  {NULL, NULL, 1, HTYPE_PSEUDO, "bin",      {(void *) PO_binary}},
  {NULL, NULL, 1, HTYPE_PSEUDO, s_cbm,      {(void *) PO_cbm}},
  {NULL, NULL, 1, HTYPE_PSEUDO, "cpu",      {(void *) CPU_Choose}},
  {NULL, NULL, 1, HTYPE_PSEUDO, s_pet,      {(void *) PO_petscii}},
  {NULL, NULL, 1, HTYPE_PSEUDO, s_raw,      {(void *) PO_raw}},
  {NULL, NULL, 1, HTYPE_PSEUDO, s_scr,      {(void *) PO_screencode}},
  {NULL, NULL, 1, HTYPE_PSEUDO, "set",      {(void *) Label_PO_set}},
  {NULL, NULL, 1, HTYPE_PSEUDO, s_zone,     {(void *) FlowPO_zone}},
  {NULL, NULL, 1, HTYPE_PSEUDO, "byte",     {(void *) PO_08}},
  {NULL, NULL, 1, HTYPE_PSEUDO, "word",     {(void *) PO_16}},
  {NULL, NULL, 1, HTYPE_PSEUDO, "fill",     {(void *) PO_fill}},
  {NULL, NULL, 1, HTYPE_PSEUDO, "text",     {(void *) PO_text}},
  {NULL, NULL, 1, HTYPE_PSEUDO, "ifdef",    {(void *) FlowPO_ifdef}},
  {NULL, NULL, 1, HTYPE_PSEUDO, "macro",    {(void *) Macro_Definition}},
  {NULL, NULL, 1, HTYPE_PSEUDO, "align",    {(void *) PO_align}},
  {NULL, NULL, 1, HTYPE_PSEUDO, "realpc",   {(void *) FlowPO_realpc}},
  {NULL, NULL, 1, HTYPE_PSEUDO, "source",   {(void *) FlowPO_source}},
  {NULL, NULL, 1, HTYPE_PSEUDO, "binary",   {(void *) PO_binary}},
  {NULL, NULL, 1, HTYPE_PSEUDO, "scrxor",   {(void *) PO_scrxor}},
  {NULL, NULL, 1, HTYPE_PSEUDO, "convtab",  {(void *) PO_convtab}},
  {NULL, NULL, 1, HTYPE_PSEUDO, "initmem",  {(void *) PO_initmem}},
  {NULL, NULL, 1, HTYPE_PSEUDO, s_subzone,  {(void *) FlowPO_subzone}},
  {NULL, NULL, 1, HTYPE_PSEUDO, "pseudopc", {(void *) FlowPO_pseudopc}},
  {NULL, NULL, 0, HTYPE_PSEUDO, s_endoffile,{(void *) FlowPO_eof}},
  /*           ^ this marks the last element */
};

/*
 * Variables
 */

treeItem* TreesROM[256];/* Array of pointers to start of item */
treeItem* TreesRAM[256];/*      trees (because of 8-bit hash) */
treeItem  WantedItem;/* temporary storage */
int       Tree_ItemCreated = FALSE;/* Global flag for tree item lookup */
int       IDString_Length;/* length of item's ID string */

/*
 * Compute IntHash value by exclusive ORing the struct's type and its name.
 * Write output to struct, reduce IntHash to ByteHash and return that.
 */
byte Tree_MakeHashes(treeItem* p) {
  byte*         ReadPointer;
  register byte b;
  u_int         IntHash = p->ID_Number;/* This *must* be unsigned! */
  /* if it's signed, the hash routine won't be effective */

  IDString_Length = 0;/* set global variable */

  ReadPointer = p->ID_String;
  while((b = *(ReadPointer++))) {
    IntHash = ((IntHash << 7) | (IntHash >> (8 * sizeof(u_int) - 7))) ^ b;
    IDString_Length++;
  }
  p->IntHash = IntHash;
  PLATFORM_INT2BYTE(IntHash);
  return((byte) IntHash);/* return ByteHash */
}

/*
 * Link a predefined data set to the (later on) read-only trees
 */
void Tree_LinkROMitem(treeItem* pItem) {
  treeItem** pPointer;
  byte       ByteHash = Tree_MakeHashes(pItem);

  pPointer = &(TreesROM[ByteHash]);/* point to treeItem pointer in table */

  while(*pPointer) {
    if(pItem->IntHash > (*pPointer)->IntHash) {/* compare IntHash */
      pPointer = &((*pPointer)->GreaterThan);
    } else {
      pPointer = &((*pPointer)->LessThanOrEqual);
    }
  }
  *pPointer = pItem;/* add new leaf to tree */
/* new items are always added as leaves, so
 * there's no need to copy a second pointer.*/
}

/*
 * Initialise ROM trees
 */
void Tree_InitROM() {
  treeItem*  t;

  Tree_ClearPointerTable(TreesROM);/* clear table (later read-only) */

  /* link strings (bytes) */
  t = PredefinedB;
  while(t->IntHash) Tree_LinkROMitem(t++);
  Tree_LinkROMitem(t);/* caution when trying to optimise this. :) */

  /* link strings (pointers) */
  t = (treeItem *) PredefinedP;
  while(t->IntHash) Tree_LinkROMitem(t++);
  Tree_LinkROMitem(t);/* caution when trying to optimise this. :) */

}

/*
 * Init one pointer table
 */
void Tree_ClearPointerTable(treeItem* pTable[]) {
  int a;

  /* Clear pointertables */
  for(a = 255; a >= 0; a--) *(pTable++) = NULL;
}

/*
 * Search for a "ROM tree" item. Compute the hash of the given string and then
 * use that to try to find a tree item that matches the given data (IntHash,
 * ID_Number, ID_String). Return pointer to tree item found.
 * If no matching item is found, return NULL.
 */
treeItem* Tree_ScanROM(byte* ID_String, int ID_Number) {
  treeItem*  pCurrentItem;
  byte       *p1, *p2;
  byte       b1, b2, ByteHash;

  WantedItem.ID_Number = ID_Number;
  WantedItem.ID_String = ID_String;
  ByteHash = Tree_MakeHashes(&WantedItem);

  pCurrentItem = TreesROM[ByteHash];/* read pointer from table */
  while(pCurrentItem) {
    if(WantedItem.IntHash > pCurrentItem->IntHash) {/* compare IntHash */
      pCurrentItem = pCurrentItem->GreaterThan;
    } else {
      if(WantedItem.IntHash == pCurrentItem->IntHash) {
        if(WantedItem.ID_Number == pCurrentItem->ID_Number) {
          p1 = WantedItem.ID_String;
          p2 = pCurrentItem->ID_String;
          do {
            b1 = *(p1++);
            b2 = *(p2++);
          } while((b1 == b2) && b1);
          if(b1 == b2) return(pCurrentItem);/* return item pointer */
        }
      }
      pCurrentItem = pCurrentItem->LessThanOrEqual;
    }
  }
  return(NULL);/* indicate failure */
}

/*
 * Search for a "RAM tree" item. Compute the hash of MiscString and then
 * use that to try to find a tree item that matches the given data (IntHash,
 * ID_Number, MiscString). Return pointer to tree item found.
 * If no matching item is found, check fCreate flag. If it is set, create a
 * new tree item, link to tree, fill with data and return its pointer (and set
 * the global "Tree_ItemCreated" flag).
 * If fCreate is clear, return NULL.
 */
treeItem* Tree_ScanRAM(int ID_Number, int fCreate) {
  treeItem** ppCurrentItem;
  treeItem*  NewLeaf;
  namedItem* NewNamed;
  byte       *p1, *p2;
  byte       b1, b2, ByteHash;

  Tree_ItemCreated = FALSE;
  WantedItem.ID_Number = ID_Number;
  WantedItem.ID_String = MiscString;
  ByteHash = Tree_MakeHashes(&WantedItem);

  ppCurrentItem = &(TreesRAM[ByteHash]);/* point into table */
  while(*ppCurrentItem) {
    if(WantedItem.IntHash > (*ppCurrentItem)->IntHash) {/* compare IntHash */
      ppCurrentItem = &((*ppCurrentItem)->GreaterThan);
    } else {
      if(WantedItem.IntHash == (*ppCurrentItem)->IntHash) {
        if(WantedItem.ID_Number == (*ppCurrentItem)->ID_Number) {
          p1 = MiscString;
          p2 = (*ppCurrentItem)->ID_String;
          do {
            b1 = *(p1++);
            b2 = *(p2++);
          } while((b1 == b2) && b1);
          if(b1 == b2) return(*ppCurrentItem);/* return pointer to item */
        }
      }
      ppCurrentItem = &((*ppCurrentItem)->LessThanOrEqual);
    }
  }
  if(fCreate == FALSE) {
    return(NULL);/* indicate failure */
  }
  /* allocate space for new item's name string */
  NewNamed = (namedItem*) ALLOC_TRY(sizeof(namedItem) + IDString_Length);
  p1 = MiscString;
  p2 = &(NewNamed->Name[0]);
  while(((*(p2++)) = (*(p1++)))) ;/* copy name string until terminator */
  /* create new item */
  NewLeaf = &(NewNamed->Item);
  NewLeaf->GreaterThan = NULL;
  NewLeaf->LessThanOrEqual = NULL;
  NewLeaf->IntHash = WantedItem.IntHash;
  NewLeaf->ID_Number = WantedItem.ID_Number;
  NewLeaf->ID_String = &(NewNamed->Name[0]);

  /* add new leaf to tree */
  *ppCurrentItem = NewLeaf;

  Tree_ItemCreated = TRUE;
  return(NewLeaf);/* return pointer to new item */
}

/*
 * Calls Tree_DumpTree for each non-zero entry of the given tree table.
 */
void Tree_Dump256(treeItem** TreeTable, int ID_Number, void (*Function)(treeItem*)) {
  int a;

  for(a = 255; a >= 0; a--) {
    if(*TreeTable) Tree_DumpTree(*TreeTable, ID_Number, Function);
    TreeTable++;
  }
}

/*
 * Call given function for each object of matching type in the given tree.
 * Calls itself recursively.
 */
void Tree_DumpTree(treeItem* Item, int ID_Number, void (*Function)(treeItem*)) {

  if(Item->ID_Number == ID_Number) {
    (Function)(Item);
  }
  if(Item->GreaterThan)
    Tree_DumpTree(Item->GreaterThan, ID_Number, Function);
  if(Item->LessThanOrEqual)
    Tree_DumpTree(Item->LessThanOrEqual, ID_Number, Function);
}
