/*
 * ACME - a crossassembler for producing 6502/65c02/65816 code.
 * Copyright (C) 1998 Marco Baye
 * Have a look at "acme.c" for further info
 */

/*
 * Input/output stuff
 */

#include "alu.h"
#include "data.h"
#include "flowpo.h"
#include "core.h"
#include "stream.h"
#include "strings.h"

/*
 * Constants
 */

char Exception_OpenQuotes[] = "Quotes still open at end of line.";
char Exception_NoVarFound[] = "'ACME' environment variable not found.";
char psvOpened[] = "File opened.\n";
char psvClosed[] = "File closed.\n";
char psvAutoClosed[] = "File auto-closed.\n";
char BIA_CloseFile[] = "file not open";
char FILE_READBINARY[] = "rb";
char FILE_WRITEBINARY[] = "wb";
char FILE_WRITETEXT[] = "w";

/*
 * Variables
 */

byte    pPathname[LNPMAX + 1];/* universal file name */
char*   psLibPath;/* Pointer to header string of library tree */
byte    QuoteChar = NO_QUOTE_CHAR;/* Current quoting character */
file*   FileList = NULL;/* pointer to linked list of file structures */

/*
 * Get byte from file and convert newline characters to zero.
 */
byte Stream_ConvertNL() {
  byte old;
  int    b;

  /* Exchange newline characters by zero */
  /* If reading LF of CRLF, read another byte */
  do {
    old = Context[nContext].OldFileByte;
    if(EndReason == RENDOFFILE) {
      Context[nContext].OldFileByte = 0;/* Fake end-of-statement */
    } else {
      /* Get byte from file */
      b = fgetc(Context[nContext].u.hFile);
      if(b == EOF) {
        /* Act upon end of file */
        EndReason = RENDOFFILE;
        Context[nContext].OldFileByte = 0;/* Fake end-of-statement */
      } else {
        Context[nContext].OldFileByte = b;
      }
    }
  } while((old == 13) && (Context[nContext].OldFileByte == 10));/* CRLF test */
/* Check for LF and CR separately. Note that setting
 * "Context[nContext].OldFileByte" to zero would confuse the
 * "old/Context[nContext].OldFileByte" stuff above.*/
  if(Context[nContext].OldFileByte == 10) return(0);
  if(Context[nContext].OldFileByte == 13) return(0);
  return(Context[nContext].OldFileByte);
}

/*
 * Get byte (low level)
 */
byte Stream_GetRawByte() {
  byte old = Context[nContext].OldRawByte;

  if(old == 0) Context[nContext].nLines++;/* Increase line counter */
  switch(Context[nContext].hByteSource) {

    case BYTESOURCE_FILE:
    /* File handler */
    if(QuoteChar != NO_QUOTE_CHAR) {
      /* Inside quotes, don't process. Only complain about zero */
      Context[nContext].OldRawByte = Stream_ConvertNL();
      if(Context[nContext].OldRawByte == 0) {
        ThrowError(Exception_OpenQuotes);
        QuoteChar = NO_QUOTE_CHAR;
      }
    } else {
      /* Outside quotes, process */
      do {
        Context[nContext].OldRawByte = Stream_ConvertNL();
        /* convert TAB to space, kill multiple spaces */
        if(Context[nContext].OldRawByte == 9)
           Context[nContext].OldRawByte = ' ';
      } while((Context[nContext].OldRawByte == ' ') && ((old == 0)
        || (old == ' ') || (old == ',')
        || (old == ':') || (old == '{')));
      /* After semicolon, skip remainder of line */
      if(Context[nContext].OldRawByte == ';')
        while((Context[nContext].OldRawByte = Stream_ConvertNL()));
    }
    break;

    case BYTESOURCE_RAM:
    /* RAM handler */
    Context[nContext].OldRawByte = *(Context[nContext].u.pRAM++);
    if(Context[nContext].OldRawByte == BLOCK_FIELDSEPARATOR) {
      EndReason = RENDOFBLOCK;
      Context[nContext].OldRawByte = 0;/* Fake end-of-statement */
    }
  }
  /* Check quoting */
  if(QuoteChar != NO_QUOTE_CHAR) {
    if(Context[nContext].OldRawByte == QuoteChar) QuoteChar = NO_QUOTE_CHAR;
  } else {
    if(Context[nContext].OldRawByte == '"')  QuoteChar = '"';
    if(Context[nContext].OldRawByte == '\'') QuoteChar = '\'';
  }
  return(Context[nContext].OldRawByte);
}

/*
 * Get byte (high level)
 */
byte GetByte(void) {
  GotByte = Stream_GetRawByte();
  if(QuoteChar == NO_QUOTE_CHAR) {
    /* Check for braces */
    if(GotByte == '}') {
      EndReason = RRIGHTBRACE;
      GotByte = 0;
    } else {
      /* Check for colon */
      if(GotByte == ':') GotByte = 0;
    }
  }
  return(GotByte);
}

/*
 * This routine reads in characters and stores them in memory, until an
 * illegal character is read or too many bytes have been received.
 * The second case produces a "string too long" error.
 * The routine zero-terminates the string and returns its length.
 * Zero lengths will produce a "missing string" error.
 */
int Stream_ReadKeyword(byte* p, int fNext) {
  int c = 0;

  if(fNext) GetByte();
  do {
    p[c] = GotByte;
    /* now: c = number of bytes present minus one */
    if(pFlagTable[GotByte] & BYTEIS_ILLEGAL) break;
    GetByte();
  } while(c++ < LSMAX);
  if(c > LSMAX) {
    ThrowError(Exception_TooLong);
    c--;
  }
  p[c] = 0;/* terminate by overwriting illegal character */
  if(c == 0) ThrowError(Exception_MissingString);
  return(c);
}

/*
 * Try to read a filename. If the flag "LibraryAllowed" is TRUE, library
 * access by using <...> quoting is possible as well. The file name given in
 * the assembler source code is converted from UNIX style to system style.
 */
int Stream_ReadFilename(byte* p, int MaxLen, int LibraryAllowed) {
  int c = 0,
      d = 0;

  SKIPSPACE;
  if(LibraryAllowed && (GotByte == '<')) {
    if(psLibPath) {
      strcpy(p, psLibPath);
      c = strlen(p);
      QuoteChar = '>';
    } else {
      ThrowError(Exception_NoVarFound);
      SkipRest();
    }
  }
  if((QuoteChar == '"') || (QuoteChar == '>')) {
    d = Stream_FinishQuoted(p, MaxLen, c);
    PLATFORM_CONVERTPATH(p + c);/* Convert to system style */
  } else ThrowError(Exception_MissingString);
  return(d);
}

/*
 * Read string and store it in memory until closing quote is reached. Does
 * not store quote. Zero-terminates string and returns its length.
 * If the string is too long or there are no closing quotes, an error will be
 * produced. "cAlready" is the number of chars already present. Storing will
 * take place after them, their number is added when comparing to MaxLen.
 */
int Stream_FinishQuoted(byte* p, int MaxLen, int cAlready) {
  do {
    p[cAlready++] = GetByte();
  } while((cAlready < MaxLen) && GotByte && (QuoteChar != NO_QUOTE_CHAR));
  if(cAlready >= MaxLen) {
    ThrowError(Exception_TooLong);
    cAlready = MaxLen;/* shrink back to limit */
  }
  p[--cAlready] = 0;/* terminate, overwriting last character (closing quote) */
  GetByte();/* proceed with next char */
  SKIPSPACE;
  return(cAlready);
}

/*
 * Try to read a comma, skipping spaces before and after. Return TRUE if comma
 * found, otherwise FALSE.
 */
int Stream_Comma() {
  SKIPSPACE;
  if(GotByte != ',') return(FALSE);
  NEXTANDSKIPSPACE;
  return(TRUE);
}

/*
 * Open file
 */
FILE* File_OpenFile(byte* Filename, char* mode) {
  FILE* Filehandle;
  file* Filestruct;

  Filehandle = fopen(Filename, mode);/* open file in given mode */
  if(Filehandle) {
/*    if(Process_Verbosity > 8) printf(psvOpened);*/
    /* link to list of currently open files */
    Filestruct = SafeAlloc(sizeof(file));
    Filestruct->FileHandle = Filehandle;
    Filestruct->Next = FileList;
    FileList = Filestruct;
  }
  return(Filehandle);
}

/*
 * Close open file
 */
void File_CloseFile(FILE* Filehandle) {
  file*  Filestruct;
  file** Pointer = &FileList;

  while((Filestruct = *Pointer)) {
    if(Filestruct->FileHandle == Filehandle) {
      fclose(Filehandle);
/*      if(Process_Verbosity > 8) printf(psvClosed);*/
      *Pointer = Filestruct->Next;
      free(Filestruct);
      return;
    }
    Pointer = &(Filestruct->Next);
  }
  BugFound(BIA_CloseFile);
}

/*
 * Scan list of open files and close them all
 */
void File_CloseAll() {
  file* Next;

  while(FileList) {
    fclose(FileList->FileHandle);
/*    if(Process_Verbosity > 8) printf(psvAutoClosed);*/
    Next = FileList->Next;
    free(FileList);
    FileList = Next;
  }
}
