/*
 * ACME - a crossassembler for producing 6502/65c02/65816 code.
 * Copyright (C) 1998 Marco Baye
 * Have a look at "acme.c" for further info
 */

/*
 * Pseudo opcodes
 */

#include "po.h"

/*
 * Constants
 */

char Exception_CannotOpenInFile[] = "Cannot open input file.";
char Exception_cbm[] = "'!cbm' is deprecated; use '!ct pet' instead.";
char Exception_MemInitTwice[] = "Memory already initialised.";
char Exception_No2ndOutfile[] = "Output file already chosen.";
char Exception_UkOutformat[] = "Unknown output format.";
char Exception_UkPO[] = "Unknown pseudo opcode.";
char psvBinary[] = "Loaded %d ($%x) bytes from file offset %d ($%x).\n";

/*
 * Parse pseudo opcodes. Has to be re-entrant.
 */
void ParsePseudoOpcode() {/* (GotByte = "!") */
  treeItem*  p;
  int      (*fn)(void);
  int        Then;

  if(Stream_ReadKeyword(MiscString, TRUE)) {
    /* Search for tree item */
    MiscString_ToLower();
    p = Tree_ScanROM(MiscString, HTYPE_PSEUDO);
    if(p) {
      fn = (int (*)()) p->Body.Pointer;
      if(fn != NULL) {
        Then = (fn)();
        if(Then == SKIP_REST) SkipRest();
        if(Then == ENSURE_EOL) EnsureEOL();
      } else ThrowError(Exception_UkPO);
    } else ThrowError(Exception_UkPO);
  }
}

/*
 * Define default value for empty memory
 */
int PO_initmem() {
  if((Pass_Flags & PASS_ISFIRST) == 0) return(SKIP_REST);
  if(Pass_Flags & PASS_MEMINIT) {
    ThrowError(Exception_MemInitTwice);
    return(SKIP_REST);
  }
  Pass_Flags |= PASS_MEMINIT;
  ALU_GetValue_Strict();
  if((ALU_Value<256) && (ALU_Value>-129)) {
    Output_InitMem(ALU_Value & 255);

    /* tricky bit */
    if(PC_Lowest != PC_Highest) {
      if(nNeedValue_Now == 0) nNeedValue_Now = 1;
    }

  } else ThrowError(Exception_TooBig);
  return(ENSURE_EOL);
}

/*
 * Select output file and format
 */
int PO_to() {
  treeItem* p;

  /* only act upon this pseudo opcode in first pass */
  if((Pass_Flags & PASS_ISFIRST)) {
    if(hOutformat == OUTFORMAT_NONE) {
      if(Stream_ReadFilename(pnfOut, LNPMAX, FALSE)) {
        if(Stream_Comma()) {/* parse output format */
          if(Stream_ReadKeyword(MiscString, FALSE)) {
            /* search for tree item */
            p = Tree_ScanROM(MiscString, HTYPE_OUTFORMAT);
            if(p) {
              hOutformat = p->Body.Opcode.Code;
              if(hOutformat == OUTFORMAT_O65) ThrowError(Exception_NotYet);
              return(ENSURE_EOL);
            } else ThrowError(Exception_UkOutformat);
          }
        } else hOutformat = OUTFORMAT_CBM;/* default */
      }
    } else ThrowError(Exception_No2ndOutfile);
  }
  return(SKIP_REST);
}

/*
 * Insert 8-bit values
 */
int PO_08() {
  do {
    Output_8b(ALU_GetValue_Medium());
  } while(Stream_Comma());
  return(ENSURE_EOL);
}

/*
 * Insert 16-bit values
 */
int PO_16() {
  do {
    Output_16b(ALU_GetValue_Medium());
  } while(Stream_Comma());
  return(ENSURE_EOL);
}

/*
 * Insert 24-bit values
 */
int PO_24() {
  do {
    Output_24b(ALU_GetValue_Medium());
  } while(Stream_Comma());
  return(ENSURE_EOL);
}

/*
 * Insert 32-bit values
 */
int PO_32() {
  do {
    Output_32b(ALU_GetValue_Medium());
  } while(Stream_Comma());
  return(ENSURE_EOL);
}

/*
 * Reserve space by sending some bytes of a given value
 */
int PO_fill() {
  value vFill = VFILL;
  int   Size = ALU_GetValue_Strict();

  if(Stream_Comma()) vFill = ALU_GetValue_Medium();
  while(Size) {
    Output_8b(vFill);
    Size--;
  }
  return(ENSURE_EOL);
}

/*
 * Insert byte until PC fits conditions
 */
int PO_align() {
  byte   vFill = VALIGN;
  value  Equal,
         Test = CPU_PC,
         And  = ALU_GetValue_Strict();

  if(!Stream_Comma()) ThrowError(Exception_Syntax);
  Equal = ALU_GetValue_Strict();
  if(Stream_Comma()) vFill = ALU_GetValue_Medium();
  while((Test & And) != Equal) {
    Output_8b(vFill);
    Test++;
  }
  return(ENSURE_EOL);
}

/*
 * Include binary file
 */
int PO_binary() {
  FILE* Filehandle;
  byte  Filename[LNPMAX + 1];/* file name */
  int   Byte,
        fSizeGiven = 0;
  value Size,
        Skip = 0;

  if(Stream_ReadFilename(Filename, LNPMAX, TRUE) == 0) return(SKIP_REST);
  if(Stream_Comma()) {
    Size = ALU_GetValue_EmptyStrict();
    fSizeGiven = ALU_Flags & MVALUE_EXISTS;
    if(Stream_Comma()) {
      Skip = ALU_GetValue_EmptyStrict();
      if((ALU_Flags & MVALUE_EXISTS) == 0) Skip = 0;
    }
  }
  if(fSizeGiven) {

    /* explicit size info, so truncate or pad file */
    if(nNeedValue_Now || nErrors) {
      /* don't insert file if it's just a waste of time */
      PC_inc += Size;
    } else {
      /* really try to open file and insert it */
      if((Filehandle = File_OpenFile(Filename, FILE_READBINARY))) {
        fseek(Filehandle, Skip, SEEK_SET);
        Byte = fgetc(Filehandle);
        while(Size) {
          if(Byte == EOF) Byte = 0;
          Output_8b(Byte);
          Size--;
          Byte = fgetc(Filehandle);
        }
        File_CloseFile(Filehandle);/* close binary file */
      } else {
        ThrowError(Exception_CannotOpenInFile);
        return(SKIP_REST);
      }
    }
  } else {

    /* no explicit size info, so read file until EOF */
    if((Filehandle = File_OpenFile(Filename, FILE_READBINARY))) {
      fseek(Filehandle, Skip, SEEK_SET);
      Byte = fgetc(Filehandle);
      while(Byte != EOF) {
        Output_8b(Byte);
        Byte = fgetc(Filehandle);
      }
      File_CloseFile(Filehandle);/* close binary file */
    } else {
      ThrowError(Exception_CannotOpenInFile);
      return(SKIP_REST);
    }
  }
  if((Pass_Flags & PASS_ISFIRST) && (Process_Verbosity > 1)) {
    printf(psvBinary, PC_inc, PC_inc, Skip, Skip);
  }
  return(ENSURE_EOL);
}

/*
 * Choose default conversion table (add block support !!!)
 */
int PO_convtab() {
  treeItem* p;

  SKIPSPACE;
  if((GotByte == '<') || (GotByte == '"')) {
    /* !!! read conversion table from file !!! */
    ThrowError(Exception_NotYet);
    return(SKIP_REST);
  }
  /* parse keyword */
  if(Stream_ReadKeyword(MiscString, FALSE) == 0) return(SKIP_REST);
  /* Search for tree item. Don't create */
  MiscString_ToLower();
  p = Tree_ScanROM(MiscString, HTYPE_CONV);
  if(p) Context_CodeTable = p->Body.Opcode.Code;/* set conversion table */
  else  ThrowError(Exception_Syntax);
  return(ENSURE_EOL);
}

/*
 * Switch to CBM mode
 */
int PO_cbm() {
  Context_CodeTable = CODETABLE_PET;
  if(Pass_Flags & PASS_ISFIRST) ThrowWarning(Exception_cbm);
  return(ENSURE_EOL);
}

/*
 * Insert string
 */
int PO_string(int hCT, byte EOR) {
  byte* p;
  byte  b;
  int   Temp = Context_CodeTable;/* buffer current conversion table */

  SKIPSPACE;
  Context_CodeTable = hCT;/* make given conversion table current one */
  do {
    if(GotByte == '"') {
      /* parse string */
      if(Stream_FinishQuoted(MiscString, LSMAX, 0)) {
        p = MiscString;
        while((b = *(p++))) {
          Output_8b(EOR ^ ConvertChar(b, Context_CodeTable));
        }
      } else return(SKIP_REST);/* error */
    } else {
      /* Parse value. No problems with single characters because the current */
      /* conversion table is temporarily set to the given one. */
      Output_8b(ALU_GetValue_Medium());
    }
  } while(Stream_Comma());

  Context_CodeTable = Temp;/* reactivate buffered conversion table */
  return(ENSURE_EOL);
}

/*
 * Insert text string (default format)
 */
int PO_text() {
  return(PO_string(Context_CodeTable, 0));
}

/*
 * Insert raw string
 */
int PO_raw() {
  return(PO_string(CODETABLE_RAW, 0));
}

/*
 * Insert PetSCII string
 */
int PO_petscii() {
  return(PO_string(CODETABLE_PET, 0));
}

/*
 * Insert screencode string
 */
int PO_screencode() {
  return(PO_string(CODETABLE_SCR, 0));
}

/*
 * Insert screencode string, EOR'd
 */
int PO_scrxor() {
  ALU_GetValue_Medium();
  if(Stream_Comma()) return(PO_string(CODETABLE_SCR, (byte) ALU_Value));
  ThrowError(Exception_Syntax);
  return(SKIP_REST);
}
