/*
 * ACME - a crossassembler for producing 6502/65c02/65816 code.
 * Copyright (C) 1998 Marco Baye
 * Have a look at "acme.c" for further info
 */

/*
 * Label stuff
 */

#include "label.h"

/*
 * Constants
 */

/* error message strings */
char Exception_LabelTwice[] = "Label already defined.";
char Exception_TooLate[] = "Too late for postfix.";
char Exception_No2ndDumpfile[] = "Label dump file already chosen.";
char Exception_CannotOpenDumpFile[] = "Cannot open label dump file.";

/*
 * Variables
 */

FILE* hfDump = NULL;/* dump file handle */

/*
 * Create new label structure
 */
label* Label_Create(int fb, int Refs) {
  label* Label = (label*) ALLOC_TRY(sizeof(label));

  /* Finish empty label item */
  Label->Flags = LABELFLAG_DEFAULT | fb;
  Label->Value = 0;
  Label->Refs = Refs;/* number of initial references */
  Label->Usage = 0;/* usage count */

  return(Label);
}

/*
 * Decrement number of references to label structure and delete if zero
 */
void Label_Remove(label* Label) {
  Label->Refs--;
  /* don't really delete as this is done by FREE_TRY() */

/* Memory leak !!! This means that in each and every
 * pass, new labels for macro parameters are created */
}

/*
 * Assign value to label. The routine acts upon the label's flag bits and
 * produces an error if needed. "p" points to the label struct, "v" is the
 * value to assign, ff are the flags to set.
 */
void Label_SetValue(label* p, value v, int ff, int fRedefine) {
  int Flags = p->Flags;

  /* value stuff */

  if((Flags & MVALUE_DEFINED) && (fRedefine == FALSE)) {
    /* Label is already defined, so compare new and old values */
    if(v != p->Value) ThrowError(Exception_LabelTwice);
  } else {
    /* Label is not yet defined OR redefinitions are allowed */
    p->Value = v;
  }

  /* flags stuff */

  /* Ensure that "unsure" labels without "isByte" state don't get that */
  if((Flags & (MVALUE_UNSURE | MVALUE_ISBYTE)) == MVALUE_UNSURE) {
    ff &= ~MVALUE_ISBYTE;
  }
  if(fRedefine) {
    Flags = (Flags & MVALUE_UNSURE) | ff;
  } else {
    if((Flags & MVALUE_FORCEBITS) == 0) {
      if((Flags & (MVALUE_UNSURE | MVALUE_DEFINED)) == 0) {
        Flags |= ff & MVALUE_FORCEBITS;
      }
    }
    Flags |= ff & ~MVALUE_FORCEBITS;
  }
  p->Flags = Flags;
}

/*
 * Parses a label definition (can be either global or local). The stringbuffer
 * holds the label name, "Len" gives the length, "Zone" its zone.
 */
void Label_ParseDefinition(zone Zone) {/* GotByte = illegal */
  label* Label;
  int    fb = Mnemo_GetForceBit();/* skips spaces */

  Label = Label_Find(Zone, fb);
  if(GotByte == '=') {

    /* label = parsed value */
    GetByte();/* proceed with next char */
    ALU_GetValue_Medium();
/* Bug fix: Don't include the above call as a parameter inside the
 * next call. It changes ALU_Flags and some compilers might not notice this.*/
    Label_SetValue(Label, ALU_Value, ALU_Flags, FALSE);
    EnsureEOL();
  } else {

    /* label = PC */
    if(CPU_fPCdefined == FALSE) {
      ThrowError(Exception_NoPC);
      CPU_fPCdefined = TRUE;
    }
    Label_SetValue(Label, CPU_PC, MVALUE_DEFINED, FALSE);
  }
}

/*
 * Search for label. If it was created, it is now given the size info "fb".
 */
label* Label_Find(zone Zone, int fb) {
  treeItem* Item;
  label*    Label;

  Item = Tree_ScanRAM(Zone | BM_LABEL, TRUE);
  if(Tree_ItemCreated) Item->Body.Pointer = Label_Create(fb, 1);

  Label = Item->Body.Pointer;
  if((Tree_ItemCreated == FALSE) && fb) {
    if((Label->Flags & MVALUE_FORCEBITS) != fb) {
      ThrowError(Exception_TooLate);/* too late for postfix */
    }
  }
  return(Label);
}

/*
 * (Re)set label
 */
int Label_PO_set() {
  label* Label;
  int    Flags,
         ForceBit;
  zone   Zone = ZONE_GLOBAL;

  SKIPSPACE;
  if(GotByte == '.') {
    Zone = Context_CurrentZone;
    GetByte();
  }
  /* after ReadKeyword: GotByte = illegal char */
  if(Stream_ReadKeyword(MiscString, FALSE) == 0) return(SKIP_REST);
  ForceBit = Mnemo_GetForceBit();/* skips spaces */
  Label = Label_Find(Zone, ForceBit);
  if(GotByte != '=') {
    ThrowError(Exception_Syntax);
    return(SKIP_REST);
  }
  /* label = parsed value */
  GetByte();/* proceed with next char */
  ALU_GetValue_Medium();
  /* clear label's force bits and set new ones */
  Flags = ALU_Flags;
  Label->Flags &= ~(MVALUE_FORCEBITS | MVALUE_ISBYTE);
  if(ForceBit) {
    Label->Flags |= ForceBit;
    Flags &= ~(MVALUE_FORCEBITS | MVALUE_ISBYTE);
  }
  Label_SetValue(Label, ALU_Value, Flags, TRUE);
  return(ENSURE_EOL);
}

/*
 * Select dump file
 */
int Label_PO_sl() {
  byte Filename[LNPMAX + 1];/* file name */

  if((Pass_Flags & PASS_ISFIRST)) {
    if(hfDump) {
      ThrowError(Exception_No2ndDumpfile);
    } else {
      if(Stream_ReadFilename(Filename, LNPMAX, FALSE)) {
        if((hfDump = File_OpenFile(Filename, FILE_WRITETEXT))) {
          PLATFORM_SETFILETYPE(Filename, FILETYPE_TEXT);
          return(ENSURE_EOL);
        }
        ThrowError(Exception_CannotOpenDumpFile);
      }
    }
  }
  return(SKIP_REST);
}

/*
 * Dump label value and flags to dump file
 */
void Label_DumpOne(treeItem* Item) {
  int    ff;
  value  v;
  label* Label = Item->Body.Pointer;

  fprintf(hfDump, "%s", Item->ID_String);/* output name */
  ff = Label->Flags;
  v  = Label->Value;
  switch(ff & MVALUE_FORCEBITS) {

    case MVALUE_FORCE16:
    fprintf(hfDump, "+2=");
    break;

    case MVALUE_FORCE16 | MVALUE_FORCE24:
    /*FALLTHROUGH*/

    case MVALUE_FORCE24:
    fprintf(hfDump, "+3=");
    break;

    default:
    fprintf(hfDump, "  =");

  }
  if(ff & MVALUE_DEFINED) fprintf(hfDump, "$%x", (u_int) v);
  else                    fprintf(hfDump, " ?");
  if(ff & MVALUE_UNSURE) fprintf(hfDump, "; ?");
  if(Label->Refs != 1) fprintf(hfDump, "; %d refs", Label->Refs);
  if(Label->Usage == 0) fprintf(hfDump, "; unused");
  fprintf(hfDump, "\n");
}

/*
 * Dump global labels into dump file
 */
void Label_DumpAll() {
  if(hfDump) {
    Tree_Dump256(TreesRAM, ZONE_GLOBAL | BM_LABEL, Label_DumpOne);
    File_CloseFile(hfDump);/* close dump file */
    hfDump = NULL;
  }
}
