/*
 * ACME - a crossassembler for producing 6502/65c02/65816 code.
 * Copyright (C) 1998 Marco Baye
 * Have a look at "acme.c" for further info
 */

/*
 * CPU stuff
 */

#include "cpu.h"

/*
 * Constants
 */

char Exception_UkCPU[] = "Unknown processor.";

/*
 * Variables
 */

int   CPU_TypeBit;/* Handle bit of current CPU (default 6502) */
int   CPU_fLongA;/* Flag for long accumulator (default off) */
int   CPU_fLongR;/* Flag for long index registers (default off) */
int   CPU_BufLongA;/* Buffer for these flags while other CPU is active */
int   CPU_BufLongR;
int   CPU_fPCdefined;/* Flag: PC is defined */
value CPU_PC;/* (Pseudo) program counter at start of statement */

/*
 * Set default values for pass
 */
void CPU_InitForPass() {
  CPU_TypeBit  = CPUBIT_6502;/* default CPU = 6502 */
  CPU_fLongA   = FALSE;/* short accu and registers */
  CPU_fLongR   = FALSE;
  CPU_BufLongA = FALSE;
  CPU_BufLongR = FALSE;
}

/*
 * Parse (re-)definitions of program counter
 */
void CPU_SetPC() {/* GotByte = "*" */
  value v;

  NEXTANDSKIPSPACE;/* proceed with next char */
  /* re-definitions of program counter change segment */
  if(GotByte == '=') {
    GetByte();/* proceed with next char */
    v = ALU_GetValue_Strict();
    if(CPU_fPCdefined) {
      /* not first segment, so show status of previous one */
      if(v < PC_Lowest) PC_Lowest = v;
      /* "Lowest" and "Highest" still needed? */
      String_EndSegment();
      if(Pass_Flags & PASS_ISFIRST) {
        Segment_Check(v);
        Segment_FindMax(v);
      }
    } else {
      /* first segment, so buffer some things */
      PC_Lowest = v;
      PC_Highest = v;
    }
    SegmentStart = v;
    CPU_PC = v;
    PC_Mem = v;
    CPU_fPCdefined = TRUE;
    EnsureEOL();
  } else {
    ThrowError(Exception_Syntax);
    SkipRest();
  }
}

/*
 * Select CPU ("!cpu" pseudo opcode) (add block support !!! caution, al/rl)
 */
int CPU_Choose() {
  treeItem* p;
  byte      handle;

  SKIPSPACE;
  if(Stream_ReadKeyword(MiscString, FALSE) == 0) return(SKIP_REST);
  /* search for tree item */
  p = Tree_ScanROM(MiscString, HTYPE_CPU);
  if(p) {
    handle = p->Body.Opcode.Code;/* get handle */
    CPU_TypeBit = handle;
    if(handle == CPUBIT_65816) {
      CPU_fLongA = CPU_BufLongA;/* use old values */
      CPU_fLongR = CPU_BufLongR;
    } else {
      CPU_fLongA = FALSE;/* other processors can't do it */
      CPU_fLongR = FALSE;
    }
  } else ThrowError(Exception_UkCPU);
  return(ENSURE_EOL);
}

/*
 * Switch to long accu ("!al" pseudo opcode) (add block support !!! caution)
 */
int CPU_SetLongA() {
  if(CPU_TypeBit == CPUBIT_65816) {
    CPU_fLongA   = TRUE;
    CPU_BufLongA = TRUE;
  } else ThrowError(Exception_WrongCPU);
  return(ENSURE_EOL);
}

/*
 * Switch to long index registers ("!rl" pseudo opcode) (add block support !!!)
 */
int CPU_SetLongR() {
  if(CPU_TypeBit == CPUBIT_65816) {
    CPU_fLongR   = TRUE;
    CPU_BufLongR = TRUE;
  } else ThrowError(Exception_WrongCPU);
  return(ENSURE_EOL);
}

/*
 * Switch to short accu ("!as" pseudo opcode) (add block support !!! caution)
 */
int CPU_SetShortA() {
  CPU_fLongA   = FALSE;
  CPU_BufLongA = FALSE;
  return(ENSURE_EOL);
}

/*
 * Switch to short index registers ("!rs" pseudo opcode) (add block support !!!)
 */
int CPU_SetShortR() {
  CPU_fLongR   = FALSE;
  CPU_BufLongR = FALSE;
  return(ENSURE_EOL);
}
