/*
 * ACME - a crossassembler for producing 6502/65c02/65816 code.
 * Copyright (C) 1998 Marco Baye
 * Have a look at "acme.c" for further info
 */

/*
 * Arithmetic/logic unit
 */

#include <math.h>
#include <stdio.h>
#include <stdlib.h>

#include "alu.h"
#include "config.h"
#include "data.h"
#include "flowpo.h"
#include "core.h"
#include "stream.h"
#include "strings.h"

/*
 * Constants
 */

char Exception_DivZero[] = "Division by zero.";
char Exception_NegExp[]  = "Exponent is negative.";
char Exception_TooOpen[]   = "Too many '('.";
char Exception_TooClosed[] = "Too many ')'.";
char Exception_UkOp[] = "Unknown operator.";
char Exception_BogusA[] = "Bogus expression, code A.";
char Exception_BogusB[] = "Bogus expression, code B.";
char BIA_Paren[] = "strange parenthesis";
char BIA_UnknownOperator[] = "invalid operator handle";

/*
 * Variables
 */

value  ALU_Value;/* Expression value */
int    ALU_Flags;/* Expression flags */
int    ALU_LeftParentheses;/* Number of parentheses still open */
int    sp_Op;/* operator stack pointer */
int    sp_Val;/* value stack pointer */
value  ValStack [MAXLEVEL];/* value stack */
int    FlagStack[MAXLEVEL];/* value flags stack */
int    OpStack  [MAXLEVEL];/* operator stack */
int    PrioStack[MAXLEVEL];/* priority stack */
int    fIndirect;/* Flag for indirect addressing */
/* (indicated by useless parentheses) */
/* Contains either 0 or MVALUE_INDIRECT */
int    State;/* deterministic finite automaton */
enum {
  STATE_EXPECTMONADIC,/* expect value or monadic operand */
  STATE_EXPECTDYADIC, /* expect dyadic operator */
  STATE_TRY2REDUCE,   /* try to reduce stacks */
  STATE_MAX_GO_ON,    /* "border value" to find those that stop parsing */
  STATE_ERROR,
  STATE_END,
};

/*
 * Prototypes
 */

inline void ALU_State_ExpectMonadic(void);
inline void ALU_State_ExpectDyadic(void);
inline void ALU_State_Try2Reduce(void);
inline void ALU_ParseQuotedCharacter(void);
inline void ALU_ParseBinaryValue(void);
inline void ALU_ParseOctalValue(void);
inline void ALU_ParseDecimalValue(void);
void ALU_ParseHexadecimalValue(void);
void ALU_GetLabelValue(zone);
void ALU_GetValue_Core(void);

#define LEFTVAL  (ValStack[sp_Val-2])
#define RIGHTVAL (ValStack[sp_Val-1])
#define LEFTFLG  (FlagStack[sp_Val-2])
#define RIGHTFLG (FlagStack[sp_Val-1])

#define PUSH_OPERATOR(x, y) ;\
  OpStack[sp_Op] = (x);\
  PrioStack[sp_Op++] = (y) ;

#define PUSH_OPERAND(x, y) ;\
  ValStack[sp_Val] = (x);\
  FlagStack[sp_Val++] = (y);

/*
 * These routines handle numerical expressions. There are operators for
 * arithmetic, logic, shift and comparison operations.
 *
 * There are four different ways to call the core routine:
 *
 * ALU_GetValue_Strict()
 * ALU_GetValue_EmptyStrict()
 * ALU_GetValue_Medium()
 * ALU_GetValue_Liberal()
 *
 * After calling one of these functions, some global variables are set:
 *
 * ALU_Value     holds the numerical result.
 * ALU_Flags     holds some additional informational flags about the result:
 *   Unsure    = A label was referenced that had its "unsure"-flag set
 *   Defined   = The result is known (no "undefined"-label was referenced)
 *   ForceBits = To enforce oversized addressing modes
 *   IsByte    = If the value fits in 8 bits.
 *   Exists    = Expression wasn't empty
 *   Indirect  = Needless parentheses indicate indirect addressing
 *
 * "Unsure" is needed for producing the same addresses in all passes; because
 * in the first pass there will almost for sure be labels that are undefined,
 * you can't simply get the addressing mode from looking at the parameter's
 * value.
 *
 * "Defined" shows that the value really could be computed - so if an undefined
 * label was referenced, this flag will be cleared.
 *
 * "Indirect" is needed to recognize unnecessary parentheses (which imply use
 * of indirect adressing modes).
 */

/*
 * This routine needs a defined value. If the result's "defined" flag is
 * clear, it throws a serious error and therefore stops assembly.
 */
value ALU_GetValue_Strict() {
  ALU_GetValue_Core();
  if(ALU_LeftParentheses) ThrowError(Exception_Syntax);
  if((ALU_Flags & MVALUE_DEFINED) == 0) ThrowSerious(Exception_NeedValue);
  return(ALU_Value);
}

/*
 * This routine needs either a defined value or no expression at all. So
 * empty expressions are accepted, but undefined ones are not.
 * If the result's "defined" flag is clear and the "exists" flag is set, it
 * throws a serious error and therefore stops assembly.
 */
value ALU_GetValue_EmptyStrict() {
  ALU_GetValue_Core();
  if(ALU_LeftParentheses) ThrowError(Exception_Syntax);
  if((ALU_Flags & MVALUE_GIVEN) == MVALUE_EXISTS)
    ThrowSerious(Exception_NeedValue);
  return(ALU_Value);
}

/*
 * The 'normal' call, it parses an expression and returns the result.
 * It the result's "exists" flag is clear (=empty expression), it throws an
 * error.
 * If the result's "defined" flag is clear, NeedValue() is called.
 */
value ALU_GetValue_Medium() {
  ALU_GetValue_Core();
  if(ALU_LeftParentheses) ThrowError(Exception_Syntax);
  if(ALU_Flags & MVALUE_EXISTS) {
    if((ALU_Flags & MVALUE_DEFINED) == 0) NeedValue();
  } else ThrowError(Exception_NeedValue);
  return(ALU_Value);
}

/*
 * This routine allows for one "(" too many. Needed when parsing indirect
 * addressing modes where internal indices have to be possible.
 */
value ALU_GetValue_Liberal() {
  ALU_GetValue_Core();
  if(ALU_Flags & MVALUE_EXISTS) {
    if((ALU_Flags & MVALUE_DEFINED) == 0) NeedValue();
  }
  return(ALU_Value);
}

/*
 * The core of it.
 */
void ALU_GetValue_Core() {
  ALU_LeftParentheses = 0;
  sp_Op = 0;/* operator stack pointer */
  sp_Val = 0;/* value stack pointer */
  State = STATE_EXPECTMONADIC;/* begin by reading value (or monadic operator) */
  fIndirect = 0;/* Contains either 0 or MVALUE_INDIRECT */

  PUSH_OPERATOR(HOP_RETURN, PRIO_RETURN);

  do {
    /* make sure stacks can take it */
    if((sp_Op == MAXLEVEL) || (sp_Val == MAXLEVEL)) {
      ThrowError(Exception_TooDeep);
      State = STATE_ERROR;
    }
    switch(State) {

      case STATE_EXPECTMONADIC:
      ALU_State_ExpectMonadic();
      break;

      case STATE_EXPECTDYADIC:
      ALU_State_ExpectDyadic();
      break;/* no fallthrough; state might have be changed to END or ERROR */

      case STATE_TRY2REDUCE:
      ALU_State_Try2Reduce();
      break;

    }
  } while(State < STATE_MAX_GO_ON);

  if(sp_Val != 1) ThrowError(Exception_BogusA);
  if(sp_Op  != 1) ThrowError(Exception_BogusB);

  if(State == STATE_END) {
    /* okay, no errors */
    ALU_Value = ValStack[0];
    ALU_Flags = FlagStack[0];
    ALU_Flags |= fIndirect;

    /* only allow *one* force bit */
    if(ALU_Flags & MVALUE_FORCE24)
      ALU_Flags &= ~(MVALUE_FORCE16 | MVALUE_FORCE08);
    if(ALU_Flags & MVALUE_FORCE16)
      ALU_Flags &= ~MVALUE_FORCE08;
    /* if value is sure, check to set ISBYTE */
    if((ALU_Flags & MVALUE_UNSURE) == 0)
      if((ALU_Value<256) && (ALU_Value>-129)) ALU_Flags |= MVALUE_ISBYTE;
    /* if there was nothing to parse, mark as undefined */
    if((ALU_Flags & MVALUE_EXISTS)==0) ALU_Flags &= ~MVALUE_DEFINED;
    /* if undefined, return zero */
    if((ALU_Flags & MVALUE_DEFINED) == 0)
      ALU_Value = 0;/* if undefined, return 0. */
  } else {
    /* State is STATE_ERROR. But actually, nobody cares. */
    /* ...they have already been reported anyway. :) */
  }

  /* make sure only one additional '(' gets through! move!!! */
  if(ALU_LeftParentheses > 1) {
    ALU_LeftParentheses = 0;
    ThrowError(Exception_TooOpen);
  }
}

/*
 * Expect operand or monadic operator (hopefully inlined)
 */
void ALU_State_ExpectMonadic() {
  int hop,
      prio;

#ifdef FDEBUG
  printf("State_ExpectMonadic\n");
#endif

  SKIPSPACE;
  switch(GotByte) {

/* Real monadic operators (state doesn't change, still ExpectMonadic) */

    case '!':/* NOT operator */
    hop = HOP_NOT;prio = PRIO_NOT;
    goto GetByteAndPushMonadic;

    case '-':/* NEGATION operator */
    hop = HOP_NEGATE;prio = PRIO_NEGATE;
    goto GetByteAndPushMonadic;

    case '<':/* LOWBYTE operator */
    hop = HOP_LOWBYTEOF;prio = PRIO_LOWBYTEOF;
    goto GetByteAndPushMonadic;

    case '>':/* HIGHBYTE operator */
    hop = HOP_HIGHBYTEOF;prio = PRIO_HIGHBYTEOF;
    goto GetByteAndPushMonadic;

    case '^':/* BANKBYTE operator */
    hop = HOP_BANKBYTEOF;prio = PRIO_BANKBYTEOF;
    goto GetByteAndPushMonadic;

/* Faked monadic operators */

    case '(':/* left parenthesis */
    hop = HOP_OPENING;prio = PRIO_OPENING;
    goto GetByteAndPushMonadic;

    case ')':/* right parenthesis, this makes "()" also throw a syntax error */
    ThrowError(Exception_Syntax);
    State = STATE_ERROR;
    break;

/* Operands (values, state changes to ExpectDyadic) */

    /* Quoted character */
    case '"':
    case '\'':
    /* The character will be converted using the current conversion table.*/
    ALU_ParseQuotedCharacter();/* GotByte = char following closing quote */
    goto NowExpectDyadic;

    /* Binary value */
    case '%':
    ALU_ParseBinaryValue();/* GotByte = non-binary character */
    goto NowExpectDyadic;

    /* Octal value */
    case '&':
    ALU_ParseOctalValue();/* GotByte = non-octal character */
    goto NowExpectDyadic;

    /* Hexadecimal value */
    case '$':
    ALU_ParseHexadecimalValue();/* GotByte = non-hexadecimal character */
    goto NowExpectDyadic;

    /* Program counter */
    case '*':
    GetByte();/* proceed with next char */
    if(CPU_fPCdefined == FALSE) {
      ThrowError(Exception_NoPC);
      CPU_fPCdefined = TRUE;
    }
    PUSH_OPERAND(CPU_PC, MVALUE_GIVEN);/* GotByte = char after closing quote */
    goto NowExpectDyadic;

    /* Local label */
    case '.':
    ValStack[sp_Val] = 0;
    if(Stream_ReadKeyword(MiscString, TRUE)) {/* GotByte = illegal character */
      ALU_GetLabelValue(Context_CurrentZone);
      goto NowExpectDyadic;
    }
    State = STATE_ERROR;
    break;

    /* Decimal values and global labels */
    default:/* (all other characters) */
    if((GotByte > 47) && (GotByte < 58)) {
      ALU_ParseDecimalValue();/* GotByte = non-decimal character */
      goto NowExpectDyadic;
    } /* goto means we don't need an "else {" here */

    if(pFlagTable[GotByte] & BYTEIS_ILLEGAL) {
      /* illegal character read - so don't go on */
      PUSH_OPERAND(0, 0);/* push pseudo value, EXISTS flag is clear */
      if(OpStack[sp_Op-1] == HOP_RETURN) {
        PUSH_OPERATOR(HOP_END, PRIO_END);
        State = STATE_TRY2REDUCE;
      } else {
        ThrowError(Exception_Syntax);
        State = STATE_ERROR;
      }
    } else {
      /* Read global label (or "NOT") */
      /* after ReadKeyword: GotByte = illegal character */
      if(Stream_ReadKeyword(MiscString, FALSE) == 3) {
        /* Check for NOT. Okay, it's hardcoded, but so what ? Sue me... */
        if((MiscString[0]=='N')&&(MiscString[1]=='O')&&(MiscString[2]=='T')) {
          PUSH_OPERATOR(HOP_NOT, PRIO_NOT);
          /* state doesn't change */
        } else {
          ALU_GetLabelValue(ZONE_GLOBAL);
          State = STATE_EXPECTDYADIC;
        }
      } else {
        ALU_GetLabelValue(ZONE_GLOBAL);
        State = STATE_EXPECTDYADIC;
      }
    }
    break;

/* no other possibilities, so here are the shared endings */

GetByteAndPushMonadic:
    GetByte();
    PUSH_OPERATOR(hop, prio);
    /* State doesn't change */
    break;

NowExpectDyadic:
    State = STATE_EXPECTDYADIC;
    break;

  }
}

/*
 * Expect dyadic operator (hopefully inlined)
 */
void ALU_State_ExpectDyadic() {
  int hop,
      prio;

#ifdef FDEBUG
printf("State_ExpectDyadic\n");
#endif

  SKIPSPACE;
  switch(GotByte) {

/* Single-character dyadic operators */

    case '^':/* "to the power of" */
    hop  = HOP_POWEROF;prio = PRIO_POWEROF;
    goto GetByteAndPushDyadic;

    case '+':/* add */
    hop  = HOP_ADD;prio = PRIO_ADD;
    goto GetByteAndPushDyadic;

    case '-':/* subtract */
    hop  = HOP_SUBTRACT;prio = PRIO_SUBTRACT;
    goto GetByteAndPushDyadic;

    case '*':/* multiply */
    hop  = HOP_MULTIPLY;prio = PRIO_MULTIPLY;
    goto GetByteAndPushDyadic;

    case '/':/* divide */
    hop  = HOP_DIVIDE;prio = PRIO_DIVIDE;
    goto GetByteAndPushDyadic;

    case '%':/* modulo */
    hop  = HOP_MODULO;prio = PRIO_MODULO;
    goto GetByteAndPushDyadic;

    case '&':/* bitwise AND */
    hop  = HOP_AND;prio = PRIO_AND;
    goto GetByteAndPushDyadic;

    case '|':/* bitwise OR */
    hop  = HOP_OR;prio = PRIO_OR;
    goto GetByteAndPushDyadic;

/* This part is commented out because there is no EXOR character defined
 *  case '#':/ * bitwise exclusive OR * /
 *  hop  = HOP_EXOR;prio = PRIO_EXOR;
 *  goto GetByteAndPushDyadic;
 */

    case '=':/* is equal */
    hop  = HOP_EQUALS;prio = PRIO_EQUALS;
    goto GetByteAndPushDyadic;

    case ')':/* closing parenthesis */
    hop  = HOP_CLOSING;prio = PRIO_CLOSING;
    goto GetByteAndPushDyadic;

/* Multi-character dyadic operators */

    /* "!=" */
    case '!':
    if(GetByte() == '=') {
      hop  = HOP_DIFFERENT;prio = PRIO_DIFFERENT;
      goto GetByteAndPushDyadic;
    } else {
      ThrowError(Exception_Syntax);
      State = STATE_ERROR;
    }
    break;

    /* "<", "<=", "<<" and "<>" */
    case '<':
    switch(GetByte()) {

      case '=':/* "<=", less or equal */
      hop  = HOP_LE;prio = PRIO_LE;
      goto GetByteAndPushDyadic;

      case '<':/* "<<", shift left */
      hop  = HOP_SL;prio = PRIO_SL;
      goto GetByteAndPushDyadic;

      case '>':/* "<>", not equal */
      hop  = HOP_DIFFERENT;prio = PRIO_DIFFERENT;
      goto GetByteAndPushDyadic;

      default:/* "<", less than */
      hop  = HOP_LT;prio = PRIO_LT;
      goto PushDyadic;
    }
    break;

    /* ">", ">=", ">>" and "><" */
    case '>':
    switch(GetByte()) {

      case '=':/* ">=", greater or equal */
      hop  = HOP_GE;prio = PRIO_GE;
      goto GetByteAndPushDyadic;

      case '>':/* ">>", logical shift right */
      hop  = HOP_LSR;prio = PRIO_LSR;
      goto GetByteAndPushDyadic;

      case '<':/* "><", not equal */
      hop  = HOP_DIFFERENT;prio = PRIO_DIFFERENT;
      goto GetByteAndPushDyadic;

      default:/* ">", greater than */
      hop  = HOP_GT;prio = PRIO_GT;
      goto PushDyadic;
    }
    break;

/* end of expression or text version of dyadic operator */
    default:
    /* check string version of operators */
    if((pFlagTable[GotByte] & BYTEIS_ILLEGAL) == 0) {
      treeItem *p;
      Stream_ReadKeyword(MiscString, FALSE);
      /* (now: GotByte = illegal character) */
      /* search for tree item */
      p = Tree_ScanROM(MiscString, HTYPE_OPERATOR);
      if(p) {
        hop  = p->Body.Opcode.Code;prio = p->Body.Opcode.Group;
        goto PushDyadic;
      } else {
        ThrowError(Exception_UkOp);
        State = STATE_ERROR;
      }
    } else {
      hop  = HOP_END;prio = PRIO_END;
      goto PushDyadic;
    }
    break;

/* no other possibilities, so here are the shared endings */

GetByteAndPushDyadic:
    GetByte();
PushDyadic:
    PUSH_OPERATOR(hop, prio);
    State = STATE_TRY2REDUCE;
    break;

  }
}

/*
 * Try to reduce stacks by performing high-priority operations
 */
void ALU_State_Try2Reduce() {

#ifdef FDEBUG
  printf("State_Try2Reduce\n");
#endif

  if(sp_Op < 2) {
#ifdef FDEBUG
    printf("Cannot reduce (too few operators)\n");
#endif
    State = STATE_EXPECTMONADIC;
    return;
  }

  if(PrioStack[sp_Op-2] < PrioStack[sp_Op-1]) {
#ifdef FDEBUG
    printf("Cannot reduce (last one has higher priority)\n");
#endif
    State = STATE_EXPECTMONADIC;
    return;
  }

  switch(OpStack[sp_Op-2]) {

/* special (pseudo) operators */

    case HOP_RETURN:
    /* don't touch fIndirect; needed for INDIRECT flag */
    sp_Op--;/* decrement operator stack pointer */
    State = STATE_END;
    break;

    case HOP_OPENING:
    fIndirect = MVALUE_INDIRECT;/* parentheses found */
    switch(OpStack[sp_Op-1]) {

      case HOP_CLOSING:/* matching parentheses */
      sp_Op -= 2;/* remove both of them */
      State = STATE_EXPECTDYADIC;
      break;

      case HOP_END:/* unmatched parenthesis */
      ALU_LeftParentheses++;/* count */
      goto RNTLObutDontTouchIndirectFlag;

      default:
      BugFound(BIA_Paren);
    }
    break;

    case HOP_CLOSING:
    ThrowError(Exception_TooClosed);
    goto RemoveNextToLastOperator;

/* monadic operators */

    case HOP_NOT:
    RIGHTVAL = ~(RIGHTVAL);
    RIGHTFLG &= ~MVALUE_ISBYTE;
    goto RemoveNextToLastOperator;

    case HOP_NEGATE:
    RIGHTVAL = -(RIGHTVAL);
    RIGHTFLG &= ~MVALUE_ISBYTE;
    goto RemoveNextToLastOperator;

    case HOP_LOWBYTEOF:
    RIGHTVAL = (RIGHTVAL) & 255;
    RIGHTFLG |= MVALUE_ISBYTE;
    RIGHTFLG &= ~MVALUE_FORCEBITS;
    goto RemoveNextToLastOperator;

    case HOP_HIGHBYTEOF:
    RIGHTVAL = ((RIGHTVAL) >> 8) & 255;
    RIGHTFLG |= MVALUE_ISBYTE;
    RIGHTFLG &= ~MVALUE_FORCEBITS;
    goto RemoveNextToLastOperator;

    case HOP_BANKBYTEOF:
    RIGHTVAL = ((RIGHTVAL) >> 16) & 255;
    RIGHTFLG |= MVALUE_ISBYTE;
    RIGHTFLG &= ~MVALUE_FORCEBITS;
    goto RemoveNextToLastOperator;

/* dyadic operators */

    case HOP_POWEROF:
    if(RIGHTVAL >= 0) {
      LEFTVAL = (value) pow((double) LEFTVAL, (double) RIGHTVAL);
    } else {
      if(RIGHTFLG & MVALUE_DEFINED) ThrowError(Exception_NegExp);
      LEFTVAL = 0;
    }
    goto HandleFlagsAndDecStacks;

    case HOP_MULTIPLY:
    LEFTVAL *= RIGHTVAL;
    goto HandleFlagsAndDecStacks;

    case HOP_DIVIDE:
    if(RIGHTVAL) LEFTVAL /= RIGHTVAL;
    else {
      if(RIGHTFLG & MVALUE_DEFINED) ThrowError(Exception_DivZero);
      LEFTVAL = 0;
    }
    goto HandleFlagsAndDecStacks;

    case HOP_MODULO:
    if(RIGHTVAL) LEFTVAL %= RIGHTVAL;
    else {
      if(RIGHTFLG & MVALUE_DEFINED) ThrowError(Exception_DivZero);
      LEFTVAL = 0;
    }
    goto HandleFlagsAndDecStacks;

    case HOP_ADD:
    LEFTVAL += RIGHTVAL;
    goto HandleFlagsAndDecStacks;

    case HOP_SUBTRACT:
    LEFTVAL -= RIGHTVAL;
    goto HandleFlagsAndDecStacks;

    case HOP_SL:
    LEFTVAL <<= RIGHTVAL;
    goto HandleFlagsAndDecStacks;

    case HOP_LSR:
    LEFTVAL >>= RIGHTVAL;
    goto HandleFlagsAndDecStacks;

    case HOP_LE:
    LEFTVAL = (LEFTVAL <= RIGHTVAL);
    goto HandleFlagsAndDecStacks;

    case HOP_LT:
    LEFTVAL = (LEFTVAL < RIGHTVAL);
    goto HandleFlagsAndDecStacks;

    case HOP_GE:
    LEFTVAL = (LEFTVAL >= RIGHTVAL);
    goto HandleFlagsAndDecStacks;

    case HOP_GT:
    LEFTVAL = (LEFTVAL > RIGHTVAL);
    goto HandleFlagsAndDecStacks;

    case HOP_DIFFERENT:
    LEFTVAL = (LEFTVAL != RIGHTVAL);
    goto HandleFlagsAndDecStacks;

    case HOP_EQUALS:
    LEFTVAL = (LEFTVAL == RIGHTVAL);
    goto HandleFlagsAndDecStacks;

    case HOP_AND:
    LEFTVAL &= RIGHTVAL;
    goto HandleFlagsAndDecStacks;

    case HOP_EXOR:
    LEFTVAL ^= RIGHTVAL;
    goto HandleFlagsAndDecStacks;

    case HOP_OR:
    LEFTVAL |= RIGHTVAL;
    goto HandleFlagsAndDecStacks;

    default:
    BugFound(BIA_UnknownOperator);
    break;

/* no other possibilities, so here are the shared endings */

/* entry point for dyadic operators */
HandleFlagsAndDecStacks:
    /* Handle flags and decrement value stack pointer */
    /* "OR" EXISTS, UNSURE and FORCEBIT flags */
    LEFTFLG |= RIGHTFLG & (MVALUE_EXISTS|MVALUE_UNSURE|MVALUE_FORCEBITS);
    LEFTFLG &= (RIGHTFLG | ~MVALUE_DEFINED);/* "AND" DEFINED flag */
    LEFTFLG &= ~MVALUE_ISBYTE;/* clear ISBYTE flag */
    sp_Val--;
    /* FALLTHROUGH */

/* entry point for monadic operators */
RemoveNextToLastOperator:
    fIndirect = 0;/* toplevel operation was something other than parentheses */
    /* FALLTHROUGH */

/* entry point for '(' operator (has set fIndirect, so don't clear now) */
RNTLObutDontTouchIndirectFlag:
    /* Remove operator and shift down next one */
    OpStack[sp_Op-2]=OpStack[sp_Op-1];
    PrioStack[sp_Op-2]=PrioStack[sp_Op-1];
    sp_Op--;/* decrement operator stack pointer */
    break;

  }
}

/*
 * Routine for parsing a quoted character. The character will be converted
 * using the current conversion table.
 */
void ALU_ParseQuotedCharacter() {
  value v;
  byte  c = GetByte();

  if(QuoteChar != NO_QUOTE_CHAR) {
    v = (value) ConvertChar(c, Context_CodeTable);
    GetByte();/* Read closing quote (hopefully) */
    if(QuoteChar == NO_QUOTE_CHAR) {
      GetByte();/* If length == 1, proceed with next byte */
    } else {
      ThrowError(Exception_TooLong);/* If longer than one character */
      /* ToDo: Skip until closing quote !!! */
    }
  } else {
    ThrowError(Exception_MissingString);/* If shorter than one char */
    v = 0;
  }
  PUSH_OPERAND(v, MVALUE_GIVEN | MVALUE_ISBYTE);
  /* GotByte = char following closing quote */
}

/*
 * Routine for parsing a binary value. Apart from "0" and "1", it also accepts
 * characters "." and "#", this is much more readable. The current value is
 * stored as soon as a character is read that is none of those given above.
 */
void ALU_ParseBinaryValue() {/* GotByte = "%" */
  value v = 0;
  int   Flags = MVALUE_GIVEN,
        fCont = TRUE,/* continue loop flag */
        c = -1;/* digit counter */

  do {
    c++;
    switch(GetByte()) {

      case '0':
      case '.':
      v <<= 1;
      break;

      case '1':
      case '#':
      v = (v << 1) + 1;
      break;

      default:
      fCont = FALSE;
    }
  } while(fCont);
  if(c > 8) {
    if(c > 16) {
      if(v < 65536) Flags |= MVALUE_FORCE24;
    } else {
      if(v <   256) Flags |= MVALUE_FORCE16;
    }
  }
  PUSH_OPERAND(v, Flags);
  /* GotByte = non-binary character */
}

/*
 * Routine for parsing an octal value. It accepts "0" to "7". The current
 * value is stored as soon as a character is read that is none of those given
 * above.
 */
void ALU_ParseOctalValue() {/* GotByte = "&" */
  value v = 0;
  int   Flags = MVALUE_GIVEN,
        c = 0;/* digit counter */

  GetByte();
  while((GotByte > 47) && (GotByte < 56)) {
    v = (v << 3) + (GotByte & 7);
    c++;
    GetByte();
  }

  if(c > 3) {
    if(c > 6) {
      if(v<65536) Flags |= MVALUE_FORCE24;
    } else {
      if(v<  256) Flags |= MVALUE_FORCE16;
    }
  }
  PUSH_OPERAND(v, Flags);
  /* GotByte = non-octal character */
}

/*
 * Routine for parsing a decimal value. It accepts "0" to "9". The current
 * value is stored as soon as a character is read that is none of those given
 * above. Unlike the others, the "decimal" routine expects the first digit to
 * be read already, because decimal values don't use any prefixes.
 * If the first two digits are "0x", this routine branches to the one for
 * parsing hexadecimal values.
 */
void ALU_ParseDecimalValue() {/* GotByte = first digit */
  value v = (GotByte & 15);

  GetByte();
  if((v == 0) && (GotByte == 'x')) {
    ALU_ParseHexadecimalValue();
  } else {
    while((GotByte > 47) && (GotByte < 58)) {
      v *= 10;
      v += (GotByte & 15);
      GetByte();
    }
    PUSH_OPERAND(v, MVALUE_GIVEN);
  }
  /* GotByte = non-decimal character */
}

/*
 * Routine for parsing a hexadecimal value. It accepts "0" to "9", "a" to "f"
 * and "A" to "F". Capital letters will be converted to lowercase letters using
 * the flagtable. The current value is stored as soon as a character is read
 * that is none of those given above.
 */
void ALU_ParseHexadecimalValue() {/* GotByte = "$" or "x" */
  value v = 0;
  byte  b;
  int   Flags = MVALUE_GIVEN,
        fCont,/* continue loop flag */
        c = -1;/* digit counter */

  do {
    c++;
    fCont = FALSE;
    b = GetByte();
    /*  first, convert "A-F" to "a-f" */
    if(pFlagTable[b] & BYTEIS_UPCASE) b |= 32;
    /* if digit, add digit value */
    if((b > 47) && (b < 58)) {
      v = (v << 4) + (b & 15);
      fCont = TRUE;/* keep going */
    }
    /* if legal ("a-f") character, add character value */
    if((b > 96) && (b < 103)) {
      v = (v << 4) + (b & 15) + 9;
      fCont = TRUE;/* keep going */
    }
  } while(fCont);
  if(c > 2) {
    if(c > 4) {
      if(v<65536) Flags |= MVALUE_FORCE24;
    } else {
      if(v<  256) Flags |= MVALUE_FORCE16;
    }
  }
  PUSH_OPERAND(v, Flags);
  /* GotByte = non-hexadecimal character */
}

/*
 * Lookup (and create, if necessary) label tree item and return value. The
 * stringbuffer holds the label's name and "Zone" its zone.
 */
void ALU_GetLabelValue(zone Zone) {
  label* Label;
  int    Flags;

  Label = Label_Find(Zone, 0);
  /* If the label was produced just now, we have to mark it as unsure */
  if(Tree_ItemCreated) Label->Flags |= MVALUE_UNSURE;
  Flags = Label->Flags;
  /* in first pass, count usage */
  if(Pass_Flags & PASS_ISFIRST) Label->Usage++;

  PUSH_OPERAND(Label->Value, Flags | MVALUE_EXISTS);
}
