/*----------------------------------------------------------------------+
|   Copyright 1999, Caldera Thin Client Systems, Inc.                   |
|   This software is licensed under the GNU Public License.             |
|   See LICENSE.TXT for further information.                            |
|                                                                       |
|   Historical Copyright                                                |
|                                                                       |
|	Copyright ( c ) 1992 Digital Research Inc.		        |
|    The software contained in this listing is proprietary to       	|
|    Digital Research Inc., Pacific Grove, California and is        	|
|    covered by U.S. and other copyright protection.  Unauthorized  	|
|    copying, adaptation, distribution, use or display is prohibited	|
|    and may be subject to civil and criminal penalties.  Disclosure	|
|    to others is prohibited.  For the terms and conditions of soft-	|
|    ware code use refer to the appropriate Digital Research        	|
|    license agreement.							|
+----------------------------------------------------------------------*/

/*----------------------------------------------------------------------+
|  file:	RADIO.C							|
|									|
|  purpose:	Initializes and draws CUA-type radio buttons and toggle	|
|       	boxes.							|
|									|
|  entry points:							|     
|               CUA_InitRbutton() : Init of bezier for radio buttons.	|
|       	CUA_ToggleBox()   : Draws crossed toggle-type buttons.	|
|       	CUA_RadioButton() : Draws circular radio buttons.	|
|									|
|  calling syntax:							|
|									|
|       void	CUA_InitRbutton( short wbox, short hbox)		|
|       void	CUA_ToggleBox( DPARMBLK far * pb_ptr )			|
|       void	CUA_RadioButton( DPARMBLK far * pb_ptr )		|
|									|
+----------------------------------------------------------------------*/
	   
/*----------------------------------------------------------------------+
|  NOTE:								|
|       This file is easily broken into two pieces for overlaying	|
|       purposes.  One module should contain the functions:		|
|       	bitplot()		called fm build_ellipse() only  |
|       	build_ellipse()		called fm CUA_InitRbutton() only|
|       	bigMulDiv()		called fm CUA_InitRbutton() only|
|       	CUA_InitRbutton()	called after opn wkstn in app	|
|       and should be located in the initialization overlay.		|
|									|
|       Another module can be broken off to contain the functions:	|
|       	bigMulDiv()		called from tog*() and rad*()	|
|       	toggl_pts()		called from CUA_ToggleBox() only|
|       	fmemcpy()   		called from CUA_ToggleBox() only|
|       	CUA_ToggleBox()		called from dr_code() in app	|
|       	CUA_RadioButton()		""	""		|
|       and should be located in an overlay accessible whenever the prog|
|       wishes to actually draw one of these things (like when dialogs	|
|       are displayed).							|
|									|
|       If you overlay, make sure the static vars are made global.	|
+----------------------------------------------------------------------*/
#include "shell.h"
#include "funcdef.h"
#include "danutil.h"

#if CUA_BUTTONS

typedef struct {
    WORD	x;
    WORD	y;
} POINT;

#define MAXRADIUS	32			/* radius in pixels	    */
#define XWIDTH		(MAXRADIUS * 2 + 1)	/* diameter in pixels	    */
#define XWORDS		((XWIDTH + 15) / 16)	/* diameter width in words  */

WORD far *	NormalRbutton;
WORD far *	SelctdRbutton;
MLOCAL	WORD	wBox, hBox ;		/* height and width of boxes/buttons*/


/*----------------------------------------------------------------------*/
short Rbutton_size( void )
{
    return(sizeof(short) * ((MAXRADIUS * 2 + 1) * XWORDS));
}

/*----------------------------------------------------------------------*/
/* A multiply and divide routine to use when overflow of the multiply	*/
/* is a possiblity.							*/
/*			Replaces SMUL_DIV()...				*/
short bigMulDiv(	short	mul1,
			short	mul2,
			short	div )
{
	return( (short)( (long)mul1 * (long)mul2 / (long)div ) ) ;
} /* bigMulDiv() */

/*--------------------------------------------------------------------------
 Set the specified bit in the NormalRbutton or SelctdRbutton array.  The 
 ell_bit array is used to blit an ellipse to the screen and the SelctdRbutton
 array is used to blit a filled ellipse to the screen.  (For CUA radio
 buttons.)
---------------------------------------------------------------------------*/
static void bitplot(	short x,
			short y,
			short filled )
{
    short	ii ;
    WORD far	*pShort ;		/* pointer to a short */
    
    ii = y * XWORDS + (x >> 4);
    pShort = ( filled != 0 ) ? (SelctdRbutton + ii) : (NormalRbutton + ii);
    *pShort |= (1 << (15 - (x & 0x000f)));
} /* bitplot */

/*--------------------------------------------------------------------------*/
/* Build the ellipses for the CUA radio buttons.			    */
static void build_ellipse(	short centerx, 
				short centery, 
				short radiusx,
				short radiusy, 
				short fillit )
{
    long	d1, d2;
    short	x, y, i;
    long	t1, t2, t3, t4, t5, t6, t7, t8, t9;
    
        /* intermediate terms to speed up loop */
    t1 = (long) radiusx * (long) radiusx;	t2 = t1 << 1;	t3 = t2 << 1;
    t4 = (long) radiusy * (long) radiusy;	t5 = t4 << 1;	t6 = t5 << 1;
    t7 = (long) radiusx * t5;			t8 = t7 << 1;	t9 = 0L;

    d1 = t2 - t7 + (t4 >> 1);	/* error terms */
    d2 = (t1 >> 1) - t8 + t5;

    x = radiusx;
    y = 0;

    while (d2 < 0) {		/* region 1 of ellipse */
	/* draw 4 points using symmetry */
	if (fillit) {
	    for (i = centerx -  x; i <= centerx + x; i++) {
		bitplot(i, centery + y, fillit );
		bitplot(i, centery - y, fillit );
	    }
	} else {
	    bitplot(centerx + x, centery + y, fillit );
	    bitplot(centerx + x, centery - y, fillit );
	    bitplot(centerx - x, centery + y, fillit );
	    bitplot(centerx - x, centery - y, fillit );
	}

	y++;		/* always move up here */
	t9 += t3;	
	if (d1 < 0) {		/* move straight up */
	    d1 += t9 + t2;
	    d2 += t9;
	} else {		/* move up and left */
	    x--;
	    t8 -= t6;
	    d1 += t9 + t2 - t8;
	    d2 += t9 + t5 - t8;
	}
    }

    do {				/* region 2 of ellipse */
	/* draw 4 points using symmetry */
	if (fillit) {
	    for (i = centerx -  x; i <= centerx + x; i++) {
		bitplot(i, centery + y, fillit ) ;
		bitplot(i, centery - y, fillit );
	    }
	} else {
	    bitplot(centerx + x, centery + y, fillit );
	    bitplot(centerx + x, centery - y, fillit );
	    bitplot(centerx - x, centery + y, fillit );
	    bitplot(centerx - x, centery - y, fillit );
	}
	
	x--;		/* always move left here */
	t8 -= t6;	
	if (d2 < 0) {	/* move up and left */
	    y++;
	    t9 += t3;
	    d2 += t9 + t5 - t8;
	} else		/* move straight left */
	    d2 += t5 - t8;
    } while (x >= 0);
} /* build_ellipse */

/*--------------------------------------------------------------------------*/
/* Initialize the ellipses for the CUA radio buttons.			    */
void CUA_InitRbutton(	short	_wbox, 
			short	_hbox )
{
    short    radiusx, radiusy ;
    short    centerx, centery ;
#define	    FILLED	1
#define	    NOT_FILLED  0

    FDB    imfdb;

    wBox = _wbox ;
    hBox = _hbox ;

    radiusx = bigMulDiv( ((wBox+1)/2), 15, 20 ) ;    
    radiusy = bigMulDiv( ((hBox+1)/2), 15, 20 ) ;
    radiusx = (radiusx < MAXRADIUS) ? radiusx : MAXRADIUS ;
    radiusy = (radiusy < MAXRADIUS) ? radiusy : MAXRADIUS ;
    centerx = radiusx + 1 ;
    centery = radiusy + 1 ;
    build_ellipse( centerx, centery, radiusx, radiusy, NOT_FILLED ) ;
    radiusx = (radiusx + 1) / 2 ;
    radiusy = (radiusy + 1) / 2 ;
    build_ellipse( centerx, centery, radiusx, radiusy, FILLED ) ;
    
    imfdb.fd_w = XWIDTH;
    imfdb.fd_h = XWIDTH;
    imfdb.fd_wdwidth = XWORDS;
    imfdb.fd_nplanes = 1;
    
    imfdb.fd_addr = (long)NormalRbutton;
    imfdb.fd_stand = STNDFORM;
    vrn_trnfm( (WORD *) &imfdb, (WORD *) &imfdb ) ;
    
    imfdb.fd_addr = (long)SelctdRbutton;
    imfdb.fd_stand = STNDFORM;
    vrn_trnfm( (WORD *) &imfdb, (WORD *) &imfdb ) ;

} /* CUA_InitRbutton */

/*--------------------------------------------------------------------------*/
void CUA_InitToggle( short _wbox, short _hbox )
{
    wBox = _wbox ;
    hBox = _hbox ;
} /* CUA_InitToggle() */

/*--------------------------------------------------------------------------*/
/* NAME: toggl_pts							    */
/*									    */
/* PURPOSE: Setup the x,y coordinate pairs for the CUA toggle box polyline. */
/*--------------------------------------------------------------------------*/
static void toggl_pts(	POINT		pts[] ,
			RECT far	*rect )
{
short    right ;
short    bottom ;
    
    right  = rect->x + bigMulDiv( wBox, 9, 10 ) - 1 ;
    bottom = rect->y + bigMulDiv( hBox, 9, 10 ) - 1 ;

    pts[0].x = pts[3].x = pts[4].x = pts[6].x = rect->x ;
    pts[0].y = pts[1].y = pts[4].y = pts[7].y = rect->y ;
    pts[1].x = pts[2].x = pts[5].x = pts[7].x = right ;
    pts[2].y = pts[3].y = pts[5].y = pts[6].y = bottom ;
} /* toggl_pts */

/*--------------------------------------------------------------------------*/
/* NAME: CUA_ToggleBox							    */
/*									    */
/* PURPOSE: Draw the CUA toggle box.  Empty box if not selected,	    */
/*				      crossed box if selected.		    */
/*--------------------------------------------------------------------------*/
void CUA_ToggleBox(	DPARMBLK far	*pb_ptr ) /* Ptr to parm block.	    */
{
    TREE	obj_ptr;		/* Pointer to object.		    */
    POINT       tog_box[8];		/* polyline coordinate pairs  	    */
    RECT	r ;			/* will contain object's pos & size */
    
    obj_ptr = pb_ptr->pb_tree + pb_ptr->pb_obj ;
    vst_lattr( MD_REPLACE, LT_SOLID, BLACK, LE_SQUARE, LE_SQUARE, 1 ) ;
    toggl_pts( tog_box, (RECT far *)&pb_ptr->pb_r );
    
    if( (pb_ptr->pb_prevstate ^ pb_ptr->pb_currstate) == 0 )
    {
	/* 900225 DLB: Make sure the background is white before drawing.    */
	fmemcpy( (void far *)&r, (void far *)&pb_ptr->pb_r, sizeof( RECT ) ) ;
	r.w = bigMulDiv( wBox, 9, 10 ) ;
	r.h = bigMulDiv( hBox, 9, 10 ) ;
	gsx_1code(S_FILL_COLOR, WHITE);
	bb_fill( MD_REPLACE, FIS_SOLID, 0, r.x, r.y, r.w, r.h ) ;
	if ( (obj_ptr->ob_state & SELECTED) &&
	     !(obj_ptr->ob_state & DISABLED) )
	    v_pline( 8, (WORD *)tog_box );/* draw CUA toggle box  */
	else
	    v_pline( 5, (WORD *)tog_box );/* draw CUA crossed box */
    }
    else {
	if ( (obj_ptr->ob_state & SELECTED) &&
	     !(obj_ptr->ob_state & DISABLED) )
	    v_pline( 8, (WORD *)tog_box );/* draw CUA toggle box  */
	else {
	    fmemcpy( (void far *)&r, (void far *)&pb_ptr->pb_r, sizeof( RECT ) ) ;
	    r.w = bigMulDiv( wBox, 9, 10 ) ;
	    r.h = bigMulDiv( hBox, 9, 10 ) ;
	    gsx_1code(S_FILL_COLOR, WHITE);
	    bb_fill( MD_REPLACE, FIS_SOLID, 0, r.x, r.y, r.w, r.h ) ;
	    v_pline( 5, (WORD *)tog_box );
	}
    }
} /* CUA_ToggleBox() */

/*--------------------------------------------------------------------------*/
/* NAME: CUA_RadioButton						    */
/*									    */
/* PURPOSE: Draw the CUA radio circle.  				    */
/*--------------------------------------------------------------------------*/
void CUA_RadioButton(	DPARMBLK far	*pb_ptr )  /* Ptr to parm block.    */
{
    TREE	obj_ptr;		/* Pointer to object.		    */
    short	radiusx ;       	/* x-axis circle radius 	    */
    short	radiusy ;       	/* y-axis circle radius 	    */
    short	index[2] ;		/* color index for vrt_cpyfm	    */
    FDB		smfdb, imfdb1, imfdb2 ;	/* source & dest. mfdb.s for cpyfm  */
    POINT	xy[4] ;
    RECT	r ;			/* will contain object's pos & size */

#define	WM_TRANSPARENT	2

    obj_ptr = pb_ptr->pb_tree + pb_ptr->pb_obj;
    
    smfdb.fd_addr = 0L ;
    
    imfdb1.fd_addr = (long)NormalRbutton;
    imfdb1.fd_w = XWIDTH;
    imfdb1.fd_h = XWIDTH;
    imfdb1.fd_wdwidth = XWORDS;
    imfdb1.fd_stand = DVCFORM;
    imfdb1.fd_nplanes = 1;

    imfdb2.fd_addr = (long)SelctdRbutton;
    imfdb2.fd_w = XWIDTH;
    imfdb2.fd_h = XWIDTH;
    imfdb2.fd_wdwidth = XWORDS;
    imfdb2.fd_stand = DVCFORM;
    imfdb2.fd_nplanes = 1;

    index[0] = 1;		/* black */
    index[1] = 0;		/* white */
    
    radiusx = bigMulDiv( ((wBox+1)/2), 15, 20 ) ;    
    radiusy = bigMulDiv( ((hBox+1)/2), 15, 20 ) ;
    radiusx = (radiusx < MAXRADIUS) ? radiusx : MAXRADIUS ;
    radiusy = (radiusy < MAXRADIUS) ? radiusy : MAXRADIUS ;

    xy[0].x = 0 ;
    xy[0].y = 0 ;
    xy[1].x = 1 + radiusx * 2 ;
    xy[1].y = 1 + radiusy * 2 ;
    xy[2].x = pb_ptr->pb_r.x ;                     
    xy[2].y = pb_ptr->pb_r.y ;                     
    xy[3].x = pb_ptr->pb_r.x + radiusx * 2 ;
    xy[3].y = pb_ptr->pb_r.y + radiusy * 2 ;
    
    if( (pb_ptr->pb_prevstate ^ pb_ptr->pb_currstate) == 0 ) {
	
	/* scott clears selected disabled  radio button */
	fmemcpy( (void far *)&r, (void far *)&pb_ptr->pb_r, sizeof( RECT ) ) ;
	gsx_1code(S_FILL_COLOR, WHITE);
	bb_fill( MD_REPLACE, FIS_SOLID, 0, r.x, r.y, wBox, hBox ) ;
	
	if( (obj_ptr->ob_state & SELECTED) && 
	    !(obj_ptr->ob_state & DISABLED) ) {
	    vrt_cpyfm( WM_TRANSPARENT, (WORD *)xy, (WORD *) &imfdb1, 
			(WORD *) &smfdb, index[0], index[1] );
	    vrt_cpyfm( WM_TRANSPARENT, (WORD *)xy, (WORD *) &imfdb2, 
			(WORD *) &smfdb, index[0], index[1] );
	}
	else {
	    vrt_cpyfm( WM_TRANSPARENT, (WORD *)xy, (WORD *) &imfdb1,
			(WORD *) &smfdb, index[0], index[1] );
	}
    }
    else {
	if( (obj_ptr->ob_state & SELECTED) && 
	    !(obj_ptr->ob_state & DISABLED) ) {
	    vrt_cpyfm( WM_TRANSPARENT, (WORD *)xy, (WORD *) &imfdb1, 
			(WORD *) &smfdb, index[0], index[1]);
	    vrt_cpyfm( WM_TRANSPARENT, (WORD *)xy, (WORD *) &imfdb2, 
			(WORD *) &smfdb, index[0], index[1]);
	}
	else {
	    /* scott clears selected disabled  radio button */
	    fmemcpy( (void far *)&r, (void far *)&pb_ptr->pb_r, sizeof( RECT ) ) ;
	    gsx_1code( S_FILL_COLOR, WHITE);
	    bb_fill( MD_REPLACE, FIS_SOLID, 0, r.x, r.y, wBox, hBox ) ;
	    
	    vrt_cpyfm( WM_TRANSPARENT, (WORD *)xy, (WORD *) &imfdb1, 
			(WORD *) &smfdb, index[0], index[1]);
	    index[0] = 0;
	    vrt_cpyfm( WM_TRANSPARENT, (WORD *)xy, (WORD *) &imfdb2, 
			(WORD *) &smfdb, index[0], index[1]);
	}
    } 
} /* CUA_RadioButton() */

#endif

/* end of radio.c */
