#include "simcom.h"
#include "simdev.h"
#include "simusr.h"
#include "gsig800.h"

/*
    ===================================================

    Please do not change the internal flash start addresses
    and sizes when applicable (currently only for 56812).
    Any changes to DSP_*I_SIZE_* and DSP_*I_START_* will
    not have any effect on programming the internal flash
    memory.

    ===================================================
*/

/* 800 version memory block sizes */
#define DSP_PR_SIZE_800 64L	/* size of on-chip bootstrap rom memory */
#define DSP_PI_SIZE_800 0x400L	/* size of on-chip program ram memory */
#define DSP_XI_SIZE_800 0x800L	/* size of on-chip x data ram */
#define DSP_XP_SIZE_800 128	/* size of on-chip x peripheral memory */

/* 800 version memory block sizes */
#define DSP_PI_SIZE_812 0x5f80L	/* size of on-chip program ram memory */
#define DSP_PE_START_812 0x6000L/* start of off-chip program ram memory */
#define DSP_XI_SIZE_812 0x800L	/* size of on-chip x data ram */
#define DSP_XR_SIZE_812 0x800L	/* size of on-chip x data ram */
#define DSP_XP_SIZE_812 128	/* size of on-chip x peripheral memory */

/* 800 version memory block sizes */
#define DSP_PI_SIZE_824 128L	/* size of on-chip program ram memory */
#define DSP_PR_SIZE_824 0x8000L	/* size of on-chip program ROM memory */
#define DSP_PE_START_824 0x8000L/* start of off-chip program ram memory */
#define DSP_XI_SIZE_824 0x800L	/* size of on-chip x data ram */
#define DSP_XI2_SIZE_824 0x600L	/* size of upper block on-chip x data ram */
#define DSP_XR_SIZE_824 0x800L	/* size of on-chip x data ram */
#define DSP_XP_SIZE_824 128	/* size of on-chip x peripheral memory */


static
unsigned long bootrom811[DSP_PR_SIZE_800] =
{
0x87D1L, 0xC000L, 0xC800L, 0xE121L, 0xDE11L, 0x8BD8L, 0x010EL, 0x8BC1L ,
0x8086L, 0x9854L, 0xA7D2L, 0x2010L, 0xA3EDL, 0x3200L, 0xA906L, 0xA3E2L ,
0x0004L, 0xA3E2L, 0x0040L, 0xA3EDL, 0x000FL, 0x87D2L, 0x03FFL, 0x6C30L ,
0x8BD8L, 0x0111L, 0x8BC1L, 0x8088L, 0xCC82L, 0x0030L, 0xABD1L, 0x807EL ,
0xB550L, 0xCE88L, 0x7CB0L, 0xA90AL, 0xCC82L, 0x0030L, 0xABE1L, 0x807EL ,
0xB561L, 0xB560L, 0xA903L, 0xCC82L, 0x0030L, 0xE521L, 0x8607L, 0xCE88L ,
0x7C30L, 0xE040L, 0x8408L, 0xE400L, 0x8E32L, 0xA662L, 0x81DDL, 0x00FFL ,
0x81D8L, 0x0001L, 0x83D8L, 0x0002L, 0xE040L, 0xA942L, 0x0000L, 0x0000L
};

extern struct dt_periph 
 dx_portb_568,
 dx_portc_568,
 dx_cop_568,
 dx_spi_568,
 dx_ssi_568,
 dx_issi_568,
 dx_timer_568,
 dx_core_56800,
 dx_core_56824,
 dx_count_56800;

#if ADSx
extern struct dt_periph dx_once_56800;
extern struct dt_periph dx_once_56824;
extern struct dt_periph dx_jtag_56800;
#endif 

static
struct dt_mperiph dx_periph_56800[] =
{
     {{"core", DSP_PORT_M, 0XffffL}, 0l, 0l, 0l, 0l, 0l, 0l, &dx_core_56800, ""},
     {{"count", 0, 0L}, 0l, 0l, 0l, 0l, 0l, 0l, &dx_count_56800, ""}
};

static
struct dt_mperiph dx_periph_56811[] =
{
    {{"core", DSP_PORT_M, 0XffffL}, 0l, 0l, 0l, 0l, 0l, 0l, &dx_core_56800, ""},
    {{"ssi", DSP_PORT_C, 0X3F00L}, 0xffedl, 0x3f00l, 0xffd0l, 0x20l, 0l, 0x200l, &dx_ssi_568, ""},
    {{"timer", DSP_PORT_C, 0Xc000L}, 0xffedl, 0xc000l, 0xffd8l, 0x18l, 0l, 0x800l, &dx_timer_568, ""},
    {{"spi1", DSP_PORT_C, 0XF0L}, 0xffedl, 0xf0l, 0xffe4l, 0x28l, 0l, 0x1000l, &dx_spi_568, "1"},
    {{"spi0", DSP_PORT_C, 0XFL}, 0xffedl, 0xfl, 0xffe0l, 0x2al, 0l, 0x2000l, &dx_spi_568, "0"},
    {{"cop",  0, 0L}, 0l, 0l, 0xfff0l, 0x2l, 0x16l, 0x4000l, &dx_cop_568, ""},
    {{"portb", DSP_PORT_B, 0XffffL}, 0xffeal, 0xffffl, 0xffeal, 0x14l, 0l, 0x8000l, &dx_portb_568, ""},
    {{"portc", DSP_PORT_C, 0XffffL}, 0xffedl, 0l, 0xffedl, 0l, 0l, 0l, &dx_portc_568, ""},
#if ADSx
    {{"once", 0, 0L}, 0l, 0l, 0l, 0l, 0l, 0l, &dx_once_56800, "", 0, 0},
    {{"jtag", 0, 0L}, 0l, 0l, 0l, 0l, 0l, 0l, &dx_jtag_56800, "", 0, 0},
#endif    
    {{"count", 0, 0L}, 0l, 0l, 0l, 0l, 0l, 0l, &dx_count_56800, ""},
};

static
struct dt_mperiph dx_periph_56812[] =
{
    {{"core", DSP_PORT_M, 0XffffL}, 0l, 0l, 0l, 0l, 0l, 0l, &dx_core_56800, ""},
    {{"ssi", DSP_PORT_C, 0X3F00L}, 0xffedl, 0x3f00l, 0xffd0l, 0x20l, 0l, 0x200l, &dx_issi_568, ""},
    {{"timer", DSP_PORT_C, 0Xc000L}, 0xffedl, 0xc000l, 0xffd8l, 0x18l, 0l, 0x800l, &dx_timer_568, ""},
    {{"spi1", DSP_PORT_C, 0XF0L}, 0xffedl, 0xf0l, 0xffe4l, 0x28l, 0l, 0x1000l, &dx_spi_568, "1"},
    {{"spi0", DSP_PORT_C, 0XFL}, 0xffedl, 0xfl, 0xffe0l, 0x2al, 0l, 0x2000l, &dx_spi_568, "0"},
    {{"cop",  0, 0L}, 0l, 0l, 0xfff0l, 0x2l, 0x16l, 0x4000l, &dx_cop_568, ""},
    {{"portb", DSP_PORT_B, 0XffffL}, 0xffeal, 0xffffl, 0xffeal, 0x14l, 0l, 0x8000l, &dx_portb_568, ""},
    {{"portc", DSP_PORT_C, 0XffffL}, 0xffedl, 0l, 0xffedl, 0l, 0l, 0l, &dx_portc_568, ""},
#if ADSx
    {{"once", 0, 0L}, 0l, 0l, 0l, 0l, 0l, 0l, &dx_once_56824, "", 0, 0},
    {{"jtag", 0, 0L}, 0l, 0l, 0l, 0l, 0l, 0l, &dx_jtag_56800, "", 0, 0},
#endif    
    {{"count", 0, 0L}, 0l, 0l, 0l, 0l, 0l, 0l, &dx_count_56800, ""},
};

static
struct dt_mperiph dx_periph_56824[] =
{
    {{"core", DSP_PORT_M, 0XffffL}, 0l, 0l, 0l, 0l, 0l, 0l, &dx_core_56824, ""},
    {{"ssi", DSP_PORT_C, 0X3F00L}, 0xffedl, 0x3f00l, 0xffd0l, 0x20l, 0l, 0x200l, &dx_issi_568, ""},
    {{"timer", DSP_PORT_C, 0Xc000L}, 0xffedl, 0xc000l, 0xffd8l, 0x18l, 0l, 0x800l, &dx_timer_568, ""},
    {{"spi1", DSP_PORT_C, 0XF0L}, 0xffedl, 0xf0l, 0xffe4l, 0x28l, 0l, 0x1000l, &dx_spi_568, "1"},
    {{"spi0", DSP_PORT_C, 0XFL}, 0xffedl, 0xfl, 0xffe0l, 0x2al, 0l, 0x2000l, &dx_spi_568, "0"},
    {{"cop",  0, 0L}, 0l, 0l, 0xfff0l, 0x2l, 0x16l, 0x4000l, &dx_cop_568, ""},
    {{"portb", DSP_PORT_B, 0XffffL}, 0xffeal, 0xffffl, 0xffeal, 0x14l, 0l, 0x8000l, &dx_portb_568, ""},
    {{"portc", DSP_PORT_C, 0XffffL}, 0xffedl, 0l, 0xffedl, 0l, 0l, 0l, &dx_portc_568, ""},
#if ADSx
    {{"once", 0, 0L}, 0l, 0l, 0l, 0l, 0l, 0l, &dx_once_56824, "", 0, 0},
    {{"jtag", 0, 0L}, 0l, 0l, 0l, 0l, 0l, 0l, &dx_jtag_56800, "", 0, 0},
#endif    
    {{"count", 0, 0L}, 0l, 0l, 0l, 0l, 0l, 0l, &dx_count_56800, ""},
};

#if ADSx
static struct dt_xpin xpin[1];
#else

static
struct dt_xpin xpin[] =
{				/* includes pin name and possible second name */
    {"reset", NULL, NULL, DSP_PORT_M, DSPT_PIM_RESET_800, DSP56800|DSP56811|DSP56812|DSP56824},
    {"modb", "irqb", NULL, DSP_PORT_M, DSPT_PIM_MODB_800, DSP56800|DSP56811|DSP56812|DSP56824},
    {"moda", "irqa", NULL, DSP_PORT_M, DSPT_PIM_MODA_800, DSP56800|DSP56811|DSP56812|DSP56824},
    {"rd", NULL, NULL, DSP_PORT_M, DSPT_PIM_RD_800, DSP56800|DSP56811|DSP56812|DSP56824},
    {"wr", NULL, NULL, DSP_PORT_M, DSPT_PIM_WR_800, DSP56800|DSP56811|DSP56812|DSP56824},
    {"ps", NULL, NULL, DSP_PORT_M, DSPT_PIM_PS_800, DSP56800|DSP56811|DSP56812|DSP56824},
    {"ds", NULL, NULL, DSP_PORT_M, DSPT_PIM_DS_800, DSP56800|DSP56811|DSP56812|DSP56824},
#if 0
    {"bs", NULL, NULL, DSP_PORT_M, DSPT_PIM_BS_800, DSP56800|DSP56811|DSP56812|DSP56824},
    {"ts", NULL, NULL, DSP_PORT_M, DSPT_PIM_TS_800, DSP56800|DSP56811|DSP56812|DSP56824},
    {"ta", NULL, NULL, DSP_PORT_M, DSPT_PIM_TA_800, DSP56800|DSP56811|DSP56812|DSP56824},
    {"br", NULL, NULL, DSP_PORT_M, DSPT_PIM_BR_800, DSP56800|DSP56811|DSP56812|DSP56824},
    {"bg", NULL, NULL, DSP_PORT_M, DSPT_PIM_BG_800, DSP56800|DSP56811|DSP56812|DSP56824},
    {"bb", NULL, NULL, DSP_PORT_M, DSPT_PIM_BB_800, DSP56800|DSP56811|DSP56812|DSP56824},
#endif
    {"pa15", "a15", NULL, DSP_PORT_A, DSP_BIT_15, DSP56800|DSP56811|DSP56812|DSP56824},
    {"pa14", "a14", NULL, DSP_PORT_A, DSP_BIT_14, DSP56800|DSP56811|DSP56812|DSP56824},
    {"pa13", "a13", NULL, DSP_PORT_A, DSP_BIT_13, DSP56800|DSP56811|DSP56812|DSP56824},
    {"pa12", "a12", NULL, DSP_PORT_A, DSP_BIT_12, DSP56800|DSP56811|DSP56812|DSP56824},
    {"pa11", "a11", NULL, DSP_PORT_A, DSP_BIT_11, DSP56800|DSP56811|DSP56812|DSP56824},
    {"pa10", "a10", NULL, DSP_PORT_A, DSP_BIT_10, DSP56800|DSP56811|DSP56812|DSP56824},
    {"pa9", "a9", NULL, DSP_PORT_A, DSP_BIT_9, DSP56800|DSP56811|DSP56812|DSP56824},
    {"pa8", "a8", NULL, DSP_PORT_A, DSP_BIT_8, DSP56800|DSP56811|DSP56812|DSP56824},
    {"pa7", "a7", NULL, DSP_PORT_A, DSP_BIT_7, DSP56800|DSP56811|DSP56812|DSP56824},
    {"pa6", "a6", NULL, DSP_PORT_A, DSP_BIT_6, DSP56800|DSP56811|DSP56812|DSP56824},
    {"pa5", "a5", NULL, DSP_PORT_A, DSP_BIT_5, DSP56800|DSP56811|DSP56812|DSP56824},
    {"pa4", "a4", NULL, DSP_PORT_A, DSP_BIT_4, DSP56800|DSP56811|DSP56812|DSP56824},
    {"pa3", "a3", NULL, DSP_PORT_A, DSP_BIT_3, DSP56800|DSP56811|DSP56812|DSP56824},
    {"pa2", "a2", NULL, DSP_PORT_A, DSP_BIT_2, DSP56800|DSP56811|DSP56812|DSP56824},
    {"pa1", "a1", NULL, DSP_PORT_A, DSP_BIT_1, DSP56800|DSP56811|DSP56812|DSP56824},
    {"pa0", "a0", NULL, DSP_PORT_A, DSP_BIT_0, DSP56800|DSP56811|DSP56812|DSP56824},
    {"pd15", "d15", NULL, DSP_PORT_D, DSP_BIT_15, DSP56800|DSP56811|DSP56812|DSP56824},
    {"pd14", "d14", NULL, DSP_PORT_D, DSP_BIT_14, DSP56800|DSP56811|DSP56812|DSP56824},
    {"pd13", "d13", NULL, DSP_PORT_D, DSP_BIT_13, DSP56800|DSP56811|DSP56812|DSP56824},
    {"pd12", "d12", NULL, DSP_PORT_D, DSP_BIT_12, DSP56800|DSP56811|DSP56812|DSP56824},
    {"pd11", "d11", NULL, DSP_PORT_D, DSP_BIT_11, DSP56800|DSP56811|DSP56812|DSP56824},
    {"pd10", "d10", NULL, DSP_PORT_D, DSP_BIT_10, DSP56800|DSP56811|DSP56812|DSP56824},
    {"pd9", "d9", NULL, DSP_PORT_D, DSP_BIT_9, DSP56800|DSP56811|DSP56812|DSP56824},
    {"pd8", "d8", NULL, DSP_PORT_D, DSP_BIT_8, DSP56800|DSP56811|DSP56812|DSP56824},
    {"pd7", "d7", NULL, DSP_PORT_D, DSP_BIT_7, DSP56800|DSP56811|DSP56812|DSP56824},
    {"pd6", "d6", NULL, DSP_PORT_D, DSP_BIT_6, DSP56800|DSP56811|DSP56812|DSP56824},
    {"pd5", "d5", NULL, DSP_PORT_D, DSP_BIT_5, DSP56800|DSP56811|DSP56812|DSP56824},
    {"pd4", "d4", NULL, DSP_PORT_D, DSP_BIT_4, DSP56800|DSP56811|DSP56812|DSP56824},
    {"pd3", "d3", NULL, DSP_PORT_D, DSP_BIT_3, DSP56800|DSP56811|DSP56812|DSP56824},
    {"pd2", "d2", NULL, DSP_PORT_D, DSP_BIT_2, DSP56800|DSP56811|DSP56812|DSP56824},
    {"pd1", "d1", NULL, DSP_PORT_D, DSP_BIT_1, DSP56800|DSP56811|DSP56812|DSP56824},
    {"pd0", "d0", NULL, DSP_PORT_D, DSP_BIT_0, DSP56800|DSP56811|DSP56812|DSP56824},
    {"pb15", NULL, NULL, DSP_PORT_B, DSP_BIT_15, DSP56811|DSP56812|DSP56824},
    {"pb14", NULL, NULL, DSP_PORT_B, DSP_BIT_14, DSP56811|DSP56812|DSP56824},
    {"pb13", NULL, NULL, DSP_PORT_B, DSP_BIT_13, DSP56811|DSP56812|DSP56824},
    {"pb12", NULL, NULL, DSP_PORT_B, DSP_BIT_12, DSP56811|DSP56812|DSP56824},
    {"pb11", NULL, NULL, DSP_PORT_B, DSP_BIT_11, DSP56811|DSP56812|DSP56824},
    {"pb10", NULL, NULL, DSP_PORT_B, DSP_BIT_10, DSP56811|DSP56812|DSP56824},
    {"pb9", NULL, NULL, DSP_PORT_B, DSP_BIT_9, DSP56811|DSP56812|DSP56824},
    {"pb8", NULL, NULL, DSP_PORT_B, DSP_BIT_8, DSP56811|DSP56812|DSP56824},
    {"pb7", NULL, NULL, DSP_PORT_B, DSP_BIT_7, DSP56811|DSP56812|DSP56824},
    {"pb6", NULL, NULL, DSP_PORT_B, DSP_BIT_6, DSP56811|DSP56812|DSP56824},
    {"pb5", NULL, NULL, DSP_PORT_B, DSP_BIT_5, DSP56811|DSP56812|DSP56824},
    {"pb4", NULL, NULL, DSP_PORT_B, DSP_BIT_4, DSP56811|DSP56812|DSP56824},
    {"pb3", NULL, NULL, DSP_PORT_B, DSP_BIT_3, DSP56811|DSP56812|DSP56824},
    {"pb2", NULL, NULL, DSP_PORT_B, DSP_BIT_2, DSP56811|DSP56812|DSP56824},
    {"pb1", NULL, NULL, DSP_PORT_B, DSP_BIT_1, DSP56811|DSP56812|DSP56824},
    {"pb0", NULL, NULL, DSP_PORT_B, DSP_BIT_0, DSP56811|DSP56812|DSP56824},
    {"pc15","tio2", NULL, DSP_PORT_C, DSP_BIT_15, DSP56811|DSP56812|DSP56824},
    {"pc14","tio1", NULL, DSP_PORT_C, DSP_BIT_14, DSP56811|DSP56812|DSP56824},
    {"pc13","srfs", NULL, DSP_PORT_C, DSP_BIT_13, DSP56811|DSP56812|DSP56824},
    {"pc12","srck", NULL, DSP_PORT_C, DSP_BIT_12, DSP56811|DSP56812|DSP56824},
    {"pc11","stfs", NULL, DSP_PORT_C, DSP_BIT_11, DSP56811|DSP56812|DSP56824},
    {"pc10","stck", NULL, DSP_PORT_C, DSP_BIT_10, DSP56811|DSP56812|DSP56824},
    {"pc9", "srd", NULL, DSP_PORT_C, DSP_BIT_9, DSP56811|DSP56812|DSP56824},
    {"pc8", "std", NULL, DSP_PORT_C, DSP_BIT_8, DSP56811|DSP56812|DSP56824},
    {"pc7", "ss1", NULL, DSP_PORT_C, DSP_BIT_7, DSP56811|DSP56812|DSP56824},
    {"pc6", "sck1", NULL, DSP_PORT_C, DSP_BIT_6, DSP56811|DSP56812|DSP56824},
    {"pc5", "mosi1", NULL, DSP_PORT_C, DSP_BIT_5, DSP56811|DSP56812|DSP56824},
    {"pc4", "miso1", NULL, DSP_PORT_C, DSP_BIT_4, DSP56811|DSP56812|DSP56824},
    {"pc3", "ss0", NULL, DSP_PORT_C, DSP_BIT_3, DSP56811|DSP56812|DSP56824},
    {"pc2", "sck0", NULL, DSP_PORT_C, DSP_BIT_2, DSP56811|DSP56812|DSP56824},
    {"pc1", "mosi0", NULL, DSP_PORT_C, DSP_BIT_1, DSP56811|DSP56812|DSP56824},
    {"pc0", "miso0", NULL, DSP_PORT_C, DSP_BIT_0, DSP56811|DSP56812|DSP56824},
#if BLM
    {"tck", NULL, NULL, DSP_PORT_M, DSPT_PIM_TCK_800, DSP56800|DSP56811|DSP56812|DSP56824},
    {"tms", NULL, NULL, DSP_PORT_M, DSPT_PIM_TMS_800, DSP56800|DSP56811|DSP56812|DSP56824},
    {"tdi", NULL, NULL, DSP_PORT_M, DSPT_PIM_TDI_800, DSP56800|DSP56811|DSP56812|DSP56824},
    {"tdo", NULL, NULL, DSP_PORT_M, DSPT_PIM_TDO_800, DSP56800|DSP56811|DSP56812|DSP56824},
    {"trst", NULL, NULL, DSP_PORT_M, DSPT_PIM_TRST_800, DSP56800|DSP56811|DSP56812|DSP56824},
#endif
    {"clko", NULL, NULL, DSP_PORT_M, DSPT_PIM_CLKO_800, DSP56800|DSP56811|DSP56812|DSP56824},
    {"extal", NULL, NULL, DSP_PORT_M, DSPT_PIM_EXTAL_800, DSP56800|DSP56811|DSP56812|DSP56824},
#if BLM
    {"t0",NULL,NULL,DSP_PER_CLOCK,DSP_TPH0,DSP56800},
    {"t1",NULL,NULL,DSP_PER_CLOCK,DSP_TPH1,DSP56800},
    {"t2",NULL,NULL,DSP_PER_CLOCK,DSP_TPH2,DSP56800},
    {"t3",NULL,NULL,DSP_PER_CLOCK,DSP_TPH3,DSP56800},
    {"phi0",NULL,NULL,DSP_PER_CLOCK,DSP_PH0,DSP56800},
    {"phi1",NULL,NULL,DSP_PER_CLOCK,DSP_PH1,DSP56800},
    {"pxab16",NULL,NULL,DSP_PER_XAB,DSP_BIT_6,DSP56800},
    {"pxab15",NULL,NULL,DSP_PER_XAB,DSP_BIT_5,DSP56800},
    {"pxab14",NULL,NULL,DSP_PER_XAB,DSP_BIT_4,DSP56800},
    {"pxab13",NULL,NULL,DSP_PER_XAB,DSP_BIT_3,DSP56800},
    {"pxab12",NULL,NULL,DSP_PER_XAB,DSP_BIT_2,DSP56800},
    {"pxab11",NULL,NULL,DSP_PER_XAB,DSP_BIT_1,DSP56800},
    {"pxab10",NULL,NULL,DSP_PER_XAB,DSP_BIT_0,DSP56800},
    {"pgdb15",NULL,NULL,DSP_PER_GDB,DSP_BIT_15,DSP56800},
    {"pgdb14",NULL,NULL,DSP_PER_GDB,DSP_BIT_14,DSP56800},
    {"pgdb13",NULL,NULL,DSP_PER_GDB,DSP_BIT_13,DSP56800},
    {"pgdb12",NULL,NULL,DSP_PER_GDB,DSP_BIT_12,DSP56800},
    {"pgdb11",NULL,NULL,DSP_PER_GDB,DSP_BIT_11,DSP56800},
    {"pgdb10",NULL,NULL,DSP_PER_GDB,DSP_BIT_10,DSP56800},
    {"pgdb9",NULL,NULL,DSP_PER_GDB,DSP_BIT_9,DSP56800},
    {"pgdb8",NULL,NULL,DSP_PER_GDB,DSP_BIT_8,DSP56800},
    {"pgdb7",NULL,NULL,DSP_PER_GDB,DSP_BIT_7,DSP56800},
    {"pgdb6",NULL,NULL,DSP_PER_GDB,DSP_BIT_6,DSP56800},
    {"pgdb5",NULL,NULL,DSP_PER_GDB,DSP_BIT_5,DSP56800},
    {"pgdb4",NULL,NULL,DSP_PER_GDB,DSP_BIT_4,DSP56800},
    {"pgdb3",NULL,NULL,DSP_PER_GDB,DSP_BIT_3,DSP56800},
    {"pgdb2",NULL,NULL,DSP_PER_GDB,DSP_BIT_2,DSP56800},
    {"pgdb1",NULL,NULL,DSP_PER_GDB,DSP_BIT_1,DSP56800},
    {"pgdb0",NULL,NULL,DSP_PER_GDB,DSP_BIT_0,DSP56800},
    {"biordw",NULL,NULL,DSP_PER_CTL,DSPT_PIM_BIORDW_800,DSP56800},
    {"res",NULL,NULL,DSP_PER_CTL,DSPT_PIM_RES_800,DSP56800},
    {"ckinstop",NULL,NULL,DSP_PER_CTL,DSPT_PIM_CKINSTOP_800,DSP56800},
    {"piack",NULL,NULL,DSP_PER_CTL,DSPT_PIM_PIACK_800,DSP56800},
    {"pmrvab",NULL,NULL,DSP_PER_CTL,DSPT_PIM_PMRVAB_800,DSP56800},
    {"cxrqz6",NULL,NULL,DSP_PER_CXRQZ,DSP_BIT_9,DSP56800},
    {"cxrqz5",NULL,NULL,DSP_PER_CXRQZ,DSP_BIT_10,DSP56800},
    {"cxrqz4",NULL,NULL,DSP_PER_CXRQZ,DSP_BIT_11,DSP56800},
    {"cxrqz3",NULL,NULL,DSP_PER_CXRQZ,DSP_BIT_12,DSP56800},
    {"cxrqz2",NULL,NULL,DSP_PER_CXRQZ,DSP_BIT_13,DSP56800},
    {"cxrqz1",NULL,NULL,DSP_PER_CXRQZ,DSP_BIT_14,DSP56800},
    {"cxrqz0",NULL,NULL,DSP_PER_CXRQZ,DSP_BIT_15,DSP56800},
    {"picvse6",NULL,NULL,DSP_PER_PICVSE,DSP_BIT_9,DSP56800},
    {"picvse5",NULL,NULL,DSP_PER_PICVSE,DSP_BIT_10,DSP56800},
    {"picvse4",NULL,NULL,DSP_PER_PICVSE,DSP_BIT_11,DSP56800},
    {"picvse3",NULL,NULL,DSP_PER_PICVSE,DSP_BIT_12,DSP56800},
    {"picvse2",NULL,NULL,DSP_PER_PICVSE,DSP_BIT_13,DSP56800},
    {"picvse1",NULL,NULL,DSP_PER_PICVSE,DSP_BIT_14,DSP56800},
    {"picvse0",NULL,NULL,DSP_PER_PICVSE,DSP_BIT_15,DSP56800},
    {"vab5",NULL,NULL,DSP_PER_VAB,DSP_BIT_6,DSP56800},
    {"vab4",NULL,NULL,DSP_PER_VAB,DSP_BIT_5,DSP56800},
    {"vab3",NULL,NULL,DSP_PER_VAB,DSP_BIT_4,DSP56800},
    {"vab2",NULL,NULL,DSP_PER_VAB,DSP_BIT_3,DSP56800},
    {"vab1",NULL,NULL,DSP_PER_VAB,DSP_BIT_2,DSP56800},
    {"vab0",NULL,NULL,DSP_PER_VAB,DSP_BIT_1,DSP56800},
    {"cxwkup0",NULL,NULL,DSP_PORT_M,DSPT_PIM_CXWKUP0_800,DSP56800|DSP56811|DSP56812|DSP56824},
    {"cxwkup1",NULL,NULL,DSP_PORT_M,DSPT_PIM_CXWKUP1_800,DSP56800|DSP56811|DSP56812|DSP56824},
    {"xtal", NULL, NULL, DSP_PORT_M, DSPT_PIM_XTAL_800, DSP56800|DSP56811|DSP56812|DSP56824},
#endif
    {"cpresz",NULL,NULL,DSP_PORT_M,DSPT_PIM_CPRESZ_800,DSP56800|DSP56811|DSP56812|DSP56824},
    {"zpreclk",NULL,NULL,DSP_PORT_M,DSPT_PIM_ZPRECLK_800,DSP56800|DSP56811|DSP56812|DSP56824},
};

#endif

static
struct dt_memory mem_56800[] =
{
    {"p", memory_map_p, DSP_MEM_PG, 0l, 0x7ffffl, 0, DSP_MEMA_EXEC | DSP_MEMA_BRK, NULL, 0x7ffffl, DSP_MEMW_SIZE16 | DSP_MEMW_WRD1, 1 },
    {"pe", 
     memory_map_pe, 
     DSP_MEM_PE, 
     0l, 
     0x7ffffl, 
     0, 
     DSP_MEMA_EXEC | DSP_MEMA_EXT | DSP_MEMA_DON | DSP_MEMA_REAL | DSP_MEMA_DISP | DSP_MEMA_BRK, 
     NULL, 
     0x7ffffl, 
     DSP_MEMW_SIZE16 | DSP_MEMW_WRD1, 1 },
    {"pr",
     memory_map_pr, 
     DSP_MEM_PR, 
     0l, DSP_PR_SIZE_800 - 1L, 
     DSP_PR_SIZE_800, 
     DSP_MEMA_ROM | DSP_MEMA_WR_PROT | DSP_MEMA_EXEC | DSP_MEMA_DON | DSP_MEMA_REAL | DSP_MEMA_DISP | DSP_MEMA_BRK,
     NULL, 
     0xffffl, 
     DSP_MEMW_SIZE16 | DSP_MEMW_WRD1, 1 
 },
    {"pi", 
     memory_map_pi, 
     DSP_MEM_PI, 
     0l, 
     DSP_PI_SIZE_800 - 1l, 
     DSP_PI_SIZE_800, 
     DSP_MEMA_MULT | DSP_MEMA_EXEC | DSP_MEMA_DON | DSP_MEMA_REAL | DSP_MEMA_DISP | DSP_MEMA_BRK, 
     NULL, 
     0xffffl, 
     DSP_MEMW_SIZE16 | DSP_MEMW_WRD1, 1 },
    {"pb", /* this is switched from X ram */
     memory_map_pb, 
     DSP_MEM_XI, 
     0l, DSP_XI_SIZE_800 - 1L, 
     0L, 
     DSP_MEMA_EXEC | DSP_MEMA_DON | DSP_MEMA_DISP | DSP_MEMA_BRK,
     NULL, 
     0xffffl, 
     DSP_MEMW_SIZE16 | DSP_MEMW_WRD1, 1  },
    {"pi", /* hole */
     memory_map_pi, 
     DSP_MEM_PI, 
     DSP_PI_SIZE_800, 
     DSP_XI_SIZE_800 - 1l, 
     0L, 
     DSP_MEMA_MULT | DSP_MEMA_NONE | DSP_MEMA_EXEC | DSP_MEMA_DON | DSP_MEMA_DISP | DSP_MEMA_BRK, 
     NULL, 
     0xffffl, 
     DSP_MEMW_SIZE16 | DSP_MEMW_WRD1, 1 },
     {"x", 
      memory_map_x, 
      DSP_MEM_XG, 
      0l, 
      0xffffl, 
      0, 
      DSP_MEMA_BRK, 
      NULL, 
      0xffffl, 
      DSP_MEMW_SIZE16 | DSP_MEMW_WRD1, 1 },
    {"xe", 
     memory_map_xe, 
     DSP_MEM_XE, 
     0l, 0xffffl, 0, 
     DSP_MEMA_EXT | DSP_MEMA_DON | DSP_MEMA_REAL | DSP_MEMA_DISP | DSP_MEMA_BRK, 
     NULL, 
     0xffffl, DSP_MEMW_SIZE16 | DSP_MEMW_WRD1, 1 },
    {"xi", 
     memory_map_xi, 
     DSP_MEM_XI, 
     0l, DSP_XI_SIZE_800 - 1l, DSP_XI_SIZE_800, 
     DSP_MEMA_MULT | DSP_MEMA_DON | DSP_MEMA_REAL | DSP_MEMA_DISP | DSP_MEMA_BRK, 
     NULL, 
     0xffffl, DSP_MEMW_SIZE16 | DSP_MEMW_WRD1, 1 },
    {"xi", /* this also doubles as p ram when omr xp bit is set */
     memory_map_xi, 
     DSP_MEM_XI, 
     0x10000l - DSP_XP_SIZE_800, 0xffffl, 
     0l, 
     DSP_MEMA_MULT | DSP_MEMA_DON | DSP_MEMA_PERIPH | DSP_MEMA_REAL | DSP_MEMA_DISP | DSP_MEMA_BRK, 
     NULL, 
     0xffffl, 
     DSP_MEMW_SIZE16 | DSP_MEMW_WRD1, 1 }

};

static
struct dt_memory mem_56811[] =
{
    {"p", memory_map_p, DSP_MEM_PG, 0l, 0xffffl, 0, DSP_MEMA_EXEC | DSP_MEMA_BRK, NULL, 0xffffl, DSP_MEMW_SIZE16 | DSP_MEMW_WRD1, 1 },
    {"pe", 
     memory_map_pe, 
     DSP_MEM_PE, 
     0l, 
     0xffffl, 
     0, 
     DSP_MEMA_EXEC | DSP_MEMA_EXT | DSP_MEMA_DON | DSP_MEMA_REAL | DSP_MEMA_DISP | DSP_MEMA_BRK, 
     NULL, 
     0xffffl, 
     DSP_MEMW_SIZE16 | DSP_MEMW_WRD1, 1 },
    {"pr",
     memory_map_pr, 
     DSP_MEM_PR, 
     0l, DSP_PR_SIZE_800 - 1L, 
     DSP_PR_SIZE_800, 
     DSP_MEMA_ROM | DSP_MEMA_WR_PROT | DSP_MEMA_EXEC | DSP_MEMA_DON | DSP_MEMA_REAL | DSP_MEMA_DISP | DSP_MEMA_BRK,
     bootrom811, 
     0xffffl, 
     DSP_MEMW_SIZE16 | DSP_MEMW_WRD1, 1 
 },
    {"pi", 
     memory_map_pi, 
     DSP_MEM_PI, 
     0l, 
     DSP_PI_SIZE_800 - 1l, 
     DSP_PI_SIZE_800, 
     DSP_MEMA_MULT | DSP_MEMA_EXEC | DSP_MEMA_DON | DSP_MEMA_REAL | DSP_MEMA_DISP | DSP_MEMA_BRK, 
     NULL, 
     0xffffl, 
     DSP_MEMW_SIZE16 | DSP_MEMW_WRD1, 1 },
    {"pb", /* this is switched from X ram */
     memory_map_pb, 
     DSP_MEM_XI, 
     0l, DSP_XI_SIZE_800 - 1L, 
     0L, 
     DSP_MEMA_EXEC | DSP_MEMA_DON | DSP_MEMA_DISP | DSP_MEMA_BRK,
     NULL, 
     0xffffl, 
     DSP_MEMW_SIZE16 | DSP_MEMW_WRD1, 1  },
    {"pi", /* hole */
     memory_map_pi, 
     DSP_MEM_PI, 
     DSP_PI_SIZE_800, 
     DSP_XI_SIZE_800 - 1l, 
     0L, 
     DSP_MEMA_MULT | DSP_MEMA_NONE | DSP_MEMA_EXEC | DSP_MEMA_DON | DSP_MEMA_DISP | DSP_MEMA_BRK, 
     NULL, 
     0xffffl, 
     DSP_MEMW_SIZE16 | DSP_MEMW_WRD1, 1 },
     {"x", 
      memory_map_x, 
      DSP_MEM_XG, 
      0l, 
      0xffffl, 
      0, 
      DSP_MEMA_BRK, 
      NULL, 
      0xffffl, 
      DSP_MEMW_SIZE16 | DSP_MEMW_WRD1, 1 },
    {"xe", 
     memory_map_xe, 
     DSP_MEM_XE, 
     0l, 0xffffl, 0, 
     DSP_MEMA_EXT | DSP_MEMA_DON | DSP_MEMA_REAL | DSP_MEMA_DISP | DSP_MEMA_BRK, 
     NULL, 
     0xffffl, DSP_MEMW_SIZE16 | DSP_MEMW_WRD1, 1 },
    {"xi", 
     memory_map_xi, 
     DSP_MEM_XI, 
     0l, DSP_XI_SIZE_800 - 1l, DSP_XI_SIZE_800, 
     DSP_MEMA_MULT | DSP_MEMA_DON | DSP_MEMA_REAL | DSP_MEMA_DISP | DSP_MEMA_BRK, 
     NULL, 
     0xffffl, DSP_MEMW_SIZE16 | DSP_MEMW_WRD1, 1 },
    {"xi", /* this also doubles as p ram when omr xp bit is set */
     memory_map_xi, 
     DSP_MEM_XI, 
     0x10000l - DSP_XP_SIZE_800, 0xffffl, 
     0l, 
     DSP_MEMA_MULT | DSP_MEMA_DON | DSP_MEMA_PERIPH | DSP_MEMA_REAL | DSP_MEMA_DISP | DSP_MEMA_BRK, 
     NULL, 
     0xffffl, 
     DSP_MEMW_SIZE16 | DSP_MEMW_WRD1, 1 }

};

static
struct dt_memory mem_56812[] =
{
    {"p", memory_map_p, DSP_MEM_PG, 0l, 0xffffl, 0, DSP_MEMA_EXEC | DSP_MEMA_BRK, NULL, 0xffffl, DSP_MEMW_SIZE16 | DSP_MEMW_WRD1, 1 },
    {"pe", 
     memory_map_pe, 
     DSP_MEM_PE, 
     0l, 
     0xffffl, 
     0, 
     DSP_MEMA_EXEC | DSP_MEMA_EXT | DSP_MEMA_DON | DSP_MEMA_REAL | DSP_MEMA_DISP | DSP_MEMA_BRK, 
     NULL, 
     0xffffl, 
     DSP_MEMW_SIZE16 | DSP_MEMW_WRD1, 1 },
    {"pi", 
     memory_map_pi, 
     DSP_MEM_PI, 
     0l, 
     DSP_PI_SIZE_812 - 1l, 
     DSP_PI_SIZE_812, 
     DSP_MEMA_MULT | DSP_MEMA_WR_PROT | DSP_MEMA_EXEC | DSP_MEMA_DON | DSP_MEMA_REAL | DSP_MEMA_DISP | DSP_MEMA_BRK, 
     NULL, 
     0xffffl, 
     DSP_MEMW_SIZE16 | DSP_MEMW_WRD1, 1 },
    {"pi", /* hole */
     memory_map_pi, 
     DSP_MEM_PI, 
     DSP_PI_SIZE_812, 
     DSP_PE_START_812 - 1l, 
     DSP_PE_START_812 - DSP_PI_SIZE_812, 
     DSP_MEMA_MULT | DSP_MEMA_NONE | DSP_MEMA_EXEC | DSP_MEMA_DON | DSP_MEMA_DISP | DSP_MEMA_BRK, 
     NULL, 
     0xffffl, 
     DSP_MEMW_SIZE16 | DSP_MEMW_WRD1, 1 },
     {"x", 
      memory_map_x, 
      DSP_MEM_XG, 
      0l, 
      0xffffl, 
      0, 
      DSP_MEMA_BRK, 
      NULL, 
      0xffffl, 
      DSP_MEMW_SIZE16 | DSP_MEMW_WRD1, 1 },
    {"xe", 
     memory_map_xe, 
     DSP_MEM_XE, 
     0l, 0xffffl, 0, 
     DSP_MEMA_EXT | DSP_MEMA_DON | DSP_MEMA_REAL | DSP_MEMA_DISP | DSP_MEMA_BRK, 
     NULL, 
     0xffffl, DSP_MEMW_SIZE16 | DSP_MEMW_WRD1, 1 },
    {"xi", 
     memory_map_xi, 
     DSP_MEM_XI, 
     0l, DSP_XI_SIZE_812 - 1l, DSP_XI_SIZE_812, 
     DSP_MEMA_MULT |DSP_MEMA_DON | DSP_MEMA_REAL | DSP_MEMA_DISP | DSP_MEMA_BRK, 
     NULL, 
     0xffffl, DSP_MEMW_SIZE16 | DSP_MEMW_WRD1, 1 },
    {"xr",
     memory_map_xr, 
     DSP_MEM_XR, 
     DSP_XI_SIZE_812,
     DSP_XI_SIZE_812+DSP_XR_SIZE_812-1L, DSP_XR_SIZE_812,
     DSP_MEMA_WR_PROT | DSP_MEMA_DON | DSP_MEMA_REAL | DSP_MEMA_DISP | DSP_MEMA_BRK, 
     NULL, 
     0xffffl, 
     DSP_MEMW_SIZE16 | DSP_MEMW_WRD1, 1 },
    {"xi",
     memory_map_xi, 
     DSP_MEM_XI, 
     0x10000l - DSP_XP_SIZE_812, 0xffffl, 
     0l, 
     DSP_MEMA_MULT | DSP_MEMA_DON | DSP_MEMA_PERIPH | DSP_MEMA_REAL | DSP_MEMA_DISP | DSP_MEMA_BRK, 
     NULL, 
     0xffffl, 
     DSP_MEMW_SIZE16 | DSP_MEMW_WRD1, 1 }

};

static
struct dt_memory mem_56824[] =
{
    {"p", memory_map_p, DSP_MEM_PG, 0l, 0xffffl, 0, DSP_MEMA_EXEC | DSP_MEMA_BRK, NULL, 0xffffl, DSP_MEMW_SIZE16 | DSP_MEMW_WRD1, 1 },
    {"pe", 
     memory_map_pe, 
     DSP_MEM_PE, 
     0l, 
     0xffffl, 
     0, 
     DSP_MEMA_EXEC | DSP_MEMA_EXT | DSP_MEMA_DON | DSP_MEMA_REAL | DSP_MEMA_DISP | DSP_MEMA_BRK, 
     NULL, 
     0xffffl, 
     DSP_MEMW_SIZE16 | DSP_MEMW_WRD1, 1 },
    {"pi", 
     memory_map_pi, 
     DSP_MEM_PI, 
     0l, 
     DSP_PI_SIZE_824 - 1l, 
     DSP_PI_SIZE_824, 
     DSP_MEMA_EXEC | DSP_MEMA_DON | DSP_MEMA_REAL | DSP_MEMA_DISP | DSP_MEMA_BRK, 
     NULL, 
     0xffffl, 
     DSP_MEMW_SIZE16 | DSP_MEMW_WRD1, 1 },
    {"pr", 
     memory_map_pr, 
     DSP_MEM_PR, 
     0l, 
     DSP_PR_SIZE_824 - 1l, 
     DSP_PR_SIZE_824, 
     DSP_MEMA_WR_PROT | DSP_MEMA_EXEC | DSP_MEMA_DON | DSP_MEMA_REAL | DSP_MEMA_DISP | DSP_MEMA_BRK, 
     NULL, 
     0xffffl, 
     DSP_MEMW_SIZE16 | DSP_MEMW_WRD1, 1 },
    {"pb", /* this is switched from X ram */
     memory_map_pb, 
     DSP_MEM_XI, 
     0l, DSP_XI_SIZE_824 - 1L, 
     0L, 
     DSP_MEMA_MULT |DSP_MEMA_EXEC | DSP_MEMA_DON | DSP_MEMA_DISP | DSP_MEMA_BRK,
     NULL, 
     0xffffl, 
     DSP_MEMW_SIZE16 | DSP_MEMW_WRD1, 1  },
    {"pb", /* this is switched from X ram */
     memory_map_pb, 
     DSP_MEM_XI, 
     0x1000L, 0x1000L+DSP_XI2_SIZE_824 - 1L, 
     0L, 
     DSP_MEMA_EXEC | DSP_MEMA_DON | DSP_MEMA_DISP | DSP_MEMA_BRK,
     NULL, 
     0xffffl, 
     DSP_MEMW_SIZE16 | DSP_MEMW_WRD1, 1  },
     {"x", 
      memory_map_x, 
      DSP_MEM_XG, 
      0l, 
      0xffffl, 
      0, 
      DSP_MEMA_BRK, 
      NULL, 
      0xffffl, 
      DSP_MEMW_SIZE16 | DSP_MEMW_WRD1, 1 },
    {"xe", 
     memory_map_xe, 
     DSP_MEM_XE, 
     0l, 0xffffl, 0, 
     DSP_MEMA_EXT | DSP_MEMA_DON | DSP_MEMA_REAL | DSP_MEMA_DISP | DSP_MEMA_BRK, 
     NULL, 
     0xffffl, DSP_MEMW_SIZE16 | DSP_MEMW_WRD1, 1 },
    {"xi", 
     memory_map_xi, 
     DSP_MEM_XI, 
     0l, DSP_XI_SIZE_824 - 1l, DSP_XI_SIZE_824, 
     DSP_MEMA_MULT |DSP_MEMA_DON | DSP_MEMA_REAL | DSP_MEMA_DISP | DSP_MEMA_BRK, 
     NULL, 
     0xffffl, DSP_MEMW_SIZE16 | DSP_MEMW_WRD1, 1 },
    {"xr",
     memory_map_xr, 
     DSP_MEM_XR, 
     DSP_XI_SIZE_824,
     DSP_XI_SIZE_824+DSP_XR_SIZE_824-1L, DSP_XR_SIZE_824,
     DSP_MEMA_WR_PROT | DSP_MEMA_DON | DSP_MEMA_REAL | DSP_MEMA_DISP | DSP_MEMA_BRK, 
     NULL, 
     0xffffl, 
     DSP_MEMW_SIZE16 | DSP_MEMW_WRD1, 1 },
    {"xi", 
     memory_map_xi, 
     DSP_MEM_XI, 
     DSP_XI_SIZE_824+DSP_XR_SIZE_824,
     DSP_XI_SIZE_824+DSP_XR_SIZE_824+DSP_XI2_SIZE_824 - 1l, DSP_XI2_SIZE_824, 
     DSP_MEMA_MULT |DSP_MEMA_DON | DSP_MEMA_REAL | DSP_MEMA_DISP | DSP_MEMA_BRK, 
     NULL, 
     0xffffl, DSP_MEMW_SIZE16 | DSP_MEMW_WRD1, 1 },
    {"xi", /* hole */
     memory_map_xi, 
     DSP_MEM_XI, 
     0x1600L,/* start address */
     0x1fffL, /* end address */
     0L, /* size to allocate 0 because hole */
     DSP_MEMA_MULT | DSP_MEMA_NONE | DSP_MEMA_EXEC | DSP_MEMA_DON | DSP_MEMA_DISP | DSP_MEMA_BRK, 
     NULL, 
     0xffffl, 
     DSP_MEMW_SIZE16 | DSP_MEMW_WRD1, 1 },
    {"xi",
     memory_map_xi, 
     DSP_MEM_XI, 
     0x10000l - DSP_XP_SIZE_824, 0xffffl, 
     0l, 
     DSP_MEMA_MULT | DSP_MEMA_DON | DSP_MEMA_PERIPH | DSP_MEMA_REAL | DSP_MEMA_DISP | DSP_MEMA_BRK, 
     NULL, 
     0xffffl, 
     DSP_MEMW_SIZE16 | DSP_MEMW_WRD1, 1 }

};

extern
struct dt_mfunc dx_mfunc_56800;
extern
struct dt_mfunc dx_mfunc_56812;
extern
struct dt_mfunc dx_mfunc_56824;
static
struct dev_xpdata pval[] =
{				/* ind,inf,outd,outf,outo,aval[32] */
    {(DSPT_PIM_RESET_800 | DSPT_PIM_IRQA_800 | DSPT_PIM_IRQB_800 | DSPT_PIM_CPRESZ_800),
     (DSPT_PIM_RESET_800 | DSPT_PIM_MODB_800 | DSPT_PIM_MODA_800 | DSPT_PIM_CPRESZ_800),
     0l, 
     DSPT_PIM_CLKO_800, 0l},	/*DSP_PORT_M */
    {0l, 0l, 0l, 0l, 0l},	/*DSP_PORT_A */
    {0l, 0l, 0l, 0l, 0l},	/*DSP_PORT_B */
    {0l, 0l, 0l, 0l, 0l},	/*DSP_PORT_C */
    {0l, 0l, 0l, 0l, 0l},	/*DSP_PORT_D */
    {0l, 0l, 0l,
     (DSP_TPH0|DSP_TPH1|DSP_TPH2|DSP_TPH3|DSP_PH0|DSP_PH1)},	/*DSP_PER_CLOCK */
    {0l, 0l, 0l,
     (DSPT_PIM_BIORDW_800|DSPT_PIM_RES_800|DSPT_PIM_CKINSTOP_800|DSPT_PIM_PIACK_800|
      DSPT_PIM_PMRVAB_800)},	/*DSP_PER_CTL */
    {0l, 0l, 0l, 0x7fl, 0l},	/*DSP_PER_XAB */
    {0l, 0l, 0l, 0l, 0l},	/*DSP_PER_GDB */
    {0xfe00l, 0xfe00l, 0l, 0l, 0l},	/*DSP_PER_CXRQZ */
    {0l, 0l, 0l, 0xfe00l, 0l},	/*DSP_PER_PICVSE */
    {0l, 0l, 0l, 0l, 0l},	/*DSP_PER_VAB */
};

static
struct dt_xpid xports[] =
{				/* may include names for partial ports or aliases */
    {"ctrl", DSP_PORT_M, DSPT_POM_CTRL_800},
    {"porta", DSP_PORT_A, 0xffffl},
    {"addr", DSP_PORT_A, 0xffffl},
    {"abus", DSP_PORT_A, 0xffffl},
    {"portd", DSP_PORT_D, 0xffffl},
    {"dbus", DSP_PORT_D, 0xffffl},
    {"data", DSP_PORT_D, 0xffffl},
    {"imc", DSP_PORT_M, DSPT_POM_IMC_800}
};

public
struct dt_var dx_56800 =
{				/* one of these for each device type */
    "56800",			/* name for this device type */
    M568MAGIC,
    DSP56800,
    DSP_WRD16_36 | DSP_ADDR19,	/* device attribute flags */
    2,				/* default device startup operating mode */
    sizeof (dx_periph_56800) / sizeof (struct dt_mperiph),	/* number of periphs */
    dx_periph_56800,		/* pointer to array of peripherals */
    sizeof (mem_56800) / sizeof (struct dt_memory),		/* nummemory */
    mem_56800,		/* pointer to array of memory structures */
    "DSPMEM",			/* virtual memory file prefix */
    &dx_mfunc_56800,		/* pointer to functions which operate on mem */
    sizeof (struct dt_gsig_56800) / sizeof (unsigned long),	/* number of global signals */
    sizeof (pval) / sizeof (struct dev_xpdata),		/* number of xportval structures */
    pval,			/* initial input port values */
    (sizeof (xpin)) / (sizeof (struct dt_xpin)),	/*number of dt_xpin structures */
    xpin,			/* pointer to array of external pin name structures */
    (sizeof (xports)) / (sizeof (struct dt_xpid)),	/* number of xpid stuctures */
    xports			/* pointer to array of external ports name stuctures */
};

public
struct dt_var dx_56811 =
{				/* one of these for each device type */
    "56811",			/* name for this device type */
    M568MAGIC,
    DSP56811,
    DSP_WRD16_36 | DSP_ADDR16,	/* device attribute flags */
    2,				/* default device startup operating mode */
    sizeof (dx_periph_56811) / sizeof (struct dt_mperiph),	/* number of periphs */
    dx_periph_56811,		/* pointer to array of peripherals */
    sizeof (mem_56811) / sizeof (struct dt_memory),		/* nummemory */
    mem_56811,		/* pointer to array of memory structures */
    "DSPMEM",			/* virtual memory file prefix */
    &dx_mfunc_56800,		/* pointer to functions which operate on mem */
    sizeof (struct dt_gsig_56800) / sizeof (unsigned long),	/* number of global signals */
    sizeof (pval) / sizeof (struct dev_xpdata),		/* number of xportval structures */
    pval,			/* initial input port values */
    (sizeof (xpin)) / (sizeof (struct dt_xpin)),	/*number of dt_xpin structures */
    xpin,			/* pointer to array of external pin name structures */
    (sizeof (xports)) / (sizeof (struct dt_xpid)),	/* number of xpid stuctures */
    xports			/* pointer to array of external ports name stuctures */
};

public
struct dt_var dx_56812 =
{				/* one of these for each device type */
    "56812",			/* name for this device type */
    M568MAGIC,
    DSP56812,
    DSP_WRD16_36 | DSP_ADDR16,	/* device attribute flags */
    2,				/* default device startup operating mode */
    sizeof (dx_periph_56812) / sizeof (struct dt_mperiph),	/* number of periphs */
    dx_periph_56812,		/* pointer to array of peripherals */
    sizeof (mem_56812) / sizeof (struct dt_memory),		/* nummemory */
    mem_56812,		/* pointer to array of memory structures */
    "DSPMEM",			/* virtual memory file prefix */
    &dx_mfunc_56812,		/* pointer to functions which operate on mem */
    sizeof (struct dt_gsig_56800) / sizeof (unsigned long),	/* number of global signals */
    sizeof (pval) / sizeof (struct dev_xpdata),		/* number of xportval structures */
    pval,			/* initial input port values */
    (sizeof (xpin)) / (sizeof (struct dt_xpin)),	/*number of dt_xpin structures */
    xpin,			/* pointer to array of external pin name structures */
    (sizeof (xports)) / (sizeof (struct dt_xpid)),	/* number of xpid stuctures */
    xports			/* pointer to array of external ports name stuctures */
};

public
struct dt_var dx_56824 =
{				/* one of these for each device type */
    "56824",			/* name for this device type */
    M568MAGIC,
    DSP56824,
    DSP_WRD16_36 | DSP_ADDR16,	/* device attribute flags */
    2,				/* default device startup operating mode */
    sizeof (dx_periph_56824) / sizeof (struct dt_mperiph),	/* number of periphs */
    dx_periph_56824,		/* pointer to array of peripherals */
    sizeof (mem_56824) / sizeof (struct dt_memory),		/* nummemory */
    mem_56824,		/* pointer to array of memory structures */
    "DSPMEM",			/* virtual memory file prefix */
    &dx_mfunc_56824,		/* pointer to functions which operate on mem */
    sizeof (struct dt_gsig_56800) / sizeof (unsigned long),	/* number of global signals */
    sizeof (pval) / sizeof (struct dev_xpdata),		/* number of xportval structures */
    pval,			/* initial input port values */
    (sizeof (xpin)) / (sizeof (struct dt_xpin)),	/*number of dt_xpin structures */
    xpin,			/* pointer to array of external pin name structures */
    (sizeof (xports)) / (sizeof (struct dt_xpid)),	/* number of xpid stuctures */
    xports			/* pointer to array of external ports name stuctures */
};

#if FULLSIM
static
int mem_dispfw56800[] =
{				/* display field widths for memory types */
    11,				/* p?? */
    11,				/* x */
    11,				/* y */
    11,				/* l */
    11,				/* none */
    11,				/* laa */
    11,				/* lab */
    11,				/* lba */
    11,				/* lbb */
    11,				/* le */
    11,				/* li */
    11,				/* pa */
    11,				/* pb */
    11,				/* pe */
    11,				/* pi */
    11,				/* pr */
    11,				/* xa */
    11,				/* xb */
    11,				/* xe */
    11,				/* xi */
    11,				/* xr */
    11,				/* ya */
    11,				/* yb */
    11,				/* ye */
    11,				/* yi */
    11,				/* yr */
    11				/* xp */
};

static
char *mem_help568[] =
{				/* display field widths for memory types */
    "pe, pi or pr depending upon address and omr value.",	/* p?? */
    "xi, xe or xr depending upon address and omr values",	/* x */
    NULL,			/* y */
    NULL,			/* l */
    NULL,			/* none */
    NULL,			/* laa */
    NULL,			/* lab */
    NULL,			/* lba */
    NULL,			/* lbb */
    NULL,			/* le */
    NULL,			/* li */
    NULL,			/* pa */
    "p bank switched from x",	/* pb */
    "external program memory",	/* pe */
    "internal program ram or rom",/* pi */
    "internal bootstrap rom",	/* pr */
    NULL,			/* xa */
    NULL,			/* xb */
    "external x data memory",	/* xe */
    "on-chip x data ram",	/* xi */
    "on-chip x data rom"	/* xr */
};

static
char *mem_help56824[] =
{				/* display field widths for memory types */
    "pe, pi or pr depending upon address and omr value.",	/* p?? */
    "xi, xe or xr depending upon address and omr values",	/* x */
    NULL,			/* y */
    NULL,			/* l */
    NULL,			/* none */
    NULL,			/* laa */
    NULL,			/* lab */
    NULL,			/* lba */
    NULL,			/* lbb */
    NULL,			/* le */
    NULL,			/* li */
    NULL,			/* pa */
    "p bank switched from x",	/* pb */
    "external program memory",	/* pe */
    "internal program ram",/* pi */
    "internal program rom",	/* pr */
    NULL,			/* xa */
    NULL,			/* xb */
    "external x data memory",	/* xe */
    "on-chip x data ram",	/* xi */
    "on-chip x data rom"	/* xr */
};

extern
struct st_periph sx_core_56800, 
 sx_portb_568,
 sx_portc_568,
 sx_cop_568,
 sx_timer_568,
 sx_ssi_568,
 sx_issi_568,
 sx_spi_568,
 sx_count_56800;

#if ADSx
extern struct st_periph sx_once_56800;
extern struct st_periph sx_once_56824;
extern struct st_periph sx_jtag_56800;
#endif 

static
struct st_periph *sx_periph_56800[] =
{
    &sx_core_56800, 
    &sx_count_56800
};

static
struct st_periph *sx_periph_56811[] =
{
    &sx_core_56800, 
    &sx_ssi_568,
    &sx_timer_568,
    &sx_spi_568,
    &sx_spi_568,
    &sx_cop_568,
    &sx_portb_568,
    &sx_portc_568,
#if ADSx
    &sx_once_56800,
    &sx_jtag_56800,
#endif    
    &sx_count_56800
};

static
struct st_periph *sx_periph_56812[] =
{
    &sx_core_56800, 
    &sx_issi_568,
    &sx_timer_568,
    &sx_spi_568,
    &sx_spi_568,
    &sx_cop_568,
    &sx_portb_568,
    &sx_portc_568,
#if ADSx
    &sx_once_56824,
    &sx_jtag_56800,
#endif    
    &sx_count_56800
};

static
struct st_periph *sx_periph_56824[] =
{
    &sx_core_56800, 
    &sx_issi_568,
    &sx_timer_568,
    &sx_spi_568,
    &sx_spi_568,
    &sx_cop_568,
    &sx_portb_568,
    &sx_portc_568,
#if ADSx
    &sx_once_56824,
    &sx_jtag_56800,
#endif    
    &sx_count_56800
};

void dsptl_sasm_56800 ();
int dspt_unasm_56800 ();

extern struct st_help hlp_56800,hlp_56811;
extern struct st_cdb cdb568;

public
struct st_var sx_56800 =
{				/* one of these for each device type */
    mem_dispfw56800,		/* memory display field width */
    mem_help568,
    sx_periph_56800,		/* pointer to array of peripherals */
    dsptl_sasm_56800,		/* Simulator assembly functions */
    dspt_unasm_56800,
    &hlp_56800,
    NULL,
    &cdb568
};

public
struct st_var sx_56811 =
{				/* one of these for each device type */
    mem_dispfw56800,		/* memory display field width */
    mem_help568,
    sx_periph_56811,		/* pointer to array of peripherals */
    dsptl_sasm_56800,		/* Simulator assembly functions */
    dspt_unasm_56800,
    &hlp_56811,
    NULL,
    &cdb568
};

public
struct st_var sx_56812 =
{				/* one of these for each device type */
    mem_dispfw56800,		/* memory display field width */
    mem_help568,
    sx_periph_56812,		/* pointer to array of peripherals */
    dsptl_sasm_56800,		/* Simulator assembly functions */
    dspt_unasm_56800,
    &hlp_56811,
    NULL,
    &cdb568
};

public
struct st_var sx_56824 =
{				/* one of these for each device type */
    mem_dispfw56800,		/* memory display field width */
    mem_help56824,
    sx_periph_56824,		/* pointer to array of peripherals */
    dsptl_sasm_56800,		/* Simulator assembly functions */
    dspt_unasm_56800,
    &hlp_56811,
    NULL,
    &cdb568
};

#endif






