//---------------------------------------------------------------------------
//
// %FILE     pit.c
// %VSS-REV  $Revision: 12 $
// %CREATED  1995.12.14
// %REVISED  $Date: 4/18/97 4:17p $
// %AUTHOR   Michael C. Draeger 
// %PROJECT  NS486SXF evaluation board software
// %PART     NS486SXF, NS486SXL
// %SUMMARY  Programmable Interval Timer module 
//     
// %VSS      $Author: Miked $ $Date: 4/18/97 4:17p $ $Revision: 12 $
//
// DESCRIPTION
//
//  Function for the Programmable Interval Timer.
//
// INPUTS
//
//   PIT_V_CLOCK - used to set clock for PIT block.  Valid values are
//                   0x00 - input clock divided by 4
//                   0x01 - input clock divided by 8
//                   0x02 - input clock divided by 16
//                   0x03 - input clock divided by 32
//                 input clock is equal to the oscillator clock divided by 2.
//                 or selevtively sysclk if in power save mode
//
//   PIT_V_TICR - programs the use of the T0 and T1 pin on the NS486SXF.
//                lower four bits are used, as per the datasheet.
//
//   WD_CLOCK_INPUT - a long integer equal to the Watchdog clock input.
//                    This clock is generated from the RTC block, and
//                    should be 1000L (1 KHz) if the proper RTC oscillator
//                    is used.
//
// HISTORY
//
/*
 *
 * $History: pit.c $ 
 * 
 * *****************  Version 12  *****************
 * User: Miked        Date: 4/18/97    Time: 4:17p
 * Updated in $/nsdemo
 *  New header (comment) changes.
 * 
 * *****************  Version 11  *****************
 * User: Miked        Date: 12/04/96   Time: 4:05p
 * Updated in $/nsdemo
 * Added one typecase and one set of paranthethis to clarify code.
 * 
 * *****************  Version 10  *****************
 * User: Miked        Date: 8/06/96    Time: 11:59a
 * Updated in $/nsdemo
 * Version 1.4.  Maintainance release.  See README.TXT for info.
 * 
 * *****************  Version 9  *****************
 * User: Miked        Date: 7/23/96    Time: 2:25p
 * Updated in $/nsdemo
 * Maintainance release.  README.TXT describes changes.
 * 
 * *****************  Version 8  *****************
 * User: Miked        Date: 7/16/96    Time: 11:54a
 * Updated in $/nsdemo
 * Updated for rev C0 release.
 * 
 * *****************  Version 7  *****************
 * User: Miked        Date: 5/03/96    Time: 2:50p
 * Updated in $/nsdemo
 * Maintainence release.
 * 
 * *****************  Version 6  *****************
 * User: Miked        Date: 4/18/96    Time: 11:56a
 * Updated in $/nsdemo
 * Modified to use global clock speed.
 * 
 * *****************  Version 5  *****************
 * User: Miked        Date: 4/18/96    Time: 10:40a
 * Updated in $/nsdemo
 * Fixed release comment.
 * 
 * *****************  Version 4  *****************
 * User: Miked        Date: 4/18/96    Time: 10:35a
 * Updated in $/nsdemo
 * Cleanup for release.
 * 
 * *****************  Version 3  *****************
 * User: Miked        Date: 4/17/96    Time: 11:32a
 * Updated in $/nsdemo
 * Renamed PITf_Delay to PIT_Delay.
 * 
 * *****************  Version 2  *****************
 * User: Miked        Date: 4/04/96    Time: 2:56p
 * Updated in $/board test
 * updated headers for VSS
 *
 */
//
// COPYRIGHT
//
//      (c) 1995, 1996, 1997 National Semiconductor Corporation
//
//---------------------------------------------------------------------------

#include "pit.h"

//---------------------------------------------------------------------------
// Globals used by this module

typedef struct
{
  ULONG TicksPerTime[3];      // PIT clock ticks per time in milliseconds
  ULONG MaxTime[3];           // maximum count in milliseconds
  WORD  CountReg[3];          // count register location for each channel
  WORD  Control[3];           // control register mask for each channel
} PIT_GLOBAL;

PRIVATE PIT_GLOBAL pitG;

//---------------------------------------------------------------------------

// INPUT        none
// OUTPUT       none
// RETURN       USHORT - FAIL if PIT not enabled in BIU
//                       SUCCESS if PIT initialized successfully
//
// DESCRIPTION  this function should be called before using the other
//              PIT functions to initialize the PIT for use

USHORT PIT_Initialize( void )
{
  
  ULONG PitClock;     // used to determine PIT clock for each timer
  BYTE  TCRval;       // temp. var used to figure out clock divisor
  BYTE  tByte;        // temp. variable

  // If PIT is not enabled in the BIU, return FAIL

  if ( ( IOR_BYTE( BIU_CONTROL1 ) & 0x08 ) != 0x08 )
    return FAIL;

  // Program the timer clock - mask off reserved bits

  IOW_BYTE( PIT_CLOCK, ( PIT_V_CLOCK & TCR_TCD ) );

  // Program the function of T0 and T1 pins
  // Set the two gate inputs low

  tByte = PIT_V_TICR;                      // value from header
  tByte &= ( TICR_T0SEL | TICR_T1SEL );    // mask off other bits
  tByte &= ( ~ ( TICR_TG0 | TICR_TG1 ) );  // clear both gates
  IOW_BYTE( PIT_TICR, tByte );             // write to register

  // get CPU speed from global.  If not valid, use default.
  if ( gTargetInfo.valid == TRUE )
    PitClock = gTargetInfo.raw_cpu_speed;
  else
    PitClock = DEFAULT_PIT_CLOCK_INPUT;
  
  // Compute clock input to PIT block
  
  TCRval = IOR_BYTE( PIT_CLOCK );     // read current clock divisor
  TCRval &= TCR_TCD;                  // discard reserved bits
  if ( TCRval == TCD_Div4 )
    PitClock /= 4;
  else if ( TCRval == TCD_Div8 )
    PitClock /= 8;
  else if ( TCRval == TCD_Div16 )
    PitClock /= 16;
  else
    PitClock /= 32;

  // This code computes the number of PIT clock ticks per 
  // millisecond (pitG.TicksPerTime[n]) by dividing the clock
  // to the pit block by 1000.  (1000ms per seconds).
  // It then computes the maximum time in milliseconds by multiplying
  // the time per ticks by 0xFFFF, which is the maximum number of ticks
  // the PIT can count. (Actually 0 is, which counts one more tick than
  // 0xFFFF, but is inconvienient to use).

  pitG.TicksPerTime[0] = ( PitClock / 1000 );
  pitG.MaxTime[0] = ( ( 0xFFFF ) / pitG.TicksPerTime[0] );
  
  pitG.TicksPerTime[1] = ( PitClock / 1000 );
  pitG.MaxTime[1] = ( ( 0xFFFF ) / pitG.TicksPerTime[1] );

  pitG.TicksPerTime[2] = ( WD_CLOCK_INPUT / 1000 );
  pitG.MaxTime[2] = ( ( 0xFFFF ) / pitG.TicksPerTime[2] );

  // Define which counter register is associated with each channel
  // This makes is easier to write to the proper timer in the
  // functions.

  pitG.CountReg[0] = PIT_COUNT0;
  pitG.CountReg[1] = PIT_COUNT1;
  pitG.CountReg[2] = PIT_COUNT2;

  // These are used to program each timer.  We only use Binary mode
  // and read/write lsb followed by msb mode to program timers, so
  // these are included with the proper timer select.  Code to
  // program timers should OR is value with the mode they want.

  pitG.Control[0] = SELECT_Counter0 | BCD_Binary | COMMAND_RWlsbmsb; 
  pitG.Control[1] = SELECT_Counter1 | BCD_Binary | COMMAND_RWlsbmsb; 
  pitG.Control[2] = SELECT_Counter2 | BCD_Binary | COMMAND_RWlsbmsb; 
  
  // All done, return

  return SUCCESS;

}

//---------------------------------------------------------------------------

// INPUT       PIT_SET * Set - pointer to structure with the following
//                USHORT  Timer  - which channel (0,1,2)
//                USHORT  Mode   - mode of operation (0-5)
//                ULONG   Time   - time to count in ms (1-maxtime)
//                USHORT  Gate   - action to gate 
//                                   PIT_GateSet - set gate
//                                   PIT_GateClear - clear gate
//                                   PIT_GateNA - no action to gate
//
// OUTPUT      none
// RETURN      USHORT - INVALID_PARAMETER if any paramer is invalid 
//                      SUCCESS if pit set correctly
//
// DESCRIPTION this function programs a PIT channel to with the given 
//             parameters.

USHORT PIT_Set( PIT_SET * Set )
{

  SEGWORDU TimerTicks;  // used to program lsb and msb of timer
  BYTE     tByte;       // scratch byte

  // Validate parameters

  if ( Set == NULL )
    return INVALID_PARAMETER;

  if ( (Set->Timer > 2) || (Set->Mode > 5) || (Set->Gate > 2) )
    return INVALID_PARAMETER;
  
  if ( (Set->Time > pitG.MaxTime[Set->Timer]) || (Set->Time == 0) )
    return INVALID_PARAMETER;

  if ( (Set->Timer == 2) && (Set->Gate != PIT_GateNA) )
    return INVALID_PARAMETER;
  
  if ( (Set->Timer == 2) && (Set->Mode != 5) )
    return INVALID_PARAMETER;

  // Convert time is msec into timer ticks

  TimerTicks.linear = (WORD) (pitG.TicksPerTime[Set->Timer] * Set->Time );

  // Program  

  IOW_BYTE ( PIT_CONTROL, 
              ( pitG.Control[Set->Timer] | ( Set->Mode << 1 ) ) 
           ); 
  IOW_BYTE ( pitG.CountReg[Set->Timer], TimerTicks.seg.lsb );
  IOW_BYTE ( pitG.CountReg[Set->Timer], TimerTicks.seg.msb );

  // Program Gate

  if ( Set->Gate == PIT_GateSet )
  {
    tByte = IOR_BYTE( PIT_TICR );
    if ( Set->Timer == 0 )
      IOW_BYTE ( PIT_TICR, (tByte | TICR_TG0) );
    else
      IOW_BYTE ( PIT_TICR, (tByte | TICR_TG1) );
  }
  else if ( Set->Gate == PIT_GateClear )
  {  
    tByte = IOR_BYTE( PIT_TICR );
    if ( Set->Timer == 0 )
      IOW_BYTE ( PIT_TICR, (tByte & (~TICR_TG0)) );
    else
      IOW_BYTE ( PIT_TICR, (tByte & (~TICR_TG1)) );
  }

  // All Set

  return SUCCESS;

}

//---------------------------------------------------------------------------

// INPUT       USHORT channel - which PIT channel (0,1)  
// OUTPUT      none
// RETURN      USHORT - INVALID_PARAMETER if invalid channel specified 
//                      SUCCESS if PIT channel canceled
//
// DESCRIPTION cancel a given PIT channel.  The WD channel (2) uses its 
//             own API for canceling so is not supported.

USHORT PIT_Cancel( USHORT channel )
{
  
  // Set Gate to 0 if channel 1 or 0.  This does nothing if the gates
  // are driven from the T0 or T1 pin, but it is easier to just clear
  // the gate bit than to check if it will matter.

  if ( channel == 0 )
    IOW_BYTE ( PIT_TICR, ( IOR_BYTE( PIT_TICR ) & (~TICR_TG0) ) );
  else if ( channel == 1 )
    IOW_BYTE ( PIT_TICR, ( IOR_BYTE( PIT_TICR ) & (~TICR_TG1) ) );
  else
    return INVALID_PARAMETER;

  // Program channel to Mode 0, with a count of 1.  If the gate is 
  // high (this will only happen if it is driven by an external
  // source), the OUT signal will go high after one clock tick.  Otherwise,
  // out will remain low.
  
  IOW_BYTE ( PIT_CONTROL, ( pitG.Control[channel] | MODE_0 ) ); 
  IOW_BYTE ( pitG.CountReg[channel], 1 ); // lsb is 1
  IOW_BYTE ( pitG.CountReg[channel], 0 ); // msb is 0

  // All done

  return SUCCESS;

}

//---------------------------------------------------------------------------

// INPUT       USHORT channel - which channel (0,1) 
//             USHORT gate - action to gate
//                                   PIT_GateSet - set gate
//                                   PIT_GateClear - clear gate
//
// OUTPUT      none
// RETURN      USHORT - INVALID_PARAMETER if parameters incorrect 
//                      SUCCESS if gate set of cleared as indicated
//
// DESCRIPTION this function allows you to set or clear the gate line 
//             of the given channel.  This only works if the channel
//             is set up to use the internally generated gate (vs. an
//             external hardware pin).  In most modes of operation the
//             gate line determines if counting in enabled.  Therefore,
//             this function allows the user to stop and start counting
//             easily.

USHORT PIT_Gate( USHORT channel, USHORT gate  )
{
  
  BYTE tByte;   // scratch byte
  
  // validate parameters

  if ( channel > 1 )
    return INVALID_PARAMETER;

  if ( ! ( ( gate == PIT_GateSet ) || ( gate == PIT_GateClear ) ) ) 
    return INVALID_PARAMETER;

  // Set or clear gate as requested

  if ( gate == PIT_GateSet )
  {
    tByte = IOR_BYTE( PIT_TICR );
    if ( channel == 0 )
      IOW_BYTE ( PIT_TICR, (tByte | TICR_TG0) );
    else
      IOW_BYTE ( PIT_TICR, (tByte | TICR_TG1) );
  }
  else if ( gate == PIT_GateClear )
  {  
    tByte = IOR_BYTE( PIT_TICR );
    if ( channel == 0 )
      IOW_BYTE ( PIT_TICR, (tByte & (~TICR_TG0)) );
    else
      IOW_BYTE ( PIT_TICR, (tByte & (~TICR_TG1)) );
  }
  
  // All done

  return SUCCESS;

}

//---------------------------------------------------------------------------

// INPUT       USHORT channel - which channel (0,1,2) 
//             PIT_STATUS * pStatus - pointer to structure for output
// OUTPUT      PIT_STATUS * pStatus - the following is filled in 
//               USHORT  Mode      - PIT mode (0-5)
//               ULONG   Time      - current time in ms (1-maxtime)
//               BOOL    TimeValid - count is loaded in PIT if TRUE
//               USHORT  Gate      - status of Gate line
//                                   PIT_GateSet - gate set
//                                   PIT_GateClear - gate clear
//                                   PIT_GateNA - not applicable
//               BOOL    Output    - TRUE if OUT signal is high
//
// RETURN        USHORT - INVALID_PARAMETER if any incorrect parameters
//                        SUCCESS if status returned
//
// DESCRIPTION   returns status information for a given PIT channel.
//               The status information is dependant on the programmed
//               mode of the PIT.

USHORT PIT_Status( USHORT channel, PIT_STATUS * pStatus )
{

  BYTE tByte;   // scratch byte
  BYTE lsb;     // lsb of count in timer
  BYTE msb;     // msb of count in timer
  WORD count;   // whole count in timer

  // validate parameters

  if ( channel > 2 )
    return INVALID_PARAMETER;

  if ( pStatus == NULL )
    return INVALID_PARAMETER;

  // issue readback comand for appropriate counter

  if ( channel == 0 )
    tByte = READBACK_Counter0;
  else if ( channel == 1)
    tByte = READBACK_Counter1;
  else
    tByte = READBACK_Counter2;
   
  IOW_BYTE ( PIT_CONTROL, ( SELECT_ReadBack |
                            READBACK_LatchCount |
                            READBACK_LatchStatus |
                            tByte ) );

  // read status byte

  tByte = IOR_BYTE ( pitG.CountReg[channel] );

  // read count
  
  if ( (tByte & STATUS_NULLCount) != STATUS_NULLCount )
  {
    pStatus->TimeValid = TRUE;
    lsb = IOR_BYTE ( pitG.CountReg[channel] );
    msb = IOR_BYTE ( pitG.CountReg[channel] );
  }
  else
  {
    pStatus->TimeValid = FALSE;
    lsb = msb = 0;
  }

  // fill in mode.  Since the datasheet lists bit 3 of the  
  // mode bits as undefined if bit 2 is a one (modes 2 and 3),
  // the code will zero out bit 3 if bit 2 is one, thus only
  // values from 0 to 5 can be returned.
  
  pStatus->Mode = (USHORT) ( ( tByte & STATUS_Mode ) >> 1 );
  if ( ( pStatus->Mode & 0x02 ) == 0x02 )
    pStatus->Mode &= ( ~(0x04) );

  // fill in output

  if ( ( tByte & STATUS_Output ) == STATUS_Output )
    pStatus->Output = TRUE;
  else
    pStatus->Output = FALSE;

  // fill in remaining time

  count = msb;
  count = (WORD) ( count << 8 );
  count |= lsb;
  
  if ( count > 0 )
    pStatus->Time = ( count / pitG.TicksPerTime[channel] );
  else
    pStatus->Time = 0;

  // fill in gate

  if ( channel == 0 )
  {
    if ( ( IOR_BYTE ( PIT_TICR ) & TICR_OUT0 ) == TICR_OUT0 )
      pStatus->Gate = PIT_GateSet;
    else
      pStatus->Gate = PIT_GateClear;
  }
  else if ( channel == 1 )
  {
    if ( ( IOR_BYTE ( PIT_TICR ) & TICR_OUT1 ) == TICR_OUT1 )
      pStatus->Gate = PIT_GateSet;
    else
      pStatus->Gate = PIT_GateClear;
  }
  else
    pStatus->Gate = PIT_GateNA;

  // All done

  return SUCCESS;

}

//---------------------------------------------------------------------------

// INPUT       USHORT timer - which channel to use (0,1) 
//             ULONG time - time in milliseconds
// OUTPUT      none 
// RETURN      USHORT - INVALID_PARAMETER if invalid timer specified
//                      FAIL if PIT setup fails (could be a invalid time
//                        parameter)
//                      SUCCESS after indicted delay is counted
//
// DESCRIPTION this function programs a PIT channel and polls until 
//             the given time has elapsed.  This can be used to implement
//             fairly accurate delays in code.  The caller must make sure
//             that the timer they wish to use is not already in use.

USHORT PIT_Delay( USHORT timer, ULONG time )
{

  PIT_SET Set;
  PIT_STATUS Status;
  BOOL loopflag;

  // Check parameters

  if ( timer >= 2 )
    return INVALID_PARAMETER;

  // Set up Timer

  Set.Timer = timer;         // channel requested
  Set.Mode =  0;             // mode 0
  Set.Time =  time;          // time requested
  Set.Gate =  PIT_GateSet;   // clear

  if ( PIT_Set( & Set ) != SUCCESS )
    return FAIL;
  
  // Loop until done

  loopflag = TRUE;

  while ( loopflag == TRUE )
  {
  
    // Get Status

    PIT_Status( 1, & Status );

    // See if output has gone high

    if ( Status.Output == TRUE )
      loopflag = FALSE;

  }

  // return

  return SUCCESS;

}

//---------------------------------------------------------------------------
// END       pit.c
//---------------------------------------------------------------------------
