//---------------------------------------------------------------------------
//
// %FILE     dram.c
// %VSS-REV  $Revision: 16 $
// %CREATED  1996.01.17
// %REVISED  $Date: 4/18/97 4:04p $
// %AUTHOR   Noreen Bell
// %PROJECT  NS486SXF evaluation board software
// %PART     NS486SXF, NS486SXL
// %SUMMARY  Dynamic RAM module
//
// %VSS      $Author: Miked $ $Date: 4/18/97 4:04p $ $Revision: 16 $
//
// DESCRIPTION
//
// INPUTS
//      Register locations.
//
//      DRAM_REFRESH            Refresh rate register
//      DRAM_RAS                RAS* timeout register
//      DRAM_BANK               Dram bank size register
//      DRAM_MASK0              Bank 0 mask register
//      DRAM_MASK1              Bank 1 mask register
//      DRAM_ADDR1              Bank 1 address register
//      DRAM_CONTROL            Dram control register
//      DRAM_STATUS             Dram status register
//
// HISTORY
//
/*
 *
 * $History: dram.c $
 * 
 * *****************  Version 16  *****************
 * User: Miked        Date: 4/18/97    Time: 4:04p
 * Updated in $/nsdemo
 * Minor changes in #defines.   New header (comment) changes.
 * 
 * *****************  Version 15  *****************
 * User: Miked        Date: 12/04/96   Time: 3:58p
 * Updated in $/nsdemo
 * Type cast some statements to get rid of compiler warnings.
 * 
 * *****************  Version 14  *****************
 * User: Miked        Date: 8/06/96    Time: 11:59a
 * Updated in $/nsdemo
 * Version 1.4.  Maintainance release.  See README.TXT for info.
 * 
 * *****************  Version 13  *****************
 * User: Miked        Date: 7/23/96    Time: 2:25p
 * Updated in $/nsdemo
 * Maintainance release.  README.TXT describes changes.
 * 
 * *****************  Version 12  *****************
 * User: Miked        Date: 7/16/96    Time: 11:54a
 * Updated in $/nsdemo
 * Updated for rev C0 release.
 * 
 * *****************  Version 11  *****************
 * User: Noreen       Date: 7/10/96    Time: 5:09p
 * Updated in $/nstest
 * Fixed bug in DRAM_Status function 
 * 
 * *****************  Version 10  *****************
 * User: Miked        Date: 5/03/96    Time: 3:16p
 * Updated in $/nsdemo
 * Fixed bug is DRAM size display.
 * 
 * *****************  Version 9  *****************
 * User: Miked        Date: 5/03/96    Time: 2:50p
 * Updated in $/nsdemo
 * Maintainence release.
 * 
 * *****************  Version 8  *****************
 * User: Noreen       Date: 4/29/96    Time: 2:04p
 * Updated in $/nsdemo
 * When calculating RAS timeout and Refresh rate based on cpu_speed added
 * check to ensure ICA cpu_speed is non-zero as well as check for valid
 * target info.
 * 
 * *****************  Version 7  *****************
 * User: Miked        Date: 4/18/96    Time: 12:51p
 * Updated in $/nsdemo
 * Added check to see if CPU speed is valid before using.
 * 
 * *****************  Version 6  *****************
 * User: Noreen       Date: 4/18/96    Time: 12:20p
 * Updated in $/nsdemo
 * Updated calculation of refresh and RAS timeout values to use global
 * cpuspeed
 * 
 * *****************  Version 5  *****************
 * User: Noreen       Date: 4/18/96    Time: 11:01a
 * Updated in $/nsdemo
 * Updated comments for release version.
 * 
 * *****************  Version 4  *****************
 * User: Miked        Date: 4/17/96    Time: 1:28p
 * Updated in $/nsdemo
 * Typed a few IO macros to avoid compile warning.
 * 
 * *****************  Version 3  *****************
 * User: Noreen       Date: 4/16/96    Time: 2:53p
 * Updated in $/nsdemo
 * Updated Bit definitions for DMA_Status function. 
 * Implemented fix for returning parity status.
 * 
 * *****************  Version 2  *****************
 * User: Noreen       Date: 4/12/96    Time: 3:00p
 * Updated in $/nstest
 * Updated headers for VSS
 *
 */
//
// COPYRIGHT
//
//      (c) 1996, 1997 National Semiconductor Corporation
//
//---------------------------------------------------------------------------

#include "dram.h"

//---------------------------------------------------------------------------
//
// FUNCTION    DRAM_Status
//
// INPUT/OUTPUT DRAMC_STATUS *pStatus - a pointer to the structure
//                                      which holds the status.
// RETURN      USHORT
//               SUCCESS if call successful
//               INVALID_PARAMETER if pointer passes to function is NULL
// DESCRIPTION
//  This function returns status information associated with the DRAM
//  controller.
//
// NOTES
//
//       typedef struct {
//               USHORT Parity;         -Disabled=0,Enabled=1,
//                                      -Enabled with NMI=2
//               USHORT Pagesize0;      -8,9,10,11,12,0(Reserved)
//               USHORT Pagesize1;      -8,9,10,11,12,0(Reserved)
//               USHORT CyclePageMiss;  -3,4
//               BOOL ExtendedCAS;      -Normal=0, Extended=1
//               USHORT RefreshRate;    -microseconds
//               USHORT RASTimeout;     -microseconds
//               USHORT ParityError;    -NoParityError=0, ParityErrorBank0=1
//                                      -ParityErrorBank1=2, ParityErrorBoth=3
//               USHORT BankSize0;      -value=0(0.5),1,2,4,8
//               USHORT BankSize1;      -value=0(0.5),1,2,4,8
//               ULONG Bank1Address;
//       } DRAMC_STATUS;
//
//---------------------------------------------------------------------------

USHORT DRAM_Status(DRAMC_STATUS * pStatus)
{

	USHORT DRAMControl, DRAMStatus, RefreshReg, RASReg;
	BYTE UpperControl, LowerControl;
	int Oscillator_Period;

	if(pStatus==NULL)
	   return INVALID_PARAMETER;

	//Get Upper and Lower Values of DRAM Control Register
	DRAMControl = IOR_WORD(DRAM_CONTROL);
	UpperControl = (BYTE) (DRAMControl >> 8);
	LowerControl = (BYTE) DRAMControl;

	if((LowerControl & (1 << 2)) == 0)
	   pStatus->Parity = Disabled;
	else if((UpperControl & (1 << 6)) == 0)
	   pStatus->Parity = Enabled_NMI;
	else
	   pStatus->Parity = Enabled;

	if((LowerControl & (1 << 7)) == 0)
	   pStatus->CyclePageMiss = ThreeCycle;
	else
	   pStatus->CyclePageMiss = FourCycle;

	if((LowerControl & (1 << 4)) == 0)
	   pStatus->ExtendedCAS = Normal;
	else
	   pStatus->ExtendedCAS = Extended;

	//Decode DRAM Page Size, Bank 1
	switch((UpperControl >> 3) & PageMask)
	{
	   case 0:
	       pStatus->Pagesize1 = 8;
	       break;

	   case 1:
	       pStatus->Pagesize1 = 9;
	       break;

	   case 2:
	       pStatus->Pagesize1 = 10;
	       break;

	   case 3:
	       pStatus->Pagesize1 = 11;
	       break;

	   case 4:
	       pStatus->Pagesize1 = 12;
	       break;

	   case 5:
	       pStatus->Pagesize1 = 0;
	       break;

	   case 6:
	       pStatus->Pagesize1 = 0;
	       break;

	   case 7:
	       pStatus->Pagesize1 = 0;
	       break;

	   default:
		break;
	}

	//Decode DRAM Page Size, Bank 0
	switch((UpperControl) & PageMask)
	{
	   case 0:
	       pStatus->Pagesize0 = 8;
	       break;

	   case 1:
	       pStatus->Pagesize0 = 9;
	       break;

	   case 2:
	       pStatus->Pagesize0 = 10;
	       break;

	   case 3:
	       pStatus->Pagesize0 = 11;
	       break;

	   case 4:
	       pStatus->Pagesize0 = 12;
	       break;

	   case 5:
	       pStatus->Pagesize0 = 0;
	       break;

	   case 6:
	       pStatus->Pagesize0 = 0;
	       break;

	   case 7:
	       pStatus->Pagesize0 = 0;
	       break;

	   default:
		break;
	}

	//Decode DRAM Bank 1 Size
	switch( ( (IOR_BYTE(DRAM_BANK) >> 4 ) & BankMask ) )
	{
	   case 0:
	       pStatus->BankSize1 = halfMb;
	       break;

	   case 1:
	       pStatus->BankSize1 = oneMb;
	       break;

	   case 2:
	       pStatus->BankSize1 = twoMb;
	       break;

	   case 3:
	       pStatus->BankSize1 = fourMb;
	       break;

	   case 4:
	       pStatus->BankSize1 = eightMb;
	       break;

	   case 5:
	       pStatus->BankSize1 = 0;
	       break;

	   case 6:
	       pStatus->BankSize1 = 0;
	       break;

	   case 7:
	       pStatus->BankSize1 = 0;
	       break;

	   default:
		break;
	}

	//Decode DRAM Bank 0 Size
	switch( ( IOR_BYTE(DRAM_BANK) & BankMask ) )
	{
	   case 0:
	       pStatus->BankSize0 = halfMb;
	       break;

	   case 1:
	       pStatus->BankSize0 = oneMb;
	       break;

	   case 2:
	       pStatus->BankSize0 = twoMb;
	       break;

	   case 3:
	       pStatus->BankSize0 = fourMb;
	       break;

	   case 4:
	       pStatus->BankSize0 = eightMb;
	       break;

	   case 5:
	       pStatus->BankSize0 = 0;
	       break;

	   case 6:
	       pStatus->BankSize0 = 0;
	       break;

	   case 7:
	       pStatus->BankSize0 = 0;
	       break;

	   default:
		break;
	}

	//Get Value of Refresh Rate Register
	RefreshReg = IOR_WORD(DRAM_REFRESH);

	//Get Value of RAS Timeout Register
	RASReg = IOR_WORD(DRAM_RAS);

	//Calculate Oscillator Period
	//Period = 1000/2cpu_speed
	if ( (gTargetInfo.valid == TRUE) && (gTargetInfo.cpu_speed != 0) )
	  Oscillator_Period = 500/gTargetInfo.cpu_speed;
	else
	  Oscillator_Period = 500/25;  // default to 25MHz

	//Refresh period = (Oscillator period) X (Refresh Register Value)
	//Divide by 1000 to get value in microseconds
	pStatus->RefreshRate = (USHORT)( (Oscillator_Period * RefreshReg) / 1000 );

	//RAS Timeout period = (Oscillator period) X (RAS Register Value)
	//Divide by 1000 to get value in microseconds
	pStatus->RASTimeout = (USHORT)( (Oscillator_Period * RASReg) / 1000 );

	DRAMStatus = IOR_WORD(DRAM_STATUS);

	//Get Status of Parity Error, only if detected
	if((DRAMStatus & (1 << 1)) != 0)
	  switch((DRAMStatus >> 1 ) & BitMask)
	  {
	      case BANK1:
		pStatus->ParityError = ParityErrorBank1;
		break;

	      case BANK0:
		pStatus->ParityError = ParityErrorBank0;
		break;

	      case BOTH:
		pStatus->ParityError = ParityErrorBoth;
		break;

	      default:
		break;
	  }
	else
	  pStatus->ParityError = NoParityError;

	//Get Value of Bank 1 Address
	//Bank1 Address = (Address Register & MaskRegister) << 19
	pStatus->Bank1Address = ((IOR_BYTE(DRAM_ADDR1) & IOR_BYTE(DRAM_MASK1)) << 19);

	return SUCCESS;

}

//---------------------------------------------------------------------------
// END       dram.c
//---------------------------------------------------------------------------
