//////////////////////////////////////////////////////////////////////
//TargetD64 - C64 archive related conversion tool and emulator frontend
//////////////////////////////////////////////////////////////////////
//Copyright (C) 1998, 1999  Karlheinz Langguth klangguth@netscape.net
//
//This program is free software; you can redistribute it and/or
//modify it under the terms of the GNU General Public License
//as published by the Free Software Foundation; either version 2
//of the License, or (at your option) any later version.
//
//This program is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//GNU General Public License for more details.
//
//You should have received a copy of the GNU General Public License
//along with this program; if not, write to the Free Software
//Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//////////////////////////////////////////////////////////////////////

//avoid multiple include of header file
#ifndef _PROFILE_HEADER
#define _PROFILE_HEADER

//whenever a profile exception occurs this class is as exception object
class CFProfileException {};

const string SECTIONMARKER("SeCtIoNsTaRt"); //this marks a start of section line
const string EMPTYENTRY("EmPtYeNtRy"); //this marks entry lines without = sign
const unsigned int ALL((unsigned int)-1);

class CFProfile
{
public:
	class CFEntryLine
	{
	friend CFProfile;

	private:
		string m_entry; //entry key
		string m_value; //value
		unsigned int m_lineNo; //lineNo of entry in file
	public:
		CFEntryLine(const string& entry = "", const string& val = "", const unsigned int lineNo = (unsigned int)-1)
			: m_entry(entry), m_value(val), m_lineNo(lineNo) {}
	public:
		inline bool operator==(const CFEntryLine& a) const
		{ return (m_entry == a.m_entry); }
		inline bool operator!=(const CFEntryLine& a) const
		{ return !(*this == a); }
		inline bool operator<(const CFEntryLine& a) const
		{ return (m_entry < a.m_entry); }
		inline bool operator<=(const CFEntryLine& a) const
		{ return ((*this == a) || (*this < a)); }
		inline bool operator>(const CFEntryLine& a) const
		{ return !(*this <= a); }
		inline bool operator>=(const CFEntryLine& a) const
		{ return !(*this < a); }
	};
	typedef multimap<string, CFEntryLine> tSectionEntry;

public:
	//determine profile name - if not existent create empty
	//if P2 is empty eval default directory of profile else use P2
	//P1 I: infix name of profile
	//P2 I: directory where profile lives
	CFProfile(const string& appName, const string& profileDir = "");

public:
	//parse the profile file
	void ParseProfile(void) throw (CFProfileException);
	//read a special entry value from profile
	//always returns FIRST value found (if multiple there)
	//P1 I: section
	//P2 I: entry
	//P3 O: value
	//R: true means entry found
	//   false means entry not found
	bool ReadProfileValue(const string& section, const string& entry, vector<string>& aValueEntry) const;
	//read a special entry value from profile
	//always return ALL values found in vector P3
	//P1 I: section
	//P2 I: entry
	//P3 O: vector of values matching entry
	//R: true means entry found
	//   false means entry not found
	bool ReadProfileValue(const string& section, const string& entry, string& value) const;
	//read a special entry value from profile
	//always return FIRST value found in vector P3 ONLY if it is an boolean value
	//P1 I: section
	//P2 I: entry
	//P3 O: vector of values matching entry
	//R: true means entry found
	//   false means entry not found
	bool ReadProfileValue(const string& section, const string& entry, bool& arg) const;
	//writes an entry into profile
	//ALWAYS changes file AND cache
	//P1 I: section
	//P2 I: entry
	//P3 I: value
	void WriteProfileValue(const string& section, const string& entry, const string& value) throw (CFProfileException);
	//Set this object as application wide profile server
	inline void SetAppProfile() { s_pAppProfile = this; }

private:
	//read first matching entry object from profile
	//always returns FIRST value found (if multiple there)
	//P1 I: section
	//P2 I: entry
	//P3 O: pointer to entry object matching the query
	//R: true means entry found
	//   false means entry not found
	bool ReadProfileValue(const string& section, const string& entry, CFEntryLine*& pEntryLine);
	//read all matching entry objects from profile
	//always returns FIRST value found (if multiple there)
	//P1 I: section
	//P2 I: entry
	//P3 O: vector of pointer to entry objects matching the query
	//R: true means entry found
	//   false means entry not found
	bool ReadProfileValue(const string& section, const string& entry, vector<CFEntryLine *>& aEntryLine);
	//opens streams to original profile file and new profile file
	//P1 O: stream of original profile file
	//P2 O: stream of new profile file
	void InitProfileChange(ifstream& in, ofstream& out) const throw (CFProfileException);
	//closes streams to original profile file and new profile file
	//and renames new profile to original profile
	//P1 I: stream of original profile file
	//P2 I: stream of new profile file
	void CommitProfileChange(ifstream& in, ofstream& out) const throw (CFProfileException);
	//copies P3 lines from P1 to P2
	//P1 I: stream of original profile file
	//P2 I: stream of new profile file
	//P3 I: number of lines to copy
	void CopyProfileLines(ifstream& in, ofstream& out, const unsigned int lines) const throw (CFProfileException);
	//skip P2 lines of P1
	//P1 I: stream of original profile file
	//P2 I: number of lines to skip
	void SkipProfileLines(ifstream& in, const unsigned int lines) const throw (CFProfileException);

public:
	//Get pointer to the application wide object server
	inline static const CFProfile* GetAppProfile(void) { return s_pAppProfile; }
private:
	//a helper function for STL algo transform
	static string PointerToString(CFEntryLine *e);


public:
	string m_profileName;

private:
	tSectionEntry m_sectionEntry;

private:
	static CFProfile* s_pAppProfile; //pointer to option object that is application wide server
};

#endif
